/**
{file:
	{name: LmiProperties.h}
	{description: Interface definition for a properties class.}
	{copyright:
		(c) 2014-2017 Vidyo, Inc.,
		433 Hackensack Avenue,
		Hackensack, NJ  07601.

		All rights reserved.

		The information contained herein is proprietary to Vidyo, Inc.
		and shall not be reproduced, copied (in whole or in part), adapted,
		modified, disseminated, transmitted, transcribed, stored in a retrieval
		system, or translated into any language in any form by any means
		without the express written consent of Vidyo, Inc.
							***** CONFIDENTIAL *****
	}
}
*/

#if !defined(LMI_PROPERTIES_H_)
#define LMI_PROPERTIES_H_

#include <Lmi/Utils/LmiTypes.h>
#include <Lmi/Utils/LmiString.h>
#include <Lmi/Utils/LmiResolution.h>
#include <Lmi/Utils/LmiMediaFormat.h>
#include <Lmi/Utils/LmiMap.h>
#include <Lmi/Utils/LmiVector.h>
#include <Lmi/Utils/LmiStack.h>
#include <Lmi/Utils/LmiStringVector.h>
#include <Lmi/Utils/LmiMisc.h>
#include <Lmi/Os/LmiFile.h>
#include <Lmi/Os/LmiTime.h>
#include <Lmi/Os/LmiSharedPtr.h>
#include <Lmi/Os/LmiMutex.h>

LMI_BEGIN_EXTERN_C

struct LmiProperties_;

/**
{type:
	{name: LmiPropertyValueType}
	{parent: LmiPropertyValue}
	{include: Lmi/Os/LmiProperties.h}
	{description: Describes the type of value stored in an LmiPropertyValue.}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_Bool}
		{description: An LmiBool.}
	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_Int}
		{description: An LmiInt.}
	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_Int16}
		{description: An LmiInt16.}
	}
  	{value:
 		{name: LMI_PROPERTYVALUE_TYPE_Int64}
 		{description: An LmiInt64.}
 	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_Uint}
		{description: An LmiUint.}
	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_Uint16}
		{description: An LmiUint16.}
	}
 	{value:
		{name: LMI_PROPERTYVALUE_TYPE_Uint64}
		{description: An LmiUint64.}
	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_SizeT}
		{description: An LmiSizeT.}
	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_Float32}
		{description: An LmiFloat32.}
	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_Float64}
		{description: An LmiFloat64.}
	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_Time}
		{description: An LmiTime.}
	}
 	{value:
		{name: LMI_PROPERTYVALUE_TYPE_Resolution}
		{description: An LmiResolution.}
	}
  	{value:
		{name: LMI_PROPERTYVALUE_TYPE_MediaFormat}
		{description: An LmiMediaFormat.}
	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_VoidPtr}
		{description: An LmiVoidPtr.}
	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_String}
		{description: An LmiString.}
	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_Properties}
		{description: An LmiProperties.}
	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_Vector}
		{description: An LmiVector(LmiPropertyValue).}
	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_SharedVoidPtr}
		{description: An LmiSharedVoidPtr.}
	}
 	{value:
		{name: LMI_PROPERTYVALUE_TYPE_WeakVoidPtr}
		{description: An LmiWeakVoidPtr.}
	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_SharedPtr}
		{description: An LmiSharedPtr.}
	}
    {value:
        {name: LMI_PROPERTYVALUE_TYPE_WeakPtr}
        {description: An LmiWeakPtr.}
	}
	{value:
		{name: LMI_PROPERTYVALUE_TYPE_Wrapper}
		{description: Any object pointer.}
	}
}
*/
typedef enum {
	LMI_PROPERTYVALUE_TYPE_Bool,
	LMI_PROPERTYVALUE_TYPE_Int,
	LMI_PROPERTYVALUE_TYPE_Int16,
	LMI_PROPERTYVALUE_TYPE_Int64,
	LMI_PROPERTYVALUE_TYPE_Uint,
	LMI_PROPERTYVALUE_TYPE_Uint16,
	LMI_PROPERTYVALUE_TYPE_Uint64,
	LMI_PROPERTYVALUE_TYPE_SizeT,
	LMI_PROPERTYVALUE_TYPE_Float32,
	LMI_PROPERTYVALUE_TYPE_Float64,
	LMI_PROPERTYVALUE_TYPE_Time,
	LMI_PROPERTYVALUE_TYPE_Resolution,
	LMI_PROPERTYVALUE_TYPE_MediaFormat,
	LMI_PROPERTYVALUE_TYPE_VoidPtr,
	LMI_PROPERTYVALUE_TYPE_String,
	LMI_PROPERTYVALUE_TYPE_Properties,
	LMI_PROPERTYVALUE_TYPE_Vector,
	LMI_PROPERTYVALUE_TYPE_SharedVoidPtr,
	LMI_PROPERTYVALUE_TYPE_WeakVoidPtr,
    LMI_PROPERTYVALUE_TYPE_SharedPtr,
    LMI_PROPERTYVALUE_TYPE_WeakPtr,
    LMI_PROPERTYVALUE_TYPE_Wrapper
} LmiPropertyValueType;

/* Forward declare LmiPropertyValue, so that LmiVector(LmiPropertyValue) can be declared before LmiPropertyValue (which contains
 an LmiVector(LmiPropertyValue)) is defined. */
typedef struct LmiPropertyValue_ LmiPropertyValue;

/* Declare the LmiVector without its header inlines since they require LmiPropertyValue to be fully defined. The header inlines
 are defined later, after LmiPropertyValue and its methods are declared. See Declare_LmiVectorInlines(LmiPropertyValue) later in the file. */
Declare_LmiVector_NoInlines(LmiPropertyValue)

typedef void *(*LmiPropertyValueWrapperConstructCopy)(void *val1, const void *val2);
typedef void *(*LmiPropertyValueWrapperAssign)(void *val1, const void *val2);
typedef LmiBool (*LmiPropertyValueWrapperEqual)(const void *val1, const void *val2);
typedef void (*LmiPropertyValueWrapperDestruct)(void *val1);

typedef struct LmiPropertyValueCallbacks_ {
    LmiPropertyValueWrapperConstructCopy copy;
    LmiPropertyValueWrapperDestruct destruct;
    LmiPropertyValueWrapperAssign assign;
    LmiPropertyValueWrapperEqual equal;
} LmiPropertyValueCallbacks;

/**
{type:
	{name: LmiPropertyValue}
	{parent: LmiProperties}
	{include: Lmi/Os/LmiProperties.h}
	{description: Describes a property value. A property value can be any of the various types described by LmiPropertyValueType.}
}
*/
struct LmiPropertyValue_ {
	LmiBool settable;
	LmiPropertyValueType type;
	union {
		LmiBool b;
		LmiInt i;
		LmiInt16 i16;
		LmiInt64 i64;
		LmiUint ui;
		LmiUint16 ui16;
		LmiUint64 ui64;
		LmiSizeT sz;
		LmiFloat32 f32;
		LmiFloat64 f64;
		LmiTime t;
		LmiResolution res;
		LmiMediaFormat fmt;
		LmiVoidPtr vp;
		LmiString s;
		struct LmiProperties_* p;
		LmiVector(LmiPropertyValue) vec;
		LmiSharedVoidPtr sp;
		LmiWeakVoidPtr wp;
	} u;
    LmiString ptrName; // Set when representing non-void shared or weak ptr
    LmiPropertyValueCallbacks *callbacks;
    LmiSizeT size;
};

/**
{function:
	{name: LmiPropertyValueConstructBool}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiBool property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructBool(LmiPropertyValue* x, LmiBool b, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: b}
		{description: The LmiBool to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructBool(LmiPropertyValue* x, LmiBool b, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructInt}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiInt property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructInt(LmiPropertyValue* x, LmiInt i, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: i}
		{description: The LmiInt to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructInt(LmiPropertyValue* x, LmiInt i, LmiBool settable);
/**
{function:
	{name: LmiPropertyValueConstructInt16}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiInt16 property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructInt16(LmiPropertyValue* x, LmiInt16 i16, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: i16}
		{description: The LmiInt16 to assign to the property value.}
	}
  	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructInt16(LmiPropertyValue* x, LmiInt16 i16, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructInt64}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiInt64 property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructInt64(LmiPropertyValue* x, LmiInt64 i64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: i64}
		{description: The LmiInt64 to assign to the property value.}
	}
  	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructInt64(LmiPropertyValue* x, LmiInt64 i64, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructUint}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiUint property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructUint(LmiPropertyValue* x, LmiUint ui, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: ui}
		{description: The LmiUint to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructUint(LmiPropertyValue* x, LmiUint ui, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructUint16}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiUint16 property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructUint16(LmiPropertyValue* x, LmiUint16 ui16, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: ui16}
		{description: The LmiUint16 to assign to the property value.}
	}
  	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructUint16(LmiPropertyValue* x, LmiUint16 ui16, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructUint64}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiUint64 property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructUint64(LmiPropertyValue* x, LmiUint64 ui64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: ui64}
		{description: The LmiUint64 to assign to the property value.}
	}
  	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructUint64(LmiPropertyValue* x, LmiUint64 ui64, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructSizeT}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiSizeT property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructSizeT(LmiPropertyValue* x, LmiSizeT sz, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: sz}
		{description: The LmiSizeT to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructSizeT(LmiPropertyValue* x, LmiSizeT sz, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructFloat32}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiFloat32 property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructFloat32(LmiPropertyValue* x, LmiFloat32 f32, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: f32}
		{description: The LmiFloat32 to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructFloat32(LmiPropertyValue* x, LmiFloat32 f32, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructFloat64}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiFloat64 property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructFloat64(LmiPropertyValue* x, LmiFloat64 f64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: f64}
		{description: The LmiFloat64 to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructFloat64(LmiPropertyValue* x, LmiFloat64 f64, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructTime}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiTime property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructTime(LmiPropertyValue* x, LmiTime t, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: t}
		{description: The LmiTime to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructTime(LmiPropertyValue* x, LmiTime t, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructResolution}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiResolution property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructResolution(LmiPropertyValue* x, LmiResolution res, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: res}
		{description: The LmiResolution to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructResolution(LmiPropertyValue* x, LmiResolution res, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructMediaFormat}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiMediaFormat property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructMediaFormat(LmiPropertyValue* x, LmiMediaFormat fmt, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: fmt}
		{description: The LmiMediaFormat to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructMediaFormat(LmiPropertyValue* x, LmiMediaFormat fmt, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructVoidPtr}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiVoidPtr property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructVoidPtr(LmiPropertyValue* x, LmiVoidPtr vp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: vp}
		{description: The LmiVoidPtr to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructVoidPtr(LmiPropertyValue* x, LmiVoidPtr vp, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructString}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiString property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructString(LmiPropertyValue* x, const LmiString* s, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: s}
		{description: The LmiString to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructString(LmiPropertyValue* x, const LmiString* s, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructStringCStr}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiString property value from a C string.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructStringCStr(LmiPropertyValue* x, const char* s, LmiBool settable, LmiAllocator* a)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: s}
		{description: The C string to assign to the LmiString property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{parameter:
		{name: a}
		{description: An allocator.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructStringCStr(LmiPropertyValue* x, const char* str, LmiBool settable, LmiAllocator* a);

/**
{function:
	{name: LmiPropertyValueConstructProperties}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiProperties property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructProperties(LmiPropertyValue* x, const LmiProperties* p, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: p}
		{description: The LmiProperties to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructProperties(LmiPropertyValue* x, const struct LmiProperties_* p, LmiBool settable);


/**
{function:
	{name: LmiPropertyValueConstructVector}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiVector(LmiPropertyValue) property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructVector(LmiPropertyValue* x, const LmiVector(LmiPropertyValue)* v, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: v}
		{description: The LmiVector(LmiPropertyValue) to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiPropertyValueAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructVector(LmiPropertyValue* x, const LmiVector(LmiPropertyValue)* p, LmiBool settable);



/**
{function:
	{name: LmiPropertyValueConstructSharedVoidPtr}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiSharedVoidPtr property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructSharedVoidPtr(LmiPropertyValue* x, const LmiSharedVoidPtr* sp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: sp}
		{description: The LmiSharedVoidPtr to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiSharedVoidPtrAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructSharedVoidPtr(LmiPropertyValue* x, const LmiSharedVoidPtr* sp, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructWeakVoidPtr}
	{parent: LmiPropertyValue}
	{description: Constructs an LmiSharedVoidPtr property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructWeakVoidPtr(LmiPropertyValue* x, const LmiWeakVoidPtr* wp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: wp}
		{description: The LmiWeakVoidPtr to assign to the property value.}
	}
	{parameter:
		{name: settable}
		{description: Defines whether or not LmiWeakVoidPtrAssign will succeed on the property value.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructWeakVoidPtr(LmiPropertyValue* x, const LmiWeakVoidPtr* wp, LmiBool settable);

/**
{function:
	{name: LmiPropertyValueConstructCopy}
	{parent: LmiPropertyValue}
	{description: Constructs a property value from another property value.}
	{prototype: LmiPropertyValue* LmiPropertyValueConstructCopy(LmiPropertyValue* x, const LmiPropertyValue* y)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: y}
		{description: A property value whose value to assign to {code:x}.}
	}
	{return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
LmiPropertyValue* LmiPropertyValueConstructCopy(LmiPropertyValue* x, const LmiPropertyValue* y);

/**
{function:
	{name: LmiPropertyValueDestruct}
	{parent: LmiPropertyValue}
	{description: Destructs a property value.}
	{prototype: void LmiPropertyValueDestruct(LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
}
*/
void LmiPropertyValueDestruct(LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueAssign}
	{parent: LmiPropertyValue}
	{description: Assigns one property value to another.}
	{prototype: LmiPropertyValue* LmiPropertyValueAssign(LmiPropertyValue* x, const LmiPropertyValue* y)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: y}
		{description: A property value whose value to assign to {code:x}.}
	}
	{return: Returns a pointer to the assigned-to property value or NULL if {code:x} is not settable or some other error occurred.}
}
*/
LmiPropertyValue* LmiPropertyValueAssign(LmiPropertyValue* x, const LmiPropertyValue* y);

/**
{function:
	{name: LmiPropertyValueEqual}
	{parent: LmiPropertyValue}
	{description: Determines whether two property values are equal.}
	{prototype: LmiBool LmiPropertyValueEqual(const LmiPropertyValue* x, const LmiPropertyValue* y)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: y}
		{description: A property value.}
	}
	{return: Return LMI_TRUE if the properties are of the same type and value, or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertyValueEqual(const LmiPropertyValue* x, const LmiPropertyValue* y);

/**
{function:
	{name: LmiPropertyValueIsSettable}
	{parent: LmiPropertyValue}
	{description: Determines whether or not a property value is settable.}
	{prototype: LmiBool LmiPropertyValueIsSettable(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns LMI_TRUE if the property value is settable or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertyValueIsSettable(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetType}
	{parent: LmiPropertyValue}
	{description: Gets the property value type of a property value.}
	{prototype: LmiPropertyValueType LmiPropertyValueGetType(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns the type of the property value.}
}
*/
LmiPropertyValueType LmiPropertyValueGetType(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetBool}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiBool of a property value.}
	{prototype: const LmiBool* LmiPropertyValueGetBool(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiBool of the property value on success, or NULL if the type of the property value is not an LmiBool.}
}
*/
const LmiBool* LmiPropertyValueGetBool(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetInt}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiInt of a property value.}
	{prototype: const LmiInt* LmiPropertyValueGetInt(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiInt of the property value on success, or NULL if the type of the property value is not an LmiInt.}
}
*/
const LmiInt* LmiPropertyValueGetInt(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetInt16}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiInt16 of a property value.}
	{prototype: const LmiInt16* LmiPropertyValueGetInt16(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiInt16 of the property value on success, or NULL if the type of the property value is not an LmiInt16.}
}
*/
const LmiInt16* LmiPropertyValueGetInt16(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetInt64}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiInt64 of a property value.}
	{prototype: const LmiInt64* LmiPropertyValueGetInt64(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiInt64 of the property value on success, or NULL if the type of the property value is not an LmiInt64.}
}
*/
const LmiInt64* LmiPropertyValueGetInt64(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetUint}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiUint of a property value.}
	{prototype: const LmiUint* LmiPropertyValueGetUint(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiUint of the property value on success, or NULL if the type of the property value is not an LmiUint.}
}
*/
const LmiUint* LmiPropertyValueGetUint(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetUint16}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiUint16 of a property value.}
	{prototype: const LmiUint16* LmiPropertyValueGetUint16(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiUint16 of the property value on success, or NULL if the type of the property value is not an LmiUint16.}
}
*/
const LmiUint16* LmiPropertyValueGetUint16(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetUint64}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiUint64 of a property value.}
	{prototype: const LmiUint64* LmiPropertyValueGetUint64(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiUint64 of the property value on success, or NULL if the type of the property value is not an LmiUint64.}
}
*/
const LmiUint64* LmiPropertyValueGetUint64(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetSizeT}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiSizeT of a property value.}
	{prototype: const LmiSizeT* LmiPropertyValueGetSizeT(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiSizeT of the property value on success, or NULL if the type of the property value is not an LmiSizeT.}
}
*/
const LmiSizeT* LmiPropertyValueGetSizeT(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetFloat32}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiFloat32 of a property value.}
	{prototype: const LmiFloat32* LmiPropertyValueGetFloat32(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiFloat32 of the property value on success, or NULL if the type of the property value is not an LmiFloat32.}
}
*/
const LmiFloat32* LmiPropertyValueGetFloat32(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetFloat64}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiFloat64 of a property value.}
	{prototype: const LmiFloat64* LmiPropertyValueGetFloat64(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiFloat64 of the property value on success, or NULL if the type of the property value is not an LmiFloat64.}
}
*/
const LmiFloat64* LmiPropertyValueGetFloat64(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetTime}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiTime of a property value.}
	{prototype: const LmiTime* LmiPropertyValueGetTime(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiTime of the property value on success, or NULL if the type of the property value is not an LmiTime.}
}
*/
const LmiTime* LmiPropertyValueGetTime(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetResolution}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiResolution of a property value.}
	{prototype: const LmiResolution* LmiPropertyValueGetResolution(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiResolution of the property value on success, or NULL if the type of the property value is not an LmiResolution.}
}
*/
const LmiResolution* LmiPropertyValueGetResolution(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetMediaFormat}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiMediaFormat of a property value.}
	{prototype: const LmiMediaFormat* LmiPropertyValueGetMediaFormat(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiMediaFormat of the property value on success, or NULL if the type of the property value is not an LmiMediaFormat.}
}
*/
const LmiMediaFormat* LmiPropertyValueGetMediaFormat(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetVoidPtr}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiVoidPtr of a property value.}
	{prototype: const LmiVoidPtr* LmiPropertyValueGetVoidPtr(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiVoidPtr of the property value on success, or NULL if the type of the property value is not an LmiVoidPtr.}
}
*/
const LmiVoidPtr* LmiPropertyValueGetVoidPtr(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetString}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiString of a property value.}
	{prototype: const LmiString* LmiPropertyValueGetString(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiString of the property value on success, or NULL if the type of the property value is not an LmiString.}
}
*/
const LmiString* LmiPropertyValueGetString(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetProperties}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiProperties of a property value.}
	{prototype: const LmiProperties* LmiPropertyValueGetProperties(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiProperties of the property value on success, or NULL if the type of the property value is not an LmiProperties.}
}
*/
const struct LmiProperties_* LmiPropertyValueGetProperties(const LmiPropertyValue* x);


/**
{function:
	{name: LmiPropertyValueGetVector}
	{parent: LmiPropertyValue}
	{description: Gets a LmiVector(LmiPropertyValue) pointer to an LmiVector(LmiPropertyValue) of a property value.}
	{prototype: const LmiVector(LmiPropertyValue)* LmiPropertyValueGetVector(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiVector(LmiPropertyValue) of the property value on success, or NULL if the type of the property value is not an LmiVector(LmiPropertyValue).}
}
*/
const LmiVector(LmiPropertyValue)* LmiPropertyValueGetVector(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueToBool}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiBool.}
	{prototype: LmiBool* LmiPropertyValueToBool(const LmiPropertyValue* x, LmiBool* b)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: b}
		{description: An LmiBool to hold a copy of the property's value.}
	}
	{return: Returns {code:b} on success, or NULL if the type of the property value is not an LmiBool or any other error occurs.}
}
*/

/**
{function:
	{name: LmiPropertyValueGetSharedVoidPtr}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiSharedVoidPtr of a property value.}
	{prototype: const LmiSharedVoidPtr* LmiPropertyValueGetSharedVoidPtr(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiSharedVoidPtr of the property value on success, or NULL if the type of the property value is not an LmiSharedVoidPtr.}
}
*/
const LmiSharedVoidPtr* LmiPropertyValueGetSharedVoidPtr(const LmiPropertyValue* x);

/**
{function:
	{name: LmiPropertyValueGetWeakVoidPtr}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiWeakVoidPtr of a property value.}
	{prototype: const LmiWeakVoidPtr* LmiPropertyValueGetWeakVoidPtr(const LmiPropertyValue* x)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiWeakVoidPtr of the property value on success, or NULL if the type of the property value is not an LmiWeakVoidPtr.}
}
*/
const LmiWeakVoidPtr* LmiPropertyValueGetWeakVoidPtr(const LmiPropertyValue* x);

LmiBool* LmiPropertyValueToBool(const LmiPropertyValue* x, LmiBool* b);

/**
{function:
	{name: LmiPropertyValueToInt}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiInt.}
	{prototype: LmiInt* LmiPropertyValueToInt(const LmiPropertyValue* x, LmiInt* i)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: i}
		{description: An LmiInt to hold a copy of the property's value.}
	}
	{return: Returns {code:i} on success, or NULL if the type of the property value is not an LmiInt or any other error occurs.}
}
*/
LmiInt* LmiPropertyValueToInt(const LmiPropertyValue* x, LmiInt* i);

/**
{function:
	{name: LmiPropertyValueToInt16}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiInt16.}
	{prototype: LmiInt16* LmiPropertyValueToInt16(const LmiPropertyValue* x, LmiInt16* i16)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: i16}
		{description: An LmiInt16 to hold a copy of the property's value.}
	}
	{return: Returns {code:i16} on success, or NULL if the type of the property value is not an LmiInt16 or any other error occurs.}
}
*/
LmiInt16* LmiPropertyValueToInt16(const LmiPropertyValue* x, LmiInt16* i16);

/**
{function:
	{name: LmiPropertyValueToInt64}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiInt64.}
	{prototype: LmiInt64* LmiPropertyValueToInt64(const LmiPropertyValue* x, LmiInt64* i64)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: i64}
		{description: An LmiInt64 to hold a copy of the property's value.}
	}
	{return: Returns {code:i64} on success, or NULL if the type of the property value is not an LmiInt64 or any other error occurs.}
}
*/
LmiInt64* LmiPropertyValueToInt64(const LmiPropertyValue* x, LmiInt64* i64);

/**
{function:
	{name: LmiPropertyValueToUint}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiUint.}
	{prototype: LmiUint* LmiPropertyValueToUint(const LmiPropertyValue* x, LmiUint* ui)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: ui}
		{description: An LmiUint to hold a copy of the property's value.}
	}
	{return: Returns {code:ui} on success, or NULL if the type of the property value is not an LmiUint or any other error occurs.}
}
*/
LmiUint* LmiPropertyValueToUint(const LmiPropertyValue* x, LmiUint* ui);

/**
{function:
	{name: LmiPropertyValueToUint16}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiUint16.}
	{prototype: LmiUint16* LmiPropertyValueToUint16(const LmiPropertyValue* x, LmiUint16* ui16)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: ui16}
		{description: An LmiUint16 to hold a copy of the property's value.}
	}
	{return: Returns {code:ui16} on success, or NULL if the type of the property value is not an LmiUint16 or any other error occurs.}
}
*/
LmiUint16* LmiPropertyValueToUint16(const LmiPropertyValue* x, LmiUint16* ui16);

/**
{function:
	{name: LmiPropertyValueToUint64}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiUint64.}
	{prototype: LmiUint64* LmiPropertyValueToUint64(const LmiPropertyValue* x, LmiUint64* ui64)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: ui64}
		{description: An LmiUint64 to hold a copy of the property's value.}
	}
	{return: Returns {code:ui16} on success, or NULL if the type of the property value is not an LmiUint64 or any other error occurs.}
}
*/
LmiUint64* LmiPropertyValueToUint64(const LmiPropertyValue* x, LmiUint64* ui64);
/**
{function:
	{name: LmiPropertyValueToSizeT}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiSizeT.}
	{prototype: LmiSizeT* LmiPropertyValueToSizeT(const LmiPropertyValue* x, LmiSizeT* sz)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: sz}
		{description: An LmiSizeT to hold a copy of the property's value.}
	}
	{return: Returns {code:sz} on success, or NULL if the type of the property value is not an LmiSizeT or any other error occurs.}
}
*/
LmiSizeT* LmiPropertyValueToSizeT(const LmiPropertyValue* x, LmiSizeT* sz);

/**
{function:
	{name: LmiPropertyValueToFloat32}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiFloat32.}
	{prototype: LmiFloat32* LmiPropertyValueToFloat32(const LmiPropertyValue* x, LmiFloat32* f32)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: f32}
		{description: An LmiFloat32 to hold a copy of the property's value.}
	}
	{return: Returns {code:f32} on success, or NULL if the type of the property value is not an LmiFloat32 or any other error occurs.}
}
*/
LmiFloat32* LmiPropertyValueToFloat32(const LmiPropertyValue* x, LmiFloat32* f32);

/**
{function:
	{name: LmiPropertyValueToFloat64}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiFloat64.}
	{prototype: LmiFloat64* LmiPropertyValueToFloat64(const LmiPropertyValue* x, LmiFloat64* f64)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: f64}
		{description: An LmiFloat64 to hold a copy of the property's value.}
	}
	{return: Returns {code:f64} on success, or NULL if the type of the property value is not an LmiFloat64 or any other error occurs.}
}
*/
LmiFloat64* LmiPropertyValueToFloat64(const LmiPropertyValue* x, LmiFloat64* f64);

/**
{function:
	{name: LmiPropertyValueToTime}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiTime.}
	{prototype: LmiTime* LmiPropertyValueToTime(const LmiPropertyValue* x, LmiTime* t)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: t}
		{description: An LmiTime to hold a copy of the property's value.}
	}
	{return: Returns {code:t} on success, or NULL if the type of the property value is not an LmiTime or any other error occurs.}
}
*/
LmiTime* LmiPropertyValueToTime(const LmiPropertyValue* x, LmiTime* t);

/**
{function:
	{name: LmiPropertyValueToResolution}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiResolution.}
	{prototype: LmiResolution* LmiPropertyValueToResolution(const LmiPropertyValue* x, LmiResolution* res)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: res}
		{description: An LmiResolution to hold a copy of the property's value.}
	}
	{return: Returns {code:t} on success, or NULL if the type of the property value is not an LmiResolution or any other error occurs.}
}
*/
LmiResolution* LmiPropertyValueToResolution(const LmiPropertyValue* x, LmiResolution* res);

/**
{function:
	{name: LmiPropertyValueToMediaFormat}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiMediaFormat.}
	{prototype: LmiMediaFormat* LmiPropertyValueToMediaFormat(const LmiPropertyValue* x, LmiMediaFormat* fmt)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: fmt}
		{description: An LmiMediaFormat to hold a copy of the property's value.}
	}
	{return: Returns {code:t} on success, or NULL if the type of the property value is not an LmiMediaFormat or any other error occurs.}
}
*/
LmiMediaFormat* LmiPropertyValueToMediaFormat(const LmiPropertyValue* x, LmiMediaFormat* fmt);

/**
{function:
	{name: LmiPropertyValueToVoidPtr}
	{parent: LmiPropertyValue}
	{description: Converts a property value to a void pointer.}
	{prototype: LmiVoidPtr* LmiPropertyValueToVoidPtr(const LmiPropertyValue* x, LmiVoidPtr* vp)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: vp}
		{description: An LmiVoidPtr to hold a copy of the property's value. }
	}
	{return: Returns {code:vp} on success, or NULL if the type of the property value is not an LmiVoidPtr or any other error occurs.}
}
*/
LmiVoidPtr* LmiPropertyValueToVoidPtr(const LmiPropertyValue* x, LmiVoidPtr* vp);

/**
{function:
	{name: LmiPropertyValueToString}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiString.}
	{prototype: LmiString* LmiPropertyValueToString(const LmiPropertyValue* x, LmiString* s)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: s}
		{description: An LmiString to hold a copy of the property's value.}
	}
	{return: Returns {code:s} on success, or NULL if the type of the property value is not an LmiString or any other error occurs.}
}
*/
LmiString* LmiPropertyValueToString(const LmiPropertyValue* x, LmiString* s);

/**
{function:
	{name: LmiPropertyValueToProperties}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiProperties.}
	{prototype: LmiProperties* LmiPropertyValueToProperties(const LmiPropertyValue* x, LmiProperties* p)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: p}
		{description: An LmiProperties to hold a copy of the property's value.}
	}
	{return: Returns {code:s} on success, or NULL if the type of the property value is not an LmiProperties or any other error occurs.}
}
*/
struct LmiProperties_* LmiPropertyValueToProperties(const LmiPropertyValue* x, struct LmiProperties_* p);

/**
{function:
	{name: LmiPropertyValueToVector}
	{parent: LmiPropertyValue}
	{description: Converts a property value to an LmiVector(LmiPropertyValue).}
	{prototype: LmiVector(LmiPropertyValue)* LmiPropertyValueToVector(const LmiPropertyValue* x, LmiVector(LmiPropertyValue)* v)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: v}
		{description: An LmiVector(LmiPropertyValue) to hold a copy of the property's value.}
	}
	{return: Returns {code:v} on success, or NULL if the type of the property value is not an LmiVector(LmiPropertyValue) or any other error occurs.}
}
*/
LmiVector(LmiPropertyValue)* LmiPropertyValueToVector(const LmiPropertyValue* x, LmiVector(LmiPropertyValue)* p);

/**
{function:
	{name: LmiPropertyValueToSharedVoidPtr}
	{parent: LmiPropertyValue}
	{description: Converts a property value to a void pointer.}
	{prototype: LmiSharedVoidPtr* LmiPropertyValueToSharedVoidPtr(const LmiPropertyValue* x, LmiSharedVoidPtr* sp)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: sp}
		{description: An LmiSharedVoidPtr to hold a copy of the property's value. }
	}
	{return: Returns {code:sp} on success, or NULL if the type of the property value is not an LmiSharedVoidPtr or any other error occurs.}
}
*/
LmiSharedVoidPtr* LmiPropertyValueToSharedVoidPtr(const LmiPropertyValue* x, LmiSharedVoidPtr* sp);

/**
{function:
	{name: LmiPropertyValueToWeakVoidPtr}
	{parent: LmiPropertyValue}
	{description: Converts a property value to a void pointer.}
	{prototype: LmiWeakVoidPtr* LmiPropertyValueToWeakVoidPtr(const LmiPropertyValue* x, LmiWeakVoidPtr* wp)}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: wp}
		{description: An LmiWeakVoidPtr to hold a copy of the property's value. }
	}
	{return: Returns {code:wp} on success, or NULL if the type of the property value is not an LmiWeakVoidPtr or any other error occurs.}
}
*/
LmiWeakVoidPtr* LmiPropertyValueToWeakVoidPtr(const LmiPropertyValue* x, LmiWeakVoidPtr* wp);

/**
{function:
	{name: LmiPropertyValuePrint}
	{parent: LmiPropertyValue}
	{description: Dump a property value to the specified file stream.}
	{prototype: void LmiPropertyValuePrint(const LmiPropertyValue* x, LmiFile* file, const char* tabs)}
	{parameter:
		{name: x}
		{description: A property vale.}
	}
	{parameter:
		{name: file}
		{description: The file stream to dump the property value\'s contents to.}
	}
 	{parameter:
		{name: tabs}
		{description: The C string containing tabs specifying how far to indent the output.}
	}
}
*/
void LmiPropertyValuePrint(const LmiPropertyValue* x, LmiFile* file, const char* tabs);

/* Generate the header inlines for the LmiVector(LmiPropertyValue) after LmiPropertyValue and its methods are declared */
Declare_LmiVector_Inlines(LmiPropertyValue)

Declare_LmiVectorEqual(LmiPropertyValue)

struct LmiPropertyListener_;

/**
{callback:
	{name: LmiPropertyListenerCallback}
	{parent: LmiPropertyListener}
	{description: Describes a callback called by LmiPropertyListener to notify an application of a property change. }
	{prototype: void (*LmiPropertyListenerCallback)(const LmiPropertyListener* x, LmiProperties* properties, const LmiString* name, const LmiPropertyValue* oldValue, const LmiPropertyValue* newValue)}
	{parameter:
		{name: x}
		{description: The property listener. }
	}
	{parameter:
		{name: properties}
		{description: The property set containing the property that triggered the listener. }
	}
	{parameter:
		{name: name}
		{description: The name of the property that triggered the listener. }
	}
	{parameter:
		{name: oldValue}
		{description: The old property value. On LMI_PROPERTYLISTENER_TRIGGER_Before calls, this is the current value of the property, unless the property was added, in which case this parameter is NULL. On LMI_PROPERTYLISTENER_TRIGGER_After calls, this is the current value of the property, or NULL if the property was added or removed.}
	}
	{parameter:
		{name: newValue}
		{description: The new property value. On LMI_PROPERTYLISTENER_TRIGGER_Before calls, this is the value the property will have subsequently, or NULL, if the property is being removed. On LMI_PROPERTYLISTENER_TRIGGER_After calls, this is the current value of the property, or NULL if the property was removed.}
	}
}
*/
typedef void (*LmiPropertyListenerCallback)(const struct LmiPropertyListener_ * x, struct LmiProperties_* properties, const LmiString* name, const LmiPropertyValue* oldValue, const LmiPropertyValue* newValue);

/**
{type:
	{name: LmiPropertyListenerTrigger}
	{parent: LmiPropertyListener}
	{include: Lmi/Os/LmiProperties.h}
	{description: Describes the trigger for a property listener to fire.}
	{value:
		{name: LMI_PROPERTYLISTENER_TRIGGER_Before}
		{description: Specifies that the listener should be called prior to the property value changing.}
	}
	{value:
		{name: LMI_PROPERTYLISTENER_TRIGGER_After}
		{description: Specifies that the listener should be called after the property value is changed.}
	}
}
*/
typedef enum {
	LMI_PROPERTYLISTENER_TRIGGER_Before,
	LMI_PROPERTYLISTENER_TRIGGER_After
} LmiPropertyListenerTrigger;

/**
{type:
	{name: LmiPropertyListener}
	{parent: LmiProperties}
	{include: Lmi/Os/LmiProperties.h}
	{description: Describes a property listener.  A property listener allows an application to be notified whenever a property is added, removed, or its value is changed.  A listener can be configured to trigger before the add, remove, or value change takes place or after (see LmiPropertyListenerTrigger for details).  A listener can be scoped to trigger on an individual property of the property set by specifying the property's name or on all properties of the property set by setting the name parameter to NULL.}
}
*/
typedef struct LmiPropertyListener_ {
	LmiBool all;
	LmiString name;
	LmiPropertyListenerTrigger trigger;
	LmiPropertyListenerCallback cb;
	LmiVoidPtr data;
} LmiPropertyListener;

/**
{function:
	{name: LmiPropertyListenerConstruct}
	{parent: LmiPropertyListener}
	{description: Constructs a property listener.}
	{prototype: LmiPropertyListener* LmiPropertyListenerConstruct(LmiPropertyListener* x, LmiPropertyListenerTrigger trigger, const LmiString* name, LmiPropertyListenerCallback cb, LmiVoidPtr data)}
	{parameter:
		{name: x}
		{description: A property listener.}
	}
	{parameter:
		{name: trigger}
		{description: The trigger to listen for.}
	}
	{parameter:
		{name: name}
		{description: The name of the property to listen to or NULL to listener to all properties.}
	}
	{parameter:
		{name: cb}
		{description: The callback to call when the property is changed.}
	}
	{parameter:
		{name: data}
		{description: A user data pointer.}
	}
	{return: Returns a pointer to the constructed object or NULL on error.}
}
*/
LmiPropertyListener* LmiPropertyListenerConstruct(LmiPropertyListener* x, LmiPropertyListenerTrigger trigger, const LmiString* name, LmiPropertyListenerCallback cb, LmiVoidPtr data);

/**
{function:
	{name: LmiPropertyListenerConstructCStr}
	{parent: LmiPropertyListener}
	{description: Constructs a property listener.}
	{prototype: LmiPropertyListener* LmiPropertyListenerConstructCStr(LmiPropertyListener* x, LmiPropertyListenerTrigger trigger, const char* name, LmiPropertyListenerCallback cb, LmiVoidPtr data, LmiAllocator* a)}
	{parameter:
		{name: x}
		{description: A property listener.}
	}
	{parameter:
		{name: trigger}
		{description: The trigger to listen for.}
	}
	{parameter:
		{name: name}
		{description: The name of the property to listen to or NULL to listen to all properties.}
	}
	{parameter:
		{name: cb}
		{description: The callback to call when the property is changed.}
	}
	{parameter:
		{name: data}
		{description: A user data pointer.}
	}
	{parameter:
		{name: a}
		{description: An allocator.}
	}
	{return: Returns a pointer to the constructed object or NULL on error.}
	{note: Path notation cannot be used in the property listeners.}
}
*/
LmiPropertyListener* LmiPropertyListenerConstructCStr(LmiPropertyListener* x, LmiPropertyListenerTrigger trigger, const char* name,
  LmiPropertyListenerCallback cb, LmiVoidPtr data, LmiAllocator* a);

/**
{function:
	{name: LmiPropertyListenerConstructCopy}
	{parent: LmiPropertyListener}
	{description: Constructs a property listener from another property listener.}
	{prototype: LmiPropertyListener* LmiPropertyListenerConstructCopy(LmiPropertyListener* x, const LmiPropertyListener* y)}
	{parameter:
		{name: x}
		{description: A property listener.}
	}
	{parameter:
		{name: y}
		{description: A property listener.}
	}
	{return: Returns a pointer to the constructed object or NULL on error.}
	{note: Path notation cannot be used in the property listeners.}
}
*/
LmiPropertyListener* LmiPropertyListenerConstructCopy(LmiPropertyListener* x, const LmiPropertyListener* y);

/**
{function:
	{name: LmiPropertyListenerDestruct}
	{parent: LmiPropertyListener}
	{description: Destructs a property listener.}
	{prototype: void LmiPropertyListenerDestruct(LmiPropertyListener* x)}
	{parameter:
		{name: x}
		{description: A property listener.}
	}
}
*/
void LmiPropertyListenerDestruct(LmiPropertyListener* x);

/**
{function:
	{name: LmiPropertyListenerAssign}
	{parent: LmiPropertyListener}
	{description: Assigns a property listener from another property listener.}
	{prototype: LmiPropertyListener* LmiPropertyListenerAssign(LmiPropertyListener* x, const LmiPropertyListener* y)}
	{parameter:
		{name: x}
		{description: A property listener.}
	}
	{parameter:
		{name: y}
		{description: A property listener.}
	}
	{return: Returns a pointer to the assigned-to object or NULL on error.}
}
*/
LmiPropertyListener* LmiPropertyListenerAssign(LmiPropertyListener* x, const LmiPropertyListener* y);

/**
{function:
	{name: LmiPropertyListenerEqual}
	{parent: LmiPropertyListener}
	{description: Determines whether two property listeners are equal.}
	{prototype: LmiBool LmiPropertyListenerEqual(const LmiPropertyListener* x, const LmiPropertyListener* y)}
	{parameter:
		{name: x}
		{description: A property listener.}
	}
	{parameter:
		{name: y}
		{description: A property listener.}
	}
	{return: Returns LMI_TRUE if the listeners are equal or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertyListenerEqual(const LmiPropertyListener* x, const LmiPropertyListener* y);

/**
{function:
	{name: LmiPropertyListenerGetTrigger}
	{parent: LmiPropertyListener}
	{description: Gets the trigger of a property listener.}
	{prototype: LmiPropertyListenerTrigger LmiPropertyListenerGetTrigger(const LmiPropertyListener* x)}
	{parameter:
		{name: x}
		{description: A property listener.}
	}
	{return: Returns the trigger for the listener.}
}
*/
LmiPropertyListenerTrigger LmiPropertyListenerGetTrigger(const LmiPropertyListener* x);

/**
{function:
	{name: LmiPropertyListenerGetName}
	{parent: LmiPropertyListener}
	{description: Gets the property name the listener is set on.}
	{prototype: const LmiString* LmiPropertyListenerGetName(const LmiPropertyListener* x)}
	{parameter:
		{name: x}
		{description: A property listener.}
	}
	{return: Returns a pointer to the name or NULL if the listener is listening to all properties.}
}
*/
const LmiString* LmiPropertyListenerGetName(const LmiPropertyListener* x);

/**
{function:
	{name: LmiPropertyListenerGetData}
	{parent: LmiPropertyListener}
	{description: Gets the user data pointer of the property listener.}
	{prototype: LmiVoidPtr LmiPropertyListenerGetData(const LmiPropertyListener* x)}
	{parameter:
		{name: x}
		{description: A property listener.}
	}
	{return: Returns the user data pointer passed to {code:LmiPropertyListenerConstruct} or {code:LmiPropertyListenerConstructCStr}.}
}
*/
LmiVoidPtr LmiPropertyListenerGetData(const LmiPropertyListener* x);

Declare_LmiMap(LmiString, LmiPropertyValue)
Declare_LmiVector(LmiPropertyListener)

/**
{type:
	{name: LmiProperties}
	{parent: Os}
	{include: Lmi/Os/LmiProperties.h}
	{description: Describes a set of properties and listeners for observing changes to those properties.}
}
*/
typedef struct LmiProperties_ {
	LmiMutex mutex;
	LmiMap(LmiString, LmiPropertyValue) map;
	LmiVector(LmiPropertyListener) listeners;
} LmiProperties;


/**
{function:
	{name: LmiPropertiesConstructDefault}
	{parent: LmiProperties}
	{description: Constructs an empty property set.}
	{prototype: LmiProperties* LmiPropertiesConstructDefault(LmiProperties* x, LmiAllocator* a)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: a}
		{description: An allocator. }
	}
	{return: Returns a pointer to the constructed object or NULL on error.}
}
*/
LmiProperties* LmiPropertiesConstructDefault(LmiProperties* x, LmiAllocator* a);
/**
{function:
	{name: LmiPropertiesConstructCopy}
	{parent: LmiProperties}
	{description: Constructs a property set from another property set.}
	{prototype: LmiProperties* LmiPropertiesConstructCopy(LmiProperties* x, const LmiProperties* y)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: y}
		{description: A property set.}
	}
	{return: Returns a pointer to the constructed object or NULL on error.}
}
*/
LmiProperties* LmiPropertiesConstructCopy(LmiProperties* x, const LmiProperties* y);
/**
{function:
	{name: LmiPropertiesDestruct}
	{parent: LmiProperties}
	{description: Destructs a property set.}
	{prototype: void LmiPropertiesDestruct(LmiProperties* x)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
}
*/
void LmiPropertiesDestruct(LmiProperties* x);

/**
{function:
	{name: LmiPropertiesAssign}
	{parent: LmiProperties}
	{description: Assigns one property set to another property set.}
	{prototype: LmiProperties* LmiPropertiesAssign(LmiProperties* x, const LmiProperties* y)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: y}
		{description: A property set.}
	}
	{return: Returns a pointer to the assigned-to object or NULL on error.}
}
*/
LmiProperties* LmiPropertiesAssign(LmiProperties* x, const LmiProperties* y);


/**
{function:
	{name: LmiPropertiesEqual}
	{parent: LmiProperties}
	{description: Compares one property set to another property set for equality.}
	{prototype: LmiBool LmiPropertiesEqual(const LmiProperties* x, const LmiProperties* y)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: y}
		{description: A property set.}
	}
	{return: Returns LMI_TRUE if the property sets are equal or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesEqual(const LmiProperties* x, const LmiProperties* y);


/**
{function:
	{name: LmiPropertiesLess}
	{parent: LmiProperties}
	{description: Compares one property set to another property set for less-than equality.}
	{prototype: LmiBool LmiPropertiesLess(const LmiProperties* x, const LmiProperties* y)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: y}
		{description: A property set.}
	}
	{return: Returns LMI_TRUE if the {code:x} is less than {code:y} or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesLess(const LmiProperties* x, const LmiProperties* y);

/**
{function:
	{name: LmiPropertiesGetAllocator}
	{parent: LmiProperties}
	{description: Gets the allocator of the property set.}
	{prototype: LmiAllocator* LmiPropertiesGetAllocator(const LmiProperties* x)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{return: Returns a pointer to the property set's allocator.}
}
*/
LmiAllocator* LmiPropertiesGetAllocator(const LmiProperties* x);

/**
{function:
	{name: LmiPropertiesAddListener}
	{parent: LmiProperties}
	{description: Adds a listener to the property set. Note: it's possible to register a property listener for a specific property even before the property has been added.}
	{prototype: LmiBool LmiPropertiesAddListener(LmiProperties* x, LmiPropertyListener* listener)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: listener}
		{description: The listener to add.}
	}
	{return: Returns LMI_TRUE if the listener was added successfully, or LMI_FALSE if an error occurred.}

	{example:
		#include <stdio.h>
		#include <Lmi/Os/LmiProperties.h>
		void PrintPropertyValue(FILE* fp, const LmiPropertyValue* v) \{
			LmiPropertyValueType type = LmiPropertyValueGetType(v);
			if (type == LMI_PROPERTYVALUE_TYPE_Bool) \{
				fprintf(fp, "%s", (*LmiPropertyValueGetBool(v)) ? "true" : "false");
			\} else if (type == LMI_PROPERTYVALUE_TYPE_Int) \{
				fprintf(fp, "%d", *LmiPropertyValueGetInt(v));
			\} else if (type == LMI_PROPERTYVALUE_TYPE_Int16) \{
				fprintf(fp, "%d", *LmiPropertyValueGetInt16(v));
			\} else if (type == LMI_PROPERTYVALUE_TYPE_Int64) \{
				fprintf(fp, "%lld", *LmiPropertyValueGetInt64(v));
			\} else if (type == LMI_PROPERTYVALUE_TYPE_Uint) \{
				fprintf(fp, "%u", *LmiPropertyValueGetUint(v));
			\} else if (type == LMI_PROPERTYVALUE_TYPE_Uint16) \{
				fprintf(fp, "%hu", *LmiPropertyValueGetUint16(v));
 			\} else if (type == LMI_PROPERTYVALUE_TYPE_Uint64) \{
				fprintf(fp, "%llu", *LmiPropertyValueGetUint64(v));
			\} else if (type == LMI_PROPERTYVALUE_TYPE_SizeT) \{
				fprintf(fp, "%zu", *LmiPropertyValueGetSizeT(v));
			\} else if (type == LMI_PROPERTYVALUE_TYPE_Float32) \{
				fprintf(fp, "%.2f", *LmiPropertyValueGetFloat32(v));
			\} else if (type == LMI_PROPERTYVALUE_TYPE_Float64) \{
				fprintf(fp, "%.5f", *LmiPropertyValueGetFloat64(v));
			\} else if (type == LMI_PROPERTYVALUE_TYPE_Time) \{
				fprintf(fp, "%lld", (long long)*LmiPropertyValueGetTime(v));
 			\} else if (type == LMI_PROPERTYVALUE_TYPE_Resolution) \{
				fprintf(fp, "%zu x %zu", LmiResolutionGetWidth(LmiPropertyValueGetResolution(v)), LmiResolutionGetHeight(LmiPropertyValueGetResolution(v)));
			\} else if (type == LMI_PROPERTYVALUE_TYPE_MediaFormat) \{
 				char mf[5];
				fprintf(fp, "'%s'", LmiMediaFormatToCStr(LmiPropertyValueGetMediaFormat(v), mf));
			\} else if (type == LMI_PROPERTYVALUE_TYPE_VoidPtr) \{
				fprintf(fp, "%p", *LmiPropertyValueGetVoidPtr(v));
			\} else if (type == LMI_PROPERTYVALUE_TYPE_String) \{
				fprintf(fp, "\\"%s\\"", LmiStringCStr(LmiPropertyValueGetString(v)));
			\}
		\}

		void BeforeCb(const LmiPropertyListener* x, LmiProperties* properties, const LmiString* name, const LmiPropertyValue* oldValue, const LmiPropertyValue* newValue) \{
			FILE* fp = (FILE*)LmiPropertyListenerGetData(x);

			fprintf(fp, "Before Event: ");

			\/\* Value Changed \*\/
			if ((oldValue != NULL) && (newValue != NULL)) \{
				fprintf(fp, "Property %s will change value from ", LmiStringCStr(name));
				PrintPropertyValue(fp, oldValue);
				fprintf(fp, " to ");
				PrintPropertyValue(fp, newValue);
				fprintf(fp, ".\n");

			\/\* Property Removed \*\/
			\} else if (oldValue != NULL) \{
				fprintf(fp, "Property %s will be removed (value currently is ", LmiStringCStr(name));
				PrintPropertyValue(fp, oldValue);
				fprintf(fp, ").\n");
			\/\* Property Added \*\/
			\} else if (newValue != NULL) \{
				fprintf(fp, "Property %s will be added (value will be ", LmiStringCStr(name));
				PrintPropertyValue(fp, newValue);
				fprintf(fp, ").\n");
			\}
		\}

		void AfterCb(const LmiPropertyListener* x, LmiProperties* properties, const LmiString* name, const LmiPropertyValue* oldValue, const LmiPropertyValue* newValue) \{
			FILE* fp = (FILE*)LmiPropertyListenerGetData(x);

			fprintf(fp, "After Event: ");

			\/\* Value changed \*\/
			if ((oldValue != NULL) && (newValue != NULL)) \{
				fprintf(fp, "Property %s has value changed to ", LmiStringCStr(name));
				PrintPropertyValue(fp, newValue);
				fprintf(fp, ".\n");
			\/\* Property Removed \*\/
			\} else if ((oldValue == NULL) && (newValue == NULL)) \{
				fprintf(fp, "Property %s was removed.\n", LmiStringCStr(name));
			\/\* Property Added \*\/
			\} else if (newValue != NULL) \{
				fprintf(fp, "Property %s was added (value is ", LmiStringCStr(name));
				PrintPropertyValue(fp, newValue);
				fprintf(fp, ").\n");
			\}
		\}

		int main() \{
			LmiAllocator* alloc;
			LmiProperties properties;
			LmiString name;
			LmiPropertyValue value, value2;
			LmiPropertyListener before, after;
			char nameCStr[] = "path1.path2.path3";

			alloc = LmiAllocatorGetDefault();
			LmiStringConstructCStr(&name, "prop1.prop2", alloc);

			\/\* Construct a property set \*\/
			LmiPropertiesConstructDefault(&properties, alloc);

			\/\* Register a "before" listener property \*\/
			LmiPropertyListenerConstruct(&before, LMI_PROPERTYLISTENER_TRIGGER_Before, &name, BeforeCb, stdout);
			LmiPropertiesAddListener(&properties, &before);

			\/\* Register a "after" listener property \*\/
			LmiPropertyListenerConstruct(&after, LMI_PROPERTYLISTENER_TRIGGER_After, &name, AfterCb, stdout);
			LmiPropertiesAddListener(&properties, &after);

			\/\* Add a property \*\/
			LmiPropertyValueConstructInt(&value, 123, LMI_TRUE);
			LmiPropertiesAddProperty(&properties, &name, &value);

			\/\* Set a property \*\/
			LmiPropertyValueConstructInt(&value2, 456, LMI_TRUE);
			LmiPropertiesSetProperty(&properties, &name, &value2);

			\/\* Another way to set property \*\/
			LmiPropertiesSetIntValueCStr(&properties, nameCStr, 42, LMI_TRUE, alloc);

			\/\* Remove a property \*\/
			LmiPropertiesRemoveProperty(&properties, &name);

			\/\* Clean-up \*\/
			LmiPropertyValueDestruct(&value2);
			LmiPropertyValueDestruct(&value);
			LmiStringDestruct(&name);
			LmiPropertyListenerDestruct(&before);
			LmiPropertyListenerDestruct(&after);
			LmiPropertiesDestruct(&properties);

			return 0;
		\}
	}
}
*/
LmiBool LmiPropertiesAddListener(LmiProperties* x, LmiPropertyListener* listener);

/**
{function:
	{name: LmiPropertiesRemoveListener}
	{parent: LmiProperties}
	{description: Removes a listener from the property set. }
	{prototype: LmiBool LmiPropertiesRemoveListener(LmiProperties* x, LmiPropertyListener* listener)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: listener}
		{description: The listener to remove.}
	}
	{return: Returns LMI_TRUE if the listener was successfully removed or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesRemoveListener(LmiProperties* x, LmiPropertyListener* listener);

/**
{function:
	{name: LmiPropertiesAddProperty}
	{parent: LmiProperties}
	{description: Adds a property to the property set.  Adding a property triggers matching property listeners.}
	{prototype: LmiBool LmiPropertiesAddProperty(LmiProperties* x, const LmiString* name, const LmiPropertyValue* v)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: name}
		{description: The name of the property.}
	}
	{parameter:
		{name: v}
		{description: The property value to set the property to.}
	}
	{return: Returns LMI_TRUE if the property does not yet exist and the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesAddProperty(LmiProperties* x, const LmiString* name, const LmiPropertyValue* v);

/**
{function:
	{name: LmiPropertiesAddPropertyCStr}
	{parent: LmiProperties}
	{description: Adds a property to the property set.  Adding a property triggers matching property listeners.}
	{prototype: LmiBool LmiPropertiesAddPropertyCStr(LmiProperties* x, const char* name, const LmiPropertyValue* v)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: name}
		{description: The name of the property. }
	}
	{parameter:
		{name: v}
		{description: The property value to set the property to.}
	}
	{return: Returns LMI_TRUE if the property does not yet exist and the property value was successful set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesAddPropertyCStr(LmiProperties* x, const char* name, const LmiPropertyValue* v);

/**
{function:
	{name: LmiPropertiesAddPropertyAtPath}
	{parent: LmiProperties}
	{description: Adds a property to the property set at the specified path.  If the last component of the path specifies a vector and index
 	  and the index is in the range [0 .. the size of vector], the property value is inserted into the vector at the index and all elements
 	  of the vector above the index are shift up.  Note that an index equal to the size of the vector adds the property value to the end of
 	  the vector.  Adding a property triggers matching property listeners.}
	{prototype: LmiBool LmiPropertiesAddPropertyAtPath(LmiProperties* x, const LmiString* path, const LmiPropertyValue* v)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: path}
		{description: The path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: v}
		{description: The property value to set the property to.}
	}
	{return: Returns LMI_TRUE if the property does not yet exist and the property value was successfully set or LMI_FALSE otherwise.}
 	{note: It is not possible to add a property to a vector with this function.}
}
*/
LmiBool LmiPropertiesAddPropertyAtPath(LmiProperties* x, const LmiString* path, const LmiPropertyValue* v);

/**
{function:
	{name: LmiPropertiesAddPropertyAtPathCStr}
	{parent: LmiProperties}
	{description: Adds a property to the property set at the specified path.  If the last component of the path specifies a vector and index
 	  and the index is in the range [0 .. the size of vector], the property value is inserted into the vector at the index and all elements
 	  of the vector above the index are shift up.  Note that an index equal to the size of the vector adds the property value to the end of
 	  the vector.  Adding a property triggers matching property listeners.}
	{prototype: LmiBool LmiPropertiesAddPropertyAtPathCStr(LmiProperties* x, const char* path, const LmiPropertyValue* v)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: path}
		{description: The path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: v}
		{description: The property value to set the property to.}
	}
	{return: Returns LMI_TRUE if the property does not yet exist and the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesAddPropertyAtPathCStr(LmiProperties* x, const char* path, const LmiPropertyValue* v);

/**
{function:
	{name: LmiPropertiesRemoveProperty}
	{parent: LmiProperties}
	{description: Removes a property from the property set. Removing a property triggers matching property listeners registered for the property. }
	{prototype: LmiBool LmiPropertiesRemoveProperty(LmiProperties* x, const LmiString* name)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: name}
		{description: The name of the property.}
	}
	{return: Returns LMI_TRUE if the property exists and the property was removed successful or LMI_FALSE otherwise.}
	{note: If path has multiple components, only the last component in the path is removed.  For example, calling this function with a path of
 	  "parent.child" will only remove "child"; "parent" will stay even if it is empty.}
}
*/
LmiBool LmiPropertiesRemoveProperty(LmiProperties* x, const LmiString* name);

/**
{function:
	{name: LmiPropertiesRemovePropertyCStr}
	{parent: LmiProperties}
	{description: Removes a property to the property set. Removing a property triggers matching property listeners.}
	{prototype: LmiBool LmiPropertiesRemovePropertyCStr(LmiProperties* x, const char* name)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: name}
		{description: The name of the property.}
	}
	{return: Returns LMI_TRUE if the property exists and the property was removed successful or LMI_FALSE otherwise.}
	{note: If path has multiple components, only the last component in the path is removed.  For example, calling this function with a path of
 	  "parent.child" will only remove "child"; "parent" will stay even if it is empty.}
}
*/
LmiBool LmiPropertiesRemovePropertyCStr(LmiProperties* x, const char* name);

/**
{function:
	{name: LmiPropertiesRemovePropertyAtPath}
	{parent: LmiProperties}
	{description: Removes a property of the property set at the specified path. Removing a property triggers matching property listeners.}
	{prototype: LmiBool LmiPropertiesRemovePropertyAtPath(LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns LMI_TRUE if the property exists and the property was removed successful or LMI_FALSE otherwise.}
	{note: If path has multiple components, only the last component in the path is removed.  For example, calling this function with a path of
 	  "parent.child" will only remove "child"; "parent" will stay even if it is empty.}
 	{note: If a property value in a vector is removed the vector shifts its elements down to fill the gap.  So for example,
 	  calling this function with a path of "vec[0] on a vector with 2 elements (vec[0] and vec[1]) will result in a vector of one element
 	  (vec[0]) with the contents of what was previously vec[1].  This can be confusing since a property value at the specified path (vec[0])
 	  exists (but with different content) after the this function completes sucessfully.}
}
*/
LmiBool LmiPropertiesRemovePropertyAtPath(LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesRemovePropertyAtPathCStr}
	{parent: LmiProperties}
	{description: Removes a property of the property set at the specified path. Removing a property triggers matching property listeners.}
	{prototype: LmiBool LmiPropertiesRemovePropertyAtPathCStr(LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns LMI_TRUE if the property exists and the property was removed successful or LMI_FALSE otherwise.}
	{note: If path has multiple components, only the last component in the path is removed.  For example, calling this function with a path of
 	  "parent.child" will only remove "child"; "parent" will stay even if it is empty.}
 	{note: If a property value in a vector is removed the vector shifts its elements down to fill the gap.  So for example,
 	  calling this function with a path of "vec[0] on a vector with 2 elements (vec[0] and vec[1]) will result in a vector of one element
 	  (vec[0]) with the contents of what was previously vec[1].  This can be confusing since a property value at the specified path (vec[0])
 	  exists (but with different content) after the this function completes sucessfully.}
}
*/
LmiBool LmiPropertiesRemovePropertyAtPathCStr(LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetPropertyConst}
	{parent: LmiProperties}
	{description: Gets the property value of a property in the property set.}
	{prototype: const LmiPropertyValue* LmiPropertiesGetPropertyConst(const LmiProperties* x, const LmiString* name)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: name}
		{description: The name of the property.}
	}
	{return: Returns a pointer to the property's value or NULL if the property was not found.}
}
*/
const LmiPropertyValue* LmiPropertiesGetPropertyConst(const LmiProperties* x, const LmiString* name);

/**
{function:
	{name: LmiPropertiesGetPropertyCStrConst}
	{parent: LmiProperties}
	{description: Gets the property value of a property in the property set.}
	{prototype: const LmiPropertyValue* LmiPropertiesGetPropertyCStrConst(const LmiProperties* x, const char* name)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: name}
		{description: The name of the property.}
	}
	{return: Returns a pointer to the property's value or NULL if the property was not found.}
}
*/
const LmiPropertyValue* LmiPropertiesGetPropertyCStrConst(const LmiProperties* x, const char* name);

/**
{function:
	{name: LmiPropertiesGetProperty}
	{parent: LmiProperties}
	{description: Gets the property value of a property in the property set.}
	{prototype: LmiPropertyValue* LmiPropertiesGetProperty(LmiProperties* x, const LmiString* name)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: name}
		{description: The name of the property.}
 	}
	{return: Returns a pointer to the property's value or NULL if the property was not found.}
}
*/
LmiPropertyValue* LmiPropertiesGetProperty(LmiProperties* x, const LmiString* name);

/**
{function:
	{name: LmiPropertiesGetPropertyCStr}
	{parent: LmiProperties}
	{description: Gets the property value of a property in the property set.}
	{prototype: LmiPropertyValue* LmiPropertiesGetPropertyCStr(LmiProperties* x, const char* name)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: name}
		{description: The name of the property.}
	}
	{return: Returns a pointer to the property's value or NULL if the property was not found.}
}
*/
LmiPropertyValue* LmiPropertiesGetPropertyCStr(LmiProperties* x, const char* name);

/**
{function:
	{name: LmiPropertiesGetPropertyAtPathConst}
	{parent: LmiProperties}
	{description: Gets the property value of a property in the property set at a specified path.}
	{prototype: const LmiPropertyValue* LmiPropertiesGetPropertyAtPathConst(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: path}
		{description: The path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to the property's value or NULL if the property was not found.}
}
*/
const LmiPropertyValue* LmiPropertiesGetPropertyAtPathConst(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetPropertyAtPathCStrConst}
	{parent: LmiProperties}
	{description: Gets the property value of a property in the property set at a specified path.}
	{prototype: const LmiPropertyValue* LmiPropertiesGetPropertyAtPathCStrConst(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: path}
		{description: The path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to the property's value or NULL if the property was not found.}
}
*/
const LmiPropertyValue* LmiPropertiesGetPropertyAtPathCStrConst(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetPropertyAtPath}
	{parent: LmiProperties}
	{description: Gets the property value of a property in the property set at a specified path.}
	{prototype: LmiPropertyValue* LmiPropertiesGetPropertyAtPath(LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: path}
		{description: The path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to the property's value or NULL if the property was not found.}
}
*/
LmiPropertyValue* LmiPropertiesGetPropertyAtPath(LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetPropertyAtPathCStr}
	{parent: LmiProperties}
	{description: Gets the property value of a property in the property set at a specified path.}
	{prototype: LmiPropertyValue* LmiPropertiesGetPropertyAtPathCStr(LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: path}
		{description: The path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to the property's value or NULL if the property was not found.}
}
*/
LmiPropertyValue* LmiPropertiesGetPropertyAtPathCStr(LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesSetProperty}
	{parent: LmiProperties}
	{description: Sets the value of a property in the property set.  If the property doesn't already exist, the property is added.
	  Setting a property triggers matching top level property listeners.}
	{prototype: LmiBool LmiPropertiesSetProperty(LmiProperties* x, const LmiString* name, const LmiPropertyValue* v)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: name}
		{description: The name of the property.}
	}
	{parameter:
		{name: v}
		{description: The property value to set the property to.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
	{note: Path will be created if does not exist.}
}
*/
LmiBool LmiPropertiesSetProperty(LmiProperties* x, const LmiString* name, const LmiPropertyValue* v);

/**
{function:
	{name: LmiPropertiesSetPropertyCStr}
	{parent: LmiProperties}
	{description: Sets the value of a property in the property set. If the property doesn't already exist, the property is added.
	  Setting a property triggers top level property listeners registered for the property.}
	{prototype: LmiBool LmiPropertiesSetPropertyCStr(LmiProperties* x, const char* name, const LmiPropertyValue* v)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: name}
		{description: The name of the property.}
	}
	{parameter:
		{name: v}
		{description: The property value to set the property to.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetPropertyCStr(LmiProperties* x, const char* name, const LmiPropertyValue* v);

/**
{function:
	{name: LmiPropertiesSetPropertyAtPath}
	{parent: LmiProperties}
	{description: Sets the value of a property in the property set at the specified path.  If the property doesn't already exist,
 	  the property is added.  Setting a property triggers matching property listeners.}
	{prototype: LmiBool LmiPropertiesSetPropertyAtPath(LmiProperties* x, const LmiString* path, const LmiPropertyValue* v)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: path}
		{description: The path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: v}
		{description: The property value to set the property to.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
	{note: Path will be created if does not exist.}
}
*/
LmiBool LmiPropertiesSetPropertyAtPath(LmiProperties* x, const LmiString* path, const LmiPropertyValue* v);

/**
{function:
	{name: LmiPropertiesSetPropertyAtPathCStr}
	{parent: LmiProperties}
	{description: Sets the value of a property in the property set at the specified path.  If the property doesn't already exist,
 	  the property is added unless it specifies a vector element (Ex. Calling this function where path is "vec[0]" and vec
	  is an empty vector will fail).  Setting a property triggers matching property listeners.}
	{prototype: LmiBool LmiPropertiesSetPropertyAtPathCStr(LmiProperties* x, const char* path, const LmiPropertyValue* v)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: path}
		{description: The path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: v}
		{description: The property value to set the property to.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
	{note: Path will be created if does not exist.
	If one of the path's components already exists and is not a type of LmiProperties, it will be replaced with LmiProperties.}
}
*/
LmiBool LmiPropertiesSetPropertyAtPathCStr(LmiProperties* x, const char* path, const LmiPropertyValue* v);

/**
{function:
	{name: LmiPropertiesMerge}
	{parent: LmiProperties}
	{description: Merges two properties. Read only properties are not updated.}
	{prototype: LmiProperties* LmiPropertiesMerge(LmiProperties* x, const LmiProperties* y)}
	{parameter:
		{name: x}
		{description: A property set that will contain a merged value.}
	}
	{parameter:
		{name: y}
		{description: A property set.}
	}
	{return: Returns a pointer to the merged object or NULL on error.}
}
*/
LmiProperties* LmiPropertiesMerge(LmiProperties* x, const LmiProperties* y);

/* Properties helper functions */
/* Setting */

/**
{function:
	{name: LmiPropertiesSetBoolValue}
	{parent: LmiProperties}
	{description: Sets LmiBool property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetBoolValue(LmiProperties* x, const LmiString* path, LmiBool b, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: b}
		{description: Value of the property of LmiBool type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetBoolValue(LmiProperties* x, const LmiString* path, LmiBool b, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetIntValue}
	{parent: LmiProperties}
	{description: Sets LmiInt property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetIntValue(LmiProperties* x, const LmiString* path, LmiInt i, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: i}
		{description: Value of the property of LmiInt type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetIntValue(LmiProperties* x, const LmiString* path, LmiInt i, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetInt16Value}
	{parent: LmiProperties}
	{description: Sets LmiInt16 property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetInt16Value(LmiProperties* x, const LmiString* path, LmiInt16 i16, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: i16}
		{description: Value of the property of LmiInt16 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetInt16Value(LmiProperties* x, const LmiString* path, LmiInt16 i16, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetInt64Value}
	{parent: LmiProperties}
	{description: Sets LmiInt64 property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetInt64Value(LmiProperties* x, const LmiString* path, LmiInt64 i64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: i64}
		{description: Value of the property of LmiInt64 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetInt64Value(LmiProperties* x, const LmiString* path, LmiInt64 i64, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetUintValue}
	{parent: LmiProperties}
	{description: Sets LmiUint property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetUintValue(LmiProperties* x, const LmiString* path, LmiUint ui, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: ui}
		{description: Value of the property of LmiUint type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetUintValue(LmiProperties* x, const LmiString* path, LmiUint ui, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetUint16Value}
	{parent: LmiProperties}
	{description: Sets LmiUint16 property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetUint16Value(LmiProperties* x, const LmiString* path, LmiUint16 ui16, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: ui16}
		{description: Value of the property of LmiUint16 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetUint16Value(LmiProperties* x, const LmiString* path, LmiUint16 ui16, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetUint64Value}
	{parent: LmiProperties}
	{description: Sets LmiUint64 property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetUint64Value(LmiProperties* x, const LmiString* path, LmiUint64 ui64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: ui64}
		{description: Value of the property of LmiUint64 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetUint64Value(LmiProperties* x, const LmiString* path, LmiUint64 ui64, LmiBool settable);



/**
{function:
	{name: LmiPropertiesSetSizeTValue}
	{parent: LmiProperties}
	{description: Sets LmiSizeT property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetSizeTValue(LmiProperties* x, const LmiString* path, LmiSizeT sz, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: sz}
		{description: Value of the property of LmiSizeT type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetSizeTValue(LmiProperties* x, const LmiString* path, LmiSizeT sz, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetFloat32Value}
	{parent: LmiProperties}
	{description: Sets LmiFloat32 property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetFloat32Value(LmiProperties* x, const LmiString* path, LmiFloat32 f32, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: f32}
		{description: Value of the property of LmiFloat32 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetFloat32Value(LmiProperties* x, const LmiString* path, LmiFloat32 f32, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetFloat64Value}
	{parent: LmiProperties}
	{description: Sets LmiFloat64 property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetFloat64Value(LmiProperties* x, const LmiString* path, LmiFloat64 f64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: f64}
		{description: Value of the property of LmiFloat64 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetFloat64Value(LmiProperties* x, const LmiString* path, LmiFloat64 f64, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetTimeValue}
	{parent: LmiProperties}
	{description: Sets LmiTime property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetTimeValue(LmiProperties* x, const LmiString* path, LmiTime t, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: t}
		{description: Value of the property of LmiTime type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetTimeValue(LmiProperties* x, const LmiString* path, LmiTime t, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetResolutionValue}
	{parent: LmiProperties}
	{description: Sets LmiResolution property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetResolutionValue(LmiProperties* x, const LmiString* path, LmiResolution res, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: res}
		{description: Value of the property of LmiResolution type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetResolutionValue(LmiProperties* x, const LmiString* path, LmiResolution res, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetMediaFormatValue}
	{parent: LmiProperties}
	{description: Sets LmiMediaFormat property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetMediaFormatValue(LmiProperties* x, const LmiString* path, LmiMediaFormat fmt, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: fmt}
		{description: Value of the property of LmiMediaFormat type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetMediaFormatValue(LmiProperties* x, const LmiString* path, LmiMediaFormat fmt, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetVoidPtrValue}
	{parent: LmiProperties}
	{description: Sets LmiVoidPtr property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetVoidPtrValue(LmiProperties* x, const LmiString* path, LmiVoidPtr vp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: vp}
		{description: Value of the property of LmiVoidPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetVoidPtrValue(LmiProperties* x, const LmiString* path, LmiVoidPtr vp, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetStringValue}
	{parent: LmiProperties}
	{description: Sets LmiString property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetStringValue(LmiProperties* x, const LmiString* path, const LmiString* s, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: s}
		{description: Value of the property of LmiString pointer type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetStringValue(LmiProperties* x, const LmiString* path, const LmiString* s, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetStringCStrValue}
	{parent: LmiProperties}
	{description: Sets C string property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetStringCStrValue(LmiProperties* x, const LmiString* path, const char* str, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: str}
		{description: Value of the property of C string type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetStringCStrValue(LmiProperties* x, const LmiString* path, const char* str, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetPropertiesValue}
	{parent: LmiProperties}
	{description: Sets LmiProperties property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetPropertiesValue(LmiProperties* x, const LmiString* path, const LmiProperties* pr, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: pr}
		{description: Value of the property of LmiProperties pointer type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetPropertiesValue(LmiProperties* x, const LmiString* path, const struct LmiProperties_* pr, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetVectorValue}
	{parent: LmiProperties}
	{description: Sets LmiVector(LmiPropertyValue) property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetVectorValue(LmiProperties* x, const LmiString* path, const LmiVector(LmiPropertyValue)* v, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: v}
		{description: Value of the property of LmiVector(LmiPropertyValue) type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetVectorValue(LmiProperties* x, const LmiString* path, const LmiVector(LmiPropertyValue)* pr, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetSharedVoidPtrValue}
	{parent: LmiProperties}
	{description: Sets LmiSharedVoidPtr property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetSharedVoidPtrValue(LmiProperties* x, const LmiString* path, const LmiSharedVoidPtr* sp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: sp}
		{description: Value of the property of LmiSharedVoidPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetSharedVoidPtrValue(LmiProperties* x, const LmiString* path, const LmiSharedVoidPtr* sp, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetWeakVoidPtrValue}
	{parent: LmiProperties}
	{description: Sets LmiWeakVoidPtr property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetWeakVoidPtrValue(LmiProperties* x, const LmiString* path, const LmiWeakVoidPtr* wp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: wp}
		{description: Value of the property of LmiWeakVoidPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetWeakVoidPtrValue(LmiProperties* x, const LmiString* path, const LmiWeakVoidPtr* wp, LmiBool settable);

/* Setting CSTR */
/**
{function:
	{name: LmiPropertiesSetBoolValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiBool property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetBoolValueCStr(LmiProperties* x, const char* path, LmiBool b, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: b}
		{description: Value of the property of LmiBool type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetBoolValueCStr(LmiProperties* x, const char* path, LmiBool b, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetIntValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiInt property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetIntValueCStr(LmiProperties* x, const char* path, LmiInt i, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: i}
		{description: Value of the property of LmiInt type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetIntValueCStr(LmiProperties* x, const char* path, LmiInt i, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetInt16ValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiInt16 property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetInt16ValueCStr(LmiProperties* x, const char* path, LmiInt16 i16, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: i16}
		{description: Value of the property of LmiInt16 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetInt16ValueCStr(LmiProperties* x, const char* path, LmiInt16 i16, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetInt64ValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiInt64 property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetInt64ValueCStr(LmiProperties* x, const char* path, LmiInt64 i64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: i64}
		{description: Value of the property of LmiInt64 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetInt64ValueCStr(LmiProperties* x, const char* path, LmiInt64 i64, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetUintValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiUint property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetUintValueCStr(LmiProperties* x, const char* path, LmiUint ui, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: ui}
		{description: Value of the property of LmiUint type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetUintValueCStr(LmiProperties* x, const char* path, LmiUint ui, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetUint16ValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiUint16 property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetUint16ValueCStr(LmiProperties* x, const char* path, LmiUint16 ui16, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: ui16}
		{description: Value of the property of LmiUint16 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetUint16ValueCStr(LmiProperties* x, const char* path, LmiUint16 ui16, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetUint64ValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiUint64 property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetUint64ValueCStr(LmiProperties* x, const char* path, LmiUint64 ui64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: ui64}
		{description: Value of the property of LmiUint64 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetUint64ValueCStr(LmiProperties* x, const char* path, LmiUint64 ui64, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetSizeTValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiSizeT property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetSizeTValueCStr(LmiProperties* x, const char* path, LmiSizeT sz, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: sz}
		{description: Value of the property of LmiSizeT type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetSizeTValueCStr(LmiProperties* x, const char* path, LmiSizeT sz, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetFloat32ValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiFloat32 property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetFloat32ValueCStr(LmiProperties* x, const char* path, LmiFloat32 f32, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: f32}
		{description: Value of the property of LmiFloat32 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetFloat32ValueCStr(LmiProperties* x, const char* path, LmiFloat32 f32, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetFloat64ValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiFloat64 property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetFloat64ValueCStr(LmiProperties* x, const char* path, LmiFloat64 f64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: f64}
		{description: Value of the property of LmiFloat64 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetFloat64ValueCStr(LmiProperties* x, const char* path, LmiFloat64 f64, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetTimeValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiTime property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetTimeValueCStr(LmiProperties* x, const char* path, LmiTime t, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: t}
		{description: Value of the property of LmiTime type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetTimeValueCStr(LmiProperties* x, const char* path, LmiTime t, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetResolutionValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiResolution property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetResolutionValueCStr(LmiProperties* x, const char* path, LmiResolution res, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: res}
		{description: Value of the property of LmiResolution type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetResolutionValueCStr(LmiProperties* x, const char* path, LmiResolution res, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetMediaFormatValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiMediaFormat property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetMediaFormatValueCStr(LmiProperties* x, const char* path, LmiMediaFormat fmt, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: fmt}
		{description: Value of the property of LmiMediaFormat type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetMediaFormatValueCStr(LmiProperties* x, const char* path, LmiMediaFormat fmt, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetVoidPtrValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiVoidPtr property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetVoidPtrValueCStr(LmiProperties* x, const char* path, LmiVoidPtr vp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: vp}
		{description: Value of the property of LmiVoidPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetVoidPtrValueCStr(LmiProperties* x, const char* path, LmiVoidPtr vp, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetStringValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiString property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetStringValueCStr(LmiProperties* x, const char* path, const LmiString* s, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: s}
		{description: Value of the property of LmiString pointer type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetStringValueCStr(LmiProperties* x, const char* path, const LmiString* s, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetStringCStrValueCStr}
	{parent: LmiProperties}
	{description: Sets C string property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetStringCStrValueCStr(LmiProperties* x, const char* path, const char* str, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: str}
		{description: Value of the property of C string type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetStringCStrValueCStr(LmiProperties* x, const char* path, const char* str, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetPropertiesValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiProperties property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetPropertiesValueCStr(LmiProperties* x, const char* path, const LmiProperties* pr, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: pr}
		{description: Value of the property of LmiProperties pointer type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetPropertiesValueCStr(LmiProperties* x, const char* path, const struct LmiProperties_* pr, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetVectorValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiVector(LmiPropertyValue) property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetVectorValueCStr(LmiProperties* x, const char* path, const LmiVector(LmiPropertyValue)* v, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: v}
		{description: Value of the property of LmiProperties pointer type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetVectorValueCStr(LmiProperties* x, const char* path, const LmiVector(LmiPropertyValue)* v, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetSharedVoidPtrValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiSharedVoidPtr property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetSharedVoidPtrValueCStr(LmiProperties* x, const char* path, const LmiSharedVoidPtr* sp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: sp}
		{description: Value of the property of LmiSharedVoidPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetSharedVoidPtrValueCStr(LmiProperties* x, const char* path, const LmiSharedVoidPtr* sp, LmiBool settable);

/**
{function:
	{name: LmiPropertiesSetWeakVoidPtrValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiWeakVoidPtr property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetWeakVoidPtrValueCStr(LmiProperties* x, const char* path, const LmiWeakVoidPtr* wp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: wp}
		{description: Value of the property of LmiWeakVoidPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesSetWeakVoidPtrValueCStr(LmiProperties* x, const char* path, const LmiWeakVoidPtr* wp, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddBoolValue}
	{parent: LmiProperties}
	{description: Adds LmiBool property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddBoolValue(LmiProperties* x, const LmiString* path, LmiBool b, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: b}
		{description: Value of the property of LmiBool type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/


LmiBool LmiPropertiesAddBoolValue(LmiProperties* x, const LmiString* path, LmiBool b, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddIntValue}
	{parent: LmiProperties}
	{description: Adds LmiInt property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddIntValue(LmiProperties* x, const LmiString* path, LmiInt i, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: i}
		{description: Value of the property of LmiInt type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddIntValue(LmiProperties* x, const LmiString* path, LmiInt i, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddInt16Value}
	{parent: LmiProperties}
	{description: Adds LmiInt16 property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddInt16Value(LmiProperties* x, const LmiString* path, LmiInt16 i16, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: i16}
		{description: Value of the property of LmiInt16 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddInt16Value(LmiProperties* x, const LmiString* path, LmiInt16 i16, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddInt64Value}
	{parent: LmiProperties}
	{description: Adds LmiInt64 property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddInt64Value(LmiProperties* x, const LmiString* path, LmiInt64 i64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: i64}
		{description: Value of the property of LmiInt64 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddInt64Value(LmiProperties* x, const LmiString* path, LmiInt64 i64, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddUintValue}
	{parent: LmiProperties}
	{description: Adds LmiUint property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddUintValue(LmiProperties* x, const LmiString* path, LmiUint ui, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: ui}
		{description: Value of the property of LmiUint type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddUintValue(LmiProperties* x, const LmiString* path, LmiUint ui, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddUint16Value}
	{parent: LmiProperties}
	{description: Adds LmiUint16 property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddUint16Value(LmiProperties* x, const LmiString* path, LmiUint16 ui16, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: ui16}
		{description: Value of the property of LmiUint16 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddUint16Value(LmiProperties* x, const LmiString* path, LmiUint16 ui16, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddUint64Value}
	{parent: LmiProperties}
	{description: Adds LmiUint64 property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddUint64Value(LmiProperties* x, const LmiString* path, LmiUint64 ui64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: ui64}
		{description: Value of the property of LmiUint64 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddUint64Value(LmiProperties* x, const LmiString* path, LmiUint64 ui64, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddSizeTValue}
	{parent: LmiProperties}
	{description: Adds LmiSizeT property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddSizeTValue(LmiProperties* x, const LmiString* path, LmiSizeT sz, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: sz}
		{description: Value of the property of LmiSizeT type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddSizeTValue(LmiProperties* x, const LmiString* path, LmiSizeT sz, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddFloat32Value}
	{parent: LmiProperties}
	{description: Adds LmiFloat32 property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddFloat32Value(LmiProperties* x, const LmiString* path, LmiFloat32 f32, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: f32}
		{description: Value of the property of LmiFloat32 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddFloat32Value(LmiProperties* x, const LmiString* path, LmiFloat32 f32, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddFloat64Value}
	{parent: LmiProperties}
	{description: Adds LmiFloat64 property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddFloat64Value(LmiProperties* x, const LmiString* path, LmiFloat64 f64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: f64}
		{description: Value of the property of LmiFloat64 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddFloat64Value(LmiProperties* x, const LmiString* path, LmiFloat64 f64, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddTimeValue}
	{parent: LmiProperties}
	{description: Adds LmiTime property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddTimeValue(LmiProperties* x, const LmiString* path, LmiTime t, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: t}
		{description: Value of the property of LmiTime type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddTimeValue(LmiProperties* x, const LmiString* path, LmiTime t, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddResolutionValue}
	{parent: LmiProperties}
	{description: Adds LmiResolution property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddResolutionValue(LmiProperties* x, const LmiString* path, LmiResolution res, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: res}
		{description: Value of the property of LmiResolution type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddResolutionValue(LmiProperties* x, const LmiString* path, LmiResolution res, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddMediaFormatValue}
	{parent: LmiProperties}
	{description: Adds LmiMediaFormat property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddMediaFormatValue(LmiProperties* x, const LmiString* path, LmiMediaFormat fmt, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: fmt}
		{description: Value of the property of LmiMediaFormat type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddMediaFormatValue(LmiProperties* x, const LmiString* path, LmiMediaFormat fmt, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddVoidPtrValue}
	{parent: LmiProperties}
	{description: Adds LmiVoidPtr property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddVoidPtrValue(LmiProperties* x, const LmiString* path, LmiVoidPtr vp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: vp}
		{description: Value of the property of LmiVoidPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddVoidPtrValue(LmiProperties* x, const LmiString* path, LmiVoidPtr vp, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddStringValue}
	{parent: LmiProperties}
	{description: Adds LmiString property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddStringValue(LmiProperties* x, const LmiString* path, const LmiString* s, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: s}
		{description: Value of the property of LmiString pointer type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddStringValue(LmiProperties* x, const LmiString* path, const LmiString* s, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddStringCStrValue}
	{parent: LmiProperties}
	{description: Adds C string property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddStringCStrValue(LmiProperties* x, const LmiString* path, const char* str, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: str}
		{description: Value of the property of C string type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddStringCStrValue(LmiProperties* x, const LmiString* path, const char* str, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddPropertiesValue}
	{parent: LmiProperties}
	{description: Adds LmiProperties property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddPropertiesValue(LmiProperties* x, const LmiString* path, const LmiProperties* pr, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: pr}
		{description: Value of the property of LmiProperties pointer type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddPropertiesValue(LmiProperties* x, const LmiString* path, const struct LmiProperties_* pr, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddVectorValue}
	{parent: LmiProperties}
	{description: Adds LmiVector(LmiPropertyValue) property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddVectorValue(LmiProperties* x, const LmiString* path, const LmiVector(LmiPropertyValue)* v, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: v}
		{description: Value of the property of LmiVector(LmiPropertyValue) type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddVectorValue(LmiProperties* x, const LmiString* path, const LmiVector(LmiPropertyValue)* v, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddSharedVoidPtrValue}
	{parent: LmiProperties}
	{description: Adds LmiSharedVoidPtr property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddSharedVoidPtrValue(LmiProperties* x, const LmiString* path, const LmiSharedVoidPtr* sp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: sp}
		{description: Value of the property of LmiSharedVoidPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddSharedVoidPtrValue(LmiProperties* x, const LmiString* path, const LmiSharedVoidPtr* sp, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddWeakVoidPtrValue}
	{parent: LmiProperties}
	{description: Adds LmiWeakVoidPtr property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddWeakVoidPtrValue(LmiProperties* x, const LmiString* path, const LmiWeakVoidPtr* wp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: wp}
		{description: Value of the property of LmiWeakVoidPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddWeakVoidPtrValue(LmiProperties* x, const LmiString* path, const LmiWeakVoidPtr* wp, LmiBool settable);

/* Setting CSTR */
/**
{function:
	{name: LmiPropertiesAddBoolValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiBool property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddBoolValueCStr(LmiProperties* x, const char* path, LmiBool b, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: b}
		{description: Value of the property of LmiBool type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddBoolValueCStr(LmiProperties* x, const char* path, LmiBool b, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddIntValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiInt property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddIntValueCStr(LmiProperties* x, const char* path, LmiInt i, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: i}
		{description: Value of the property of LmiInt type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddIntValueCStr(LmiProperties* x, const char* path, LmiInt i, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddInt16ValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiInt16 property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddInt16ValueCStr(LmiProperties* x, const char* path, LmiInt16 i16, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: i16}
		{description: Value of the property of LmiInt16 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddInt16ValueCStr(LmiProperties* x, const char* path, LmiInt16 i16, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddInt64ValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiInt64 property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddInt64ValueCStr(LmiProperties* x, const char* path, LmiInt64 i64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: i64}
		{description: Value of the property of LmiInt64 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddInt64ValueCStr(LmiProperties* x, const char* path, LmiInt64 i64, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddUintValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiUint property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddUintValueCStr(LmiProperties* x, const char* path, LmiUint ui, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: ui}
		{description: Value of the property of LmiUint type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddUintValueCStr(LmiProperties* x, const char* path, LmiUint ui, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddUint16ValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiUint16 property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddUint16ValueCStr(LmiProperties* x, const char* path, LmiUint16 ui16, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: ui16}
		{description: Value of the property of LmiUint16 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddUint16ValueCStr(LmiProperties* x, const char* path, LmiUint16 ui16, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddUint64ValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiUint64 property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddUint64ValueCStr(LmiProperties* x, const char* path, LmiUint64 ui64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: ui64}
		{description: Value of the property of LmiUint64 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddUint64ValueCStr(LmiProperties* x, const char* path, LmiUint64 ui64, LmiBool settable);
/**
{function:
	{name: LmiPropertiesAddSizeTValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiSizeT property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddSizeTValueCStr(LmiProperties* x, const char* path, LmiSizeT sz, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: sz}
		{description: Value of the property of LmiSizeT type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddSizeTValueCStr(LmiProperties* x, const char* path, LmiSizeT sz, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddFloat32ValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiFloat32 property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddFloat32ValueCStr(LmiProperties* x, const char* path, LmiFloat32 f32, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: f32}
		{description: Value of the property of LmiFloat32 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddFloat32ValueCStr(LmiProperties* x, const char* path, LmiFloat32 f32, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddFloat64ValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiFloat64 property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddFloat64ValueCStr(LmiProperties* x, const char* path, LmiFloat64 f64, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: f64}
		{description: Value of the property of LmiFloat64 type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddFloat64ValueCStr(LmiProperties* x, const char* path, LmiFloat64 f64, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddTimeValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiTime property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddTimeValueCStr(LmiProperties* x, const char* path, LmiTime t, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: t}
		{description: Value of the property of LmiTime type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddTimeValueCStr(LmiProperties* x, const char* path, LmiTime t, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddResolutionValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiResolution property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddResolutionValueCStr(LmiProperties* x, const char* path, LmiResolution res, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: res}
		{description: Value of the property of LmiResolution type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddResolutionValueCStr(LmiProperties* x, const char* path, LmiResolution res, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddMediaFormatValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiMediaFormat property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddMediaFormatValueCStr(LmiProperties* x, const char* path, LmiMediaFormat fmt, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: fmt}
		{description: Value of the property of LmiMediaFormat type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddMediaFormatValueCStr(LmiProperties* x, const char* path, LmiMediaFormat fmt, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddVoidPtrValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiVoidPtr property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddVoidPtrValueCStr(LmiProperties* x, const char* path, LmiVoidPtr vp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: vp}
		{description: Value of the property of LmiVoidPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddVoidPtrValueCStr(LmiProperties* x, const char* path, LmiVoidPtr vp, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddStringValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiString property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddStringValueCStr(LmiProperties* x, const char* path, const LmiString* s, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: s}
		{description: Value of the property of LmiString pointer type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddStringValueCStr(LmiProperties* x, const char* path, const LmiString* s, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddStringCStrValueCStr}
	{parent: LmiProperties}
	{description: Adds C string property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddStringCStrValueCStr(LmiProperties* x, const char* path, const char* str, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: str}
		{description: Value of the property of C string type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddStringCStrValueCStr(LmiProperties* x, const char* path, const char* str, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddPropertiesValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiProperties property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddPropertiesValueCStr(LmiProperties* x, const char* path, const LmiProperties* pr, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: pr}
		{description: Value of the property of LmiProperties pointer type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddPropertiesValueCStr(LmiProperties* x, const char* path, const struct LmiProperties_* pr, LmiBool settable);
/**
{function:
	{name: LmiPropertiesAddVectorValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiVector(LmiPropertyValue) property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddVectorValueCStr(LmiProperties* x, const char* path, const LmiVector(LmiPropertyValue)* v, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: v}
		{description: Value of the property of LmiVector(LmiPropertyValue) type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddVectorValueCStr(LmiProperties* x, const char* path, const LmiVector(LmiPropertyValue)* v, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddSharedVoidPtrValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiSharedVoidPtr property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddSharedVoidPtrValueCStr(LmiProperties* x, const char* path, const LmiSharedVoidPtr* sp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: sp}
		{description: Value of the property of LmiSharedVoidPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddSharedVoidPtrValueCStr(LmiProperties* x, const char* path, const LmiSharedVoidPtr* sp, LmiBool settable);

/**
{function:
	{name: LmiPropertiesAddWeakVoidPtrValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiWeakVoidPtr property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddWeakVoidPtrValueCStr(LmiProperties* x, const char* path, const LmiWeakVoidPtr* wp, LmiBool settable)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: wp}
		{description: Value of the property of LmiWeakVoidPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
LmiBool LmiPropertiesAddWeakVoidPtrValueCStr(LmiProperties* x, const char* path, const LmiWeakVoidPtr* wp, LmiBool settable);

/* Getting */
/**
{function:
	{name: LmiPropertiesIsSettableValue}
	{parent: LmiProperties}
	{description: Determines whether or not a property value in a path is settable.}
	{prototype: const LmiBool* LmiPropertiesIsSettableValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns pointer to the boolean, which is LMI_TRUE if the property value is settable or LMI_FALSE otherwise. NULL on failure.}
}
*/
const LmiBool* LmiPropertiesIsSettableValue(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetTypeValue}
	{parent: LmiProperties}
	{description: Gets the property value type of a property value.}
	{prototype: const LmiPropertyValueType* LmiPropertiesGetTypeValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to the type of the property value or NULL on failure.}
}
*/
const LmiPropertyValueType* LmiPropertiesGetTypeValue(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetBoolValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiBool of a property value.}
	{prototype: const LmiBool* LmiPropertiesGetBoolValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiBool of the property value on success, or NULL if the type of the property value is not an LmiBool or property is not set.}
}
*/
const LmiBool* LmiPropertiesGetBoolValue(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetIntValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiInt of a property value.}
	{prototype: const LmiInt* LmiPropertiesGetIntValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiInt of the property value on success, or NULL if the type of the property value is not an LmiInt or property is not set.}
}
*/
const LmiInt* LmiPropertiesGetIntValue(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetInt16Value}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiInt16 of a property value.}
	{prototype: const LmiInt16* LmiPropertiesGetInt16Value(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiInt16 of the property value on success, or NULL if the type of the property value is not an LmiInt16 or property is not set.}
}
*/
const LmiInt16* LmiPropertiesGetInt16Value(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetInt64Value}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiInt64 of a property value.}
	{prototype: const LmiInt64* LmiPropertiesGetInt64Value(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiInt64 of the property value on success, or NULL if the type of the property value is not an LmiInt64 or property is not set.}
}
*/
const LmiInt64* LmiPropertiesGetInt64Value(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetUintValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiUint of a property value.}
	{prototype: const LmiUint* LmiPropertiesGetUintValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiUint of the property value on success, or NULL if the type of the property value is not an LmiUint or property is not set.}
}
*/
const LmiUint* LmiPropertiesGetUintValue(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetUint16Value}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiUint16 of a property value.}
	{prototype: const LmiUint16* LmiPropertiesGetUint16Value(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiUint16 of the property value on success, or NULL if the type of the property value is not an LmiUint16 or property is not set.}
}
*/
const LmiUint16* LmiPropertiesGetUint16Value(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetUint64Value}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiUint64 of a property value.}
	{prototype: const LmiUint64* LmiPropertiesGetUint64Value(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiUint64 of the property value on success, or NULL if the type of the property value is not an LmiUint64 or property is not set.}
}
*/
const LmiUint64* LmiPropertiesGetUint64Value(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetSizeTValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiSizeT of a property value.}
	{prototype: const LmiSizeT* LmiPropertiesGetSizeTValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiSizeT of the property value on success, or NULL if the type of the property value is not an LmiSizeT or property is not set.}
}
*/
const LmiSizeT* LmiPropertiesGetSizeTValue(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetFloat32Value}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiFloat32 of a property value.}
	{prototype: const LmiFloat32* LmiPropertiesGetFloat32Value(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiFloat32 of the property value on success, or NULL if the type of the property value is not an LmiFloat32 or property is not set.}
}
*/
const LmiFloat32* LmiPropertiesGetFloat32Value(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetFloat64Value}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiFloat64 of a property value.}
	{prototype: const LmiFloat64* LmiPropertiesGetFloat64Value(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiFloat64 of the property value on success, or NULL if the type of the property value is not an LmiFloat64 or property is not set.}
}
*/
const LmiFloat64* LmiPropertiesGetFloat64Value(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetTimeValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiTime of a property value.}
	{prototype: const LmiTime* LmiPropertiesGetTimeValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiTime of the property value on success, or NULL if the type of the property value is not an LmiTime or property is not set.}
}
*/
const LmiTime* LmiPropertiesGetTimeValue(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetResolutionValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiResolution of a property value.}
	{prototype: const LmiResolution* LmiPropertiesGetResolutionValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiResolution of the property value on success, or NULL if the type of the property value is not an LmiResolution or property is not set.}
}
*/
const LmiResolution* LmiPropertiesGetResolutionValue(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetMediaFormatValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiMediaFormat of a property value.}
	{prototype: const LmiMediaFormat* LmiPropertiesGetMediaFormatValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiMediaFormat of the property value on success, or NULL if the type of the property value is not an LmiMediaFormat or property is not set.}
}
*/
const LmiMediaFormat* LmiPropertiesGetMediaFormatValue(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetVoidPtrValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiVoidPtr of a property value.}
	{prototype: const LmiVoidPtr* LmiPropertiesGetVoidPtrValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiVoidPtr of the property value on success, or NULL if the type of the property value is not an LmiVoidPtr or property is not set.}
}
*/
const LmiVoidPtr* LmiPropertiesGetVoidPtrValue(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetStringValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiString of a property value.}
	{prototype: const LmiString* LmiPropertiesGetStringValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiString of the property value on success, or NULL if the type of the property value is not an LmiString or property is not set.}
}
*/
const LmiString* LmiPropertiesGetStringValue(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetPropertiesValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiProperties of a property value.}
	{prototype: const LmiProperties* LmiPropertiesGetPropertiesValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiProperties of the property value on success, or NULL if the type of the property value is not an LmiProperties or property is not set.}
}
*/
const LmiProperties* LmiPropertiesGetPropertiesValue(const LmiProperties* x, const LmiString* path);


/**
{function:
	{name: LmiPropertiesGetVectorValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiVector(LmiPropertyValue) of a property value.}
	{prototype: const LmiVector(LmiPropertyValue)* LmiPropertiesGetVectorValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiVector(LmiPropertyValue) of the property value on success, or NULL if the type of the property value is not an LmiVector(LmiPropertyValue) or property is not set.}
}
*/
const LmiVector(LmiPropertyValue)* LmiPropertiesGetVectorValue(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetSharedVoidPtrValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiSharedVoidPtr of a property value.}
	{prototype: const LmiSharedVoidPtr* LmiPropertiesGetSharedVoidPtrValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiSharedVoidPtr of the property value on success, or NULL if the type of the property value is not an LmiSharedVoidPtr or property is not set.}
}
*/
const LmiSharedVoidPtr* LmiPropertiesGetSharedVoidPtrValue(const LmiProperties* x, const LmiString* path);

/**
{function:
	{name: LmiPropertiesGetWeakVoidPtrValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiWeakVoidPtr of a property value.}
	{prototype: const LmiWeakVoidPtr* LmiPropertiesGetWeakVoidPtrValue(const LmiProperties* x, const LmiString* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiWeakVoidPtr of the property value on success, or NULL if the type of the property value is not an LmiWeakVoidPtr or property is not set.}
}
*/
const LmiWeakVoidPtr* LmiPropertiesGetWeakVoidPtrValue(const LmiProperties* x, const LmiString* path);

/* Getting CStr */
/**
{function:
	{name: LmiPropertiesIsSettableValueCStr}
	{parent: LmiProperties}
	{description: Determines whether or not a property value in a path is settable.}
	{prototype: const LmiBool* LmiPropertiesIsSettableValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns pointer to the boolean, which is LMI_TRUE if the property value is settable or LMI_FALSE otherwise. NULL on failure.}
}
*/
const LmiBool* LmiPropertiesIsSettableValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetTypeValueCStr}
	{parent: LmiProperties}
	{description: Gets the property value type of a property value.}
	{prototype: const LmiPropertyValueType* LmiPropertiesGetTypeValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to the type of the property value or NULL on failure.}
}
*/
const LmiPropertyValueType* LmiPropertiesGetTypeValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetBoolValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiBool of a property value.}
	{prototype: const LmiBool* LmiPropertiesGetBoolValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiBool of the property value on success, or NULL if the type of the property value is not an LmiBool or property is not set.}
}
*/
const LmiBool* LmiPropertiesGetBoolValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetIntValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiInt of a property value.}
	{prototype: const LmiInt* LmiPropertiesGetIntValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiInt of the property value on success, or NULL if the type of the property value is not an LmiInt or property is not set.}
}
*/
const LmiInt* LmiPropertiesGetIntValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetInt16ValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiInt16 of a property value.}
	{prototype: const LmiInt16* LmiPropertiesGetInt16ValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiInt16 of the property value on success, or NULL if the type of the property value is not an LmiInt16 or property is not set.}
}
*/
const LmiInt16* LmiPropertiesGetInt16ValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetInt64ValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiInt64 of a property value.}
	{prototype: const LmiInt64* LmiPropertiesGetInt64ValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiInt64 of the property value on success, or NULL if the type of the property value is not an LmiInt64 or property is not set.}
}
*/
const LmiInt64* LmiPropertiesGetInt64ValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetUintValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiUint of a property value.}
	{prototype: const LmiUint* LmiPropertiesGetUintValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiUint of the property value on success, or NULL if the type of the property value is not an LmiUint or property is not set.}
}
*/
const LmiUint* LmiPropertiesGetUintValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetUint16ValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiUint16 of a property value.}
	{prototype: const LmiUint16* LmiPropertiesGetUint16ValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiUint16 of the property value on success, or NULL if the type of the property value is not an LmiUint16 or property is not set.}
}
*/
const LmiUint16* LmiPropertiesGetUint16ValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetUint64ValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiUint64 of a property value.}
	{prototype: const LmiUint64* LmiPropertiesGetUint64ValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiUint64 of the property value on success, or NULL if the type of the property value is not an LmiUint64 or property is not set.}
}
*/
const LmiUint64* LmiPropertiesGetUint64ValueCStr(const LmiProperties* x, const char* path);
/**
{function:
	{name: LmiPropertiesGetSizeTValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiSizeT of a property value.}
	{prototype: const LmiSizeT* LmiPropertiesGetSizeTValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiSizeT of the property value on success, or NULL if the type of the property value is not an LmiSizeT or property is not set.}
}
*/
const LmiSizeT* LmiPropertiesGetSizeTValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetFloat32ValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiFloat32 of a property value.}
	{prototype: const LmiFloat32* LmiPropertiesGetFloat32ValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiFloat32 of the property value on success, or NULL if the type of the property value is not an LmiFloat32 or property is not set.}
}
*/
const LmiFloat32* LmiPropertiesGetFloat32ValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetFloat64ValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiFloat64 of a property value.}
	{prototype: const LmiFloat64* LmiPropertiesGetFloat64ValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiFloat64 of the property value on success, or NULL if the type of the property value is not an LmiFloat64 or property is not set.}
}
*/
const LmiFloat64* LmiPropertiesGetFloat64ValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetTimeValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiTime of a property value.}
	{prototype: const LmiTime* LmiPropertiesGetTimeValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiTime of the property value on success, or NULL if the type of the property value is not an LmiTime or property is not set.}
}
*/
const LmiTime* LmiPropertiesGetTimeValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetResolutionValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiResolution of a property value.}
	{prototype: const LmiResolution* LmiPropertiesGetResolutionValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiResolution of the property value on success, or NULL if the type of the property value is not an LmiResolution or property is not set.}
}
*/
const LmiResolution* LmiPropertiesGetResolutionValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetMediaFormatValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiMediaFormat of a property value.}
	{prototype: const LmiMediaFormat* LmiPropertiesGetMediaFormatValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiMediaFormat of the property value on success, or NULL if the type of the property value is not an LmiMediaFormat or property is not set.}
}
*/
const LmiMediaFormat* LmiPropertiesGetMediaFormatValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetVoidPtrValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiVoidPtr of a property value.}
	{prototype: const LmiVoidPtr* LmiPropertiesGetVoidPtrValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiVoidPtr of the property value on success, or NULL if the type of the property value is not an LmiVoidPtr or property is not set.}
}
*/
const LmiVoidPtr* LmiPropertiesGetVoidPtrValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetStringValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiString of a property value.}
	{prototype: const LmiString* LmiPropertiesGetStringValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiString of the property value on success, or NULL if the type of the property value is not an LmiString or property is not set.}
}
*/
const LmiString* LmiPropertiesGetStringValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetPropertiesValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiProperties of a property value.}
	{prototype: const LmiProperties* LmiPropertiesGetPropertiesValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiProperties of the property value on success, or NULL if the type of the property value is not an LmiProperties or property is not set.}
}
*/
const LmiProperties* LmiPropertiesGetPropertiesValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetVectorValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiVector(LmiPropertyValue) of a property value.}
	{prototype: const LmiVector(LmiPropertyValue)* LmiPropertiesGetVectorValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiVector(LmiPropertyValue) of the property value on success, or NULL if the type of the property value is not an LmiVector(LmiPropertyValue) or property is not set.}
}
*/
const LmiVector(LmiPropertyValue)* LmiPropertiesGetVectorValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetSharedVoidPtrValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiSharedVoidPtr of a property value.}
	{prototype: const LmiSharedVoidPtr* LmiPropertiesGetSharedVoidPtrValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiSharedVoidPtr of the property value on success, or NULL if the type of the property value is not an LmiSharedVoidPtr or property is not set.}
}
*/
const LmiSharedVoidPtr* LmiPropertiesGetSharedVoidPtrValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetWeakVoidPtrValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiWeakVoidPtr of a property value.}
	{prototype: const LmiWeakVoidPtr* LmiPropertiesGetWeakVoidPtrValueCStr(const LmiProperties* x, const char* path)}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: path}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiWeakVoidPtr of the property value on success, or NULL if the type of the property value is not an LmiWeakVoidPtr or property is not set.}
}
*/
const LmiWeakVoidPtr* LmiPropertiesGetWeakVoidPtrValueCStr(const LmiProperties* x, const char* path);

/**
{function:
	{name: LmiPropertiesGetAllNames}
	{parent: LmiProperties}
	{description: Gets all the names of the property values in the property set.}
	{prototype: LmiBool LmiPropertiesGetAllNames(const LmiProperties* x, LmiVector(LmiString) *names)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: names}
		{description: The names.}
	}
	{return: Returns LMI_TRUE if successful or LMI_FALSE otherwise.}
}
*/
LmiBool LmiPropertiesGetAllNames(const LmiProperties* x, LmiVector(LmiString) *names);

/**
{function:
	{name: LmiPropertiesPrint}
	{parent: LmiProperties}
	{description: Print all the property values in the property set to the specified file stream.}
	{prototype: void LmiPropertiesPrint(const LmiProperties* x, LmiFile* file, const char* tabs)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: file}
		{description: The file stream to dump the property set\'s contents to.}
	}
 	{parameter:
		{name: tabs}
		{description: The C string containing tabs specifying how far to indent the output.}
	}
}
*/
void LmiPropertiesPrint(const LmiProperties* x, LmiFile* file, const char* tabs);

/**
{function:
	{name: LmiPropertiesToXml}
	{parent: LmiProperties}
	{description: Export the properties as an XML document.}
	{prototype: const LmiString* LmiPropertiesToXml(const LmiProperties* x, LmiString* xml)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: xml}
		{description: A string into which the properties are encoded as XML.}
	}
 {return: Pointer to a string representing {code:x}, or NULL on failure.}
}
*/
const LmiString* LmiPropertiesToXml(const LmiProperties* x, LmiString* xml);

/**
{function:
	{name: LmiPropertiesToJson}
	{parent: LmiProperties}
	{description: Export the properties as an JSON document.}
	{prototype: const LmiString* LmiPropertiesToJson(const LmiProperties* x, LmiString* xml)}
	{parameter:
		{name: x}
		{description: A property set.}
	}
	{parameter:
		{name: xml}
		{description: A string into which the properties are encoded as JSON.}
	}
 {return: Pointer to a string representing {code:x}, or NULL on failure.}
}
*/
const LmiString* LmiPropertiesToJson(const LmiProperties* x, LmiString* xml);

/**
{function visibility="private":
	{name: LmiPropertiesReadFromString}
	{parent: LmiProperties}
	{description: Read properties from a compact string.}
	{prototype: LmiBool LmiPropertiesReadFromString(LmiProperties *props, const char *s)}
	{parameter: {name: props} {description: The properties object.}}
	{parameter: {name: s} {description: The string.}}
	{return: Whether the string parsed successfully.}
	{note: Types are auto-detected from syntax.}
	{note: Grammar:
		{p: ROOT}
		{p: -> PROP [',' PROP]*}
		{p: PROP}
		{p: -> NAME ('=' VALUE | '\{' ROOT '\}' )}
		{p: NAME}
		{p: -> (ALNUM | '_')+}
		{p: VALUE}
		{p: -> 'T'}
		{p: -> 'F'}
		{p: -> ['+' | '-'] DIGIT+ ['.' DIGIT*]}
		{p: -> '"' PRINTABLE* '"'}
	}
}
*/
LmiBool LmiPropertiesReadFromString(LmiProperties *props, const char *s);

/////////////// Shared pointer macros

/* LmiProperties for shared pointers */
// LmiPropertyValue

/* Magic MACROs to define functions correctly */
#define LmiPropertyValueFunc_(T_,Func_) LmiTemplate1(LmiClassFunc(LmiPropertyValue, Func_), T_)

#define LmiPropertyValueConstructSharedPtr(T_) LmiPropertyValueFunc_(T_, ConstructSharedPtr)

/**
{function:
    {name: LmiPropertyValueConstructSharedPtr}
    {parent: LmiPropertyValue}
    {description: Constructs an LmiSharedPtr property value corresponding to a specified pointer.}
    {prototype: LmiPropertyValue* LmiPropertyValueConstructSharedPtr(T_)(LmiPropertyValue * x, const LmiSharedPtr(T_) *sp, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the shared pointer.}}
    {parameter: {name: x} {description: A property value.}}
    {parameter: {name: sp} {description: The LmiSharedPtr to assign to the property value.}}
    {parameter: {name: settable} {description: Defines if property can be overwritten.}}
    {return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
#define Declare_LmiPropertyValueConstructSharedPtr(T_) \
LMI_INLINE_DECLARATION LmiPropertyValue* LmiPropertyValueConstructSharedPtr(T_)(LmiPropertyValue * x, const LmiSharedPtr(T_) *sp, LmiBool settable);

#define Define_LmiPropertyValueConstructSharedPtr(T_) \
LmiPropertyValue *LmiPropertyValueConstructSharedPtr(T_)(LmiPropertyValue *x, const LmiSharedPtr(T_) *sp, LmiBool settable) \
{                                                                           \
    x->settable = settable;                                                 \
    x->type = LMI_PROPERTYVALUE_TYPE_SharedPtr;                             \
    if (LmiStringConstructCStr(&x->ptrName,                                 \
                           LmiQuoteValue(T_),                               \
                           LmiAllocatorGetDefault()) == NULL)         \
        return NULL;                                                        \
    if (LmiSharedVoidPtrConstructFromSharedPtr(T_)(&x->u.sp, sp) == NULL) { \
        LmiStringDestruct(&x->ptrName);                                     \
        return NULL;                                                        \
    }                                                                       \
    return x;                                                               \
}

#define LmiPropertyValueGetSharedPtr(T_) LmiPropertyValueFunc_(T_, GetSharedPtr)

/**
{function:
	{name: LmiPropertyValueGetSharedPtr}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiSharedPtr of a property value.}
	{prototype: const LmiSharedPtr(T_) *LmiPropertyValueGetSharedPtr(T_)(const LmiPropertyValue *x)}
	{template-parameter:
		{name: T_} {description: The type pointed to by the shared pointer.}}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiSharedPtr of the property value on success, or NULL if the type of the property value is not a correctly typified LmiSharedPtr.}
}
*/
#define Declare_LmiPropertyValueGetSharedPtr(T_) \
LMI_INLINE_DECLARATION const LmiSharedPtr(T_) *LmiPropertyValueGetSharedPtr(T_)(const LmiPropertyValue *x);

#define Define_LmiPropertyValueGetSharedPtr(T_) \
const LmiSharedPtr(T_)* LmiPropertyValueGetSharedPtr(T_)(const LmiPropertyValue* x) \
{                                                           \
    if (x->type != LMI_PROPERTYVALUE_TYPE_SharedPtr) {      \
        return NULL;                                        \
    }                                                       \
    if (!LmiStringEqualCStr(&x->ptrName,                    \
                           LmiQuoteValue(T_))) {            \
        return NULL;                                        \
    }                                                       \
    return (LmiSharedPtr(T_)*)&x->u.sp;                     \
}

#define LmiPropertyValueDereferenceSharedPtr(T_) LmiPropertyValueFunc_(T_, DereferenceSharedPtr)

/**
{function:
	{name: LmiPropertyValueDereferenceSharedPtr}
	{parent: LmiPropertyValue}
	{description: Gets a pointer to a data stored in an LmiSharedPtr of a property value.}
	{prototype: T_ *LmiPropertyValueDereferenceSharedPtr(T_)(const LmiPropertyValue *x)}
	{template-parameter:
		{name: T_} {description: The type pointed to by the shared pointer.}}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to a data stored in an LmiSharedPtr of the property value on success, or NULL if the type of the property value is not a correctly typified LmiSharedPtr.}
}
*/
#define Declare_LmiPropertyValueDereferenceSharedPtr(T_) \
LMI_INLINE_DECLARATION T_ *LmiPropertyValueDereferenceSharedPtr(T_)(const LmiPropertyValue *x);

#define Define_LmiPropertyValueDereferenceSharedPtr(T_) \
T_* LmiPropertyValueDereferenceSharedPtr(T_)(const LmiPropertyValue* x) \
{                                                                       \
    if (x->type != LMI_PROPERTYVALUE_TYPE_SharedPtr) {                  \
        return NULL;                                                    \
    }                                                                   \
    if (!LmiStringEqualCStr(&x->ptrName,                                \
                           LmiQuoteValue(T_))) {                        \
        return NULL;                                                    \
    }                                                                   \
    return LmiSharedPtrGet(T_)((LmiSharedPtr(T_)*)&x->u.sp);            \
}

#define LmiPropertyValueToSharedPtr(T_) LmiPropertyValueFunc_(T_, ToSharedPtr)

/**
{function:
	{name: LmiPropertyValueToSharedPtr}
	{parent: LmiPropertyValue}
	{description: Converts a property value to a shared pointer.}
	{prototype: LmiSharedPtr(T_) *LmiPropertyValueToSharedPtr(T_)(const LmiPropertyValue *x, LmiSharedPtr(T_) *sp)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the shared pointer.}}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: sp}
		{description: An typified LmiSharedPtr to hold a copy of the property's value. }
	}
	{return: Returns {code:sp} on success, or NULL if the type of the property value is not a correctly typified LmiSharedPtr or any other error occurs.}
}
*/
#define Declare_LmiPropertyValueToSharedPtr(T_) \
LMI_INLINE_DECLARATION LmiSharedPtr(T_) *LmiPropertyValueToSharedPtr(T_)(const LmiPropertyValue *x, LmiSharedPtr(T_) *sp);

#define Define_LmiPropertyValueToSharedPtr(T_) \
LmiSharedPtr(T_) *LmiPropertyValueToSharedPtr(T_)(const LmiPropertyValue *x, LmiSharedPtr(T_) *sp) \
{                                                                                           \
    if (x->type == LMI_PROPERTYVALUE_TYPE_SharedPtr) {                                      \
        if (!LmiStringEqualCStr(&x->ptrName, LmiQuoteValue(T_))) {                          \
            return NULL;                                                                    \
        }                                                                                   \
        return LmiSharedPtrAssignFromSharedVoidPtr(T_)(sp, &x->u.sp);                       \
    }                                                                                       \
    return NULL;                                                                            \
}

// LmiProperties

#define LmiPropertiesFunc_(T_,Func_) LmiTemplate1(LmiClassFunc(LmiProperties, Func_), T_)

//// Set

#define LmiPropertiesSetSharedPtrValue(T_) LmiPropertiesFunc_(T_, SetSharedPtrValue)

/**
{function:
	{name: LmiPropertiesSetSharedPtrValue}
	{parent: LmiProperties}
	{description: Sets LmiSharedPtr property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetSharedPtrValue(T_)(LmiProperties* x, const LmiString* name, const LmiSharedPtr(T_)* sp, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the shared pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: sp}
		{description: Value of the property of LmiSharedPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
#define Declare_LmiPropertiesSetSharedPtrValue(T_) \
LMI_INLINE_DECLARATION LmiBool LmiPropertiesSetSharedPtrValue(T_)(LmiProperties* x, const LmiString* name, const LmiSharedPtr(T_)* sp, LmiBool settable);

#define Define_LmiPropertiesSetSharedPtrValue(T_) \
LmiBool LmiPropertiesSetSharedPtrValue(T_)(LmiProperties* x, const LmiString* name, const LmiSharedPtr(T_)* sp, LmiBool settable) \
{                                                                       \
	LmiPropertyValue p;                                                 \
	LmiBool r = LMI_FALSE;                                              \
	if (!LmiPropertyValueConstructSharedPtr(T_)(&p, sp, settable)) {    \
		return LMI_FALSE;                                               \
	}                                                                   \
	r = LmiPropertiesSetProperty(x, name, &p);                          \
	LmiPropertyValueDestruct(&p);                                       \
	return r;                                                           \
}

#define LmiPropertiesSetSharedPtrValueCStr(T_) LmiPropertiesFunc_(T_, SetSharedPtrValueCStr)

/**
{function:
	{name: LmiPropertiesSetSharedPtrValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiSharedPtr property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetSharedPtrValueCStr(T_)(LmiProperties* x, const char* name, const LmiSharedPtr(T_)* sp, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the shared pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: sp}
		{description: Value of the property of LmiSharedPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
#define Declare_LmiPropertiesSetSharedPtrValueCStr(T_) \
LMI_INLINE_DECLARATION LmiBool LmiPropertiesSetSharedPtrValueCStr(T_)(LmiProperties* x, const char* name, const LmiSharedPtr(T_)* sp, LmiBool settable);

#define Define_LmiPropertiesSetSharedPtrValueCStr(T_) \
LmiBool LmiPropertiesSetSharedPtrValueCStr(T_)(LmiProperties* x, const char* name, const LmiSharedPtr(T_)* sp, LmiBool settable) \
{                                                                       \
	LmiPropertyValue p;                                                 \
	LmiBool r = LMI_FALSE;                                              \
	if (!LmiPropertyValueConstructSharedPtr(T_)(&p, sp, settable)) {    \
		return LMI_FALSE;                                               \
	}                                                                   \
	r = LmiPropertiesSetPropertyCStr(x, name, &p);                      \
	LmiPropertyValueDestruct(&p);                                       \
	return r;                                                           \
}

//// Add
#define LmiPropertiesAddSharedPtrValue(T_) LmiPropertiesFunc_(T_, AddSharedPtrValue)

/**
{function:
	{name: LmiPropertiesAddSharedPtrValue}
	{parent: LmiProperties}
	{description: Adds LmiSharedPtr property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddSharedPtrValue(T_)(LmiProperties* x, const LmiString* name, const LmiSharedPtr(T_)* sp, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the shared pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: sp}
		{description: Value of the property of LmiSharedPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
#define Declare_LmiPropertiesAddSharedPtrValue(T_) \
LMI_INLINE_DECLARATION LmiBool LmiPropertiesAddSharedPtrValue(T_)(LmiProperties* x, const LmiString* name, const LmiSharedPtr(T_)* sp, LmiBool settable);

#define Define_LmiPropertiesAddSharedPtrValue(T_) \
LmiBool LmiPropertiesAddSharedPtrValue(T_)(LmiProperties* x, const LmiString* name, const LmiSharedPtr(T_)* sp, LmiBool settable) \
{                                                                       \
	LmiPropertyValue p;                                                 \
	LmiBool r = LMI_FALSE;                                              \
	if (!LmiPropertyValueConstructSharedPtr(T_)(&p, sp, settable)) {    \
		return LMI_FALSE;                                               \
	}                                                                   \
	r = LmiPropertiesAddProperty(x, name, &p);                          \
	LmiPropertyValueDestruct(&p);                                       \
	return r;                                                           \
}


#define LmiPropertiesAddSharedPtrValueCStr(T_) LmiPropertiesFunc_(T_, AddSharedPtrValueCStr)

/**
{function:
	{name: LmiPropertiesAddSharedPtrValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiSharedPtr property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddSharedPtrValueCStr(T_)(LmiProperties* x, const char* name, const LmiSharedPtr(T_)* sp, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the shared pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: sp}
		{description: Value of the property of LmiSharedPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
#define Declare_LmiPropertiesAddSharedPtrValueCStr(T_) \
LMI_INLINE_DECLARATION LmiBool LmiPropertiesAddSharedPtrValueCStr(T_)(LmiProperties* x, const char* name, const LmiSharedPtr(T_)* sp, LmiBool settable);

#define Define_LmiPropertiesAddSharedPtrValueCStr(T_) \
LmiBool LmiPropertiesAddSharedPtrValueCStr(T_)(LmiProperties* x, const char* name, const LmiSharedPtr(T_)* sp, LmiBool settable) \
{                                                                       \
	LmiPropertyValue p;                                                 \
	LmiBool r = LMI_FALSE;                                              \
	if (!LmiPropertyValueConstructSharedPtr(T_)(&p, sp, settable)) {    \
		return LMI_FALSE;                                               \
	}                                                                   \
	r = LmiPropertiesAddPropertyCStr(x, name, &p);                      \
	LmiPropertyValueDestruct(&p);                                       \
	return r;                                                           \
}

//// Get
#define LmiPropertiesGetSharedPtrValue(T_) LmiPropertiesFunc_(T_, GetSharedPtrValue)

/**
{function:
	{name: LmiPropertiesGetSharedPtrValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiSharedPtr of a property value.}
	{prototype: const LmiSharedPtr(T_)* LmiPropertiesGetSharedPtrValue(T_)(const LmiProperties* x, const LmiString* name)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the shared pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiSharedPtr of the property value on success, or NULL if the type of the property value is not an LmiSharedPtr or property is not set.}
}
*/
#define Declare_LmiPropertiesGetSharedPtrValue(T_) \
LMI_INLINE_DECLARATION const LmiSharedPtr(T_)* LmiPropertiesGetSharedPtrValue(T_)(const LmiProperties* x, const LmiString* name);

#define Define_LmiPropertiesGetSharedPtrValue(T_) \
const LmiSharedPtr(T_)* LmiPropertiesGetSharedPtrValue(T_)(const LmiProperties* x, const LmiString* name) \
{                                                                           \
	const LmiPropertyValue* p;                                              \
	p = LmiPropertiesGetPropertyConst(x, name);                             \
	if (!p)                                                                 \
    {                                                                       \
		return NULL;                                                        \
	}                                                                       \
	return LmiPropertyValueGetSharedPtr(T_)(p);                             \
}

#define LmiPropertiesGetSharedPtrValueCStr(T_) LmiPropertiesFunc_(T_, GetSharedPtrValueCStr)

/**
{function:
	{name: LmiPropertiesGetSharedPtrValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiSharedPtr of a property value.}
	{prototype: const LmiSharedPtr(T_)* LmiPropertiesGetSharedPtrValueCStr(T_)(const LmiProperties* x, const char* name)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the shared pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiSharedPtr of the property value on success, or NULL if the type of the property value is not an LmiSharedPtr or property is not set.}
}
*/
#define Declare_LmiPropertiesGetSharedPtrValueCStr(T_) \
LMI_INLINE_DECLARATION const LmiSharedPtr(T_)* LmiPropertiesGetSharedPtrValueCStr(T_)(const LmiProperties* x, const char* name);

#define Define_LmiPropertiesGetSharedPtrValueCStr(T_) \
const LmiSharedPtr(T_)* LmiPropertiesGetSharedPtrValueCStr(T_)(const LmiProperties* x, const char* name) \
{                                                                           \
	const LmiPropertyValue* p;                                              \
	p = LmiPropertiesGetPropertyCStrConst(x, name);                         \
	if (!p)                                                                 \
    {                                                                       \
		return NULL;                                                        \
	}                                                                       \
	return LmiPropertyValueGetSharedPtr(T_)(p);                             \
}

#define LmiPropertiesDereferenceSharedPtr(T_) LmiPropertiesFunc_(T_, DereferenceSharedPtr)

/**
{function:
	{name: LmiPropertiesDereferenceSharedPtr}
	{parent: LmiProperties}
	{description: Gets a pointer to a data of an LmiSharedPtr of a property value.}
	{prototype: T_* LmiPropertiesDereferenceSharedPtr(T_)(const LmiProperties* x, const LmiString* name)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the shared pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to a data held in an LmiSharedPtr of the property value on success, or NULL if the type of the property value is not an LmiSharedPtr or property is not set.}
}
*/

#define Declare_LmiPropertiesDereferenceSharedPtr(T_) \
LMI_INLINE_DECLARATION T_* LmiPropertiesDereferenceSharedPtr(T_)(const LmiProperties* x, const char* name);

#define Define_LmiPropertiesDereferenceSharedPtr(T_) \
T_* LmiPropertiesDereferenceSharedPtr(T_)(const LmiProperties* x, const char* name) \
{                                                                           \
	const LmiPropertyValue* p;                                              \
	p = LmiPropertiesGetPropertyCStrConst(x, name);                         \
	if (!p)                                                                 \
    {                                                                       \
		return NULL;                                                        \
	}                                                                       \
    return LmiPropertyValueDereferenceSharedPtr(T_)(p);                     \
}

#define LmiPropertiesDereferenceSharedPtrCStr(T_) LmiPropertiesFunc_(T_, DereferenceSharedPtrCStr)

/**
{function:
	{name: LmiPropertiesDereferenceSharedPtrCStr}
	{parent: LmiProperties}
	{description: Gets a pointer to a data of an LmiSharedPtr of a property value.}
	{prototype: T_* LmiPropertiesDereferenceSharedPtrCStr(T_)(const LmiProperties* x, const char* name)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the shared pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to a data held in an LmiSharedPtr of the property value on success, or NULL if the type of the property value is not an LmiSharedPtr or property is not set.}
}
*/

#define Declare_LmiPropertiesDereferenceSharedPtrCStr(T_) \
LMI_INLINE_DECLARATION T_* LmiPropertiesDereferenceSharedPtrCStr(T_)(const LmiProperties* x, const char* name);

#define Define_LmiPropertiesDereferenceSharedPtrCStr(T_) \
T_* LmiPropertiesDereferenceSharedPtrCStr(T_)(const LmiProperties* x, const char* name) \
{                                                                           \
	const LmiPropertyValue* p;                                              \
	p = LmiPropertiesGetPropertyCStrConst(x, name);                         \
	if (!p)                                                                 \
    {                                                                       \
		return NULL;                                                        \
	}                                                                       \
    return LmiPropertyValueDereferenceSharedPtr(T_)(p);                     \
}

//// Register class

#define Declare_LmiPropertyValueSharedPtrFuncs(T_)  \
    Declare_LmiPropertyValueConstructSharedPtr(T_)  \
    Declare_LmiPropertyValueGetSharedPtr(T_)        \
    Declare_LmiPropertyValueToSharedPtr(T_)         \
    Declare_LmiPropertyValueDereferenceSharedPtr(T_)

#define Declare_LmiPropertiesSharedPtrFuncs(T_)     \
    Declare_LmiPropertiesSetSharedPtrValue(T_)      \
    Declare_LmiPropertiesSetSharedPtrValueCStr(T_)  \
    Declare_LmiPropertiesAddSharedPtrValue(T_)      \
    Declare_LmiPropertiesAddSharedPtrValueCStr(T_)  \
    Declare_LmiPropertiesGetSharedPtrValue(T_)      \
    Declare_LmiPropertiesGetSharedPtrValueCStr(T_)  \
    Declare_LmiPropertiesDereferenceSharedPtr(T_)   \
    Declare_LmiPropertiesDereferenceSharedPtrCStr(T_)

/* Non-inlined LmiProperties functions */
#define Define_LmiPropertyValueSharedPtrFuncs(T_) \
    Define_LmiPropertyValueConstructSharedPtr(T_) \
    Define_LmiPropertyValueToSharedPtr(T_)        \
    Define_LmiPropertyValueGetSharedPtr(T_)       \
    Define_LmiPropertyValueDereferenceSharedPtr(T_)

#define Define_LmiPropertiesSharedPtrFuncs(T_)      \
    Define_LmiPropertiesSetSharedPtrValue(T_)       \
    Define_LmiPropertiesSetSharedPtrValueCStr(T_)   \
    Define_LmiPropertiesAddSharedPtrValue(T_)       \
    Define_LmiPropertiesAddSharedPtrValueCStr(T_)   \
    Define_LmiPropertiesGetSharedPtrValue(T_)       \
    Define_LmiPropertiesGetSharedPtrValueCStr(T_)   \
    Define_LmiPropertiesDereferenceSharedPtr(T_)    \
    Define_LmiPropertiesDereferenceSharedPtrCStr(T_)

#define Declare_LmiPropertiesSharedPtr(T_)      \
    Declare_LmiPropertyValueSharedPtrFuncs(T_)  \
    Declare_LmiPropertiesSharedPtrFuncs(T_)

#define Define_LmiPropertiesSharedPtr(T_)      \
    Define_LmiPropertyValueSharedPtrFuncs(T_)  \
    Define_LmiPropertiesSharedPtrFuncs(T_)

/////////////// Weak pointer macros


/* LmiProperties for weak pointers*/
// LmiPropertyValue

/* Magic MACROs to define functions correctly */

#define LmiPropertyValueConstructWeakPtr(T_) LmiPropertyValueFunc_(T_, ConstructWeakPtr)

/**
{function:
    {name: LmiPropertyValueConstructWeakPtr}
    {parent: LmiPropertyValue}
    {description: Constructs an LmiWeakPtr property value corresponding to a specified pointer.}
    {prototype: LmiPropertyValue* LmiPropertyValueConstructWeakPtr(T_)(LmiPropertyValue * x, const LmiWeakPtr(T_) *wp, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the weak pointer.}}
    {parameter: {name: x} {description: A property value.}}
    {parameter: {name: wp} {description: The LmiWeakPtr to assign to the property value.}}
    {parameter: {name: settable} {description: Defines if property can be overwritten.}}
    {return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
#define Declare_LmiPropertyValueConstructWeakPtr(T_) \
LMI_INLINE_DECLARATION LmiPropertyValue* LmiPropertyValueConstructWeakPtr(T_)(LmiPropertyValue * x, const LmiWeakPtr(T_) *wp, LmiBool settable);

#define Define_LmiPropertyValueConstructWeakPtr(T_) \
LmiPropertyValue *LmiPropertyValueConstructWeakPtr(T_)(LmiPropertyValue *x, const LmiWeakPtr(T_) *wp, LmiBool settable) \
{                                                                           \
    x->settable = settable;                                                 \
    x->type = LMI_PROPERTYVALUE_TYPE_WeakPtr;                               \
    if (LmiStringConstructCStr(&x->ptrName,                                 \
                           LmiQuoteValue(T_),                               \
                           LmiAllocatorGetDefault()) == NULL)         \
        return NULL;                                                        \
    if (LmiWeakVoidPtrConstructFromWeakPtr(T_)(&x->u.wp, wp) == NULL) {     \
        LmiStringDestruct(&x->ptrName);                                     \
        return NULL;                                                        \
    }                                                                       \
    return x;                                                               \
}

#define LmiPropertyValueGetWeakPtr(T_) LmiPropertyValueFunc_(T_, GetWeakPtr)

/**
{function:
	{name: LmiPropertyValueGetWeakPtr}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an LmiWeakPtr of a property value.}
	{prototype: const LmiWeakPtr(T_) *LmiPropertyValueGetWeakPtr(T_)(const LmiPropertyValue *x)}
	{template-parameter:
		{name: T_} {description: The type pointed to by the weak pointer.}}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an LmiWeakPtr of the property value on success, or NULL if the type of the property value is not a correctly typified LmiWeakPtr.}
}
*/
#define Declare_LmiPropertyValueGetWeakPtr(T_) \
LMI_INLINE_DECLARATION const LmiWeakPtr(T_) *LmiPropertyValueGetWeakPtr(T_)(const LmiPropertyValue *x);

#define Define_LmiPropertyValueGetWeakPtr(T_) \
const LmiWeakPtr(T_)* LmiPropertyValueGetWeakPtr(T_)(const LmiPropertyValue* x) \
{                                                           \
    if (x->type != LMI_PROPERTYVALUE_TYPE_WeakPtr ||        \
        !LmiStringEqualCStr(&x->ptrName,                    \
                             LmiQuoteValue(T_))) {          \
        return NULL;                                        \
    }                                                       \
    return (LmiWeakPtr(T_)*)&x->u.wp;                       \
}

#define LmiPropertyValueToWeakPtr(T_) LmiPropertyValueFunc_(T_, ToWeakPtr)

/**
{function:
	{name: LmiPropertyValueToWeakPtr}
	{parent: LmiPropertyValue}
	{description: Converts a property value to a weak pointer.}
	{prototype: LmiWeakPtr(T_) *LmiPropertyValueToWeakPtr(T_)(const LmiPropertyValue *x, LmiWeakPtr(T_) *wp)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the weak pointer.}}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: wp}
		{description: An typified LmiWeakPtr to hold a copy of the property's value. }
	}
	{return: Returns {code:wp} on success, NULL if the type of the property value is not a correctly specifiedor or other error occurs.}
}
*/
#define Declare_LmiPropertyValueToWeakPtr(T_) \
LMI_INLINE_DECLARATION LmiWeakPtr(T_) *LmiPropertyValueToWeakPtr(T_)(const LmiPropertyValue *x, LmiWeakPtr(T_) *wp);

#define Define_LmiPropertyValueToWeakPtr(T_) \
LmiWeakPtr(T_) *LmiPropertyValueToWeakPtr(T_)(const LmiPropertyValue *x, LmiWeakPtr(T_) *wp)\
{                                                                                           \
    if (x->type == LMI_PROPERTYVALUE_TYPE_WeakPtr) {                                        \
        if (!LmiStringEqualCStr(&x->ptrName, LmiQuoteValue(T_))) {                          \
            return NULL;                                                                    \
        }                                                                                   \
        return (LmiWeakPtr(T_)*)LmiWeakPtrAssignFromWeakVoidPtr(T_)(wp, &x->u.wp);          \
    }                                                                                       \
    return NULL;                                                                            \
}

#define LmiPropertyValueDereferenceWeakPtr(T_) LmiPropertyValueFunc_(T_, DereferenceWeakPtr)

/**
{function:
	{name: LmiPropertyValueDereferenceWeakPtr}
	{parent: LmiPropertyValue}
	{description: Gets a pointer to data stored in an LmiWeakPtr of a property value.}
	{prototype: T_ *LmiPropertyValueDereferenceWeakPtr(T_)(const LmiPropertyValue *x)}
	{template-parameter:
		{name: T_} {description: The type pointed to by the weak pointer.}}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to a data stored in an LmiWeakPtr of the property value on success. NULL if the type of the property value is not a correctly typified LmiWeakPtr or if LmiWeakPtr is expired.}
}
*/
#define Declare_LmiPropertyValueDereferenceWeakPtr(T_) \
LMI_INLINE_DECLARATION T_* LmiPropertyValueDereferenceWeakPtr(T_)(const LmiPropertyValue *x);

#define Define_LmiPropertyValueDereferenceWeakPtr(T_) \
T_* LmiPropertyValueDereferenceWeakPtr(T_)(const LmiPropertyValue* x) \
{                                                           \
    LmiSharedPtr(T_) sp;                                    \
    T_ *t;                                                  \
    if (x->type != LMI_PROPERTYVALUE_TYPE_WeakPtr ||        \
        !LmiStringEqualCStr(&x->ptrName,                    \
                             LmiQuoteValue(T_))) {          \
        return NULL;                                        \
    }                                                       \
    if (LmiSharedPtrConstructFromWeakVoidPtr(T_)(&sp,       \
                                &x->u.wp) == NULL) {        \
        return NULL;                                        \
    }                                                       \
    t = LmiSharedPtrGet(T_)(&sp);                           \
    LmiSharedPtrDestruct(T_)(&sp);                          \
    return t;                                               \
}

// LmiProperties

//// Set

#define LmiPropertiesSetWeakPtrValue(T_) LmiPropertiesFunc_(T_, SetWeakPtrValue)

/**
{function:
	{name: LmiPropertiesSetWeakPtrValue}
	{parent: LmiProperties}
	{description: Sets LmiWeakPtr property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetWeakPtrValue(T_)(LmiProperties* x, const LmiString* name, const LmiWeakPtr(T_) *wp, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the weak pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: wp}
		{description: Value of the property of LmiWeakPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
#define Declare_LmiPropertiesSetWeakPtrValue(T_) \
LMI_INLINE_DECLARATION LmiBool LmiPropertiesSetWeakPtrValue(T_)(LmiProperties* x, const LmiString* name, const LmiWeakPtr(T_) *wp, LmiBool settable);

#define Define_LmiPropertiesSetWeakPtrValue(T_) \
LmiBool LmiPropertiesSetWeakPtrValue(T_)(LmiProperties* x, const LmiString* name, const LmiWeakPtr(T_) *wp, LmiBool settable) \
{                                                                       \
	LmiPropertyValue p;                                                 \
	LmiBool r = LMI_FALSE;                                              \
	if (!LmiPropertyValueConstructWeakPtr(T_)(&p, wp, settable)) {      \
		return LMI_FALSE;                                               \
	}                                                                   \
	r = LmiPropertiesSetProperty(x, name, &p);                          \
	LmiPropertyValueDestruct(&p);                                       \
	return r;                                                           \
}

#define LmiPropertiesSetWeakPtrValueCStr(T_) LmiPropertiesFunc_(T_, SetWeakPtrValueCStr)

/**
{function:
	{name: LmiPropertiesSetWeakPtrValueCStr}
	{parent: LmiProperties}
	{description: Sets LmiWeakPtr property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetWeakPtrValueCStr(T_)(LmiProperties* x, const char* name, const LmiWeakPtr(T_) *wp, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the weak pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: wp}
		{description: Value of the property of LmiWeakPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
#define Declare_LmiPropertiesSetWeakPtrValueCStr(T_) \
LMI_INLINE_DECLARATION LmiBool LmiPropertiesSetWeakPtrValueCStr(T_)(LmiProperties* x, const char* name, const LmiWeakPtr(T_) *wp, LmiBool settable);

#define Define_LmiPropertiesSetWeakPtrValueCStr(T_) \
LmiBool LmiPropertiesSetWeakPtrValueCStr(T_)(LmiProperties* x, const char* name, const LmiWeakPtr(T_) *wp, LmiBool settable) \
{                                                                       \
	LmiPropertyValue p;                                                 \
	LmiBool r = LMI_FALSE;                                              \
	if (!LmiPropertyValueConstructWeakPtr(T_)(&p, wp, settable)) {      \
		return LMI_FALSE;                                               \
	}                                                                   \
	r = LmiPropertiesSetPropertyCStr(x, name, &p);                      \
	LmiPropertyValueDestruct(&p);                                       \
	return r;                                                           \
}

//// Add
#define LmiPropertiesAddWeakPtrValue(T_) LmiPropertiesFunc_(T_, AddWeakPtrValue)

/**
{function:
	{name: LmiPropertiesAddWeakPtrValue}
	{parent: LmiProperties}
	{description: Adds LmiWeakPtr property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddWeakPtrValue(T_)(LmiProperties* x, const LmiString* name, const LmiWeakPtr(T_) *wp, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the weak pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: wp}
		{description: Value of the property of LmiWeakPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
#define Declare_LmiPropertiesAddWeakPtrValue(T_) \
LMI_INLINE_DECLARATION LmiBool LmiPropertiesAddWeakPtrValue(T_)(LmiProperties* x, const LmiString* name, const LmiWeakPtr(T_) *wp, LmiBool settable);

#define Define_LmiPropertiesAddWeakPtrValue(T_) \
LmiBool LmiPropertiesAddWeakPtrValue(T_)(LmiProperties* x, const LmiString* name, const LmiWeakPtr(T_) *wp, LmiBool settable) \
{                                                                       \
	LmiPropertyValue p;                                                 \
	LmiBool r = LMI_FALSE;                                              \
	if (!LmiPropertyValueConstructWeakPtr(T_)(&p, wp, settable)) {      \
		return LMI_FALSE;                                               \
	}                                                                   \
	r = LmiPropertiesAddProperty(x, name, &p);                          \
	LmiPropertyValueDestruct(&p);                                       \
	return r;                                                           \
}


#define LmiPropertiesAddWeakPtrValueCStr(T_) LmiPropertiesFunc_(T_, AddWeakPtrValueCStr)

/**
{function:
	{name: LmiPropertiesAddWeakPtrValueCStr}
	{parent: LmiProperties}
	{description: Adds LmiWeakPtr property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddWeakPtrValueCStr(T_)(LmiProperties* x, const char* name, const LmiWeakPtr(T_) *wp, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the weak pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: wp}
		{description: Value of the property of LmiWeakPtr type.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
#define Declare_LmiPropertiesAddWeakPtrValueCStr(T_) \
LMI_INLINE_DECLARATION LmiBool LmiPropertiesAddWeakPtrValueCStr(T_)(LmiProperties* x, const char* name, const LmiWeakPtr(T_) *wp, LmiBool settable);

#define Define_LmiPropertiesAddWeakPtrValueCStr(T_) \
LmiBool LmiPropertiesAddWeakPtrValueCStr(T_)(LmiProperties* x, const char* name, const LmiWeakPtr(T_) *wp, LmiBool settable) \
{                                                                       \
	LmiPropertyValue p;                                                 \
	LmiBool r = LMI_FALSE;                                              \
	if (!LmiPropertyValueConstructWeakPtr(T_)(&p, wp, settable)) {      \
		return LMI_FALSE;                                               \
	}                                                                   \
	r = LmiPropertiesAddPropertyCStr(x, name, &p);                      \
	LmiPropertyValueDestruct(&p);                                       \
	return r;                                                           \
}

//// Get
#define LmiPropertiesGetWeakPtrValue(T_) LmiPropertiesFunc_(T_, GetWeakPtrValue)

/**
{function:
	{name: LmiPropertiesGetWeakPtrValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiWeakPtr of a property value.}
	{prototype: const LmiWeakPtr(T_)* LmiPropertiesGetWeakPtrValue(T_)(const LmiProperties* x, const LmiString* name)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the weak pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiWeakPtr of the property value on success, or NULL if the type of the property value is not an LmiWeakPtr or property is not set.}
}
*/
#define Declare_LmiPropertiesGetWeakPtrValue(T_) \
LMI_INLINE_DECLARATION const LmiWeakPtr(T_)* LmiPropertiesGetWeakPtrValue(T_)(const LmiProperties* x, const LmiString* name);

#define Define_LmiPropertiesGetWeakPtrValue(T_) \
const LmiWeakPtr(T_)* LmiPropertiesGetWeakPtrValue(T_)(const LmiProperties* x, const LmiString* name) \
{                                                                           \
	const LmiPropertyValue* p;                                              \
	p = LmiPropertiesGetPropertyConst(x, name);                             \
	if (!p)                                                                 \
    {                                                                       \
		return NULL;                                                        \
	}                                                                       \
	return LmiPropertyValueGetWeakPtr(T_)(p);                               \
}

#define LmiPropertiesGetWeakPtrValueCStr(T_) LmiPropertiesFunc_(T_, GetWeakPtrValueCStr)

/**
{function:
	{name: LmiPropertiesGetWeakPtrValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to an LmiWeakPtr of a property value.}
	{prototype: const LmiWeakPtr(T_)* LmiPropertiesGetWeakPtrValueCStr(T_)(const LmiProperties* x, const char* name)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the weak pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to an LmiWeakPtr of the property value on success, or NULL if the type of the property value is not an LmiWeakPtr or property is not set.}
}
*/
#define Declare_LmiPropertiesGetWeakPtrValueCStr(T_) \
LMI_INLINE_DECLARATION const LmiWeakPtr(T_)* LmiPropertiesGetWeakPtrValueCStr(T_)(const LmiProperties* x, const char* name);

#define Define_LmiPropertiesGetWeakPtrValueCStr(T_) \
const LmiWeakPtr(T_)* LmiPropertiesGetWeakPtrValueCStr(T_)(const LmiProperties* x, const char* name) \
{                                                                           \
	const LmiPropertyValue* p;                                              \
	p = LmiPropertiesGetPropertyCStrConst(x, name);                         \
	if (!p)                                                                 \
    {                                                                       \
		return NULL;                                                        \
	}                                                                       \
	return LmiPropertyValueGetWeakPtr(T_)(p);                               \
}

#define LmiPropertiesDereferenceWeakPtr(T_) LmiPropertiesFunc_(T_, DereferenceWeakPtr)

/**
{function:
	{name: LmiPropertiesDereferenceWeakPtr}
	{parent: LmiProperties}
	{description: Gets a pointer to a data of an LmiWeakPtr of a property value.}
	{prototype: T_* LmiPropertiesDereferenceWeakPtr(T_)(const LmiProperties* x, const LmiString* name)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the weak pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to a data stored in an LmiWeakPtr of the property value on success. NULL if the type of the property value is not a correctly typified LmiWeakPtr or if LmiWeakPtr is expired.}
}
*/
#define Declare_LmiPropertiesDereferenceWeakPtr(T_) \
LMI_INLINE_DECLARATION T_* LmiPropertiesDereferenceWeakPtr(T_)(const LmiProperties* x, const LmiString* name);

#define Define_LmiPropertiesDereferenceWeakPtr(T_) \
T_* LmiPropertiesDereferenceWeakPtr(T_)(const LmiProperties* x, const LmiString* name) \
{                                                                           \
	const LmiPropertyValue* p;                                              \
	p = LmiPropertiesGetPropertyConst(x, name);                             \
	if (!p)                                                                 \
    {                                                                       \
		return NULL;                                                        \
	}                                                                       \
	return LmiPropertyValueDereferenceWeakPtr(T_)(p);                       \
}

#define LmiPropertiesDereferenceWeakPtrCStr(T_) LmiPropertiesFunc_(T_, DereferenceWeakPtrCStr)

/**
{function:
	{name: LmiPropertiesDereferenceWeakPtrCStr}
	{parent: LmiProperties}
	{description: Gets a pointer to a data of an LmiWeakPtr of a property value.}
	{prototype: T_* LmiPropertiesDereferenceWeakPtrCStr(T_)(const LmiProperties* x, const char* name)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the weak pointer.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to a data stored in an LmiWeakPtr of the property value on success. NULL if the type of the property value is not a correctly typified LmiWeakPtr or if LmiWeakPtr is expired.}
}
*/
#define Declare_LmiPropertiesDereferenceWeakPtrCStr(T_) \
LMI_INLINE_DECLARATION T_* LmiPropertiesDereferenceWeakPtrCStr(T_)(const LmiProperties* x, const char* name);

#define Define_LmiPropertiesDereferenceWeakPtrCStr(T_) \
T_* LmiPropertiesDereferenceWeakPtrCStr(T_)(const LmiProperties* x, const char* name) \
{                                                                           \
	const LmiPropertyValue* p;                                              \
	p = LmiPropertiesGetPropertyCStrConst(x, name);                         \
	if (!p)                                                                 \
    {                                                                       \
		return NULL;                                                        \
	}                                                                       \
	return LmiPropertyValueDereferenceWeakPtr(T_)(p);                       \
}

//// Register class

#define Declare_LmiPropertyValueWeakPtrFuncs(T_)  \
    Declare_LmiPropertyValueConstructWeakPtr(T_)  \
    Declare_LmiPropertyValueGetWeakPtr(T_)        \
    Declare_LmiPropertyValueToWeakPtr(T_)         \
    Declare_LmiPropertyValueDereferenceWeakPtr(T_)

#define Declare_LmiPropertiesWeakPtrFuncs(T_)     \
    Declare_LmiPropertiesSetWeakPtrValue(T_)      \
    Declare_LmiPropertiesSetWeakPtrValueCStr(T_)  \
    Declare_LmiPropertiesAddWeakPtrValue(T_)      \
    Declare_LmiPropertiesAddWeakPtrValueCStr(T_)  \
    Declare_LmiPropertiesGetWeakPtrValue(T_)      \
    Declare_LmiPropertiesGetWeakPtrValueCStr(T_)  \
    Declare_LmiPropertiesDereferenceWeakPtr(T_)   \
    Declare_LmiPropertiesDereferenceWeakPtrCStr(T_)

/* Non-inlined LmiProperties functions */
#define Define_LmiPropertyValueWeakPtrFuncs(T_) \
    Define_LmiPropertyValueConstructWeakPtr(T_) \
    Define_LmiPropertyValueToWeakPtr(T_)        \
    Define_LmiPropertyValueGetWeakPtr(T_)       \
    Define_LmiPropertyValueDereferenceWeakPtr(T_)

#define Define_LmiPropertiesWeakPtrFuncs(T_)      \
    Define_LmiPropertiesSetWeakPtrValue(T_)       \
    Define_LmiPropertiesSetWeakPtrValueCStr(T_)   \
    Define_LmiPropertiesAddWeakPtrValue(T_)       \
    Define_LmiPropertiesAddWeakPtrValueCStr(T_)   \
    Define_LmiPropertiesGetWeakPtrValue(T_)       \
    Define_LmiPropertiesGetWeakPtrValueCStr(T_)   \
    Define_LmiPropertiesDereferenceWeakPtr(T_)    \
    Define_LmiPropertiesDereferenceWeakPtrCStr(T_)

#define Declare_LmiPropertiesWeakPtr(T_)      \
    Declare_LmiPropertyValueWeakPtrFuncs(T_)  \
    Declare_LmiPropertiesWeakPtrFuncs(T_)

#define Define_LmiPropertiesWeakPtr(T_)      \
    Define_LmiPropertyValueWeakPtrFuncs(T_)  \
    Define_LmiPropertiesWeakPtrFuncs(T_)

// Any type wrapper

#define LmiPropertyValueConstruct(T_) LmiPropertyValueFunc_(T_, Construct)

/**
{function:
    {name: LmiPropertyValueConstruct}
    {parent: LmiPropertyValue}
    {description: Constructs an property value corresponding to a specified type.}
    {prototype: LmiPropertyValue* LmiPropertyValueConstruct(T_)(LmiPropertyValue * x, const T_ *val, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The pointed type to be stored.}}
    {parameter: {name: x} {description: A property value.}}
    {parameter: {name: val} {description: The value to assign to the property value.}}
    {parameter: {name: settable} {description: Defines if property value can be overwritten.}}
    {return: Returns a pointer to the constructed property value or NULL on error.}
}
*/
#define Declare_LmiPropertyValueConstruct(T_) \
LMI_INLINE_DECLARATION LmiPropertyValue* LmiPropertyValueConstruct(T_)(LmiPropertyValue * x, const T_ *val, LmiBool settable);

#define Define_LmiPropertyValueConstruct(T_) \
LmiPropertyValue *LmiPropertyValueConstruct(T_)(LmiPropertyValue *x, const T_ *val, LmiBool settable) \
{                                                                                   \
    x->settable = settable;                                                         \
    x->type = LMI_PROPERTYVALUE_TYPE_Wrapper;                                       \
    x->size = sizeof(T_);                                                           \
    if (LmiStringConstructCStr(&x->ptrName,                                         \
                           LmiQuoteValue(T_),                                       \
                           LmiAllocatorGetDefault()) == NULL)                 		\
        return NULL;                                                                \
    x->u.vp = LmiAllocatorAllocate(LmiAllocatorGetDefault(), sizeof(T_));     		\
    if (T_ ## ConstructCopy((T_*)x->u.vp, val) == NULL) {							\
        LmiAllocatorDeallocate(LmiAllocatorGetDefault(),                      		\
                                   x->u.vp, sizeof(T_));							\
        LmiStringDestruct(&x->ptrName);                                             \
        return NULL;                                                                \
    }                                                                               \
    x->callbacks = &LmiPropertyValueCallbacks_ ## T_;                               \
    return x;                                                                       \
}

#define LmiPropertyValueGet(T_) LmiPropertyValueFunc_(T_, Get)

/**
{function:
	{name: LmiPropertyValueGet}
	{parent: LmiPropertyValue}
	{description: Gets a const pointer to an specific type of a property value.}
	{prototype: const T_ *LmiPropertyValueGet(T_)(const LmiPropertyValue *x)}
	{template-parameter:
		{name: T_} {description: The associated type.}}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{return: Returns a pointer to an associated type of the property value on success, or NULL if the type of the property value is not a correctly typified.}
}
*/
#define Declare_LmiPropertyValueGet(T_) \
LMI_INLINE_DECLARATION const T_ *LmiPropertyValueGet(T_)(const LmiPropertyValue *x);

#define Define_LmiPropertyValueGet(T_) \
const T_* LmiPropertyValueGet(T_)(const LmiPropertyValue* x)    \
{                                                               \
    if (x->type != LMI_PROPERTYVALUE_TYPE_Wrapper) {            \
        return NULL;                                            \
    }                                                           \
    if (!LmiStringEqualCStr(&x->ptrName,                        \
                           LmiQuoteValue(T_))) {                \
        return NULL;                                            \
    }                                                           \
    return (T_*)x->u.vp;                                        \
}

#define LmiPropertyValueTo(T_) LmiPropertyValueFunc_(T_, To)

/**
{function:
	{name: LmiPropertyValueTo}
	{parent: LmiPropertyValue}
	{description: Converts a property value to a pointer.}
	{prototype: T_ *LmiPropertyValueTo(T_)(const LmiPropertyValue *x, T_ *p)}
    {template-parameter:
        {name: T_} {description: The pointed type.}}
	{parameter:
		{name: x}
		{description: A property value.}
	}
	{parameter:
		{name: p}
		{description: A pointer to hold a copy of the property's value. }
	}
	{return: Returns {code:p} on success, or NULL if the type of the property value is not a correctly typified or any other error occurs.}
}
*/
#define Declare_LmiPropertyValueTo(T_) \
LMI_INLINE_DECLARATION T_ *LmiPropertyValueTo(T_)(const LmiPropertyValue *x, T_ *p);

#define Define_LmiPropertyValueTo(T_) \
T_ *LmiPropertyValueTo(T_)(const LmiPropertyValue *x, T_ *p)        \
{                                                                   \
    if (x->type == LMI_PROPERTYVALUE_TYPE_Wrapper) {                \
        if (!LmiStringEqualCStr(&x->ptrName, LmiQuoteValue(T_))) {  \
            return NULL;                                            \
        }                                                           \
        return LmiAssign(T_)(p, (T_*)x->u.vp);                      \
    }                                                               \
    return NULL;                                                    \
}

// LmiProperties

#define LmiPropertiesFunc_(T_,Func_) LmiTemplate1(LmiClassFunc(LmiProperties, Func_), T_)

//// Set

#define LmiPropertiesSetValue(T_) LmiPropertiesFunc_(T_, SetValue)

/**
{function:
	{name: LmiPropertiesSetValue}
	{parent: LmiProperties}
	{description: Sets property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetValue(T_)(LmiProperties* x, const LmiString* name, const T_* p, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The type to store.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: p}
		{description: Pointer to the value.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
#define Declare_LmiPropertiesSetValue(T_) \
LMI_INLINE_DECLARATION LmiBool LmiPropertiesSetValue(T_)(LmiProperties* x, const LmiString* name, const T_* p, LmiBool settable);

#define Define_LmiPropertiesSetValue(T_) \
LmiBool LmiPropertiesSetValue(T_)(LmiProperties* x, const LmiString* name, const T_* p, LmiBool settable) \
{                                                           \
	LmiPropertyValue pv;                                    \
	LmiBool r = LMI_FALSE;                                  \
	if (!LmiPropertyValueConstruct(T_)(&pv, p, settable)) { \
		return LMI_FALSE;                                   \
	}                                                       \
	r = LmiPropertiesSetProperty(x, name, &pv);             \
	LmiPropertyValueDestruct(&pv);                          \
	return r;                                               \
}

#define LmiPropertiesSetValueCStr(T_) LmiPropertiesFunc_(T_, SetValueCStr)

/**
{function:
	{name: LmiPropertiesSetValueCStr}
	{parent: LmiProperties}
	{description: Sets property value to the specified path.}
	{prototype: LmiBool LmiPropertiesSetValueCStr(T_)(LmiProperties* x, const char* name, const T_* p, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The type to store.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: p}
		{description: Pointer to the value.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully set or LMI_FALSE otherwise.}
}
*/
#define Declare_LmiPropertiesSetValueCStr(T_) \
LMI_INLINE_DECLARATION LmiBool LmiPropertiesSetValueCStr(T_)(LmiProperties* x, const char* name, const T_* p, LmiBool settable);

#define Define_LmiPropertiesSetValueCStr(T_) \
LmiBool LmiPropertiesSetValueCStr(T_)(LmiProperties* x, const char* name, const T_* p, LmiBool settable) \
{                                                           \
	LmiPropertyValue pv;                                    \
	LmiBool r = LMI_FALSE;                                  \
	if (!LmiPropertyValueConstruct(T_)(&pv, p, settable)) { \
		return LMI_FALSE;                                   \
	}                                                       \
	r = LmiPropertiesSetPropertyCStr(x, name, &pv);         \
	LmiPropertyValueDestruct(&pv);                          \
	return r;                                               \
}

//// Add
#define LmiPropertiesAddValue(T_) LmiPropertiesFunc_(T_, AddValue)

/**
{function:
	{name: LmiPropertiesAddValue}
	{parent: LmiProperties}
	{description: Adds a property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddValue(T_)(LmiProperties* x, const LmiString* name, const T_* p, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The type to store.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: p}
		{description: Pointer to the value.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
#define Declare_LmiPropertiesAddValue(T_) \
LMI_INLINE_DECLARATION LmiBool LmiPropertiesAddValue(T_)(LmiProperties* x, const LmiString* name, const T_* p, LmiBool settable);

#define Define_LmiPropertiesAddValue(T_) \
LmiBool LmiPropertiesAddValue(T_)(LmiProperties* x, const LmiString* name, const T_* p, LmiBool settable) \
{                                                           \
	LmiPropertyValue pv;                                    \
	LmiBool r = LMI_FALSE;                                  \
	if (!LmiPropertyValueConstruct(T_)(&pv, p, settable)) { \
		return LMI_FALSE;                                   \
	}                                                       \
	r = LmiPropertiesAddProperty(x, name, &pv);             \
	LmiPropertyValueDestruct(&pv);                          \
	return r;                                               \
}


#define LmiPropertiesAddValueCStr(T_) LmiPropertiesFunc_(T_, AddValueCStr)

/**
{function:
	{name: LmiPropertiesAddValueCStr}
	{parent: LmiProperties}
	{description: Adds a property value to the specified path. See LmiPropertiesAddProperty for details on behavior.}
	{prototype: LmiBool LmiPropertiesAddValueCStr(T_)(LmiProperties* x, const char* name, const T_* p, LmiBool settable)}
    {template-parameter:
        {name: T_} {description: The type to store.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{parameter:
		{name: p}
		{description: Pointer to the value.}
	}
	{parameter:
		{name: settable}
		{description: Defines if property can be overwritten.}
	}
	{return: Returns LMI_TRUE if the property value was successfully added or LMI_FALSE on failure or if property exists.}
}
*/
#define Declare_LmiPropertiesAddValueCStr(T_) \
LMI_INLINE_DECLARATION LmiBool LmiPropertiesAddValueCStr(T_)(LmiProperties* x, const char* name, const T_* p, LmiBool settable);

#define Define_LmiPropertiesAddValueCStr(T_) \
LmiBool LmiPropertiesAddValueCStr(T_)(LmiProperties* x, const char* name, const T_* p, LmiBool settable) \
{                                                           \
	LmiPropertyValue pv;                                    \
	LmiBool r = LMI_FALSE;                                  \
	if (!LmiPropertyValueConstruct(T_)(&pv, p, settable)) { \
		return LMI_FALSE;                                   \
	}                                                       \
	r = LmiPropertiesAddPropertyCStr(x, name, &pv);         \
	LmiPropertyValueDestruct(&pv);                          \
	return r;                                               \
}

//// Get
#define LmiPropertiesGetValue(T_) LmiPropertiesFunc_(T_, GetValue)

/**
{function:
	{name: LmiPropertiesGetValue}
	{parent: LmiProperties}
	{description: Gets a const pointer to a value of a property value.}
	{prototype: const T_* LmiPropertiesGetValue(T_)(const LmiProperties* x, const LmiString* name)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the stored object.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to a value of the property value on success, or NULL if the type of the property value is not a correct one or property is not set.}
}
*/
#define Declare_LmiPropertiesGetValue(T_) \
LMI_INLINE_DECLARATION const T_* LmiPropertiesGetValue(T_)(const LmiProperties* x, const LmiString* name);

#define Define_LmiPropertiesGetValue(T_) \
const T_* LmiPropertiesGetValue(T_)(const LmiProperties* x, const LmiString* name) \
{                                                                           \
	const LmiPropertyValue* p;                                              \
	p = LmiPropertiesGetPropertyConst(x, name);                             \
	if (!p)                                                                 \
    {                                                                       \
		return NULL;                                                        \
	}                                                                       \
	return LmiPropertyValueGet(T_)(p);                                      \
}

#define LmiPropertiesGetValueCStr(T_) LmiPropertiesFunc_(T_, GetValueCStr)

/**
{function:
	{name: LmiPropertiesGetValueCStr}
	{parent: LmiProperties}
	{description: Gets a const pointer to a value of a property value.}
	{prototype: const T_* LmiPropertiesGetValueCStr(T_)(const LmiProperties* x, const char* name)}
    {template-parameter:
        {name: T_} {description: The type pointed to by the stored object.}}
	{parameter:
		{name: x}
		{description: A properties object.}
	}
	{parameter:
		{name: name}
		{description: The name or path to the property.  Paths are used to locate a property in a hierarchy of nested properties.  Paths are 
 		  specified using dotted notation (i.e. parent.child) where each component of the path is delimited by a dot ('.'). Path delimiters can 
 		  be escaped with a backslash ('\\'). For example, "file\\.txt" specifies a simple name rather than a path.  The elements of a property 
		  of type vector can be indexed by enclosing the index in square brackets.  For example, parent.child[5].}
	}
	{return: Returns a pointer to a value of the property value on success, or NULL if the type of the property value is not a correct one or property is not set.}
}
*/
#define Declare_LmiPropertiesGetValueCStr(T_) \
LMI_INLINE_DECLARATION const T_* LmiPropertiesGetValueCStr(T_)(const LmiProperties* x, const char* name);

#define Define_LmiPropertiesGetValueCStr(T_) \
const T_* LmiPropertiesGetValueCStr(T_)(const LmiProperties* x, const char* name) \
{                                                                           \
	const LmiPropertyValue* p;                                              \
	p = LmiPropertiesGetPropertyCStrConst(x, name);                         \
	if (!p)                                                                 \
    {                                                                       \
		return NULL;                                                        \
	}                                                                       \
	return LmiPropertyValueGet(T_)(p);                                      \
}


#define Declare_LmiPropertiesValueCallbacks(T_)

#define Define_LmiPropertiesValueCallbacks(T_) \
    LmiPropertyValueCallbacks LmiPropertyValueCallbacks_ ## T_ = {                  \
        .assign     = (LmiPropertyValueWrapperAssign)          T_ ## Assign,        \
        .copy       = (LmiPropertyValueWrapperConstructCopy)   T_ ## ConstructCopy, \
        .destruct   = (LmiPropertyValueWrapperDestruct)        T_ ## Destruct,      \
        .equal      = (LmiPropertyValueWrapperEqual)           T_ ## Equal          \
    };                                                                              \

//// Register class

#define Declare_LmiPropertyValueFuncs(T_)   \
    Declare_LmiPropertyValueConstruct(T_)   \
    Declare_LmiPropertyValueGet(T_)         \
    Declare_LmiPropertyValueTo(T_)

#define Declare_LmiPropertiesFuncs(T_)     \
    Declare_LmiPropertiesSetValue(T_)      \
    Declare_LmiPropertiesSetValueCStr(T_)  \
    Declare_LmiPropertiesAddValue(T_)      \
    Declare_LmiPropertiesAddValueCStr(T_)  \
    Declare_LmiPropertiesGetValue(T_)      \
    Declare_LmiPropertiesGetValueCStr(T_)

/* Non-inlined LmiProperties functions */
#define Define_LmiPropertyValueFuncs(T_) \
    Define_LmiPropertyValueConstruct(T_) \
    Define_LmiPropertyValueGet(T_)       \
    Define_LmiPropertyValueTo(T_)

#define Define_LmiPropertiesFuncs(T_)      \
    Define_LmiPropertiesSetValue(T_)       \
    Define_LmiPropertiesSetValueCStr(T_)   \
    Define_LmiPropertiesAddValue(T_)       \
    Define_LmiPropertiesAddValueCStr(T_)   \
    Define_LmiPropertiesGetValue(T_)       \
    Define_LmiPropertiesGetValueCStr(T_)

#define Declare_LmiProperties(T_)           \
    Declare_LmiPropertiesValueCallbacks(T_) \
    Declare_LmiPropertyValueFuncs(T_)       \
    Declare_LmiPropertiesFuncs(T_)

#define Define_LmiProperties(T_)            \
    Define_LmiPropertiesValueCallbacks(T_)  \
    Define_LmiPropertyValueFuncs(T_)        \
    Define_LmiPropertiesFuncs(T_)

LMI_END_EXTERN_C

#endif

