/**
{file:
    {name: LmiSleepInhibitor.h}
    {description: Interface to the LmiSystemSleepInhibitor and LmiDisplaySleepInhibitor classes.  These classes allow an application to prevent
 	  the system\'s and display\'s idle-timers from triggering a system/display sleep.}
    {copyright:
    	(c) 2018 Vidyo, Inc.,
    	433 Hackensack Avenue,
    	Hackensack, NJ  07601.

    	All rights reserved.

    	The information contained herein is proprietary to Vidyo, Inc.
    	and shall not be reproduced, copied (in whole or in part), adapted,
    	modified, disseminated, transmitted, transcribed, stored in a retrieval
    	system, or translated into any language in any form by any means
    	without the express written consent of Vidyo, Inc.
    	                  ***** CONFIDENTIAL *****
    }
}
*/
#ifndef LMI_SLEEPINHIBITOR_H_
#define LMI_SLEEPINHIBITOR_H_

#include <Lmi/Utils/LmiTypes.h>
#include <Lmi/Utils/LmiAllocator.h>

/**
{type visibility="public":
	{name: LmiSystemSleepInhibitor}
 	{parent: Os}
 	{include: Lmi/Os/LmiSleepInhibitor.h}
 	{description: Allow an application to prevent the system\'s idle-timer from triggering a system sleep.}
}
*/
typedef struct {
	LmiAllocator* alloc;
	void* implementation;
} LmiSystemSleepInhibitor;

LMI_BEGIN_EXTERN_C

/**
{function visibility="public":
	{name: LmiSystemSleepInhibitorConstructDefault}
	{parent: LmiSystemSleepInhibitor}
 	{description: Construct a system sleep inhibitor. From construction to the destruction of the system sleep inhibitor the systems\'s idle-timer is prevented from triggering a system sleep.}
	{prototype: LmiSystemSleepInhibitor* LmiSystemSleepInhibitorConstructDefault(LmiSystemSleepInhibitor* x, LmiAllocator* alloc)}
	{parameter:
		{name: x}
		{description: A system sleep inhibitor object to construct.}
	}
	{parameter:
		{name: alloc}
		{description: An allocator.}
	}
	{return: A pointer to the constructed system sleep inhibitor object or NULL on failure.}
}
*/
LmiSystemSleepInhibitor* LmiSystemSleepInhibitorConstructDefault(LmiSystemSleepInhibitor* x, LmiAllocator* alloc);

/**
{function visibility="public":
	{name: LmiSystemSleepInhibitorDestruct}
	{parent: LmiSystemSleepInhibitor}
	{description: Destruct a system sleep inhibitor object.  When all system sleep inhibitors have been destructed, the system\'s idle-timer will once again trigger system sleep after the system\'s idle-timer timeout period.}
	{prototype: void LmiSystemSleepInhibitorDestruct(LmiSystemSleepInhibitor* x)}
	{parameter:
		{name: x}
		{description: A pointer to the system sleep inhibitor object to destruct.}
	}
}
*/
void LmiSystemSleepInhibitorDestruct(LmiSystemSleepInhibitor* x);

/* Private OS dependent interface */
LmiSystemSleepInhibitor* LmiSystemSleepInhibitorConstructDefault_(LmiSystemSleepInhibitor* x, LmiAllocator* alloc);
void LmiSystemSleepInhibitorDestruct_(LmiSystemSleepInhibitor* x);

LMI_END_EXTERN_C

/**
{type visibility="public":
	{name: LmiDisplaySleepInhibitor}
	{parent: Os}
 	{include: Lmi/Os/LmiSleepInhibitor.h}
	{description: Allow an application to prevent the display\'s idle-timer from triggering a display sleep.}
}
*/
typedef struct {
	LmiAllocator* alloc;
	void* implementation;
} LmiDisplaySleepInhibitor;

LMI_BEGIN_EXTERN_C
/**
{function visibility="public":
	{name: LmiDisplaySleepInhibitorConstructDefault}
	{parent: LmiDisplaySleepInhibitor}
 	{description: Construct a display sleep inhibitor. From construction to the destruction of the display sleep inhibitor the display\'s idle-timer is prevented from triggering a display sleep.}
	{prototype: LmiDisplaySleepInhibitor* LmiDisplaySleepInhibitorConstructDefault(LmiDisplaySleepInhibitor* x, LmiAllocator* alloc)}
	{parameter:
		{name: x}
		{description: A display sleep inhibitor object to construct.}
	}
	{parameter:
		{name: alloc}
		{description: An allocator.}
	}
	{return: A pointer to the constructed display sleep inhibitor object or NULL on failure.}
}
*/
LmiDisplaySleepInhibitor* LmiDisplaySleepInhibitorConstructDefault(LmiDisplaySleepInhibitor* x, LmiAllocator* alloc);

/**
{function visibility="public":
	{name: LmiDisplaySleepInhibitorDestruct}
	{parent: LmiDisplaySleepInhibitor}
	{description: Destruct a display sleep inhibitor object.  When all display sleep inhibitors have been destructed, the display\'s idle-timer will once again trigger display sleep after the display\'s idle-timer timeout period.}
	{prototype: void LmiDisplaySleepInhibitorDestruct(LmiDisplaySleepInhibitor* x)}
	{parameter:
		{name: x}
		{description: A pointer to the display sleep inhibitor object to destruct.}
	}
}
*/
void LmiDisplaySleepInhibitorDestruct(LmiDisplaySleepInhibitor* x);

LmiBool LmiSleepInhibitorInitialize(void);
void LmiSleepInhibitorUninitialize(void);

/* Private OS dependent interface */
LmiBool LmiSleepInhibitorInitialize_(void);
void LmiSleepInhibitorUninitialize_(void);
LmiDisplaySleepInhibitor* LmiDisplaySleepInhibitorConstructDefault_(LmiDisplaySleepInhibitor* x, LmiAllocator* alloc);
void LmiDisplaySleepInhibitorDestruct_(LmiDisplaySleepInhibitor* x);

LMI_END_EXTERN_C

#endif /* LMI_SLEEPINHIBITOR_H_ */
