/**
{file:
	{name: VidyoCameraControlCapabilities.h}
	{description: The camera control capabilities module.}
	{copyright:
		(c) 2006-2021 Vidyo, Inc.,
		433 Hackensack Avenue,
		Hackensack, NJ  07601.

		All rights reserved.

		The information contained herein is proprietary to Vidyo, Inc.
		and shall not be reproduced, copied (in whole or in part), adapted,
		modified, disseminated, transmitted, transcribed, stored in a retrieval
		system, or translated into any language in any form by any means
		without the express written consent of Vidyo, Inc.}
}
*/

#ifndef VIDYO_CAMERACONTROLCAPABILITIES_H_
#define VIDYO_CAMERACONTROLCAPABILITIES_H_

#include <Lmi/Utils/LmiTypes.h>
#include <Lmi/Utils/LmiVector.h>
#include <Lmi/Utils/LmiAllocator.h>
#include <Lmi/Video/Common/LmiCameraControl.h>


LMI_BEGIN_EXTERN_C

/**
{type:
    {name: VidyoCameraControlDirection}
    {parent: VidyoCameraControlCapabilities}
    {description: The camera control direction.}
    {value: {name: VIDYO_CAMERACONTROLDIRECTION_PanLeft} {description: }}
    {value: {name: VIDYO_CAMERACONTROLDIRECTION_PanRight} {description: }}
    {value: {name: VIDYO_CAMERACONTROLDIRECTION_TiltUp} {description: }}
    {value: {name: VIDYO_CAMERACONTROLDIRECTION_TiltDown} {description: }}
    {value: {name: VIDYO_CAMERACONTROLDIRECTION_ZoomIn} {description: }}
    {value: {name: VIDYO_CAMERACONTROLDIRECTION_ZoomOut} {description: }}
    {value: {name: VIDYO_CAMERACONTROLDIRECTION_None} {description: }}
}
*/
typedef enum {
    VIDYO_CAMERACONTROLDIRECTION_PanLeft,
    VIDYO_CAMERACONTROLDIRECTION_PanRight,
    VIDYO_CAMERACONTROLDIRECTION_TiltUp,
    VIDYO_CAMERACONTROLDIRECTION_TiltDown,
    VIDYO_CAMERACONTROLDIRECTION_ZoomIn,
    VIDYO_CAMERACONTROLDIRECTION_ZoomOut,
    VIDYO_CAMERACONTROLDIRECTION_None
} VidyoCameraControlDirection;

/**
{type visibility="public":
    {name: VidyoCameraControlAction}
    {parent: VidyoCameraControlCapabilities}
    {description: Represents the enumeration of camera control action.}
    {value: {name: VIDYO_CAMERACONTROLACTION_Start} {description: Start control action for continuous move.}}
    {value: {name: VIDYO_CAMERACONTROLACTION_Stop} {description: Stop contraol action for stopping the move.}}
    {value: {name: VIDYO_CAMERACONTROLACTION_Nudge} {description: Nudge control action for move the camera to a direction.}}
    {value: {name: VIDYO_CAMERACONTROLACTION_Visca} {description: Visca contraol action for sending visca command.}}
    {value: {name: VIDYO_CAMERACONTROLACTION_Preset} {description: Preset control action for sending the preset index.}}
}
*/
typedef enum {
    VIDYO_CAMERACONTROLACTION_Start,
    VIDYO_CAMERACONTROLACTION_Stop,
    VIDYO_CAMERACONTROLACTION_Nudge,
    VIDYO_CAMERACONTROLACTION_Visca,
    VIDYO_CAMERACONTROLACTION_Preset
} VidyoCameraControlAction;

/**
{type:
	{name: VidyoCameraPreset}
	{parent: Device}
	{description: Represents camera preset structure.}
	{member:
		{name: index}
		{type: LmiUint}
		{description: Represents preset index.}
	}
	{member:
		{name: name}
		{type: LmiString}
		{description: Represents friendly name associate with preset index.}
	}
}
*/
typedef struct
{
	LmiUint index;
	LmiString name;
} VidyoCameraPreset;

/**
{type:
    {name: VidyoCameraControlCommand}
    {parent: Device}
    {description: Represents the camera control structure.}
    {member:
        {name: action}
        {type: VidyoCameraControlAction}
        {description: Represents the camera control action, refer VidyoCameraControlAction.}
    }
    {member:
        {name: direction}
        {type: VidyoCameraControlDirection}
        {description: Represents the direction of camera control, refer VidyoCameraControlDirection.}
    }
    {member:
        {name: viscaCmd}
        {type: LmiString}
        {description: Represents the visca command.}
    }
    {member:
        {name: presetId}
        {type: LmiUint}
        {description: Represents the preset id.}
    }
}
*/
typedef struct  {
    VidyoCameraControlAction action;
    VidyoCameraControlDirection direction;
    LmiString viscaCmd;
    LmiUint presetId;
} VidyoCameraControlCommand;

/**
{function visibility="private":
	{name: VidyoCameraPresetConstruct}
	{parent: VidyoCameraPreset}
	{description: Constructs an VidyoCameraPreset object.}
	{prototype: VidyoCameraPreset* VidyoCameraPresetConstruct(VidyoCameraPreset *preset, LmiUint index, const LmiString *name, LmiAllocator *alloc)}
	{parameter:
		{name: preset}
		{description: The VidyoCameraPreset object to be constructed.}
	}
	{parameter:
		{name: index}
		{description: The index number of preset.}
	}
	{parameter:
		{name: name}
		{description: The friendly name of preset.}
	}
	{parameter:
		{name: alloc}
		{description: The LmiAllocator object.}
	}
	{return: VidyoCameraPreset object on success otherwise NULL.}
}
*/
VidyoCameraPreset* VidyoCameraPresetConstruct(VidyoCameraPreset *preset, LmiUint index, const LmiString *name, LmiAllocator *alloc);

/**
{function visibility="private":
	{name: VidyoCameraPresetConstructDefault}
	{parent: VidyoCameraPreset}
	{description: Constructs default VidyoCameraPreset object.}
	{prototype: VidyoCameraPreset* VidyoCameraPresetConstructDefault(VidyoCameraPreset *preset, LmiAllocator *alloc)}
	{parameter:
		{name: preset}
		{description: The VidyoCameraPreset object to be constructed.}
	}
	{parameter:
		{name: alloc}
		{description: The LmiAllocator object.}
	}
	{return: VidyoCameraPreset object on success otherwise NULL.}
}
*/
VidyoCameraPreset* VidyoCameraPresetConstructDefault(VidyoCameraPreset *preset, LmiAllocator *alloc);

/**
{function visibility="private":
	{name: VidyoCameraPresetConstructCopy}
	{parent: VidyoCameraPreset}
	{description: Constructs an VidyoCameraPreset object as a copy of an existing object.}
	{prototype: VidyoCameraPreset* VidyoCameraPresetConstructCopy(VidyoCameraPreset *d, const VidyoCameraPreset *s)}
	{parameter:
		{name: d}
		{description: The VidyoCameraPreset object to construct.}
	}
	{parameter:
		{name: s}
		{description: The VidyoCameraPreset object to make a copy of.}
	}
	{return: VidyoCameraPreset object on success otherwise NULL.}
}
*/
VidyoCameraPreset* VidyoCameraPresetConstructCopy(VidyoCameraPreset *d, const VidyoCameraPreset *s);

/**
{function visibility="private":
	{name: VidyoCameraPresetAssign}
	{parent: VidyoCameraPreset}
	{description: Assign a VidyoCameraPreset object as a copy of an existing object.}
	{prototype: VidyoCameraPreset* VidyoCameraPresetAssign(VidyoCameraPreset *d, const VidyoCameraPreset *s)}
	{parameter:
		{name: d}
		{description: The VidyoCameraPreset object to be constructed as a copy.}
	}
	{parameter:
		{name: s}
		{description: An existing VidyoCameraPreset object.}
	}
	{return: VidyoCameraPreset object on success otherwise NULL.}
}
*/
VidyoCameraPreset* VidyoCameraPresetAssign(VidyoCameraPreset *d, const VidyoCameraPreset *s);

/**
{function visibility="private":
	{name: VidyoCameraPresetEqual}
	{parent: VidyoCameraPreset}
	{description: Compare two VidyoCameraPreset objects for equality.}
	{prototype: LmiBool VidyoCameraPresetEqual(const VidyoCameraPreset *preset, const VidyoCameraPreset *other)}
	{parameter:
		{name: preset}
		{description: The first VidyoCameraPreset object.}
	}
	{parameter:
		{name: other}
		{description: The second VidyoCameraPreset object.}
	}
	{return: A boolean indicating whether the two objects are equal.}
}
*/
LmiBool VidyoCameraPresetEqual(const VidyoCameraPreset *preset, const VidyoCameraPreset *other);

/**
{function visibility="private":
	{name: VidyoCameraPresetDestruct}
	{parent: VidyoCameraPreset}
	{description: Destructs the VidyoCameraPreset object.}
	{prototype: void VidyoCameraPresetDestruct(VidyoCameraPreset *preset)}
	{parameter:
		{name: preset}
		{description: The VidyoCameraPreset object to be destructed.}
	}
}
*/
void VidyoCameraPresetDestruct(VidyoCameraPreset *preset);

/**
{function visibility="private":
    {name: VidyoCameraPresetGetIndex}
    {parent: VidyoCameraPreset}
    {description: Gets the preset index.}
    {prototype: LmiUint VidyoCameraPresetGetIndex(const VidyoCameraPreset *preset)}
    {parameter: {name: preset} {description: The VidyoCameraPreset object.}}
    {return: The index number of preset.}
}
*/
LMI_INLINE_DECLARATION LmiUint VidyoCameraPresetGetIndex(const VidyoCameraPreset *preset);

/**
{function visibility="private":
    {name: VidyoCameraPresetGetName}
    {parent: VidyoCameraPreset}
    {description: Gets the preset friendly name.}
    {prototype: const LmiString *VidyoCameraPresetGetName(const VidyoCameraPreset *preset)}
    {parameter: {name: preset} {description: The VidyoCameraPreset object.}}
    {return: The friendly name of preset.}
}
*/
LMI_INLINE_DECLARATION const LmiString *VidyoCameraPresetGetName(const VidyoCameraPreset *preset);

Declare_LmiVector(VidyoCameraPreset)
Declare_LmiVectorEqual(VidyoCameraPreset)

/**
{function visibility="private":
    {name: VidyoCameraPresetTranslateFromLmiCameraPreset}
    {parent: VidyoCameraPreset}
    {description: Translate preset information from LmiCameraPreset to VidyoCameraPreset object.}
    {prototype: LmiBool VidyoCameraPresetTranslateFromLmiCameraPreset(LmiVector(VidyoCameraPreset) *vidyoPreset,  const LmiVector(LmiCameraPreset) *lmiPreset, LmiAllocator *alloc)}
    {parameter: {name: vidyoPreset} {description: The vector of VidyoCameraPreset object.}}
    {parameter: {name: lmiPreset} {description: The vector of LmiCameraPreset object.}}
    {parameter: {name: alloc} {description: The LmiAllocator object.}}
    {return: A boolean indicating if translation  is successful or not.}
}
*/
LmiBool VidyoCameraPresetTranslateFromLmiCameraPreset(LmiVector(VidyoCameraPreset) *vidyoPreset, const LmiVector(LmiCameraPreset) *lmiPreset, LmiAllocator *alloc);

/**
{function visibility="private":
    {name: VidyoCameraPresetTranslateToLmiCameraPreset}
    {parent: VidyoCameraPreset}
    {description: Translate preset information from VidyoCameraPreset to LmiCameraPreset object.}
    {prototype: LmiBool VidyoCameraPresetTranslateToLmiCameraPreset(LmiVector(LmiCameraPreset) *lmiPreset,  const LmiVector(VidyoCameraPreset) *vidyoPreset, LmiAllocator *alloc)}
    {parameter: {name: lmiPreset} {description: The vector of LmiCameraPreset object.}}
    {parameter: {name: vidyoPreset} {description: The vector of VidyoCameraPreset object.}}
    {parameter: {name: alloc} {description: The LmiAllocator object.}}
    {return: A boolean indicating if translation  is successful or not.}
}
*/
LmiBool VidyoCameraPresetTranslateToLmiCameraPreset(LmiVector(LmiCameraPreset) *lmiPreset, const LmiVector(VidyoCameraPreset) *vidyoPreset, LmiAllocator *alloc);

/**
{type:
    {name: VidyoCameraControlCapabilities}
    {parent: Device}
    {include: Lmi/VidyoClient/VidyoCameraControlCapabilities.h}
    {description: Represents the capability of a camera control.}
    {member:
        {name: panTiltHasRubberBand}
        {type: LmiBool}
        {description:}
    }
    {member:
        {name: panTiltHasContinuousMove}
        {type: LmiBool}
        {description:}
    }
    {member:
        {name: panTiltHasNudge}
        {type: LmiBool}
        {description:}
    }
    {member:
        {name: zoomHasRubberBand}
        {type: LmiBool}
        {description:}
    }
    {member:
        {name: zooomHasContinuousMove}
        {type: LmiBool}
        {description:}
    }
    {member:
        {name: zoomHasNudge}
        {type: LmiBool}
        {description:}
    }
    {member:
        {name: hasPhotoCapture}
        {type: LmiBool}
        {description:}
    }
    {member:
        {name: hasViscaSupport}
        {type: LmiBool}
        {description: Camera supports VISCA Protocol.}
    }
    {member:
        {name: hasPresetSupport}
        {type: LmiBool}
        {description: Camera supports Presets.}
    }
}
*/
typedef struct {
    LmiBool panTiltHasRubberBand; // for future use
    LmiBool panTiltHasContinuousMove;
    LmiBool panTiltHasNudge;
    
    LmiBool zoomHasRubberBand; // for future use
    LmiBool zooomHasContinuousMove;
    LmiBool zoomHasNudge;
    
    LmiBool hasPhotoCapture;  // for future use
    LmiBool hasViscaSupport;
    LmiBool hasPresetSupport;
}  VidyoCameraControlCapabilities;

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesConstructDefault}
    {parent: VidyoCameraControlCapabilities}
    {description: Construct an VidyoCameraControlCapabilities object with default values.}
    {prototype: VidyoCameraControlCapabilities *VidyoCameraControlCapabilitiesConstructDefault(VidyoCameraControlCapabilities *caps, LmiAllocator *alloc)}
    {parameter: {name: caps} {description: The VidyoCameraControlCapabilities object to construct.}}
    {parameter: {name: alloc} {description: The allocator to use.}}
    {return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoCameraControlCapabilities *VidyoCameraControlCapabilitiesConstructDefault(VidyoCameraControlCapabilities *caps, LmiAllocator *alloc);

/**
{function visibility="private":
	{name: VidyoCameraControlCapabilitiesConstructCopy}
	{parent: VidyoCameraControlCapabilities}
	{description: Constructs an VidyoCameraControlCapabilities object as a copy of an existing objec.}
	{prototype: VidyoCameraControlCapabilities *VidyoCameraControlCapabilitiesConstructCopy(VidyoCameraControlCapabilities *d, VidyoCameraControlCapabilities *s)}
	{parameter: {name: d} {description: The VidyoCameraControlCapabilities object to construct.}}
	{parameter: {name: s} {description: The VidyoCameraControlCapabilities object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoCameraControlCapabilities *VidyoCameraControlCapabilitiesConstructCopy(VidyoCameraControlCapabilities *d, VidyoCameraControlCapabilities *s);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesConstruct}
    {parent: VidyoCameraControlCapabilities}
    {description: Constructs an VidyoCameraControlCapabilities object with capabilities.}
    {prototype: VidyoCameraControlCapabilities *VidyoCameraControlCapabilitiesConstruct(VidyoCameraControlCapabilities *caps, LmiBool panTiltRubberBand, LmiBool panTiltMove, LmiBool panTiltNudge, LmiBool zoomRubberBand, LmiBool zoomMove, LmiBool zoomNudge, LmiBool photoCapture, LmiBool viscaSupport, LmiBool presetSupport)}
    {parameter: {name: caps} {description: The VidyoCameraControlCapabilities object to construct.}}
    {parameter: {name: panTiltRubberBand} {description: Indicate that camera support PanTiltRubberBand control type.}}
    {parameter: {name: panTiltMove} {description: Indicate that camera support PanTiltContinuousMove control type.}}
    {parameter: {name: panTiltNudge} {description: Indicate that camera support PanTiltNudge control type.}}
    {parameter: {name: zoomRubberBand} {description: Indicate that camera support ZoomRubberBand control type.}}
    {parameter: {name: zoomMove} {description: Indicate that camera support ZoomContinuousMove control type.}}
    {parameter: {name: zoomNudge} {description: Indicate that camera support ZoomNudge control type.}}
    {parameter: {name: photoCapture} {description: Indicate that camera support PhotoCapture.}}
    {parameter: {name: viscaSupport} {description: Indicate that camera support Visca protocol.}}
    {parameter: {name: presetSupport} {description: Indicate that camera support Preset.}}
    {return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoCameraControlCapabilities *VidyoCameraControlCapabilitiesConstruct(VidyoCameraControlCapabilities *caps, LmiBool panTiltRubberBand, LmiBool panTiltMove, LmiBool panTiltNudge, LmiBool zoomRubberBand, LmiBool zoomMove, LmiBool zoomNudge, LmiBool photoCapture, LmiBool viscaSupport, LmiBool presetSupport);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesDestruct}
    {parent: VidyoCameraControlCapabilities}
    {description: Destructs an VidyoCameraControlCapabilities object.}
    {prototype: void VidyoCameraControlCapabilitiesDestruct(VidyoCameraControlCapabilities *caps)}
    {parameter: {name: caps} {description: The VidyoCameraControlCapabilities object to destruct.}}
}
*/
LMI_INLINE_DECLARATION void VidyoCameraControlCapabilitiesDestruct(VidyoCameraControlCapabilities *caps);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesAssign}
    {parent: VidyoCameraControlCapabilities}
    {description: Assigns one VidyoCameraControlCapabilities object the value of another.}
    {prototype: VidyoCameraControlCapabilities *VidyoCameraControlCapabilitiesAssign(VidyoCameraControlCapabilities *caps, const VidyoCameraControlCapabilities *other)}
    {parameter: {name: caps} {description: The VidyoCameraControlCapabilities object to assign to.}}
    {parameter: {name: other} {description: The VidyoCameraControlCapabilities object to make a copy of.}}
    {return: A pointer to the destination object on success, or NULL on failure.}
}
*/
LMI_INLINE_DECLARATION VidyoCameraControlCapabilities *VidyoCameraControlCapabilitiesAssign(VidyoCameraControlCapabilities *caps, const VidyoCameraControlCapabilities *other);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesEqual}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: LmiBool VidyoCameraControlCapabilitiesEqual(const VidyoCameraControlCapabilities *caps, const VidyoCameraControlCapabilities *other)}
    {parameter: {name: caps} {description: #}}
    {parameter: {name: other} {description: #}}
    {return: LMI_TRUE it objects are equal, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoCameraControlCapabilitiesEqual(const VidyoCameraControlCapabilities *caps, const VidyoCameraControlCapabilities *other);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesTranslateFromLmiControlCaps}
    {parent: VidyoCameraControlCapabilities}
    {description: Construct an VidyoCameraControlCapabilities from LmiCameraControlCapabilities.}
    {prototype: VidyoCameraControlCapabilities *VidyoCameraControlCapabilitiesTranslateFromLmiControlCaps(VidyoCameraControlCapabilities *caps, const LmiCameraControlCapabilities *lmiCaps)}
    {parameter: {name: caps} {description: The VidyoCameraControlCapabilities object to construct.}}
    {parameter: {name: lmiCaps} {description: The LmiCameraControlCapabilities object for initialization of VidyoCameraControlCapabilities object.}}
    {return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoCameraControlCapabilities *VidyoCameraControlCapabilitiesTranslateFromLmiControlCaps(VidyoCameraControlCapabilities *caps, const LmiCameraControlCapabilities *lmiCaps);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesTranslateToLmiControlCaps}
    {parent: VidyoCameraControlCapabilities}
    {description: Construct an VidyoCameraControlCapabilities from LmiCameraControlCapabilities.}
    {prototype: void VidyoCameraControlCapabilitiesTranslateToLmiControlCaps(const VidyoCameraControlCapabilities *caps, LmiCameraControlCapabilities *lmiCaps)}
    {parameter: {name: caps} {description: The VidyoCameraControlCapabilities object to construct.}}
    {parameter: {name: lmiCaps} {description: The LmiCameraControlCapabilities object for initialization of VidyoCameraControlCapabilities object.}}
}
 */
void VidyoCameraControlCapabilitiesTranslateToLmiControlCaps(const VidyoCameraControlCapabilities *caps, LmiCameraControlCapabilities *lmiCaps);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesSetPanTiltRubberBand}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: void VidyoCameraControlCapabilitiesSetPanTiltRubberBand(VidyoCameraControlCapabilities *caps, LmiBool enable)}
    {parameter: {name: caps} {description: #}}
    {parameter: {name: enable} {description: #}}
}
*/
LMI_INLINE_DECLARATION void VidyoCameraControlCapabilitiesSetPanTiltRubberBand(VidyoCameraControlCapabilities *caps, LmiBool enable);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesSetPanTiltContinuousMove}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: void VidyoCameraControlCapabilitiesSetPanTiltContinuousMove(VidyoCameraControlCapabilities *caps, LmiBool enable)}
    {parameter: {name: caps} {description: #}}
    {parameter: {name: enable} {description: #}}
}
*/
LMI_INLINE_DECLARATION void VidyoCameraControlCapabilitiesSetPanTiltContinuousMove(VidyoCameraControlCapabilities *caps, LmiBool enable);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesSetPanTiltNudge}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: void VidyoCameraControlCapabilitiesSetPanTiltNudge(VidyoCameraControlCapabilities *caps, LmiBool enable)}
    {parameter: {name: caps} {description: #}}
    {parameter: {name: enable} {description: #}}
}
*/
LMI_INLINE_DECLARATION void VidyoCameraControlCapabilitiesSetPanTiltNudge(VidyoCameraControlCapabilities *caps, LmiBool enable);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesSetZoomRubberBand}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: void VidyoCameraControlCapabilitiesSetZoomRubberBand(VidyoCameraControlCapabilities *caps, LmiBool enable)}
    {parameter: {name: caps} {description: #}}
    {parameter: {name: enable} {description: #}}
}
*/
LMI_INLINE_DECLARATION void VidyoCameraControlCapabilitiesSetZoomRubberBand(VidyoCameraControlCapabilities *caps, LmiBool enable);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesSetZoomContinuousMove}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: void VidyoCameraControlCapabilitiesSetZoomContinuousMove(VidyoCameraControlCapabilities *caps, LmiBool enable)}
    {parameter: {name: caps} {description: #}}
    {parameter: {name: enable} {description: #}}
}
*/
LMI_INLINE_DECLARATION void VidyoCameraControlCapabilitiesSetZoomContinuousMove(VidyoCameraControlCapabilities *caps, LmiBool enable);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesSetZoomNudge}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: void VidyoCameraControlCapabilitiesSetZoomNudge(VidyoCameraControlCapabilities *caps, LmiBool enable)}
    {parameter: {name: caps} {description: #}}
    {parameter: {name: enable} {description: #}}
}
*/
LMI_INLINE_DECLARATION void VidyoCameraControlCapabilitiesSetZoomNudge(VidyoCameraControlCapabilities *caps, LmiBool enable);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesSetPhotoCapture}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: void VidyoCameraControlCapabilitiesSetPhotoCapture(VidyoCameraControlCapabilities *caps, LmiBool enable)}
    {parameter: {name: caps} {description: #}}
    {parameter: {name: enable} {description: #}}
}
*/
LMI_INLINE_DECLARATION void VidyoCameraControlCapabilitiesSetPhotoCapture(VidyoCameraControlCapabilities *caps, LmiBool enable);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesSetViscaSupport}
    {parent: VidyoCameraControlCapabilities}
    {description: Set whether Visca protocol is supported.}
    {prototype: void VidyoCameraControlCapabilitiesSetViscaSupport(VidyoCameraControlCapabilities *caps, LmiBool enable)}
    {parameter: {name: caps} {description: The camera capabilities object.}}
    {parameter: {name: enable} {description: Whether Visca protocol is supported.}}
}
*/
LMI_INLINE_DECLARATION void VidyoCameraControlCapabilitiesSetViscaSupport(VidyoCameraControlCapabilities *caps, LmiBool enable);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesClear}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: void VidyoCameraControlCapabilitiesClear(VidyoCameraControlCapabilities *caps)}
    {parameter: {name: caps} {description: #}}
}
*/
void VidyoCameraControlCapabilitiesClear(VidyoCameraControlCapabilities *caps);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesHasPanTiltRubberBand}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: LmiBool VidyoCameraControlCapabilitiesHasPanTiltRubberBand(const VidyoCameraControlCapabilities *caps)}
    {parameter: {name: caps} {description: #}}
    {return: #}
}
*/
LMI_INLINE_DECLARATION LmiBool VidyoCameraControlCapabilitiesHasPanTiltRubberBand(const VidyoCameraControlCapabilities *caps);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesHasPanTiltContinuousMove}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: LmiBool VidyoCameraControlCapabilitiesHasPanTiltContinuousMove(const VidyoCameraControlCapabilities *caps)}
    {parameter: {name: caps} {description: #}}
    {return: #}
}
*/
LMI_INLINE_DECLARATION LmiBool VidyoCameraControlCapabilitiesHasPanTiltContinuousMove(const VidyoCameraControlCapabilities *caps);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesHasPanTiltNudge}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: LmiBool VidyoCameraControlCapabilitiesHasPanTiltNudge(const VidyoCameraControlCapabilities *caps)}
    {parameter: {name: caps} {description: #}}
    {return: #}
}
*/
LMI_INLINE_DECLARATION LmiBool VidyoCameraControlCapabilitiesHasPanTiltNudge(const VidyoCameraControlCapabilities *caps);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesHasZoomRubberBand}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: LmiBool VidyoCameraControlCapabilitiesHasZoomRubberBand(const VidyoCameraControlCapabilities *caps)}
    {parameter: {name: caps} {description: #}}
    {return: #}
}
*/
LMI_INLINE_DECLARATION LmiBool VidyoCameraControlCapabilitiesHasZoomRubberBand(const VidyoCameraControlCapabilities *caps);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesHasZoomContinuousMove}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: LmiBool VidyoCameraControlCapabilitiesHasZoomContinuousMove(const VidyoCameraControlCapabilities *caps)}
    {parameter: {name: caps} {description: #}}
    {return: #}
}
*/
LMI_INLINE_DECLARATION LmiBool VidyoCameraControlCapabilitiesHasZoomContinuousMove(const VidyoCameraControlCapabilities *caps);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesHasZoomNudge}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: LmiBool VidyoCameraControlCapabilitiesHasZoomNudge(const VidyoCameraControlCapabilities *caps)}
    {parameter: {name: caps} {description: #}}
    {return: #}
}
*/
LMI_INLINE_DECLARATION LmiBool VidyoCameraControlCapabilitiesHasZoomNudge(const VidyoCameraControlCapabilities *caps);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesHasPhotoCapture}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: LmiBool VidyoCameraControlCapabilitiesHasPhotoCapture(const VidyoCameraControlCapabilities *caps)}
    {parameter: {name: caps} {description: #}}
    {return: #}
}
*/
LMI_INLINE_DECLARATION LmiBool VidyoCameraControlCapabilitiesHasPhotoCapture(const VidyoCameraControlCapabilities *caps);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesHasViscaSupport}
    {parent: VidyoCameraControlCapabilities}
    {description: Check whether Visca protocol is supported.}
    {prototype: LmiBool VidyoCameraControlCapabilitiesHasViscaSupport(const VidyoCameraControlCapabilities *caps)}
    {parameter: {name: caps} {description: The camera capabilities object.}}
    {return: A boolean indicating if Visca protocol is supported.}
}
*/
LMI_INLINE_DECLARATION LmiBool VidyoCameraControlCapabilitiesHasViscaSupport(const VidyoCameraControlCapabilities *caps);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesHasPanTilt}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: LmiBool VidyoCameraControlCapabilitiesHasPanTilt(const VidyoCameraControlCapabilities *caps)}
    {parameter: {name: caps} {description: #}}
    {return: #}
}
*/
LMI_INLINE_DECLARATION LmiBool VidyoCameraControlCapabilitiesHasPanTilt(const VidyoCameraControlCapabilities *caps);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesHasZoom}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: LmiBool VidyoCameraControlCapabilitiesHasZoom(const VidyoCameraControlCapabilities *caps)}
    {parameter: {name: caps} {description: #}}
    {return: #}
}
*/
LMI_INLINE_DECLARATION LmiBool VidyoCameraControlCapabilitiesHasZoom(const VidyoCameraControlCapabilities *caps);

/**
{function visibility="private":
    {name: VidyoCameraControlCapabilitiesHasAny}
    {parent: VidyoCameraControlCapabilities}
    {description: #}
    {prototype: LmiBool VidyoCameraControlCapabilitiesHasAny(const VidyoCameraControlCapabilities *caps)}
    {parameter: {name: caps} {description: #}}
    {return: #}
}
*/
LMI_INLINE_DECLARATION LmiBool VidyoCameraControlCapabilitiesHasAny(const VidyoCameraControlCapabilities *caps);

/**
{function visibility="private":
    {name: VidyoCameraControlDirectionCStr}
    {parent: VidyoCameraControlCapabilities}
    {description: Return the string value of VidyoCameraControlDirection.}
    {prototype: const char * VidyoCameraControlDirectionCStr(VidyoCameraControlDirection direction)}
    {parameter: {name: direction} {description: The camera control direction.}}
    {return: Returns direction as string.}
}
*/
const char *VidyoCameraControlDirectionCStr(VidyoCameraControlDirection direction);

/**
{function visibility="private":
    {name: VidyoCameraControlCommandConstruct}
    {parent: VidyoCameraControlCommand}
    {description: Construct an VidyoCameraControlCommand object.}
    {prototype: VidyoCameraControlCommand* VidyoCameraControlCommandConstruct(VidyoCameraControlCommand* obj, VidyoCameraControlAction action, VidyoCameraControlDirection direction, LmiString* viscaCmd, LmiUint presetId, LmiAllocator* alloc)}
    {parameter: {name: obj} {description: Represents the VidyoCameraControlCommand object to construct.}}
    {parameter: {name: action} {description: Represents the control action of camera, refer VidyoCameraControlAction. }}
    {parameter: {name: direction} {description: Represents the control direction of camera, refer VidyoCameraControlDirection.}}
    {parameter: {name: viscaCmd} {description: Represents the visca command.}}
    {parameter: {name: presetId} {description: Represents the preset index.}}
    {parameter: {name: alloc} {description: Represents the LmiAllocator object.}}
    {return: VidyoCameraControlCommand object on success otherwise NULL.}
}
*/
VidyoCameraControlCommand* VidyoCameraControlCommandConstruct(VidyoCameraControlCommand* obj, VidyoCameraControlAction action, VidyoCameraControlDirection direction, LmiString* viscaCmd, LmiUint presetId, LmiAllocator* alloc);

/**
 {function visibility="private":
    {name:VidyoCameraControlCommandConstructDefault}
    {parent: VidyoCameraControlCommand}
    {description: Default construct for VidyoCameraControlCommand.}
    {prototype: VidyoCameraControlCommand* VidyoCameraControlCommandConstructDefault(VidyoCameraControlCommand* obj, LmiAllocator* alloc)}
    {parameter: {name: obj} {description: Represents the VidyoCameraControlCommand object to construct.}}
    {parameter: {name: alloc} {description: Represents the LmiAllocator object.}}
    {return: VidyoCameraControlCommand object on success otherwise NULL.}
}
*/
VidyoCameraControlCommand* VidyoCameraControlCommandConstructDefault(VidyoCameraControlCommand* obj, LmiAllocator* alloc);

/**
{function visibility="private":
    {name: VidyoCameraControlCommandConstructCopy}
    {parent: VidyoCameraControlCommand}
    {description: Constructs an VidyoCameraControlCommand object as a copy of an existing object.}
    {prototype: VidyoCameraControlCommand* VidyoCameraControlCommandConstructCopy(VidyoCameraControlCommand* d, const VidyoCameraControlCommand* s)}
    {parameter: {name: d} {description: Represents the VidyoCameraControlCommand Object to construct.}}
    {parameter: {name: s} {description: Represents the VidyoCameraControlCommand object to make a copy of.}}
    {return: VidyoCameraControlCommand object on success otherwise NULL.}
}
*/
VidyoCameraControlCommand* VidyoCameraControlCommandConstructCopy(VidyoCameraControlCommand* d, const VidyoCameraControlCommand* s);

/**
{function visibility="private":
    {name: VidyoCameraControlCommandAssign}
    {parent: VidyoCameraControlCommand}
    {description: Assigns one VidyoCameraControlCommand object the value of another.}
    {prototype: VidyoCameraControlCommand* VidyoCameraControlCommandAssign(VidyoCameraControlCommand* d, const VidyoCameraControlCommand* s)}
    {parameter: {name: d} {description: Represents the VidyoCameraControlCommand Object to assign.}}
    {parameter: {name: s} {description: Represents the VidyoCameraControlCommand object from which to assign.}}
    {return: VidyoCameraControlCommand object on success otherwise NULL.}
}
*/
VidyoCameraControlCommand* VidyoCameraControlCommandAssign(VidyoCameraControlCommand* d, const VidyoCameraControlCommand* s);

/**
{function visibility="private":
    {name: VidyoCameraControlCommandDestruct}
    {parent: VidyoCameraControlCommand}
    {description: Destructs an VidyoCameraControlCommand object.}
    {prototype: void VidyoCameraControlCommandDestruct(VidyoCameraControlCommand* obj)}
    {parameter: {name: obj} {description: The VidyoCameraControlCommand object to destruct.}}
}
*/
void VidyoCameraControlCommandDestruct(VidyoCameraControlCommand* obj);

Declare_LmiVector(VidyoCameraControlCommand)

LMI_END_EXTERN_C

#if LMI_INLINE_NEED_HEADER_FILE_DEFINITIONS
#include "VidyoCameraControlCapabilitiesInline.h"
#endif

#endif /* VIDYO_CAMERACONTROLCAPABILITIES_H_ */
