/**
{file:
	{name: VidyoEndpoint.h}
	{description: High level client library. }
	{copyright:
		(c) 2014-2016 Vidyo, Inc.,
		433 Hackensack Avenue,
		Hackensack, NJ  07601.
		All rights reserved.
		The information contained herein is proprietary to Vidyo, Inc.
		and shall not be reproduced, copied (in whole or in part), adapted,
		modified, disseminated, transmitted, transcribed, stored in a retrieval
		system, or translated into any language in any form by any means
		without the express written consent of Vidyo, Inc.}
}
*/
#ifndef VIDYO_ENDPOINT_H_
#define VIDYO_ENDPOINT_H_

#include <Lmi/Utils/LmiAllocator.h>
#include <Lmi/Utils/LmiMisc.h>
#include <Lmi/Utils/LmiTypes.h>
#include <Lmi/Utils/LmiVector.h>
#include <Lmi/Utils/LmiList.h>
#include <Lmi/Utils/LmiUtils.h>
#include <Lmi/Os/LmiMallocAllocator.h>
#include <Lmi/Os/LmiOs.h>
#include <Lmi/Os/LmiLog.h>
#include <Lmi/Os/LmiSharedPtr.h>

LMI_BEGIN_EXTERN_C

struct VidyoEndpointImpl_;
typedef struct VidyoEndpointImpl_ VidyoEndpointImpl;
Declare_LmiSharedPtr(VidyoEndpointImpl)
Declare_LmiSharedPtrWrapper(VidyoEndpoint, VidyoEndpointImpl)
Declare_LmiWeakPtrWrapper(VidyoEndpointHandle, VidyoEndpoint, VidyoEndpointImpl)

LMI_END_EXTERN_C

#include "VidyoProperty.h"
#include "VidyoLogRecord.h"
#include "VidyoLogger.h"
#include "VidyoTransport.h"
#include "VidyoNetworkInterface.h"
#include "VidyoCodec.h"
#include "VidyoForwardDecl.h"

#include "VidyoAudioFrame.h"
#include "VidyoVideoFrame.h"
#include "VidyoDevice.h"

/* sinks */
#include "VidyoLocalSpeaker.h"
#include "VidyoRemoteSpeaker.h"
#include "VidyoLocalRenderer.h"
#include "VidyoRemoteRenderer.h"
#include "VidyoVirtualRenderer.h"

/* sources */
#include "VidyoLocalMicrophone.h"
#include "VidyoRemoteMicrophone.h"
#include "VidyoLocalCamera.h"
#include "VidyoRemoteCamera.h"
#include "VidyoLocalWindowShare.h"
#include "VidyoRemoteWindowShare.h"
#include "VidyoLocalMonitor.h"
#include "VidyoVirtualVideoSource.h"
#include "VidyoVirtualAudioSource.h"
#include "VidyoUser.h"
#include "VidyoCall.h"
#include "VidyoRoom.h"
#include "VidyoParticipant.h"
#include "VidyoStatistics.h"

LMI_BEGIN_EXTERN_C

/**
{package apigroup="connector,simple":
	{name: VidyoClient}
	{parent: Lmi}
	{include: Lmi/VidyoClient/VidyoEndpoint.h}
	{library: VidyoClient}
	{description: The VidyoClient package encapsulates all APIs required to build a vidyo conferencing application.}
}
*/

/**
{package apigroup="connector,simple":
	{name: Endpoint}
	{parent: VidyoClient}
	{include: Lmi/VidyoClient/VidyoEndpoint.h}
	{description: The VidyoClient package encapsulates all APIs required to build a vidyo conferencing application.}
}
*/


/**
{type apigroup="simple" reference-counted="yes":
	{name: VidyoEndpoint}
	{parent: Endpoint}
	{description: }
}
*/

/**
{type visibility="private":
	{name: VidyoEndpointHandle}
	{parent: VidyoEndpoint}
	{description: }
}
*/

/**
{type visibility="public":
	{name: VidyoEndpointErrorCode}
	{parent: VidyoEndpoint}
	{description: Provide the advanced error code to the application.}
	{value:
		{name: VIDYO_ENDPOINTERRORCODE_NotAllowed} 
		{description: Not allowed.}}
	{value:
		{name: VIDYO_ENDPOINTERRORCODE_WrongState}
		{description: Called in wrong state.}}
	{value:
		{name: VIDYO_ENDPOINTERRORCODE_InvalidInput}
		{description: Invalid input.}}
	{value:
		{name: VIDYO_ENDPOINTERRORCODE_Misc}
		{description: Miscellaneous remote error.}}
}
*/
typedef enum
{
	VIDYO_ENDPOINTERRORCODE_NotAllowed,
	VIDYO_ENDPOINTERRORCODE_WrongState,
	VIDYO_ENDPOINTERRORCODE_InvalidInput,
	VIDYO_ENDPOINTERRORCODE_Misc
} VidyoEndpointErrorCode;

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnError}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered to provide advanced error codes to the application.}
	{prototype: void(*VidyoEndpointOnError)(VidyoEndpoint *e, VidyoEndpointErrorCode error, const LmiString* apiName)}
	{parameter:
		{name: e}
		{description: Pointer to the VidyoEndpoint that triggered the callback.}
	}
	{parameter:
		{name: error}
		{description: Error code, see VidyoEndpointErrorCode.}
	}
	{parameter:
		{name: apiName}
		{description: Name of the api where error occured.}
	}
}
*/
typedef void(*VidyoEndpointOnError)(VidyoEndpoint *e, VidyoEndpointErrorCode error, const LmiString* apiName);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLog}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered for every log record. }
	{prototype: void (*VidyoEndpointOnLog)(VidyoEndpoint* e, const VidyoLogRecord* logRecord)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: logRecord}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLog)(VidyoEndpoint* e, const VidyoLogRecord* logRecord);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalCameraAdded}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local camera becomes available to a VidyoEndpoint.}
	{prototype: void (*VidyoEndpointOnLocalCameraAdded)(VidyoEndpoint* e, VidyoLocalCamera* localCamera)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localCamera}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalCameraAdded)(VidyoEndpoint* e, VidyoLocalCamera* localCamera);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalCameraRemoved}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local camera is no longer available to a VidyoEndpoint.}
	{prototype: void (*VidyoEndpointOnLocalCameraRemoved)(VidyoEndpoint* e, VidyoLocalCamera* localCamera)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localCamera}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalCameraRemoved)(VidyoEndpoint* e, VidyoLocalCamera* localCamera);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalCameraSelected}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time there is a change in which local camera is selected for use.}
	{prototype: void (*VidyoEndpointOnLocalCameraSelected)(VidyoEndpoint* e, VidyoLocalCamera* localCamera)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localCamera}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalCameraSelected)(VidyoEndpoint* e, VidyoLocalCamera* localCamera);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalCameraStateUpdated}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time there is a change in the state of the camera such as being suspended or disabled.}
	{prototype: void (*VidyoEndpointOnLocalCameraStateUpdated)(VidyoEndpoint* e, VidyoLocalCamera* localCamera, VidyoDeviceState state)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localCamera}
		{description: }
	}
	{parameter:
		{name: state}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalCameraStateUpdated)(VidyoEndpoint* e, VidyoLocalCamera* localCamera, VidyoDeviceState state);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnRemoteCameraAdded}
	{parent: VidyoEndpoint}
		{description: }
	{prototype: void (*VidyoEndpointOnRemoteCameraAdded)(VidyoEndpoint* e, VidyoRemoteCamera* remoteCamera, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteCamera}
		{description: }
	}
	{parameter:
		{name: user}
		{description: }
	}
	{parameter:
		{name: call}
		{description: }
	}
	{parameter:
		{name: room}
		{description: }
	}
	{parameter:
		{name: participant}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteCameraAdded)(VidyoEndpoint* e, VidyoRemoteCamera* remoteCamera, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnRemoteCameraRemoved}
	{parent: VidyoEndpoint}
		{description: }
	{prototype: void (*VidyoEndpointOnRemoteCameraRemoved)(VidyoEndpoint* e, VidyoRemoteCamera* remoteCamera, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteCamera}
		{description: }
	}
	{parameter:
		{name: user}
		{description: }
	}
	{parameter:
		{name: call}
		{description: }
	}
	{parameter:
		{name: room}
		{description: }
	}
	{parameter:
		{name: participant}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteCameraRemoved)(VidyoEndpoint* e, VidyoRemoteCamera* remoteCamera, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnRemoteCameraStateUpdated}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: void (*VidyoEndpointOnRemoteCameraStateUpdated)(VidyoEndpoint* e, VidyoRemoteCamera* remoteCamera, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant, VidyoDeviceState state)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteCamera}
		{description: }
	}
	{parameter:
		{name: user}
		{description: }
	}
	{parameter:
		{name: call}
		{description: }
	}
	{parameter:
		{name: room}
		{description: }
	}
	{parameter:
		{name: participant}
		{description: }
	}
	{parameter:
		{name: state}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteCameraStateUpdated)(VidyoEndpoint* e, VidyoRemoteCamera* remoteCamera, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant, VidyoDeviceState state);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalMicrophoneAdded}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local microphone becomes available to a VidyoEndpoint.}
	{prototype: void (*VidyoEndpointOnLocalMicrophoneAdded)(VidyoEndpoint* e, VidyoLocalMicrophone* localMicrophone)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMicrophone}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalMicrophoneAdded)(VidyoEndpoint* e, VidyoLocalMicrophone* localMicrophone);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalMicrophoneRemoved}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local microphone is no longer available to a VidyoEndpoint.}
	{prototype: void (*VidyoEndpointOnLocalMicrophoneRemoved)(VidyoEndpoint* e, VidyoLocalMicrophone* localMicrophone)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMicrophone}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalMicrophoneRemoved)(VidyoEndpoint* e, VidyoLocalMicrophone* localMicrophone);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalMicrophoneSelected}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time there is a change in which local microphone is selected for use.}
	{prototype: void (*VidyoEndpointOnLocalMicrophoneSelected)(VidyoEndpoint* e, VidyoLocalMicrophone* localMicrophone)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMicrophone}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalMicrophoneSelected)(VidyoEndpoint* e, VidyoLocalMicrophone* localMicrophone);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalMicrophoneStateUpdated}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time there is a change in the state of the microphone such as being suspended or disabled.}
	{prototype: void (*VidyoEndpointOnLocalMicrophoneStateUpdated)(VidyoEndpoint* e, VidyoLocalMicrophone* localMicrophone, VidyoDeviceState state)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMicrophone}
		{description: }
	}
	{parameter:
		{name: state}
		{description: New state of the Local microphone.
			If microphone is muted locally by the local user using application or hard/soft muted by moderator, microphone will be paused instead of stopped.
			So, VIDYO_DEVICESTATE_Paused will be reported and VIDYO_DEVICESTATE_Resumed when unmuted.
			VIDYO_DEVICESTATE_Stopped will be reported when device is unselected and VIDYO_DEVICESTATE_Started when device is selected.}
	}
}
*/
typedef void(*VidyoEndpointOnLocalMicrophoneStateUpdated)(VidyoEndpoint* e, VidyoLocalMicrophone* localMicrophone, VidyoDeviceState state);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnRemoteMicrophoneAdded}
	{parent: VidyoEndpoint}
		{description: }
	{prototype: void (*VidyoEndpointOnRemoteMicrophoneAdded)(VidyoEndpoint* e, VidyoRemoteMicrophone* remoteMicrophone, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteMicrophone}
		{description: }
	}
	{parameter:
		{name: user}
		{description: }
	}
	{parameter:
		{name: call}
		{description: }
	}
	{parameter:
		{name: room}
		{description: }
	}
	{parameter:
		{name: participant}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteMicrophoneAdded)(VidyoEndpoint* e, VidyoRemoteMicrophone* remoteMicrophone, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnRemoteMicrophoneRemoved}
	{parent: VidyoEndpoint}
		{description: }
	{prototype: void (*VidyoEndpointOnRemoteMicrophoneRemoved)(VidyoEndpoint* e, VidyoRemoteMicrophone* remoteMicrophone, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteMicrophone}
		{description: }
	}
	{parameter:
		{name: user}
		{description: }
	}
	{parameter:
		{name: call}
		{description: }
	}
	{parameter:
		{name: room}
		{description: }
	}
	{parameter:
		{name: participant}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteMicrophoneRemoved)(VidyoEndpoint* e, VidyoRemoteMicrophone* remoteMicrophone, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnRemoteMicrophoneStateUpdated}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: void (*VidyoEndpointOnRemoteMicrophoneStateUpdated)(VidyoEndpoint* e, VidyoRemoteMicrophone* remoteMicrophone, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant, VidyoDeviceState state)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteMicrophone}
		{description: }
	}
	{parameter:
		{name: user}
		{description: }
	}
	{parameter:
		{name: call}
		{description: }
	}
	{parameter:
		{name: room}
		{description: }
	}
	{parameter:
		{name: participant}
		{description: }
	}
	{parameter:
		{name: state}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteMicrophoneStateUpdated)(VidyoEndpoint* e, VidyoRemoteMicrophone* remoteMicrophone, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant, VidyoDeviceState state);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalSpeakerAdded}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local speaker becomes available to a VidyoEndpoint.}
	{prototype: void (*VidyoEndpointOnLocalSpeakerAdded)(VidyoEndpoint* e, VidyoLocalSpeaker* localSpeaker)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localSpeaker}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalSpeakerAdded)(VidyoEndpoint* e, VidyoLocalSpeaker* localSpeaker);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalSpeakerRemoved}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local speaker is no longer available to a VidyoEndpoint.}
	{prototype: void (*VidyoEndpointOnLocalSpeakerRemoved)(VidyoEndpoint* e, VidyoLocalSpeaker* localSpeaker)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localSpeaker}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalSpeakerRemoved)(VidyoEndpoint* e, VidyoLocalSpeaker* localSpeaker);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalSpeakerSelected}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time there is a change in which local speaker is selected for use.}
	{prototype: void (*VidyoEndpointOnLocalSpeakerSelected)(VidyoEndpoint* e, VidyoLocalSpeaker* localSpeaker)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localSpeaker}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalSpeakerSelected)(VidyoEndpoint* e, VidyoLocalSpeaker* localSpeaker);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalSpeakerStateUpdated}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time there is a change in the state of the speaker such as being suspended or disabled.}
	{prototype: void (*VidyoEndpointOnLocalSpeakerStateUpdated)(VidyoEndpoint* e, VidyoLocalSpeaker* localSpeaker, VidyoDeviceState state)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localSpeaker}
		{description: }
	}
	{parameter:
		{name: state}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalSpeakerStateUpdated)(VidyoEndpoint* e, VidyoLocalSpeaker* localSpeaker, VidyoDeviceState state);


/**
{callback:
	{name: VidyoEndpointOnRemoteSpeakerAdded}
	{parent: VidyoEndpoint}
		{description: }
	{prototype: void (*VidyoEndpointOnRemoteSpeakerAdded)(VidyoEndpoint* e, VidyoRemoteSpeaker* remoteSpeaker, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteSpeaker}
		{description: }
	}
	{parameter:
		{name: user}
		{description: }
	}
	{parameter:
		{name: call}
		{description: }
	}
	{parameter:
		{name: room}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteSpeakerAdded)(VidyoEndpoint* e, VidyoRemoteSpeaker* remoteSpeaker, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room);

/**
{callback:
	{name: VidyoEndpointOnRemoteSpeakerRemoved}
	{parent: VidyoEndpoint}
		{description: }
	{prototype: void (*VidyoEndpointOnRemoteSpeakerRemoved)(VidyoEndpoint* e, VidyoRemoteSpeaker* remoteSpeaker, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteSpeaker}
		{description: }
	}
	{parameter:
		{name: user}
		{description: }
	}
	{parameter:
		{name: call}
		{description: }
	}
	{parameter:
		{name: room}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteSpeakerRemoved)(VidyoEndpoint* e, VidyoRemoteSpeaker* remoteSpeaker, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room);

/**
{callback apigroup="simple":
 	{name: VidyoEndpointOnRemoteSpeakerStateUpdated}
 	{parent: VidyoEndpoint}
 	{description: }
 	{prototype: void (*VidyoEndpointOnRemoteSpeakerStateUpdated)(VidyoEndpoint* e, VidyoRemoteSpeaker* remoteSpeaker, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, VidyoDeviceState state)}
 	{parameter:
 		{name: e}
 		{description: }
 	}
 	{parameter:
 		{name: remoteSpeaker}
 		{description: }
 	}
 	{parameter:
 		{name: user}
 		{description: }
 	}
	{parameter:
 		{name: call}
 		{description: }
 	}
 	{parameter:
 		{name: room}
 		{description: }
 	}
 	{parameter:
 		{name: state}
 		{description: }
 	}
}
*/
typedef void(*VidyoEndpointOnRemoteSpeakerStateUpdated)(VidyoEndpoint* e, VidyoRemoteSpeaker* remoteSpeaker, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, VidyoDeviceState state);

/**
{callback:
	{name: VidyoEndpointOnLocalRendererAdded}
	{parent: VidyoEndpoint}
		{description: }
	{prototype: void (*VidyoEndpointOnLocalRendererAdded)(VidyoEndpoint* e, VidyoLocalRenderer* localRenderer)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localRenderer}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalRendererAdded)(VidyoEndpoint* e, VidyoLocalRenderer* localRenderer);

/**
{callback:
	{name: VidyoEndpointOnLocalRendererRemoved}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: void (*VidyoEndpointOnLocalRendererRemoved)(VidyoEndpoint* e, VidyoLocalRenderer* localRenderer)}
	{parameter:
		{name: e}
	{description: }
	}
	{parameter:
		{name: localRenderer}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalRendererRemoved)(VidyoEndpoint* e, VidyoLocalRenderer* localRenderer);

/**
{callback:
	{name: VidyoEndpointOnLocalRendererSelected}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: void (*VidyoEndpointOnLocalRendererSelected)(VidyoEndpoint* e, VidyoLocalRenderer* localRenderer)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localRenderer}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalRendererSelected)(VidyoEndpoint* e, VidyoLocalRenderer* localRenderer);

/**
{callback:
	{name: VidyoEndpointOnLocalRendererStateUpdated}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: void (*VidyoEndpointOnLocalRendererStateUpdated)(VidyoEndpoint* e, VidyoLocalRenderer* localRenderer, VidyoDeviceState state)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localRenderer}
		{description: }
	}
	{parameter:
		{name: state}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalRendererStateUpdated)(VidyoEndpoint* e, VidyoLocalRenderer* localRenderer, VidyoDeviceState state);

/**
{callback:
	{name: VidyoEndpointOnRemoteRendererAdded}
	{parent: VidyoEndpoint}
		{description: }
	{prototype: void (*VidyoEndpointOnRemoteRendererAdded)(VidyoEndpoint* e, VidyoRemoteRenderer* remoteRenderer, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteRenderer}
		{description: }
	}
	{parameter:
		{name: user}
		{description: }
	}
	{parameter:
		{name: call}
		{description: }
	}
	{parameter:
		{name: room}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteRendererAdded)(VidyoEndpoint* e, VidyoRemoteRenderer* remoteRenderer, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room);

/**
{callback:
	{name: VidyoEndpointOnRemoteRendererRemoved}
	{parent: VidyoEndpoint}
		{description: }
	{prototype: void (*VidyoEndpointOnRemoteRendererRemoved)(VidyoEndpoint* e, VidyoRemoteRenderer* remoteRenderer, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteRenderer}
		{description: }
	}
	{parameter:
		{name: user}
		{description: }
	}
	{parameter:
		{name: call}
		{description: }
	}
	{parameter:
		{name: room}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteRendererRemoved)(VidyoEndpoint* e, VidyoRemoteRenderer* remoteRenderer, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room);

/**
{callback apigroup="simple":
 	{name: VidyoEndpointOnRemoteRendererStateUpdated}
 	{parent: VidyoEndpoint}
 	{description: }
 	{prototype: void (*VidyoEndpointOnRemoteRendererStateUpdated)(VidyoEndpoint* e, VidyoRemoteRenderer* remoteRenderer, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, VidyoDeviceState state)}
 	{parameter:
 		{name: e}
 		{description: }
 	}
 	{parameter:
 		{name: remoteRenderer}
 		{description: }
 	}
 	{parameter:
 		{name: user}
 		{description: }
 	}
 	{parameter:
 		{name: call}
 		{description: }
 	}
 	{parameter:
 		{name: room}
 		{description: }
 	}
 	{parameter:
 		{name: state}
 		{description: }
 	}
}
*/
typedef void(*VidyoEndpointOnRemoteRendererStateUpdated)(VidyoEndpoint* e, VidyoRemoteRenderer* remoteRenderer, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, VidyoDeviceState state);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalWindowShareAdded}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local window becomes available for a VidyoEndpoint to share in a conference.}
	{prototype: void (*VidyoEndpointOnLocalWindowShareAdded)(VidyoEndpoint* e, VidyoLocalWindowShare* localWindowShare)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localWindowShare}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalWindowShareAdded)(VidyoEndpoint* e, VidyoLocalWindowShare* localWindowShare);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalWindowShareRemoved}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local window is no longer available for a VidyoEndpoint to share in a conference.}
	{prototype: void (*VidyoEndpointOnLocalWindowShareRemoved)(VidyoEndpoint* e, VidyoLocalWindowShare* localWindowShare)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localWindowShare}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalWindowShareRemoved)(VidyoEndpoint* e, VidyoLocalWindowShare* localWindowShare);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalWindowShareSelected}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time there is a change in which local window is selected for sharing.}
	{prototype: void (*VidyoEndpointOnLocalWindowShareSelected)(VidyoEndpoint* e, VidyoLocalWindowShare* localWindowShare)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localWindowShare}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalWindowShareSelected)(VidyoEndpoint* e, VidyoLocalWindowShare* localWindowShare);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalWindowShareStateUpdated}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time there is a change in the state of the window such as being suspended or disabled.}
	{prototype: void (*VidyoEndpointOnLocalWindowShareStateUpdated)(VidyoEndpoint* e, VidyoLocalWindowShare* localWindowShare, VidyoDeviceState state)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localWindowShare}
		{description: }
	}
	{parameter:
		{name: state}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalWindowShareStateUpdated)(VidyoEndpoint* e, VidyoLocalWindowShare* localWindowShare, VidyoDeviceState state);


/**
{callback apigroup="simple":
	{name: VidyoEndpointOnRemoteWindowShareAdded}
	{parent: VidyoEndpoint}
		{description: }
	{prototype: void (*VidyoEndpointOnRemoteWindowShareAdded)(VidyoEndpoint* e, VidyoRemoteWindowShare* remoteWindowShare, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteWindowShare}
		{description: }
	}
	{parameter:
		{name: user}
		{description: }
	}
	{parameter:
		{name: call}
		{description: }
	}
	{parameter:
		{name: room}
		{description: }
	}
	{parameter:
		{name: participant}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteWindowShareAdded)(VidyoEndpoint* e, VidyoRemoteWindowShare* remoteWindowShare, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnRemoteWindowShareRemoved}
	{parent: VidyoEndpoint}
		{description: }
	{prototype: void (*VidyoEndpointOnRemoteWindowShareRemoved)(VidyoEndpoint* e, VidyoRemoteWindowShare* remoteWindowShare, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteWindowShare}
		{description: }
	}
	{parameter:
		{name: user}
		{description: }
	}
	{parameter:
		{name: call}
		{description: }
	}
	{parameter:
		{name: room}
		{description: }
	}
	{parameter:
		{name: participant}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteWindowShareRemoved)(VidyoEndpoint* e, VidyoRemoteWindowShare* remoteWindowShare, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnRemoteWindowShareStateUpdated}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: void (*VidyoEndpointOnRemoteWindowShareStateUpdated)(VidyoEndpoint* e, VidyoRemoteWindowShare* remoteWindowShare, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant, VidyoDeviceState state)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteWindowShare}
		{description: }
	}
	{parameter:
		{name: user}
		{description: }
	}
	{parameter:
		{name: call}
		{description: }
	}
	{parameter:
		{name: room}
		{description: }
	}
	{parameter:
		{name: participant}
		{description: }
	}
	{parameter:
		{name: state}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteWindowShareStateUpdated)(VidyoEndpoint* e, VidyoRemoteWindowShare* remoteWindowShare, const VidyoUser* user, const VidyoCall* call, const VidyoRoom* room, const VidyoParticipant* participant, VidyoDeviceState state);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalMonitorAdded}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local monitor becomes available to a VidyoEndpoint.}
	{prototype: void (*VidyoEndpointOnLocalMonitorAdded)(VidyoEndpoint* e, VidyoLocalMonitor* localMonitor)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMonitor}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalMonitorAdded)(VidyoEndpoint* e, VidyoLocalMonitor* localMonitor);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalMonitorRemoved}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local monitor is no longer available to a VidyoEndpoint.}
	{prototype: void (*VidyoEndpointOnLocalMonitorRemoved)(VidyoEndpoint* e, VidyoLocalMonitor* localMonitor)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMonitor}
		{description: }
	}
}
*/
typedef void (*VidyoEndpointOnLocalMonitorRemoved)(VidyoEndpoint* e, VidyoLocalMonitor* localMonitor);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalMonitorSelected}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time there is a change in which local monitor is selected for sharing.}
	{prototype: void (*VidyoEndpointOnLocalMonitorSelected)(VidyoEndpoint* e, VidyoLocalMonitor* localMonitor)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMonitor}
		{description: }
	}
}
*/
typedef void (*VidyoEndpointOnLocalMonitorSelected)(VidyoEndpoint* e, VidyoLocalMonitor* localMonitor);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalMonitorStateUpdated}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time there is a change in the state of the monitor such as being suspended or disabled.}
	{prototype: void (*VidyoEndpointOnLocalMonitorStateUpdated)(VidyoEndpoint* e, VidyoLocalMonitor* localMonitor, VidyoDeviceState state)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMonitor}
		{description: }
	}
	{parameter:
		{name: state}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalMonitorStateUpdated)(VidyoEndpoint* e, VidyoLocalMonitor* localMonitor, VidyoDeviceState state);

/**
{callback apigroup="simple":
    {name: VidyoEndpointOnVirtualVideoSourceAdded}
    {parent: VidyoEndpoint}
    {description: Callback that is triggered each time a virtual video source becomes available for a VidyoEndpoint to add to a conference.}
    {prototype: void (*VidyoEndpointOnVirtualVideoSourceAdded)(VidyoEndpoint* e, VidyoVirtualVideoSource* virtualVideoSource)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {parameter:
        {name: virtualVideoSource}
        {description: The VidyoVirtualVideoSource object that has been added.}
    }
}
*/
typedef void(*VidyoEndpointOnVirtualVideoSourceAdded)(VidyoEndpoint* e, VidyoVirtualVideoSource* virtualVideoSource);

/**
{callback apigroup="simple":
    {name: VidyoEndpointOnVirtualVideoSourceRemoved}
    {parent: VidyoEndpoint}
    {description: Callback that is triggered each time a virtual video source is no longer available for a VidyoEndpoint to add to a conference.}
    {prototype: void (*VidyoEndpointOnVirtualVideoSourceRemoved)(VidyoEndpoint* e, VidyoVirtualVideoSource* virtualVideoSource)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {parameter:
        {name: virtualVideoSource}
        {description: The VidyoVirtualVideoSource object that has been removed.}
    }
}
*/
typedef void(*VidyoEndpointOnVirtualVideoSourceRemoved)(VidyoEndpoint* e, VidyoVirtualVideoSource* virtualVideoSource);

/**
{callback apigroup="simple":
    {name: VidyoEndpointOnVirtualVideoSourceStateUpdated}
    {parent: VidyoEndpoint}
    {description: Callback that is triggered each time there is a change in the state of the virtual video source such as being started, stopped or configuration changed.}
    {prototype: void (*VidyoEndpointOnVirtualVideoSourceStateUpdated)(VidyoEndpoint* e, VidyoVirtualVideoSource* virtualVideoSource, VidyoDeviceState state)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {parameter:
        {name: virtualVideoSource}
        {description: The VidyoVirtualVideoSource object that has been changed its state or configuration.}
    }
    {parameter:
        {name: state}
        {description: The state of the virtual video source that shows what was changed.}
    }
}
*/
typedef void(*VidyoEndpointOnVirtualVideoSourceStateUpdated)(VidyoEndpoint* e, VidyoVirtualVideoSource* virtualVideoSource, VidyoDeviceState state);

/**
{callback apigroup="simple":
    {name: VidyoEndpointOnVirtualVideoSourceExternalMediaBufferReleased}
    {parent: VidyoEndpoint}
    {description: Callback that is triggered to notify application that VidyoClient is done with the buffer that holds raw video frame. This buffer is the same one that was passed to VidyoClient via function VidyoVirtualVideoSourceSendFrameWithExternalData.}
    {prototype: void(*VidyoEndpointOnVirtualVideoSourceExternalMediaBufferReleased)(VidyoEndpoint* e, VidyoVirtualVideoSource* virtualVideoSource, LmiUint8* buffer, LmiSizeT size)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint that triggered the callback.}
    }
    {parameter:
        {name: virtualVideoSource}
        {description: The VidyoVirtualVideoSource object that is associated with this callback.}
    }
    {parameter:
        {name: buffer}
        {description: The buffer (allocated by application that holds raw data of a video frame) that is no longer being used by VidyoClient.}
    }
    {parameter:
        {name: size}
        {description: The size of the buffer.}
    }
}
*/
typedef void(*VidyoEndpointOnVirtualVideoSourceExternalMediaBufferReleased)(VidyoEndpoint* e, VidyoVirtualVideoSource* virtualVideoSource, LmiUint8* buffer, LmiSizeT size);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalCameraFrame}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local camera produces a video frame.}
	{prototype: void (*VidyoEndpointOnLocalCameraFrame)(VidyoEndpoint* e, VidyoLocalCamera* localCamera, const VidyoVideoFrame *videoFrame)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localCamera}
		{description: }
	}
	{parameter:
		{name: videoFrame}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalCameraFrame)(VidyoEndpoint* e, VidyoLocalCamera* localCamera, const VidyoVideoFrame *videoFrame);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalMicrophoneFrame}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local microphone produces an audio frame.}
	{prototype: void (*VidyoEndpointOnLocalMicrophoneFrame)(VidyoEndpoint* e, VidyoLocalMicrophone* localMicrophone, const VidyoAudioFrame *audioFrame)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMicrophone}
		{description: }
	}
	{parameter:
		{name: audioFrame}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalMicrophoneFrame)(VidyoEndpoint* e, VidyoLocalMicrophone* localMicrophone, const VidyoAudioFrame *audioFrame);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalMicrophoneEnergy}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local microphone produces an audio frame with energy present.}
	{prototype: void (*VidyoEndpointOnLocalMicrophoneEnergy)(VidyoEndpoint* e, VidyoLocalMicrophone* localMicrophone, LmiInt audioEnergy)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMicrophone}
		{description: }
	}
	{parameter:
		{name: audioEnergy}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalMicrophoneEnergy)(VidyoEndpoint* e, VidyoLocalMicrophone* localMicrophone, LmiInt audioEnergy);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalMonitorFrame}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local camera produces a video frame.}
	{prototype: void (*VidyoEndpointOnLocalMonitorFrame)(VidyoEndpoint* e, VidyoLocalMonitor* localMonitor, const VidyoVideoFrame *videoFrame)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMonitor}
		{description: }
	}
	{parameter:
		{name: videoFrame}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalMonitorFrame)(VidyoEndpoint* e, VidyoLocalMonitor* localMonitor, const VidyoVideoFrame *videoFrame);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnLocalWindowShareFrame}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a local camera produces a video frame.}
	{prototype: void (*VidyoEndpointOnLocalWindowShareFrame)(VidyoEndpoint* e, VidyoLocalWindowShare* localWindowShare, const VidyoVideoFrame *videoFrame)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localWindowShare}
		{description: }
	}
	{parameter:
		{name: videoFrame}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnLocalWindowShareFrame)(VidyoEndpoint* e, VidyoLocalWindowShare* localWindowShare, const VidyoVideoFrame *videoFrame);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnRemoteCameraFrame}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a remote camera produces a video frame.}
	{prototype: void (*VidyoEndpointOnRemoteCameraFrame)(VidyoEndpoint* e, VidyoRemoteCamera* remoteCamera, const VidyoParticipant *participant, const VidyoVideoFrame *videoFrame)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteCamera}
		{description: }
	}
	{parameter:
		{name: participant}
		{description: }
	}
	{parameter:
		{name: videoFrame}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteCameraFrame)(VidyoEndpoint* e, VidyoRemoteCamera* remoteCamera, const VidyoParticipant *participant, const VidyoVideoFrame *videoFrame);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnRemoteMicrophoneFrame}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a remote camera produces a video frame.}
	{prototype: void (*VidyoEndpointOnRemoteMicrophoneFrame)(VidyoEndpoint* e, VidyoRemoteMicrophone* remoteMicrophone, const VidyoParticipant *participant, const VidyoAudioFrame *audioFrame)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteMicrophone}
		{description: }
	}
	{parameter:
		{name: participant}
		{description: }
	}
	{parameter:
		{name: audioFrame}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteMicrophoneFrame)(VidyoEndpoint* e, VidyoRemoteMicrophone* remoteMicrophone, const VidyoParticipant *participant, const VidyoAudioFrame *audioFrame);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnRemoteMicrophoneEnergy}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a remote camera produces a video frame.}
	{prototype: void (*VidyoEndpointOnRemoteMicrophoneEnergy)(VidyoEndpoint* e, VidyoRemoteMicrophone* remoteMicrophone, const VidyoParticipant *participant, LmiInt audioEnergy)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteMicrophone}
		{description: }
	}
	{parameter:
		{name: participant}
		{description: }
	}
	{parameter:
		{name: audioEnergy}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteMicrophoneEnergy)(VidyoEndpoint* e, VidyoRemoteMicrophone* remoteMicrophone, const VidyoParticipant *participant, LmiInt audioEnergy);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnRemoteWindowShareFrame}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a remote camera produces a video frame.}
	{prototype: void (*VidyoEndpointOnRemoteWindowShareFrame)(VidyoEndpoint* e, VidyoRemoteWindowShare* remoteWindowShare, const VidyoParticipant *participant, const VidyoVideoFrame *videoFrame)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: remoteWindowShare}
		{description: }
	}
	{parameter:
		{name: participant}
		{description: }
	}
	{parameter:
		{name: videoFrame}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnRemoteWindowShareFrame)(VidyoEndpoint* e, VidyoRemoteWindowShare* remoteWindowShare, const VidyoParticipant *participant, const VidyoVideoFrame *videoFrame);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnNetworkInterfaceAdded}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a network interface is available to a VidyoEndpoint.}
	{prototype: void (*VidyoEndpointOnNetworkInterfaceAdded)(VidyoEndpoint* e, VidyoNetworkInterface* networkInterface)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: networkInterface}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnNetworkInterfaceAdded)(VidyoEndpoint* e, VidyoNetworkInterface* networkInterface);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnNetworkInterfaceRemoved}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a network interface is no longer available to a VidyoEndpoint.}
	{prototype: void (*VidyoEndpointOnNetworkInterfaceRemoved)(VidyoEndpoint* e, VidyoNetworkInterface* networkInterface)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: networkInterface}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnNetworkInterfaceRemoved)(VidyoEndpoint* e, VidyoNetworkInterface* networkInterface);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnNetworkInterfaceSelected}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time there is a change in which network interface is selected for use.}
	{prototype: void (*VidyoEndpointOnNetworkInterfaceSelected)(VidyoEndpoint* e, VidyoNetworkInterface* networkInterface, VidyoNetworkInterfaceTransportType transportType)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: networkInterface}
		{description: }
	}
	{parameter:
		{name: transportType}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnNetworkInterfaceSelected)(VidyoEndpoint* e, VidyoNetworkInterface* networkInterface, VidyoNetworkInterfaceTransportType transportType);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnNetworkInterfaceStateUpdated}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time there is a change in the state of the network interface such as being up or down.}
	{prototype: void (*VidyoEndpointOnNetworkInterfaceStateUpdated)(VidyoEndpoint* e, VidyoNetworkInterface* networkInterface, VidyoNetworkInterfaceState state)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: networkInterface}
		{description: }
	}
	{parameter:
		{name: state}
		{description: }
	}
}
*/
typedef void(*VidyoEndpointOnNetworkInterfaceStateUpdated)(VidyoEndpoint* e, VidyoNetworkInterface* networkInterface, VidyoNetworkInterfaceState state);

/**
{type apigroup="simple":
	{name: VidyoEndpointReconnectFailReason}
	{parent: VidyoEndpoint}
	{description: This type indicates the reason why an attempt to reconnect failed.}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_Cancelled} {description: Media support was disabled (by calling VidyoRoomDisableMedia) before an attempt to enable it completed.}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_NoConference} {description: There was an error determining which conference to join.}}
	{value:{name: VIDYO_ENDPOINTRECONNECTFAILREASON_SignalingConnectionLost} {description: Signaling Connection lost on current transport.}}
	{value:{name: VIDYO_ENDPOINTRECONNECTFAILREASON_MediaConnectionLost} {description: Media Connection lost on current transport.}}
	{value:{name: VIDYO_ENDPOINTRECONNECTFAILREASON_NoResponse} {description: Signaling messages were sent to enable media, but no response was received within the allotted amount of time.}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_InvalidPassword} {description: Media support was disabled because the room password has changed.}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_InvalidRoomPin} {description: Media support was disabled because the room pin has changed.}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_RoomIsFull} {description: Media support was disabled because the room is full.}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_InvalidRoomKey} {description: Media support was disabled because the room is invalid.}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_RoomDisabled} {description: Media support was disabled because the room is disabled.}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_AllLinesInUse} {description: Media support was disabled because all lines are in use.}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_ConnectionError} {description: Media support was disabled because the of a connection error.}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_InvalidRoom} {description: }}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_ConferenceLocked} {description: Media support was disabled because the room is locked.}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_SeatLicenseExpired} {description: Media support was disabled because the seat license has expired.}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_NotLicensed} {description: Media support was disabled because the endpoint is not licensed.}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_Rejected} {description: Attempt to enable media was rejected by the server}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_UnknownError} {description: Media support was disabled as there an unknown fault from the portal.}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_MiscLocalError} {description: Some miscellaneous problem occurred with this application.}}
	{value: {name: VIDYO_ENDPOINTRECONNECTFAILREASON_MiscRemoteError} {description: Some miscellaneous problem occurred with the conferencing service.}}
}
*/
typedef enum
{
	VIDYO_ENDPOINTRECONNECTFAILREASON_Cancelled,
	VIDYO_ENDPOINTRECONNECTFAILREASON_NoConference,
	VIDYO_ENDPOINTRECONNECTFAILREASON_SignalingConnectionLost,
	VIDYO_ENDPOINTRECONNECTFAILREASON_MediaConnectionLost,
	VIDYO_ENDPOINTRECONNECTFAILREASON_NoResponse,
	VIDYO_ENDPOINTRECONNECTFAILREASON_InvalidPassword,
	VIDYO_ENDPOINTRECONNECTFAILREASON_InvalidRoomPin,
	VIDYO_ENDPOINTRECONNECTFAILREASON_RoomIsFull,
	VIDYO_ENDPOINTRECONNECTFAILREASON_InvalidRoomKey,
	VIDYO_ENDPOINTRECONNECTFAILREASON_RoomDisabled,
	VIDYO_ENDPOINTRECONNECTFAILREASON_AllLinesInUse,
	VIDYO_ENDPOINTRECONNECTFAILREASON_ConnectionError,
	VIDYO_ENDPOINTRECONNECTFAILREASON_InvalidRoom,
	VIDYO_ENDPOINTRECONNECTFAILREASON_ConferenceLocked,
	VIDYO_ENDPOINTRECONNECTFAILREASON_SeatLicenseExpired,
	VIDYO_ENDPOINTRECONNECTFAILREASON_NotLicensed,
	VIDYO_ENDPOINTRECONNECTFAILREASON_Rejected,
	VIDYO_ENDPOINTRECONNECTFAILREASON_UnknownError,
	VIDYO_ENDPOINTRECONNECTFAILREASON_MiscLocalError,
	VIDYO_ENDPOINTRECONNECTFAILREASON_MiscRemoteError
} VidyoEndpointReconnectFailReason;
/**
 {callback:
	{name: VidyoEndpointOnReconnecting}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered when the reconnecting attempt has failed and new attempt will be started. }
	{prototype: void (*VidyoEndpointOnReconnecting)(VidyoEndpoint *e, LmiUint attempt, LmiUint attemptTimeout,  VidyoEndpointReconnectFailReason reason)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object that triggered the callback.}
	}
	{parameter:
		{name: attempt}
		{description: The attempt number.}
	}
	{parameter:
		{name: attemptTimeout}
		{description: The time in seconds after which attempt to reconnect will start..}
	}
	{parameter:
		{name: reason}
		{description: The previous reason of reconnecting failure.}
	}
}
*/
typedef void(*VidyoEndpointOnReconnecting)(VidyoEndpoint *e, LmiUint attempt, LmiUint attemptTimeout, VidyoEndpointReconnectFailReason reason);

/**
 {callback:
	{name: VidyoEndpointOnReconnected}
	{parent: VidyoEndpoint}
	{description: This callback will be triggered when media was enabled and the user rejoined to the call/conference. }
	{prototype: void (*VidyoEndpointOnReconnected)(VidyoEndpoint *e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object that triggered the callback.}
	}
}
*/
typedef void(*VidyoEndpointOnReconnected)(VidyoEndpoint* e);

/**
 {callback:
	{name: VidyoEndpointOnConferenceLost}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered when reconnecting failed and there no more attempt to reconnect. }
	{prototype: void (*VidyoEndpointOnConferenceLost)(VidyoEndpoint* e, VidyoEndpointReconnectFailReason reason)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object that triggered the callback.}
	}
	{parameter:
		{name: reason}
		{description: Reason why the established connection became disconnected.}
	}
}
*/
typedef void(*VidyoEndpointOnConferenceLost)(VidyoEndpoint* e, VidyoEndpointReconnectFailReason reason);

/**
 {callback:
	{name: VidyoEndpointOnRemoteCameraFirstFrameReceived}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time the first frame for remote camera received.}
	{prototype: void(*VidyoEndpointOnRemoteCameraFirstFrameReceived)(VidyoEndpoint* e, VidyoRemoteCamera* remoteCamera)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint that triggered the callback.}
	}
	{parameter:
		{name: remoteCamera}
		{description: Remote camera which frame was received for.}
	}
}
*/
typedef void(*VidyoEndpointOnRemoteCameraFirstFrameReceived)(VidyoEndpoint* e, VidyoRemoteCamera* remoteCamera);

/**
 {callback:
	{name: VidyoEndpointOnRemoteWindowShareFirstFrameReceived}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time the first frame for remote window share received.}
	{prototype: void(*VidyoEndpointOnRemoteWindowShareFirstFrameReceived)(VidyoEndpoint* e, VidyoRemoteWindowShare* remoteWindowShare)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint that triggered the callback.}
	}
	{parameter:
		{name: remoteWindowShare}
		{description: Remote window share which frame was received for.}
	}
}
*/
typedef void(*VidyoEndpointOnRemoteWindowShareFirstFrameReceived)(VidyoEndpoint* e, VidyoRemoteWindowShare* remoteWindowShare);

/**
 {callback:
	{name: VidyoEndpointOnGetWhitelistedAudioDevices}
	{parent: VidyoEndpoint}
	{description: Callback that reports the names of the audio devices that are whitelisted.}
	{prototype: void(*VidyoEndpointOnGetWhitelistedAudioDevices)(VidyoEndpoint* e, const LmiVector(LmiString)* audioDevices)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint that triggered the callback.}
	}
	{parameter:
		{name: audioDevices}
		{description: The array with the names of the audio devices that are whitelisted.}
	}
}
*/
typedef void(*VidyoEndpointOnGetWhitelistedAudioDevices)(VidyoEndpoint* e, const LmiVector(LmiString)* audioDevices);

/**
{type:
	{name: VidyoEndpointMediaState}
	{parent: VidyoEndpoint}
	{description: }
	{value: {name: VIDYO_ENDPOINTMEDIASTATE_Idle} {description: }}
	{value: {name: VIDYO_ENDPOINTMEDIASTATE_GreenRoom} {description: }}
	{value: {name: VIDYO_ENDPOINTMEDIASTATE_Live} {description: }}
}
*/
typedef enum {
	VIDYO_ENDPOINTMEDIASTATE_Idle,
	VIDYO_ENDPOINTMEDIASTATE_GreenRoom,
	VIDYO_ENDPOINTMEDIASTATE_Live
} VidyoEndpointMediaState;

/**
{function apigroup="simple":
	{name: VidyoClientInitialize}
	{parent: Endpoint}
	{description: }
	{prototype: LmiBool VidyoClientInitialize(void)}
	{return: }
}
*/
LmiBool VidyoClientInitialize(void);

/**
{function apigroup="simple":
	{name: VidyoClientUninitialize}
	{parent: Endpoint}
	{description: }
	{prototype: void VidyoClientUninitialize(void)}
}
*/
void VidyoClientUninitialize(void);

/**
{function apigroup="simple":
	{name: VidyoClientSetExperimentalOptions}
	{parent: Endpoint}
	{description: }
	{prototype: LmiBool VidyoClientSetExperimentalOptions(const char *options)}
	{parameter:
		{name: options}
		{description: The experimental optons in JSON format.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoClientSetExperimentalOptions(const char *options);

/**
{function apigroup="simple":
	{name: VidyoClientGetVersion}
	{parent: Endpoint}
	{description: }
	{prototype: const LmiString *VidyoClientGetVersion(void)}
	{return: Version of the VidyoClient library. }
}
*/
const LmiString *VidyoClientGetVersion(void);

/**
{function apigroup="simple":
	{name: VidyoClientGetVersionWithoutBuildNumber}
	{parent: Endpoint}
	{description: }
	{prototype: const LmiString *VidyoClientGetVersionWithoutBuildNumber(void)}
	{return: Version of the VidyoClient library. }
}
*/
const LmiString *VidyoClientGetVersionWithoutBuildNumber(void);

/**
{function apigroup="simple":
	{name: VidyoClientGetBuildTag}
	{parent: Endpoint}
	{description: }
	{prototype: const LmiString *VidyoClientGetBuildTag(void)}
	{return: Build tag of the VidyoClient library. }
}
*/
const LmiString *VidyoClientGetBuildTag(void);

/**
{function apigroup="simple":
	{name: VidyoEndpointConstruct}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: VidyoEndpoint* VidyoEndpointConstruct(VidyoEndpoint* e, const LmiViewId* viewId, VidyoLocalRendererViewStyle viewStyle, LmiUint remoteParticipants, const char* consoleLogFilter, const char* logFileFilter, const char* logFileName, LmiAllocator* alloc)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: viewId}
		{description: A platform specific parent view ID where the VidyoEndpoint's rendering window will be added as a child window. It will be used to render preview and composite remote participants. When running custom layout on macOS, the core animation layer should be disabled on this view.}
	}
	{parameter:
		{name: viewStyle}
		{description: Type of the composite renderer which represents the visual style and behaviour. }
	}
	{parameter:
		{name: remoteParticipants}
		{description: Number of remote participants to composite into the window. Setting the value to 0 will render preview only. }
	}
	{parameter:
		{name: consoleLogFilter}
		{description: }
	}
	{parameter:
		{name: logFileFilter}
		{description: }
	}
	{parameter:
		{name: logFileName}
		{description: }
	}
	{parameter:
		{name: alloc}
		{description: }
	}
	{return: }
	{note: There are two mutually exclusive alternatives to rendering, composited and custom. A composited renderer is used when the viewId is passed to this constructor or when VidyoEndpointAssignViewToCompositeRenderer() is called later. Custom rendring allows configurable layouts where each source is rendered into a view provided by the application. It uses VidyoEndpointAssignViewToLocalCamera(), VidyoEndpointAssignViewToRemoteCamera() etc. APIs. However, once the choice is made, the other cannot be used. }
	{note: If viewId is valid the device detection will start implicitly and default devices will be selected, otherwise the viewId can be assigned later using VidyoEndpointAssignViewToCompositeRenderer() but the device detection will not start until the appropriate event listeners are registered such as VidyoEndpointRegisterLocalCameraEventListener(). }
	{note: LmiViewId is HWND on windows, NSView on mac, UIView on iOS, window on X11, ViewGroup on android.}
}
*/
VidyoEndpoint *VidyoEndpointConstruct(VidyoEndpoint *e, const LmiViewId* viewId, VidyoLocalRendererViewStyle viewStyle, LmiUint remoteParticipants, const char *consoleLogFilter, const char *logFileFilter, const char *logFileName, LmiAllocator *alloc);

/**
{function apigroup="simple":
	{name: VidyoEndpointConstructCopy}
	{parent: VidyoEndpoint}
	{description: Constructs an VidyoEndpoint object as a copy of another.}
	{prototype: VidyoEndpoint *VidyoEndpointConstructCopy(VidyoEndpoint *obj, const VidyoEndpoint *other)}
	{parameter: {name: obj} {description: The VidyoEndpoint object to construct.}}
	{parameter: {name: other} {description: The call object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/

/**
{function apigroup="simple":
	{name: VidyoEndpointDestruct}
	{parent: VidyoEndpoint}
	{description: Destructs an VidyoEndpoint object.}
	{prototype: void VidyoEndpointDestruct(VidyoEndpoint *obj)}
	{parameter: {name: obj} {description: The VidyoEndpoint object to destruct.}}
}
*/

/**
{function visibility="private":
	{name: VidyoEndpointAssign}
	{parent: VidyoEndpoint}
	{description: Assigns one VidyoEndpoint object the value of another. }
	{prototype: VidyoEndpoint *VidyoEndpointAssign(VidyoEndpoint *obj, const VidyoEndpoint *other)}
	{parameter: {name: obj} {description: The VidyoEndpoint object.}}
	{parameter: {name: other} {description: The call object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/

/**
{function visibility="private":
	{name: VidyoEndpointGetHandle}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: VidyoEndpointHandle* VidyoEndpointGetHandle(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
	{return: }
}
*/
VidyoEndpointHandle* VidyoEndpointGetHandle(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointStartLocalCameraDetection}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointStartLocalCameraDetection(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointStartLocalCameraDetection(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointStartLocalMicrophoneDetection}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointStartLocalMicrophoneDetection(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointStartLocalMicrophoneDetection(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointStartLocalSpeakerDetection}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointStartLocalSpeakerDetection(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointStartLocalSpeakerDetection(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointStartLocalWindowShareDetection}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointStartLocalWindowShareDetection(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointStartLocalWindowShareDetection(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointStartLocalMonitorDetection}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointStartLocalMonitorDetection(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointStartLocalMonitorDetection(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointStopLocalCameraDetection}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointStopLocalCameraDetection(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
}
*/
void VidyoEndpointStopLocalCameraDetection(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointStopLocalMicrophoneDetection}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointStopLocalMicrophoneDetection(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
}
*/
void VidyoEndpointStopLocalMicrophoneDetection(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointStopLocalSpeakerDetection}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointStopLocalSpeakerDetection(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
}
*/
void VidyoEndpointStopLocalSpeakerDetection(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointStopLocalWindowShareDetection}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointStopLocalWindowShareDetection(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
}
*/
void VidyoEndpointStopLocalWindowShareDetection(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointStopLocalMonitorDetection}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointStopLocalMonitorDetection(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
}
*/
void VidyoEndpointStopLocalMonitorDetection(VidyoEndpoint* e);

/**
{function visibility="private":
	{name: VidyoEndpointSetUserData}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: void VidyoEndpointSetUserData(VidyoEndpoint* e, LmiVoidPtr userData)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: userData}
		{description: }
	}
}
*/
void VidyoEndpointSetUserData(VidyoEndpoint* e, LmiVoidPtr userData);

/**
{function visibility="private":
	{name: VidyoEndpointGetUserData}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiVoidPtr VidyoEndpointGetUserData(const VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
	{return: }
}
*/
LmiVoidPtr VidyoEndpointGetUserData(const VidyoEndpoint* e);

/**
 {type apigroup="simple":
	{name: VidyoEndpointMode}
	{parent: VidyoEndpoint}
	{description: Operating modes of the VidyoEndpoint.}
	{value: {name: VIDYO_ENDPOINTMODE_Default}
		{description: The default operating mode.}}
	{value: {name: VIDYO_ENDPOINTMODE_Foreground}
		{description: Runs in the foreground.}}
	{value: {name: VIDYO_ENDPOINTMODE_Background}
		{description: Runs in the background.}}
	{note: The default is currently VIDYO_ENDPOINTMODE_Foreground.}
}
*/
typedef enum
{
	VIDYO_ENDPOINTMODE_Default,
	VIDYO_ENDPOINTMODE_Foreground,
	VIDYO_ENDPOINTMODE_Background
} VidyoEndpointMode;

/**
 {function apigroup="simple":
	{name: VidyoEndpointSetMode}
	{parent: VidyoEndpoint}
	{description: Sets the operating mode of the VidyoEndpoint. This API have to be called before renderer is destructed: i.e. VidyoEndpointHideView will be called in case VIDYO_ENDPOINTMODE_Background and after renderer is constructed: i.e. VidyoEndpointAssignViewToCompositeRenderer, VidyoEndpointAssignViewToLocalCamera, VidyoEndpointAssignViewToRemoteCamera will be called in case VIDYO_ENDPOINTMODE_Foreground. This API is required only when user in the conference. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSetMode(VidyoEndpoint* e, VidyoEndpointMode mode)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: mode}
		{description: Desired operating mode.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetMode(VidyoEndpoint *e, VidyoEndpointMode mode);

/**
 {function apigroup="simple":
	{name: VidyoEndpointAssignViewToCompositeRenderer}
	{parent: VidyoEndpoint}
	{description: Composites the preview and remote participants in a view. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointAssignViewToCompositeRenderer(VidyoEndpoint *e, const LmiViewId* viewId, VidyoLocalRendererViewStyle viewStyle, LmiUint remoteParticipants)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID where the participant will be rendered.}
	}
	{parameter:
		{name: viewStyle}
		{description: Type of the composite renderer which represents the visual style and behaviour. }
	}
	{parameter:
		{name: remoteParticipants}
		{description: Number of remote participants to composite into the window. Setting the value to 0 will render preview only. }
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: This API will work only if custom rendering has not been used. ex: VidyoEndpointAssignViewToLocalCamera() was never called.}
	{note: The view will not be shown until VidyoEndpointShowViewAt is called with valid parameters.}
}
*/
LmiBool VidyoEndpointAssignViewToCompositeRenderer(VidyoEndpoint *e, const LmiViewId* viewId, VidyoLocalRendererViewStyle viewStyle, LmiUint remoteParticipants);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterLogEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about log events.}
	{prototype: LmiBool VidyoEndpointRegisterLogEventListener(VidyoEndpoint* e, VidyoEndpointOnLog onLog, const char* filter)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onLog}
		{description: Callback that is triggered for every new log record.}
	}
	{parameter:
		{name: filter}
		{description:
	   {p: A space-separated (or comma-separated)
		 sequence of names of log levels, each optionally followed by a
		 category.  Categories are
		 separated from levels by the character \'@\'.}
	   {p: Either a level or category may be specified as \'all\' or \'*\'
	     (interchangeably), meaning all levels or categories.  A level specified
		 without a category is equivalent to a level@*.}
	   {p: The levels are: \'fatal\', \'error\', \'warning\', \'info\', \'debug\', 
		 \'sent\', \'received\', \'enter\', and '\leave\'.}
	   {p: The level may be prefixed by \'-\' or \'!\' (interchangeably), meaning to remove
		 the given level (or levels, see below) from the given category.
		 The string \'none\' is equivalent to \'-all\'.}
	   {p: The first level in the list may be prefixed by \'+\'.
	     If the first level begins with a \'+\' or \'-\'/\'!\', the string
		 is used to modify the listener\'s existing levels and 45categories.
		 Otherwise, unless the {code: add} parameter is true, the levels and categories 
 		 specified in the string replace all existing levels and categories for the listener.  
 		 The leading character \'+\' may be present before other levels, but is ignored.}
	   {p: Unless they are preceded by \'=\', level names imply other levels
		 as well.  In particular, each of the levels \'fatal\', \'error\', \'warning\',
		 \'info\', and \'debug\' implies the levels higher than it, when specified
		 positively; \'sent\' and \'received\' always imply each other; and \'enter\' and
		 \'leave\' always imply each other.}
	   {p: See VidyoEndpointGetLogCategories for determining the names and descriptions of registered categories.}
	   {p: Level names are case-insensitive; category names are case-sensitive.}
	   {p: Example: "all" - all levels, all categories.}
	   {p: Example: "all -enter" - all categories, all levels but enter and leave.}
	   {p: Example: "*@VidyoClient" - all levels for the VidyoClient category.}
	   {p: Example: "-*@VidyoClient" - remove all levels from the VidyoClient category.}
	   {p: Example: "debug" - all categories, fatal, error, warning, info, and debug levels.}
	   {p: Example: "=debug" - all categories, debug level only.}
	   {p: Example: "all -=enter" - all categories, all levels but enter (leave is still included).}
	   {p: Example: "error@VidyoClient" - fatal and error only for the VidyoClient category.}
	   {p: Example: "warning debug@VidyoClient" - fatal, error, and warning for all categories; additionally, info and debug for the VidyoClient category.}
	}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterLogEventListener(VidyoEndpoint* e, VidyoEndpointOnLog onLog, const char* filter);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterLogEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters log event notifications.}
	{prototype: LmiBool VidyoEndpointUnregisterLogEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterLogEventListener(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterLocalCameraEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about camera events. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointRegisterLocalCameraEventListener(VidyoEndpoint* e, VidyoEndpointOnLocalCameraAdded onAdded, VidyoEndpointOnLocalCameraRemoved onRemoved, VidyoEndpointOnLocalCameraSelected onSelected, VidyoEndpointOnLocalCameraStateUpdated onStateUpdated)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered for every new or existing camera added.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when the camera is removed.}
	}
	{parameter:
		{name: onSelected}
		{description: Callback that is triggered when a camera is selected for use.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when a camera state is updated.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterLocalCameraEventListener(VidyoEndpoint* e, VidyoEndpointOnLocalCameraAdded onAdded, VidyoEndpointOnLocalCameraRemoved onRemoved, VidyoEndpointOnLocalCameraSelected onSelected, VidyoEndpointOnLocalCameraStateUpdated onStateUpdated);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterLocalCameraEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters local camera event notifications. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointUnregisterLocalCameraEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterLocalCameraEventListener(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterLocalCameraFrameListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about camera frames. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointRegisterLocalCameraFrameListener(VidyoEndpoint* e, const VidyoLocalCamera* localCamera, VidyoEndpointOnLocalCameraFrame onFrame, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: localCamera}
		{description: The VidyoLocalCamera for which to receive the frames.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered for every frame from a remote camera.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested frame.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested frame.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested frame.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: The extra parameters are used to determin the steam quality and the resulting frame might not match exactly what is requested.}
}
*/
LmiBool VidyoEndpointRegisterLocalCameraFrameListener(VidyoEndpoint* e, const VidyoLocalCamera* localCamera, VidyoEndpointOnLocalCameraFrame onFrame, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterLocalCameraFrameListener}
	{parent: VidyoEndpoint}
	{description: Unregisters local camera frame notifications. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointUnregisterLocalCameraFrameListener(VidyoEndpoint* e, const VidyoLocalCamera* localCamera)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: localCamera}
		{description: The VidyoLocalCamera for which to stop notifications.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterLocalCameraFrameListener(VidyoEndpoint* e, const VidyoLocalCamera* localCamera);

/**
{type visibility="public":
	{name: VidyoEndpointGoogleAnalyticsEventCategory}
	{parent: VidyoEndpoint}
	{description: Represents the event category of google analytics service.}
	{value:
		{name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTCATEGORY_Login}
		{description: The login event category.}}
	{value:
		{name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTCATEGORY_UserType}
		{description: The user type event category.}}
	{value:
		{name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTCATEGORY_JoinConference}
		{description: The join Conference event category.}}
	{value:
		{name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTCATEGORY_ConferenceEnd}
		{description: The conference end event category.}}
	{value:
		{name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTCATEGORY_InCallCodec}
		{description: The in-call codec event category.}}
	{value:
		{name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTCATEGORY_None}
		{description: Unknown event category.}}
}
*/
typedef enum {
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTCATEGORY_Login,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTCATEGORY_UserType,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTCATEGORY_JoinConference,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTCATEGORY_ConferenceEnd,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTCATEGORY_InCallCodec,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTCATEGORY_None
} VidyoEndpointGoogleAnalyticsEventCategory;

/**
{type visibility="public":
    {name: VidyoEndpointGoogleAnalyticsEventAction}
    {parent: VidyoEndpoint}
    {description: Represents the event action of google analytics service.}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginSuccess}
        {description: Action for success login.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginAttempt}
        {description: Action for login attempt.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginFailedAuthentication}
        {description: Action for login failed: authentication failed.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginFailedConnect}
        {description: Action for login failed: failed to connect.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginFailedResponseTimeout}
        {description: Action for login failed: response timeout.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginFailedMiscError}
        {description: Action for login failed: misc error.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginFailedWebProxyAuthRequired}
        {description: Action for login failed: webproxy authentication required.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginFailedUnsupportedTenantVersion}
        {description: Action for login failed: unsupported tenant version.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_UserTypeGuest}
        {description: Action to indicate if user type is guest.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_UserTypeRegularToken}
        {description: Action to indicate if user type is regular: token.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_UserTypeRegularPassword}
        {description: Action to indicate if user type is regular: password.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_UserTypeRegularSaml}
        {description: Action to indicate if user type is regular: saml.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_UserTypeRegularExtdata}
        {description: Action to indicate if user type is regular: extdata.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceSuccess}
        {description: Action to indicate join conference success.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceAttempt}
        {description:  Action to indicate join conference attempt.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceReconnectRequests}
        {description: Action to indicate join conference reconnect requests.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedConnectionError}
        {description: Action to indicate join conference failed due to connection error.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedWrongPin}
        {description: Action to indicate join conference failed due to wrong Pin.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedRoomFull}
        {description: Action to indicate join conference failed because room  is full.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedRoomDisabled}
        {description: Action to indicate join conference failed because room disabled.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedConferenceLocked}
        {description: Action to indicate join conference failed because conference is locked.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedUnknownError}
        {description: Action to indicate join conference failed due to unknown error.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_ConferenceEndLeft}
        {description: Action to indicate conference left.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_ConferenceEndBooted}
        {description: Action to indicate conference end booted.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_ConferenceEndSignalingConnectionLost}
        {description: Action to indicate  conference end signaling connection lost.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_ConferenceEndMediaConnectionLost}
        {description: Action to indicate conference end failed media connection lost.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_ConferenceEndUnknownError}
        {description: Action to indicate conference end unknown error.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_InCallCodecVideoH264}
        {description: Action to indicate H264 video codec is used in a conference call.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_InCallCodecVideoH264SVC}
        {description: Action to indicate H264-SVC video codec is used in a conference call.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_InCallCodecAudioSPEEXRED}
        {description: Action to indicate SPEEX RED audio codec is used in a conference call.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_All}
        {description: All actions in category should be sent.}}
    {value:
        {name: VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_Unknown}
        {description: Unknown event action.}}
}
*/
typedef enum
{
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginSuccess,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginAttempt,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginFailedAuthentication,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginFailedConnect,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginFailedResponseTimeout,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginFailedMiscError,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginFailedWebProxyAuthRequired,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_LoginFailedUnsupportedTenantVersion,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_UserTypeGuest,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_UserTypeRegularToken,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_UserTypeRegularPassword,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_UserTypeRegularSaml,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_UserTypeRegularExtdata,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceSuccess,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceAttempt,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceReconnectRequests,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedConnectionError,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedWrongPin,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedRoomFull,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedRoomDisabled,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedConferenceLocked,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedUnknownError,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_ConferenceEndLeft,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_ConferenceEndBooted,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_ConferenceEndSignalingConnectionLost,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_ConferenceEndMediaConnectionLost,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_ConferenceEndUnknownError,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_InCallCodecVideoH264,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_InCallCodecVideoH264SVC,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_InCallCodecAudioSPEEXRED,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_All,
    VIDYO_ENDPOINTGOOGLEANALYTICSEVENTACTION_Unknown
} VidyoEndpointGoogleAnalyticsEventAction;

/**
{type:
    {name: VidyoEndpointGoogleAnalyticsEventTable}
    {parent: VidyoEndpoint}
    {description: The event action from event table.}
    {member:
        {name: eventCategory}
        {type: VidyoEndpointGoogleAnalyticsEventCategory}
        {description: The event category.}
    }
    {member:
        {name: eventAction}
        {type: VidyoEndpointGoogleAnalyticsEventAction}
        {description: The event action.}
    }
    {member:
        {name: enable}
        {type: LmiBool}
        {description: An enable flag for action.}
    }
}
*/
typedef struct
{
    VidyoEndpointGoogleAnalyticsEventCategory eventCategory;
    VidyoEndpointGoogleAnalyticsEventAction eventAction;
    LmiBool enable;
} VidyoEndpointGoogleAnalyticsEventTable;


/**
{function visibility="private":
    {name: VidyoEndpointGoogleAnalyticsEventTableConstructDefault}
    {parent: VidyoEndpointGoogleAnalyticsEventTable}
    {description: Constructs default event table object.}
    {prototype: VidyoEndpointGoogleAnalyticsEventTable *VidyoEndpointGoogleAnalyticsEventTableConstructDefault(VidyoEndpointGoogleAnalyticsEventTable* eventTable, LmiAllocator* alloc)}
    {parameter:
        {name: eventTable}
        {description: Table record of correspondings action.}
    }
    {parameter:
        {name: alloc}
        {description: LmiAllocator object.}
    }
    {return: Pointer to the assigned object on success, or NULL on failure.}
}
*/
VidyoEndpointGoogleAnalyticsEventTable* VidyoEndpointGoogleAnalyticsEventTableConstructDefault(VidyoEndpointGoogleAnalyticsEventTable* eventTable, LmiAllocator* alloc);

/**
{function visibility="private":
    {name: VidyoEndpointGoogleAnalyticsEventTableConstructCopy}
    {parent: VidyoEndpointGoogleAnalyticsEventTable}
    {description: Constructs copy of event table object.}
    {prototype: VidyoEndpointGoogleAnalyticsEventTable *VidyoEndpointGoogleAnalyticsEventTableConstructCopy(VidyoEndpointGoogleAnalyticsEventTable* d, const VidyoEndpointGoogleAnalyticsEventTable* s)}
    {parameter:
        {name: d}
        {description: The object to construct.}
    }
    {parameter:
        {name: s}
        {description: The call object to make a copy of.}
    }
    {return: Pointer to the assigned object on success, or NULL on failure.}
}
*/
VidyoEndpointGoogleAnalyticsEventTable* VidyoEndpointGoogleAnalyticsEventTableConstructCopy(VidyoEndpointGoogleAnalyticsEventTable* d, const VidyoEndpointGoogleAnalyticsEventTable* s);

/**
{function visibility="private":
    {name: VidyoEndpointGoogleAnalyticsEventTableAssign}
    {parent: VidyoEndpointGoogleAnalyticsEventTable}
    {description: Assigns one event table object the value of another.}
    {prototype: VidyoEndpointGoogleAnalyticsEventTable *VidyoEndpointGoogleAnalyticsEventTableAssign(VidyoEndpointGoogleAnalyticsEventTable* d, const VidyoEndpointGoogleAnalyticsEventTable* s)}
    {parameter:
        {name: d}
        {description: The object to be constructed.}
    }
    {parameter:
        {name: s}
        {description: The call object to make a copy of.}
    }
    {return: Pointer to the assigned object on success, or NULL on failure.}
}
*/
VidyoEndpointGoogleAnalyticsEventTable *VidyoEndpointGoogleAnalyticsEventTableAssign(VidyoEndpointGoogleAnalyticsEventTable* d, const VidyoEndpointGoogleAnalyticsEventTable* s);

/**
{function visibility="private":
    {name: VidyoEndpointGoogleAnalyticsEventTableDestruct}
    {parent: VidyoEndpointGoogleAnalyticsEventTable}
    {description: Destructs event table object.}
    {prototype: void VidyoEndpointGoogleAnalyticsEventTableDestruct(VidyoEndpointGoogleAnalyticsEventTable* eventTable)}
    {parameter:
        {name: eventTable}
        {description: The object to destruct.}
    }
}
*/
void VidyoEndpointGoogleAnalyticsEventTableDestruct(VidyoEndpointGoogleAnalyticsEventTable* eventTable);

Declare_LmiVector(VidyoEndpointGoogleAnalyticsEventTable)

/**
{callback apigroup="simple":
    {name: VidyoEndpointOnGetAnalyticsEventTable}
    {parent: VidyoEndpoint}
    {description: Callback that is triggered each time when result of VidyoEndpointGetGoogleAnalyticsEventTable is received.}
    {prototype: void(*VidyoEndpointOnGetAnalyticsEventTable)(VidyoEndpoint* e, const LmiVector(VidyoEndpointGoogleAnalyticsEventTable)* eventTables)}
    {parameter:
        {name: e}
        {description: Pointer to the VidyoEndpoint that triggered the callback.}}
    {parameter:
        {name: eventTables}
        {description: Event table records for correspondings actions.}}
}
*/
typedef void(*VidyoEndpointOnGetAnalyticsEventTable)(VidyoEndpoint* e, const LmiVector(VidyoEndpointGoogleAnalyticsEventTable)* eventTables);

/**
{type:
	{name:  VidyoEndpointGoogleAnalyticsOptions}
	{parent: VidyoEndpoint}
	{description: Represent the options for google analytics.}
	{member: {name: id} {type: LmiString} {description: Represents the measurement is for google analytics service.}}
	{member: {name: key} {type: LmiString} {description: Represents the configured API secret key for google analytics service.}}
}
*/
typedef struct
{
	LmiString id;
	LmiString key;
} VidyoEndpointGoogleAnalyticsOptions;

/**
{function visibility="private":
    {name: VidyoEndpointGoogleAnalyticsOptionsConstructDefault}
    {parent: VidyoEndpointGoogleAnalyticsOptions}
    {description: Constructs the default VidyoEndpointGoogleAnalyticsOptions object.}
    {prototype: VidyoEndpointGoogleAnalyticsOptions* VidyoEndpointGoogleAnalyticsOptionsConstructDefault(VidyoEndpointGoogleAnalyticsOptions* options, LmiAllocator* alloc)}
    {parameter:
        {name: options}
        {description: Represents the VidyoEndpointGoogleAnalyticsOptions object to construct.}
    }
    {parameter:
        {name: alloc}
        {description: Represents the LmiAllocator object.}
    }
    {return: Pointer to the assigned object on success, or NULL on failure.}
}
*/
VidyoEndpointGoogleAnalyticsOptions* VidyoEndpointGoogleAnalyticsOptionsConstructDefault(VidyoEndpointGoogleAnalyticsOptions* options, LmiAllocator* alloc);

/**
{function visibility="private":
    {name: VidyoEndpointGoogleAnalyticsOptionsConstructCopy}
    {parent: VidyoEndpointGoogleAnalyticsOptions}
    {description: Constructs an VidyoEndpointGoogleAnalyticsOptions object as a copy of an existing object.}
    {prototype: VidyoEndpointGoogleAnalyticsOptions* VidyoEndpointGoogleAnalyticsOptionsConstructCopy(VidyoEndpointGoogleAnalyticsOptions* d, const VidyoEndpointGoogleAnalyticsOptions* s)}
    {parameter:
        {name: d}
        {description: The object to construct.}
    }
    {parameter:
        {name: s}
        {description: The object to make copy of.}
    }
    {return: The pointer to constructed object on success, or NULL on failure}
}
*/
VidyoEndpointGoogleAnalyticsOptions* VidyoEndpointGoogleAnalyticsOptionsConstructCopy(VidyoEndpointGoogleAnalyticsOptions* d, const VidyoEndpointGoogleAnalyticsOptions* s);

/**
{function visibility="private":
    {name: VidyoEndpointGoogleAnalyticsOptionsAssign}
    {parent: VidyoEndpointGoogleAnalyticsOptions}
    {description: Assigns one VidyoEndpointGoogleAnalyticsOptions object the value of another.}
    {prototype: VidyoEndpointGoogleAnalyticsOptions* VidyoEndpointGoogleAnalyticsOptionsAssign(VidyoEndpointGoogleAnalyticsOptions* d, const VidyoEndpointGoogleAnalyticsOptions* s)}
    {parameter:
        {name: d}
        {description: The object to assign to.}
    }
    {parameter:
        {name: s}
        {description: The object to assign from.}
    }
    {return: The pointer to constructed object on success, or NULL on failure}
}
*/
VidyoEndpointGoogleAnalyticsOptions* VidyoEndpointGoogleAnalyticsOptionsAssign(VidyoEndpointGoogleAnalyticsOptions* d, const VidyoEndpointGoogleAnalyticsOptions* s);

/**
{function visibility="private":
    {name: VidyoEndpointGoogleAnalyticsOptionsDestruct}
    {parent: VidyoEndpointGoogleAnalyticsOptions}
    {description: Destructs an VidyoEndpointGoogleAnalyticsOptions object.}
    {prototype: void VidyoEndpointGoogleAnalyticsOptionsDestruct(VidyoEndpointGoogleAnalyticsOptions* options)}
    {parameter:
        {name: options}
        {description: The VidyoEndpointGoogleAnalyticsOptions object.}
    }
}
*/
void VidyoEndpointGoogleAnalyticsOptionsDestruct(VidyoEndpointGoogleAnalyticsOptions* options);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnGetGoogleAnalyticsOptions}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time when result of VidyoEndpointGetGoogleAnalyticsOptions is received.}
	{prototype: void(*VidyoEndpointOnGetGoogleAnalyticsOptions)(VidyoEndpoint* e, const VidyoEndpointGoogleAnalyticsOptions* options)}
	{parameter:
		{name: e}
		{description: Pointer to the VidyoEndpoint that triggered the callback.}}
	{parameter:
		{name: options}
		{description: Represents the VidyoEndpointGoogleAnalyticsOptions object.}}
}
*/
typedef void(*VidyoEndpointOnGetGoogleAnalyticsOptions)(VidyoEndpoint* e, const VidyoEndpointGoogleAnalyticsOptions* options);

/**
 {function apigroup="simple":
    {name: VidyoEndpointStartGoogleAnalyticsService}
    {parent: VidyoEndpoint}
    {description: Start sending of analytics in VidyoEndpoint.}
    {prototype: LmiBool VidyoEndpointStartGoogleAnalyticsService(VidyoEndpoint* e, const VidyoEndpointGoogleAnalyticsOptions* options)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}}
	{parameter:
		{name: options}
		{description: A structure that represent the google analytics options, plz refer VidyoEndpointGoogleAnalyticsOptions. If the user set to NULL, Vidyo Client will use default option to use google analytics.}}
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointStartGoogleAnalyticsService(VidyoEndpoint* e, const VidyoEndpointGoogleAnalyticsOptions* options);

/**
 {function apigroup="simple":
    {name: VidyoEndpointStopGoogleAnalyticsService}
    {parent: VidyoEndpoint}
    {description: Stop sending of Google analytics in VidyoEndpoint.}
    {prototype: LmiBool VidyoEndpointStopGoogleAnalyticsService(VidyoEndpoint *e)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}}
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointStopGoogleAnalyticsService(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointIsGoogleAnalyticsServiceEnabled}
	{parent: VidyoEndpoint}
	{description: Checks whether Google Analytics service is running.}
	{prototype: LmiBool VidyoEndpointIsGoogleAnalyticsServiceEnabled(VidyoEndpoint *e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointIsGoogleAnalyticsServiceEnabled(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetGoogleAnalyticsOptions}
	{parent: VidyoEndpoint}
	{description: Returns the configuration of Google Analytics 4 service.}
	{prototype: LmiBool VidyoEndpointGetGoogleAnalyticsOptions(VidyoEndpoint *e, VidyoEndpointOnGetGoogleAnalyticsOptions onGetGoogleAnalyticsOptions)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}}
	{parameter:
		 {name: onGetGoogleAnalyticsOptions}
		 {description: Callback called upon completion.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointGetGoogleAnalyticsOptions(VidyoEndpoint *e, VidyoEndpointOnGetGoogleAnalyticsOptions onGetGoogleAnalyticsOptions);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetGoogleAnalyticsEventTable}
	{parent: VidyoEndpoint}
	{description: This is an asynchronous interface. Get the analytics event table.}
	{prototype: LmiBool VidyoEndpointGetGoogleAnalyticsEventTable(VidyoEndpoint *e, VidyoEndpointOnGetAnalyticsEventTable onGetAnalyticsEventTableCallback)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onGetAnalyticsEventTableCallback}
		{description: Callback called upon completion.}
	}
	{return: LMI_TRUE if the request has been scheduled successfully, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoEndpointGetGoogleAnalyticsEventTable(VidyoEndpoint *e, VidyoEndpointOnGetAnalyticsEventTable onGetAnalyticsEventTableCallback);

/**
 {function apigroup="simple":
    {name: VidyoEndpointGoogleAnalyticsControlEventAction}
    {parent: VidyoEndpoint}
    {description: Control type of event action and categories.}
    {prototype: LmiBool VidyoEndpointGoogleAnalyticsControlEventAction(VidyoEndpoint *e, VidyoEndpointGoogleAnalyticsEventCategory eventCategory, VidyoEndpointGoogleAnalyticsEventAction eventAction, LmiBool enable)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {parameter:
        {name: eventCategory}
        {description: Type of event category.}
    }
    {parameter:
        {name: eventAction}
        {description: Type of event action}
    }
    {parameter:
        {name: enable}
        {description: Event action state flag.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointGoogleAnalyticsControlEventAction(VidyoEndpoint *e, VidyoEndpointGoogleAnalyticsEventCategory eventCategory, VidyoEndpointGoogleAnalyticsEventAction eventAction, LmiBool enable);

/**
 {function apigroup="simple":
	{name: VidyoEndpointStartInsightsService}
	{parent: VidyoEndpoint}
	{description: Start sending of Insights in VidyoEndpoint.}
	{prototype: LmiBool VidyoEndpointStartInsightsService(VidyoEndpoint *e, const LmiString* serverUrl)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}}
	{parameter:
		{name: serverUrl}
		{description: Url of Insights server.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointStartInsightsService(VidyoEndpoint *e, const LmiString* serverUrl);

/**
 {function apigroup="simple":
	{name: VidyoEndpointStopInsightsService}
	{parent: VidyoEndpoint}
	{description: Stop sending of Insights in VidyoEndpoint.}
	{prototype: LmiBool VidyoEndpointStopInsightsService(VidyoEndpoint *e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointStopInsightsService(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointIsInsightsServiceEnabled}
	{parent: VidyoEndpoint}
	{description: Checks whether Insights service is running.}
	{prototype: LmiBool VidyoEndpointIsInsightsServiceEnabled(VidyoEndpoint *e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointIsInsightsServiceEnabled(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetInsightsServiceUrl}
	{parent: VidyoEndpoint}
	{description: Returns Insights Url if service is running.}
	{prototype: const LmiString *VidyoEndpointGetInsightsServiceUrl(VidyoEndpoint *e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}}
	{return: Insights Url on success, NULL on failure.}
}
*/
const LmiString *VidyoEndpointGetInsightsServiceUrl(VidyoEndpoint *e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSelectLocalCamera}
	{parent: VidyoEndpoint}
	{description: Selects the camera to be used. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSelectLocalCamera(VidyoEndpoint* e, const VidyoLocalCamera* localCamera)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: localCamera}
		{description: Camera to be selected.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectLocalCamera(VidyoEndpoint* e, const VidyoLocalCamera* localCamera);

/**
 {function apigroup="simple":
	 {name: VidyoEndpointSelectVideoContentShare}
	 {parent: VidyoEndpoint}
	 {description: Selects the camera to be used for video content share. On, iOS, this API must be called from main(UI) thread.}
	 {prototype: LmiBool VidyoEndpointSelectVideoContentShare(VidyoEndpoint* e, const VidyoLocalCamera* localCamera)}
	 {parameter:
		 {name: e}
		 {description: The VidyoEndpoint object.}
	 }
	 {parameter:
		 {name: localCamera}
		 {description: Camera to be selected for video content share.}
	 }
	 {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointSelectVideoContentShare(VidyoEndpoint* e, const VidyoLocalCamera* localCamera);

/**
 {function apigroup="simple":
	{name: VidyoEndpointAssignViewToLocalCamera}
	{parent: VidyoEndpoint}
	{description: Shows a preview of a local camera in a view. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointAssignViewToLocalCamera(VidyoEndpoint* e, const LmiViewId* viewId, const VidyoLocalCamera* localCamera, LmiBool displayCropped, LmiBool allowZoom)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID where the participant will be rendered.}
	}
	{parameter:
		{name: localCamera}
		{description: The VidyoLocalCamera that will be shown.}
	}
	{parameter:
		{name: displayCropped}
		{description: Render the image as cropped. Otherwise letterbox.}
	}
	{parameter:
		{name: allowZoom}
		{description: Allow zooming of the image with mouse or touch.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: This API will work only if compositing had been disabled by passing a NULL LmiViewId to VidyoEndpointConstruct.}
	{note: The view will not be shown until VidyoEndpointShowViewAt is called with valid parameters.}
}
*/
LmiBool VidyoEndpointAssignViewToLocalCamera(VidyoEndpoint* e, const LmiViewId* viewId, const VidyoLocalCamera* localCamera, LmiBool displayCropped, LmiBool allowZoom);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterLocalMicrophoneEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about microphone events. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointRegisterLocalMicrophoneEventListener(VidyoEndpoint* e, VidyoEndpointOnLocalMicrophoneAdded onAdded, VidyoEndpointOnLocalMicrophoneRemoved onRemoved, VidyoEndpointOnLocalMicrophoneSelected onSelected, VidyoEndpointOnLocalMicrophoneStateUpdated onStateUpdated)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered for every new or existing microphone added.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when the microphone is removed.}
	}
	{parameter:
		{name: onSelected}
		{description: Callback that is triggered when a microphone is selected for use.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when a microphone state is updated.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterLocalMicrophoneEventListener(VidyoEndpoint* e, VidyoEndpointOnLocalMicrophoneAdded onAdded, VidyoEndpointOnLocalMicrophoneRemoved onRemoved, VidyoEndpointOnLocalMicrophoneSelected onSelected, VidyoEndpointOnLocalMicrophoneStateUpdated onStateUpdated);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterLocalMicrophoneEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters local microphone event notifications. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointUnregisterLocalMicrophoneEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterLocalMicrophoneEventListener(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterLocalMicrophoneFrameListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about microphone frames. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointRegisterLocalMicrophoneFrameListener(VidyoEndpoint* e, const VidyoLocalMicrophone* localMicrophone, VidyoEndpointOnLocalMicrophoneFrame onFrame)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: localMicrophone}
		{description: The VidyoLocalMicrophone for which to receive the frames.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered for every frame from a microphone.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterLocalMicrophoneFrameListener(VidyoEndpoint* e, const VidyoLocalMicrophone* localMicrophone, VidyoEndpointOnLocalMicrophoneFrame onFrame);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterLocalMicrophoneFrameListener}
	{parent: VidyoEndpoint}
	{description: Unregisters local microphone frame notifications. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointUnregisterLocalMicrophoneFrameListener(VidyoEndpoint* e, const VidyoLocalMicrophone* localMicrophone)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: localMicrophone}
		{description: The VidyoLocalMicrophone for which to stop notifications.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterLocalMicrophoneFrameListener(VidyoEndpoint* e, const VidyoLocalMicrophone* localMicrophone);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterLocalMicrophoneEnergyListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about microphone energies.}
	{prototype: LmiBool VidyoEndpointRegisterLocalMicrophoneEnergyListener(VidyoEndpoint* e, VidyoEndpointOnLocalMicrophoneEnergy onEnergy)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onEnergy}
		{description: Callback that is triggered for every frame with energy from a microphone.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterLocalMicrophoneEnergyListener(VidyoEndpoint* e, VidyoEndpointOnLocalMicrophoneEnergy onEnergy);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterLocalMicrophoneEnergyListener}
	{parent: VidyoEndpoint}
	{description: Unregisters local microphone energy notifications.}
	{prototype: LmiBool VidyoEndpointUnregisterLocalMicrophoneEnergyListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterLocalMicrophoneEnergyListener(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSelectLocalMicrophone}
	{parent: VidyoEndpoint}
	{description: Selects the microphone to be used. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSelectLocalMicrophone(VidyoEndpoint* e, const VidyoLocalMicrophone* localMicrophone)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: localMicrophone}
		{description: Microphone to be selected.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectLocalMicrophone(VidyoEndpoint* e, const VidyoLocalMicrophone* localMicrophone);

/**
 {function apigroup="simple":
	 {name: VidyoEndpointSelectAudioContentShare}
	 {parent: VidyoEndpoint}
	 {description: Selects the microphone to be used for audio content share. On, iOS, this API must be called from main(UI) thread.}
	 {prototype: LmiBool VidyoEndpointSelectAudioContentShare(VidyoEndpoint* e, const VidyoLocalMicrophone* localMicrophone)}
	 {parameter:
		 {name: e}
		 {description: The VidyoEndpoint object.}
	 }
	 {parameter:
		 {name: localMicrophone}
		 {description: Microphone to be selected for audio content share.}
	 }
	 {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointSelectAudioContentShare(VidyoEndpoint* e, const VidyoLocalMicrophone* localMicrophone);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterLocalSpeakerEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about speaker events. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointRegisterLocalSpeakerEventListener(VidyoEndpoint* e, VidyoEndpointOnLocalSpeakerAdded onAdded, VidyoEndpointOnLocalSpeakerRemoved onRemoved, VidyoEndpointOnLocalSpeakerSelected onSelected, VidyoEndpointOnLocalSpeakerStateUpdated onStateUpdated)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered for every new or existing speaker added.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when the speaker is removed.}
	}
	{parameter:
		{name: onSelected}
		{description: Callback that is triggered when a speaker is selected for use.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when a speaker state is updated.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterLocalSpeakerEventListener(VidyoEndpoint* e, VidyoEndpointOnLocalSpeakerAdded onAdded, VidyoEndpointOnLocalSpeakerRemoved onRemoved, VidyoEndpointOnLocalSpeakerSelected onSelected, VidyoEndpointOnLocalSpeakerStateUpdated onStateUpdated);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterLocalSpeakerEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters local speaker event notifications. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointUnregisterLocalSpeakerEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterLocalSpeakerEventListener(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSelectLocalSpeaker}
	{parent: VidyoEndpoint}
	{description: Selects the speaker to be used. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSelectLocalSpeaker(VidyoEndpoint* e, const VidyoLocalSpeaker* localSpeaker)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: localSpeaker}
		{description: Speaker to be selected.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectLocalSpeaker(VidyoEndpoint* e, const VidyoLocalSpeaker* localSpeaker);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterLocalMonitorEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about monitor events. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointRegisterLocalMonitorEventListener(VidyoEndpoint* e, VidyoEndpointOnLocalMonitorAdded onAdded, VidyoEndpointOnLocalMonitorRemoved onRemoved, VidyoEndpointOnLocalMonitorSelected onSelected, VidyoEndpointOnLocalMonitorStateUpdated onStateUpdated)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered for every new or existing monitor added.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when the monitor is removed.}
	}
	{parameter:
		{name: onSelected}
		{description: Callback that is triggered when a monitor is selected for use.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when a monitor state is updated.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterLocalMonitorEventListener(VidyoEndpoint* e, VidyoEndpointOnLocalMonitorAdded onAdded, VidyoEndpointOnLocalMonitorRemoved onRemoved, VidyoEndpointOnLocalMonitorSelected onSelected, VidyoEndpointOnLocalMonitorStateUpdated onStateUpdated);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterLocalMonitorEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters local monitor event notifications. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointUnregisterLocalMonitorEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterLocalMonitorEventListener(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterLocalMonitorFrameListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about monitor frames. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointRegisterLocalMonitorFrameListener(VidyoEndpoint* e, const VidyoLocalMonitor* localMonitor, VidyoEndpointOnLocalMonitorFrame onFrame, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: localMonitor}
		{description: The VidyoLocalMonitor for which to receive the frames.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered for every frame from a monitor.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested frame.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested frame.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested frame.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterLocalMonitorFrameListener(VidyoEndpoint* e, const VidyoLocalMonitor* localMonitor, VidyoEndpointOnLocalMonitorFrame onFrame, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterLocalMonitorFrameListener}
	{parent: VidyoEndpoint}
	{description: Unregisters local monitor frame notifications. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointUnregisterLocalMonitorFrameListener(VidyoEndpoint* e, const VidyoLocalMonitor* localMonitor)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: localMonitor}
		{description: The VidyoLocalMonitor for which to stop notifications.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterLocalMonitorFrameListener(VidyoEndpoint* e, const VidyoLocalMonitor* localMonitor);

/**
 {function apigroup="simple":
	{name: VidyoEndpointAssignViewToLocalMonitor}
	{parent: VidyoEndpoint}
	{description: Shows a VidyoLocalMonitor in a view. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointAssignViewToLocalMonitor(VidyoEndpoint* e, const LmiViewId* viewId, VidyoLocalMonitor* localMonitor, LmiBool displayCropped, LmiBool allowZoom)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID where the VidyoLocalMonitor will be rendered.}
	}
	{parameter:
		{name: localMonitor}
		{description: The VidyoLocalMonitor that will be shown.}
	}
	{parameter:
		{name: displayCropped}
		{description: Render the image as cropped. Otherwise letterbox.}
	}
	{parameter:
		{name: allowZoom}
		{description: Allow zooming of the image with mouse or touch.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: This API will work only if compositing had been disabled by passing a NULL LmiViewId to VidyoEndpointConstruct.}
	{note: The view will not be shown until VidyoEndpointShowViewAt is called with valid parameters.}
}
*/
LmiBool VidyoEndpointAssignViewToLocalMonitor(VidyoEndpoint* e, const LmiViewId* viewId, VidyoLocalMonitor* localMonitor, LmiBool displayCropped, LmiBool allowZoom);

/**
 {function apigroup="simple" deprecated="yes" replaced-by="VidyoEndpointSelectLocalMonitorAdvanced":
	{name: VidyoEndpointSelectLocalMonitor}
	{parent: VidyoEndpoint}
	{description: Selects the monitor to be used. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSelectLocalMonitor(VidyoEndpoint* e, const VidyoLocalMonitor* localMonitor)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: localMonitor}
		{description: Monitor to be selected.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectLocalMonitor(VidyoEndpoint* e, const VidyoLocalMonitor* localMonitor);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterLocalWindowShareEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about window events. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointRegisterLocalWindowShareEventListener(VidyoEndpoint* e, VidyoEndpointOnLocalWindowShareAdded onAdded, VidyoEndpointOnLocalWindowShareRemoved onRemoved, VidyoEndpointOnLocalWindowShareSelected onSelected, VidyoEndpointOnLocalWindowShareStateUpdated onStateUpdated)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered for every new or existing window added.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when the window is removed.}
	}
	{parameter:
		{name: onSelected}
		{description: Callback that is triggered when a window is selected for use.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when a window state changes.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterLocalWindowShareEventListener(VidyoEndpoint* e, VidyoEndpointOnLocalWindowShareAdded onAdded, VidyoEndpointOnLocalWindowShareRemoved onRemoved, VidyoEndpointOnLocalWindowShareSelected onSelected, VidyoEndpointOnLocalWindowShareStateUpdated onStateUpdated);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterLocalWindowShareEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters local window share event notifications. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointUnregisterLocalWindowShareEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterLocalWindowShareEventListener(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterLocalWindowShareFrameListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about window frames. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointRegisterLocalWindowShareFrameListener(VidyoEndpoint* e, const VidyoLocalWindowShare* localWindowShare, VidyoEndpointOnLocalWindowShareFrame onFrame, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: localWindowShare}
		{description: The VidyoLocalWindowShare for which to receive the frames.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered for every frame from a window.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested frame.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested frame.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested frame.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterLocalWindowShareFrameListener(VidyoEndpoint* e, const VidyoLocalWindowShare* localWindowShare, VidyoEndpointOnLocalWindowShareFrame onFrame, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterLocalWindowShareFrameListener}
	{parent: VidyoEndpoint}
	{description: Unregisters local window share frame notifications. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointUnregisterLocalWindowShareFrameListener(VidyoEndpoint* e, const VidyoLocalWindowShare* localWindowShare)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: localWindowShare}
		{description: The VidyoLocalWindowShare for which to stop notifications.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterLocalWindowShareFrameListener(VidyoEndpoint* e, const VidyoLocalWindowShare* localWindowShare);

/**
 {function apigroup="simple":
	{name: VidyoEndpointAssignViewToLocalWindowShare}
	{parent: VidyoEndpoint}
	{description: Shows a VidyoLocalWindowShare in a view. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointAssignViewToLocalWindowShare(VidyoEndpoint* e, const LmiViewId* viewId, VidyoLocalWindowShare* localWindowShare, LmiBool displayCropped, LmiBool allowZoom)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID where the VidyoLocalWindowShare will be rendered.}
	}
	{parameter:
		{name: localWindowShare}
		{description: The VidyoLocalWindowShare that will be shown.}
	}
	{parameter:
		{name: displayCropped}
		{description: Render the image as cropped. Otherwise letterbox.}
	}
	{parameter:
		{name: allowZoom}
		{description: Allow zooming of the image with mouse or touch.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: This API will work only if compositing had been disabled by passing a NULL LmiViewId to VidyoEndpointConstruct.}
	{note: The view will not be shown until VidyoEndpointShowViewAt is called with valid parameters.}
}
*/
LmiBool VidyoEndpointAssignViewToLocalWindowShare(VidyoEndpoint* e, const LmiViewId* viewId, VidyoLocalWindowShare* localWindowShare, LmiBool displayCropped, LmiBool allowZoom);

/**
 {function apigroup="simple" deprecated="yes" replaced-by="VidyoEndpointSelectLocalWindowShareAdvanced":
	{name: VidyoEndpointSelectLocalWindowShare}
	{parent: VidyoEndpoint}
	{description: Selects the window to be used. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSelectLocalWindowShare(VidyoEndpoint* e, const VidyoLocalWindowShare* localWindowShare)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: localWindowShare}
		{description: Window to be selected.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectLocalWindowShare(VidyoEndpoint *e, const VidyoLocalWindowShare *localWindowShare);

/**
{function apigroup="simple":
	{name: VidyoEndpointShowWindowSharePreview}
	{parent: VidyoEndpoint}
	{description: Sets the flag to show/hide the window share preview. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointShowWindowSharePreview(VidyoEndpoint* e, LmiBool preview)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: preview}
		{description: Flag to show/hide the window share preview.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointShowWindowSharePreview(VidyoEndpoint* e, LmiBool preview);

/**
 {function apigroup="simple":
	{name: VidyoEndpointShowPreview}
	{parent: VidyoEndpoint}
	{description: Sets the flag to show/hide the camera preview for composite layout only. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointShowPreview(VidyoEndpoint* e, LmiBool preview)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: preview}
		{description: Flag to show/hide the camera preview.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointShowPreview(VidyoEndpoint* e, LmiBool preview);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSetCameraPrivacy}
	{parent: VidyoEndpoint}
	{description: Sets the flag to show/hide the local camera for remote participants for composite/custom layout and local camera preview for composite layout only. Will also unmute a soft mute(Hunter only). On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSetCameraPrivacy(VidyoEndpoint* e, LmiBool privacy)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: privacy}
		{description: Flag to show/hide the camera.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetCameraPrivacy(VidyoEndpoint* e, LmiBool privacy);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSelectDefaultCamera}
	{parent: VidyoEndpoint}
	{description: Resets the camera selection to pick the Default camera. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSelectDefaultCamera(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectDefaultCamera(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointCycleCamera}
	{parent: VidyoEndpoint}
	{description: Cycles camera selection to the next available camera. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointCycleCamera(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointCycleCamera(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSetMicrophonePrivacy}
	{parent: VidyoEndpoint}
	{description: Sets the flag to mute/unmute the microphone. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSetMicrophonePrivacy(VidyoEndpoint* e, LmiBool privacy)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: privacy}
		{description: Flag to mute/unmute the microphone.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetMicrophonePrivacy(VidyoEndpoint* e, LmiBool privacy);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSelectDefaultMicrophone}
	{parent: VidyoEndpoint}
	{description: Resets the microphone selection to pick the Default microphone. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSelectDefaultMicrophone(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectDefaultMicrophone(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointCycleMicrophone}
	{parent: VidyoEndpoint}
	{description: Cycles microphone selection to the next available microphone. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointCycleMicrophone(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointCycleMicrophone(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSetSpeakerPrivacy}
	{parent: VidyoEndpoint}
	{description: Sets the flag to mute/unmute the speaker. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSetSpeakerPrivacy(VidyoEndpoint* e, LmiBool privacy)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: privacy}
		{description: Flag to mute/unmute the speaker.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetSpeakerPrivacy(VidyoEndpoint* e, LmiBool privacy);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSelectDefaultSpeaker}
	{parent: VidyoEndpoint}
	{description: Resets the speaker selection to pick the Default speaker. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSelectDefaultSpeaker(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectDefaultSpeaker(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointCycleSpeaker}
	{parent: VidyoEndpoint}
	{description: Cycles speaker selection to the next available speaker. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointCycleSpeaker(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointCycleSpeaker(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterRemoteCameraEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about camera events from other participants.}
	{prototype: LmiBool VidyoEndpointRegisterRemoteCameraEventListener(VidyoEndpoint* e, VidyoEndpointOnRemoteCameraAdded onAdded, VidyoEndpointOnRemoteCameraRemoved onRemoved, VidyoEndpointOnRemoteCameraStateUpdated onStateUpdated)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered when another participant adds a camera into a conference.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when another participant removes a camera from a conference.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when another participant updates a camera state in a conference.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterRemoteCameraEventListener(VidyoEndpoint* e, VidyoEndpointOnRemoteCameraAdded onAdded, VidyoEndpointOnRemoteCameraRemoved onRemoved, VidyoEndpointOnRemoteCameraStateUpdated onStateUpdated);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterRemoteCameraEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters remote camera event notifications.}
	{prototype: LmiBool VidyoEndpointUnregisterRemoteCameraEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterRemoteCameraEventListener(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterRemoteCameraFrameListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about camera frames from other participants.}
	{prototype: LmiBool VidyoEndpointRegisterRemoteCameraFrameListener(VidyoEndpoint* e, const VidyoRemoteCamera* remoteCamera, VidyoEndpointOnRemoteCameraFrame onFrame, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: remoteCamera}
		{description: The VidyoRemoteCamera for which to receive the frames.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered for every frame from a remote camera.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested frame.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested frame.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested frame.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: The extra parameters are used to determin the steam quality and the resulting frame might not match exactly what is requested.}
}
*/
LmiBool VidyoEndpointRegisterRemoteCameraFrameListener(VidyoEndpoint* e, const VidyoRemoteCamera* remoteCamera, VidyoEndpointOnRemoteCameraFrame onFrame, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterRemoteCameraFrameListener}
	{parent: VidyoEndpoint}
	{description: Unregisters remote camera frame notifications.}
	{prototype: LmiBool VidyoEndpointUnregisterRemoteCameraFrameListener(VidyoEndpoint* e, const VidyoRemoteCamera* remoteCamera)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: remoteCamera}
		{description: The VidyoRemoteCamera for which to stop notifications.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterRemoteCameraFrameListener(VidyoEndpoint* e, const VidyoRemoteCamera* remoteCamera);

/**
 {function apigroup="simple":
	{name: VidyoEndpointAssignViewToRemoteCamera}
	{parent: VidyoEndpoint}
	{description: Shows a participant in a view.}
	{prototype: LmiBool VidyoEndpointAssignViewToRemoteCamera(VidyoEndpoint* e, const LmiViewId* viewId, const VidyoRemoteCamera* remoteCamera, LmiBool displayCropped, LmiBool allowZoom)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID where the participant will be rendered.}
	}
	{parameter:
		{name: remoteCamera}
		{description: The VidyoRemoteCamera that will be shown.}
	}
	{parameter:
		{name: displayCropped}
		{description: Render the image as cropped. Otherwise letterbox.}
	}
	{parameter:
		{name: allowZoom}
		{description: Allow zooming of the image with mouse or touch.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: This API will work only if compositing had been disabled by passing a NULL LmiViewId to VidyoEndpointConstruct.}
	{note: The application needs to make sure the sources are hidden when the local CPU and bandwidth are limited, by monitoring the VidyoEndpointOnResourcesChanged callback and showing/hiding sources based on avaialble resources.}
	{note: The view will not be shown until VidyoEndpointShowViewAt is called with valid parameters.}
}
*/
LmiBool VidyoEndpointAssignViewToRemoteCamera(VidyoEndpoint* e, const LmiViewId* viewId, const VidyoRemoteCamera* remoteCamera, LmiBool displayCropped, LmiBool allowZoom);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterRemoteMicrophoneEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about microphone events from other participants.}
	{prototype: LmiBool VidyoEndpointRegisterRemoteMicrophoneEventListener(VidyoEndpoint* e, VidyoEndpointOnRemoteMicrophoneAdded onAdded, VidyoEndpointOnRemoteMicrophoneRemoved onRemoved, VidyoEndpointOnRemoteMicrophoneStateUpdated onStateUpdated)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered when another participant adds a microphone into a conference.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when another participant removes a microphone from a conference.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when another participant updates a microphone state in a conference.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterRemoteMicrophoneEventListener(VidyoEndpoint* e, VidyoEndpointOnRemoteMicrophoneAdded onAdded, VidyoEndpointOnRemoteMicrophoneRemoved onRemoved, VidyoEndpointOnRemoteMicrophoneStateUpdated onStateUpdated);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterRemoteMicrophoneEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters remote microphone event notifications.}
	{prototype: LmiBool VidyoEndpointUnregisterRemoteMicrophoneEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterRemoteMicrophoneEventListener(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterRemoteMicrophoneFrameListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about microphone frames from other participants.}
	{prototype: LmiBool VidyoEndpointRegisterRemoteMicrophoneFrameListener(VidyoEndpoint* e, const VidyoRemoteMicrophone* remoteMicrophone, VidyoEndpointOnRemoteMicrophoneFrame onFrame)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: remoteMicrophone}
		{description: The VidyoRemoteMicrophone for which to receive the frames.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered for every frame from a remote microphone.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterRemoteMicrophoneFrameListener(VidyoEndpoint* e, const VidyoRemoteMicrophone* remoteMicrophone, VidyoEndpointOnRemoteMicrophoneFrame onFrame);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterRemoteMicrophoneFrameListener}
	{parent: VidyoEndpoint}
	{description: Unregisters remote microphone frame notifications.}
	{prototype: LmiBool VidyoEndpointUnregisterRemoteMicrophoneFrameListener(VidyoEndpoint* e, const VidyoRemoteMicrophone* remoteMicrophone)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: remoteMicrophone}
		{description: The VidyoRemoteMicrophone for which to stop notifications.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterRemoteMicrophoneFrameListener(VidyoEndpoint* e, const VidyoRemoteMicrophone* remoteMicrophone);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterRemoteMicrophoneEnergyListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about microphone energies from other participants.}
	{prototype: LmiBool VidyoEndpointRegisterRemoteMicrophoneEnergyListener(VidyoEndpoint* e, VidyoEndpointOnRemoteMicrophoneEnergy onEnergy)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onEnergy}
		{description: Callback that is triggered for every frame with energy from a remote microphone.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterRemoteMicrophoneEnergyListener(VidyoEndpoint* e, VidyoEndpointOnRemoteMicrophoneEnergy onEnergy);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterRemoteMicrophoneEnergyListener}
	{parent: VidyoEndpoint}
	{description: Unregisters remote microphone energy notifications.}
	{prototype: LmiBool VidyoEndpointUnregisterRemoteMicrophoneEnergyListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterRemoteMicrophoneEnergyListener(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterRemoteWindowShareEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about window-shares from other participants.}
	{prototype: LmiBool VidyoEndpointRegisterRemoteWindowShareEventListener(VidyoEndpoint* e, VidyoEndpointOnRemoteWindowShareAdded onAdded, VidyoEndpointOnRemoteWindowShareRemoved onRemoved, VidyoEndpointOnRemoteWindowShareStateUpdated onStateUpdated)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered when another participant adds a window-share into a conference.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when another participant removes a window-share from a conference.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when another participant updates a window-share state in a conference.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterRemoteWindowShareEventListener(VidyoEndpoint* e, VidyoEndpointOnRemoteWindowShareAdded onAdded, VidyoEndpointOnRemoteWindowShareRemoved onRemoved, VidyoEndpointOnRemoteWindowShareStateUpdated onStateUpdated);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterRemoteWindowShareEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters remote window share event notifications.}
	{prototype: LmiBool VidyoEndpointUnregisterRemoteWindowShareEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterRemoteWindowShareEventListener(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterRemoteWindowShareFrameListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about window-share frames from other participants.}
	{prototype: LmiBool VidyoEndpointRegisterRemoteWindowShareFrameListener(VidyoEndpoint* e, const VidyoRemoteWindowShare* remoteWindowShare, VidyoEndpointOnRemoteWindowShareFrame onFrame, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: remoteWindowShare}
		{description: The VidyoRemoteWindowShare for which to receive the frames.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered for every frame from a remote window share.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested frame.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested frame.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested frame.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: The extra parameters are used to determin the steam quality and the resulting frame might not match exactly what is requested.}
}
*/
LmiBool VidyoEndpointRegisterRemoteWindowShareFrameListener(VidyoEndpoint* e, const VidyoRemoteWindowShare* remoteWindowShare, VidyoEndpointOnRemoteWindowShareFrame onFrame, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterRemoteWindowShareFrameListener}
	{parent: VidyoEndpoint}
	{description: Unregisters remote window share frame notifications.}
	{prototype: LmiBool VidyoEndpointUnregisterRemoteWindowShareFrameListener(VidyoEndpoint* e, const VidyoRemoteWindowShare* remoteWindowShare)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: remoteWindowShare}
		{description: The VidyoRemoteWindowShare for which to stop notifications.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterRemoteWindowShareFrameListener(VidyoEndpoint* e, const VidyoRemoteWindowShare* remoteWindowShare);

/**
 {function apigroup="simple":
	{name: VidyoEndpointAssignViewToRemoteWindowShare}
	{parent: VidyoEndpoint}
	{description: Shows a participant in a view. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointAssignViewToRemoteWindowShare(VidyoEndpoint* e, const LmiViewId* viewId, const VidyoRemoteWindowShare* remoteWindowShare, LmiBool displayCropped, LmiBool allowZoom)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID where the participant will be rendered.}
	}
	{parameter:
		{name: remoteWindowShare}
		{description: The VidyoRemoteWindowShare that will be shown.}
	}
	{parameter:
		{name: displayCropped}
		{description: Render the image as cropped. Otherwise letterbox.}
	}
	{parameter:
		{name: allowZoom}
		{description: Allow zooming of the image with mouse or touch.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: This API will work only if compositing had been disabled by passing a NULL LmiViewId to VidyoEndpointConstruct.}
	{note: The view will not be shown until VidyoEndpointShowViewAt is called with valid parameters.}
}
*/
LmiBool VidyoEndpointAssignViewToRemoteWindowShare(VidyoEndpoint* e, const LmiViewId* viewId, const VidyoRemoteWindowShare* remoteWindowShare, LmiBool displayCropped, LmiBool allowZoom);

/**
 {function apigroup="simple":
    {name: VidyoEndpointAssignViewToVirtualVideoSource}
    {parent: VidyoEndpoint}
    {description: Shows a virtual video source in a view. On, iOS, this API must be called from main(UI) thread.}
    {prototype: LmiBool VidyoEndpointAssignViewToVirtualVideoSource(VidyoEndpoint* e, const LmiViewId* viewId, const VidyoVirtualVideoSource* virtualVideoSource, LmiBool displayCropped, LmiBool allowZoom)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {parameter:
        {name: viewId}
        {description: A platform specific view ID where the source will be rendered.}
    }
    {parameter:
        {name: virtualVideoSource}
        {description: The VidyoVirtualVideoSource that will be shown.}
    }
    {parameter:
        {name: displayCropped}
        {description: Render the image as cropped. Otherwise letterbox.}
    }
    {parameter:
        {name: allowZoom}
        {description: Allow zooming of the image with mouse or touch.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
    {note: This API will work only if compositing had been disabled by passing a NULL LmiViewId to VidyoEndpointConstruct.}
    {note: The view will not be shown until VidyoEndpointShowViewAt is called with valid parameters.}
}
*/
LmiBool VidyoEndpointAssignViewToVirtualVideoSource(VidyoEndpoint* e, const LmiViewId* viewId, const VidyoVirtualVideoSource* virtualVideoSource, LmiBool displayCropped, LmiBool allowZoom);

/**
 {function apigroup="simple":
	{name: VidyoEndpointHideView}
	{parent: VidyoEndpoint}
	{description: Hides a view which is used for custom layout of sources. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointHideView(VidyoEndpoint* e, const LmiViewId* viewId)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: viewId}
		{description: The view ID of the view to hide.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointHideView(VidyoEndpoint* e, const LmiViewId* viewId);

/**
 {function apigroup="simple":
	{name: VidyoEndpointShowViewAt}
	{parent: VidyoEndpoint}
	{description: Sets the position of the view every time it's changed on the screen. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointShowViewAt(VidyoEndpoint* e, const LmiViewId* viewId, LmiInt x, LmiInt y, LmiUint width, LmiUint height)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID to be positioned.}
	}
	{parameter:
		{name: x}
		{description: X coordinate. }
	}
	{parameter:
		{name: y}
		{description: Y coordinate. }
	}
	{parameter:
		{name: width}
		{description: Width of the view. }
	}
	{parameter:
		{name: height}
		{description: Height of the view. }
	}
	{return: }
	{note: The view will not be shown until this funciton is called with valid parameters.}
	{note: If viewId is NULL then the composite renderer constructed in VidyoEndpointConstruct will be positioned.}
}
*/
LmiBool VidyoEndpointShowViewAt(VidyoEndpoint* e, const LmiViewId* viewId, LmiInt x, LmiInt y, LmiUint width, LmiUint height);

/**
 {function apigroup="simple":
 {name: VidyoEndpointShowViewAtPoints}
 {parent: VidyoEndpoint}
 {description: Sets the position of the view every time it's changed on the screen. On, iOS, this API must be called from main(UI) thread.}
 {prototype: LmiBool VidyoEndpointShowViewAtPoints(VidyoEndpoint* e, const LmiViewId* viewId, LmiInt x, LmiInt y, LmiUint width, LmiUint height)}
 {parameter:
 	{name: e}
 	{description: The VidyoEndpoint object.}
 }
 {parameter:
 	{name: viewId}
 	{description: A platform specific view ID to be positioned.}
 }
 {parameter:
 	{name: x}
 	{description: X coordinate (in points).}
 }
 {parameter:
 	{name: y}
 	{description: Y coordinate (in points).}
 }
 {parameter:
 	{name: width}
 	{description: Width of the view (in points).}
 }
 {parameter:
 	{name: height}
 	{description: Height of the view (in points).}
 }
 {return: }
 {note: The view will not be shown until this funciton is called with valid parameters.}
 {note: If viewId is NULL then the composite renderer constructed in VidyoEndpointConstruct will be positioned.}
 }
 */
LmiBool VidyoEndpointShowViewAtPoints(VidyoEndpoint* e, const LmiViewId* viewId, LmiInt x, LmiInt y, LmiUint width, LmiUint height);
/**
 {function apigroup="simple":
	{name: VidyoEndpointShowViewLabel}
	{parent: VidyoEndpoint}
	{description: Shows or hides the label on the view which displays the participant name or preview. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointShowViewLabel(VidyoEndpoint* e, const LmiViewId* viewId, LmiBool showLabel)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID to be positioned.}
	}
	{parameter:
		{name: showLabel}
		{description: Boolean to show or hide the label. }
	}
	{return: }
	{note: If viewId is NULL then the composite renderer constructed in VidyoEndpointConstruct will be assumed.}
}
*/
LmiBool VidyoEndpointShowViewLabel(VidyoEndpoint* e, const LmiViewId* viewId, LmiBool showLabel);

/**
 {function apigroup="simple":
	{name: VidyoEndpointShowAudioMeters}
	{parent: VidyoEndpoint}
	{description: Shows or hides the audio meters in the view. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointShowAudioMeters(VidyoEndpoint* e, const LmiViewId* viewId, LmiBool showMeters)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID to be positioned.}
	}
	{parameter:
		{name: showMeters}
		{description: Boolean to show or hide the audio meters. }
	}
	{return: }
	{note: If viewId is NULL then the composite renderer constructed in VidyoEndpointConstruct will be assumed.}	
}
*/
LmiBool VidyoEndpointShowAudioMeters(VidyoEndpoint* e, const LmiViewId* viewId, LmiBool showMeters);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSetViewBackgroundColor}
	{parent: VidyoEndpoint}
	{description: Shows or hides the label on the view which displays the participant name or preview. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSetViewBackgroundColor(VidyoEndpoint* e, const LmiViewId* viewId, LmiUint8 red, LmiUint8 green, LmiUint8 blue)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID to be positioned.}
	}
	{parameter:
		{name: red}
		{description: Red component of the color 0-255.}
	}
	{parameter:
		{name: green}
		{description: Green component of the color 0-255.}
	}
	{parameter:
		{name: blue}
		{description: Blue component of the color 0-255.}
	}
	{return: }
	{note: If viewId is NULL then the composite renderer constructed in VidyoEndpointConstruct will be assumed.}
}
*/
LmiBool VidyoEndpointSetViewBackgroundColor(VidyoEndpoint* e, const LmiViewId* viewId, LmiUint8 red, LmiUint8 green, LmiUint8 blue);


/**
 {function apigroup="simple":
	{name: VidyoEndpointSetViewAnimationSpeed}
	{parent: VidyoEndpoint}
	{description: Sets the speed of transition animaiton for windows moving around inside the view. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSetViewAnimationSpeed(VidyoEndpoint* e, const LmiViewId* viewId, LmiUint speedPercentage)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID to be positioned.}
	}
	{parameter:
		{name: speedPercentage}
		{description: Percentage of animation speed with default of 50, 100 for slowest and 0 for instant. }
	}
	{return: }
	{note: Not supported in WebRTC.}
	{note: If viewId is NULL then the composite renderer constructed in VidyoEndpointConstruct will be assumed.}
}
*/
LmiBool VidyoEndpointSetViewAnimationSpeed(VidyoEndpoint* e, const LmiViewId* viewId, LmiUint speedPercentage);

/**
{function apigroup="simple":
	{name: VidyoEndpointStartLocationDetection}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointStartLocationDetection(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointStartLocationDetection(VidyoEndpoint* e);
/**
{function apigroup="simple":
	{name: VidyoEndpointStopLocationDetection}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: void VidyoEndpointStopLocationDetection(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
}
*/
void VidyoEndpointStopLocationDetection(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterNetworkInterfaceEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about network interface events.}
	{prototype: LmiBool VidyoEndpointRegisterNetworkInterfaceEventListener(VidyoEndpoint* e, VidyoEndpointOnNetworkInterfaceAdded onAdded, VidyoEndpointOnNetworkInterfaceRemoved onRemoved, VidyoEndpointOnNetworkInterfaceSelected onSelected, VidyoEndpointOnNetworkInterfaceStateUpdated onStateUpdated)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered for every new or existing network interface added.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when the network interface is removed.}
	}
	{parameter:
		{name: onSelected}
		{description: Callback that is triggered when a network interface is selected for use.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when a network interface state changes.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterNetworkInterfaceEventListener(VidyoEndpoint* e, VidyoEndpointOnNetworkInterfaceAdded onAdded, VidyoEndpointOnNetworkInterfaceRemoved onRemoved, VidyoEndpointOnNetworkInterfaceSelected onSelected, VidyoEndpointOnNetworkInterfaceStateUpdated onStateUpdated);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterNetworkInterfaceEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters network interface event notifications.}
	{prototype: LmiBool VidyoEndpointUnregisterNetworkInterfaceEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterNetworkInterfaceEventListener(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSelectDefaultNetworkInterfaceForSignaling}
	{parent: VidyoEndpoint}
	{description: Selects the default ANY network interface to be used for Signaling.}
	{prototype: LmiBool VidyoEndpointSelectDefaultNetworkInterfaceForSignaling(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectDefaultNetworkInterfaceForSignaling(VidyoEndpoint *e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSelectNetworkInterfaceForSignaling}
	{parent: VidyoEndpoint}
	{description: Selects the network interface to be used for Signaling.}
	{prototype: LmiBool VidyoEndpointSelectNetworkInterfaceForSignaling(VidyoEndpoint* e, const VidyoNetworkInterface* networkInterface)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: networkInterface}
		{description: Network interface to be selected.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectNetworkInterfaceForSignaling(VidyoEndpoint *e, const VidyoNetworkInterface *networkInterface);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSelectDefaultNetworkInterfaceForMedia}
	{parent: VidyoEndpoint}
	{description: Selects the default ANY network interface to be used for Media.}
	{prototype: LmiBool VidyoEndpointSelectDefaultNetworkInterfaceForMedia(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectDefaultNetworkInterfaceForMedia(VidyoEndpoint* e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSelectNetworkInterfaceForMedia}
	{parent: VidyoEndpoint}
	{description: Selects the network interface to be used for Media.}
	{prototype: LmiBool VidyoEndpointSelectNetworkInterfaceForMedia(VidyoEndpoint* e, const VidyoNetworkInterface* networkInterface)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: networkInterface}
		{description: Network interface to be selected.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectNetworkInterfaceForMedia(VidyoEndpoint *e, const VidyoNetworkInterface *networkInterface);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetLocation}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointSetLocation(VidyoEndpoint* e, LmiFloat64 latitude, LmiFloat64 longitude)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: latitude}
		{description: }
	}
	{parameter:
		{name: longitude}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointSetLocation(VidyoEndpoint* e, LmiFloat64 latitude, LmiFloat64 longitude);

/**
{function:
	{name: VidyoEndpointSetPool}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointSetPool(VidyoEndpoint* e, const LmiString* name)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: name}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointSetPool(VidyoEndpoint* e, const LmiString* name);

/**
{function:
	{name: VidyoEndpointValidateLicenseKey}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointValidateLicenseKey(VidyoEndpoint *e, const LmiString *licenseKey, const LmiString *userName)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: licenseKey}
		{description: }
	}
 	{parameter:
 		{name: userName}
 		{description: }
 	}
	{return: }
}
*/
LmiBool VidyoEndpointValidateLicenseKey(VidyoEndpoint *e, const LmiString *licenseKey, const LmiString *userName);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetId}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: const LmiString* VidyoEndpointGetId(const VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
	{return: }
}
*/
const LmiString *VidyoEndpointGetId(const VidyoEndpoint *e);

/**
{function:
	{name: VidyoEndpointGetApplicationTag}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: const LmiString* VidyoEndpointGetApplicationTag(const VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: }
	}
	{return: }
}
*/
const LmiString *VidyoEndpointGetApplicationTag(const VidyoEndpoint *e);

/**
{function:
	{name: VidyoEndpointSetApplicationTag}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointSetApplicationTag(VidyoEndpoint* e, const char* applicationTag)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: applicationTag}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointSetApplicationTag(VidyoEndpoint *e, const char *applicationTag);

/**
{function apigroup="simple":
	{name: VidyoEndpointIsDebugEnabled}
	{parent: VidyoEndpoint}
	{description: Checks if debug mode is enabled on the endpoint.}
	{prototype: LmiBool VidyoEndpointIsDebugEnabled(VidyoEndpoint *e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE if enabled, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoEndpointIsDebugEnabled(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointEnableDebug}
	{parent: VidyoEndpoint}
	{description: Enabled debug mode on the endpoint which will start debug logging and enable statistics.}
	{prototype: LmiBool VidyoEndpointEnableDebug(VidyoEndpoint* e, LmiUint port, const char *logFilter)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: port}
		{description: Debug port will output debug logging.  }
	}
	{parameter:
		{name: logFilter}
		{description: Log filter to use on the debug port. }
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointEnableDebug(VidyoEndpoint *e, LmiUint port, const char *logFilter);

/**
{function apigroup="simple":
	{name: VidyoEndpointDisableDebug}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: void VidyoEndpointDisableDebug(VidyoEndpoint *e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
}
*/
void VidyoEndpointDisableDebug(VidyoEndpoint *e);

/**
{function:
	{name: VidyoEndpointSetDebugFilter}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointSetDebugFilter(VidyoEndpoint* e, const char* debugLogFilter)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: debugLogFilter}
		{description: Log filter to use on the debug port. }
	}
	{return: }
}
*/
LmiBool VidyoEndpointSetDebugFilter(VidyoEndpoint *e, const char *debugLogFilter);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetConsoleLoggerFilter}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointSetConsoleLoggerFilter(VidyoEndpoint* e, const char* consoleLogFilter)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: consoleLogFilter}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointSetConsoleLoggerFilter(VidyoEndpoint* e, const char* consoleLogFilter);
/**
{function:
	{name: VidyoEndpointSetConsoleLoggerLevelAndCategory}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointSetConsoleLoggerLevelAndCategory(VidyoEndpoint* e, const char* levelName, const char* categoryName, LmiBool on)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: levelName}
		{description: }
	}
	{parameter:
		{name: categoryName}
		{description: }
	}
	{parameter:
		{name: on}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointSetConsoleLoggerLevelAndCategory(VidyoEndpoint* e, const char* levelName, const char* categoryName, LmiBool on);

/**
 {type visibility="public":
	{name: VidyoEndpointLoggerType}
	{parent: VidyoEndpoint}
	{description: Type of the logger.}
	{value: {name: VIDYO_ENDPOINTLOGGERTYPE_CONSOLE}
		{description: Logger type is console logger.}}
	{value: {name: VIDYO_ENDPOINTLOGGERTYPE_FILE}
		{description: Logger type is file logger.}}
	{value: {name: VIDYO_ENDPOINTLOGGERTYPE_FEEDBACK}
		{description: Logger type is feedback logger.}}
	{value: {name: VIDYO_ENDPOINTLOGGERTYPE_HTTP}
		{description: Logger type is HTTP logger.}}
}
*/
typedef enum
{
	VIDYO_ENDPOINTLOGGERTYPE_CONSOLE,
	VIDYO_ENDPOINTLOGGERTYPE_FILE,
	VIDYO_ENDPOINTLOGGERTYPE_FEEDBACK,
	VIDYO_ENDPOINTLOGGERTYPE_HTTP
} VidyoEndpointLoggerType;

/**
 {type visibility="public":
	{name: VidyoEndpointGetLoggerCategoryListResult}
	{parent: VidyoEndpoint}
	{description: Result of getting the logger category list.}
	{value: {name: VIDYO_ENDPOINTGETLOGGERCATEGORYLISTRESULT_SUCCESS}
		{description: Logger category list was got successfully.}}
	{value: {name: VIDYO_ENDPOINTGETLOGGERCATEGORYLISTRESULT_MiscError}
		{description: The getting of logger category list failed some miscellaneous reason.}}
}
*/
typedef enum
{
	VIDYO_ENDPOINTGETLOGGERCATEGORYLISTRESULT_SUCCESS,
	VIDYO_ENDPOINTGETLOGGERCATEGORYLISTRESULT_MiscError
} VidyoEndpointGetLoggerCategoryListResult;

/**
{function visibility="private":
    {name: VidyoEndpointSetLogLevel}
    {parent: VidyoEndpoint}
    {description: Setting of log properties according to the input string.}
    {prototype: LmiBool VidyoEndpointSetLogLevel(VidyoEndpoint *e, VidyoEndpointLoggerType loggerType, const char *logFilter)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {parameter:
        {name: loggerType}
        {description: loggerType console, file, feedback, http.}
    }
    {parameter:
        {name: logFilter}
        {description: logging string settings.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetLogLevel(VidyoEndpoint *e, VidyoEndpointLoggerType loggerType, const char *logFilter);

/**
 {callback:
	{name: VidyoEndpointOnGetLoggerCategoryListComplete}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered when getting the logger category list completes.}
	{prototype: void (*VidyoEndpointOnGetLoggerCategoryListComplete)(VidyoEndpoint* e, VidyoEndpointLoggerType loggerType, const LmiVector(VidyoLoggerCategory) *categories, VidyoEndpointGetLoggerCategoryListResult result)}
	{parameter:
		{name: e}
		{description: The endpoint object.}
	}
	{parameter:
		{name: loggerType}
		{description: Type of the logger.}
	}
	{parameter:
		{name: categories}
		{description: Array of logger categories.}
	}
	{parameter:
		{name: result}
		{description: Result of getting the logger category list.}
	}
}
*/
typedef void (*VidyoEndpointOnGetLoggerCategoryListComplete)(VidyoEndpoint* e, VidyoEndpointLoggerType loggerType, const LmiVector(VidyoLoggerCategory) *categories, VidyoEndpointGetLoggerCategoryListResult result);

/**
{function visibility="private":
	{name: VidyoEndpointGetConsoleLoggerCategoryList}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointGetConsoleLoggerCategoryList(VidyoEndpoint* e, LmiVector(VidyoLoggerCategory)* categories)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: categories}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointGetConsoleLoggerCategoryList(VidyoEndpoint* e, LmiVector(VidyoLoggerCategory)* categories);

/**
{function apigroup="simple":
	{name: VidyoEndpointEnableFileLogger}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointEnableFileLogger(VidyoEndpoint *e, const char *logFileName, const char *logFileFilter)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: logFileName}
		{description: }
	}
	{parameter:
		{name: logFileFilter}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointEnableFileLogger(VidyoEndpoint *e, const char *logFileName, const char *logFileFilter);

/**
 {function apigroup="simple":
 	{name: VidyoEndpointSetFileLoggerUTC}
 	{parent: VidyoEndpoint}
	{description: }
 	{prototype: LmiBool VidyoEndpointSetFileLoggerUTC(VidyoEndpoint* e, LmiBool utc)}
 	{parameter:
 		{name: e}
 		{description: }
 	}
 	{parameter:
 		{name: utc}
 	{description: }
 	}
 	{return: }
 }
 */
LmiBool VidyoEndpointSetFileLoggerUTC(VidyoEndpoint* e, LmiBool utc);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSetConsoleLoggerUTC}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointSetConsoleLoggerUTC(VidyoEndpoint* e, LmiBool utc)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: utc}
	{description: }
	}
	{return: }
 }
 */
LmiBool VidyoEndpointSetConsoleLoggerUTC(VidyoEndpoint* e, LmiBool utc);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetFileLoggerFilter}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointSetFileLoggerFilter(VidyoEndpoint* e, const char* logFileFilter)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: logFileFilter}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointSetFileLoggerFilter(VidyoEndpoint* e, const char* logFileFilter);

/**
{function:
	{name: VidyoEndpointSetFileLoggerLevelAndCategory}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointSetFileLoggerLevelAndCategory(VidyoEndpoint* e, const char* levelName, const char* categoryName, LmiBool on)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: levelName}
		{description: }
	}
	{parameter:
		{name: categoryName}
		{description: }
	}
	{parameter:
		{name: on}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointSetFileLoggerLevelAndCategory(VidyoEndpoint* e, const char* levelName, const char* categoryName, LmiBool on);

/**
{function visibility="public"  deprecated="yes" replaced-by="VidyoEndpointGetFileLoggerCategoryListAsync":
	{name: VidyoEndpointGetFileLoggerCategoryList}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointGetFileLoggerCategoryList(VidyoEndpoint* e, LmiVector(VidyoLoggerCategory)* categories)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: categories}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointGetFileLoggerCategoryList(VidyoEndpoint* e, LmiVector(VidyoLoggerCategory)* categories);

/**
{function visibility="public":
	{name: VidyoEndpointGetFileLoggerCategoryListAsync}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointGetFileLoggerCategoryListAsync(VidyoEndpoint* e, VidyoEndpointOnGetLoggerCategoryListComplete onComplete)}
	{parameter:
		{name: e}
		{description: The endpoint object.}
	}
	{parameter:
		{name: onComplete}
		{description: The callback that is triggered when getting the logger category list completes.}
	}
	{return:  LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointGetFileLoggerCategoryListAsync(VidyoEndpoint* e, VidyoEndpointOnGetLoggerCategoryListComplete onComplete);

/**
{function visibility="public":
	{name: VidyoEndpointSetFileLoggerRotationParameters}
	{parent: VidyoEndpoint}
	{description: Sets log file size limit, how often to initiate the wrap and number of log files.}
	{prototype: LmiBool VidyoEndpointSetFileLoggerRotationParameters(VidyoEndpoint* e, LmiSizeT maxSize, LmiTime wrapPeriod, LmiSizeT wrapCount)}
	{parameter:
		{name: e}
		{description: The endpoint object.}
	}
	{parameter:
		{name: maxSize}
		{description: The number of bytes of logged messages to save.}
	}
	{parameter:
		{name: wrapPeriod}
		{description: How often to initiate the wrap in nanoseconds. If this value is LMI_TIME_INFINITE, the log file will not be wrapped based on time.}
	}
	{parameter:
		{name: wrapCount}
		{description: The maximum number of files to keep.}
	}
	{return:  LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetFileLoggerRotationParameters(VidyoEndpoint* e, LmiSizeT maxSize, LmiTime wrapPeriod, LmiSizeT wrapCount);

/**
{function:
	{name: VidyoEndpointSetLogFilePath}
	{parent: VidyoEndpoint}
	{description: Sets new path to log file.}
	{prototype: LmiBool VidyoEndpointSetLogFilePath(VidyoEndpoint* e, const char* filename)}
	{parameter:
		{name: e}
		{description: The endpoint object.}
	}
	{parameter:
		{name: filename}
		{description: The name of the file to log to. Note: Can be passed just filename or with full path so that location will be changed.}
	}
	{return: LMI_TRUE on success, else LMI_FALSE.}
}
*/
LmiBool VidyoEndpointSetLogFilePath(VidyoEndpoint* e, const char* filename);

/**
{function:
	{name: VidyoEndpointStartNextLogFile}
	{parent: VidyoEndpoint}
	{description: Option to stop logging into current file and start logging to the next one.}
	{prototype: LmiBool VidyoEndpointStartNextLogFile(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The endpoint object.}
	}
	{return:  LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointStartNextLogFile(VidyoEndpoint* e);

/**
 {type visibility="public":
	{name: VidyoEndpointLogCompressionResult}
	{parent: VidyoEndpoint}
	{description: Result of file log compression.}
	{value: {name: VIDYO_ENDPOINTLOGCOMPRESSIONRESULT_Success}
		{description: File Compressed successfully.}}
	{value: {name: VIDYO_ENDPOINTLOGCOMPRESSIONRESULT_MiscError}
		{description: The compression failed some miscellaneous reason.}}
}
*/
typedef enum
{
	VIDYO_ENDPOINTLOGCOMPRESSIONRESULT_Success,
	VIDYO_ENDPOINTLOGCOMPRESSIONRESULT_MiscError
} VidyoEndpointLogCompressionResult;

/**
 {callback:
	{name: VidyoEndpointOnLogCompressionComplete}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered when the file log compression completes. }
	{prototype: void (*VidyoEndpointOnLogCompressionComplete)(VidyoEndpoint* e, LmiString *compressedFileName, VidyoEndpointLogCompressionResult result)}
	{parameter:
		{name: e}
		{description:}
	}
	{parameter:
		{name: compressedFileName}
		{description: Name if the compression output file.}
	}
	{parameter:
		{name: result}
		{description: Result of the file compression.}
	}
}
*/
typedef void(*VidyoEndpointOnLogCompressionComplete)(VidyoEndpoint* e, LmiString *compressedFileName, VidyoEndpointLogCompressionResult result);

/**
{function:
	{name: VidyoEndpointCompressLogs}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointCompressLogs(VidyoEndpoint *e, LmiString *compressedFileName, VidyoEndpointOnLogCompressionComplete onComplete)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: compressedFileName}
		{description: Name if the compression output file.}
	}
	{parameter:
		{name: onComplete}
		{description: Callback that is triggered when the compression completes.}
	}
	{return: }
}
*/
LmiBool VidyoEndpointCompressLogs(VidyoEndpoint *e, LmiString *compressedFileName, VidyoEndpointOnLogCompressionComplete onComplete);

/**
{function visibility="private":
	{name: VidyoEndpointAddAudioCodec}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointAddAudioCodec(VidyoEndpoint* e, const VidyoCodecAudio* audioCodec)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: audioCodec}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointAddAudioCodec(VidyoEndpoint* e, const VidyoCodecAudio* audioCodec);

/**
{function visibility="private":
	{name: VidyoEndpointAddVideoCodec}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointAddVideoCodec(VidyoEndpoint* e, const VidyoCodecVideo* videoCodec)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: videoCodec}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointAddVideoCodec(VidyoEndpoint* e, const VidyoCodecVideo* videoCodec);

/**
{function visibility="private":
	{name: VidyoEndpointSetAudioVideoCodecPreference}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointSetAudioVideoCodecPreference(VidyoEndpoint* e, const LmiVector(VidyoCodecVideo)* videoCodecs, const LmiVector(VidyoCodecAudio)* audioCodecs)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: videoCodecs}
		{description: }
	}
	{parameter:
		{name: audioCodecs}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointSetAudioVideoCodecPreference(VidyoEndpoint* e, const LmiVector(VidyoCodecVideo)* videoCodecs, const LmiVector(VidyoCodecAudio)* audioCodecs);

/**
{function visibility="private":
	{name: VidyoEndpointGetAudioVideoCodecPreference}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointGetAudioVideoCodecPreference(VidyoEndpoint* e, LmiVector(VidyoCodecVideo)* videoCodecs, LmiVector(VidyoCodecAudio)* audioCodecs)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: videoCodecs}
		{description: }
	}
	{parameter:
		{name: audioCodecs}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointGetAudioVideoCodecPreference(VidyoEndpoint* e, LmiVector(VidyoCodecVideo)* videoCodecs, LmiVector(VidyoCodecAudio)* audioCodecs);

/**
{function visibility="private":
	{name: VidyoEndpointRegisterVidyoTransport}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointRegisterVidyoTransport(VidyoEndpoint* e, const VidyoTransport* transport)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: transport}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointRegisterVidyoTransport(VidyoEndpoint* e, const VidyoTransport* transport);

/**
{function visibility="private":
	{name: VidyoEndpointSetWatermark}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointSetWatermark(VidyoEndpoint *e, const VidyoLmiVideoFrame *watermark)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: watermark}
		{description: }
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetWatermark(VidyoEndpoint *e, const VidyoLmiVideoFrame *watermark);

/**
{function:
	{name: VidyoEndpointCreateRenderer}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointCreateRenderer(VidyoEndpoint* e, const VidyoLocalMonitor* monitor, LmiUint width, LmiUint height)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: monitor}
		{description: }
	}
	{parameter:
		{name: width}
		{description: }
	}
	{parameter:
		{name: height}
		{description: }
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointCreateRenderer(VidyoEndpoint* e, const VidyoLocalMonitor* monitor, LmiUint width, LmiUint height);

/**
{function visibility="private":
	{name: VidyoEndpointCreateRendererFullScreen}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointCreateRendererFullScreen(VidyoEndpoint* e, VidyoLocalMonitor* monitor)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: monitor}
		{description: }
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointCreateRendererFullScreen(VidyoEndpoint* e, VidyoLocalMonitor* monitor);

/**
{function:
{name: VidyoEndpointCreateRendererMultiScreen}
{parent: VidyoEndpoint}
{description: }
{prototype: LmiBool VidyoEndpointCreateRendererMultiScreen(VidyoEndpoint* e)}
{parameter:
{name: e}
{description: }
}
{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointCreateRendererMultiScreen(VidyoEndpoint* e);

/**
{function visibility="private":
	{name: VidyoEndpointCreateRendererFromConferenceScene}
	{parent: VidyoEndpoint}
	{description: VidyoLocalRenderer object constructed from conference scene should also get notified about stream/slot changes by calling VidyoLocalRendererSetStreamParameters API.}
	{prototype: LmiBool VidyoEndpointCreateRendererFromConferenceScene(VidyoEndpoint* e, VidyoLmiConferenceScene* conferenceScene, LmiAllocator* alloc)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: conferenceScene}
		{description: }
	}
	{parameter:
		{name: alloc}
		{description: }
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointCreateRendererFromConferenceScene(VidyoEndpoint* e, VidyoLmiConferenceScene* conferenceScene, LmiAllocator* alloc);

/**
{function:
	{name: VidyoEndpointCreateRendererFromViewId}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointCreateRendererFromViewId(VidyoEndpoint* e, const LmiViewId* viewId, LmiInt x, LmiInt y, LmiUint width, LmiUint height)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: viewId}
		{description: A platform specific parent view ID where the rendering window will be added as a child window.}
	}
	{parameter:
		{name: x}
		{description: X coordinate of the child window within the parent view ID. }
	}
	{parameter:
		{name: y}
		{description: Y coordinate of the child window within the parent view ID. }
	}
	{parameter:
		{name: width}
		{description: Width of the child window within the parent view ID. }
	}
	{parameter:
		{name: height}
		{description: Height of the child window within the parent view ID. }
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: LmiViewId is HWND on windows, NSView on mac, UIView on iOS, window on X11, ViewGroup on android.}
}
*/
LmiBool VidyoEndpointCreateRendererFromViewId(VidyoEndpoint* e, const LmiViewId* viewId, LmiInt x, LmiInt y, LmiUint width, LmiUint height);

/**
{function:
	{name: VidyoEndpointRemoveRenderer}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: void VidyoEndpointRemoveRenderer(VidyoEndpoint* e, const VidyoLocalRenderer* localRenderer)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localRenderer}
		{description: }
	}
}
*/
void VidyoEndpointRemoveRenderer(VidyoEndpoint* e, const VidyoLocalRenderer* localRenderer);

/**
{function visibility="private":
	{name: VidyoEndpointGetLocalCameras}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointGetLocalCameras(VidyoEndpoint* e, LmiVector(VidyoLocalCamera)* localCameras)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localCameras}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointGetLocalCameras(VidyoEndpoint* e, LmiVector(VidyoLocalCamera)* localCameras);

/**
{function visibility="private":
	{name: VidyoEndpointGetLocalMicrophones}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointGetLocalMicrophones(VidyoEndpoint* e, LmiVector(VidyoLocalMicrophone)* localMicrophones)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMicrophones}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointGetLocalMicrophones(VidyoEndpoint* e, LmiVector(VidyoLocalMicrophone)* localMicrophones);

/**
{function visibility="private":
	{name: VidyoEndpointGetLocalSpeakers}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointGetLocalSpeakers(VidyoEndpoint* e, LmiVector(VidyoLocalSpeaker)* localSpeakers)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localSpeakers}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointGetLocalSpeakers(VidyoEndpoint* e, LmiVector(VidyoLocalSpeaker)* localSpeakers);

/**
{function visibility="private":
	{name: VidyoEndpointGetLocalRenderers}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointGetLocalRenderers(VidyoEndpoint* e, LmiVector(VidyoLocalRenderer)* localRenderers)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localRenderers}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointGetLocalRenderers(VidyoEndpoint* e, LmiVector(VidyoLocalRenderer)* localRenderers);

/**
{function visibility="private":
	{name: VidyoEndpointGetLocalWindowShares}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointGetLocalWindowShares(VidyoEndpoint* e, LmiVector(VidyoLocalWindowShare)* localWindowShares)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localWindowShares}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointGetLocalWindowShares(VidyoEndpoint* e, LmiVector(VidyoLocalWindowShare)* localWindowShares);

/**
{function visibility="private":
	{name: VidyoEndpointGetLocalMonitors}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointGetLocalMonitors(VidyoEndpoint* e, LmiVector(VidyoLocalMonitor)* localMonitors)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMonitors}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoEndpointGetLocalMonitors(VidyoEndpoint* e, LmiVector(VidyoLocalMonitor)* localMonitors);

/**
 {function:
	{name: VidyoEndpointRegisterLocalRendererEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about renderer events. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointRegisterLocalRendererEventListener(VidyoEndpoint* e, VidyoEndpointOnLocalRendererAdded onAdded, VidyoEndpointOnLocalRendererRemoved onRemoved, VidyoEndpointOnLocalRendererSelected onSelected, VidyoEndpointOnLocalRendererStateUpdated onStateUpdated)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered for every new or existing window added.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when the window is removed.}
	}
	{parameter:
		{name: onSelected}
		{description: Callback that is triggered when a window is selected for use.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when a window state changes.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterLocalRendererEventListener(VidyoEndpoint* e, VidyoEndpointOnLocalRendererAdded onAdded, VidyoEndpointOnLocalRendererRemoved onRemoved, VidyoEndpointOnLocalRendererSelected onSelected, VidyoEndpointOnLocalRendererStateUpdated onStateUpdated);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterLocalRendererEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters local renderer event notifications. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointUnregisterLocalRendererEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterLocalRendererEventListener(VidyoEndpoint* e);

/**
 {function:
	{name: VidyoEndpointRegisterRemoteRendererEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about renderer events.}
	{prototype: LmiBool VidyoEndpointRegisterRemoteRendererEventListener(VidyoEndpoint* e, VidyoEndpointOnRemoteRendererAdded onAdded, VidyoEndpointOnRemoteRendererRemoved onRemoved, VidyoEndpointOnRemoteRendererStateUpdated onStateUpdated)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onAdded}
		{description: .}
	}
	{parameter:
		{name: onRemoved}
		{description: }
	}
 	{parameter:
 		{name: onStateUpdated}
 		{description:}
 	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterRemoteRendererEventListener(VidyoEndpoint* e, VidyoEndpointOnRemoteRendererAdded onAdded, VidyoEndpointOnRemoteRendererRemoved onRemoved, VidyoEndpointOnRemoteRendererStateUpdated onStateUpdated);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterRemoteRendererEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters remote renderer event notifications.}
	{prototype: LmiBool VidyoEndpointUnregisterRemoteRendererEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterRemoteRendererEventListener(VidyoEndpoint* e);

/**
 {function:
	{name: VidyoEndpointRegisterRemoteSpeakerEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about speaker events.}
	{prototype: LmiBool VidyoEndpointRegisterRemoteSpeakerEventListener(VidyoEndpoint* e, VidyoEndpointOnRemoteSpeakerAdded onAdded, VidyoEndpointOnRemoteSpeakerRemoved onRemoved, VidyoEndpointOnRemoteSpeakerStateUpdated onStateUpdated)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered for every new or existing window added.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when the window is removed.}
	}
 	{parameter:
 		{name: onStateUpdated}
 		{description:}
 	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterRemoteSpeakerEventListener(VidyoEndpoint* e, VidyoEndpointOnRemoteSpeakerAdded onAdded, VidyoEndpointOnRemoteSpeakerRemoved onRemoved, VidyoEndpointOnRemoteSpeakerStateUpdated onStateUpdated);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterRemoteSpeakerEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters remote speaker event notifications.}
	{prototype: LmiBool VidyoEndpointUnregisterRemoteSpeakerEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointUnregisterRemoteSpeakerEventListener(VidyoEndpoint* e);

/**
{function:
	{name: VidyoEndpointLocalCameraSelectAdvanced}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointLocalCameraSelectAdvanced(VidyoEndpoint *e, const VidyoLocalCamera *localCamera)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localCamera}
		{description: }
	}
}
*/
void VidyoEndpointLocalCameraSelectAdvanced(VidyoEndpoint *e, const VidyoLocalCamera *localCamera);

/**
{function:
	{name: VidyoEndpointLocalCameraUnselectAdvanced}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointLocalCameraUnselectAdvanced(VidyoEndpoint *e, const VidyoLocalCamera *localCamera)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localCamera}
		{description: }
	}
}
*/
void VidyoEndpointLocalCameraUnselectAdvanced(VidyoEndpoint *e, const VidyoLocalCamera *localCamera);

/**
{function:
	{name: VidyoEndpointLocalMicrophoneSelectAdvanced}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointLocalMicrophoneSelectAdvanced(VidyoEndpoint *e, const VidyoLocalMicrophone *localMicrophone, LmiBool previewedPreCall)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMicrophone}
		{description: }
	}
	{parameter:
		{name: previewedPreCall}
		{description: }
	}
}
*/
void VidyoEndpointLocalMicrophoneSelectAdvanced(VidyoEndpoint *e, const VidyoLocalMicrophone *localMicrophone, LmiBool previewedPreCall);

/**
{function:
	{name: VidyoEndpointLocalMicrophoneUnselectAdvanced}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointLocalMicrophoneUnselectAdvanced(VidyoEndpoint *e, const VidyoLocalMicrophone *localMicrophone)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMicrophone}
		{description: }
	}
}
*/
void VidyoEndpointLocalMicrophoneUnselectAdvanced(VidyoEndpoint *e, const VidyoLocalMicrophone *localMicrophone);

/**
{function:
	{name: VidyoEndpointLocalSpeakerSelectAdvanced}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointLocalSpeakerSelectAdvanced(VidyoEndpoint *e, const VidyoLocalSpeaker *localSpeaker)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localSpeaker}
		{description: }
	}
}
*/
void VidyoEndpointLocalSpeakerSelectAdvanced(VidyoEndpoint *e, const VidyoLocalSpeaker *localSpeaker);

/**
{function:
	{name: VidyoEndpointLocalSpeakerUnselectAdvanced}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointLocalSpeakerUnselectAdvanced(VidyoEndpoint *e, const VidyoLocalSpeaker *localSpeaker)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localSpeaker}
		{description: }
	}
}
*/
void VidyoEndpointLocalSpeakerUnselectAdvanced(VidyoEndpoint *e, const VidyoLocalSpeaker *localSpeaker);

/**
{function:
	{name: VidyoEndpointLocalWindowShareSelectAdvanced}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointLocalWindowShareSelectAdvanced(VidyoEndpoint *e, const VidyoLocalWindowShare *localWindowShare)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localWindowShare}
		{description: }
	}
}
*/
void VidyoEndpointLocalWindowShareSelectAdvanced(VidyoEndpoint *e, const VidyoLocalWindowShare *localWindowShare);

/**
{function:
	{name: VidyoEndpointLocalWindowShareUnselectAdvanced}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointLocalWindowShareUnselectAdvanced(VidyoEndpoint *e, const VidyoLocalWindowShare *localWindowShare)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localWindowShare}
		{description: }
	}
}
*/
void VidyoEndpointLocalWindowShareUnselectAdvanced(VidyoEndpoint *e, const VidyoLocalWindowShare *localWindowShare);

/**
{function:
	{name: VidyoEndpointVirtualCameraSelectAdvanced}
	{parent: VidyoEndpoint}
	{description: Select a specific virtual video source which is a camera and add it to all remote renderers. The real camera will be unselected. On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointVirtualCameraSelectAdvanced(VidyoEndpoint *e, const VidyoVirtualVideoSource *virtualCamera)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object}
	}
	{parameter:
		{name: virtualCamera}
		{description: The VidyoVirtualVideoSource object to select. NULL to unselect the selected virtual camera.}
	}
}
*/
void VidyoEndpointVirtualCameraSelectAdvanced(VidyoEndpoint *e, const VidyoVirtualVideoSource *virtualCamera);

/**
{function:
    {name: VidyoEndpointVirtualSourceWindowShareSelectAdvanced}
    {parent: VidyoEndpoint}
    {description: Select a specific virtual video source which is a window share and add it to all remote renderers. The local window share will be unselected. On, iOS, this API must be called from main(UI) thread.}
    {prototype: void VidyoEndpointVirtualSourceWindowShareSelectAdvanced(VidyoEndpoint *e, const VidyoVirtualVideoSource *virtualWindowShare)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {parameter:
        {name: virtualWindowShare}
        {description: The VidyoVirtualVideoSource object to select. NULL to unselect the selected virtual window share.}
    }
}
*/
void VidyoEndpointVirtualSourceWindowShareSelectAdvanced(VidyoEndpoint *e, const VidyoVirtualVideoSource *virtualWindowShare);

/**
{function:
	{name: VidyoEndpointVirtualVideoSourceUnselectAdvanced}
	{parent: VidyoEndpoint}
	{description: Unselect a specific virtual video source. On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointVirtualVideoSourceUnselectAdvanced(VidyoEndpoint *e, const VidyoVirtualVideoSource *virtualVideoSource)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: virtualVideoSource}
		{description: The VidyoVirtualVideoSource object to unselect.}
	}
}
*/
void VidyoEndpointVirtualVideoSourceUnselectAdvanced(VidyoEndpoint *e, const VidyoVirtualVideoSource *virtualVideoSource);

/**
{function:
	{name: VidyoEndpointLocalMonitorSelectAdvanced}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointLocalMonitorSelectAdvanced(VidyoEndpoint *e, const VidyoLocalMonitor *localMonitor)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMonitor}
		{description: }
	}
}
*/
void VidyoEndpointLocalMonitorSelectAdvanced(VidyoEndpoint *e, const VidyoLocalMonitor *localMonitor);

/**
{function:
	{name: VidyoEndpointLocalMonitorUnselectAdvanced}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointLocalMonitorUnselectAdvanced(VidyoEndpoint *e, const VidyoLocalMonitor *localMonitor)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localMonitor}
		{description: }
	}
}
*/
void VidyoEndpointLocalMonitorUnselectAdvanced(VidyoEndpoint *e, const VidyoLocalMonitor *localMonitor);

/**
{function:
	{name: VidyoEndpointLocalRendererSelectAdvanced}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointLocalRendererSelectAdvanced(VidyoEndpoint *e, const VidyoLocalRenderer *localRenderer)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localRenderer}
		{description: }
	}
}
*/
void VidyoEndpointLocalRendererSelectAdvanced(VidyoEndpoint *e, const VidyoLocalRenderer *localRenderer);

/**
{function:
	{name: VidyoEndpointLocalRendererUnselectAdvanced}
	{parent: VidyoEndpoint}
	{description: On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointLocalRendererUnselectAdvanced(VidyoEndpoint *e, const VidyoLocalRenderer *localRenderer)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: localRenderer}
		{description: }
	}
}
*/
void VidyoEndpointLocalRendererUnselectAdvanced(VidyoEndpoint *e, const VidyoLocalRenderer *localRenderer);

/**
{type:
	{name: VidyoEndpointTone}
	{parent: VidyoEndpoint}
	{description: }
	{value: {name: VIDYO_ENDPOINTTONE_Join} {description: }}
	{value: {name: VIDYO_ENDPOINTTONE_Leave} {description: }}
	{value: {name: VIDYO_ENDPOINTTONE_MsgRecv} {description: }}
    {value: {name: VIDYO_ENDPOINTTONE_MsgSend} {description: }}
    {value: {name: VIDYO_ENDPOINTTONE_Ring} {description: }}
    {value: {name: VIDYO_ENDPOINTTONE_Max} {description: }}
}
*/
typedef enum {
	VIDYO_ENDPOINTTONE_Join,
	VIDYO_ENDPOINTTONE_Leave,
	VIDYO_ENDPOINTTONE_MsgRecv,
    VIDYO_ENDPOINTTONE_MsgSend,
    VIDYO_ENDPOINTTONE_Ring,
    VIDYO_ENDPOINTTONE_Max
} VidyoEndpointTone;

LmiBool LmiVidyoEndpointSetTone(VidyoEndpoint* e, VidyoEndpointTone tone, const LmiString* filePath);
VidyoLmiAudioFrame* LmiVidyoEndpointGetTone(VidyoEndpoint* e, VidyoEndpointTone tone);

struct LmiResourceManager_;
struct LmiResourceManager_* VidyoEndpointGetResourceManager(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetStats}
	{parent: VidyoEndpoint}
	{description: Fill in data in VidyoEndpointStats from an endpoint.}
	{prototype: VidyoEndpointStats *VidyoEndpointGetStats(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object to extract stats.}}
	{return: Pointer to the endpoint stats on Success or NULL on failure.}
}
*/
VidyoEndpointStats *VidyoEndpointGetStats(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetStatsJson}
	{parent: VidyoEndpoint}
	{description: Fill in data in VidyoEndpointStats from an endpoint.}
	{prototype: LmiString *VidyoEndpointGetStatsJson(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object to extract stats.}}
	{return: Pointer to the json encoded string containing the stats on Success or NULL on failure.}
}
*/
LmiString *VidyoEndpointGetStatsJson(VidyoEndpoint *e);

/**
 {function apigroup="simple":
	{name: VidyoEndpointDisable}
	{parent: VidyoEndpoint}
	{description: Stop background tasks associated with an endpoint. On, iOS, this API must be called from main(UI) thread.}
	{prototype: void VidyoEndpointDisable(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object to disable.}}
 }
 */
void VidyoEndpointDisable(VidyoEndpoint *e);

/**
 {type:
	{name: VidyoEndpointTradeOffProfile}
	{parent: VidyoEndpoint}
	{description: }
	{value: {name: VIDYO_ENDPOINTTRADEOFFPROFILE_High} {description: }}
	{value: {name: VIDYO_ENDPOINTTRADEOFFPROFILE_Medium} {description: }}
	{value: {name: VIDYO_ENDPOINTTRADEOFFPROFILE_Low} {description: }}
    {value: {name: VIDYO_ENDPOINTTRADEOFFPROFILE_LowAlt} {description: an alternative low configuration driving CPU tradeoff lower than VIDYO_ENDPOINTTRADEOFFPROFILE_Low}}
 }
 */
typedef enum {
	VIDYO_ENDPOINTTRADEOFFPROFILE_High,
	VIDYO_ENDPOINTTRADEOFFPROFILE_Medium,
	VIDYO_ENDPOINTTRADEOFFPROFILE_Low,
    VIDYO_ENDPOINTTRADEOFFPROFILE_LowAlt
} VidyoEndpointTradeOffProfile;

/**
 {function:
	{name: VidyoEndpointSetCpuTradeOffProfile}
	{parent: VidyoEndpoint}
	{description: Set CPU profile for the endpoint.}
	{prototype: LmiBool VidyoEndpointSetCpuTradeOffProfile(VidyoEndpoint *e, VidyoEndpointTradeOffProfile profile)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: profile} {description: The CPU tradeoff profile.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointSetCpuTradeOffProfile(VidyoEndpoint *e, VidyoEndpointTradeOffProfile profile);

/**
 {function:
	{name: VidyoEndpointGetCpuTradeOffProfile}
	{parent: VidyoEndpoint}
	{description: Get current CPU profile setting.}
	{prototype: VidyoEndpointTradeOffProfile VidyoEndpointGetCpuTradeOffProfile(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: The CUP tradeoff profile.}
 }
 */
VidyoEndpointTradeOffProfile VidyoEndpointGetCpuTradeOffProfile(VidyoEndpoint *e);


/**
{function apigroup="simple":
	{name: VidyoEndpointSetAdvancedOptions}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointSetAdvancedOptions(VidyoEndpoint *e, const char *options)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter:
		{name: options}
		{description: The experimental optons in JSON format.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetAdvancedOptions(VidyoEndpoint *e, const char *options);

/**
{function apigroup="simple":
 	{name: VidyoEndpointSetCameraNudgeConfiguration}
 	{parent: VidyoEndpoint}
 	{description: Sets camera PTZ nudge configuration.}
 	{prototype: LmiBool VidyoEndpointSetCameraNudgeConfiguration(VidyoEndpoint *e, const char *cameraNudgeConfig)}
 	{parameter: {name: e} {description: The endpoint object.}}
 	{parameter:
 		{name: cameraNudgeConfig}
 		{description: Camera PTZ nudge configuration in JSON format.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointSetCameraNudgeConfiguration(VidyoEndpoint *e, const char *cameraNudgeConfig);

/**
 {type:
 	{name: VidyoEndpointBaseTransportType}
 	{parent: VidyoEndpoint}
 	{description: Base transport type. }
 	{value: {name: VIDYO_ENDPOINTBASETRANSPORTTYPE_TLS}
 		{description: Use TLS as base transport.}}
 	{value: {name: VIDYO_ENDPOINTBASETRANSPORTTYPE_TCP}
 		{description: Use TCP as base transport.}}
 	{value: {name: VIDYO_ENDPOINTBASETRANSPORTTYPE_UDP}
 		{description: Use UDP as base transport.}}
 }
 */
typedef enum
{
	VIDYO_ENDPOINTBASETRANSPORTTYPE_TLS,
	VIDYO_ENDPOINTBASETRANSPORTTYPE_TCP,
	VIDYO_ENDPOINTBASETRANSPORTTYPE_UDP
} VidyoEndpointBaseTransportType;

/**
 {function:
 	{name: VidyoEndpointSetTURNAddressCredentials}
 	{parent: VidyoEndpoint}
 	{description: Set TURN server and credentials information.}
 	{prototype: LmiBool VidyoEndpointSetTURNAddressCredentials(VidyoEndpoint* e, const char* turnAddress, LmiUint port, const char* username, const char* password, VidyoEndpointBaseTransportType baseTransport)}
 	{parameter:
 		{name: e}
 		{description: The endpoint object.}
 	}
 	{parameter:
 		{name: turnAddress}
 		{description: TURN server FQDN or IP address.}
 	}
 	{parameter:
 		{name: port}
 		{description: TURN server port.}
 	}
 	{parameter:
 		{name: username}
 		{description: TURN server user name.}
 	}
 	{parameter:
 		{name: password}
 		{description: TURN server password.}
 	}
 	{parameter:
 		{name: baseTransport}
 		{description: Base transport type.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointSetTURNAddressCredentials(VidyoEndpoint* e, const char* turnAddress, LmiUint port, const char* username, const char* password, VidyoEndpointBaseTransportType baseTransport);

/**
 {function:
 	{name: VidyoEndpointSetTURNServerCertificateRequired}
 	{parent: VidyoEndpoint}
 	{description: Set TURN server certificate requirement.}
 	{prototype: LmiBool VidyoEndpointSetTURNServerCertificateRequired(VidyoEndpoint* e, LmiBool required)}
 	{parameter:
 		{name: e}
 		{description: The endpoint object.}
 	}
 	{parameter:
 		{name: required}
 		{description: LMI_TRUE if TURN server certificate is required.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointSetTURNServerCertificateRequired(VidyoEndpoint* e, LmiBool required);

/**
 {function:
 	{name: VidyoEndpointSetMaxSendBitRate}
 	{parent: VidyoEndpoint}
 	{description: Sets the max send bit rate for all sources.}
 	{prototype: void VidyoEndpointSetMaxSendBitRate(VidyoEndpoint* e, LmiUint bitRate)}
 	{parameter:
 		{name: e}
 		{description: The endpoint object.}
 	}
 	{parameter:
 		{name: bitRate}
 		{description: The max send bit rate for all sources, in bps.}
 	}
 	{note: This maximum bitrate will be distributed among all the local sources.}
 }
 */
void VidyoEndpointSetMaxSendBitRate(VidyoEndpoint* e, LmiUint bitRate);

/**
{function:
    {name: VidyoEndpointGetMaxSendBitRate}
    {parent: VidyoEndpoint}
    {description: Return the max send bit rate configured by application. }
    {prototype: LmiUint VidyoEndpointGetMaxSendBitRate(VidyoEndpoint* e)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {return: the max send bit rate configured by application, in bps}
}
*/
LmiUint VidyoEndpointGetMaxSendBitRate(VidyoEndpoint* e);

/**
{function:
    {name: VidyoEndpointSetMaxReceiveBitRate}
    {parent: VidyoEndpoint}
    {description: Request server to limit bit rate sent to this client.}
    {prototype: void VidyoEndpointSetMaxReceiveBitRate(VidyoEndpoint* e, LmiUint bitRate)}
    {parameter:
        {name: e}
        {description: The endpoint object.}
    }
    {parameter:
        {name: bitRate}
        {description: The maximum receiving bit rate request sent to server, in bps.}
    }
}
 */
void VidyoEndpointSetMaxReceiveBitRate(VidyoEndpoint* e, LmiUint bitRate);

/**
{function:
    {name: VidyoEndpointGetMaxReceiveBitRate}
    {parent: VidyoEndpoint}
    {description: Return the max receive bit rate configured by application. }
    {prototype: LmiUint VidyoEndpointGetMaxReceiveBitRate(VidyoEndpoint* e)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {return: the max receive bit rate configured by application, in kbps}
}
*/
LmiUint VidyoEndpointGetMaxReceiveBitRate(VidyoEndpoint* e);

/*
{function visibility="private":
	{name: VidyoEndpointGetLicenseKey}
	{parent: VidyoEndpoint}
	{description: Get the license key of the endpoint.}
	{prototype: LmiBool VidyoEndpointGetLicenseKey(VidyoEndpoint* e, LmiString *licenseKey)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: licenseKey}{description: The lincense key for the endpoint}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointGetLicenseKey(VidyoEndpoint* e, LmiString *licenseKey);

/**
{function visibility="public":
 	{name: VidyoEndpointSetLicenseKey}
 	{parent: VidyoEndpoint}
 	{description: Set the license key of the endpoint.}
 	{prototype: LmiBool VidyoEndpointSetLicenseKey(VidyoEndpoint* e, LmiString *licenseKey)}
 	{parameter: {name: e} {description: The endpoint object.}}
 	{parameter: {name: licenseKey}{description: The lincense key for the endpoint}}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetLicenseKey(VidyoEndpoint* e, LmiString *licenseKey);

/**
 {function:
 {name: VidyoEndpointSetProductInfo}
 {parent: VidyoEndpoint}
 {description: Set the product information of the endpoint.}
 {prototype: LmiBool VidyoEndpointSetProductInfo(VidyoEndpoint *e, const LmiVector(VidyoProperty) *productInfo, const LmiVector(VidyoProperty) *supportedFeature)}
 {parameter: {name: e} {description: The endpoint object.}}
 {parameter: {name: productInfo}{description: The product info of the endpoint}}
 {parameter: {name: supportedFeature}{description: The features this endpoint can support}}
 {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointSetProductInfo(VidyoEndpoint *e, const LmiVector(VidyoProperty)* productInfo, const LmiVector(VidyoProperty) *supportedFeature);

/**
 {function  deprecated="yes" replaced-by="VidyoEndpointGetProductInfoAsync":
 {name: VidyoEndpointGetProductInfo}
 {parent: VidyoEndpoint}
 {description: Get the product information of the endpoint.}
 {prototype: LmiBool VidyoEndpointGetProductInfo(VidyoEndpoint *e, LmiVector(VidyoProperty) *productInfo, LmiVector(VidyoProperty) *supportedFeature)}
 {parameter: {name: e} {description: The endpoint object.}}
 {parameter: {name: productInfo}{description: The product info of the endpoint}}
 {parameter: {name: supportedFeature}{description: The features this endpoint can support}}
 {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointGetProductInfo(VidyoEndpoint *e, LmiVector(VidyoProperty) *productInfo, LmiVector(VidyoProperty) *supportedFeature);

/**
 {callback:
	{name: VidyoEndpointOnGetProductInfoComplete}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered when getting the product information completes. }
	{prototype: void (*VidyoEndpointOnGetProductInfoComplete)(VidyoEndpoint* e, const LmiVector(VidyoProperty) *productInfo, const LmiVector(VidyoProperty) *supportedFeature)}
	{parameter:
		{name: e}
		{description: The endpoint object.}
	}
	{parameter:
		{name: productInfo}
		{description: Array of product information.}
	}
	{parameter:
		{name: supportedFeature}
		{description: Array of supported features.}
	}
}
*/
typedef void (*VidyoEndpointOnGetProductInfoComplete)(VidyoEndpoint* e, const LmiVector(VidyoProperty) *productInfo, const LmiVector(VidyoProperty) *supportedFeature);

/**
 {function:
	 {name: VidyoEndpointGetProductInfoAsync}
	 {parent: VidyoEndpoint}
	 {description: Get the product information of the endpoint.}
	 {prototype: LmiBool VidyoEndpointGetProductInfoAsync(VidyoEndpoint *e, VidyoEndpointOnGetProductInfoComplete onComplete)}
	 {parameter: {name: e} {description: The endpoint object.}}
	 {parameter: {name: onComplete}{description: The callback that is triggered when getting the product information completes.}}
	 {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoEndpointGetProductInfoAsync(VidyoEndpoint *e, VidyoEndpointOnGetProductInfoComplete onComplete);

/**
{type:
	{name: VidyoClientAppLogLevel}
	{parent: VidyoEndpoint}
	{description: }
	{value: {name: VIDYO_CLIENTAPPLOGLEVEL_Info} {description: }}
	{value: {name: VIDYO_CLIENTAPPLOGLEVEL_Debug} {description: }}
	{value: {name: VIDYO_CLIENTAPPLOGLEVEL_Warning} {description: }}
	{value: {name: VIDYO_CLIENTAPPLOGLEVEL_Error} {description: }}
}
*/
typedef enum {
	VIDYO_CLIENTAPPLOGLEVEL_Info,
	VIDYO_CLIENTAPPLOGLEVEL_Debug,
	VIDYO_CLIENTAPPLOGLEVEL_Warning,
	VIDYO_CLIENTAPPLOGLEVEL_Error
} VidyoClientAppLogLevel;

/**
{function:
	{name: VidyoEndpointAppLog}
	{parent: VidyoEndpoint}
	{description: Log an application log message into the VidyoClient logs under the log category 'Application'. }
	{prototype: void VidyoEndpointAppLog(VidyoEndpoint* e, VidyoClientAppLogLevel logLevel, LmiString* logMsg)}
	{parameter:
		{name: e}
		{description: }
	}
	{parameter:
		{name: logLevel}
		{description: Specify logging level using enum, allowed values defined in VidyoClientAppLogLevel}
	}
	{parameter:
		{name: logMsg}
		{description: Log message to write to logfile}
	}
}
*/
void VidyoEndpointAppLog(VidyoEndpoint* e, VidyoClientAppLogLevel logLevel, LmiString* logMsg);

/**
{function:
    {name: VidyoEndpointSetFontFileName}
    {parent: VidyoEndpoint}
    {description: Set font file name. On, iOS, this API must be called from main(UI) thread.}
    {prototype: LmiBool VidyoEndpointSetFontFileName(VidyoEndpoint* e, const char *fontFileName)}
    {parameter:
        {name: e}
        {description: }
    }
    {parameter:
        {name: fontFileName}
        {description: Pointer to char array that contains font file name.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetFontFileName(VidyoEndpoint* e, const char *fontFileName);


/**
{function:
 	{name: VidyoEndpointRequestToSpeak}
 	{parent: VidyoEndpoint}
 	{description: Request to speak after sever paused audio stream from selected local microphone. On, iOS, this API must be called from main(UI) thread.}
 	{prototype: LmiBool VidyoEndpointRequestToSpeak(VidyoEndpoint* e)}
 	{parameter:
 		{name: e}
 		{description: The VidyoEndpoint object.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRequestToSpeak(VidyoEndpoint* e);

/**
{function:
 	{name: VidyoEndpointRequestToResumeVideo}
 	{parent: VidyoEndpoint}
 	{description: Request to resume sending video after sever paused video stream from selected local camera. Will resume both local and remote streams and show preview in case it is on. Allowed only in case of soft mute. On, iOS, this API must be called from main(UI) thread.}
 	{prototype: LmiBool VidyoEndpointRequestToResumeVideo(VidyoEndpoint* e)}
 	{parameter:
 		{name: e}
 		{description: The VidyoEndpoint object.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRequestToResumeVideo(VidyoEndpoint* e);

/**
{function apigroup="simple":
    {name: VidyoEndpointRegisterVirtualVideoSourceEventListener}
    {parent: VidyoEndpoint}
    {description: Registers to get notified about virtual video source events. On, iOS, this API must be called from main(UI) thread.}
    {prototype: LmiBool VidyoEndpointRegisterVirtualVideoSourceEventListener(VidyoEndpoint* e, VidyoEndpointOnVirtualVideoSourceAdded onAdded, VidyoEndpointOnVirtualVideoSourceRemoved onRemoved, VidyoEndpointOnVirtualVideoSourceStateUpdated onStateUpdated, VidyoEndpointOnVirtualVideoSourceExternalMediaBufferReleased onExternalMediaBufferReleased)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {parameter:
        {name: onAdded}
        {description: Callback that is triggered for every new virtual video source added.}
    }
    {parameter:
        {name: onRemoved}
        {description: Callback that is triggered when the virtual video source is removed.}
    }
    {parameter:
        {name: onStateUpdated}
        {description: Callback that is triggered when a virtual video source state changes.}
    }
    {parameter:
        {name: onExternalMediaBufferReleased}
        {description: Callback that is triggered each time when VidyoClient is done with an application allocated buffer that holds raw video frame.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterVirtualVideoSourceEventListener(VidyoEndpoint* e, VidyoEndpointOnVirtualVideoSourceAdded onAdded, VidyoEndpointOnVirtualVideoSourceRemoved onRemoved, VidyoEndpointOnVirtualVideoSourceStateUpdated onStateUpdated, VidyoEndpointOnVirtualVideoSourceExternalMediaBufferReleased onExternalMediaBufferReleased);

/**
{function apigroup="simple":
    {name: VidyoEndpointUnregisterVirtualVideoSourceEventListener}
    {parent: VidyoEndpoint}
    {description: Unregisters virtual video source event notifications. On, iOS, this API must be called from main(UI) thread.}
    {prototype: LmiBool VidyoEndpointUnregisterVirtualVideoSourceEventListener(VidyoEndpoint* e)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointUnregisterVirtualVideoSourceEventListener(VidyoEndpoint* e);

/**
{function:
    {name: VidyoEndpointCreateVirtualVideoSource}
    {parent: VidyoEndpoint}
    {description: Create a virtual video source. The user can create a maximum of 10 virtual video sources. On, iOS, this API must be called from main(UI) thread.}
    {prototype: LmiBool VidyoEndpointCreateVirtualVideoSource(VidyoEndpoint* e, VidyoVirtualVideoSourceType type, const char *id, const char *name)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {parameter:
        {name: type}
        {description: Type of the virtual video source.}
    }
    {parameter:
        {name: id}
        {description: Unique ID.}
    }
    {parameter:
        {name: name}
        {description: Name of the virtual video source.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointCreateVirtualVideoSource(VidyoEndpoint* e, VidyoVirtualVideoSourceType type, const char *id, const char *name);

/**
{function:
    {name: VidyoEndpointDestroyVirtualVideoSource}
    {parent: VidyoEndpoint}
    {description: Destroy virtual video source. On, iOS, this API must be called from main(UI) thread.}
    {prototype: LmiBool VidyoEndpointDestroyVirtualVideoSource(VidyoEndpoint* e, const VidyoVirtualVideoSource* virtualVideoSource)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {parameter:
        {name: virtualVideoSource}
        {description: The virtual video source that will be destroyed.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointDestroyVirtualVideoSource(VidyoEndpoint* e, const VidyoVirtualVideoSource* virtualVideoSource);

/**
{function:
    {name: VidyoEndpointSelectVirtualCamera}
    {parent: VidyoEndpoint}
    {description: Selects the virtual video source that will be shown to remote participants. The real camera will be unselected if it's selected. On, iOS, this API must be called from main(UI) thread.}
    {prototype: LmiBool VidyoEndpointSelectVirtualCamera(VidyoEndpoint* e, const VidyoVirtualVideoSource* virtualCamera)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {parameter:
        {name: virtualCamera}
        {description: The virtual video source that will be shown to remote participants.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectVirtualCamera(VidyoEndpoint* e, const VidyoVirtualVideoSource* virtualCamera);

/**
{function:
    {name: VidyoEndpointSelectVirtualSourceWindowShare}
    {parent: VidyoEndpoint}
    {description: Selects the virtual video source which is window share that will be shown to remote participants. The local window share will be unselected if it's selected. On, iOS, this API must be called from main(UI) thread.}
    {prototype: LmiBool VidyoEndpointSelectVirtualSourceWindowShare(VidyoEndpoint* e, const VidyoVirtualVideoSource* virtualWindowShare)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {parameter:
        {name: virtualWindowShare}
        {description: The virtual video source that will be shown to remote participants.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectVirtualSourceWindowShare(VidyoEndpoint* e, const VidyoVirtualVideoSource* virtualWindowShare);

/**
{callback:
	{name: VidyoEndpointOnVirtualAudioSourceAdded}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a virtual audio source becomes available for a VidyoEndpoint to add into a conference.}
	{prototype: void(*VidyoEndpointOnVirtualAudioSourceAdded)(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource)}
	{parameter: {name: e} {description: The VidyoEndpoint that triggered the callback.}}
	{parameter: {name: virtualAudioSource} {description: The VidyoVirtualAudioSource object that was created and now available for a VidyoEndpoint to add into a conference.}}
}
*/
typedef void(*VidyoEndpointOnVirtualAudioSourceAdded)(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource);

/**
{callback:
	{name: VidyoEndpointOnVirtualAudioSourceRemoved}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time a virtual audio source is no longer available for a VidyoEndpoint to add into a conference.}
	{prototype: void(*VidyoEndpointOnVirtualAudioSourceRemoved)(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource)}
	{parameter: {name: e} {description: The VidyoEndpoint that triggered the callback.}}
	{parameter: {name: virtualAudioSource} {description: The VidyoVirtualAudioSource object that was removed from the VidyoEndpoint.}}
}
*/
typedef void(*VidyoEndpointOnVirtualAudioSourceRemoved)(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource);

/**
{callback:
	{name: VidyoEndpointOnVirtualAudioSourceSelected}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered when specific virtual audio source selected.}
	{prototype: void(*VidyoEndpointOnVirtualAudioSourceSelected)(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource)}
	{parameter: {name: e} {description: The VidyoEndpoint that triggered the callback.}}
	{parameter: {name: virtualAudioSource} {description: The virtual audio source object..}}
}
*/
typedef void(*VidyoEndpointOnVirtualAudioSourceSelected)(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource);

/**
{callback:
	{name: VidyoEndpointOnVirtualAudioSourceStateUpdated}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time there is a change in the state of the window such as being suspended or disabled.}
	{prototype: void(*VidyoEndpointOnVirtualAudioSourceStateUpdated)(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource, VidyoDeviceState state)}
	{parameter: {name: e} {description: The VidyoEndpoint that triggered the callback.}}
	{parameter: {name: virtualAudioSource} {description: The virtual audio source object..}}
	{parameter: {name: state} {description: New state of the virtual audio source.}}
}
*/
typedef void(*VidyoEndpointOnVirtualAudioSourceStateUpdated)(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource, VidyoDeviceState state);

/**
{callback:
	{name: VidyoEndpointOnVirtualAudioSourceExternalMediaBufferReleased}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered to notify application that VidyoClient is done with the buffer that holds raw audio frame. This buffer is the same one that was passed to VidyoClient via function VidyoVirtualAudioSourceSendFrameWithExternalData.}
	{prototype: void(*VidyoEndpointOnVirtualAudioSourceExternalMediaBufferReleased)(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource, LmiUint8* buffer, LmiSizeT size)}
	{parameter: {name: e} {description: The VidyoEndpoint that triggered the callback.}}
	{parameter: {name: virtualAudioSource} {description: The VidyoVirtualAudioSource object.}}
	{parameter: {name: buffer} {description: The buffer (allocated by application that holds raw data of the audio  frame) that is no longer being used by VidyoClient.}}
	{parameter: {name: size} {description: The size of the buffer.}}
}
*/
typedef void(*VidyoEndpointOnVirtualAudioSourceExternalMediaBufferReleased)(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource, LmiUint8* buffer, LmiSizeT size);

/**
{function:
	{name: VidyoEndpointRegisterVirtualAudioSourceEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about virtual audio source events. This is a synchronous interface.}
	{prototype: LmiBool VidyoEndpointRegisterVirtualAudioSourceEventListener(VidyoEndpoint* e, VidyoEndpointOnVirtualAudioSourceAdded onAdded, VidyoEndpointOnVirtualAudioSourceRemoved onRemoved, VidyoEndpointOnVirtualAudioSourceStateUpdated onStateUpdated, VidyoEndpointOnVirtualAudioSourceSelected onSelected, VidyoEndpointOnVirtualAudioSourceExternalMediaBufferReleased onExternalMediaBufferReleased)}
	{parameter: {name: e} {description: The VidyoEndpoint object.}}
	{parameter: {name: onAdded} {description: Callback that is triggered each time when virtual audio source is created.}}
	{parameter: {name: onRemoved} {description: Callback that is triggered each time when virtual audio source is removed.}}
	{parameter: {name: onStateUpdated} {description: Callback that is triggered each time when virtual audio source chagnes state.}}
	{parameter: {name: onSelected} {description: Callback that is triggered each time when virtual audio source is selected.}}
	{parameter: {name: onExternalMediaBufferReleased} {description: Callback that is triggered each time when VidyoClient is done with an application allocated buffer that holds raw audio frame data.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterVirtualAudioSourceEventListener(VidyoEndpoint* e, VidyoEndpointOnVirtualAudioSourceAdded onAdded, VidyoEndpointOnVirtualAudioSourceRemoved onRemoved, VidyoEndpointOnVirtualAudioSourceStateUpdated onStateUpdated, VidyoEndpointOnVirtualAudioSourceSelected onSelected, VidyoEndpointOnVirtualAudioSourceExternalMediaBufferReleased onExternalMediaBufferReleased);

/**
{function:
	{name: VidyoEndpointUnregisterVirtualAudioSourceEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters virtual audio source event notifications. This is a synchronous interface.}
	{prototype: LmiBool VidyoEndpointUnregisterVirtualAudioSourceEventListener(VidyoEndpoint* e)}
	{parameter: {name: e} {description: The VidyoEndpoint object.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointUnregisterVirtualAudioSourceEventListener(VidyoEndpoint* e);

/**
{function:
	{name: VidyoEndpointCreateVirtualAudioSource}
	{parent: VidyoEndpoint}
	{description: Create virtual audio source. This is an asynchronous interface. The VidyoEndpointOnVirtualAudioSourceAdded will trigger.}
	{prototype: LmiBool VidyoEndpointCreateVirtualAudioSource(VidyoEndpoint* e, const LmiString *name, const LmiString *id, const LmiString *configuration)}
	{parameter: {name: e} {description: The VidyoEndpoint object.}}
	{parameter: {name: name} {description: The name for the virtual audio source.}}
	{parameter: {name: id} {description: The id for the virtual audio source.}}
	{parameter: {name: configuration} {description: The string in JSON format that contain audio configuration that will be used for virtual audio source.
		{p: format - Specifies the media format. Possible values are: fl32 or in16. The default is fl32.}
		{p: Example: \{\'format\' : \'in16\'\}.}
		{p: sampleRate - Specifies the audio sample rate. Possible values are: 48000, 32000 and 16000. The default is 32000.}
		{p: Example: \{\'sampleRate\' : 48000\}.}
		{p: numberOfChannels - Specifies the number of channels. Possible values are: 1 or 2. The default is 1.}
		{p: Example: \{\'numberOfChannels\' : 2\}.}
		{p: frameInterval - Specifies the frame interval in ms. Possible values are: 20 or 40. The default is 20.}
		{p: Example: \{\'frameInterval\' : 40\}.}
	}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointCreateVirtualAudioSource(VidyoEndpoint* e, const LmiString *name, const LmiString *id, const LmiString *configuration);

/**
{function:
	{name: VidyoEndpointDestroyVirtualAudioSource}
	{parent: VidyoEndpoint}
	{description: Destroy virtual audio source. This is an asynchronous interface. The VidyoEndpointOnVirtualAudioSourceRemoved will trigger.}
	{prototype: LmiBool VidyoEndpointDestroyVirtualAudioSource(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource)}
	{parameter: {name: e} {description: The VidyoEndpoint object.}}
	{parameter: {name: virtualAudioSource} {description: The virtual audio source that will be destroyed.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointDestroyVirtualAudioSource(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource);

/**
{function :
	{name: VidyoEndpointSelectVirtualMicrophone}
	{parent: VidyoEndpoint}
	{description: Selects the virtual audio source which will be used as microphone. The real miicrophone will be unselected if it's selected. This is a synchronous interface. The VidyoEndpointOnVirtualAudioSourceStateUpdated will trigger.}
	{prototype: LmiBool VidyoEndpointSelectVirtualMicrophone(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource)}
	{parameter: {name: e} {description: The VidyoEndpoint object.}}
	{parameter: {name: virtualAudioSource} {description: The virtual audio source that will be used as microphone.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectVirtualMicrophone(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource);

/**
{function:
	{name: VidyoEndpointSelectVirtualAudioContentShare}
	{parent: VidyoEndpoint}
	{description: Selects the virtual audio source which will be used for content share. The selected microphone which currently using for content share will be unselected. This is a synchronous interface. The VidyoEndpointOnVirtualAudioSourceStateUpdated will trigger.}
	{prototype: LmiBool VidyoEndpointSelectVirtualAudioContentShare(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource)}
	{parameter: {name: e} {description: The VidyoEndpoint object.}}
	{parameter: {name: virtualAudioSource} {description: The virtual audio source that will be used for content share.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectVirtualAudioContentShare(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource);

/**
{function:
	{name: VidyoEndpointSendVirtualAudioSourceFrame}
	{parent: VidyoEndpoint}
	{description: Sends already constructed audio frame to selected virtual audio source and then send it to local and/or remote streams. This is a synchronous interface.}
	{prototype: LmiBool VidyoEndpointSendVirtualAudioSourceFrame(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource, const VidyoAudioFrame* audioFrame)}
	{parameter: {name: e} {description: The VidyoEndpoint object.}}
	{parameter: {name: virtualAudioSource} {description: The virtual audio source.}}
	{parameter: {name: audioFrame} {description: The audio frame to be sent.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSendVirtualAudioSourceFrame(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource, const VidyoAudioFrame* audioFrame);

/**
{function:
	{name: VidyoEndpointSendVirtualAudioSourceFrameWithExternalData}
	{parent: VidyoEndpoint}
	{description: Pass raw buffer to selected virtual audio source, based on which the audio frame will be constructed and then send it to local and/or remote streams. This is an asynchronous interface. The VidyoEndpointOnVirtualAudioSourceExternalMediaBufferReleased will trigger when buffer is not needed anymore.}
	{prototype: LmiBool VidyoEndpointSendVirtualAudioSourceFrameWithExternalData(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource, const LmiUint8* buffer, LmiSizeT nbSamples, LmiTime elapsedTime)}
	{parameter: {name: e} {description: The VidyoEndpoint object.}}
	{parameter: {name: virtualAudioSource} {description: The virtual audio source.}}
	{parameter: {name: buffer} {description: The buffer with audio data to be sent.}}
	{parameter: {name: nbSamples} {description: The number of sample frames that will be represented in the audio data.}}
	{parameter: {name: elapsedTime} {description: The elapsed time.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSendVirtualAudioSourceFrameWithExternalData(VidyoEndpoint* e, const VidyoVirtualAudioSource* virtualAudioSource, const LmiUint8* buffer, LmiSizeT nbSamples, LmiTime elapsedTime);

/**
{function:
    {name: VidyoEndpointSetRendererOptionsForViewId}
    {parent: VidyoEndpoint}
    {description: Sets up the local renderer and conference tiles displaying options.}
    {prototype: LmiBool VidyoEndpointSetRendererOptionsForViewId(VidyoEndpoint* e, const LmiViewId* viewId, const char *options)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {parameter:
        {name: viewId}
        {description: A platform-specific view ID.}
    }
    {parameter:
        {name: options}
        {description: A char array that contains options in the JSON format.
            {p: EnableVerticalVideoCentering - Enable or disable the vertical centering of the video screen. Default is true.}
            {p: Example: \{\'EnableVerticalVideoCentering\':true\}.}
            {p: EnableFECCIconCustomLayout - Enable or disable the displaying of custom FECC icon in the right bottom corner. Default is false.}
            {p: Example: \{\'EnableFECCIconCustomLayout\':false\}.}
            {p: ViewingDistance - Sets viewing distance in feet. Default is 2.5f. Minimum and Maximum values are 1f & 15f respectively.}
            {p: Example: \{\'ViewingDistance\':2.5\}.}
            {p: ShowAudioTiles - Enable or disable the showing of audio tiles. Default is true.}
            {p: Example: \{\'ShowAudioTiles\':false\}.}
            {p: SetPipPosition - Sets the position of the preview when it is displayed using picture-in-picture mode. 
                        Possible values for x: "PipPositionLeft", "PipPositionCenter", "PipPositionRight".
                        Possible values for y: "PipPositionBottom", "PipPositionCenter", "PipPositionTop".
                        lockPip is a boolean indicating whether to lock the preview in this position (by default is false).
                        When false, the user may drag the preview to another location using the mouse or a touch.}
            {p: Example: \{\'SetPipPosition\':\{\'x\':\'PipPositionRight\', \'y\': \'PipPositionTop\', \'lockPip\':true\}\}.}
            {p: SetPixelDensity - Sets the rendering environment's pixel density.}
            {p: Example: \{\'SetPixelDensity\':326.0f.
                        However for latest rendering environment, being released for iOS only, this option is used to choose surface density with possible values as
                        0.5 (everything will be smaller) / 1.0 (default) / 2.0 (everything will be larger).\}}
            {p: SetTouchAsInputDevice - Enable or disable touch as a input device. Default is false.}
            {p: Example: \{\'SetTouchAsInputDevice\':false\}.}
            {p: EnableExpandedCameraControl - Enable or disable expanded camera control. Default is true for mobile devices and false for non-mobile devices.}
            {p: Example: \{\'EnableExpandedCameraControl\':false\}.}
            {p: BorderStyle - Sets the border style of the preview. None: Do not put a border on any tiles. Same: Put the same colored border on each tile. Highlight: Put a border on each tile, but tiles associated with the highest ranked participant will have a different colored border than the others. Flash: Tiles associated with the highest ranked participant have a different colored border than the others. When the highest ranked participant changes, the border of the new highest ranked participant flashes momentarily.
                        Possible values for BorderStyle: "None", "Same", "Highlight", "Flash"}
            {p: Example: \{\'BorderStyle\':\'Same\'}\}
            {p: enablePreviewMirroring - Specifies whether mirroring will be enabled or disabled for local cameras. Possible values are true or false. The default is true.}
            {p: Example: \{\'EnablePreviewMirroring\' : true\}.}
        }
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetRendererOptionsForViewId(VidyoEndpoint* e, const LmiViewId* viewId, const char *options);

/**
{function:
    {name: VidyoEndpointGetRendererOptionsForViewId}
    {parent: VidyoEndpoint}
    {description: Gets the local renderer and conference tiles displaying options.}
    {prototype: LmiString* VidyoEndpointGetRendererOptionsForViewId(VidyoEndpoint* e, const LmiViewId* viewId)}
    {parameter:
        {name: e}
        {description: The VidyoEndpoint object.}
    }
    {parameter:
        {name: viewId}
        {description: A platform-specific view ID.}
    }
    {return: Pointer to LmiString that contains the local renderer and conference tiles displaying options in JSON format.}
}
*/
LmiString* VidyoEndpointGetRendererOptionsForViewId(VidyoEndpoint* e, const LmiViewId* viewId);

/**
{function:
	{name: VidyoEndpointAddMessageClass}
	{parent: VidyoEndpoint}
	{description: Add message class that application should receive.}
	{prototype: LmiBool VidyoEndpointAddMessageClass(VidyoEndpoint *e, const char *messageClass)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: messageClass}
		{description: Name of message class. MSGCLASS_HUNTER is added by default so application will receive only these message class.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointAddMessageClass(VidyoEndpoint *e, const char *messageClass);

/**
{function:
	{name: VidyoEndpointSetReceiveRawMessage}
	{parent: VidyoEndpoint}
	{description: Tell the library to send the raw chat messages to the application.}
	{prototype: LmiBool VidyoEndpointSetReceiveRawMessage(VidyoEndpoint *e, LmiBool receiveRawMessage)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: receiveRawMessage}
		{description: Set it to LMI_TRUE to receive raw message with header, LMI_FALSE to reseive only the message body.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetReceiveRawMessage(VidyoEndpoint *e, LmiBool receiveRawMessage);

/**
{function visibility="private":
	{name: VidyoEndpointSetMaxCropping}
	{parent: VidyoEndpoint}
	{description: This function sets the maximum percentage of a dimension that can be cropped from a video. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSetMaxCropping(VidyoEndpoint *e, LmiUint maxCropPct)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: maxCropPct}
		{description: When fitting a video into a tile, this sets the maximum percentage that can be cropped to achieve a better fit.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetMaxCropping(VidyoEndpoint *e, LmiUint maxCropPct);

/**
{callback:
	{name: VidyoEndpointOnDisableVideoOnLowBwStarted}
	{parent: VidyoEndpoint}
	{description: Callback that will be triggered when disable video on low bandwidth is started.}
	{prototype: void (*VidyoEndpointOnDisableVideoOnLowBwStarted)(const VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: Pointer to the VidyoEndpoint that triggered the callback. }
	}
}
*/
typedef void (*VidyoEndpointOnDisableVideoOnLowBwStarted)(const VidyoEndpoint* e);

/**
{callback:
	{name: VidyoEndpointOnDisableVideoOnLowBwEnded}
	{parent: VidyoEndpoint}
	{description: Callback that will be triggered when disable video on low bandwidth is ended.}
	{prototype: void (*VidyoEndpointOnDisableVideoOnLowBwEnded)(const VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: Pointer to the VidyoEndpoint that triggered the callback. }
	}
}
*/
typedef void (*VidyoEndpointOnDisableVideoOnLowBwEnded)(const VidyoEndpoint* e);

/**
{function:
	{name: VidyoEndpointRegisterDisableVideoOnLowBwEventListener}
	{parent: VidyoEndpoint}
	{description: Register callbacks for disable video on low bandwidth.}
	{prototype: void VidyoEndpointRegisterDisableVideoOnLowBwEventListener(VidyoEndpoint* e, VidyoEndpointOnDisableVideoOnLowBwStarted onStarted, VidyoEndpointOnDisableVideoOnLowBwEnded onEnded)}
	{parameter: {name: e} {description: VidyoEndpoint object.}}
	{parameter: {name: onStarted} {description: Callback that will be triggered when disable video on low bandwidth is started.}}
	{parameter: {name: onEnded} {description: Callback that will be triggered when disable video on low bandwidth is ended.}}
}
*/
void VidyoEndpointRegisterDisableVideoOnLowBwEventListener(VidyoEndpoint* e, VidyoEndpointOnDisableVideoOnLowBwStarted onStarted, VidyoEndpointOnDisableVideoOnLowBwEnded onEnded);

/**
{function:
	{name: VidyoEndpointUnregisterDisableVideoOnLowBwEventListener}
	{parent: VidyoEndpoint}
	{description: Unregister callbacks for disable video on low bandwidth.}
	{prototype: void VidyoEndpointUnregisterDisableVideoOnLowBwEventListener(VidyoEndpoint* e)}
	{parameter: {name: e} {description: VidyoEndpoint object.}}
}
*/
void VidyoEndpointUnregisterDisableVideoOnLowBwEventListener(VidyoEndpoint* e);

/**
 {function:
	{name: VidyoEndpointRegisterReconnectEventListener}
	{parent: VidyoEndpoint}
	{description: Registers to get notified about reconnecting events. Note: You should not perform long action in the callbacks.}
	{prototype: LmiBool VidyoEndpointRegisterReconnectEventListener(VidyoEndpoint* e, VidyoEndpointOnReconnecting onReconnecting, VidyoEndpointOnReconnected onReconnected, VidyoEndpointOnConferenceLost onConferenceLost)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onReconnecting}
		{description: Callback that is triggered each time the new attempt to reconnect will be started.}
	}
	{parameter:
		{name: onReconnected}
		{description: Callback that is triggered when reconnecting was successful.}
	}
	{parameter:
		{name: onConferenceLost}
		{description: Callback that is triggered when reconnecting failed.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterReconnectEventListener(VidyoEndpoint* e, VidyoEndpointOnReconnecting onReconnecting, VidyoEndpointOnReconnected onReconnected, VidyoEndpointOnConferenceLost onConferenceLost);

/**
 {function:
	{name: VidyoEndpointUnregisterReconnectEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters reconnecting event notifications.}
	{prototype: LmiBool VidyoEndpointUnregisterReconnectEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointUnregisterReconnectEventListener(VidyoEndpoint* e);

/**
 {function:
	{name: VidyoEndpointSwapStreamsBetweenViews}
	{parent: VidyoEndpoint}
	{description: Swap streams of the remote video sources between views.}
	{prototype: LmiBool VidyoEndpointSwapStreamsBetweenViews(VidyoEndpoint* e, const LmiViewId* firstView, const LmiViewId* secondView)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: firstView}
		{description: A platform specific view ID where the first remote video sources is displayed.}
	}
	{parameter:
		{name: secondView}
		{description: A platform specific view ID where the second remote video sources is displayed.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSwapStreamsBetweenViews(VidyoEndpoint* e, const LmiViewId* firstView, const LmiViewId* secondView);

/**
 {function:
	{name: VidyoEndpointRequestVideoForRemoteCamera}
	{parent: VidyoEndpoint}
	{description: Requests video for remote camera before assign.}
	{prototype: LmiBool VidyoEndpointRequestVideoForRemoteCamera(VidyoEndpoint* e, const VidyoRemoteCamera* remoteCamera, LmiUint width, LmiUint height, LmiTime frameInterval, VidyoEndpointOnRemoteCameraFirstFrameReceived onFirstFrameReceived)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: remoteCamera}
		{description: The VidyoRemoteCamera for which to receive the video.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested video.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested video.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested video.}
	}
	{parameter:
		{name: onFirstFrameReceived}
		{description: Callback that is triggered when first frame from a remote camera received.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRequestVideoForRemoteCamera(VidyoEndpoint* e, const VidyoRemoteCamera* remoteCamera, LmiUint width, LmiUint height, LmiTime frameInterval, VidyoEndpointOnRemoteCameraFirstFrameReceived onFirstFrameReceived);

/**
 {function:
	{name: VidyoEndpointRequestVideoForRemoteWindowShare}
	{parent: VidyoEndpoint}
	{description: Requests video for remote window share before assign.}
	{prototype: LmiBool VidyoEndpointRequestVideoForRemoteWindowShare(VidyoEndpoint* e, const VidyoRemoteWindowShare* remoteWindowShare, LmiUint width, LmiUint height, LmiTime frameInterval, VidyoEndpointOnRemoteWindowShareFirstFrameReceived onFirstFrameReceived)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: remoteWindowShare}
		{description: The VidyoRemoteWindowShare for which to receive the video.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested video.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested video.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested video.}
	}
	{parameter:
		{name: onFirstFrameReceived}
		{description: Callback that is triggered when first frame from a remote window share received.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRequestVideoForRemoteWindowShare(VidyoEndpoint* e, const VidyoRemoteWindowShare* remoteWindowShare, LmiUint width, LmiUint height, LmiTime frameInterval, VidyoEndpointOnRemoteWindowShareFirstFrameReceived onFirstFrameReceived);

/**
	{function apigroup="simple":
	{name: VidyoEndpointSetOptions}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: LmiBool VidyoEndpointSetOptions(VidyoEndpoint *e, const char *options)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter:
		{name: options}
		{description: The options in JSON format.
			{p: conferenceReferenceNumber - a special identifier that will show up in the Portal CDR record for the conference. Should be set before joining the conference.}
			{p: Example: \{\'conferenceReferenceNumber\' : \'reference number\'\}.}
			{p: microphoneMaxBoostLevel - sets max boost level that will be applied only in the conference for the selected microphone in case the system is set to the higher value. Possible values are devices specific, e.g. 0 - 20 dB. Default is 12 dB.}
			{p: Example: \{\'microphoneMaxBoostLevel\' : 10\}.}
			{p: minMicrophoneVolume - sets min volume that will be applied only in the conference for the selected microphone in case the system is set to the lower value. Possible values are 0 - 100 percentage. Default is 50 per cent.}
			{p: Example: \{\'minMicrophoneVolume\' : 70\}.}
			{p: Possible values for audio device modes are
				audioSharedModeBoth - use both audio devices (Speaker and Microphone) in shared mode.
				audioExclusiveModeBoth - use both audio devices (Speaker and Microphone) in exclusive mode.
				audioExclusiveModeMic - use microphone in exclusive mode and speaker in shared mode.
				By default VidyoClient is working with audioSharedModeBoth. A new mode will be applied only for the next conference or in case another device is selected.}
			{p: Example: \{\'audioSharedModeBoth\' : true\}.}
			{p: Example: \{\'audioExclusiveModeBoth\' : true\}.}
			{p: Example: \{\'audioExclusiveModeMic\' : true\}.}
			{p: preferredAudioCodec - sets VidyoClient to use preferred audio codec. Possible value are OPUS RED, OPUS and SPEEX RED. Default is OPUS RED.}
			{p: Example: \{\'preferredAudioCodec\' : \'SPEEX RED\'\}.}
            {p: AudioBitrateMultiplier - A multiplier to be used when calculating the audio bitrate. Possible values are 0,1, and 2. Default is 2.}
            {p: Example: \{\'AudioBitrateMultiplier\':1\}.}
            {p: AudioPacketInterval - It decies audio frame duration to encoder. It sets time interval between two packet of audio in miliseconds Possible values are 20, and 40 ms. Default is 40 ms.}
            {p: Example: \{\'AudioPacketInterval\':40\}.}
            {p: AudioPacketLossPercentage - it values is used to configure encoder. If given higher value then encoder is configured to resistant to loss on the expense of quality. Possible values are [0, 10, 20, 30] in percentage. Default is 10%.}
            {p: Example: \{\'AudioPacketLossPercentage\':20\}.}
            {p: statRate - sets VidyoRoom statistics gather interval in seconds. Possible values are 2, 5, 10, 20 and 30. Default is 10.}
            {p: Example: \{\'statRate\':10\}.}
			{p: enableStaticShareSvc - Specifies whether SVC will be enabled or disabled for static sharing. Possible values are true or false. The default is false. Note: SVC is enabled by default for interactive video and high frame rate sharing.}
			{p: Example: \{\'enableStaticShareSvc\' : true\}.}
		}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetOptions(VidyoEndpoint *e, const char *options);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetOptions}
	{parent: VidyoEndpoint}
	{description: }
	{prototype: const LmiString *VidyoEndpointGetOptions(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: JSON encoded string on success, NULL on failure.}
}
*/
const LmiString *VidyoEndpointGetOptions(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetConferenceNumber}
	{parent: VidyoEndpoint}
	{description: Sets conference number - a special identifier that will show up in the Portal CDR record for the conference. Should be set before joining the conference.}
	{prototype: void VidyoEndpointSetConferenceNumber(VidyoEndpoint *e, const LmiString *conferenceNumber)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: conferenceNumber} {description: Conference number to be set.}}
}
*/
void VidyoEndpointSetConferenceNumber(VidyoEndpoint *e, const LmiString *conferenceNumber);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetConferenceNumber}
	{parent: VidyoEndpoint}
	{description: Gets conference number - a special identifier that is showed up in the Portal CDR record for the conference.}
	{prototype: const LmiString* VidyoEndpointGetConferenceNumber(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: conference number on success, NULL on failure.}
}
*/
const LmiString* VidyoEndpointGetConferenceNumber(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetMinMicrophoneVolume}
	{parent: VidyoEndpoint}
	{description: Sets min volume that will be applied only in the conference for the selected microphone in case the system is set to the lower value.}
	{prototype: void VidyoEndpointSetMinMicrophoneVolume(VidyoEndpoint *e, LmiUint volumePercent)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: volumePercent} {description: Volume percent for which possible values are 0 - 100 percentage. Default is 50 percent.}}
}
*/
void VidyoEndpointSetMinMicrophoneVolume(VidyoEndpoint *e, LmiUint volumePercent);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetMinMicrophoneVolume}
	{parent: VidyoEndpoint}
	{description: Gets minimum microphone value that is applied only in the conference for the selected microphone in case the system is set to the lower value.}
	{prototype: LmiUint VidyoEndpointGetMinMicrophoneVolume(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: Minimum microphone value.}
}
*/
LmiUint VidyoEndpointGetMinMicrophoneVolume(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetMicrophoneMaxBoostLevel}
	{parent: VidyoEndpoint}
	{description: Sets max boost level that will be applied only in the conference for the selected microphone in case the system is set to the higher value.}
	{prototype: void VidyoEndpointSetMicrophoneMaxBoostLevel(VidyoEndpoint *e, LmiFloat64 boostLevel)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: boostLevel} {description: Boost level for which possible values are devices specific, e.g. 0 - 20 dB. Default is 12 dB.}}
}
*/
void VidyoEndpointSetMicrophoneMaxBoostLevel(VidyoEndpoint *e, LmiFloat64 boostLevel);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetMicrophoneMaxBoostLevel}
	{parent: VidyoEndpoint}
	{description: Gets maximum microphone boost level that is applied only in the conference for the selected microphone in case the system is set to the higher value.}
	{prototype: LmiFloat64 VidyoEndpointGetMicrophoneMaxBoostLevel(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: Maximum microphone boost level.}
}
*/
LmiFloat64 VidyoEndpointGetMicrophoneMaxBoostLevel(VidyoEndpoint* e);

/**
{type visibility="public":
	{name: VidyoEndpointAudioMode}
	{parent: VidyoEndpoint}
	{description: Indicates the audio mode.}
	{value: {name: VIDYO_ENDPOINTAUDIOMODE_SharedModeBoth}
		{description: Both audio devices (Speaker and Microphone) used in shared mode.}}
	{value: {name: VIDYO_ENDPOINTAUDIOMODE_ExclusiveModeBoth}
		{description: Both audio devices (Speaker and Microphone) used in exclusive mode.}}
	{value: {name: VIDYO_ENDPOINTAUDIOMODE_ExclusiveModeMicrophone}
		{description: Microphone is used in exclusive mode and speaker in shared mode..}}
	{value: {name: VIDYO_ENDPOINTAUDIOMODE_Unknown}
		{description: Unknown audio mode.}}
}
*/
typedef enum
{
	VIDYO_ENDPOINTAUDIOMODE_SharedModeBoth,
	VIDYO_ENDPOINTAUDIOMODE_ExclusiveModeBoth,
	VIDYO_ENDPOINTAUDIOMODE_ExclusiveModeMicrophone,
	VIDYO_ENDPOINTAUDIOMODE_Unknown
} VidyoEndpointAudioMode;


/**
{function apigroup="simple":
	{name: VidyoEndpointSetAudioMode}
	{parent: VidyoEndpoint}
	{description: Enables or disables usage of audio devices (speaker and microphone) in different audio modes. By default SharedModeBoth is enabled.}
	{prototype: LmiBool VidyoEndpointSetAudioMode(VidyoEndpoint* e, VidyoEndpointAudioMode audioMode)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: audioMode} {description: Indicates which audio mode is enabled/disabled for both audio devices. For possible values of the audio mode, refer VidyoEndpointAudioMode.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetAudioMode(VidyoEndpoint* e, VidyoEndpointAudioMode audioMode);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetAudioMode}
	{parent: VidyoEndpoint}
	{description: Gets the audio mode for both the audio devices (speaker and microphone).}
	{prototype: VidyoEndpointAudioMode VidyoEndpointGetAudioMode(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: Audio Mode, refer VidyoEndpointAudioMode.}
}
*/
VidyoEndpointAudioMode VidyoEndpointGetAudioMode(VidyoEndpoint* e);

/**
 {type visibility="public":
	{name: VidyoEndpointPreferredAudioCodec}
	{parent: VidyoEndpoint}
	{description: Preferred audio codec to be set.}
	{value: {name: VIDYO_ENDPOINTPREFERREDAUDIOCODEC_OpusRed}
		{description: Preferred audio codec is OPUS RED.}}
	{value: {name: VIDYO_ENDPOINTPREFERREDAUDIOCODEC_Opus}
		{description: Preferred audio codec is OPUS.}}
	{value: {name: VIDYO_ENDPOINTPREFERREDAUDIOCODEC_SpeexRed}
		{description: Preferred audio codec is SPEEX RED.}}
	{value: {name: VIDYO_ENDPOINTPREFERREDAUDIOCODEC_Unknown}
		{description: Preferred audio codec is Unknown.}}
}
*/
typedef enum
{
	VIDYO_ENDPOINTPREFERREDAUDIOCODEC_OpusRed,
	VIDYO_ENDPOINTPREFERREDAUDIOCODEC_Opus,
	VIDYO_ENDPOINTPREFERREDAUDIOCODEC_SpeexRed,
	VIDYO_ENDPOINTPREFERREDAUDIOCODEC_Unknown
} VidyoEndpointPreferredAudioCodec;

/*
{function apigroup="simple":
	{name: VidyoEndpointSetPreferredAudioCodec}
	{parent: VidyoEndpoint}
	{description: Sets VidyoClient to use preferred audio codec.}
	{prototype: LmiBool VidyoEndpointSetPreferredAudioCodec(VidyoEndpointPreferredAudioCodec preferredAudioCodec)}
	{parameter:
		{parameter: {name: e}
		{description: The endpoint object.}}
		{name: preferredAudioCodec}
		{description: Preferred audio codec to be set. For possible values of the audio codec, refer VidyoEndpointPreferredAudioCodec.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoEndpointSetPreferredAudioCodec(VidyoEndpoint* e, VidyoEndpointPreferredAudioCodec preferredAudioCodec);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetPreferredAudioCodec}
	{parent: VidyoEndpoint}
	{description: Gets preferred audio codec type.}
	{prototype: VidyoEndpointPreferredAudioCodec VidyoEndpointGetPreferredAudioCodec(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: Preferred audio codec, refer VidyoEndpointPreferredAudioCodec.}
}
*/
VidyoEndpointPreferredAudioCodec VidyoEndpointGetPreferredAudioCodec(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetAudioPacketInterval}
	{parent: VidyoEndpoint}
	{description: Sets time interval between two packet of audio in miliseconds. Default is 40 ms.}
	{prototype: LmiBool VidyoEndpointSetAudioPacketInterval(VidyoEndpoint* e, LmiUint packetInterval)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: packetInterval} {description: Packet interval for which possible values are 20, and 40 ms.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetAudioPacketInterval(VidyoEndpoint* e, LmiUint packetInterval);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetAudioPacketInterval}
	{parent: VidyoEndpoint}
	{description: Gets time interval between two packet of audio in miliseconds.}
	{prototype: LmiUint VidyoEndpointGetAudioPacketInterval(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: Audio packet interval in miliseconds.}
}
*/
LmiUint VidyoEndpointGetAudioPacketInterval(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetAudioPacketLossPercentage}
	{parent: VidyoEndpoint}
	{description: Sets value that is used to configure encoder. If given higher value then encoder is configured to resistant to loss on the expense of quality. Default is 10%.}
	{prototype: LmiBool VidyoEndpointSetAudioPacketLossPercentage(VidyoEndpoint* e, LmiUint lossPercentage)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: lossPercentage} {description: Loss percantage for which possible values are [0, 10, 20, 30] in percentage.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetAudioPacketLossPercentage(VidyoEndpoint* e, LmiUint lossPercentage);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetAudioPacketLossPercentage}
	{parent: VidyoEndpoint}
	{description: Gets audio loss percantage - value that is used to configure encoder.}
	{prototype: LmiUint VidyoEndpointGetAudioPacketLossPercentage(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: Audio loss percentage.}
}
*/
LmiUint VidyoEndpointGetAudioPacketLossPercentage(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetAudioBitrateMultiplier}
	{parent: VidyoEndpoint}
	{description: Sets multiplier value to be used when calculating the audio bitrate. Default is 2.}
	{prototype: LmiBool VidyoEndpointSetAudioBitrateMultiplier(VidyoEndpoint* e, LmiUint bitrateMultiplier)}
		{parameter: {name: e} {description: The endpoint object.}}
		{parameter:{name: bitrateMultiplier} {description: Bitrate multiplier for which possible values are 0, 1 and 2.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetAudioBitrateMultiplier(VidyoEndpoint* e, LmiUint bitrateMultiplier);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetAudioBitrateMultiplier}
	{parent: VidyoEndpoint}
	{description: Gets audio bitrate multiplier which is used when calculating the audio bitrate.}
	{prototype: LmiUint VidyoEndpointGetAudioBitrateMultiplier(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: Audio bitrate multiplier.}
}
*/
LmiUint VidyoEndpointGetAudioBitrateMultiplier(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetStatisticRate}
	{parent: VidyoEndpoint}
	{description: Sets conference room statistics gather interval in seconds. Default is 10.}
	{prototype: LmiBool VidyoEndpointSetStatisticRate(VidyoEndpoint* e, LmiInt64 statRate)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: statRate} {description: Statistics gather interval for which possible values are 2, 5, 10, 20 and 30.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSetStatisticRate(VidyoEndpoint* e, LmiInt64 statRate);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetStatisticRate}
	{parent: VidyoEndpoint}
	{description: Gets conference room statistics gather interval in seconds.}
	{prototype: LmiInt64 VidyoEndpointGetStatisticRate(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: Statistics gather interval.}
}
*/
LmiInt64 VidyoEndpointGetStatisticRate(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetStaticShareSvcState}
	{parent: VidyoEndpoint}
	{description: Enables or disables SVC. The default is false. Note: SVC is enabled by default for interactive video and high frame rate sharing.}
	{prototype: LmiBool VidyoEndpointSetStaticShareSvcState(VidyoEndpoint* e, LmiBool enable)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: enable} {description: Indicates whether enable SVC or not.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoEndpointSetStaticShareSvcState(VidyoEndpoint* e, LmiBool enable);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetStaticShareSvcState}
	{parent: VidyoEndpoint}
	{description: Gets whether SVC is enabled.}
	{prototype: LmiBool VidyoEndpointGetStaticShareSvcState(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: LMI_TRUE if SVC enabled, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoEndpointGetStaticShareSvcState(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetAutoReconnect}
	{parent: VidyoEndpoint}
	{description: Enables or disables auto-reconnect.}
	{prototype: LmiBool VidyoEndpointSetAutoReconnect(VidyoEndpoint *e, LmiBool enable)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: enable} {description: Indicates whether auto-reconnect is enabled or not.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoEndpointSetAutoReconnect(VidyoEndpoint *e, LmiBool enable);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetAutoReconnect}
	{parent: VidyoEndpoint}
	{description: Gets state of auto-reconnect. It's either enabled or disabled.}
	{prototype: LmiBool VidyoEndpointGetAutoReconnect(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: LMI_TRUE if auto-reconnect enabled, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoEndpointGetAutoReconnect(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetAutoReconnectMaxAttempts}
	{parent: VidyoEndpoint}
	{description: Sets maximum attempts to be done for reconnecting. By default, it is set to 3.}
	{prototype: LmiBool VidyoEndpointSetAutoReconnectMaxAttempts(VidyoEndpoint *e, LmiUint maxAttempts)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: maxAttempts} {description: Number of maximum attempts to be done for reconnecting. Min value is 1 and the max is 4.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoEndpointSetAutoReconnectMaxAttempts(VidyoEndpoint *e, LmiUint maxAttempts);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetAutoReconnectMaxAttempts}
	{parent: VidyoEndpoint}
	{description: Gets maximum attempts to be done for reconnecting.}
	{prototype: LmiUint VidyoEndpointGetAutoReconnectMaxAttempts(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: amount of maximum attempts to be done for reconnecting}
}
*/
LmiUint VidyoEndpointGetAutoReconnectMaxAttempts(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetAutoReconnectAttemptBackOff}
	{parent: VidyoEndpoint}
	{description: Sets back off for next reconnect attempt. By default, it is set to 5 seconds.}
	{prototype: LmiBool VidyoEndpointSetAutoReconnectAttemptBackOff(VidyoEndpoint *e, LmiUint backOff)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: backOff} {description: Time in seconds before next reconnect attempt. Min value is 2 and the max is 20 seconds.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoEndpointSetAutoReconnectAttemptBackOff(VidyoEndpoint *e, LmiUint backOff);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetAutoReconnectAttemptBackOff}
	{parent: VidyoEndpoint}
	{description: Gets back off for next reconnect attempt.}
	{prototype: LmiUint VidyoEndpointGetAutoReconnectAttemptBackOff(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: the back off for next reconnect attempt.}
}
*/
LmiUint VidyoEndpointGetAutoReconnectAttemptBackOff(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetDisableVideoOnLowBandwidth}
	{parent: VidyoEndpoint}
	{description: When enabled, all video feeds (video and content share) will be shut off to the client when the user experiences prolonged low network bandwidth.}
	{prototype: LmiBool VidyoEndpointSetDisableVideoOnLowBandwidth(VidyoEndpoint *e, LmiBool enable)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: enable} {description: Indicates whether disable video on low bandwidth is enabled or not.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoEndpointSetDisableVideoOnLowBandwidth(VidyoEndpoint *e, LmiBool enable);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetDisableVideoOnLowBandwidth}
	{parent: VidyoEndpoint}
	{description: Gets state of video on low bandwidth. It's either enabled or disabled.}
	{prototype: LmiBool VidyoEndpointGetDisableVideoOnLowBandwidth(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: LMI_TRUE if enabled video on low bandwidth, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoEndpointGetDisableVideoOnLowBandwidth(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetDisableVideoOnLowBandwidthResponseTime}
	{parent: VidyoEndpoint}
	{description: Sets time it takes to shut off video due to low bandwidth. By default, it is set to 30 seconds.}
	{prototype: LmiBool VidyoEndpointSetDisableVideoOnLowBandwidthResponseTime(VidyoEndpoint *e, LmiUint responseTime)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: responseTime} {description: Time it takes to shut off video due to low bandwidth.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoEndpointSetDisableVideoOnLowBandwidthResponseTime(VidyoEndpoint *e, LmiUint responseTime);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetDisableVideoOnLowBandwidthResponseTime}
	{parent: VidyoEndpoint}
	{description: Gets time it takes to shut off video due to low bandwidth.}
	{prototype: LmiUint VidyoEndpointGetDisableVideoOnLowBandwidthResponseTime(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: the time it takes to shut off video due to low bandwidth.}
}
*/
LmiUint VidyoEndpointGetDisableVideoOnLowBandwidthResponseTime(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetDisableVideoOnLowBandwidthSampleTime}
	{parent: VidyoEndpoint}
	{description: Sets how often to check for low bandwidth. By default, it is set to 5 seconds.}
	{prototype: LmiBool VidyoEndpointSetDisableVideoOnLowBandwidthSampleTime(VidyoEndpoint *e, LmiUint sampleTime)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: sampleTime} {description: How often to check for low bandwidth. Must divide evenly into responseTime.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoEndpointSetDisableVideoOnLowBandwidthSampleTime(VidyoEndpoint *e, LmiUint sampleTime);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetDisableVideoOnLowBandwidthSampleTime}
	{parent: VidyoEndpoint}
	{description: Gets how often checking for low bandwidth happens.}
	{prototype: LmiUint VidyoEndpointGetDisableVideoOnLowBandwidthSampleTime(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: how often check for low bandwidth happens.}
}
*/
LmiUint VidyoEndpointGetDisableVideoOnLowBandwidthSampleTime(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetDisableVideoOnLowBandwidthRecoveryTime}
	{parent: VidyoEndpoint}
	{description: Sets time it takes to recover when video is shut off. By default, it is set to 0 seconds which means never.}
	{prototype: LmiBool VidyoEndpointSetDisableVideoOnLowBandwidthRecoveryTime(VidyoEndpoint *e, LmiUint recoveryTime)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: recoveryTime} {description: Time it takes to recover when video is shut off. Recovery can only happen when there are no available video sources.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoEndpointSetDisableVideoOnLowBandwidthRecoveryTime(VidyoEndpoint *e, LmiUint recoveryTime);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetDisableVideoOnLowBandwidthRecoveryTime}
	{parent: VidyoEndpoint}
	{description: Gets time it takes to recover when video is shut off.}
	{prototype: LmiUint VidyoEndpointGetDisableVideoOnLowBandwidthRecoveryTime(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: time it takes to recover when video is shut off}
}
*/
LmiUint VidyoEndpointGetDisableVideoOnLowBandwidthRecoveryTime(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetDisableVideoOnLowBandwidthThreshold}
	{parent: VidyoEndpoint}
	{description: Sets low bandwidth threshold. By default, it is set to 100kpbs.}
	{prototype: LmiBool VidyoEndpointSetDisableVideoOnLowBandwidthThreshold(VidyoEndpoint *e, LmiUint kbps)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: kbps} {description: Threshold for low bandwidth in kilobits/sec.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoEndpointSetDisableVideoOnLowBandwidthThreshold(VidyoEndpoint *e, LmiUint kbps);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetDisableVideoOnLowBandwidthThreshold}
	{parent: VidyoEndpoint}
	{description: Gets low bandwidth threshold.}
	{prototype: LmiUint VidyoEndpointGetDisableVideoOnLowBandwidthThreshold(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: the low bandwidth threshold}
}
*/
LmiUint VidyoEndpointGetDisableVideoOnLowBandwidthThreshold(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointSetDisableVideoOnLowBandwidthAudioStreams}
	{parent: VidyoEndpoint}
	{description: Sets number of audio streams requested while video is shut off. By default, it is set to 3.}
	{prototype: LmiBool VidyoEndpointSetDisableVideoOnLowBandwidthAudioStreams(VidyoEndpoint *e, LmiUint audioStreams)}
	{parameter: {name: e} {description: The endpoint object.}}
	{parameter: {name: audioStreams} {description:  Number of audio streams requested while video is shut off. Min value is 1 and max is 3.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoEndpointSetDisableVideoOnLowBandwidthAudioStreams(VidyoEndpoint *e, LmiUint audioStreams);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetDisableVideoOnLowBandwidthAudioStreams}
	{parent: VidyoEndpoint}
	{description: Gets number of audio streams requested while video is shut off.}
	{prototype: LmiUint VidyoEndpointGetDisableVideoOnLowBandwidthAudioStreams(VidyoEndpoint *e)}
	{parameter: {name: e} {description: The endpoint object.}}
	{return: the number of audio streams requested while video is shut off.}
}
*/
LmiUint VidyoEndpointGetDisableVideoOnLowBandwidthAudioStreams(VidyoEndpoint *e);

/**
{function apigroup="simple":
	{name: VidyoEndpointRegisterErrorEventListener}
	{parent: VidyoEndpoint}
	{description: Registers the advanced error event listeners.}
	{prototype: LmiBool VidyoEndpointRegisterErrorEventListener(VidyoEndpoint* e, VidyoEndpointOnError onErrorCallback)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onErrorCallback}
		{description: Callback that is triggered when an error occurs.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRegisterErrorEventListener(VidyoEndpoint* e, VidyoEndpointOnError onErrorCallback);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterErrorEventListener}
	{parent: VidyoEndpoint}
	{description: Unregisters advanced error event notifications.}
	{prototype: LmiBool VidyoEndpointUnregisterErrorEventListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointUnregisterErrorEventListener(VidyoEndpoint* e);

/**
{function apigroup="simple":
	{name: VidyoEndpointAddAudioDeviceToWhitelist}
	{parent: VidyoEndpoint}
	{description: Adds audio device to whitelist.}
	{prototype: LmiBool VidyoEndpointAddAudioDeviceToWhitelist(VidyoEndpoint* e, const LmiString* deviceName)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: deviceName}
		{description: The name of audio device that will be whitelisted.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointAddAudioDeviceToWhitelist(VidyoEndpoint* e, const LmiString* deviceName);

/**
{function apigroup="simple":
	{name: VidyoEndpointRemoveAudioDeviceFromWhitelist}
	{parent: VidyoEndpoint}
	{description: Removes audio device from whitelist.}
	{prototype: LmiBool VidyoEndpointRemoveAudioDeviceFromWhitelist(VidyoEndpoint* e, const LmiString* deviceName)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: deviceName}
		{description: The name of audio device that will be removed from whitelist.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointRemoveAudioDeviceFromWhitelist(VidyoEndpoint* e, const LmiString* deviceName);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetWhitelistedAudioDevices}
	{parent: VidyoEndpoint}
	{description: Gets the names of the audio devices that are whitelisted.}
	{prototype: LmiBool VidyoEndpointGetWhitelistedAudioDevices(VidyoEndpoint* e, VidyoEndpointOnGetWhitelistedAudioDevices onGetCallback)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onGetCallback}
		{description: Callback that will report an array with the names of audio devices that are whitelisted.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointGetWhitelistedAudioDevices(VidyoEndpoint* e, VidyoEndpointOnGetWhitelistedAudioDevices onGetCallback);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnGetActiveNetworkInterface}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time when result of VidyoEndpointGetActiveNetworkInterface is received.}
	{prototype: void (*VidyoEndpointOnGetActiveNetworkInterface)(VidyoEndpoint* e, const VidyoNetworkInterface *signalingInterface, const VidyoNetworkInterface *mediaInterface)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint that triggered the callback.}}
	{parameter:
		{name: signalingInterface}
		{description: The zeroed-out VidyoNetworkInterface object for signaling interface.}}
	{parameter:
		{name: mediaInterface}
		{description: The zeroed-out VidyoNetworkInterface object for media interface.}}
}
*/
typedef void (*VidyoEndpointOnGetActiveNetworkInterface)(VidyoEndpoint* e, const VidyoNetworkInterface *signalingInterface, const VidyoNetworkInterface *mediaInterface);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetActiveNetworkInterface}
	{parent: VidyoEndpoint}
	{description: This is an asynchronous interface. Get the NetworkInterface for media and signaling.}
	{prototype: LmiBool VidyoEndpointGetActiveNetworkInterface(VidyoEndpoint* e, VidyoEndpointOnGetActiveNetworkInterface onGetActiveNetworkInterfaceCallback)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}}
	{parameter:
		{name: onGetActiveNetworkInterfaceCallback}
		{description: Callback called upon completion.}}
	{return: LMI_TRUE on success, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoEndpointGetActiveNetworkInterface(VidyoEndpoint* e, VidyoEndpointOnGetActiveNetworkInterface onGetActiveNetworkInterfaceCallback);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnGetAutoReconnectSetting}
	{parent: VidyoEndpoint}
	{description: Callback that is triggered each time when result of VidyoEndpointGetAutoReconnectSetting is received.}
	{prototype: void (*VidyoEndpointOnGetAutoReconnectSetting)(VidyoEndpoint *e, LmiBool enableAutoReconnect, LmiUint reconnectBackoff, LmiUint maxReconnectAttempts)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint that triggered the callback.}}
	{parameter:
		{name: enableAutoReconnect}
		{description: Represents the auto reconnect is enabled or not.}}
	{parameter:
		{name: reconnectBackoff}
		{description: Represents the time in seconds before next reconnect attempt.}}
	{parameter:
		{name: maxReconnectAttempts}
		{description: Represents the maximum reconnect attempts.}}
}
*/
typedef void (*VidyoEndpointOnGetAutoReconnectSetting)(VidyoEndpoint *e, LmiBool enableAutoReconnect, LmiUint reconnectBackoff, LmiUint maxReconnectAttempts);

/**
{function apigroup="simple":
	{name: VidyoEndpointGetAutoReconnectSetting}
	{parent: VidyoEndpoint}
	{description: This is an asynchronous interface. Gets the auto-reconnect setting.}
	{prototype: LmiBool VidyoEndpointGetAutoReconnectSetting(VidyoEndpoint *e, VidyoEndpointOnGetAutoReconnectSetting onGetAutoReconnectSettingCallback)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}}
	{parameter:
		{name: onGetAutoReconnectSettingCallback}
		{description: Callback called upon completion.}}
	{return: LMI_TRUE on success, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoEndpointGetAutoReconnectSetting(VidyoEndpoint *e, VidyoEndpointOnGetAutoReconnectSetting onGetAutoReconnectSettingCallback);

/**
{type:
	{name: VidyoEndpointCameraEffectType}
	{parent: VidyoEndpoint}
	{description: Provides the supported camera background effects and used to enable or disable effect.}
	{value:
		{name: VIDYO_ENDPOINTCAMERAEFFECTTYPE_None}
		{description: Disables active effects.}}
	{value:
		{name: VIDYO_ENDPOINTCAMERAEFFECTTYPE_Blur}
		{description: Enables background blur effect.}}
	{value:
		{name: VIDYO_ENDPOINTCAMERAEFFECTTYPE_VirtualBackground}
		{description: Enables virtual background effect.}}
}
*/
typedef enum
{
	VIDYO_ENDPOINTCAMERAEFFECTTYPE_None,
	VIDYO_ENDPOINTCAMERAEFFECTTYPE_Blur,
	VIDYO_ENDPOINTCAMERAEFFECTTYPE_VirtualBackground
} VidyoEndpointCameraEffectType;

/**
{type:
	{name:  VidyoEndpointCameraEffectInfo}
	{parent: Endpoint}
	{description: Represent the configuration of camera background effect.}
	{member:
		{name: effectType}
		{type: VidyoEndpointCameraEffectType}
		{description: Type of the camera background effect. In case VIDYO_CONNECTORCAMERAEFFECTTYPE_None is set no need to set other fields.}
	}
	{member:
		{name: token}
		{type: LmiString}
		{description: The token for enabling the camera background effect. The blank string is NOT allowed. In case the token is not valid or expired enabling of camera background effect will fail.}
	}
	{member:
		{name: pathToResources}
		{type: LmiString}
		{description: The path to resources like a model and its configuration. A directory with supported resources is packaged with VidyoClient library.}
	}
	{member:
		{name: pathToEffect}
		{type: LmiString}
		{description: The path to effect. A directory with supported effect is packaged with VidyoClient library.}
	}
	{member:
		{name: blurIntensity}
		{type: LmiUint}
		{description: Blur intensity. Valid valies is [3-8]. Default value is 5. Used only if effectType VIDYO_CONNECTORCAMERAEFFECTTYPE_Blur}
	}
	{member:
		{name: virtualBackgroundPicture}
		{type: LmiString}
		{description: The path to the picture that will be used for the virtual background. Used only if effectType VIDYO_CONNECTORCAMERAEFFECTTYPE_VirtualBackground.}
	}
}
*/
typedef struct
{
	VidyoEndpointCameraEffectType effectType;
	LmiString token;
	LmiString pathToResources;
	LmiString pathToEffect;
	LmiUint blurIntensity;
	LmiString virtualBackgroundPicture;
} VidyoEndpointCameraEffectInfo;

VidyoEndpointCameraEffectInfo* VidyoEndpointCameraEffectInfoConstructDefault(VidyoEndpointCameraEffectInfo* effectInfo, LmiAllocator* alloc);
VidyoEndpointCameraEffectInfo* VidyoEndpointCameraEffectInfoConstructCopy(VidyoEndpointCameraEffectInfo* dst, const VidyoEndpointCameraEffectInfo* src);
void VidyoEndpointCameraEffectInfoDestruct(VidyoEndpointCameraEffectInfo* effectInfo);

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnGetCameraBackgroundEffectInfo}
	{parent: VidyoEndpoint}
	{description: Callback that reports the current configuration of camera background effect to the application.}
	{prototype: void (*VidyoEndpointOnGetCameraBackgroundEffectInfo)(VidyoEndpoint* e,  const VidyoEndpointCameraEffectInfo* effectInfo)}
	{parameter:
		{name: e}
		{description: Pointer to the VidyoEndpoint that triggered the callback. }
	}
	{parameter:
		{name: effectInfo}
		{description: Represents the current configuration of camera background effect.}
	}
}
*/
typedef void (*VidyoEndpointOnGetCameraBackgroundEffectInfo)(VidyoEndpoint* e, const VidyoEndpointCameraEffectInfo* effectInfo);

/**
{function:
	{name: VidyoEndpointGetCameraBackgroundEffect}
	{parent: VidyoEndpoint}
	{description: Gets the current configuration of camera background effect. This functionality is supported only for macOS 10.14 and above, iOS 10.0 and above and Windows VC++ 15 and above.}
	{prototype: LmiBool VidyoEndpointGetCameraBackgroundEffect(VidyoEndpoint* e, VidyoEndpointOnGetCameraBackgroundEffectInfo onGetEffectInfo)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onGetEffectInfo}
		{description: Callback that reports the current configuration of camera background effect to the application. In case some error occurs NULL will be reported.}
	}
	{return: Returns LMI_TRUE on success or LMI_FALSE otherwise.}
}
*/
LmiBool VidyoEndpointGetCameraBackgroundEffect(VidyoEndpoint* e, VidyoEndpointOnGetCameraBackgroundEffectInfo onGetEffectInfo);

/**
{function:
	{name: VidyoEndpointSetCameraBackgroundEffect}
	{parent: VidyoEndpoint}
	{description: Enables the specified by VidyoEndpointCameraEffectType camera background effect or disables active effect if VIDYO_ENDPOINTCAMERAEFFECTTYPE_None passed. The application must provide token to enable this feature as part of VidyoConnectorCameraEffectInfo structure. This functionality is supported only for macOS 10.14 and above, iOS 10.0 and above and Windows VC++ 15 and above.}
	{prototype: LmiBool VidyoEndpointSetCameraBackgroundEffect(VidyoEndpoint* e, const VidyoEndpointCameraEffectInfo* effectInfo)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: effectInfo}
		{description: Represents the configuration of camera background effect.}
	}
	{return: Returns LMI_TRUE on success or LMI_FALSE otherwise.}
}
*/
LmiBool VidyoEndpointSetCameraBackgroundEffect(VidyoEndpoint* e, const VidyoEndpointCameraEffectInfo* effectInfo);

/**
{function:
	{name: VidyoEndpointSetBlurIntensity}
	{parent: VidyoEndpoint}
	{description: Sets the intensity of the background blur effect. It is allowed to change blur intensity while the effect is enabled. This functionality is supported only for macOS 10.14 and above, iOS 10.0 and above and Windows VC++ 15 and above.}
	{prototype: LmiBool VidyoEndpointSetBlurIntensity(VidyoEndpoint* e, LmiUint intensity)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: intensity}
		{description: The intensity of the background blur effect. Valid valies is [3-8]. Default value is 5.}
	}
	{return: Returns LMI_TRUE on success or LMI_FALSE otherwise.}
}
*/
LmiBool VidyoEndpointSetBlurIntensity(VidyoEndpoint* e, LmiUint intensity);

/**
{function:
	{name: VidyoEndpointSetVirtualBackgroundPicture}
	{parent: VidyoEndpoint}
	{description: Sets path and file name to the picture that will be used for the virtual background effect. It is allowed to change picture while the effect is enabled. This functionality is supported only for macOS 10.14 and above, iOS 10.0 and above and Windows VC++ 15 and above.}
	{prototype: LmiBool VidyoEndpointSetVirtualBackgroundPicture(VidyoEndpoint* e, const LmiString* pathToPicture)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: pathToPicture}
		{description: Path and file name to the picture. Currently, only PNG and JPG is supported..}
	}
	{return: Returns LMI_TRUE on success or LMI_FALSE otherwise.}
}
*/
LmiBool VidyoEndpointSetVirtualBackgroundPicture(VidyoEndpoint* e, const LmiString* pathToPicture);

/**
{type:
	{name: VidyoEndpointCameraEffectError}
	{parent: VidyoEndpoint}
	{description: Provides the error codes for camera background effect.}
	{value:
		{name: VIDYO_ENDPOINTCAMERAEFFECTERROR_None}
		{description: No errors.}}
	{value:
		{name: VIDYO_ENDPOINTCAMERAEFFECTERROR_InvalidToken}
		{description: The token is empty or expired.}}
	{value:
		{name: VIDYO_ENDPOINTCAMERAEFFECTERROR_InvalidResources}
		{description: The path to resources is empty or does not contain some files.}}
	{value:
		{name: VIDYO_ENDPOINTCAMERAEFFECTERROR_InvalidEffect}
		{description: The path to effect is empty  or does not contain some files.}}
	{value:
		{name: VIDYO_ENDPOINTCAMERAEFFECTERROR_InvalidBlurIntensity}
		{description: The blur intensity is out of allowed range.}}
	{value:
		{name: VIDYO_ENDPOINTCAMERAEFFECTERROR_InvalidBackgroundPicture}
		{description: The path to background picture is empty or file doesn't exist or has unsupported format.}}
	{value:
		{name: VIDYO_ENDPOINTCAMERAEFFECTERROR_LoadEffectFailed}
		{description: Failed to load effect.}}
	{value:
		{name: VIDYO_ENDPOINTCAMERAEFFECTERROR_UnsupportedOpenGLVersion}
		{description: Effect was not started due to unsupported OpenGL version.}}
	{value:
		{name: VIDYO_ENDPOINTCAMERAEFFECTERROR_UnsupportedLibraryVersion}
		{description: Effect was not started due to unsupported Banuba library version.}}
	{value:
		{name: VIDYO_ENDPOINTCAMERAEFFECTERROR_NotAllowedForCurrectEffect}
		{description: Opperation is not allowed with the current effect.}}
	{value:
		{name: VIDYO_ENDPOINTCAMERAEFFECTERROR_MiscError}
		{description: Miscellaneous error.}}
}
*/

typedef enum
{
	VIDYO_ENDPOINTCAMERAEFFECTERROR_None,
	VIDYO_ENDPOINTCAMERAEFFECTERROR_InvalidToken,
	VIDYO_ENDPOINTCAMERAEFFECTERROR_InvalidResources,
	VIDYO_ENDPOINTCAMERAEFFECTERROR_InvalidEffect,
	VIDYO_ENDPOINTCAMERAEFFECTERROR_InvalidBlurIntensity,
	VIDYO_ENDPOINTCAMERAEFFECTERROR_InvalidBackgroundPicture,
	VIDYO_ENDPOINTCAMERAEFFECTERROR_LoadEffectFailed,
	VIDYO_ENDPOINTCAMERAEFFECTERROR_UnsupportedOpenGLVersion,
	VIDYO_ENDPOINTCAMERAEFFECTERROR_UnsupportedLibraryVersion,
	VIDYO_ENDPOINTCAMERAEFFECTERROR_NotAllowedForCurrectEffect,
	VIDYO_ENDPOINTCAMERAEFFECTERROR_MiscError
} VidyoEndpointCameraEffectError;

/**
{callback apigroup="simple":
	{name: VidyoEndpointOnCameraEffectError}
	{parent: VidyoEndpoint}
	{description: Callback that reports camera effect errors.}
	{prototype: void (*VidyoEndpointOnCameraEffectError)(VidyoEndpoint* e,  VidyoEndpointCameraEffectError error)}
	{parameter:
		{name: e}
		{description: Pointer to the VidyoEndpoint that triggered the callback.}
	}
	{parameter:
		{name: error}
		{description: The error occurred during applying or changing the camera effect.}
	}
}
*/
typedef void (*VidyoEndpointOnCameraEffectError)(VidyoEndpoint* e, VidyoEndpointCameraEffectError error);

/**
{function:
	{name: VidyoEndpointRegisterCameraEffectErrorListener}
	{parent: VidyoEndpoint}
	{description: Registers callback to notify the user in case of any errors occur during enabling camera effect.}
	{prototype: LmiBool VidyoEndpointRegisterCameraEffectErrorListener(VidyoEndpoint* e, VidyoEndpointOnCameraEffectError onError)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: onError}
		{description: The callback that will reports camera effect errors.}
	}
	{return: Returns LMI_TRUE on success or LMI_FALSE otherwise.}
}
*/
LmiBool VidyoEndpointRegisterCameraEffectErrorListener(VidyoEndpoint* e, VidyoEndpointOnCameraEffectError onError);

/**
{function:
	{name: VidyoEndpointUnregisterCameraEffectErrorListener}
	{parent: VidyoEndpoint}
	{description: Unregisters callback to notify the user in case of any errors occur during enabling camera effect.}
	{prototype: LmiBool VidyoEndpointUnregisterCameraEffectErrorListener(VidyoEndpoint* e)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{return: Returns LMI_TRUE on success or LMI_FALSE otherwise.}
}
*/
LmiBool VidyoEndpointUnregisterCameraEffectErrorListener(VidyoEndpoint* e);

/**
{type:
	{name:  VidyoEndpointShareOptions}
	{parent: Endpoint}
	{description: Represent the options for sharing.}
	{member: {name: enableAudio} {type: LmiBool} {description: A variable that indicates whether the sharing will be with or without audio.}}
	{member: {name: enableHighFramerate} {type: LmiBool} {description: A variable that indicates whether the sharing will be with or without high framerate share.}}
}
*/
typedef struct
{
	LmiBool enableAudio;
	LmiBool enableHighFramerate;
} VidyoEndpointShareOptions;

VidyoEndpointShareOptions* VidyoEndpointShareOptionsConstructDefault(VidyoEndpointShareOptions* options, LmiAllocator* alloc);
VidyoEndpointShareOptions* VidyoEndpointShareOptionsConstructCopy(VidyoEndpointShareOptions* dst, const VidyoEndpointShareOptions* src);
void VidyoEndpointShareOptionsDestruct(VidyoEndpointShareOptions* options);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSelectLocalMonitorAdvanced}
	{parent: VidyoEndpoint}
	{description: Selects the local monitor to share in a conference. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSelectLocalMonitorAdvanced(VidyoEndpoint* e, const VidyoLocalMonitor* localMonitor, VidyoEndpointShareOptions* options)}
	{parameter: {name: e} {description: The VidyoEndpoint object.}}
	{parameter: {name: localMonitor} {description: Monitor to be selected.}}
	{parameter: {name: options} {description: A structure that represent the sharing options.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectLocalMonitorAdvanced(VidyoEndpoint* e, const VidyoLocalMonitor* localMonitor, VidyoEndpointShareOptions* options);

/**
 {function apigroup="simple":
	{name: VidyoEndpointSelectLocalWindowShareAdvanced}
	{parent: VidyoEndpoint}
	{description: Selects the local window to share in a conference. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSelectLocalWindowShareAdvanced(VidyoEndpoint* e, const VidyoLocalWindowShare* localWindowShare, VidyoEndpointShareOptions* options)}
	{parameter: {name: e} {description: The VidyoEndpoint object.}}
	{parameter: {name: localWindowShare} {description: Window to be selected.}}
	{parameter: {name: options} {description: A structure that represent the sharing options.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectLocalWindowShareAdvanced(VidyoEndpoint* e, const VidyoLocalWindowShare* localWindowShare, VidyoEndpointShareOptions* options);

/**
 {function:
	{name: VidyoEndpointInsightsNotifyApplicationEvent}
	{parent: VidyoEndpoint}
	{description: Notify endpoint of an event. This data will be sent to Loki analytics. Event is automatically timestamped. }
	{prototype: LmiBool VidyoEndpointInsightsNotifyApplicationEvent(VidyoEndpoint* e, const char* eventName, const LmiVector(LmiString)* parameters)}
	{parameter: {name: e} {description: The VidyoEndpoint object.}}
	{parameter: {name: eventName} {description: Event name.}}
	{parameter: {name: parameters} {description: Event parameters as string vector. Can be NULL if no parameters are needed.}}
	{return: LMI_TRUE if event is logged by the system, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointInsightsNotifyApplicationEvent(VidyoEndpoint* e, const char* eventName, const LmiVector(LmiString)* parameters);

/**
{function apigroup="simple":
	{name: VidyoEndpointSelectVirtualRenderer}
	{parent: VidyoEndpoint}
	{description: Selects the virtual renderer to receive all events required for rendering. On, iOS, this API must be called from main(UI) thread.}
	{prototype: LmiBool VidyoEndpointSelectVirtualRenderer(VidyoEndpoint* e, const VidyoVirtualRenderer* renderer)}
	{parameter:
		{name: e}
		{description: The VidyoEndpoint object.}
	}
	{parameter:
		{name: renderer}
		{description: The virtual renderer object to select, NULL to unselect.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointSelectVirtualRenderer(VidyoEndpoint* e, const VidyoVirtualRenderer* renderer);

/**
 {function apigroup="simple":
	{name: VidyoEndpointRegisterPlugin}
	{parent: VidyoEndpoint}
	{description: Registers plugin.}
	{prototype: LmiVoidPtr VidyoEndpointRegisterPlugin(VidyoEndpoint* e, const LmiVoidPtr plugin)}
	{parameter: {name: e} {description: The VidyoEndpoint object.}}
	{parameter: {name: plugin} {description: Plugin to register.}}
	{return: Newly registered plugin handle or NULL on failure.}
}
*/
LmiVoidPtr VidyoEndpointRegisterPlugin(VidyoEndpoint* e, const LmiVoidPtr plugin);

/**
 {function apigroup="simple":
	{name: VidyoEndpointUnregisterPlugin}
	{parent: VidyoEndpoint}
	{description: Unregisters plugin.}
	{prototype: LmiBool VidyoEndpointUnregisterPlugin(VidyoEndpoint* e, const LmiVoidPtr plugin)}
	{parameter: {name: e} {description: The VidyoEndpoint object.}}
	{parameter: {name: plugin} {description: Plugin to unregister.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoEndpointUnregisterPlugin(VidyoEndpoint* e, const LmiVoidPtr plugin);

/**
{type visibility="public":
    {name: VidyoEndpointNGRActionType}
    {parent: VidyoEndpoint}
    {description: Lists available actions for next generation renderer library.}
    {value:
        {name: VIDYO_ENDPOINTNGRACTIONTYPE_SetLayout}
        {description: Sets the layout.}}
    {value:
        {name: VIDYO_ENDPOINTNGRACTIONTYPE_SetViewSize}
        {description: Sets the view size.}}
    {value:
        {name: VIDYO_ENDPOINTNGRACTIONTYPE_SetDebugInfoVisible}
        {description: Sets the visibility of debug information.}}
    {value:
        {name: VIDYO_ENDPOINTNGRACTIONTYPE_SetLabelVisible}
        {description: Sets the visibility of labels on renderer.}}
    {value:
        {name: VIDYO_ENDPOINTNGRACTIONTYPE_SetAudioMeterVisible}
        {description: Sets the visibility of audio meter on tiles.}}
    {value:
        {name: VIDYO_ENDPOINTNGRACTIONTYPE_SetBackgroundColor}
        {description: Sets the background color.}}
    {value:
        {name: VIDYO_ENDPOINTNGRACTIONTYPE_SetDensity}
        {description: Sets the density.}}
    {value:
        {name: VIDYO_ENDPOINTNGRACTIONTYPE_SetMaxRemoteParticipants}
        {description: Sets the max remote participants.}}
    {value:
        {name: VIDYO_ENDPOINTNGRACTIONTYPE_PinParticipant}
        {description: Pin the participants.}}
    {value:
        {name: VIDYO_ENDPOINTNGRACTIONTYPE_ShowAudioTiles}
        {description: Sets whether to show audio only tiles.}}
    {value:
        {name: VIDYO_ENDPOINTNGRACTIONTYPE_Invalid}
        {description: Invalid action.}}
}
*/

typedef enum
{
    VIDYO_ENDPOINTNGRACTIONTYPE_SetLayout,
    VIDYO_ENDPOINTNGRACTIONTYPE_SetViewSize,
    VIDYO_ENDPOINTNGRACTIONTYPE_SetDebugInfoVisible,
    VIDYO_ENDPOINTNGRACTIONTYPE_SetLabelVisible,
    VIDYO_ENDPOINTNGRACTIONTYPE_SetAudioMeterVisible,
    VIDYO_ENDPOINTNGRACTIONTYPE_SetBackgroundColor,
    VIDYO_ENDPOINTNGRACTIONTYPE_SetDensity,
    VIDYO_ENDPOINTNGRACTIONTYPE_SetMaxRemoteParticipants,
    VIDYO_ENDPOINTNGRACTIONTYPE_PinParticipant,
    VIDYO_ENDPOINTNGRACTIONTYPE_ShowAudioTiles,
    VIDYO_ENDPOINTNGRACTIONTYPE_Invalid
} VidyoEndpointNGRActionType;

/**
{callback apigroup="simple":
    {name: VidyoEndpointOnNGRResult}
    {parent: VidyoEndpoint}
    {description: Callback providing final response for actions requested to next generation renderer.}
    {prototype: void(*VidyoEndpointOnNGRResult)(const VidyoEndpoint* e, VidyoEndpointNGRActionType type, LmiBool success, const LmiString* error)}
    {parameter:
        {name: e}
        {description: Represent the pointer to VidyoEndpoint object that triggered the callback. }}
    {parameter:
        {name: type}
        {description: Represent the type of  action. Refer VidyoEndpointNGRActionType.}}
    {parameter:
        {name: success}
        {description: Represent the final response.}}
    {parameter:
        {name: error}
        {description: Represent the description of error, empty otherwise.}}
}
*/
typedef void(*VidyoEndpointOnNGRResult)(const VidyoEndpoint* e, VidyoEndpointNGRActionType type, LmiBool success, const LmiString* error);

/**
{function apigroup="simple":
    {name: VidyoEndpointRegisterNGRResultEventListener}
    {parent: VidyoEndpoint}
    {description: This is a synchronous interface. Register a callback to receive the final response for the Next Generation Renderer action.}
    {prototype: LmiBool VidyoEndpointRegisterNGRResultEventListener(VidyoEndpoint* e, VidyoEndpointOnNGRResult onNGRResult)}
    {parameter: 
        {name: e}
        {description: VidyoEndpoint object.}}
    {parameter: 
        {name: onNGRResult}
        {description: Callback to be triggered upon receive the final response from next generation renderer library.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoEndpointRegisterNGRResultEventListener(VidyoEndpoint* e, VidyoEndpointOnNGRResult onNGRResult);

/**
{function apigroup="simple":
    {name: VidyoEndpointUnregisterNGRResultEventListener}
    {parent: VidyoEndpoint}
    {description: This is a synchronous interface. Unregister the callback to stop receiving the final response for the next generation renderer action.}
    {prototype: LmiBool VidyoEndpointUnregisterNGRResultEventListener(VidyoEndpoint* e)}
    {parameter: 
        {name: e}
        {description: VidyoEndpoint object.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoEndpointUnregisterNGRResultEventListener(VidyoEndpoint* e);

LMI_END_EXTERN_C

#if LMI_INLINE_NEED_HEADER_FILE_DEFINITIONS
#include "VidyoEndpointInline.h"
#endif

#endif /* VIDYO_ENDPOINT_H_ */
