/** 
{file:
	{name: VidyoRemoteCamera.h}
	{description: RemoteCamera device. }
	{copyright:
		(c) 2013-2016 Vidyo, Inc.,
		433 Hackensack Avenue,
		Hackensack, NJ  07601.
		All rights reserved.
		The information contained herein is proprietary to Vidyo, Inc.
		and shall not be reproduced, copied (in whole or in part), adapted,
		modified, disseminated, transmitted, transcribed, stored in a retrieval
		system, or translated into any language in any form by any means
		without the express written consent of Vidyo, Inc.}
}
*/
#ifndef VIDYO_REMOTECAMERA_H_ 
#define VIDYO_REMOTECAMERA_H_ 

#include <Lmi/Utils/LmiTypes.h>
#include <Lmi/Utils/LmiVector.h>
#include <Lmi/Utils/LmiList.h>
#include <Lmi/Os/LmiSharedPtr.h>

LMI_BEGIN_EXTERN_C

struct VidyoRemoteCameraImpl_;
typedef struct VidyoRemoteCameraImpl_ VidyoRemoteCameraImpl;
Declare_LmiSharedPtr(VidyoRemoteCameraImpl)
Declare_LmiSharedPtrWrapper(VidyoRemoteCamera, VidyoRemoteCameraImpl)
Declare_LmiWeakPtrWrapper(VidyoRemoteCameraHandle, VidyoRemoteCamera, VidyoRemoteCameraImpl)
Declare_LmiVector(VidyoRemoteCamera)
Declare_LmiList(VidyoRemoteCamera)

LMI_END_EXTERN_C

#include "VidyoParticipant.h"
#include "VidyoLocalRenderer.h"
#include "VidyoForwardDecl.h"
#include "VidyoDevice.h"
#include "VidyoCameraControlCapabilities.h"

LMI_BEGIN_EXTERN_C


/**
{type apigroup="connector,simple" reference-counted="yes":
	{name: VidyoRemoteCamera}
	{parent: Device}
	{include: Lmi/VidyoClient/VidyoRemoteCamera.h}
	{description: This object represents a camera which belongs to the remote participant.}
	{member:
		{name: id}
		{type: LmiString}
		{description: Unique ID.}
	}
	{member:
		{name: name}
		{type: LmiString}
		{description: Name.}
	}
}
*/

/**
{callback visibility="public":
	{name: VidyoRemoteCameraOnRemoteCameraFrame}
	{parent: VidyoRemoteCamera}
	{description: Callback that is triggered each time a remote camera produces a video frame.}
	{prototype: void (*VidyoRemoteCameraOnRemoteCameraFrame)(VidyoRemoteCamera* c, const VidyoParticipant *participant, const VidyoVideoFrame *videoFrame)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object.}
	}
	{parameter:
		{name: participant}
		{description: The remote participant object.}
	}
	{parameter:
		{name: videoFrame}
		{description: The remote camera video frame.}
	}
}
*/
typedef void (*VidyoRemoteCameraOnRemoteCameraFrame)(VidyoRemoteCamera* c, const VidyoParticipant *participant, const VidyoVideoFrame *videoFrame);

/**
{type:
	{name: VidyoRemoteCameraMode}
	{parent: VidyoRemoteCamera}
	{description: }
	{value: {name: VIDYO_REMOTECAMERAMODE_Dynamic} {description: }}
	{value: {name: VIDYO_REMOTECAMERAMODE_Static} {description: }}
}
*/
typedef enum {
	VIDYO_REMOTECAMERAMODE_Dynamic,
	VIDYO_REMOTECAMERAMODE_Static
} VidyoRemoteCameraMode;

/**
{type apigroup="connector,simple":
	{name: VidyoRemoteCameraPosition}
	{parent: VidyoRemoteCamera}
	{description: Position of the camera.}
	{value: {name: VIDYO_REMOTECAMERAPOSITION_Unknown} {description: Position is unknown. }}
	{value: {name: VIDYO_REMOTECAMERAPOSITION_Front} {description: Facing towards the screen. }}
	{value: {name: VIDYO_REMOTECAMERAPOSITION_Back} {description: Facing away from the screen. }}
}
*/
typedef enum {
	VIDYO_REMOTECAMERAPOSITION_Unknown,
	VIDYO_REMOTECAMERAPOSITION_Front,
	VIDYO_REMOTECAMERAPOSITION_Back
} VidyoRemoteCameraPosition;


/**
{callback apigroup="beta":
	{name: VidyoRemoteCameraOnFrameSizeUpdate}
	{parent: VidyoRemoteCamera}
	{description: Callback that is triggered when video frame from a remote camera changes it's size.}
	{prototype: void (*VidyoRemoteCameraOnFrameSizeUpdate)(VidyoRemoteCamera* c, const LmiSizeT width, const LmiSizeT height)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera that triggered the callback.}
	}
	{parameter:
		{name: width}
		{description: The width of the video frame.}
	}
	{parameter:
		{name: height}
		{description: The height of the video frame.}
	}
}
*/
typedef void(*VidyoRemoteCameraOnFrameSizeUpdate)(VidyoRemoteCamera* c, const LmiSizeT width, const LmiSizeT height);

/**
{callback visibility="private":
	{name: VidyoRemoteCameraOnFirstFrameReceived}
	{parent: VidyoRemoteCamera}
	{description: Callback that is triggered when first video frame from a remote camera received.}
	{prototype: void (*VidyoRemoteCameraOnFirstFrameReceived)(VidyoRemoteCamera* c)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera that triggered the callback.}
	}
}
*/
typedef void(*VidyoRemoteCameraOnFirstFrameReceived)(VidyoRemoteCamera* c);

/**
{function visibility="private":
	{name: VidyoRemoteCameraConstruct}
	{parent: VidyoRemoteCamera}
	{description: }
	{prototype: VidyoRemoteCamera* VidyoRemoteCameraConstruct(VidyoRemoteCamera* c, VidyoLmiRemoteSource* remoteSource, VidyoParticipant *participant, LmiAllocator* alloc)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object. }
	}
	{parameter:
		{name: remoteSource}
		{description: }
	}
	{parameter:
		{name: participant}
		{description: }
	}
	{parameter:
		{name: alloc}
		{description: }
	}
	{return: }
}
*/
VidyoRemoteCamera* VidyoRemoteCameraConstruct(VidyoRemoteCamera* c, VidyoLmiRemoteSource* remoteSource, VidyoParticipant *participant, LmiAllocator* alloc);

/**
{function apigroup="connector,simple":
	{name: VidyoRemoteCameraConstructCopy}
	{parent: VidyoRemoteCamera}
	{description: Constructs an VidyoRemoteCamera object as a copy of another.}
	{prototype: VidyoRemoteCamera *VidyoRemoteCameraConstructCopy(VidyoRemoteCamera *obj, const VidyoRemoteCamera *other)}
	{parameter: {name: obj} {description: The VidyoRemoteCamera object to construct.}}
	{parameter: {name: other} {description: The call object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/

/**
{function apigroup="connector,simple":
	{name: VidyoRemoteCameraDestruct}
	{parent: VidyoRemoteCamera}
	{description: Destructs an VidyoRemoteCamera object.}
	{prototype: void VidyoRemoteCameraDestruct(VidyoRemoteCamera *obj)}
	{parameter: {name: obj} {description: The VidyoRemoteCamera object to destruct.}}
}
*/

/**
{function apigroup="connector,simple" visibility="private":
	{name: VidyoRemoteCameraAssign}
	{parent: VidyoRemoteCamera}
	{description: Assigns one VidyoRemoteCamera object the value of another. }
	{prototype: VidyoRemoteCamera *VidyoRemoteCameraAssign(VidyoRemoteCamera *obj, const VidyoRemoteCamera *other)}
	{parameter: {name: obj} {description: The VidyoRemoteCamera object.}}
	{parameter: {name: other} {description: The call object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/

/**
{function apigroup="connector,simple":
	{name: VidyoRemoteCameraGetName}
	{parent: VidyoRemoteCamera}
	{description: Gets the name of the camera.}
	{prototype: const LmiString* VidyoRemoteCameraGetName(const VidyoRemoteCamera* c)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object. }
	}
	{return: Name of the camera.}
}
*/
const LmiString* VidyoRemoteCameraGetName(const VidyoRemoteCamera* c);

/**
{function apigroup="connector,simple":
	{name: VidyoRemoteCameraGetId}
	{parent: VidyoRemoteCamera}
	{description: Gets the unique ID of the camera.}
	{prototype: const LmiString* VidyoRemoteCameraGetId(const VidyoRemoteCamera* c)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object. }
	}
	{return: Unique ID of the camera.}
}
*/
const LmiString* VidyoRemoteCameraGetId(const VidyoRemoteCamera* c);

/**
{function apigroup="connector,simple":
	{name: VidyoRemoteCameraGetPosition}
	{parent: VidyoRemoteCamera}
	{description: Gets the position of the camera.}
	{prototype: VidyoRemoteCameraPosition VidyoRemoteCameraGetPosition(const VidyoRemoteCamera* c)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object. }
	}
	{return: Position of the camera.}
}
*/
VidyoRemoteCameraPosition VidyoRemoteCameraGetPosition(const VidyoRemoteCamera* c);

/**
{function apigroup="connector,simple":
	{name: VidyoRemoteCameraIsControllable}
	{parent: VidyoRemoteCamera}
	{description: Checks if the remote camera is controllable.}
	{prototype: LmiBool VidyoRemoteCameraIsControllable(const VidyoRemoteCamera *c)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object. }
	}
	{return: LMI_TRUE if controllable, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRemoteCameraIsControllable(const VidyoRemoteCamera *c);

/**
{function apigroup="connector,simple" visibility="public":
    {name: VidyoRemoteCameraGetControlCapabilities}
    {parent: VidyoRemoteCamera}
    {description: Gets camera control capabilities.}
    {prototype: const VidyoCameraControlCapabilities *VidyoRemoteCameraGetControlCapabilities(const VidyoRemoteCamera *c)}
    {parameter:
        {name: c}
        {description: The VidyoRemoteCamera object.}
    }
    {return: The VidyoCameraControlCapabilities object.}
}
*/
const VidyoCameraControlCapabilities *VidyoRemoteCameraGetControlCapabilities(const VidyoRemoteCamera *c);

/**
{function apigroup="connector,simple":
	{name: VidyoRemoteCameraControlPTZNudge}
	{parent: VidyoRemoteCamera}
	{description: Send the PTZ Nudge commands to the remote camera.}
	{prototype: LmiBool VidyoRemoteCameraControlPTZNudge(VidyoRemoteCamera *c, LmiInt pan, LmiInt tilt, LmiInt zoom)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object. }
	}
	{parameter:
		{name: pan}
		{description: Number of nudges to pan the camera. }
	}
	{parameter:
		{name: tilt}
		{description: Number of nudges to tilt the camera. }
	}
	{parameter:
		{name: zoom}
		{description: Number of nudges to zoom the camera. }
	}
	{return: LMI_TRUE when the control command was sent, LMI_FALSE otherwise.}
	{note: Positive numbers increase the nudge, and negative numbers decrease the nudge for left/right up/down zoom in/out. }
}
*/
LmiBool VidyoRemoteCameraControlPTZNudge(VidyoRemoteCamera *c, LmiInt pan, LmiInt tilt, LmiInt zoom);

/**
{function apigroup="connector,simple":
    {name: VidyoRemoteCameraControlPTZStart}
    {parent: VidyoRemoteCamera}
    {description: Send the PTZ Start commands to the remote camera.}
    {prototype: LmiBool VidyoRemoteCameraControlPTZStart(VidyoRemoteCamera *c, VidyoCameraControlDirection direction, LmiTime timeout)}
    {parameter:
        {name: c}
        {description: The VidyoRemoteCamera object. }
    }
    {parameter:
        {name: direction}
        {description: The VidyoCameraControlDirection enum that indicates direction of move.}
    }
    {parameter:
        {name: timeout}
        {description: Amount of time in nanoseconds at which camera will be moving in the specified directions.}
    }
    {return: LMI_TRUE when the control command was sent, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRemoteCameraControlPTZStart(VidyoRemoteCamera *c, VidyoCameraControlDirection direction, LmiTime timeout);

/**
{function apigroup="connector,simple":
    {name: VidyoRemoteCameraControlPTZStop}
    {parent: VidyoRemoteCamera}
    {description: Send the PTZ Stop commands to the remote camera.}
    {prototype: LmiBool VidyoRemoteCameraControlPTZStop(VidyoRemoteCamera *c)}
    {parameter:
        {name: c}
        {description: The VidyoRemoteCamera object. }
    }
    {return: LMI_TRUE when the control command was sent, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRemoteCameraControlPTZStop(VidyoRemoteCamera *c);

/**
{function apigroup="connector,simple" visibility="public":
	{name: VidyoRemoteCameraShowCameraControl}
	{parent: VidyoRemoteCamera}
	{description: Shows camera control when rendering remote camera. }
	{prototype: LmiBool VidyoRemoteCameraShowCameraControl(VidyoRemoteCamera* c, LmiBool show)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object.}
	}
	{parameter:
		{name: show}
		{description: Show or hide camera controls. }
	}
	{return: LMI_TRUE on success.}
	{note: Applies to Composite and Custom rendering layouts. }
}
*/
LmiBool VidyoRemoteCameraShowCameraControl(VidyoRemoteCamera* c, LmiBool show);

/**
{function visibility="private":
	{name: VidyoRemoteCameraSetUserData}
	{parent: VidyoRemoteCamera}
	{description: }
	{prototype: void VidyoRemoteCameraSetUserData(VidyoRemoteCamera* c, LmiVoidPtr userData)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object. }
	}
	{parameter:
		{name: userData}
		{description: }
	}
}
*/
void VidyoRemoteCameraSetUserData(VidyoRemoteCamera* c, LmiVoidPtr userData);

/**
{function visibility="private":
	{name: VidyoRemoteCameraGetUserData}
	{parent: VidyoRemoteCamera}
	{description: }
	{prototype: LmiVoidPtr VidyoRemoteCameraGetUserData(const VidyoRemoteCamera* c)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object. }
	}
	{return: }
}
*/
LmiVoidPtr VidyoRemoteCameraGetUserData(const VidyoRemoteCamera* c);

/**
{function visibility="public":
	{name: VidyoRemoteCameraRegisterFrameEventListener}
	{parent: VidyoRemoteCamera}
	{description: Registers to get notified about a new remote camera frame.}
	{prototype: LmiBool VidyoRemoteCameraRegisterFrameEventListener(VidyoRemoteCamera* c, VidyoRemoteCameraOnRemoteCameraFrame onFrame)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered each time a remote camera produces a video frame.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRemoteCameraRegisterFrameEventListener(VidyoRemoteCamera* c, VidyoRemoteCameraOnRemoteCameraFrame onFrame);

/**
 {function apigroup="public":
	{name: VidyoRemoteCameraUnregisterFrameEventListener}
	{parent: VidyoRemoteCamera}
	{description: Unregisters remote camera frame notifications.}
	{prototype: LmiBool VidyoRemoteCameraUnregisterFrameEventListener(VidyoRemoteCamera* c)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoRemoteCameraUnregisterFrameEventListener(VidyoRemoteCamera* c);

/**
{function visibility="private":
	{name: VidyoRemoteCameraGetParticipant}
	{parent: VidyoRemoteCamera}
	{description: }
	{prototype: VidyoParticipant *VidyoRemoteCameraGetParticipant(VidyoRemoteCamera* c)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object. }
	}
	{return: }
}
*/
VidyoParticipant *VidyoRemoteCameraGetParticipant(VidyoRemoteCamera *c);

/**
{function:
	{name: VidyoRemoteCameraAddToLocalRenderer}
	{parent: VidyoRemoteCamera}
	{description: }
	{prototype: LmiVoidPtr VidyoRemoteCameraAddToLocalRenderer(VidyoRemoteCamera* c, const VidyoLocalRenderer* localRenderer, VidyoRemoteCameraMode mode)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object. }
	}
	{parameter:
		{name: localRenderer}
		{description: }
	}
	{parameter:
		{name: mode}
		{description: }
	}
	{return: }
}
*/
LmiVoidPtr VidyoRemoteCameraAddToLocalRenderer(VidyoRemoteCamera* c, const VidyoLocalRenderer* localRenderer, VidyoRemoteCameraMode mode);

/**
{function:
	{name: VidyoRemoteCameraRemoveFromLocalRenderer}
	{parent: VidyoRemoteCamera}
	{description: }
	{prototype: LmiBool VidyoRemoteCameraRemoveFromLocalRenderer(VidyoRemoteCamera* c, const VidyoLocalRenderer* localRenderer)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object. }
	}
	{parameter:
		{name: localRenderer}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoRemoteCameraRemoveFromLocalRenderer(VidyoRemoteCamera* c, const VidyoLocalRenderer* localRenderer);

/**
{function:
	{name: VidyoRemoteCameraSetPositionInLocalRenderer}
	{parent: VidyoRemoteCamera}
	{description: }
	{prototype: LmiBool VidyoRemoteCameraSetPositionInLocalRenderer(VidyoRemoteCamera *c, const VidyoLocalRenderer *localRenderer, LmiInt x, LmiInt y, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object. }
	}
	{parameter:
		{name: localRenderer}
		{description: }
	}
	{parameter:
		{name: x}
		{description: }
	}
	{parameter:
		{name: y}
		{description: }
	}
	{parameter:
		{name: width}
		{description: }
	}
	{parameter:
		{name: height}
		{description: }
	}
	{parameter:
		{name: frameInterval}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoRemoteCameraSetPositionInLocalRenderer(VidyoRemoteCamera *c, const VidyoLocalRenderer *localRenderer, LmiInt x, LmiInt y, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
{function:
	{name: VidyoRemoteCameraGetLocalRenderingStreamId}
	{parent: VidyoRemoteCamera}
	{description: }
	{prototype: LmiVoidPtr VidyoRemoteCameraGetLocalRenderingStreamId(VidyoRemoteCamera* c, LmiInt index)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object. }
	}
	{parameter:
		{name: index}
		{description: }
	}
	{return: }
}
*/
LmiVoidPtr VidyoRemoteCameraGetLocalRenderingStreamId(VidyoRemoteCamera *c, LmiInt index);

/**
{function apigroup="beta":
	{name: VidyoRemoteCameraRegisterFrameSizeListener}
	{parent: VidyoRemoteCamera}
	{description: Registers to get notified about camera frame size changes.}
	{prototype: LmiBool VidyoRemoteCameraRegisterFrameSizeListener(VidyoRemoteCamera* c, VidyoRemoteCameraOnFrameSizeUpdate onFrameSizeUpdate)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object.}
	}
	{parameter:
		{name: onFrameSizeUpdate}
		{description: Callback that is triggered whenever frame size changes.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRemoteCameraRegisterFrameSizeListener(VidyoRemoteCamera* c, VidyoRemoteCameraOnFrameSizeUpdate onFrameSizeUpdate);

/**
{function apigroup="beta":
	{name: VidyoRemoteCameraUnregisterFrameSizeListener}
	{parent: VidyoRemoteCamera}
	{description: Unregisters frame size event notifications.}
	{prototype: LmiBool VidyoRemoteCameraUnregisterFrameSizeListener(VidyoRemoteCamera* c)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRemoteCameraUnregisterFrameSizeListener(VidyoRemoteCamera* c);


/**
{function visibility="private":
	{name: VidyoRemoteCameraConstructFromDevice}
	{parent: VidyoRemoteCamera}
	{description: }
	{prototype: VidyoRemoteCamera *VidyoRemoteCameraConstructFromDevice(VidyoRemoteCamera* vidyoRemoteCamera, const VidyoDevice *device, LmiAllocator* alloc)}
	{parameter:
		{name: vidyoRemoteCamera}
		{description: }
	}
	{parameter:
		{name: device}
		{description: }
	}
	{parameter:
		{name: alloc}
		{description: }
	}
	{return: }
}
*/
VidyoRemoteCamera *VidyoRemoteCameraConstructFromDevice(VidyoRemoteCamera* vidyoRemoteCamera, const VidyoDevice *device, LmiAllocator* alloc);

/**
{function visibility="private":
	{name: VidyoRemoteCameraGetStats}
	{parent: VidyoRemoteCamera}
	{description: Fill in stats data in VidyoRemoteVideoSourceStats from a remote camera.}
	{prototype: LmiBool VidyoRemoteCameraGetStats(VidyoRemoteCamera *camera, VidyoRemoteVideoSourceStats *stats)}
	{parameter: {name: camera} {description: The remote camera object to extract stats.}}
	{parameter: {name: stats} {description: The stats object to fill.}}
	{return: LMI_TRUE on success.}
}
*/
LmiBool VidyoRemoteCameraGetStats(VidyoRemoteCamera *camera, VidyoRemoteVideoSourceStats *stats);

/**
 {function visibility="private":
	{name: VidyoRemoteCameraRequestVideo}
	{parent: VidyoRemoteCamera}
	{description: Requests video for remote camera before assign.}
	{prototype: LmiBool VidyoRemoteCameraRequestVideo(VidyoRemoteCamera* c, LmiUint width, LmiUint height, LmiTime frameInterval, VidyoRemoteCameraOnFirstFrameReceived onFirstFrameReceived)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested video.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested video.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested video.}
	}
	{parameter:
		{name: onFirstFrameReceived}
		{description: Callback that is triggered when first frame from a remote camera received.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRemoteCameraRequestVideo(VidyoRemoteCamera* c, LmiUint width, LmiUint height, LmiTime frameInterval, VidyoRemoteCameraOnFirstFrameReceived onFirstFrameReceived);

/**
{callback apigroup="connector,simple":
	{name: VidyoRemoteCameraOnPresetUpdated}
	{parent: VidyoRemoteCamera}
	{description: Callback that is triggered each time a remote camera updates preset information.}
	{prototype: typedef void (*VidyoRemoteCameraOnPresetUpdated)(VidyoRemoteCamera* c, const LmiVector(VidyoCameraPreset) *presets)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object.}
	}
	{parameter:
		{name: presets}
		{description: Ordered list of preset information.}
	}
}
*/
typedef void (*VidyoRemoteCameraOnPresetUpdated)(VidyoRemoteCamera* c, const LmiVector(VidyoCameraPreset) *presets);

/**
{function apigroup="connector,simple":
	{name: VidyoRemoteCameraRegisterPresetEventListener}
	{parent: VidyoRemoteCamera}
	{description: Registers a callback to receive notifications about camera preset information.}
	{prototype: LmiBool VidyoRemoteCameraRegisterPresetEventListener(VidyoRemoteCamera* c, VidyoRemoteCameraOnPresetUpdated onPresetUpdated)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object.}
	}
	{parameter:
		{name: onPresetUpdated}
		{description: Callback that is triggered whenever camera preset information changes.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRemoteCameraRegisterPresetEventListener(VidyoRemoteCamera* c, VidyoRemoteCameraOnPresetUpdated onPresetUpdated);

/**
{function apigroup="connector,simple":
	{name: VidyoRemoteCameraUnregisterPresetEventListener}
	{parent: VidyoRemoteCamera}
	{description: Unregisters callback to get notified about camera preset information.}
	{prototype: LmiBool VidyoRemoteCameraUnregisterPresetEventListener(VidyoRemoteCamera* c)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRemoteCameraUnregisterPresetEventListener(VidyoRemoteCamera* c);

/**
{function apigroup="connector,simple":
	{name: VidyoRemoteCameraActivatePreset}
	{parent: VidyoRemoteCamera}
	{description: Send the preset index to the remote camera.}
	{prototype: LmiBool VidyoRemoteCameraActivatePreset(VidyoRemoteCamera *c, LmiUint index)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object.}}
	{parameter:
		{name: index}
		{description: Represents the preset index.}}
	{return: LMI_TRUE when the activate preset command was sent successfully, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRemoteCameraActivatePreset(VidyoRemoteCamera *c, LmiUint index);

/**
{function apigroup="connector,simple":
	{name: VidyoRemoteCameraViscaControl}
	{parent: VidyoRemoteCamera}
	{description: Sends the visca message to the remote camera.}
	{prototype: LmiBool VidyoRemoteCameraViscaControl(VidyoRemoteCamera *c, const LmiString *viscaCmd, const LmiString *viscaCmdId)}
	{parameter:
		{name: c}
		{description: The VidyoRemoteCamera object.}}
	{parameter:
		{name: viscaCmd}
		{description: Represents the visca command, with maximum length as 128 characters. This should be any valid visca command.For example: "81 01 04 3f 02 02 ff"}}
	{parameter:
		{name: viscaCmdId}
		{description: Represents the visca command identifier, with maximum length as 128 characters. Application can set NULL OR any valid string identifier, which can be used at remote end for debugging.}}
	{return: LMI_TRUE when the visca command was sent successfully, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRemoteCameraViscaControl(VidyoRemoteCamera *c, const LmiString *viscaCmd, const LmiString *viscaCmdId);

LMI_END_EXTERN_C

#if LMI_INLINE_NEED_HEADER_FILE_DEFINITIONS
#include "VidyoRemoteCameraInline.h"
#endif

#endif /* VIDYO_REMOTECAMERA_H_ */
