#ifndef BNB_EFFECT_PLAYER_WRAP_EFFECT_PLAYER_H
#define BNB_EFFECT_PLAYER_WRAP_EFFECT_PLAYER_H

#include <stdbool.h>
#include <stdint.h>

#include <bnb/common_types.h>
#include <bnb/error.h>

#ifdef __cplusplus
extern "C"
{
#endif

/*
 * EffectPlayer API
 */
typedef struct effect_holder effect_holder_t;
typedef struct full_image_holder full_image_holder_t;
typedef struct effect_player_holder effect_player_holder_t;
typedef struct effect_manager_holder effect_manager_holder_t;

typedef enum
{
    bnb_nn_mode_automatically,
    bnb_nn_mode_enable,
    bnb_nn_mode_disable
} bnb_nn_mode_t;

typedef enum
{
    bnb_consistency_mode_synchronous,
    bnb_consistency_mode_synchronous_when_effect_loaded,
    bnb_consistency_mode_asynchronous_inconsistent,
    bnb_consistency_mode_asynchronous_consistent
} bnb_consistency_mode_t;

typedef struct
{
    int32_t fx_width;
    int32_t fx_height;

    bnb_nn_mode_t nn_enable;

    bnb_face_search_mode_t face_search;

    bool js_debugger_enable;

    bool manual_audio;
} bnb_effect_player_configuration_t;

typedef uintptr_t listener_id_t;
/* listeners */
typedef void(*bnb_effect_activation_completion_listener_t)(const char* url, void* user_data);
typedef void(*bnb_error_listener_t)(const char* domain, const char* message, void* user_data);

/*
 * full_image_t
 */
full_image_holder_t* bnb_full_image_from_yuv_nv12_img(
    bnb_image_format_t format,
    uint8_t* y_plane,
    int32_t y_stride,
    uint8_t* uv_plane,
    int32_t uv_stride,
    bnb_error**);
full_image_holder_t* bnb_full_image_from_yuv_i420_img(
    bnb_image_format_t format,
    uint8_t* y_plane,
    int32_t y_row_stride,
    int32_t y_pixel_stride,
    uint8_t* u_plane,
    int32_t u_row_stride,
    int32_t u_pixel_stride,
    uint8_t* v_plane,
    int32_t v_row_stride,
    int32_t v_pixel_stride,
    bnb_error**);
full_image_holder_t* bnb_full_image_from_bpc8_img(
    bnb_image_format_t format,
    bnb_pixel_format_t pixel_format,
    uint8_t* data,
    int32_t stride,
    bnb_error**);

void bnb_full_image_get_format(full_image_holder_t* image, bnb_image_format_t* format, bnb_error**);

/* should be called after bnb_effect_player_push_frame() */
void bnb_full_image_release(full_image_holder_t* img, bnb_error**);

/*
 * effect_player
 */
void bnb_effect_player_load_glad_functions(void* proc, bnb_error**);

effect_player_holder_t* bnb_effect_player_create(bnb_effect_player_configuration_t* cfg, bnb_error**);
void bnb_effect_player_destroy(effect_player_holder_t* ep, bnb_error**);

void bnb_effect_player_set_render_consistency_mode(effect_player_holder_t* ep, bnb_consistency_mode_t mode, bnb_error**);

void bnb_effect_player_surface_created(effect_player_holder_t* ep, int32_t width, int32_t height, bnb_error**);
void bnb_effect_player_surface_changed(effect_player_holder_t* ep, int32_t width, int32_t height, bnb_error**);
void bnb_effect_player_surface_destroyed(effect_player_holder_t* ep, bnb_error**);

void bnb_effect_player_push_frame(effect_player_holder_t* ep, full_image_holder_t* img, bnb_error**);

int64_t bnb_effect_player_draw(effect_player_holder_t* ep, bnb_error**);

void bnb_effect_player_playback_play(effect_player_holder_t* ep, bnb_error**);
void bnb_effect_player_playback_pause(effect_player_holder_t* ep, bnb_error**);
void bnb_effect_player_playback_stop(effect_player_holder_t* ep, bnb_error**);

void bnb_effect_player_enable_audio(effect_player_holder_t* ep, bool enable, bnb_error**);
void bnb_effect_player_set_effect_volume(effect_player_holder_t* ep, float volume, bnb_error**);

listener_id_t bnb_effect_player_add_effect_activation_completion_listener(effect_player_holder_t* ep, bnb_effect_activation_completion_listener_t l, void*, bnb_error**);
void bnb_effect_player_remove_effect_activation_completion_listener(effect_player_holder_t* ep, listener_id_t l, bnb_error**);

effect_manager_holder_t* bnb_effect_player_get_effect_manager(effect_player_holder_t* ep, bnb_error**);

/*
 * effect_manager
 */
effect_holder_t* bnb_effect_manager_load_effect(effect_manager_holder_t* em, const char* effect_path, bnb_error**);
effect_holder_t* bnb_effect_manager_load_effect_async(effect_manager_holder_t* em, const char* effect_path, bnb_error**);

effect_holder_t* bnb_effect_manager_get_current_effect(effect_manager_holder_t* em, bnb_error**);

void bnb_effect_manager_set_effect_size(effect_manager_holder_t* em, int width, int height, bnb_error**);

listener_id_t bnb_effect_manager_add_error_listener(effect_manager_holder_t* em, bnb_error_listener_t l, void* user_data, bnb_error**);
void bnb_effect_manager_remove_error_listener(effect_manager_holder_t* em, listener_id_t l, bnb_error**);

/*
 * effect
 */
/* returned string must be freed */
char* bnb_effect_get_url(effect_holder_t* effect, bnb_error**);
void bnb_effect_call_js_method(effect_holder_t* effect, const char* method, const char* param, bnb_error**);
void bnb_effect_reset(effect_holder_t* effect, bnb_error**);

#ifdef __cplusplus
} //extern "C"
#endif

#endif /* BNB_EFFECT_PLAYER_WRAP_EFFECT_PLAYER_H */
