/**
{file:
    {name: LmiDirectoryEntry.h}
    {description: Objects to provide access to directory entries on disk or other persistent storage.}
    {copyright:
    	(c) 2018 Vidyo, Inc.,
    	433 Hackensack Avenue,
    	Hackensack, NJ 07601.

    	All rights reserved.

    	The information contained herein is proprietary to Vidyo, Inc.
    	and shall not be reproduced, copied (in whole or in part), adapted,
    	modified, disseminated, transmitted, transcribed, stored in a retrieval
    	system, or translated into any language in any form by any means
    	without the express written consent of Vidyo, Inc.
    	                  ***** CONFIDENTIAL *****
    }
}
*/
#ifndef LMI_DIRECTORYENTRY_H_
#define LMI_DIRECTORYENTRY_H_

#include <Lmi/Utils/LmiTypes.h>
#include <Lmi/Utils/LmiAllocator.h>
#include <Lmi/Utils/LmiVector.h>
#include <Lmi/Os/LmiPath.h>

/**
{type:
	{name: LmiDirectoryEntryType}
	{parent: LmiDirectoryEntry}
	{include: Lmi/Os/LmiDirectoryEntry.h}
	{description: The type of a directory entry.}
 	{value: {name: LMI_DIRECTORYENTRY_TYPE_Directory} {description: Indicates the directory entry is a directory (LmiDirectory).}}
 	{value: {name: LMI_DIRECTORYENTRY_TYPE_File} {description: Indicates the directory entry is a file (LmiFile).}}
 	{value: {name: LMI_DIRECTORYENTRY_TYPE_Other} {description: Indicates the directory entry is an unknown type.}}
}
*/
enum {
	LMI_DIRECTORYENTRY_TYPE_Directory = 1 << 16,
	LMI_DIRECTORYENTRY_TYPE_File = 1 << 17,
	/* Insert future entry types here */
	LMI_DIRECTORYENTRY_TYPE_Other = 1 << 31
};
typedef int LmiDirectoryEntryType;

/**
{type:
	{name: LmiDirectoryEntry}
	{parent: Os}
	{include: Lmi/Os/LmiDirectoryEntry.h}
	{description: A directory entry.}
}
*/
typedef struct {
	LmiPath path;
	LmiDirectoryEntryType type;
} LmiDirectoryEntry;

LMI_BEGIN_EXTERN_C

/* TODO: Add dockblocks */
LmiDirectoryEntry* LmiDirectoryEntryConstruct(LmiDirectoryEntry* x, LmiDirectoryEntryType type, const LmiPath* path);
LmiDirectoryEntry* LmiDirectoryEntryConstructCopy(LmiDirectoryEntry* x, const LmiDirectoryEntry* y);
void LmiDirectoryEntryDestruct(LmiDirectoryEntry* x);
LmiDirectoryEntry* LmiDirectoryEntryAssign(LmiDirectoryEntry* x, const LmiDirectoryEntry* y);
LmiBool LmiDirectoryEntryEqual(const LmiDirectoryEntry* x, const LmiDirectoryEntry* y);
LmiBool LmiDirectoryEntrySwap(LmiDirectoryEntry* x, LmiDirectoryEntry* y);
LmiDirectoryEntryType LmiDirectoryEntryGetType(const LmiDirectoryEntry* x);
const LmiPath* LmiDirectoryEntryGetPath(const LmiDirectoryEntry* x);
const char* LmiDirectoryEntryGetPathCStr(const LmiDirectoryEntry* x);
LmiBool LmiDirectoryEntrySetPath(LmiDirectoryEntry* x, const LmiPath* path);
LmiAllocator* LmiDirectoryEntryGetAllocator(const LmiDirectoryEntry* x);

LMI_END_EXTERN_C

Declare_LmiVector(LmiDirectoryEntry)

#endif
