/** 
{file:
	{name: VidyoConnector.h}
	{description: High level client library. }
	{copyright:
		(c) 2014-2016 Vidyo, Inc.,
		433 Hackensack Avenue,
		Hackensack, NJ  07601.
		All rights reserved.
		The information contained herein is proprietary to Vidyo, Inc.
		and shall not be reproduced, copied (in whole or in part), adapted,
		modified, disseminated, transmitted, transcribed, stored in a retrieval
		system, or translated into any language in any form by any means
		without the express written consent of Vidyo, Inc.}
}
*/
#ifndef VIDYO_CONNECTOR_H_
#define VIDYO_CONNECTOR_H_

#include <Lmi/Utils/LmiAllocator.h>
#include <Lmi/Utils/LmiMisc.h>
#include <Lmi/Utils/LmiTypes.h>
#include <Lmi/Utils/LmiVector.h>
#include <Lmi/Utils/LmiList.h>
#include <Lmi/Utils/LmiUtils.h>
#include <Lmi/Os/LmiMallocAllocator.h>
#include <Lmi/Os/LmiOs.h>
#include <Lmi/Os/LmiLog.h>
#include <Lmi/Os/LmiSharedPtr.h>


LMI_BEGIN_EXTERN_C

struct VidyoConnectorImpl_;
typedef struct VidyoConnectorImpl_ VidyoConnectorImpl;
Declare_LmiSharedPtr(VidyoConnectorImpl)
Declare_LmiSharedPtrWrapper(VidyoConnector, VidyoConnectorImpl)
Declare_LmiWeakPtrWrapper(VidyoConnectorHandle, VidyoConnector, VidyoConnectorImpl)

LMI_END_EXTERN_C

#include "VidyoLogRecord.h"
#include "VidyoParticipant.h"
#include "VidyoChatMessage.h"
#include "VidyoNetworkInterface.h"

#include "VidyoVideoFrame.h"
/* devices */
#include "VidyoLocalCamera.h"
#include "VidyoLocalMicrophone.h"
#include "VidyoLocalSpeaker.h"
#include "VidyoLocalWindowShare.h"
#include "VidyoVirtualVideoSource.h"
#include "VidyoVirtualAudioSource.h"
#include "VidyoLocalMonitor.h"
#include "VidyoRemoteCamera.h"
#include "VidyoRemoteMicrophone.h"
#include "VidyoRemoteWindowShare.h"
#include "VidyoVirtualRenderer.h"
#include "VidyoUtils.h"

LMI_BEGIN_EXTERN_C

/**
{package apigroup="connector,simple":
	{name: Connector}
	{parent: VidyoClient}
	{include: Lmi/VidyoClient/VidyoConnector.h}
	{description: Package that encapsulates all the APIs required to build a simple videoconferencing application.}
}
*/

/**
 {type apigroup="connector" reference-counted="yes":
	{name: VidyoConnector}
	{parent: Connector}
	{description: The videoconferencing API that handles all aspects of the media (audio/video) connection.}
}
*/

/**
{function apigroup="connector":
	{name: VidyoConnectorInitialize}
	{parent: Connector}
	{description: Initialize the VidyoConnector API.}
	{prototype: LmiBool VidyoConnectorInitialize(void)}
	{return: }
	{note: This function is required: Any application that uses the VidyoConnector API must initialize it first, typically when the application starts.}
}
*/
LmiBool VidyoConnectorInitialize(void);

/**
{function apigroup="connector":
	{name: VidyoConnectorUninitialize}
	{parent: Connector}
	{description: Uninitialize the VidyoConnector API.}
	{prototype: void VidyoConnectorUninitialize(void)}
	{note: This function is required: Any application that uses the VidyoConnector API must uninitialize it once it is no longer needed or before the application exits.}
}
*/
void VidyoConnectorUninitialize(void);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetExperimentalOptions}
	{parent: Connector}
	{description: Sets up experimental options on the VidyoConnector API.}
	{prototype: LmiBool VidyoConnectorSetExperimentalOptions(const char *options)}
	{parameter:
		{name: options}
		{description: The experimental optons in JSON format.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: This function is for experimental features and should not be used unless specifically instructed by Vidyo.}
}
*/
LmiBool VidyoConnectorSetExperimentalOptions(const char *options);

/**
{type apigroup="connector" visibility="private":
	{name: VidyoConnectorHandle}
	{parent: VidyoConnector}
	{description: Weak reference to a VidyoConnector object.}
}
*/

/**
 {type apigroup="connector" visibility="public":
	{name: VidyoConnectorFailReason}
	{parent: VidyoConnector}
	{description: Reasons why a connection attempt failed.}
	{value: {name: VIDYO_CONNECTORFAILREASON_ConnectionFailed}
		{description: The destination could not be reached.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_ConnectionLost}
		{description: The transport connection was lost prior to completing the login procedure.}}
 	{value: {name: VIDYO_CONNECTORFAILREASON_ConnectionTimeout}
		{description: The signaling connection timed-out prior to completing the login procedure.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_NoResponse}
		{description: The service was successfully contacted, but the service ignored the user's
			request to log in or it did not answer in a reasonable amount of time.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_Terminated}
		{description: The service was successfully contacted, but the service closed the connection
			or it refused to continue processing the login request.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_InvalidToken}
		{description: The token that was provided is unknown to the service or unauthorized to log in.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_InvalidRoomPin}
		{description: The room pin that was provided is invalid or unauthorized.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_UnableToCreateResource} {description: The server was unable to create the resource.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_NoResponseFromResource} {description: The server did
		not respond in a reasonable amount of time to the request to connect to the resource.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_InvalidResourceId} {description: The resource ID was invalid.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_ResourceFull} {description: The resource
		is at full capacity, and it cannot accept additional members.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_NotMember} {description: The resource allows
		only certain users to enter, and the current user is not one of them.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_Banned} {description: The user has
		been banned from entering the resource.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_MediaNotEnabled} {description: The client could not enable media.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_MediaFailed} {description: An established media connection failed.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_MiscLocalError} {description: The server
		rejected the user's request to enter the resource, due to a miscellaneous problem with the request.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_MiscRemoteError} {description: The server
		rejected the user's request to enter the resource, due to a miscellaneous problem of its own.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_MiscError}
		{description: The login failed for some other miscellaneous reason.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_InvalidDisplayNameLength}
		{description: Length of the display name is longer than 256 characters.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_RoomLocked}
		{description: An established media connection failed because the room is locked and it cannot accept additional members.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_RoomDisabled}
		{description: An established media connection failed because the room is disabled.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_AllLinesInUse}
		{description: An established media connection failed because all lines are in use.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_SeatLicenseExpired}
		{description: An established media connection failed because the seat license has expired.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_NotLicensed}
		{description: An established media connection failed because the endpoint is not licensed.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_Rejected}
		{description: Attempt to enable media was rejected by the server.}}
	{value: {name: VIDYO_CONNECTORFAILREASON_BrowserVersionNotSupported}
		{description: VidyoClient library does not support this browser versoin.}}
}
*/
typedef enum
{
	VIDYO_CONNECTORFAILREASON_ConnectionFailed,
	VIDYO_CONNECTORFAILREASON_ConnectionLost,
	VIDYO_CONNECTORFAILREASON_ConnectionTimeout,
	VIDYO_CONNECTORFAILREASON_NoResponse,
	VIDYO_CONNECTORFAILREASON_Terminated,
	VIDYO_CONNECTORFAILREASON_InvalidToken,
	VIDYO_CONNECTORFAILREASON_InvalidRoomPin,
	VIDYO_CONNECTORFAILREASON_UnableToCreateResource,
	VIDYO_CONNECTORFAILREASON_NoResponseFromResource,
	VIDYO_CONNECTORFAILREASON_InvalidResourceId,
	VIDYO_CONNECTORFAILREASON_ResourceFull,
	VIDYO_CONNECTORFAILREASON_NotMember,
	VIDYO_CONNECTORFAILREASON_Banned,
	VIDYO_CONNECTORFAILREASON_MediaNotEnabled,
	VIDYO_CONNECTORFAILREASON_MediaFailed,
	VIDYO_CONNECTORFAILREASON_MiscLocalError,
	VIDYO_CONNECTORFAILREASON_MiscRemoteError,
	VIDYO_CONNECTORFAILREASON_MiscError,
	VIDYO_CONNECTORFAILREASON_InvalidDisplayNameLength,
	VIDYO_CONNECTORFAILREASON_RoomLocked,
	VIDYO_CONNECTORFAILREASON_RoomDisabled,
	VIDYO_CONNECTORFAILREASON_AllLinesInUse,
	VIDYO_CONNECTORFAILREASON_SeatLicenseExpired,
	VIDYO_CONNECTORFAILREASON_NotLicensed,
	VIDYO_CONNECTORFAILREASON_Rejected,
	VIDYO_CONNECTORFAILREASON_BrowserVersionNotSupported
} VidyoConnectorFailReason;

/**
 {type apigroup="connector" visibility="public":
	{name: VidyoConnectorDisconnectReason}
	{parent: VidyoConnector}
	{description: Reasons why an established connection was disconnected.}
	{value: {name: VIDYO_CONNECTORDISCONNECTREASON_Disconnected}
		{description: The user disconnected gracefully.}}
	{value: {name: VIDYO_CONNECTORDISCONNECTREASON_ConnectionLost}
		{description: The transport connection was lost.}}
 	{value: {name: VIDYO_CONNECTORDISCONNECTREASON_ConnectionTimeout}
		{description: The signaling connection timed-out.}}
	{value: {name: VIDYO_CONNECTORDISCONNECTREASON_NoResponse}
		{description: The service did not respond in a reasonable amount of time to a request by the user.}}
	{value: {name: VIDYO_CONNECTORDISCONNECTREASON_Terminated}
		{description: The service closed the connection or otherwise terminated the login session.}}
	{value: {name: VIDYO_CONNECTORDISCONNECTREASON_Booted}
		{description: The user was kicked out of the room. This could be the result of an administrator's action.}}
	{value: {name: VIDYO_CONNECTORDISCONNECTREASON_MiscLocalError}
		{description: A miscellaneous problem occurred with this client library.}}
	{value: {name: VIDYO_CONNECTORDISCONNECTREASON_MiscRemoteError}
		{description: A miscellaneous problem occurred with the conferencing service.}}
	{value: {name: VIDYO_CONNECTORDISCONNECTREASON_MiscError}
		{description: The connection was closed for some other miscellaneous reason.}}
}
*/
typedef enum
{
	VIDYO_CONNECTORDISCONNECTREASON_Disconnected,
	VIDYO_CONNECTORDISCONNECTREASON_ConnectionLost,
	VIDYO_CONNECTORDISCONNECTREASON_ConnectionTimeout,
	VIDYO_CONNECTORDISCONNECTREASON_NoResponse,
	VIDYO_CONNECTORDISCONNECTREASON_Terminated,
	VIDYO_CONNECTORDISCONNECTREASON_Booted,
	VIDYO_CONNECTORDISCONNECTREASON_MiscLocalError,
	VIDYO_CONNECTORDISCONNECTREASON_MiscRemoteError,
	VIDYO_CONNECTORDISCONNECTREASON_MiscError
} VidyoConnectorDisconnectReason;

/**
 {type apigroup="connector" visibility="public":
	{name: VidyoConnectorState}
	{parent: VidyoConnector}
	{description: States of a connection, mostly representing the progress of a connection attempt.}
	{value: {name: VIDYO_CONNECTORSTATE_Idle}
			{description: Not ready to connect, no devices.}
	}
	{value: {name: VIDYO_CONNECTORSTATE_Ready}
			{description: Ready to connect.}
	}
	{value: {name: VIDYO_CONNECTORSTATE_EstablishingConnection}
			{description: Establishing a connection to the server.}
	}
	{value: {name: VIDYO_CONNECTORSTATE_FindingResource}
			{description: Finding a resource.}
	}
	{value: {name: VIDYO_CONNECTORSTATE_ConnectingToResource}
			{description: Connecting to a resource.}
	}
	{value: {name: VIDYO_CONNECTORSTATE_EnablingMedia}
			{description: Enabling media.}
	}
	{value: {name: VIDYO_CONNECTORSTATE_Connected}
			{description: Successfully connected.}
	}
	{value: {name: VIDYO_CONNECTORSTATE_Disconnecting}
			{description: Disconnecting.}
	}
	{value: {name: VIDYO_CONNECTORSTATE_Disabling}
			{description: Disabling.}
	}
	{value: {name: VIDYO_CONNECTORSTATE_Dead}
			{description: VidyoConnector can't be used any more and must be destructed.}
	}
}
*/
typedef enum
{
	VIDYO_CONNECTORSTATE_Idle,
	VIDYO_CONNECTORSTATE_Ready,
	VIDYO_CONNECTORSTATE_EstablishingConnection,
	VIDYO_CONNECTORSTATE_FindingResource,
	VIDYO_CONNECTORSTATE_ConnectingToResource,
	VIDYO_CONNECTORSTATE_EnablingMedia,
	VIDYO_CONNECTORSTATE_Connected,
	VIDYO_CONNECTORSTATE_Disconnecting,
	VIDYO_CONNECTORSTATE_Disabling,
	VIDYO_CONNECTORSTATE_Dead
} VidyoConnectorState;

/**
 {type apigroup="connector" visibility="public":
	{name: VidyoConnectorMode}
	{parent: VidyoConnector}
	{description: Operating modes of a VidyoConnector object.}
	{value: {name: VIDYO_CONNECTORMODE_Default}
		{description: The default operating mode.}}
	{value: {name: VIDYO_CONNECTORMODE_Foreground}
		{description: Runs in the foreground.}}
	{value: {name: VIDYO_CONNECTORMODE_Background}
		{description: Runs in the background.}}
	{note: The default is currently VIDYO_CONNECTORMODE_Foreground.}
}
*/
typedef enum
{
	VIDYO_CONNECTORMODE_Default,
	VIDYO_CONNECTORMODE_Foreground,
	VIDYO_CONNECTORMODE_Background
} VidyoConnectorMode;

/**
 {type apigroup="connector" visibility="public":
	{name: VidyoConnectorViewStyle}
	{parent: VidyoConnector}
	{description: Styles of the composite renderer, which apply to its layout and behaviour.}
	{value: {name: VIDYO_CONNECTORVIEWSTYLE_Default}
		{description: The default style.}}
    {value: {name: VIDYO_CONNECTORVIEWSTYLE_Tiles}
        {description: Participants are shown in sizes that best reflect their order of importance.}}
    {value: {name: VIDYO_CONNECTORVIEWSTYLE_NGRGrid}
        {description: Next generation renderer with grid layout.}}
    {value: {name: VIDYO_CONNECTORVIEWSTYLE_NGRSpeaker}
        {description: Next generation renderer with speaker layout.}}
    {note: The default is VIDYO_CONNECTORVIEWSTYLE_Tiles for all platforms. Currently, iOS supports VIDYO_CONNECTORVIEWSTYLE_NGRGrid & VIDYO_CONNECTORVIEWSTYLE_NGRSpeaker layout, If used in other platforms, no action will be taken and LMI_FALSE will be return. The transition between the existing view style i.e. default/tiles, and the new generation view style, as well as the reverse, is not supported for the entire lifecycle of the connector object.}
}
*/
typedef enum
{
	VIDYO_CONNECTORVIEWSTYLE_Default,
	VIDYO_CONNECTORVIEWSTYLE_Tiles,
    VIDYO_CONNECTORVIEWSTYLE_NGRGrid,
    VIDYO_CONNECTORVIEWSTYLE_NGRSpeaker,
} VidyoConnectorViewStyle;

/**
 {type apigroup="connector" visibility="public":
	{name: VidyoConnectorTradeOffProfile}
	{parent: VidyoConnector}
	{description: CPU tradeoff profile. }
	{value: {name: VIDYO_CONNECTORTRADEOFFPROFILE_High}
		{description: High CPU usage.}}
	{value: {name: VIDYO_CONNECTORTRADEOFFPROFILE_Medium}
		{description: Balanced CPU usage.}}
	{value: {name: VIDYO_CONNECTORTRADEOFFPROFILE_Low}
		{description: Low CPU usage.}}
	{note: The default is currently VIDYO_CONNECTORTRADEOFFPROFILE_Medium.}
 }
 */
typedef enum
{
	VIDYO_CONNECTORTRADEOFFPROFILE_High,
	VIDYO_CONNECTORTRADEOFFPROFILE_Medium,
	VIDYO_CONNECTORTRADEOFFPROFILE_Low
} VidyoConnectorTradeOffProfile;

/**
 {type apigroup="connector" visibility="public":
 	{name: VidyoConnectorBaseTransportType}
 	{parent: VidyoConnector}
 	{description: Base transport type. }
 	{value: {name: VIDYO_CONNECTORBASETRANSPORTTYPE_TLS}
 		{description: Use TLS as base transport.}}
 	{value: {name: VIDYO_CONNECTORBASETRANSPORTTYPE_TCP}
 		{description: Use TCP as base transport.}}
 	{value: {name: VIDYO_CONNECTORBASETRANSPORTTYPE_UDP}
 		{description: Use UDP as base transport.}}
 }
 */
typedef enum
{
	VIDYO_CONNECTORBASETRANSPORTTYPE_TLS,
	VIDYO_CONNECTORBASETRANSPORTTYPE_TCP,
	VIDYO_CONNECTORBASETRANSPORTTYPE_UDP
} VidyoConnectorBaseTransportType;

/**
{type apigroup="connector" visibility="public":
	{name: VidyoConnectorModerationResult}
	{parent: VidyoConnector}
	{description: This type indicates the result of moderation request.}
	{value:
		{name: VIDYO_CONNECTORMODERATIONRESULT_OK}
		{description: Request was successful.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONRESULT_NoResponse}
		{description: The server did not respond in a reasonable amount of time for request.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONRESULT_Unauthorized}
		{description: The user did not have permission for request.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONRESULT_NotOwnerOfRoom}
		{description: The user is not owner of the room.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONRESULT_NotAcceptable}
		{description: Request is not acceptable.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONRESULT_NotAllowed}
		{description: Request is not allowed.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONRESULT_Conflict}
		{description: Request has a conflict.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONRESULT_InvalidInput}
		{description: Request parameter is invalid.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONRESULT_OutOfResources}
		{description: The request cannot completed at the current time because it is out of some resource.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONRESULT_UserIsOffline}
		{description: Requested user is offline.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONRESULT_RoomFull}
		{description: Room is currently full.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONRESULT_MiscLocalError}
		{description: Request failed due to some miscellaneous problem with the request.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONRESULT_MiscRemoteError}
		{description: The server is rejecting the request due to some miscellaneous problem of its own.}
	}
}
*/
typedef enum
{
	VIDYO_CONNECTORMODERATIONRESULT_OK,
	VIDYO_CONNECTORMODERATIONRESULT_NoResponse,
	VIDYO_CONNECTORMODERATIONRESULT_Unauthorized,
	VIDYO_CONNECTORMODERATIONRESULT_NotOwnerOfRoom,
	VIDYO_CONNECTORMODERATIONRESULT_NotAcceptable,
	VIDYO_CONNECTORMODERATIONRESULT_NotAllowed,
	VIDYO_CONNECTORMODERATIONRESULT_Conflict,
	VIDYO_CONNECTORMODERATIONRESULT_InvalidInput,
	VIDYO_CONNECTORMODERATIONRESULT_OutOfResources,
	VIDYO_CONNECTORMODERATIONRESULT_UserIsOffline,
	VIDYO_CONNECTORMODERATIONRESULT_RoomFull,
	VIDYO_CONNECTORMODERATIONRESULT_MiscLocalError,
	VIDYO_CONNECTORMODERATIONRESULT_MiscRemoteError
} VidyoConnectorModerationResult;

/**
{type apigroup="simple" visibility="public":
	{name: VidyoConnectorConferenceMode}
	{parent: VidyoConnector}
	{description: Sets the type of conference mode.}
	{value: {name: VIDYO_CONNECTORCONFERENCEMODE_LOBBY} {description: Indicate the user has joined the room which has Lobby Room property. The user is waiting in lobby because the host or presenter has not joined yet }}
	{value: {name: VIDYO_CONNECTORCONFERENCEMODE_LECTURE} {description: Indicate the user has joined the room which has Lecture Mode property. The lecture mode features will be enforced on this room. }}
	{value: {name: VIDYO_CONNECTORCONFERENCEMODE_GROUP} {description:Indicate the user is in room with property Group mode, which is a regular conference room with no special property }}
}
*/
typedef enum
{
	VIDYO_CONNECTORCONFERENCEMODE_LOBBY,
	VIDYO_CONNECTORCONFERENCEMODE_LECTURE,
	VIDYO_CONNECTORCONFERENCEMODE_GROUP
} VidyoConnectorConferenceMode;

/**
{type visibility="public":
	{name: VidyoConnectorConferenceHostState}
	{parent: VidyoConnector}
	{description: Tell us the joining status of the host or owner of the room hosting the conference.}
	{value: {name: VIDYO_CONNECTORCONFERENCEHOSTSTATE_JOINED} {description: Tells us if the host has joined}}
	{value: {name: VIDYO_CONNECTORCONFERENCEHOSTSTATE_LEFT} {description: Tells us if the host has left the conference}}
}
*/
typedef enum
{
	VIDYO_CONNECTORCONFERENCEHOSTSTATE_JOINED,
	VIDYO_CONNECTORCONFERENCEHOSTSTATE_LEFT
} VidyoConnectorConferenceHostState;

/**
 {type apigroup="connector" visibility="public":
	{name: VidyoConnectorRecordingServiceResult}
	{parent: VidyoConnector}
	{description: Results of recording service requests.}
	{value: {name: VIDYO_CONNECTORRECORDINGSERVICERESULT_Success}
		{description: The request has succeeded.}}
	{value: {name: VIDYO_CONNECTORRECORDINGSERVICERESULT_InvalidArgument}
		{description: One or more input arguments are invalid.}}
	{value: {name: VIDYO_CONNECTORRECORDINGSERVICERESULT_GeneralFailure}
		{description: Media support was disabled or there is an unknown fault on the portal.}}
	{value: {name: VIDYO_CONNECTORRECORDINGSERVICERESULT_SeatLicenseExpired}
		{description: The seat license has expired.}}
	{value: {name: VIDYO_CONNECTORRECORDINGSERVICERESULT_NotLicensed}
		{description: The endpoint is not licensed.}}
	{value: {name: VIDYO_CONNECTORRECORDINGSERVICERESULT_ResourceNotAvailable}
		{description: There are no available resources of the portal or recording servers.}}
	{value: {name: VIDYO_CONNECTORRECORDINGSERVICERESULT_ControlMeetingFailure}
		{description: Miscellaneous portal error occurred.}}
}
*/
typedef enum
{
	VIDYO_CONNECTORRECORDINGSERVICERESULT_Success,
	VIDYO_CONNECTORRECORDINGSERVICERESULT_InvalidArgument,
	VIDYO_CONNECTORRECORDINGSERVICERESULT_GeneralFailure,
	VIDYO_CONNECTORRECORDINGSERVICERESULT_SeatLicenseExpired,
	VIDYO_CONNECTORRECORDINGSERVICERESULT_NotLicensed,
	VIDYO_CONNECTORRECORDINGSERVICERESULT_ResourceNotAvailable,
	VIDYO_CONNECTORRECORDINGSERVICERESULT_ControlMeetingFailure
} VidyoConnectorRecordingServiceResult;

/**
{type apigroup="connector" visibility="public":
	{name: VidyoConnectorSearchResult}
	{parent: VidyoConnector}
	{description: Provide the result of user search request.}
	{value:
		{name: VIDYO_CONNECTORSEARCHRESULT_Ok} 
		{description: The user search concluded properly.}}
	{value:
		{name: VIDYO_CONNECTORSEARCHRESULT_NoRecords}
		{description: No record found for user search request.}}
	{value:
		{name: VIDYO_CONNECTORSEARCHRESULT_NoResponse}
		{description: No response from server for user search request.}}
	{value:
		{name: VIDYO_CONNECTORSEARCHRESULT_MiscLocalError}
		{description: Not able to search for some other miscellaneous local error.}}
	{value:
		{name: VIDYO_CONNECTORSEARCHRESULT_MiscRemoteError}
		{description: Not able to search for some other miscellaneous remote error.}}
}
*/
typedef enum
{
	VIDYO_CONNECTORSEARCHRESULT_Ok,
	VIDYO_CONNECTORSEARCHRESULT_NoRecords,
	VIDYO_CONNECTORSEARCHRESULT_NoResponse,
	VIDYO_CONNECTORSEARCHRESULT_MiscLocalError,
	VIDYO_CONNECTORSEARCHRESULT_MiscRemoteError
} VidyoConnectorSearchResult;

/**
{type apigroup="connector" visibility="public":
	{name: VidyoConnectorErrorCode}
	{parent: VidyoConnector}
	{description: Provide the advanced error code to the application.}
	{value:
		{name: VIDYO_CONNECTORERRORCODE_NotAllowed} 
		{description: Not allowed.}}
	{value:
		{name: VIDYO_CONNECTORERRORCODE_WrongState}
		{description: Called in wrong state.}}
	{value:
		{name: VIDYO_CONNECTORERRORCODE_InvalidInput}
		{description: Invalid input.}}
	{value:
		{name: VIDYO_CONNECTORERRORCODE_Misc}
		{description: Miscellaneous remote error.}}
}
*/
typedef enum
{
	VIDYO_CONNECTORERRORCODE_NotAllowed,
	VIDYO_CONNECTORERRORCODE_WrongState,
	VIDYO_CONNECTORERRORCODE_InvalidInput,
	VIDYO_CONNECTORERRORCODE_Misc
} VidyoConnectorErrorCode;

/**
{type apigroup="connector" visibility="public":
	{name: VidyoConnectorRecordingState}
	{parent: VidyoConnector}
	{description: Provide the status of recording.}
	{value:
		{name: VIDYO_CONNECTORRECORDINGSTATE_NotRecording} 
		{description: Recording is in stop state.}}
	{value:
		{name: VIDYO_CONNECTORRECORDINGSTATE_RecordingPaused}
		{description: Recording is in pause state.}}
	{value:
		{name: VIDYO_CONNECTORRECORDINGSTATE_Recording}
		{description: Recording is in start/resume state.}}
}
*/
typedef enum
{
	VIDYO_CONNECTORRECORDINGSTATE_NotRecording,
	VIDYO_CONNECTORRECORDINGSTATE_RecordingPaused,
	VIDYO_CONNECTORRECORDINGSTATE_Recording
} VidyoConnectorRecordingState;

/**
{type apigroup="connector" visibility="public":
	{name: VidyoConnectorWebcastingState}
	{parent: VidyoConnector}
	{description: Provide the status of webcasting/broadcasting.}
	{value:
		{name: VIDYO_CONNECTORWEBCASTINGSTATE_NotWebcasting} 
		{description: Room is not broadcast or webcasting is not going on.}}
	{value:
		{name: VIDYO_CONNECTORWEBCASTINGSTATE_Webcasting}
		{description: Room is being broadcast or webcasting is going on.}}
}
*/
typedef enum
{
	VIDYO_CONNECTORWEBCASTINGSTATE_NotWebcasting,
	VIDYO_CONNECTORWEBCASTINGSTATE_Webcasting
} VidyoConnectorWebcastingState;

/**
{type apigroup="connector" visibility="public":
	{name:  VidyoConnectorConnectionProperties}
	{parent: Connector}
	{description: Represents the connection properties structure, which are based on information from the portal.}
	{member:
		{name: roomName}
		{type: LmiString}
		{description: Name of the room.}
	}
	{member:
		{name: recordingState}
		{type: VidyoConnectorRecordingState}
		{description: State of Recording. See enum VidyoConnectorRecordingState.}
	}
	{member:
		{name: webcastingState}
		{type: VidyoConnectorWebcastingState}
		{description: State of Webcasting. See enum VidyoConnectorWebcastingState.}
	}
	{member:
		{name: isRoomLocked}
		{type: LmiBool}
		{description: Return LMI_TRUE if room is lock otherwise LMI_FALSE.}
	}
	{member:
		{name: hasModeratorPin}
		{type: LmiBool}
		{description: Return LMI_TRUE if moderation pin is set otherwise LMI_FALSE.}
	}
	{member:
		{name: hasRoomPin}
		{type: LmiBool}
		{description: Return LMI_TRUE if room pin is set otherwise LMI_FALSE.}
	}
	{member:
		{name: isVp9Enabled}
		{type: LmiBool}
		{description: Return LMI_TRUE if VP9 is enabled otherwise LMI_FALSE.}
	}
	{member:
		{name: minimumRoomPinLength}
		{type: LmiInt}
		{description: Return the minimum room pin length if minimumRoomPinLength is set by default.}
	}
	{member:
		{name: maximumRoomPinLength}
		{type: LmiInt}
		{description: Return the maximum room pin length if maximumRoomPinLength is set by default.}
	}
	{member:
		{name: maxSendBandWidth}
		{type: LmiUint}
		{description: Represents the maximum send bandwidth.}
	}
	{member:
		{name: maxReceiveBandWidth}
		{type: LmiUint}
		{description: Represents the maximum receive bandwidth.}
	}
}
*/
typedef struct
{
	LmiString roomName;
	VidyoConnectorRecordingState recordingState;
	VidyoConnectorWebcastingState webcastingState;
	LmiBool isRoomLocked;
	LmiBool hasModeratorPin;
	LmiBool hasRoomPin;
	LmiBool isVp9Enabled;
	LmiInt minimumRoomPinLength;
	LmiInt maximumRoomPinLength;
	LmiUint maxSendBandWidth;
	LmiUint maxReceiveBandWidth;
} VidyoConnectorConnectionProperties;

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLog}
	{parent: VidyoConnector}
	{description: Callback that is triggered for every log record. }
	{prototype: void (*VidyoConnectorOnLog)(VidyoConnector* c, const VidyoLogRecord* logRecord)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: logRecord}
		{description: The VidyoLogRecord object containing log record data.}
	}
}
*/
typedef void(*VidyoConnectorOnLog)(VidyoConnector* c, const VidyoLogRecord* logRecord);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalCameraAdded}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local camera becomes available to a VidyoConnector.}
	{prototype: void(*VidyoConnectorOnLocalCameraAdded)(VidyoConnector* c, VidyoLocalCamera* localCamera)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localCamera}
		{description: Local camera that was added to the VidyoConnector.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalCameraAdded)(VidyoConnector* c, VidyoLocalCamera* localCamera);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalCameraRemoved}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local camera is no longer available to a VidyoConnector.}
	{prototype: void(*VidyoConnectorOnLocalCameraRemoved)(VidyoConnector* c, VidyoLocalCamera* localCamera)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localCamera}
		{description: Local camera that was removed from the VidyoConnector.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalCameraRemoved)(VidyoConnector* c, VidyoLocalCamera* localCamera);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalCameraSelected}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time there is a change in which local camera is selected for use.}
	{prototype: void(*VidyoConnectorOnLocalCameraSelected)(VidyoConnector* c, VidyoLocalCamera* localCamera)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localCamera}
		{description: Local camera that was selected. NULL if none of them remain selected.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalCameraSelected)(VidyoConnector* c, VidyoLocalCamera* localCamera);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalCameraStateUpdated}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time there is a change in the state of the camera such as being suspended or disabled.}
	{prototype: void(*VidyoConnectorOnLocalCameraStateUpdated)(VidyoConnector* c, VidyoLocalCamera* localCamera, VidyoDeviceState state)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localCamera}
		{description: Local camera that was selected or NULL if nothing was selected.}
	}
	{parameter:
		{name: state}
		{description: New state of the Local camera.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalCameraStateUpdated)(VidyoConnector* c, VidyoLocalCamera* localCamera, VidyoDeviceState state);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalMicrophoneAdded}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local microphone becomes available to a VidyoConnector.}
	{prototype: void(*VidyoConnectorOnLocalMicrophoneAdded)(VidyoConnector* c, VidyoLocalMicrophone* localMicrophone)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localMicrophone}
		{description: Local microphone that was added to the VidyoConnector.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalMicrophoneAdded)(VidyoConnector* c, VidyoLocalMicrophone* localMicrophone);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalMicrophoneRemoved}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local microphone is no longer available to a VidyoConnector.}
	{prototype: void(*VidyoConnectorOnLocalMicrophoneRemoved)(VidyoConnector* c, VidyoLocalMicrophone* localMicrophone)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localMicrophone}
		{description: Local microphone that was removed from the VidyoConnector.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalMicrophoneRemoved)(VidyoConnector* c, VidyoLocalMicrophone* localMicrophone);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalMicrophoneSelected}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time there is a change in which local microphone is selected for use.}
	{prototype: void(*VidyoConnectorOnLocalMicrophoneSelected)(VidyoConnector* c, VidyoLocalMicrophone* localMicrophone)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localMicrophone}
		{description: Local microphone that was selected. NULL if none of them remain selected.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalMicrophoneSelected)(VidyoConnector* c, VidyoLocalMicrophone* localMicrophone);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalMicrophoneStateUpdated}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time there is a change in the state of the microphone such as being suspended or disabled.}
	{prototype: void(*VidyoConnectorOnLocalMicrophoneStateUpdated)(VidyoConnector* c, VidyoLocalMicrophone* localMicrophone, VidyoDeviceState state)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localMicrophone}
		{description: Local microphone that was selected or NULL if nothing was selected.}
	}
	{parameter:
		{name: state}
		{description: New state of the Local microphone.
			If microphone is muted locally by the local user using application or hard/soft muted by moderator, microphone will be paused instead of stopped.
			So, VIDYO_DEVICESTATE_Paused will be reported and VIDYO_DEVICESTATE_Resumed when unmuted.
			VIDYO_DEVICESTATE_Stopped will be reported when device is unselected and VIDYO_DEVICESTATE_Started when device is selected.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalMicrophoneStateUpdated)(VidyoConnector* c, VidyoLocalMicrophone* localMicrophone, VidyoDeviceState state);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalSpeakerAdded}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local speaker becomes available to a VidyoConnector.}
	{prototype: void(*VidyoConnectorOnLocalSpeakerAdded)(VidyoConnector* c, VidyoLocalSpeaker* localSpeaker)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localSpeaker}
		{description: Local speaker that was added to the VidyoConnector.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalSpeakerAdded)(VidyoConnector* c, VidyoLocalSpeaker* localSpeaker);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalSpeakerRemoved}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local speaker is no longer available to a VidyoConnector.}
	{prototype: void(*VidyoConnectorOnLocalSpeakerRemoved)(VidyoConnector* c, VidyoLocalSpeaker* localSpeaker)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localSpeaker}
		{description: Local speaker that was removed from the VidyoConnector.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalSpeakerRemoved)(VidyoConnector* c, VidyoLocalSpeaker* localSpeaker);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalSpeakerSelected}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time there is a change in which local speaker is selected for use.}
	{prototype: void(*VidyoConnectorOnLocalSpeakerSelected)(VidyoConnector* c, VidyoLocalSpeaker* localSpeaker)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localSpeaker}
		{description: Local speaker that was selected. NULL if none of them remain selected.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalSpeakerSelected)(VidyoConnector* c, VidyoLocalSpeaker* localSpeaker);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalSpeakerStateUpdated}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time there is a change in the state of the speaker such as being suspended or disabled.}
	{prototype: void(*VidyoConnectorOnLocalSpeakerStateUpdated)(VidyoConnector* c, VidyoLocalSpeaker* localSpeaker, VidyoDeviceState state)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localSpeaker}
		{description: Local speaker that was selected or NULL if nothing was selected.}
	}
	{parameter:
		{name: state}
		{description: New state of the Local speaker.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalSpeakerStateUpdated)(VidyoConnector* c, VidyoLocalSpeaker* localSpeaker, VidyoDeviceState state);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalMonitorAdded}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local monitor becomes available to a VidyoConnector.}
	{prototype: void(*VidyoConnectorOnLocalMonitorAdded)(VidyoConnector* c, VidyoLocalMonitor* localMonitor)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localMonitor}
		{description: Local monitor that was added to the VidyoConnector.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalMonitorAdded)(VidyoConnector* c, VidyoLocalMonitor* localMonitor);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalMonitorRemoved}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local monitor is no longer available to a VidyoConnector.}
	{prototype: void(*VidyoConnectorOnLocalMonitorRemoved)(VidyoConnector* c, VidyoLocalMonitor* localMonitor)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localMonitor}
		{description: Local monitor that was removed from the VidyoConnector.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalMonitorRemoved)(VidyoConnector* c, VidyoLocalMonitor* localMonitor);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalMonitorSelected}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time there is a change in which local monitor is selected for sharing.}
	{prototype: void(*VidyoConnectorOnLocalMonitorSelected)(VidyoConnector* c, VidyoLocalMonitor* localMonitor)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localMonitor}
		{description: Local monitor that was selected. NULL if none of them remain selected.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalMonitorSelected)(VidyoConnector* c, VidyoLocalMonitor* localMonitor);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalMonitorStateUpdated}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time there is a change in the state of the monitor such as being suspended or disabled.}
	{prototype: void(*VidyoConnectorOnLocalMonitorStateUpdated)(VidyoConnector* c, VidyoLocalMonitor* localMonitor, VidyoDeviceState state)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localMonitor}
		{description: Local monitor that was selected or NULL if nothing was selected. }
	}
	{parameter:
		{name: state}
		{description: New state of the Local monitor.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalMonitorStateUpdated)(VidyoConnector* c, VidyoLocalMonitor* localMonitor, VidyoDeviceState state);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalWindowShareAdded}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local window becomes available for a VidyoConnector to share in a conference.}
	{prototype: void(*VidyoConnectorOnLocalWindowShareAdded)(VidyoConnector* c, VidyoLocalWindowShare* localWindowShare)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localWindowShare}
		{description: Local, shareable window that was added to the VidyoConnector.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalWindowShareAdded)(VidyoConnector* c, VidyoLocalWindowShare* localWindowShare);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalWindowShareRemoved}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local window is no longer available for a VidyoConnector to share in a conference.}
	{prototype: void(*VidyoConnectorOnLocalWindowShareRemoved)(VidyoConnector* c, VidyoLocalWindowShare* localWindowShare)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localWindowShare}
		{description: Local window that was removed from the VidyoConnector. The window is no longer shareable.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalWindowShareRemoved)(VidyoConnector* c, VidyoLocalWindowShare* localWindowShare);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalWindowShareSelected}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time there is a change in which local window is selected for sharing.}
	{prototype: void(*VidyoConnectorOnLocalWindowShareSelected)(VidyoConnector* c, VidyoLocalWindowShare* localWindowShare)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localWindowShare}
		{description: Local window that was selected. NULL if none of them remain selected.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalWindowShareSelected)(VidyoConnector* c, VidyoLocalWindowShare* localWindowShare);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLocalWindowShareStateUpdated}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time there is a change in the state of the window such as being suspended or disabled.}
	{prototype: void(*VidyoConnectorOnLocalWindowShareStateUpdated)(VidyoConnector* c, VidyoLocalWindowShare* localWindowShare, VidyoDeviceState state)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localWindowShare}
		{description: Local window that was selected or NULL if nothing was selected.}
	}
	{parameter:
		{name: state}
		{description: New state of the Local window.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalWindowShareStateUpdated)(VidyoConnector* c, VidyoLocalWindowShare* localWindowShare, VidyoDeviceState state);

/**
{callback apigroup="connector":
    {name: VidyoConnectorOnVirtualVideoSourceAdded}
    {parent: VidyoConnector}
    {description: Callback that is triggered each time a virtual video source becomes available for a VidyoConnector to add into a conference.}
    {prototype: void(*VidyoConnectorOnVirtualVideoSourceAdded)(VidyoConnector* c, VidyoVirtualVideoSource* virtualVideoSource)}
    {parameter:
        {name: c}
        {description: The VidyoConnector that triggered the callback.}
    }
    {parameter:
        {name: virtualVideoSource}
        {description: The VidyoVirtualVideoSource object that was added to the VidyoConnector.}
    }
}
*/
typedef void(*VidyoConnectorOnVirtualVideoSourceAdded)(VidyoConnector* c, VidyoVirtualVideoSource* virtualVideoSource);

/**
{callback apigroup="connector":
    {name: VidyoConnectorOnVirtualVideoSourceRemoved}
    {parent: VidyoConnector}
    {description: Callback that is triggered each time a virtual video source is no longer available for a VidyoConnector to add into a conference.}
    {prototype: void(*VidyoConnectorOnVirtualVideoSourceRemoved)(VidyoConnector* c, VidyoVirtualVideoSource* virtualVideoSource)}
    {parameter:
        {name: c}
        {description: The VidyoConnector that triggered the callback.}
    }
    {parameter:
        {name: virtualVideoSource}
        {description: The VidyoVirtualVideoSource object that was removed from the VidyoConnector.}
    }
}
*/
typedef void(*VidyoConnectorOnVirtualVideoSourceRemoved)(VidyoConnector* c, VidyoVirtualVideoSource* virtualVideoSource);

/**
{callback apigroup="connector":
    {name: VidyoConnectorOnVirtualVideoSourceStateUpdated}
    {parent: VidyoConnector}
    {description: Callback that is triggered each time there is a change in the state of the window such as being suspended or disabled.}
    {prototype: void(*VidyoConnectorOnVirtualVideoSourceStateUpdated)(VidyoConnector* c, VidyoVirtualVideoSource* virtualVideoSource, VidyoDeviceState state)}
    {parameter:
        {name: c}
        {description: The VidyoConnector that triggered the callback.}
    }
    {parameter:
        {name: virtualVideoSource}
        {description: Local window that was selected or NULL if nothing was selected.}
    }
    {parameter:
        {name: state}
        {description: New state of the Local window.}
    }
}
*/
typedef void(*VidyoConnectorOnVirtualVideoSourceStateUpdated)(VidyoConnector* c, VidyoVirtualVideoSource* virtualVideoSource, VidyoDeviceState state);

/**
{callback apigroup="connector":
    {name: VidyoConnectorOnVirtualVideoSourceExternalMediaBufferReleased}
    {parent: VidyoConnector}
    {description: Callback that is triggered to notify application that VidyoClient is done with the buffer that holds raw video frame. This buffer is the same one that was passed to VidyoClient via function VidyoVirtualVideoSourceSendFrameWithExternalData.}
    {prototype: void(*VidyoConnectorOnVirtualVideoSourceExternalMediaBufferReleased)(VidyoConnector* c, VidyoVirtualVideoSource* virtualVideoSource, LmiUint8* buffer, LmiSizeT size)}
    {parameter:
        {name: c}
        {description: The VidyoConnector that triggered the callback.}
    }
    {parameter:
        {name: virtualVideoSource}
        {description: The VidyoVirtualVideoSource object that is associated with this callback.}
    }
    {parameter:
        {name: buffer}
        {description: The buffer (allocated by application that holds raw data of a video frame) that is no longer being used by VidyoClient.}
    }
    {parameter:
        {name: size}
        {description: The size of the buffer.}
    }
}
*/
typedef void(*VidyoConnectorOnVirtualVideoSourceExternalMediaBufferReleased)(VidyoConnector* c, VidyoVirtualVideoSource* virtualVideoSource, LmiUint8* buffer, LmiSizeT size);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnVirtualAudioSourceAdded}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a virtual audio source becomes available for a VidyoConnector to add into a conference.}
	{prototype: void(*VidyoConnectorOnVirtualAudioSourceAdded)(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource)}
	{parameter: {name: c} {description: The VidyoConnector that triggered the callback.}}
	{parameter: {name: virtualAudioSource} {description: The VidyoVirtualAudioSource object that was created and now available for a VidyoConnector to add into a conference.}}
}
*/
typedef void(*VidyoConnectorOnVirtualAudioSourceAdded)(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnVirtualAudioSourceRemoved}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a virtual audio source is no longer available for a VidyoConnector to add into a conference.}
	{prototype: void(*VidyoConnectorOnVirtualAudioSourceRemoved)(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource)}
	{parameter: {name: c} {description: The VidyoConnector that triggered the callback.}}
	{parameter: {name: virtualAudioSource} {description: The VidyoVirtualAudioSource object that was removed from the VidyoConnector.}}
}
*/
typedef void(*VidyoConnectorOnVirtualAudioSourceRemoved)(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnVirtualAudioSourceSelected}
	{parent: VidyoConnector}
	{description: Callback that is triggered when specific virtual audio source selected.}
	{prototype: void(*VidyoConnectorOnVirtualAudioSourceSelected)(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource)}
	{parameter: {name: c} {description: The VidyoConnector that triggered the callback.}}
	{parameter: {name: virtualAudioSource} {description: The virtual audio source object..}}
}
*/
typedef void(*VidyoConnectorOnVirtualAudioSourceSelected)(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnVirtualAudioSourceStateUpdated}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time there is a change in the state of the window such as being suspended or disabled.}
	{prototype: void(*VidyoConnectorOnVirtualAudioSourceStateUpdated)(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource, VidyoDeviceState state)}
	{parameter: {name: c} {description: The VidyoConnector that triggered the callback.}}
	{parameter: {name: virtualAudioSource} {description: The virtual audio source object..}}
	{parameter: {name: state} {description: New state of the virtual audio source.}}
}
*/
typedef void(*VidyoConnectorOnVirtualAudioSourceStateUpdated)(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource, VidyoDeviceState state);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnVirtualAudioSourceExternalMediaBufferReleased}
	{parent: VidyoConnector}
	{description: Callback that is triggered to notify application that VidyoClient is done with the buffer that holds raw audio frame. This buffer is the same one that was passed to VidyoClient via function VidyoVirtualAudioSourceSendFrameWithExternalData.}
	{prototype: void(*VidyoConnectorOnVirtualAudioSourceExternalMediaBufferReleased)(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource, LmiUint8* buffer, LmiSizeT size)}
	{parameter: {name: c} {description: The VidyoConnector that triggered the callback.}}
	{parameter: {name: virtualAudioSource} {description: The VidyoVirtualAudioSource object.}}
	{parameter: {name: buffer} {description: The buffer (allocated by application that holds raw data of the audio  frame) that is no longer being used by VidyoClient.}}
	{parameter: {name: size} {description: The size of the buffer.}}
}
*/
typedef void(*VidyoConnectorOnVirtualAudioSourceExternalMediaBufferReleased)(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource, LmiUint8* buffer, LmiSizeT size);

/**
{function apigroup="connector":
	{name: VidyoConnectorRegisterVirtualAudioSourceEventListener}
	{parent: VidyoConnector}
	{description: Registers to get notified about virtual audio source events. This is a synchronous interface. This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterVirtualAudioSourceEventListener(VidyoConnector* c, VidyoConnectorOnVirtualAudioSourceAdded onAdded, VidyoConnectorOnVirtualAudioSourceRemoved onRemoved, VidyoConnectorOnVirtualAudioSourceStateUpdated onStateUpdated, VidyoConnectorOnVirtualAudioSourceSelected onSelected, VidyoConnectorOnVirtualAudioSourceExternalMediaBufferReleased onExternalMediaBufferReleased)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: onAdded} {description: Callback that is triggered each time when virtual audio source is created.}}
	{parameter: {name: onRemoved} {description: Callback that is triggered each time when virtual audio source is removed.}}
	{parameter: {name: onStateUpdated} {description: Callback that is triggered each time when virtual audio source chagnes state.}}
	{parameter: {name: onSelected} {description: Callback that is triggered each time when virtual audio source is selected.}}
	{parameter: {name: onExternalMediaBufferReleased} {description: Callback that is triggered each time when VidyoClient is done with an application allocated buffer that holds raw audio frame data.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterVirtualAudioSourceEventListener(VidyoConnector* c, VidyoConnectorOnVirtualAudioSourceAdded onAdded, VidyoConnectorOnVirtualAudioSourceRemoved onRemoved, VidyoConnectorOnVirtualAudioSourceStateUpdated onStateUpdated, VidyoConnectorOnVirtualAudioSourceSelected onSelected, VidyoConnectorOnVirtualAudioSourceExternalMediaBufferReleased onExternalMediaBufferReleased);

/**
{function apigroup="connector":
	{name: VidyoConnectorUnregisterVirtualAudioSourceEventListener}
	{parent: VidyoConnector}
	{description: Unregisters virtual audio source event notifications. This is a synchronous interface. This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterVirtualAudioSourceEventListener(VidyoConnector* c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterVirtualAudioSourceEventListener(VidyoConnector* c);

/**
{function apigroup="connector":
	{name: VidyoConnectorCreateVirtualAudioSource}
	{parent: VidyoConnector}
	{description: Create virtual audio source. This is an asynchronous interface. This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE. The VidyoConnectorOnVirtualAudioSourceAdded will trigger.}
	{prototype: LmiBool VidyoConnectorCreateVirtualAudioSource(VidyoConnector* c, const LmiString *name, const LmiString *id, const LmiString *configuration)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: name} {description: The name for the virtual audio source.}}
	{parameter: {name: id} {description: The id for the virtual audio source.}}
	{parameter: {name: configuration} {description: The string in JSON format that contain audio configuration that will be used for virtual audio source.
		{p: format - Specifies the media format. Possible values are: fl32 or in16. The default is fl32.}
		{p: Example: \{\'format\' : \'in16\'\}.}
		{p: sampleRate - Specifies the audio sample rate. Possible values are: 48000, 32000 and 16000. The default is 32000.}
		{p: Example: \{\'sampleRate\' : 48000\}.}
		{p: numberOfChannels - Specifies the number of channels. Possible values are: 1 or 2. The default is 1.}
		{p: Example: \{\'numberOfChannels\' : 2\}.}
		{p: frameInterval - Specifies the frame interval in ms. Possible values are: 20 or 40. The default is 20.}
		{p: Example: \{\'frameInterval\' : 40\}.}
	}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorCreateVirtualAudioSource(VidyoConnector* c, const LmiString *name, const LmiString *id, const LmiString* configuration);

/**
{function apigroup="connector":
	{name: VidyoConnectorDestroyVirtualAudioSource}
	{parent: VidyoConnector}
	{description: Destroy virtual audio source. This is an asynchronous interface. This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE. The VidyoConnectorOnVirtualAudioSourceRemoved will trigger.}
	{prototype: LmiBool VidyoConnectorDestroyVirtualAudioSource(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: virtualAudioSource} {description: The virtual audio source that will be destroyed.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorDestroyVirtualAudioSource(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource);

/**
{function apigroup="connector":
	{name: VidyoConnectorSelectVirtualMicrophone}
	{parent: VidyoConnector}
	{description: Selects the virtual audio source which will be used as microphone. The real miicrophone will be unselected if it's selected. This is a synchronous interface. This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE. The VidyoConnectorOnVirtualAudioSourceStateUpdated will trigger.}
	{prototype: LmiBool VidyoConnectorSelectVirtualMicrophone(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: virtualAudioSource} {description: The virtual audio source that will be used as microphone.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectVirtualMicrophone(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource);

/**
{function apigroup="connector":
	{name: VidyoConnectorSelectVirtualAudioContentShare}
	{parent: VidyoConnector}
	{description: Selects the virtual audio source which will be used for content share. The selected microphone which currently using for content share will be unselected. This is a synchronous interface. This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE. The VidyoConnectorOnVirtualAudioSourceStateUpdated will trigger.}
	{prototype: LmiBool VidyoConnectorSelectVirtualAudioContentShare(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: virtualAudioSource} {description: The virtual audio source that will be used for content share.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectVirtualAudioContentShare(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource);

/**
{function apigroup="connector":
	{name: VidyoConnectorSendVirtualAudioSourceFrame}
	{parent: VidyoConnector}
	{description: Sends already constructed audio frame to selected virtual audio source and then send it to local and/or remote streams. This is a synchronous interface. This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSendVirtualAudioSourceFrame(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource, const VidyoAudioFrame* audioFrame)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: virtualAudioSource} {description: The virtual audio source.}}
	{parameter: {name: audioFrame} {description: The audio frame to be sent.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSendVirtualAudioSourceFrame(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource, const VidyoAudioFrame* audioFrame);

/**
{function apigroup="connector":
	{name: VidyoConnectorSendVirtualAudioSourceFrameWithExternalData}
	{parent: VidyoConnector}
	{description: Pass raw buffer to selected virtual audio source, based on which the audio frame will be constructed and then send it to local and/or remote streams. This is an asynchronous interface. This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE. The VidyoConnectorOnVirtualAudioSourceExternalMediaBufferReleased will trigger when buffer is not needed anymore.}
	{prototype: LmiBool VidyoConnectorSendVirtualAudioSourceFrameWithExternalData(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource, const LmiUint8* buffer, LmiSizeT nbSamples, LmiTime elapsedTime)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: virtualAudioSource} {description: The virtual audio source.}}
	{parameter: {name: buffer} {description: The buffer with audio data to be sent.}}
	{parameter: {name: nbSamples} {description: The number of sample frames that will be represented in the audio data.}}
	{parameter: {name: elapsedTime} {description: The elapsed time.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSendVirtualAudioSourceFrameWithExternalData(VidyoConnector* c, const VidyoVirtualAudioSource* virtualAudioSource, const LmiUint8* buffer, LmiSizeT nbSamples, LmiTime elapsedTime);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRemoteCameraAdded}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time one of the other participants adds a remote camera into a conference.}
	{prototype: void(*VidyoConnectorOnRemoteCameraAdded)(VidyoConnector* c, VidyoRemoteCamera* remoteCamera, const VidyoParticipant* participant)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteCamera}
		{description: Remote camera that was added.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant who owns the remote camera.}
	}
}
*/
typedef void(*VidyoConnectorOnRemoteCameraAdded)(VidyoConnector* c, VidyoRemoteCamera* remoteCamera, const VidyoParticipant* participant);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRemoteCameraRemoved}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time one of the other participants removes a remote camera from a conference.}
	{prototype: void(*VidyoConnectorOnRemoteCameraRemoved)(VidyoConnector* c, VidyoRemoteCamera* remoteCamera, const VidyoParticipant* participant)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteCamera}
		{description: Remote camera that was removed.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant who owns the remote camera.}
	}
}
*/
typedef void(*VidyoConnectorOnRemoteCameraRemoved)(VidyoConnector* c, VidyoRemoteCamera* remoteCamera, const VidyoParticipant* participant);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRemoteCameraStateUpdated}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time one of the other participants remote camera changes state in a conference.}
	{prototype: void(*VidyoConnectorOnRemoteCameraStateUpdated)(VidyoConnector* c, VidyoRemoteCamera* remoteCamera, const VidyoParticipant* participant, VidyoDeviceState state)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteCamera}
		{description: Remote camera that was removed.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant who owns the remote camera.}
	}
	{parameter:
		{name: state}
		{description: The new state of the remote camera.}
	}
}
*/
typedef void(*VidyoConnectorOnRemoteCameraStateUpdated)(VidyoConnector* c, VidyoRemoteCamera* remoteCamera, const VidyoParticipant* participant, VidyoDeviceState state);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRemoteMicrophoneAdded}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time one of the other participants adds a remote microphone into a conference.}
	{prototype: void(*VidyoConnectorOnRemoteMicrophoneAdded)(VidyoConnector* c, VidyoRemoteMicrophone* remoteMicrophone, const VidyoParticipant* participant)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteMicrophone}
		{description: Remote microphone that was added.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant who owns the remote microphone.}
	}
}
*/
typedef void(*VidyoConnectorOnRemoteMicrophoneAdded)(VidyoConnector* c, VidyoRemoteMicrophone* remoteMicrophone, const VidyoParticipant* participant);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRemoteMicrophoneRemoved}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time one of the other participants removes a remote microphone from a conference.}
	{prototype: void(*VidyoConnectorOnRemoteMicrophoneRemoved)(VidyoConnector* c, VidyoRemoteMicrophone* remoteMicrophone, const VidyoParticipant* participant)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteMicrophone}
		{description: Remote microphone that was removed.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant who owns the remote microphone.}
	}
}
*/
typedef void(*VidyoConnectorOnRemoteMicrophoneRemoved)(VidyoConnector* c, VidyoRemoteMicrophone* remoteMicrophone, const VidyoParticipant* participant);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRemoteMicrophoneStateUpdated}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time one of the other participants remote microphone changes state in a conference.}
	{prototype: void(*VidyoConnectorOnRemoteMicrophoneStateUpdated)(VidyoConnector* c, VidyoRemoteMicrophone* remoteMicrophone, const VidyoParticipant* participant, VidyoDeviceState state)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteMicrophone}
		{description: Remote microphone that was removed.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant who owns the remote microphone.}
	}
	{parameter:
		{name: state}
		{description: The new state of the remote microphone.}
	}
}
*/
typedef void(*VidyoConnectorOnRemoteMicrophoneStateUpdated)(VidyoConnector* c, VidyoRemoteMicrophone* remoteMicrophone, const VidyoParticipant* participant, VidyoDeviceState state);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRemoteWindowShareAdded}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time one of the other participants adds, or shares, a remote window into a conference.}
	{prototype: void(*VidyoConnectorOnRemoteWindowShareAdded)(VidyoConnector* c, VidyoRemoteWindowShare* remoteWindowShare, const VidyoParticipant* participant)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteWindowShare}
		{description: Remote window that was added, or shared.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant who owns the remote window.}
	}
}
*/
typedef void(*VidyoConnectorOnRemoteWindowShareAdded)(VidyoConnector* c, VidyoRemoteWindowShare* remoteWindowShare, const VidyoParticipant* participant);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRemoteWindowShareRemoved}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time one of the other participants removes a remote window from a conference, or stops sharing it.}
	{prototype: void(*VidyoConnectorOnRemoteWindowShareRemoved)(VidyoConnector* c, VidyoRemoteWindowShare* remoteWindowShare, const VidyoParticipant* participant)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteWindowShare}
		{description: Remote window that was removed, or no longer shared.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant who owns the remote window.}
	}
}
*/
typedef void(*VidyoConnectorOnRemoteWindowShareRemoved)(VidyoConnector* c, VidyoRemoteWindowShare* remoteWindowShare, const VidyoParticipant* participant);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRemoteWindowShareStateUpdated}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time one of the other participants remote window changes state in a conference.}
	{prototype: void(*VidyoConnectorOnRemoteWindowShareStateUpdated)(VidyoConnector* c, VidyoRemoteWindowShare* remoteWindowShare, const VidyoParticipant* participant, VidyoDeviceState state)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteWindowShare}
		{description: Remote window that was removed, or no longer shared.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant who owns the remote window.}
	}
	{parameter:
		{name: state}
		{description: The new state of the remote window.}
	}
}
*/
typedef void(*VidyoConnectorOnRemoteWindowShareStateUpdated)(VidyoConnector* c, VidyoRemoteWindowShare* remoteWindowShare, const VidyoParticipant* participant, VidyoDeviceState state);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnLocalCameraFrame}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local camera produces a video frame.}
	{prototype: void (*VidyoConnectorOnLocalCameraFrame)(VidyoConnector* c, VidyoLocalCamera* localCamera, const VidyoVideoFrame *videoFrame)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localCamera}
		{description: Local camera that produced the frame.}
	}
	{parameter:
		{name: videoFrame}
		{description: The video frame that contains image data.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalCameraFrame)(VidyoConnector* c, VidyoLocalCamera* localCamera, const VidyoVideoFrame *videoFrame);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnLocalMicrophoneFrame}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local microphone produces an audio frame.}
	{prototype: void (*VidyoConnectorOnLocalMicrophoneFrame)(VidyoConnector* c, VidyoLocalMicrophone* localMicrophone, const VidyoAudioFrame *audioFrame)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localMicrophone}
		{description: Local microphone that produced the frame.}
	}
	{parameter:
		{name: audioFrame}
		{description: The audio frame that contains sound data.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalMicrophoneFrame)(VidyoConnector* c, VidyoLocalMicrophone* localMicrophone, const VidyoAudioFrame *audioFrame);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnLocalMicrophoneEnergy}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local microphone produces an audio frame with energy present.}
	{prototype: void (*VidyoConnectorOnLocalMicrophoneEnergy)(VidyoConnector* c, VidyoLocalMicrophone* localMicrophone, LmiInt audioEnergy)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localMicrophone}
		{description: Local microphone that produced the frame.}
	}
	{parameter:
		{name: audioEnergy}
		{description: Energy of the audio frame in dBFS (full scale).}
	}
	{note: This callback can be used to draw volume bars in the UI.}
}
*/
typedef void(*VidyoConnectorOnLocalMicrophoneEnergy)(VidyoConnector* c, VidyoLocalMicrophone* localMicrophone, LmiInt audioEnergy);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnLocalMonitorFrame}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local camera produces a video frame.}
	{prototype: void (*VidyoConnectorOnLocalMonitorFrame)(VidyoConnector* c, VidyoLocalMonitor* localMonitor, const VidyoVideoFrame *videoFrame)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localMonitor}
		{description: Local monitor that produced the frame.}
	}
	{parameter:
		{name: videoFrame}
		{description: The video frame that contains image data.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalMonitorFrame)(VidyoConnector* c, VidyoLocalMonitor* localMonitor, const VidyoVideoFrame *videoFrame);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnLocalWindowShareFrame}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a local camera produces a video frame.}
	{prototype: void (*VidyoConnectorOnLocalWindowShareFrame)(VidyoConnector* c, VidyoLocalWindowShare* localWindowShare, const VidyoVideoFrame *videoFrame)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: localWindowShare}
		{description: Local window share that produced the frame.}
	}
	{parameter:
		{name: videoFrame}
		{description: The video frame that contains image data.}
	}
}
*/
typedef void(*VidyoConnectorOnLocalWindowShareFrame)(VidyoConnector* c, VidyoLocalWindowShare* localWindowShare, const VidyoVideoFrame *videoFrame);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnRemoteCameraFrame}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a remote camera produces a video frame.}
	{prototype: void (*VidyoConnectorOnRemoteCameraFrame)(VidyoConnector* c, VidyoRemoteCamera* remoteCamera, const VidyoParticipant *participant, const VidyoVideoFrame *videoFrame)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteCamera}
		{description: Remote camera that produced the frame.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant that produced the frame.}
	}
	{parameter:
		{name: videoFrame}
		{description: The video frame that contains image data.}
	}
}
*/
typedef void(*VidyoConnectorOnRemoteCameraFrame)(VidyoConnector* c, VidyoRemoteCamera* remoteCamera, const VidyoParticipant *participant, const VidyoVideoFrame *videoFrame);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnRemoteMicrophoneFrame}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a remote microphone produces an audio frame.}
	{prototype: void (*VidyoConnectorOnRemoteMicrophoneFrame)(VidyoConnector* c, VidyoRemoteMicrophone* remoteMicrophone, const VidyoParticipant *participant, const VidyoAudioFrame *audioFrame)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteMicrophone}
		{description: Remote microphone that produced the frame.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant that produced the frame.}
	}
	{parameter:
		{name: audioFrame}
		{description: The audio frame that contains sound data.}
	}
}
*/
typedef void (*VidyoConnectorOnRemoteMicrophoneFrame)(VidyoConnector* c, VidyoRemoteMicrophone* remoteMicrophone, const VidyoParticipant *participant, const VidyoAudioFrame *audioFrame);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnRemoteMicrophoneEnergy}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a remote microphone produces an frame with energy.}
	{prototype: void (*VidyoConnectorOnRemoteMicrophoneEnergy)(VidyoConnector* c, VidyoRemoteMicrophone* remoteMicrophone, const VidyoParticipant *participant, LmiInt audioEnergy)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteMicrophone}
		{description: Remote microphone that produced the frame.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant that produced the frame.}
	}
	{parameter:
		{name: audioEnergy}
		{description: The audio frame that contains sound data.}
	}
	{note: This callback can be used to draw volume bars in the UI. However, audio sources are dynamically selected on the server and therefore the audio energy will only be available for active speakers. The UI needs to take this into consideration and clear out the volume bars for inactive speakers after a short timeout.}
}
*/
typedef void (*VidyoConnectorOnRemoteMicrophoneEnergy)(VidyoConnector* c, VidyoRemoteMicrophone* remoteMicrophone, const VidyoParticipant *participant, LmiInt audioEnergy);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnRemoteWindowShareFrame}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a remote camera produces a video frame.}
	{prototype: void (*VidyoConnectorOnRemoteWindowShareFrame)(VidyoConnector* c, VidyoRemoteWindowShare* remoteWindowShare, const VidyoParticipant *participant, const VidyoVideoFrame *videoFrame)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteWindowShare}
		{description: Remote window share that produced the frame.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant that produced the frame.}
	}
	{parameter:
		{name: videoFrame}
		{description: The video frame that contains image data.}
	}
}
*/
typedef void(*VidyoConnectorOnRemoteWindowShareFrame)(VidyoConnector* c, VidyoRemoteWindowShare* remoteWindowShare, const VidyoParticipant *participant, const VidyoVideoFrame *videoFrame);

/**
{callback apigroup="beta,enterprise":
 	{name: VidyoConnectorOnRemoteSpeakerAdded}
 	{parent: VidyoConnector}
 	{description: Callback that is triggered when server is ready to accept audio stream from local microphone.}
 	{prototype: void(*VidyoConnectorOnRemoteSpeakerAdded)(VidyoConnector* c, VidyoRemoteSpeaker* remoteSpeaker)}
 	{parameter:
 		{name: c}
 		{description: The VidyoConnector that triggered the callback.}
 	}
 	{parameter:
 		{name: remoteSpeaker}
 		{description: Remote speaker that was added.}
 	}
}
*/
typedef void(*VidyoConnectorOnRemoteSpeakerAdded)(VidyoConnector* c, VidyoRemoteSpeaker* remoteSpeaker);

/**
{callback apigroup="beta,enterprise":
 	{name: VidyoConnectorOnRemoteSpeakerRemoved}
 	{parent: VidyoConnector}
 	{description: Callback that is triggered when server stops receiving audio stream from local microphone.}
 	{prototype: void(*VidyoConnectorOnRemoteSpeakerRemoved)(VidyoConnector* c, VidyoRemoteSpeaker* remoteSpeaker)}
 	{parameter:
 		{name: c}
 		{description: The VidyoConnector that triggered the callback.}
 	}
 	{parameter:
 		{name: remoteSpeaker}
 		{description: Remote speaker that was removed.}
 	}
}
*/
typedef void(*VidyoConnectorOnRemoteSpeakerRemoved)(VidyoConnector* c, VidyoRemoteSpeaker* remoteSpeaker);

/**
{callback apigroup="beta,enterprise":
 	{name: VidyoConnectorOnRemoteSpeakerStateUpdated}
 	{parent: VidyoConnector}
 	{description: Callback that is triggered when server makes a state change on accepting audio stream.}
 	{prototype: void(*VidyoConnectorOnRemoteSpeakerStateUpdated)(VidyoConnector* c, VidyoRemoteSpeaker* remoteSpeaker, VidyoDeviceState state)}
 	{parameter:
 		{name: c}
 		{description: The VidyoConnector that triggered the callback.}
 	}
 	{parameter:
 		{name: remoteSpeaker}
 		{description: Remote speaker that has a state change.}
 	}
 	{parameter:
 		{name: state}
 		{description: The new state of the remote speaker.}
 	}
}
*/
typedef void(*VidyoConnectorOnRemoteSpeakerStateUpdated)(VidyoConnector* c, VidyoRemoteSpeaker* remoteSpeaker, VidyoDeviceState state);

/**
{callback apigroup="beta,enterprise":
 	{name: VidyoConnectorOnRemoteRendererAdded}
 	{parent: VidyoConnector}
 	{description: Callback that is triggered when server is ready to accept video stream from local camera.}
 	{prototype: void(*VidyoConnectorOnRemoteRendererAdded)(VidyoConnector* c, VidyoRemoteRenderer* remoteRenderer)}
 	{parameter:
 		{name: c}
 		{description: The VidyoConnector that triggered the callback.}
 	}
 	{parameter:
 		{name: remoteRenderer}
 		{description: Remote renderer that was added.}
 	}
}
*/
typedef void(*VidyoConnectorOnRemoteRendererAdded)(VidyoConnector* c, VidyoRemoteRenderer* remoteRenderer);

/**
{callback apigroup="beta,enterprise":
 	{name: VidyoConnectorOnRemoteRendererRemoved}
 	{parent: VidyoConnector}
 	{description: Callback that is triggered when server stops receiving video stream from local camera.}
 	{prototype: void(*VidyoConnectorOnRemoteRendererRemoved)(VidyoConnector* c, VidyoRemoteRenderer* remoteRenderer)}
 	{parameter:
 		{name: c}
		{description: The VidyoConnector that triggered the callback.}
 	}
 	{parameter:
 		{name: remoteRenderer}
 		{description: Remote renderer that was removed.}
 	}
}
*/
typedef void(*VidyoConnectorOnRemoteRendererRemoved)(VidyoConnector* c, VidyoRemoteRenderer* remoteRenderer);

/**
{callback apigroup="beta,enterprise":
 	{name: VidyoConnectorOnRemoteRendererStateUpdated}
 	{parent: VidyoConnector}
 	{description: Callback that is triggered when server makes a state change on accepting audio stream.}
 	{prototype: void(*VidyoConnectorOnRemoteRendererStateUpdated)(VidyoConnector* c, VidyoRemoteRenderer* remoteRenderer, VidyoDeviceState state)}
 	{parameter:
 		{name: c}
 		{description: The VidyoConnector that triggered the callback.}
 	}
 	{parameter:
 		{name: remoteRenderer}
 		{description: Remote renderer that has a state change.}
 	}
 	{parameter:
 		{name: state}
 		{description: The new state of the remote renderer.}
 	}
}
*/
typedef void(*VidyoConnectorOnRemoteRendererStateUpdated)(VidyoConnector* c, VidyoRemoteRenderer* remoteRenderer, VidyoDeviceState state);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnNetworkInterfaceAdded}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a network interface is available to a VidyoConnector.}
	{prototype: void (*VidyoConnectorOnNetworkInterfaceAdded)(VidyoConnector* c, VidyoNetworkInterface* networkInterface)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: networkInterface}
		{description: The VidyoNetworkInterface which described the interface.}
	}
}
*/
typedef void(*VidyoConnectorOnNetworkInterfaceAdded)(VidyoConnector* c, VidyoNetworkInterface* networkInterface);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnNetworkInterfaceRemoved}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a network interface is no longer available to a VidyoConnector.}
	{prototype: void (*VidyoConnectorOnNetworkInterfaceRemoved)(VidyoConnector* c, VidyoNetworkInterface* networkInterface)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: networkInterface}
		{description: The VidyoNetworkInterface which described the interface.}
	}
}
*/
typedef void(*VidyoConnectorOnNetworkInterfaceRemoved)(VidyoConnector* c, VidyoNetworkInterface* networkInterface);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnNetworkInterfaceSelected}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time there is a change in which network interface is selected for use.}
	{prototype: void (*VidyoConnectorOnNetworkInterfaceSelected)(VidyoConnector* c, VidyoNetworkInterface* networkInterface, VidyoNetworkInterfaceTransportType transportType)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: networkInterface}
		{description: The VidyoNetworkInterface which described the interface.}
	}
	{parameter:
		{name: transportType}
		{description: The Transport type for which this interface was selected.}
	}
}
*/
typedef void(*VidyoConnectorOnNetworkInterfaceSelected)(VidyoConnector* c, VidyoNetworkInterface* networkInterface, VidyoNetworkInterfaceTransportType transportType);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnNetworkInterfaceStateUpdated}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time there is a change in the state of the network interface such as being up or down.}
	{prototype: void (*VidyoConnectorOnNetworkInterfaceStateUpdated)(VidyoConnector* c, VidyoNetworkInterface* networkInterface, VidyoNetworkInterfaceState state)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: networkInterface}
		{description: The VidyoNetworkInterface which described the interface.}
	}
	{parameter:
		{name: state}
		{description: The new state of the interface.}
	}
}
*/
typedef void(*VidyoConnectorOnNetworkInterfaceStateUpdated)(VidyoConnector* c, VidyoNetworkInterface* networkInterface, VidyoNetworkInterfaceState state);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnSuccess}
	{parent: VidyoConnector}
	{description: Callback that is triggered when the connection is successfully established. }
	{prototype: void (*VidyoConnectorOnSuccess)(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
}
*/
typedef void(*VidyoConnectorOnSuccess)(VidyoConnector* c);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnFailure}
	{parent: VidyoConnector}
	{description: Callback that is triggered when the connection attempt has failed. }
	{prototype: void (*VidyoConnectorOnFailure)(VidyoConnector* c, VidyoConnectorFailReason reason)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: reason}
		{description: Reason why the connection could not be established.}
	}
}
*/
typedef void(*VidyoConnectorOnFailure)(VidyoConnector* c, VidyoConnectorFailReason reason);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnDisconnected}
	{parent: VidyoConnector}
	{description: Callback that is triggered when an established connection is disconnected. }
	{prototype: void (*VidyoConnectorOnDisconnected)(VidyoConnector* c, VidyoConnectorDisconnectReason reason)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: reason}
		{description: Reason why the established connection became disconnected.}
	}
}
*/
typedef void(*VidyoConnectorOnDisconnected)(VidyoConnector* c, VidyoConnectorDisconnectReason reason);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnParticipantJoined}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time another participant joins a conference.}
	{prototype: void (*VidyoConnectorOnParticipantJoined)(VidyoConnector* c, VidyoParticipant* participant)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant who joined the conference.}
	}
}
*/
typedef void(*VidyoConnectorOnParticipantJoined)(VidyoConnector* c, VidyoParticipant* participant);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnParticipantLeft}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time an existing participant leaves a conference.}
	{prototype: void (*VidyoConnectorOnParticipantLeft)(VidyoConnector* c, VidyoParticipant* participant)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant who left the conference.}
	}
}
*/
typedef void(*VidyoConnectorOnParticipantLeft)(VidyoConnector* c, VidyoParticipant* participant);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnDynamicParticipantChanged}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time the order of dyanmic participants has changed, based on active speech detection.}
	{prototype: void (*VidyoConnectorOnDynamicParticipantChanged)(VidyoConnector* c, LmiVector(VidyoParticipant)* participants)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: participants}
		{description: Ordered list of most recently selected participants in the conference.}
	}
}
*/
typedef void(*VidyoConnectorOnDynamicParticipantChanged)(VidyoConnector* c, LmiVector(VidyoParticipant)* participants);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnLoudestParticipantChanged}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time one of the other participants becomes the loudest, based on active speech detection.}
	{prototype: void (*VidyoConnectorOnLoudestParticipantChanged)(VidyoConnector* c, const VidyoParticipant* participant, LmiBool audioOnly)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: participant}
		{description: New loudest participant.}
	}
	{parameter:
		{name: audioOnly}
		{description: LMI_TRUE if the participant does not have video streams.}
	}
}
*/
typedef void(*VidyoConnectorOnLoudestParticipantChanged)(VidyoConnector* c, const VidyoParticipant* participant, LmiBool audioOnly);


/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnChatMessageReceived}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a chat message is received from one of the other participants.}
	{prototype: void (*VidyoConnectorOnChatMessageReceived)(VidyoConnector* c, const VidyoParticipant* participant, VidyoChatMessage* chatMessage)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant who sent the chat message. It can be NULL when this information is unknown to the client due to timing.}
	}
	{parameter:
		{name: chatMessage}
		{description: Pointer to the VidyoChatMessage object containing the message, and sender name and id.}
	}
}
*/
typedef void(*VidyoConnectorOnChatMessageReceived)(VidyoConnector* c, const VidyoParticipant* participant, VidyoChatMessage* chatMessage);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnAvailableResourcesChanged}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time the available local resources change. If there is no enough bandwidth for sources bandwidthSend and bandwidthReceive will be 0.
		If has enough bandwidth for sources bandwidthSend and bandwidthReceive will be 1-70%. If has more bandwidth than required for sources bandwidthSend and bandwidthReceive will be 70-100%.}
	{prototype: void (*VidyoConnectorOnAvailableResourcesChanged)(VidyoConnector* c, LmiUint cpuEncode, LmiUint cpuDecode, LmiUint bandwidthSend, LmiUint bandwidthReceive)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: cpuEncode}
		{description: Percent of maximum CPU available to encode the streams.}
	}
	{parameter:
		{name: cpuDecode}
		{description: Percent of maximum CPU available to decode the streams.}
	}
	{parameter:
		{name: bandwidthSend}
		{description: Percentage of maximum bandwidth avaialble to transmit the streams.}
	}
	{parameter:
		{name: bandwidthReceive}
		{description: Percentage of maximum bandwidth avaialble to receive the streams.}
	}
}
*/
typedef void(*VidyoConnectorOnAvailableResourcesChanged)(VidyoConnector* c, LmiUint cpuEncode, LmiUint cpuDecode, LmiUint bandwidthSend, LmiUint bandwidthReceive);

/**
 {callback apigroup="connector" deprecated="yes" replaced-by="VidyoConnectorOnConnectionPropertiesChanged":
	{name: VidyoConnectorRecorderInCall}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time a recorder joins or leaves the call.}
	{prototype: void (*VidyoConnectorRecorderInCall)(VidyoConnector* c, LmiBool hasRecorder, LmiBool isPaused )}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: hasRecorder}
		{description: LMI_TRUE when a recorder joined, LMI_FALSE when no recorder are in the call}
	}
    {parameter:
        {name: isPaused}
        {description: LMI_TRUE when a recorder joined but is paused, LMI_FALSE otherwise}
    }
}
*/
typedef void(*VidyoConnectorRecorderInCall)(VidyoConnector* c, LmiBool hasRecorder, LmiBool isPaused);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnConnectionPropertiesChanged}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when connection properties is changed.}
	{prototype: void (*VidyoConnectorOnConnectionPropertiesChanged)(VidyoConnector* c, const VidyoConnectorConnectionProperties* connectionProperties)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: connectionProperties}
		{description: VidyoConnectorConnectionProperties object.}
	}
}
*/
typedef void (*VidyoConnectorOnConnectionPropertiesChanged)(VidyoConnector* c, const VidyoConnectorConnectionProperties* connectionProperties);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnMaxRemoteSourcesChanged}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time the local client's maximum decoding capability changes.}
	{prototype: void (*VidyoConnectorOnMaxRemoteSourcesChanged)(VidyoConnector* c, LmiUint maxRemoteSources)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: maxRemoteSources}
		{description: Maximum number of remote video sources that can be recived both statically and dynamically.}
	}
	{note: When the application is managing the statically viewed sources through the VidyoConnectorAssign*() API, such as VidyoConnectorAssignViewToRemoteCamera(), the application is responsible for keeping track of the number of these views. When the number of maximum remote participants that can be decoded falls below what the applicaiton has statically viewed, the application must choose which static sources to enable or disable to meet the resource manager constraints.}
}
*/
typedef void(*VidyoConnectorOnMaxRemoteSourcesChanged)(VidyoConnector* c, LmiUint maxRemoteSources);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnWebProxyCredentialsRequest}
	{parent: VidyoConnector}
	{description: Callback that is triggered when Web Proxy Server requires password.}
	{prototype: void (*VidyoConnectorOnWebProxyCredentialsRequest)(VidyoConnector* c, LmiString* webProxyAddress)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
 	{parameter:
 		{name: webProxyAddress}
 		{description: .}
 	}
}
*/
typedef void(*VidyoConnectorOnWebProxyCredentialsRequest)(VidyoConnector* c, LmiString* webProxyAddress);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnReconnecting}
	{parent: VidyoConnector}
	{description: Callback that is triggered when the reconnecting attempt has failed and new attempt will be started. }
	{prototype: void (*VidyoConnectorOnReconnecting)(VidyoConnector *c, LmiUint attempt, LmiUint attemptTimeout,  VidyoConnectorFailReason reason)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: attempt}
		{description: The attempt number.}
	}
	{parameter:
		{name: attemptTimeout}
		{description: The time in seconds after which attempt to reconnect will start..}
	}
	{parameter:
		{name: reason}
		{description: The previous reason of reconnecting failure.}
	}
}
*/
typedef void(*VidyoConnectorOnReconnecting)(VidyoConnector *c, LmiUint attempt, LmiUint attemptTimeout, VidyoConnectorFailReason reason);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnReconnected}
	{parent: VidyoConnector}
	{description: This callback will be triggered when the connection was established and the user rejoined to the call/conference. }
	{prototype: void (*VidyoConnectorOnReconnected)(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
}
*/
typedef void(*VidyoConnectorOnReconnected)(VidyoConnector* c);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnConferenceLost}
	{parent: VidyoConnector}
	{description: Callback that is triggered when reconnecting failed and there no more attempt to reconnect. }
	{prototype: void (*VidyoConnectorOnConferenceLost)(VidyoConnector* c, VidyoConnectorFailReason reason)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: reason}
		{description: Reason why the established connection became disconnected.}
	}
}
*/
typedef void(*VidyoConnectorOnConferenceLost)(VidyoConnector* c, VidyoConnectorFailReason reason);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRemoteCameraFirstFrameReceived}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time the first frame for remote camera received.}
	{prototype: void(*VidyoConnectorOnRemoteCameraFirstFrameReceived)(VidyoConnector* c, VidyoRemoteCamera* remoteCamera)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteCamera}
		{description: Remote camera which frame was received for.}
	}
}
*/
typedef void(*VidyoConnectorOnRemoteCameraFirstFrameReceived)(VidyoConnector* c, VidyoRemoteCamera* remoteCamera);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRemoteWindowShareFirstFrameReceived}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time the first frame for remote window share received.}
	{prototype: void(*VidyoConnectorOnRemoteWindowShareFirstFrameReceived)(VidyoConnector* c, VidyoRemoteWindowShare* remoteWindowShare)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: remoteWindowShare}
		{description: Remote window share which frame was received for.}
	}
}
*/
typedef void(*VidyoConnectorOnRemoteWindowShareFirstFrameReceived)(VidyoConnector* c, VidyoRemoteWindowShare* remoteWindowShare);

/**
{callback apigroup="simple":
	{name: VidyoConnectorOnRequestModeratorRoleResult}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when the result of requesting moderator role is received.}
	{prototype: void (*VidyoConnectorOnRequestModeratorRoleResult)(VidyoConnector* c,  VidyoConnectorModerationResult result)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback. }
	}
	{parameter:
		{name: result}
		{description: Result of the moderator role request.}
	}
}
*/
typedef void (*VidyoConnectorOnRequestModeratorRoleResult)(VidyoConnector* c, VidyoConnectorModerationResult result);

/**
{callback apigroup="simple":
	{name: VidyoConnectorOnRemoveModeratorRoleResult}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when the result of removing moderator role is received.}
	{prototype: void (*VidyoConnectorOnRemoveModeratorRoleResult)(VidyoConnector* c, VidyoConnectorModerationResult result)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback. }
	}
	{parameter:
		{name: result}
		{description: Result of the remove moderator role request.}
	}
}
*/
typedef void (*VidyoConnectorOnRemoveModeratorRoleResult)(VidyoConnector* c, VidyoConnectorModerationResult result);

/**
{callback apigroup="simple":
	{name: VidyoConnectorOnInviteResult}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when the result of invitation is received.}
	{prototype: void(*VidyoConnectorOnInviteResult)(VidyoConnector* c, const LmiString* inviteeId, VidyoConnectorModerationResult result)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback. }
	}
	{parameter:
		{name: inviteeId}
		{description: Id of the user who was invited. }
	}
	{parameter:
		{name: result}
		{description: The result of Invitation. }
	}
}
*/
typedef void(*VidyoConnectorOnInviteResult)(VidyoConnector* c, const LmiString* inviteeId, VidyoConnectorModerationResult result);

/**
{callback apigroup="simple":
	{name: VidyoConnectorOnLockRoomResult}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when the result of lock room request is received.}
	{prototype: void(*VidyoConnectorOnLockRoomResult)(VidyoConnector* c, VidyoConnectorModerationResult result)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: The result of lock room request.}
	}
}
*/
typedef void(*VidyoConnectorOnLockRoomResult)(VidyoConnector* c, VidyoConnectorModerationResult result);

/**
{callback apigroup="simple":
	{name: VidyoConnectorOnUnlockRoomResult}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when result of unlock room request is received.}
	{prototype: void(*VidyoConnectorOnUnlockRoomResult)(VidyoConnector* c,  VidyoConnectorModerationResult result)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: The result of unlock room request.}
	}
}
*/
typedef void(*VidyoConnectorOnUnlockRoomResult)(VidyoConnector* c, VidyoConnectorModerationResult result);

/**
{callback apigroup="simple":
	{name: VidyoConnectorOnSetRoomPINResult}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when result of set room pin is received.}
	{prototype: void(*VidyoConnectorOnSetRoomPINResult)(VidyoConnector* c, VidyoConnectorModerationResult result)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: The result of set room pin request.}
	}
}
*/
typedef void(*VidyoConnectorOnSetRoomPINResult)(VidyoConnector* c, VidyoConnectorModerationResult result);

/**
{callback apigroup="simple":
	{name: VidyoConnectorOnRemoveRoomPINResult}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when result of remove room pin is received.}
	{prototype: void(*VidyoConnectorOnRemoveRoomPINResult)(VidyoConnector* c, VidyoConnectorModerationResult result)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: The result of remove room pin request.}
	}
}
*/
typedef void(*VidyoConnectorOnRemoveRoomPINResult)(VidyoConnector* c, VidyoConnectorModerationResult result);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnGetRecordingServiceProfiles}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when result of recording service profiles is received.}
	{prototype: void(*VidyoConnectorOnGetRecordingServiceProfiles)(VidyoConnector* c, const LmiVector(LmiString)* profiles, const LmiVector(LmiString)* prefixes, VidyoConnectorRecordingServiceResult result)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: profiles}
		{description: Vector that contains descriptions of available recording profiles, NULL if the request has failed.}
	}
	{parameter:
		{name: prefixes}
		{description: Vector that contains prefixes of available recording profiles, NULL if the request has failed.}
	}
	{parameter:
		{name: result}
		{description: Result of the request. Can be one of VidyoConnectorRecordingServiceResult values.}
	}
}
*/
typedef void(*VidyoConnectorOnGetRecordingServiceProfiles)(VidyoConnector* c, const LmiVector(LmiString)* profiles, const LmiVector(LmiString)* prefixes, VidyoConnectorRecordingServiceResult result);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRecordingServiceStartResult}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when result of recording service start is received.}
	{prototype: void(*VidyoConnectorOnRecordingServiceStartResult)(VidyoConnector* c, VidyoConnectorModerationResult result)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: Result of the request. Can be one of VidyoConnectorModerationResult values.}
	}
}
*/
typedef void(*VidyoConnectorOnRecordingServiceStartResult)(VidyoConnector* c, VidyoConnectorModerationResult result);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRecordingServiceStopResult}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when result of recording service stop is received.}
	{prototype: void(*VidyoConnectorOnRecordingServiceStopResult)(VidyoConnector* c, VidyoConnectorModerationResult result)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: Result of the request. Can be one of VidyoConnectorModerationResult values.}
	}
}
*/
typedef void(*VidyoConnectorOnRecordingServiceStopResult)(VidyoConnector* c, VidyoConnectorModerationResult result);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRecordingServicePauseResult}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when result of recording service pause is received.}
	{prototype: void(*VidyoConnectorOnRecordingServicePauseResult)(VidyoConnector* c, VidyoConnectorModerationResult result)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: Result of the request. Can be one of VidyoConnectorModerationResult values.}
	}
}
*/
typedef void(*VidyoConnectorOnRecordingServicePauseResult)(VidyoConnector* c, VidyoConnectorModerationResult result);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRecordingServiceResumeResult}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when result of recording service resume is received.}
	{prototype: void(*VidyoConnectorOnRecordingServiceResumeResult)(VidyoConnector* c, VidyoConnectorModerationResult result)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: Result of the request. Can be one of VidyoConnectorModerationResult values.}
	}
}
*/
typedef void(*VidyoConnectorOnRecordingServiceResumeResult)(VidyoConnector* c, VidyoConnectorModerationResult result);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnConferenceModeChanged}
	{parent: VidyoConnector}
	{description: Change to Callback that will be triggered when conference mode changes.} 
	{prototype:  void(*VidyoConnectorOnConferenceModeChanged)(VidyoConnector *c, VidyoConnectorConferenceMode mode)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback. }
	}
	{parameter:
		{name: mode}
		{description: The New conference mode i.e. lecture, lobby or group}
	}
}
*/
typedef void(*VidyoConnectorOnConferenceModeChanged)(VidyoConnector *c, VidyoConnectorConferenceMode mode);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnConferenceHostStatusChanged}
	{parent: VidyoConnector}
	{description: Callback that will be triggered when conference host joins or leaves.}
	{prototype: void(*VidyoConnectorOnConferenceHostStatusChanged)(VidyoConnector *c, const VidyoParticipant* host, VidyoConnectorConferenceHostState hostState)}
	{parameter:
	{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback. }
	}
	{parameter:
		{name: host}
		{description: The participant info of the host}
	}
	{parameter:
		{name: hostState}
		{description: The host status i.e. joined or left}
	}
}
*/
typedef void(*VidyoConnectorOnConferenceHostStatusChanged)(VidyoConnector *c, const VidyoParticipant* host, VidyoConnectorConferenceHostState hostState);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnUserSearchResults}
	{parent: VidyoConnector}
	{description: Callback that is triggered for each user search operation.}
	{prototype:  void (*VidyoConnectorOnUserSearchResults)(VidyoConnector* c, const LmiString* searchText, LmiUint startIndex, VidyoConnectorSearchResult searchResult, const LmiVector(VidyoContactInfo)* contacts, LmiSizeT numRecords)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: searchText}
		{description: The search text provided to search users.}
	}
	{parameter:
		{name: startIndex}
		{description: The starting index from where the result should be retrieved.}
	}
	{parameter:
		{name: searchResult}
		{description: Result of the user search request.}
	}
	{parameter:
		{name: contacts}
		{description: The contacts found in the search.}
	}
	{parameter:
		{name: numRecords}
		{description: The number of records in the search.}
	}
}
*/
typedef void (*VidyoConnectorOnUserSearchResults)(VidyoConnector* c, const LmiString* searchText,
    LmiUint startIndex, VidyoConnectorSearchResult searchResult, const LmiVector(VidyoContactInfo)* contacts,
	LmiSizeT numRecords);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnError}
	{parent: VidyoConnector}
	{description: Callback that is triggered to provide advanced error codes to the application.}
	{prototype: void(*VidyoConnectorOnError)(VidyoConnector *c, VidyoConnectorErrorCode error, const LmiString* apiName)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: error}
		{description: Error code, see VidyoConnectorErrorCode.}
	}
	{parameter:
		{name: apiName}
		{description: Name of the api where error occured.}
	}
}
*/
typedef void(*VidyoConnectorOnError)(VidyoConnector *c, VidyoConnectorErrorCode error, const LmiString* apiName);

/**
{type apigroup="connector" visibility="public":
	{name: VidyoConnectorModerationActionType}
	{parent: VidyoConnector}
	{description: This type indicates the action of moderation request.}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_Mute}
		{description: Mute video or audio action.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_Unmute}
		{description: Unmute video or audio action.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_StartLectureMode}
		{description: Start lecture mode action.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_StopLectureMode}
		{description: Stop lecture mode action.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_SetPresenter}
		{description: Set presenter action.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_RemovePresenter}
		{description: Remove presenter action.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_RaiseHand}
		{description: Raise hand action.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_UnraiseHand}
		{description: Unraise hand action.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_DismissRaisedHand}
		{description: Dismiss raised hand action.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_DismissAllRaisedHands}
		{description: Dismiss all raised hands action.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_SetModeratorPin}
		{description: Set moderator pin.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_RemoveModeratorPin}
		{description: REmove moderator pin.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_DisconnectAll}
		{description: Disconnect all participants.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_DisconnectOne}
		{description: Disconnect one participant.}
	}
	{value:
		{name: VIDYO_CONNECTORMODERATIONACTIONTYPE_Invalid}
		{description: Wrong action.}
	}
}
*/
typedef enum
{
	VIDYO_CONNECTORMODERATIONACTIONTYPE_Mute,
	VIDYO_CONNECTORMODERATIONACTIONTYPE_Unmute,
	VIDYO_CONNECTORMODERATIONACTIONTYPE_StartLectureMode,
	VIDYO_CONNECTORMODERATIONACTIONTYPE_StopLectureMode,
	VIDYO_CONNECTORMODERATIONACTIONTYPE_SetPresenter,
	VIDYO_CONNECTORMODERATIONACTIONTYPE_RemovePresenter,
	VIDYO_CONNECTORMODERATIONACTIONTYPE_RaiseHand,
	VIDYO_CONNECTORMODERATIONACTIONTYPE_UnraiseHand,
	VIDYO_CONNECTORMODERATIONACTIONTYPE_DismissRaisedHand,
	VIDYO_CONNECTORMODERATIONACTIONTYPE_DismissAllRaisedHands,
	VIDYO_CONNECTORMODERATIONACTIONTYPE_SetModeratorPin,
	VIDYO_CONNECTORMODERATIONACTIONTYPE_RemoveModeratorPin,
	VIDYO_CONNECTORMODERATIONACTIONTYPE_DisconnectAll,
	VIDYO_CONNECTORMODERATIONACTIONTYPE_DisconnectOne,
	VIDYO_CONNECTORMODERATIONACTIONTYPE_Invalid
} VidyoConnectorModerationActionType;

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnModerationResult}
	{parent: VidyoConnector}
	{description: Callback that triggered when the moderation API is complete.}
	{prototype: void(*VidyoConnectorOnModerationResult)(const VidyoConnector* c, const VidyoParticipant* participant, VidyoConnectorModerationResult result, VidyoConnectorModerationActionType action, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: participant}
		{description: Participant object. Note: will be NULL when moderation request for all participants.}
	}
	{parameter:
		{name: result}
		{description: Result of the moderator request.}
	}
	{parameter:
		{name: action}
		{description: Action that was performed.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
}
*/
typedef void(*VidyoConnectorOnModerationResult)(const VidyoConnector* c, const VidyoParticipant* participant, VidyoConnectorModerationResult result, VidyoConnectorModerationActionType action, const LmiString* requestId);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnGetWhitelistedAudioDevices}
	{parent: VidyoConnector}
	{description: Callback that reports the names of the audio devices that are whitelisted.}
	{prototype: void(*VidyoConnectorOnGetWhitelistedAudioDevices)(VidyoConnector* c, const LmiVector(LmiString)* audioDevices)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: audioDevices}
		{description: The array with the names of the audio devices that are whitelisted.}
	}
}
*/
typedef void(*VidyoConnectorOnGetWhitelistedAudioDevices)(VidyoConnector* c, const LmiVector(LmiString)* audioDevices);

/**
 {function apigroup="connector":
	{name: VidyoConnectorConstruct}
	{parent: VidyoConnector}
	{description: Constructs the VidyoConnector. On, iOS, this API must be called from main(UI) thread.}
	{prototype: VidyoConnector* VidyoConnectorConstruct(VidyoConnector* c, const LmiViewId* viewId, VidyoConnectorViewStyle viewStyle, LmiUint remoteParticipants, const char* logFileFilter, const char* logFileName, const LmiVoidPtr userData)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform-specific view ID where the VidyoConnector's rendering window will be added as a child window. It will be used to render the preview and composite remote participants.  When running custom layout on macOS, the core animation layer should be disabled on this view.}
	}
	{parameter:
		{name: viewStyle}
		{description: Type of the composite renderer which represents the visual style and behaviour. }
	}
	{parameter:
		{name: remoteParticipants}
		{description: Number of remote participants to composite into the window. Setting the value to 0 (zero) will render the preview only.}
	}
	{parameter:
		{name: logFileFilter}
		{description:
	   {p: A space-separated (or comma-separated)
		 sequence of names of log levels, each optionally followed by a
		 category.  Categories are
		 separated from levels by the character \'@\'.}
	   {p: Either a level or category may be specified as \'all\' or \'*\'
	     (interchangeably), meaning all levels or categories.  A level specified
		 without a category is equivalent to a level@*.}
	   {p: The levels are: \'fatal\', \'error\', \'warning\', \'info\', \'debug\', 
		 \'sent\', \'received\', \'enter\', and '\leave\'.}
	   {p: The level may be prefixed by \'-\' or \'!\' (interchangeably), meaning to remove
		 the given level (or levels, see below) from the given category.
		 The string \'none\' is equivalent to \'-all\'.}
	   {p: The first level in the list may be prefixed by \'+\'.
	     If the first level begins with a \'+\' or \'-\'/\'!\', the string
		 is used to modify the listener\'s existing levels and categories.
		 Otherwise, unless the {code: add} parameter is true, the levels and categories 
 		 specified in the string replace all existing levels and categories for the listener.  
 		 The leading character \'+\' may be present before other levels, but is ignored.}
	   {p: Unless they are preceded by \'=\', level names imply other levels
		 as well.  In particular, each of the levels \'fatal\', \'error\', \'warning\',
		 \'info\', and \'debug\' implies the levels higher than it, when specified
		 positively; \'sent\' and \'received\' always imply each other; and \'enter\' and
		 \'leave\' always imply each other.}
	   {p: See VidyoConnectorGetLogCategories for determining the names and descriptions of registered categories.}
	   {p: Level names are case-insensitive; category names are case-sensitive.}
	   {p: Example: "all" - all levels, all categories.}
	   {p: Example: "all -enter" - all categories, all levels but enter and leave.}
	   {p: Example: "*@VidyoClient" - all levels for the VidyoClient category.}
	   {p: Example: "-*@VidyoClient" - remove all levels from the VidyoClient category.}
	   {p: Example: "debug" - all categories, fatal, error, warning, info, and debug levels.}
	   {p: Example: "=debug" - all categories, debug level only.}
	   {p: Example: "all -=enter" - all categories, all levels but enter (leave is still included).}
	   {p: Example: "error@VidyoClient" - fatal and error only for the VidyoClient category.}
	   {p: Example: "warning debug@VidyoClient" - fatal, error, and warning for all categories; additionally, info and debug for the VidyoClient category.}
	}}
	{parameter:
		{name: logFileName}
		{description: Full path to the file where the log should be stored; otherwise, NULL or empty string, in order to use the default OS-dependent writable path. }
	}
	{parameter:
		{name: userData}
		{description: Arbitrary user data that can be retrieved later.}
	}
	{return: The constructed object on success. NULL on failure.}
	{note: There are two mutually exclusive alternatives to rendering, composited and custom. A composited renderer is used when the viewId is passed to this constructor or when VidyoConnectorAssignViewToCompositeRenderer() is called later. Custom rendring allows configurable layouts where each source is rendered into a view provided by the application. It uses VidyoConnectorAssignViewToLocalCamera(), VidyoConnectorAssignViewToRemoteCamera() etc. APIs. However, once the choice is made, the other cannot be used. }
	{note: If viewId is valid the device detection will start implicitly and default devices will be selected, otherwise the viewId can be assigned later using VidyoConnectorAssignViewToCompositeRenderer() but the device detection will not start until the appropriate event listeners are registered such as VidyoConnectorRegisterLocalCameraEventListener(). }
	{note: LmiViewId is HWND on Windows, NSView on Mac OS X, UIView on iOS, window on X11, ViewGroup on Android.}
	{note: The view will not be shown until VidyoConnectorShowViewAt is called with valid parameters.}
}
*/
VidyoConnector* VidyoConnectorConstruct(VidyoConnector* c, const LmiViewId* viewId, VidyoConnectorViewStyle viewStyle, LmiUint remoteParticipants, const char* logFileFilter, const char* logFileName, const LmiVoidPtr userData);

/**
 {function apigroup="connector":
	{name: VidyoConnectorConstructCopy}
	{parent: VidyoConnector}
	{description: Constructs an VidyoConnector object as a copy of another.}
	{prototype: VidyoConnector *VidyoConnectorConstructCopy(VidyoConnector* c, const VidyoConnector *other)}
	{parameter: {name: c} {description: The VidyoConnector object to construct.}}
	{parameter: {name: other} {description: The VidyoConnector object from which to construct a copy.}}
	{return: The constructed object on success. NULL on failure.}
}
*/

/**
 {function apigroup="connector":
	{name: VidyoConnectorDestruct}
	{parent: VidyoConnector}
	{description: Destructs an VidyoConnector object.}
	{prototype: void VidyoConnectorDestruct(VidyoConnector* c)}
	{parameter: {name: c} {description: The VidyoConnector object to destruct.}}
}
*/

/**
 {function apigroup="connector":
	{name: VidyoConnectorAssign}
	{parent: VidyoConnector}
	{description: Assigns one VidyoConnector object the value of another. }
	{prototype: VidyoConnector *VidyoConnectorAssign(VidyoConnector* c, const VidyoConnector *other)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: other} {description: The VidyoConnector object from which to assign.}}
	{return: The destination object on success. NULL on failure.}
}
*/

/**
 {function apigroup="connector" visibility="private":
	{name: VidyoConnectorSetUserData}
	{parent: VidyoConnector}
	{description: Associates user data with a VidyoConnector object, so that such data may be retrieved later.}
	{prototype: void VidyoConnectorSetUserData(VidyoConnector* c, LmiVoidPtr userData)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: userData}
		{description: Arbitrary data.}
	}
}
*/
void VidyoConnectorSetUserData(VidyoConnector* c, LmiVoidPtr userData);

/**
 {function apigroup="connector" visibility="private":
	{name: VidyoConnectorGetUserData}
	{parent: VidyoConnector}
	{description: Retreives any user data that was associated with a VidyoConnector object.}
	{prototype: LmiVoidPtr VidyoConnectorGetUserData(const VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: User data on success. NULL on failure.}
}
*/
LmiVoidPtr VidyoConnectorGetUserData(const VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSetMode}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets the operating mode of the VidyoConnector. This API have to be called before renderer is destructed: i.e. VidyoConnectorHideView will be called in case VIDYO_CONNECTORMODE_Background and after renderer is constructed: i.e. VidyoConnectorAssignViewToCompositeRenderer, VidyoConnectorAssignViewToLocalCamera, VidyoConnectorAssignViewToRemoteCamera will be called in case VIDYO_CONNECTORMODE_Foreground. This API is required only when user in the conference. On, iOS, this API must be called from main(UI) thread.
                    This API is allowed in all states of VidyoConnector. It won't do anything in VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead states of VidyoConnector but it will still return LMI_TRUE.
 }
	{prototype: LmiBool VidyoConnectorSetMode(VidyoConnector* c, VidyoConnectorMode mode)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: mode}
		{description: Desired operating mode.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetMode(VidyoConnector* c, VidyoConnectorMode mode);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSetLocation}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets the geographic location of the endpoint using the VidyoConnector.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetLocation(VidyoConnector* c, LmiFloat64 latitude, LmiFloat64 longitude)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: latitude}
		{description: Latitude of the geographic coordinate.}
	}
	{parameter:
		{name: longitude}
		{description: Longitude of the geographic coordinate.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: Setting the location manually overrides other proximity detection mechanisms that are used by VidyoConnector to determine the best routing performance. }
}
*/
LmiBool VidyoConnectorSetLocation(VidyoConnector* c, LmiFloat64 latitude, LmiFloat64 longitude);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSetPool}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets a specific media pool to use for audio and video. 
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetPool(VidyoConnector* c, const char *name)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: name}
		{description: Name of the pool.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: Setting the pool name manually overrides other proximity detection mechanisms that are used by VidyoConnector to determine the best routing performance. The pool name can be retrived by contacting support. }
}
*/
LmiBool VidyoConnectorSetPool(VidyoConnector* c, const char *name);

/**
 {function apigroup="connector" deprecated="yes" replaced-by="VidyoConnectorSetMaxSendBitRate":
	{name: VidyoConnectorSetMaxBitRate}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets the max send bit rate for all sources. 
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling and 
					VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetMaxBitRate(VidyoConnector* c, LmiUint bitRate)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: bitRate}
		{description: The max send bit rate for all sources. }
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
    {note: This maximum bit rate will be distributed among all the local sources.}
}
*/
LmiBool VidyoConnectorSetMaxBitRate(VidyoConnector* c, LmiUint bitRate);

/**
{function apigroup="connector":
    {name: VidyoConnectorSetMaxSendBitRate}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Sets the max send bit rate for all sources.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling and 
					VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorSetMaxSendBitRate(VidyoConnector* c, LmiUint bitRate)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {parameter:
        {name: bitRate}
        {description: The max send bit rate for all sources, in bps. }
    }
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
    {note: This maximum bit rate will be distributed among all the local sources.}
 }
 */
LmiBool VidyoConnectorSetMaxSendBitRate(VidyoConnector* c, LmiUint bitRate);

/**
{function apigroup="connector":
    {name: VidyoConnectorGetMaxSendBitRate}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Return the max send bit rate configured by application.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiUint VidyoConnectorGetMaxSendBitRate(VidyoConnector* c)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {return: the max send bit rate configured by application, in bps}
}
*/
LmiUint VidyoConnectorGetMaxSendBitRate(VidyoConnector* c);

/**
{function apigroup="beta,enterprise":
    {name: VidyoConnectorSetMaxReceiveBitRate}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Request server to limit bit rate sent to this client.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CNNECTORSTATE_Disabling 
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorSetMaxReceiveBitRate(VidyoConnector* c, LmiUint bitRate)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {parameter:
        {name: bitRate}
        {description: The maximum receiving bit rate request sent to server, in bps. }
    }
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetMaxReceiveBitRate(VidyoConnector* c, LmiUint bitRate);

/**
{function apigroup="beta,enterprise":
    {name: VidyoConnectorGetMaxReceiveBitRate}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Return the max receive bit rate configured by application.
    				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiUint VidyoConnectorGetMaxReceiveBitRate(VidyoConnector* c)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {return: the max receive bit rate configured by application, in kbps}
}
*/
LmiUint VidyoConnectorGetMaxReceiveBitRate(VidyoConnector* c);

/**
{function apigroup="connector":
	{name: VidyoConnectorEnableDebug}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Enabled debug mode which will start debug logging and enable statistics.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorEnableDebug(VidyoConnector* c, LmiUint port, const char *logFilter)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: port}
		{description: Local listening port for debugging.}
	}
	{parameter:
		{name: logFilter}
		{description: Log filter to use on the debug port. Default log filter will be use if null or empty string is passed.
		{p: Example: warning debug@VidyoClient debug@VidyoConnector all@LmiPortalSession all@LmiPortalMembership debug@LmiResourceManager all@LmiIce all@LmiSignaling.}}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorEnableDebug(VidyoConnector* c, LmiUint port, const char *logFilter);

/**
{function apigroup="connector":
	{name: VidyoConnectorDisableDebug}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Disable debug mode. 
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDisableDebug(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorDisableDebug(VidyoConnector* c);

/**
{function apigroup="connector":
	{name: VidyoConnectorRegisterModerationResultEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about result of moderation APIs.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterModerationResultEventListener(VidyoConnector* c, VidyoConnectorOnModerationResult onModerationResult)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onModerationResult}
		{description: Callback that is triggered when the moderation API is complete.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterModerationResultEventListener(VidyoConnector* c, VidyoConnectorOnModerationResult onModerationResult);

/**
{function apigroup="connector":
	{name: VidyoConnectorUnregisterModerationResultEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregistering notifications about moderation result events.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterModerationResultEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterModerationResultEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorAssignViewToCompositeRenderer}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Composites the preview and remote participants in a view. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disconnecting. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorAssignViewToCompositeRenderer(VidyoConnector *c, const LmiViewId* viewId, VidyoConnectorViewStyle viewStyle, LmiUint remoteParticipants)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID where the participant will be rendered.}
	}
	{parameter:
		{name: viewStyle}
		{description: Type of the composite renderer which represents the visual style and behaviour. }
	}
	{parameter:
		{name: remoteParticipants}
		{description: Number of remote participants to composite into the window. Setting the value to 0 will render preview only. }
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: This API will work only if custom rendering has not been used. ex: VidyoConnectorAssignViewToLocalCamera() was never called.}
	{note: The view will not be shown until VidyoConnectorShowViewAt is called with valid parameters.}
}
*/
LmiBool VidyoConnectorAssignViewToCompositeRenderer(VidyoConnector *c, const LmiViewId* viewId, VidyoConnectorViewStyle viewStyle, LmiUint remoteParticipants);

/**
 {type apigroup="connector" visibility="public":
    {name: VidyoConnectorLogLevel}
    {parent: VidyoConnector}
    {description: Log levels of VidyoClient library.}
    {value: {name: VIDYO_CONNECTORLOGLEVEL_PRODUCTION}
        {description: The production log mode.}}
    {value: {name: VIDYO_CONNECTORLOGLEVEL_DEBUG}
        {description: The debug log mode.}}
    {value: {name: VIDYO_CONNECTORLOGLEVEL_INVALID}
        {description: The Invalid log level. This value should not be used when using VidyoConnectorSetLogLevel. It is used to indicate errors when VidyoConnectorGetLogLevel is used.}}
}
*/
typedef enum
{
    VIDYO_CONNECTORLOGLEVEL_PRODUCTION,
    VIDYO_CONNECTORLOGLEVEL_DEBUG,
    VIDYO_CONNECTORLOGLEVEL_INVALID,
} VidyoConnectorLogLevel;

/**
 {type apigroup="connector" visibility="public":
    {name: VidyoConnectorLoggerType}
    {parent: VidyoConnector}
    {description: Logger type of VidyoClient library.}
    {value: {name: VIDYO_CONNECTORLOGGERTYPE_CONSOLE}
     {description: The console logger.}}
    {value: {name: VIDYO_CONNECTORLOGGERTYPE_FILE}
        {description: The file logger.}}
    {value: {name: VIDYO_CONNECTORLOGGERTYPE_FEEDBACK}
        {description: The feedback logger.}}
    {value: {name: VIDYO_CONNECTORLOGGERTYPE_HTTP}
        {description: The http logger.}}
}
*/
typedef enum
{
    VIDYO_CONNECTORLOGGERTYPE_CONSOLE,
    VIDYO_CONNECTORLOGGERTYPE_FILE,
    VIDYO_CONNECTORLOGGERTYPE_FEEDBACK,
    VIDYO_CONNECTORLOGGERTYPE_HTTP
} VidyoConnectorLoggerType;

/**
 {function apigroup="connector" visibility="public":
    {name: VidyoConnectorSetLogLevel}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Allow application to replace log level of logger(console, file, feedback, http) as production or debug.
        This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.
    }
    {prototype: LmiBool VidyoConnectorSetLogLevel(VidyoConnector *c, VidyoConnectorLoggerType loggerType, VidyoConnectorLogLevel logLevel)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
	{parameter:
        {name: loggerType}
        {description: loggerType console, file, feedback, http.}
	}
    {parameter:
        {name: logLevel}
        {description: log level production or debug.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetLogLevel(VidyoConnector *c, VidyoConnectorLoggerType loggerType, VidyoConnectorLogLevel logLevel);

/**
{function apigroup="connector" visibility="public":
    {name: VidyoConnectorGetLogLevel}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Allow application to get log level i.e production or debug.
        This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return VIDYO_CONNECTORLOGLEVEL_INVALID.
    }
    {prototype: VidyoConnectorLogLevel VidyoConnectorGetLogLevel(VidyoConnector *c, VidyoConnectorLoggerType loggerType)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {parameter:
        {name: loggerType}
        {description: logger type ( console, file, feedback, http).}
    }
    {return: VidyoConnectorLogLevel.}
}
*/
VidyoConnectorLogLevel VidyoConnectorGetLogLevel(VidyoConnector *c, VidyoConnectorLoggerType loggerType);

/**
 {function apigroup="connector":
    {name: VidyoConnectorSetAdvancedLogOptions}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Allow application to set advanced log options in addition to standard ones. The given log levels and categories string will be added to current log level the application is running with. The string should be valid for this API to work correctly. e.g debug@VidyoConnector debug@VidyoClient.}
    {prototype: LmiBool VidyoConnectorSetAdvancedLogOptions(VidyoConnector *c, VidyoConnectorLoggerType loggerType, const LmiString* advancedLogFilter)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
	{parameter:
        {name: loggerType}
        {description: The loggerType console, file, feedback, http.}
	}
    {parameter:
        {name: advancedLogFilter}
        {description: advanced log options in addition to standard ones.
       {p: A space-separated (or comma-separated)
         sequence of names of log levels, each optionally followed by a
         category.  Categories are
         separated from levels by the character \'@\'.}
       {p: Either a level or category may be specified as \'all\' or \'*\'
         (interchangeably), meaning all levels or categories.  A level specified
         without a category is equivalent to a level@*.}
       {p: The levels are: \'fatal\', \'error\', \'warning\', \'info\', \'debug\',
         \'sent\', \'received\', \'enter\', and '\leave\'.}
       {p: The level may be prefixed by \'-\' or \'!\' (interchangeably), meaning to remove
         the given level (or levels, see below) from the given category.
         The string \'none\' is equivalent to \'-all\'.}
       {p: The first level in the list may be prefixed by \'+\'.
         If the first level begins with a \'+\' or \'-\'/\'!\', the string
         is used to modify the listener\'s existing levels and categories.
         Otherwise, unless the {code: add} parameter is true, the levels and categories
          specified in the string replace all existing levels and categories for the listener.
          The leading character \'+\' may be present before other levels, but is ignored.}
       {p: Unless they are preceded by \'=\', level names imply other levels
         as well.  In particular, each of the levels \'fatal\', \'error\', \'warning\',
         \'info\', and \'debug\' implies the levels higher than it, when specified
         positively; \'sent\' and \'received\' always imply each other; and \'enter\' and
         \'leave\' always imply each other.}
       {p: See VidyoConnectorGetLogCategories for determining the names and descriptions of registered categories.}
       {p: Level names are case-insensitive; category names are case-sensitive.}
       {p: Example: "all" - all levels, all categories.}
       {p: Example: "all -enter" - all categories, all levels but enter and leave.}
       {p: Example: "*@VidyoClient" - all levels for the VidyoClient category.}
       {p: Example: "-*@VidyoClient" - remove all levels from the VidyoClient category.}
       {p: Example: "debug" - all categories, fatal, error, warning, info, and debug levels.}
       {p: Example: "=debug" - all categories, debug level only.}
       {p: Example: "all -=enter" - all categories, all levels but enter (leave is still included).}
       {p: Example: "error@VidyoClient" - fatal and error only for the VidyoClient category.}
       {p: Example: "warning debug@VidyoClient" - fatal, error, and warning for all categories; additionally, info and debug for the VidyoClient category.}
    }}
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetAdvancedLogOptions(VidyoConnector *c, VidyoConnectorLoggerType loggerType, const LmiString* advancedLogFilter);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterLogEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about log events.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterLogEventListener(VidyoConnector* c, VidyoConnectorOnLog onLog, const char* filter)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onLog}
		{description: Callback that is triggered for every new log record.}
	}
	{parameter:
		{name: filter}
		{description:
	   {p: A space-separated (or comma-separated)
		 sequence of names of log levels, each optionally followed by a
		 category.  Categories are
		 separated from levels by the character \'@\'.}
	   {p: Either a level or category may be specified as \'all\' or \'*\'
	     (interchangeably), meaning all levels or categories.  A level specified
		 without a category is equivalent to a level@*.}
	   {p: The levels are: \'fatal\', \'error\', \'warning\', \'info\', \'debug\', 
		 \'sent\', \'received\', \'enter\', and '\leave\'.}
	   {p: The level may be prefixed by \'-\' or \'!\' (interchangeably), meaning to remove
		 the given level (or levels, see below) from the given category.
		 The string \'none\' is equivalent to \'-all\'.}
	   {p: The first level in the list may be prefixed by \'+\'.
	     If the first level begins with a \'+\' or \'-\'/\'!\', the string
		 is used to modify the listener\'s existing levels and 45categories.
		 Otherwise, unless the {code: add} parameter is true, the levels and categories 
 		 specified in the string replace all existing levels and categories for the listener.  
 		 The leading character \'+\' may be present before other levels, but is ignored.}
	   {p: Unless they are preceded by \'=\', level names imply other levels
		 as well.  In particular, each of the levels \'fatal\', \'error\', \'warning\',
		 \'info\', and \'debug\' implies the levels higher than it, when specified
		 positively; \'sent\' and \'received\' always imply each other; and \'enter\' and
		 \'leave\' always imply each other.}
	   {p: See VidyoConnectorGetLogCategories for determining the names and descriptions of registered categories.}
	   {p: Level names are case-insensitive; category names are case-sensitive.}
	   {p: Example: "all" - all levels, all categories.}
	   {p: Example: "all -enter" - all categories, all levels but enter and leave.}
	   {p: Example: "*@VidyoClient" - all levels for the VidyoClient category.}
	   {p: Example: "-*@VidyoClient" - remove all levels from the VidyoClient category.}
	   {p: Example: "debug" - all categories, fatal, error, warning, info, and debug levels.}
	   {p: Example: "=debug" - all categories, debug level only.}
	   {p: Example: "all -=enter" - all categories, all levels but enter (leave is still included).}
	   {p: Example: "error@VidyoClient" - fatal and error only for the VidyoClient category.}
	   {p: Example: "warning debug@VidyoClient" - fatal, error, and warning for all categories; additionally, info and debug for the VidyoClient category.}
	}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterLogEventListener(VidyoConnector* c, VidyoConnectorOnLog onLog, const char* filter);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterLogEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters log event notifications. 
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterLogEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterLogEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterLocalCameraEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about local camera events. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterLocalCameraEventListener(VidyoConnector* c, VidyoConnectorOnLocalCameraAdded onAdded, VidyoConnectorOnLocalCameraRemoved onRemoved, VidyoConnectorOnLocalCameraSelected onSelected, VidyoConnectorOnLocalCameraStateUpdated onStateUpdated)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered for every new or existing camera added.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when the camera is removed.}
	}
	{parameter:
		{name: onSelected}
		{description: Callback that is triggered when a camera is selected for use.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when a camera state is updated.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterLocalCameraEventListener(VidyoConnector* c, VidyoConnectorOnLocalCameraAdded onAdded, VidyoConnectorOnLocalCameraRemoved onRemoved, VidyoConnectorOnLocalCameraSelected onSelected, VidyoConnectorOnLocalCameraStateUpdated onStateUpdated);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterLocalCameraEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters local camera event notifications. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterLocalCameraEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterLocalCameraEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterLocalCameraFrameListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about camera frames.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterLocalCameraFrameListener(VidyoConnector* c, VidyoConnectorOnLocalCameraFrame onFrame, const VidyoLocalCamera* localCamera, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered for every frame from a remote camera.}
	}
	{parameter:
		{name: localCamera}
		{description: The VidyoLocalCamera for which to receive the frames.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested frame.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested frame.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested frame.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: The extra parameters are used to determin the steam quality and the resulting frame might not match exactly what is requested.}
}
*/
LmiBool VidyoConnectorRegisterLocalCameraFrameListener(VidyoConnector* c, VidyoConnectorOnLocalCameraFrame onFrame, const VidyoLocalCamera* localCamera, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterLocalCameraFrameListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters local camera frame notifications. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterLocalCameraFrameListener(VidyoConnector* c, const VidyoLocalCamera* localCamera)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: localCamera}
		{description: The VidyoLocalCamera for which to stop notifications.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorUnregisterLocalCameraFrameListener(VidyoConnector* c, const VidyoLocalCamera* localCamera);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnPresenterChanged}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when presenter changed.}
	{prototype: void (*VidyoConnectorOnPresenterChanged)(VidyoConnector *c, const VidyoParticipant* participant)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: participant}
		{description: The current presenter.}
	}
}
*/
typedef void (*VidyoConnectorOnPresenterChanged)(VidyoConnector *c, const VidyoParticipant* participant);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnHandRaised}
	{parent: VidyoConnector}
	{description:Callback that is triggered for the moderator every time someone raises their hand .}
	{prototype: void (*VidyoConnectorOnHandRaised)(VidyoConnector *c, const LmiVector(VidyoParticipant)* participant)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: participant}
		{description: Participants  whose hand was raised}
	}
}
*/
typedef void(*VidyoConnectorOnHandRaised)(VidyoConnector *c, const LmiVector(VidyoParticipant)* participant);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterLectureModeEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers the Lecture mode feature callbacks.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterLectureModeEventListener(VidyoConnector *c, VidyoConnectorOnPresenterChanged presenterChanged, VidyoConnectorOnHandRaised handRaised)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: presenterChanged}
		{description: Callback that is triggered when presenter changed.}
	}
	{parameter:
		{name: handRaised}
		{description: Callback that is triggered for Moderator when some hand raised.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterLectureModeEventListener(VidyoConnector *c, VidyoConnectorOnPresenterChanged presenterChanged, VidyoConnectorOnHandRaised handRaised);


/**
{type visibility="public":
    {name: VidyoConnectorGoogleAnalyticsEventCategory}
    {parent: VidyoConnector}
    {description: Represents the event category of google analytics service.}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTCATEGORY_Login}
        {description: The login event category.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTCATEGORY_UserType}
        {description: The user type event category.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTCATEGORY_JoinConference}
        {description: The join conference event category.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTCATEGORY_ConferenceEnd}
        {description: The conference end event category.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTCATEGORY_InCallCodec}
        {description: The in-call codec event category.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTCATEGORY_None}
        {description: Unknown event category.}}
}
*/
typedef enum {
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTCATEGORY_Login,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTCATEGORY_UserType,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTCATEGORY_JoinConference,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTCATEGORY_ConferenceEnd,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTCATEGORY_InCallCodec,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTCATEGORY_None
} VidyoConnectorGoogleAnalyticsEventCategory;

/**
{type visibility="public":
    {name: VidyoConnectorGoogleAnalyticsEventAction}
    {parent: VidyoConnector}
    {description: Represents the event action of google analytics service.}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginSuccess}
        {description: The event indicates that login was successful.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginAttempt}
        {description: This event indicates a login attempt.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginFailedAuthentication}
        {description: This event indicates that login failed due to authentication.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginFailedConnect}
        {description: This event indicates that login failed due to connect.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginFailedResponseTimeout}
        {description: This event indicates that login failed due to response timeout.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginFailedMiscError}
        {description: This event indicates that login failed due to miscellaneous error.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginFailedWebProxyAuthRequired}
        {description: This event indicates that login failed due to webproxy authentication.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginFailedUnsupportedTenantVersion}
        {description: This event indicates that login failed due to unsupported tenant version.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_UserTypeGuest}
        {description: This event indicates that user type is guest.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_UserTypeRegularToken}
        {description: This event indicates that user type is regular with token.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_UserTypeRegularPassword}
        {description: This event indicates that user type is regular with password.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_UserTypeRegularSaml}
        {description: This event indicates that user type is regular with saml.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_UserTypeRegularExtdata}
        {description: This event indicates that user type is regular with extended data.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceSuccess}
        {description: This event indicates that join conference is successful.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceAttempt}
        {description: This event indicates that join conference attempt is made.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceReconnectRequests}
        {description: This event indicates that join conference with reconnect requested.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedConnectionError}
        {description: This Event indicates that join conference is failed due to connection error.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedWrongPin}
        {description: This event indicates that join conference is failed due to wrong pin.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedRoomFull}
        {description: This event indicates that join conference is failed because room is full.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedRoomDisabled}
        {description: This event indicates that join conference is failed because room is disabled.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedConferenceLocked}
        {description: This event indicates that join conference is failed because conference room is locked.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedUnknownError}
        {description: This event indicates that  join conference is failed due to unknown error.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_ConferenceEndLeft}
        {description: This event indicates that conference is ended because user is left.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_ConferenceEndBooted}
        {description: This event indicates that conference is ended because user is booted.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_ConferenceEndSignalingConnectionLost}
        {description: This event indicates that conference is ended because signaling connection is lost.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_ConferenceEndMediaConnectionLost}
        {description: This event indicates that conference is ended because media connection is lost.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_ConferenceEndUnknownError}
        {description: This event indicates that conference is ended with unknown error.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_InCallCodecVideoH264}
        {description: This event indicates that H264 video codec is used in a conference call.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_InCallCodecVideoH264SVC}
        {description: This event indicates that H264-SVC video codec is used in a conference call.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_InCallCodecAudioSPEEXRED}
        {description: This event indicates that SPEEX RED audio codec is used in a conference call.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_All}
        {description: This event includes all the above mentioned events.}}
    {value:
        {name: VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_Unknown}
        {description: Unknown event action.}}
}
*/
typedef enum
{
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginSuccess,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginAttempt,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginFailedAuthentication,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginFailedConnect,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginFailedResponseTimeout,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginFailedMiscError,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginFailedWebProxyAuthRequired,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_LoginFailedUnsupportedTenantVersion,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_UserTypeGuest,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_UserTypeRegularToken,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_UserTypeRegularPassword,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_UserTypeRegularSaml,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_UserTypeRegularExtdata,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceSuccess,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceAttempt,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceReconnectRequests,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedConnectionError,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedWrongPin,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedRoomFull,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedRoomDisabled,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedConferenceLocked,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_JoinConferenceFailedUnknownError,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_ConferenceEndLeft,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_ConferenceEndBooted,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_ConferenceEndSignalingConnectionLost,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_ConferenceEndMediaConnectionLost,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_ConferenceEndUnknownError,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_InCallCodecVideoH264,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_InCallCodecVideoH264SVC,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_InCallCodecAudioSPEEXRED,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_All,
    VIDYO_CONNECTORGOOGLEANALYTICSEVENTACTION_Unknown
} VidyoConnectorGoogleAnalyticsEventAction;

/**
{type apigroup="connector" visibility="public":
    {name: VidyoConnectorGoogleAnalyticsEventTable}
    {parent: Connector}
    {description: Represents the google analytics event table.}
    {member:
        {name: eventCategory}
        {type: VidyoConnectorGoogleAnalyticsEventCategory}
        {description: Represents the event category of google analytics.}
    }
    {member:
        {name: eventAction}
        {type: VidyoConnectorGoogleAnalyticsEventAction}
        {description: Represents the event action of google analytics.}
    }
    {member:
    {name: enable}
        {type: LmiBool}
        {description: Represents the event action enable flag.}
    }
}
*/
typedef struct
{
    VidyoConnectorGoogleAnalyticsEventCategory eventCategory;
    VidyoConnectorGoogleAnalyticsEventAction eventAction;
    LmiBool enable;
} VidyoConnectorGoogleAnalyticsEventTable;

/**
{function visibility="private":
    {name: VidyoConnectorGoogleAnalyticsEventTableConstructDefault}
    {parent: VidyoConnectorGoogleAnalyticsEventTable}
    {description: Constructs the default VidyoConnectorGoogleAnalyticsEventTable object.}
    {prototype: VidyoConnectorGoogleAnalyticsEventTable* VidyoConnectorGoogleAnalyticsEventTableConstructDefault(VidyoConnectorGoogleAnalyticsEventTable* evantTable, LmiAllocator* alloc)}
    {parameter:
        {name: evantTable}
        {description: The VidyoConnectorGoogleAnalyticsEventTable object.}
    }
    {parameter:
        {name: alloc}
        {description: The LmiAllocator object.}
    }
    {return: The pointer to constructed object on success, or NULL on failure}
}
*/
VidyoConnectorGoogleAnalyticsEventTable* VidyoConnectorGoogleAnalyticsEventTableConstructDefault(VidyoConnectorGoogleAnalyticsEventTable* evantTable, LmiAllocator* alloc);

/**
{function visibility="private":
    {name: VidyoConnectorGoogleAnalyticsEventTableConstructCopy}
    {parent: VidyoConnectorGoogleAnalyticsEventTable}
    {description: Constructs an VidyoConnectorGoogleAnalyticsEventTable object as a copy of an existing object.}
    {prototype: VidyoConnectorGoogleAnalyticsEventTable* VidyoConnectorGoogleAnalyticsEventTableConstructCopy(VidyoConnectorGoogleAnalyticsEventTable* d, const VidyoConnectorGoogleAnalyticsEventTable* s)}
    {parameter:
        {name: d}
        {description: The object to construct.}
    }
    {parameter:
        {name: s}
        {description: The object to make copy of.}
    }
    {return: The pointer to constructed object on success, or NULL on failure}
}
*/
VidyoConnectorGoogleAnalyticsEventTable* VidyoConnectorGoogleAnalyticsEventTableConstructCopy(VidyoConnectorGoogleAnalyticsEventTable* d, const VidyoConnectorGoogleAnalyticsEventTable* s);

/**
{function visibility="private":
    {name: VidyoConnectorGoogleAnalyticsEventTableAssign}
    {parent: VidyoConnectorGoogleAnalyticsEventTable}
    {description: Assigns one VidyoConnectorGoogleAnalyticsEventTable object the value of another. .}
    {prototype: VidyoConnectorGoogleAnalyticsEventTable* VidyoConnectorGoogleAnalyticsEventTableAssign(VidyoConnectorGoogleAnalyticsEventTable* d, const VidyoConnectorGoogleAnalyticsEventTable* s)}
    {parameter:
        {name: d}
        {description: The object to assign to.}
    }
    {parameter:
        {name: s}
        {description: The object to assign from.}
    }
    {return: The pointer to constructed object on success, or NULL on failure}
}
*/
VidyoConnectorGoogleAnalyticsEventTable* VidyoConnectorGoogleAnalyticsEventTableAssign(VidyoConnectorGoogleAnalyticsEventTable* d, const VidyoConnectorGoogleAnalyticsEventTable* s);

/**
{function visibility="private":
    {name: VidyoConnectorGoogleAnalyticsEventTableDestruct}
    {parent: VidyoConnectorGoogleAnalyticsEventTable}
    {description: Destructs an VidyoConnectorGoogleAnalyticsEventTable object.}
    {prototype: void VidyoConnectorGoogleAnalyticsEventTableDestruct(VidyoConnectorGoogleAnalyticsEventTable* eventTable)}
    {parameter:
        {name: eventTable}
        {description: The VidyoConnectorGoogleAnalyticsEventTable object.}
    }
}
*/
void VidyoConnectorGoogleAnalyticsEventTableDestruct(VidyoConnectorGoogleAnalyticsEventTable* eventTable);

Declare_LmiVector(VidyoConnectorGoogleAnalyticsEventTable)

/**
{callback apigroup="simple":
    {name: VidyoConnectorOnGetGoogleAnalyticsEventTable}
    {parent: VidyoConnector}
    {description: Callback that is triggered each time when result of VidyoConnectorGetGoogleAnalyticsEventTable is received.}
    {prototype: void(*VidyoConnectorOnGetGoogleAnalyticsEventTable)(VidyoConnector* c, const LmiVector(VidyoConnectorGoogleAnalyticsEventTable)* eventTable)}
    {parameter:
        {name: c}
        {description: Pointer to the VidyoConnector that triggered the callback.}}
    {parameter:
        {name: eventTable}
        {description: Vector of VidyoConnectorGoogleAnalyticsEventTable object.}}
}
*/
typedef void(*VidyoConnectorOnGetGoogleAnalyticsEventTable)(VidyoConnector* c, const LmiVector(VidyoConnectorGoogleAnalyticsEventTable)* eventTable);

/**
{function apigroup="simple":
    {name: VidyoConnectorGetGoogleAnalyticsEventTable}
    {parent: VidyoConnector}
    {description: This is an asynchronous interface. Get the analytics event table.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorGetGoogleAnalyticsEventTable(VidyoConnector *c, VidyoConnectorOnGetGoogleAnalyticsEventTable onGetGoogleAnalyticsEventTable)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}}
    {parameter:
        {name: onGetGoogleAnalyticsEventTable}
        {description: Callback called upon completion.}}
    {return: LMI_TRUE if the request has been scheduled successfully, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorGetGoogleAnalyticsEventTable(VidyoConnector *c, VidyoConnectorOnGetGoogleAnalyticsEventTable onGetGoogleAnalyticsEventTable);

/**
{function apigroup="simple":
	{name: VidyoConnectorGoogleAnalyticsControlEventAction}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Control type of event action and categories of google analytics.
			This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorGoogleAnalyticsControlEventAction(VidyoConnector *c, VidyoConnectorGoogleAnalyticsEventCategory eventCategory, VidyoConnectorGoogleAnalyticsEventAction eventAction, LmiBool enable)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{parameter:
		{name: eventCategory}
		{description: Type of event category. refer VidyoConnectorGoogleAnalyticsEventCategory.}}
	{parameter:
		{name: eventAction}
		{description: Type of event action. refer VidyoConnectorGoogleAnalyticsEventAction.}}
	{parameter:
		{name: enable}
		{description: Event action state flag.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorGoogleAnalyticsControlEventAction(VidyoConnector *c, VidyoConnectorGoogleAnalyticsEventCategory eventCategory, VidyoConnectorGoogleAnalyticsEventAction eventAction, LmiBool enable);

/**
{type apigroup="connector" visibility="public":
    {name:  VidyoConnectorGoogleAnalyticsOptions}
    {parent: Connector}
    {description: Represent the options for google analytics.}
    {member: {name: id} {type: LmiString} {description: Represents the measurement is for google analytics service.}}
    {member: {name: key} {type: LmiString} {description: Represents the configured API secret key for google analytics service.}}
}
*/
typedef struct
{
    LmiString id;
    LmiString key;
} VidyoConnectorGoogleAnalyticsOptions;

/**
{function visibility="private":
    {name: VidyoConnectorGoogleAnalyticsOptionsConstructDefault}
    {parent: VidyoConnectorGoogleAnalyticsOptions}
    {description: Constructs the default VidyoConnectorGoogleAnalyticsOptions object.}
    {prototype: VidyoConnectorGoogleAnalyticsOptions* VidyoConnectorGoogleAnalyticsOptionsConstructDefault(VidyoConnectorGoogleAnalyticsOptions* options, LmiAllocator* alloc)}
    {parameter:
        {name: options}
        {description: Represents the VidyoConnectorGoogleAnalyticsOptions object to construct.}
    }
    {parameter:
        {name: alloc}
        {description: Represents the LmiAllocator object.}
    }
    {return: The pointer to constructed object on success, or NULL on failure}
}
*/
VidyoConnectorGoogleAnalyticsOptions* VidyoConnectorGoogleAnalyticsOptionsConstructDefault(VidyoConnectorGoogleAnalyticsOptions* options, LmiAllocator* alloc);

/**
{function visibility="private":
    {name: VidyoConnectorGoogleAnalyticsOptionsConstructCopy}
    {parent: VidyoConnectorGoogleAnalyticsOptions}
    {description: Constructs an VidyoConnectorGoogleAnalyticsOptions object as a copy of an existing object.}
    {prototype: VidyoConnectorGoogleAnalyticsOptions* VidyoConnectorGoogleAnalyticsOptionsConstructCopy(VidyoConnectorGoogleAnalyticsOptions* d, const VidyoConnectorGoogleAnalyticsOptions* s)}
    {parameter:
        {name: d}
        {description: The object to construct.}
    }
    {parameter:
        {name: s}
        {description: The object to make copy of.}
    }
    {return: The pointer to constructed object on success, or NULL on failure}
}
*/
VidyoConnectorGoogleAnalyticsOptions* VidyoConnectorGoogleAnalyticsOptionsConstructCopy(VidyoConnectorGoogleAnalyticsOptions* d, const VidyoConnectorGoogleAnalyticsOptions* s);

/**
{function visibility="private":
    {name: VidyoConnectorGoogleAnalyticsOptionsAssign}
    {parent: VidyoConnectorGoogleAnalyticsOptions}
    {description: Assigns one VidyoConnectorGoogleAnalyticsOptions object the value of another.}
    {prototype: VidyoConnectorGoogleAnalyticsOptions* VidyoConnectorGoogleAnalyticsOptionsAssign(VidyoConnectorGoogleAnalyticsOptions* d, const VidyoConnectorGoogleAnalyticsOptions* s)}
    {parameter:
        {name: d}
        {description: The object to assign to.}
    }
    {parameter:
        {name: s}
        {description: The object to assign from.}
    }
    {return: The pointer to constructed object on success, or NULL on failure}
}
*/
VidyoConnectorGoogleAnalyticsOptions* VidyoConnectorGoogleAnalyticsOptionsAssign(VidyoConnectorGoogleAnalyticsOptions* d, const VidyoConnectorGoogleAnalyticsOptions* s);

/**
{function visibility="private":
    {name: VidyoConnectorGoogleAnalyticsOptionsDestruct}
    {parent: VidyoConnectorGoogleAnalyticsOptions}
    {description: Destructs an VidyoConnectorGoogleAnalyticsOptions object.}
    {prototype: void VidyoConnectorGoogleAnalyticsOptionsDestruct(VidyoConnectorGoogleAnalyticsOptions* options)}
    {parameter:
        {name: options}
        {description: The VidyoConnectorGoogleAnalyticsOptions object.}
    }
}
*/
void VidyoConnectorGoogleAnalyticsOptionsDestruct(VidyoConnectorGoogleAnalyticsOptions* options);

/**
{callback apigroup="simple":
    {name: VidyoConnectorOnGetGoogleAnalyticsOptions}
    {parent: VidyoConnector}
    {description: Callback that is triggered each time when result of VidyoConnectorGetGoogleAnalyticsOptions is received.}
    {prototype: void(*VidyoConnectorOnGetGoogleAnalyticsOptions)(VidyoConnector* c, const VidyoConnectorGoogleAnalyticsOptions* options)}
    {parameter:
        {name: c}
        {description: Pointer to the VidyoConnector that triggered the callback.}}
    {parameter:
        {name: options}
        {description: Represents the VidyoConnectorGoogleAnalyticsOptions object.}}
}
*/
typedef void(*VidyoConnectorOnGetGoogleAnalyticsOptions)(VidyoConnector* c, const VidyoConnectorGoogleAnalyticsOptions* options);

/**
{function apigroup="connector":
	{name: VidyoConnectorStartGoogleAnalyticsService}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Start google analytics service.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorStartGoogleAnalyticsService(VidyoConnector* c, const VidyoConnectorGoogleAnalyticsOptions* options)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{parameter:
		{name: options}
		{description: A structure that represent the google analytics options, plz refer VidyoConnectorGoogleAnalyticsOptions. If the user set to NULL, Vidyo Client will use default option to use google analytics.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorStartGoogleAnalyticsService(VidyoConnector* c, const VidyoConnectorGoogleAnalyticsOptions* options);

/**
{function apigroup="connector":
	{name: VidyoConnectorStopGoogleAnalyticsService}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Stop google analytics service.
			This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorStopGoogleAnalyticsService(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorStopGoogleAnalyticsService(VidyoConnector* c);

/**
{function apigroup="connector":
	{name: VidyoConnectorIsGoogleAnalyticsServiceEnabled}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Checks whether Google Analytics service is running.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorIsGoogleAnalyticsServiceEnabled(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorIsGoogleAnalyticsServiceEnabled(VidyoConnector* c);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetGoogleAnalyticsOptions}
	{parent: VidyoConnector}
	{description: This is a asynchronous interface. It returns google analytics configured options.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorGetGoogleAnalyticsOptions(VidyoConnector* c, VidyoConnectorOnGetGoogleAnalyticsOptions onGetGoogleAnalyticsOptions)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{parameter:
		{name: onGetGoogleAnalyticsOptions}
		{description: Callback called upon completion.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorGetGoogleAnalyticsOptions(VidyoConnector* c, VidyoConnectorOnGetGoogleAnalyticsOptions onGetGoogleAnalyticsOptions);

/**
{function apigroup="connector":
	{name: VidyoConnectorInsightsNotifyEvent}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Notify the events as a part of analytics service. The event data will be sent to Loki analytics server if analytics service is started with VIDYO_CONNECTORANALYTICSSERVICETYPE_VidyoInsights. Event is automatically timestamped.
		This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorInsightsNotifyEvent(VidyoConnector* c, const char* eventName, const LmiVector(LmiString)* parameters)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{parameter:
		{name: eventName}
		{description: Event name.}}
	{parameter:
		{name: parameters}
		{description: Pointer to string vector event parameters. Can be NULL if no parameters are needed.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorInsightsNotifyEvent(VidyoConnector* c, const char* eventName, const LmiVector(LmiString)* parameters);

/**
{function apigroup="connector":
	{name: VidyoConnectorStartInsightsService}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Start insights service.
		This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorStartInsightsService(VidyoConnector* c, const LmiString* serverUrl)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{parameter:
		{name: serverUrl}
		{description: Url of Insights server.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorStartInsightsService(VidyoConnector* c, const LmiString* serverUrl);

/**
{function apigroup="connector":
	{name: VidyoConnectorStopInsightsService}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Stop insights service.
		This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorStopInsightsService(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorStopInsightsService(VidyoConnector* c);

/**
{function apigroup="connector":
	{name: VidyoConnectorIsInsightsServiceEnabled}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Checks whether Insights service is running.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorIsInsightsServiceEnabled(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorIsInsightsServiceEnabled(VidyoConnector* c);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetInsightsServiceUrl}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. It returns Insights Url if service is running.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: const LmiString* VidyoConnectorGetInsightsServiceUrl(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{return: Insights Url on success, NULL on failure.}
}
*/
const LmiString* VidyoConnectorGetInsightsServiceUrl(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterLectureModeEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters the Lecture mode part callbacks.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterLectureModeEventListener(VidyoConnector *c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorUnregisterLectureModeEventListener(VidyoConnector *c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorStartLectureMode}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Switch room mode to the Lecture mode. 
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorStartLectureMode(VidyoConnector *c, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorStartLectureMode(VidyoConnector *c, const LmiString* requestId);

/**
 {function apigroup="connector":
	{name: VidyoConnectorStopLectureMode}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Switch room mode to the Group mode.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorStopLectureMode(VidyoConnector *c, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorStopLectureMode(VidyoConnector *c, const LmiString* requestId);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSetPresenter}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Set a specific participant as a presenter.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetPresenter(VidyoConnector *c, const VidyoParticipant* participant, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: participant}
		{description: The VidyoParticipant object who will be a new presenter.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetPresenter(VidyoConnector *c, const VidyoParticipant* participant, const LmiString* requestId);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRemovePresenter}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Remove presenter rights from a specific participant.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRemovePresenter(VidyoConnector *c, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRemovePresenter(VidyoConnector *c, const LmiString* requestId);

/**
 {callback apigroup="connector":
	{name: VidyoConnectorOnRaiseHandResponse}
	{parent: VidyoConnector}
	{description: Callback that notifies if raise hand request was approved or dismissed.}
	{prototype: void (*VidyoConnectorOnRaiseHandResponse)(VidyoConnector *c, const VidyoParticipantHandState handState)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: handState}
		{description: State of the VidyoParticipantHandState.}
	}
}
*/
typedef void (*VidyoConnectorOnRaiseHandResponse)(VidyoConnector *c, const VidyoParticipantHandState handState);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRaiseHand}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Each participant can send a moderation request to raise their hand to request to be unmuted. The Moderator and Presenter will be notified.}
	{prototype: LmiBool VidyoConnectorRaiseHand(VidyoConnector *c, VidyoConnectorOnRaiseHandResponse raiseHandResponse, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: raiseHandResponse}
		{description: Callback that is triggered each time when response of hand raise request.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRaiseHand(VidyoConnector *c, VidyoConnectorOnRaiseHandResponse raiseHandResponse, const LmiString* requestId);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnraiseHand}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Each participant can send a moderation request to remove his raised hand to cancel the request to be unmuted.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnraiseHand(VidyoConnector *c, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnraiseHand(VidyoConnector *c, const LmiString* requestId);

/**
 {function apigroup="connector":
	{name: VidyoConnectorApproveRaisedHand}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. The Moderator can send a moderation request to approve a raised hand, in which case the participant whose hand was raised will be notified.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorApproveRaisedHand(VidyoConnector *c, const VidyoParticipant* participant, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: participant}
		{description: Selected participant whose raised hand request is approved.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorApproveRaisedHand(VidyoConnector *c, const VidyoParticipant* participant, const LmiString* requestId);

/**
 {function apigroup="connector":
	{name: VidyoConnectorDismissRaisedHand}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. The Moderator or the Presenter can send a moderation request to decline a raised hands, in which case the participant whose hand was raised will be notified.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDismissRaisedHand(VidyoConnector *c, const LmiVector(VidyoParticipant)* participants, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: participants}
		{description: List of selected participants whose raised hand request is going to dismiss.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorDismissRaisedHand(VidyoConnector *c, const LmiVector(VidyoParticipant)* participants, const LmiString* requestId);

/**
 {function apigroup="connector":
	{name: VidyoConnectorDismissAllRaisedHands}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. The Moderator or the Presenter can send a moderation request to reject all hands raised, in which case all participants whose hand was raised will be notified.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDismissAllRaisedHands(VidyoConnector *c, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorDismissAllRaisedHands(VidyoConnector *c, const LmiString* requestId);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSelectLocalCamera}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Selects the local camera to use. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSelectLocalCamera(VidyoConnector* c, const VidyoLocalCamera* localCamera)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: localCamera}
		{description: Local camera to select. NULL if none of them should be selected.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectLocalCamera(VidyoConnector* c, const VidyoLocalCamera* localCamera);

/**
 {function apigroup="connector":
	 {name: VidyoConnectorSelectVideoContentShare}
	 {parent: VidyoConnector}
	 {description: This is an asynchronous interface. Selects the local camera to use for video content share. On, iOS, this API must be called from main(UI) thread.
	 				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	 {prototype: LmiBool VidyoConnectorSelectVideoContentShare(VidyoConnector* c, const VidyoLocalCamera* localCamera)}
	 {parameter:
		 {name: c}
		 {description: The VidyoConnector object.}
	 }
	 {parameter:
		 {name: localCamera}
		 {description: Local camera to select. NULL if none of them should be selected.}
	 }
	 {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorSelectVideoContentShare(VidyoConnector* c, const VidyoLocalCamera* localCamera);

/**
 {function apigroup="connector":
	{name: VidyoConnectorAssignViewToLocalCamera}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Assigns the view in which to show the preview from a local camera. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorAssignViewToLocalCamera(VidyoConnector* c, const LmiViewId* viewId, const VidyoLocalCamera* localCamera, LmiBool displayCropped, LmiBool allowZoom)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID where the participant will be rendered.}
	}
	{parameter:
		{name: localCamera}
		{description: The VidyoLocalCamera that will be shown.}
	}
	{parameter:
		{name: displayCropped}
		{description: Render the image as cropped. Otherwise letterbox.}
	}
	{parameter:
		{name: allowZoom}
		{description: Allow zooming of the image with mouse or touch.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: This API will work only if compositing had been disabled by passing a NULL LmiViewId to VidyoConnectorConstruct.}
	{note: The view will not be shown until VidyoConnectorShowViewAt is called with valid parameters.}
}
*/
LmiBool VidyoConnectorAssignViewToLocalCamera(VidyoConnector* c, const LmiViewId* viewId, const VidyoLocalCamera* localCamera, LmiBool displayCropped, LmiBool allowZoom);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterLocalMicrophoneEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about local microphone events. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterLocalMicrophoneEventListener(VidyoConnector* c, VidyoConnectorOnLocalMicrophoneAdded onAdded, VidyoConnectorOnLocalMicrophoneRemoved onRemoved, VidyoConnectorOnLocalMicrophoneSelected onSelected, VidyoConnectorOnLocalMicrophoneStateUpdated onStateUpdated)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered for every new or existing microphone added.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when the microphone is removed.}
	}
	{parameter:
		{name: onSelected}
		{description: Callback that is triggered when a microphone is selected for use.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when a microphone state is updated.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
*/
LmiBool VidyoConnectorRegisterLocalMicrophoneEventListener(VidyoConnector* c, VidyoConnectorOnLocalMicrophoneAdded onAdded, VidyoConnectorOnLocalMicrophoneRemoved onRemoved, VidyoConnectorOnLocalMicrophoneSelected onSelected, VidyoConnectorOnLocalMicrophoneStateUpdated onStateUpdated);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterLocalMicrophoneEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters local microphone event notifications. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterLocalMicrophoneEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterLocalMicrophoneEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterLocalMicrophoneFrameListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about microphone frames. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterLocalMicrophoneFrameListener(VidyoConnector* c, VidyoConnectorOnLocalMicrophoneFrame onFrame, const VidyoLocalMicrophone* localMicrophone)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered for every frame from a microphone.}
	}
	{parameter:
		{name: localMicrophone}
		{description: The VidyoLocalMicrophone for which to receive the frames.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterLocalMicrophoneFrameListener(VidyoConnector* c, VidyoConnectorOnLocalMicrophoneFrame onFrame, const VidyoLocalMicrophone* localMicrophone);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterLocalMicrophoneFrameListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters local microphone frame notifications. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterLocalMicrophoneFrameListener(VidyoConnector* c, const VidyoLocalMicrophone* localMicrophone)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: localMicrophone}
		{description: The VidyoLocalMicrophone for which to stop notifications.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorUnregisterLocalMicrophoneFrameListener(VidyoConnector* c, const VidyoLocalMicrophone* localMicrophone);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterLocalMicrophoneEnergyListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about microphone energies.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterLocalMicrophoneEnergyListener(VidyoConnector* c, VidyoConnectorOnLocalMicrophoneEnergy onEnergy)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onEnergy}
		{description: Callback that is triggered for every frame from a microphone that has energy.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterLocalMicrophoneEnergyListener(VidyoConnector* c, VidyoConnectorOnLocalMicrophoneEnergy onEnergy);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterLocalMicrophoneEnergyListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters local microphone energy notifications.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterLocalMicrophoneEnergyListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorUnregisterLocalMicrophoneEnergyListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSelectLocalMicrophone}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Selects the local microphone to use. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSelectLocalMicrophone(VidyoConnector* c, const VidyoLocalMicrophone* localMicrophone)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: localMicrophone}
		{description: Local microphone to select. NULL if none of them should be selected.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectLocalMicrophone(VidyoConnector* c, const VidyoLocalMicrophone* localMicrophone);

/**
 {function apigroup="connector":
	 {name: VidyoConnectorSelectAudioContentShare}
	 {parent: VidyoConnector}
	 {description: This is an asynchronous interface. Selects the local microphone to use for audio content share. On, iOS, this API must be called from main(UI) thread.
	 				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	 {prototype: LmiBool VidyoConnectorSelectAudioContentShare(VidyoConnector* c, const VidyoLocalMicrophone* localMicrophone)}
	 {parameter:
		 {name: c}
		 {description: The VidyoConnector object.}
	 }
	 {parameter:
		 {name: localMicrophone}
		 {description: Local microphone to select for audio content share. NULL if none of them should be selected.}
	 }
	 {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorSelectAudioContentShare(VidyoConnector* c, const VidyoLocalMicrophone* localMicrophone);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterLocalSpeakerEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about local speaker events. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterLocalSpeakerEventListener(VidyoConnector* c, VidyoConnectorOnLocalSpeakerAdded onAdded, VidyoConnectorOnLocalSpeakerRemoved onRemoved, VidyoConnectorOnLocalSpeakerSelected onSelected, VidyoConnectorOnLocalSpeakerStateUpdated onStateUpdated)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered for every new or existing speaker added.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when the speaker is removed.}
	}
	{parameter:
		{name: onSelected}
		{description: Callback that is triggered when a speaker is selected for use.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when a speaker state is updated.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterLocalSpeakerEventListener(VidyoConnector* c, VidyoConnectorOnLocalSpeakerAdded onAdded, VidyoConnectorOnLocalSpeakerRemoved onRemoved, VidyoConnectorOnLocalSpeakerSelected onSelected, VidyoConnectorOnLocalSpeakerStateUpdated onStateUpdated);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterLocalSpeakerEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters local speaker event notifications. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterLocalSpeakerEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterLocalSpeakerEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSelectLocalSpeaker}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Selects the local speaker to use. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSelectLocalSpeaker(VidyoConnector* c, const VidyoLocalSpeaker* localSpeaker)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: localSpeaker}
		{description: Local speaker to select. NULL if none of them should be selected.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectLocalSpeaker(VidyoConnector* c, const VidyoLocalSpeaker* localSpeaker);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterLocalMonitorEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about local monitor events. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterLocalMonitorEventListener(VidyoConnector* c, VidyoConnectorOnLocalMonitorAdded onAdded, VidyoConnectorOnLocalMonitorRemoved onRemoved, VidyoConnectorOnLocalMonitorSelected onSelected, VidyoConnectorOnLocalMonitorStateUpdated onStateUpdated)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered for every new or existing monitor added.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when the monitor is removed.}
	}
	{parameter:
		{name: onSelected}
		{description: Callback that is triggered when a monitor is selected for use.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when a monitor state is updated.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterLocalMonitorEventListener(VidyoConnector* c, VidyoConnectorOnLocalMonitorAdded onAdded, VidyoConnectorOnLocalMonitorRemoved onRemoved, VidyoConnectorOnLocalMonitorSelected onSelected, VidyoConnectorOnLocalMonitorStateUpdated onStateUpdated);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterLocalMonitorEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters local monitor event notifications. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterLocalMonitorEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterLocalMonitorEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterLocalMonitorFrameListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about monitor frames. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterLocalMonitorFrameListener(VidyoConnector* c, VidyoConnectorOnLocalMonitorFrame onFrame, const VidyoLocalMonitor* localMonitor, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered for every frame from a monitor.}
	}
	{parameter:
		{name: localMonitor}
		{description: The VidyoLocalMonitor for which to receive the frames.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested frame.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested frame.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested frame.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterLocalMonitorFrameListener(VidyoConnector* c, VidyoConnectorOnLocalMonitorFrame onFrame, const VidyoLocalMonitor* localMonitor, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterLocalMonitorFrameListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters local monitor frame notifications. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterLocalMonitorFrameListener(VidyoConnector* c, const VidyoLocalMonitor* localMonitor)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: localMonitor}
		{description: The VidyoLocalMonitor for which to stop notifications.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorUnregisterLocalMonitorFrameListener(VidyoConnector* c, const VidyoLocalMonitor* localMonitor);

/**
 {function apigroup="connector":
	{name: VidyoConnectorAssignViewToLocalMonitor}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Shows a VidyoLocalMonitor in a view. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorAssignViewToLocalMonitor(VidyoConnector* c, const LmiViewId* viewId, VidyoLocalMonitor* localMonitor, LmiBool displayCropped, LmiBool allowZoom)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID where the VidyoLocalMonitor will be rendered.}
	}
	{parameter:
		{name: localMonitor}
		{description: The VidyoLocalMonitor that will be shown.}
	}
	{parameter:
		{name: displayCropped}
		{description: Render the image as cropped. Otherwise letterbox.}
	}
	{parameter:
		{name: allowZoom}
		{description: Allow zooming of the image with mouse or touch.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: This API will work only if compositing had been disabled by passing a NULL LmiViewId to VidyoConnectorConstruct.}
	{note: The view will not be shown until VidyoConnectorShowViewAt is called with valid parameters.}
}
*/
LmiBool VidyoConnectorAssignViewToLocalMonitor(VidyoConnector* c, const LmiViewId* viewId, VidyoLocalMonitor* localMonitor, LmiBool displayCropped, LmiBool allowZoom);

/**
 {function apigroup="connector" deprecated="yes" replaced-by="VidyoConnectorSelectLocalMonitorAdvanced":
	{name: VidyoConnectorSelectLocalMonitor}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Selects the local monitor to share in a conference. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSelectLocalMonitor(VidyoConnector* c, const VidyoLocalMonitor *localMonitor)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: localMonitor}
		{description: Local monitor to share in a conference. NULL if none of them should be shared.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectLocalMonitor(VidyoConnector* c, const VidyoLocalMonitor *localMonitor);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterLocalWindowShareEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about local shareable-window events. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterLocalWindowShareEventListener(VidyoConnector* c, VidyoConnectorOnLocalWindowShareAdded onAdded, VidyoConnectorOnLocalWindowShareRemoved onRemoved, VidyoConnectorOnLocalWindowShareSelected onSelected, VidyoConnectorOnLocalWindowShareStateUpdated onStateUpdated)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered for every new or existing window added.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when the window is removed.}
	}
	{parameter:
		{name: onSelected}
		{description: Callback that is triggered when a window is selected for use.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when a window state is updated.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterLocalWindowShareEventListener(VidyoConnector* c, VidyoConnectorOnLocalWindowShareAdded onAdded, VidyoConnectorOnLocalWindowShareRemoved onRemoved, VidyoConnectorOnLocalWindowShareSelected onSelected, VidyoConnectorOnLocalWindowShareStateUpdated onStateUpdated);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterLocalWindowShareEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters local window share event notifications. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterLocalWindowShareEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterLocalWindowShareEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterLocalWindowShareFrameListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about window frames. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterLocalWindowShareFrameListener(VidyoConnector* c, VidyoConnectorOnLocalWindowShareFrame onFrame, const VidyoLocalWindowShare* localWindowShare, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered for every frame from a window.}
	}
	{parameter:
		{name: localWindowShare}
		{description: The VidyoLocalWindowShare for which to receive the frames.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested frame.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested frame.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested frame.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterLocalWindowShareFrameListener(VidyoConnector* c, VidyoConnectorOnLocalWindowShareFrame onFrame, const VidyoLocalWindowShare* localWindowShare, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterLocalWindowShareFrameListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters local window share frame notifications. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterLocalWindowShareFrameListener(VidyoConnector* c, const VidyoLocalWindowShare* localWindowShare)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: localWindowShare}
		{description: The VidyoLocalWindowShare for which to stop notifications.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorUnregisterLocalWindowShareFrameListener(VidyoConnector* c, const VidyoLocalWindowShare* localWindowShare);

/**
 {function apigroup="connector":
	{name: VidyoConnectorAssignViewToLocalWindowShare}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Shows a VidyoLocalWindowShare in a view. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorAssignViewToLocalWindowShare(VidyoConnector* c, const LmiViewId* viewId, VidyoLocalWindowShare* localWindowShare, LmiBool displayCropped, LmiBool allowZoom)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID where the VidyoLocalWindowShare will be rendered.}
	}
	{parameter:
		{name: localWindowShare}
		{description: The VidyoLocalWindowShare that will be shown.}
	}
	{parameter:
		{name: displayCropped}
		{description: Render the image as cropped. Otherwise letterbox.}
	}
	{parameter:
		{name: allowZoom}
		{description: Allow zooming of the image with mouse or touch.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: This API will work only if compositing had been disabled by passing a NULL LmiViewId to VidyoConnectorConstruct.}
	{note: The view will not be shown until VidyoConnectorShowViewAt is called with valid parameters.}
}
*/
LmiBool VidyoConnectorAssignViewToLocalWindowShare(VidyoConnector* c, const LmiViewId* viewId, VidyoLocalWindowShare* localWindowShare, LmiBool displayCropped, LmiBool allowZoom);

/**
 {function apigroup="connector" deprecated="yes" replaced-by="VidyoConnectorSelectLocalWindowShareAdvanced":
	{name: VidyoConnectorSelectLocalWindowShare}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Selects the local window to share in a conference. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSelectLocalWindowShare(VidyoConnector* c, const VidyoLocalWindowShare* localWindowShare)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: localWindowShare}
		{description: Local window to share in a conference. NULL if none of them should be shared.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectLocalWindowShare(VidyoConnector* c, const VidyoLocalWindowShare* localWindowShare);

/**
{function apigroup="connector":
	{name: VidyoConnectorShowWindowSharePreview}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets the flag to show/hide the window share preview. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorShowWindowSharePreview(VidyoConnector* c, LmiBool preview)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: preview}
		{description: Flag to show/hide the window share preview.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorShowWindowSharePreview(VidyoConnector* c, LmiBool preview);

/**
 {function apigroup="connector":
	{name: VidyoConnectorShowPreview}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets the flag to show/hide the local camera preview for composite layout only. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorShowPreview(VidyoConnector* c, LmiBool preview)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: preview}
		{description: Flag to show/hide the local camera preview.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorShowPreview(VidyoConnector* c, LmiBool preview);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSetCameraPrivacy}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Sets the flag to show/hide the local camera for remote participants for composite/custom layout and local camera preview for composite layout only. Will also unmute a soft mute(Hunter only). On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetCameraPrivacy(VidyoConnector* c, LmiBool privacy)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: privacy}
		{description: Flag to show/hide the local camera.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetCameraPrivacy(VidyoConnector* c, LmiBool privacy);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSelectDefaultCamera}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Resets the camera selection to pick the Default camera. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSelectDefaultCamera(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectDefaultCamera(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorCycleCamera}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Cycles camera selection to the next available camera. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorCycleCamera(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorCycleCamera(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSetMicrophonePrivacy}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Sets the flag to mute/unmute the microphone. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetMicrophonePrivacy(VidyoConnector* c, LmiBool privacy)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: privacy}
		{description: Flag to mute/unmute the microphone.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetMicrophonePrivacy(VidyoConnector* c, LmiBool privacy);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSelectDefaultMicrophone}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Resets the microphone selection to pick the Default microphone. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSelectDefaultMicrophone(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectDefaultMicrophone(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorCycleMicrophone}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Cycles microphone selection to the next available microphone. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorCycleMicrophone(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorCycleMicrophone(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSetSpeakerPrivacy}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Sets the flag to mute/unmute the speaker. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetSpeakerPrivacy(VidyoConnector* c, LmiBool privacy)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: privacy}
		{description: Flag to mute/unmute the speaker.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetSpeakerPrivacy(VidyoConnector* c, LmiBool privacy);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSelectDefaultSpeaker}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Resets the speaker selection to pick the Default speaker. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSelectDefaultSpeaker(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectDefaultSpeaker(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorCycleSpeaker}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Cycles speaker selection to the next available speaker. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorCycleSpeaker(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorCycleSpeaker(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterParticipantEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about participant events.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterParticipantEventListener(VidyoConnector* c, VidyoConnectorOnParticipantJoined onJoined, VidyoConnectorOnParticipantLeft onLeft, VidyoConnectorOnDynamicParticipantChanged onDynamicChanged, VidyoConnectorOnLoudestParticipantChanged onLoudestChanged)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onJoined}
		{description: Callback that is triggered when another participant joins a conference.}
	}
	{parameter:
		{name: onLeft}
		{description: Callback that is triggered when an existing participant leaves a conference.}
	}
	{parameter:
		{name: onDynamicChanged}
		{description: Callback that is triggered when the order of participants has changed, based on their importance according to active speech detection.}
	}
	{parameter:
		{name: onLoudestChanged}
		{description: Callback that is triggered when a new participant becomes the loudest, based on active speech detection.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterParticipantEventListener(VidyoConnector* c, VidyoConnectorOnParticipantJoined onJoined, VidyoConnectorOnParticipantLeft onLeft, VidyoConnectorOnDynamicParticipantChanged onDynamicChanged, VidyoConnectorOnLoudestParticipantChanged onLoudestChanged);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterParticipantEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters participant event notifications.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterParticipantEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterParticipantEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterMessageEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about message events.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterMessageEventListener(VidyoConnector* c, VidyoConnectorOnChatMessageReceived onChatMessageReceived)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onChatMessageReceived}
		{description: Callback that is triggered when a chat message is received from a participant.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterMessageEventListener(VidyoConnector* c, VidyoConnectorOnChatMessageReceived onChatMessageReceived);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterMessageEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters message event notifications.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterMessageEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterMessageEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterRemoteCameraEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about remote camera events.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterRemoteCameraEventListener(VidyoConnector* c, VidyoConnectorOnRemoteCameraAdded onAdded, VidyoConnectorOnRemoteCameraRemoved onRemoved, VidyoConnectorOnRemoteCameraStateUpdated onStateUpdated)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered each time one of the other participants adds a remote camera into a conference.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered each time one of the other participants removes a remote camera from a conference.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered each time one of the other participants remote camera chagnes state in a conference.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterRemoteCameraEventListener(VidyoConnector* c, VidyoConnectorOnRemoteCameraAdded onAdded, VidyoConnectorOnRemoteCameraRemoved onRemoved, VidyoConnectorOnRemoteCameraStateUpdated onStateUpdated);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterRemoteCameraEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters remote camera event notifications.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterRemoteCameraEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterRemoteCameraEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterRemoteCameraFrameListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about camera frames from other participants.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterRemoteCameraFrameListener(VidyoConnector* c, VidyoConnectorOnRemoteCameraFrame onFrame, const VidyoRemoteCamera* remoteCamera, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered for every frame from a remote camera.}
	}
	{parameter:
		{name: remoteCamera}
		{description: The VidyoRemoteCamera for which to receive the frames.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested frame.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested frame.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested frame.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: The extra parameters are used to determin the steam quality and the resulting frame might not match exactly what is requested.}
}
*/
LmiBool VidyoConnectorRegisterRemoteCameraFrameListener(VidyoConnector* c, VidyoConnectorOnRemoteCameraFrame onFrame, const VidyoRemoteCamera* remoteCamera, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterRemoteCameraFrameListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters remote camera frame notifications.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterRemoteCameraFrameListener(VidyoConnector* c, const VidyoRemoteCamera* remoteCamera)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: remoteCamera}
		{description: The VidyoRemoteCamera for which to stop notifications.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorUnregisterRemoteCameraFrameListener(VidyoConnector* c, const VidyoRemoteCamera* remoteCamera);

/**
 {function apigroup="connector":
	{name: VidyoConnectorAssignViewToRemoteCamera}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Assigns the view in which to show the remote camera of one of the other participants.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorAssignViewToRemoteCamera(VidyoConnector* c, const LmiViewId* viewId, const VidyoRemoteCamera* remoteCamera, LmiBool displayCropped, LmiBool allowZoom)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID where the participant will be rendered.}
	}
	{parameter:
		{name: remoteCamera}
		{description: The VidyoRemoteCamera that will be shown.}
	}
	{parameter:
		{name: displayCropped}
		{description: Render the image as cropped. Otherwise letterbox.}
	}
	{parameter:
		{name: allowZoom}
		{description: Allow zooming of the image with mouse or touch.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: This API will work only if compositing had been disabled by passing a NULL LmiViewId to VidyoConnectorConstruct.}
	{note: The application needs to make sure the sources are hidden when the local CPU and bandwidth are limited, by monitoring the VidyoConnectorOnResourcesChanged callback and showing/hiding sources based on avaialble resources.}
	{note: The view will not be shown until VidyoConnectorShowViewAt is called with valid parameters.}
}
*/
LmiBool VidyoConnectorAssignViewToRemoteCamera(VidyoConnector* c, const LmiViewId* viewId, const VidyoRemoteCamera* remoteCamera, LmiBool displayCropped, LmiBool allowZoom);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterRemoteMicrophoneEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about remote microphone events. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterRemoteMicrophoneEventListener(VidyoConnector* c, VidyoConnectorOnRemoteMicrophoneAdded onAdded, VidyoConnectorOnRemoteMicrophoneRemoved onRemoved, VidyoConnectorOnRemoteMicrophoneStateUpdated onStateUpdated)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered when another participant adds a microphone into a conference.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when another participant removes a microphone from a conference.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when another participant microphone changes state in a conference.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterRemoteMicrophoneEventListener(VidyoConnector* c, VidyoConnectorOnRemoteMicrophoneAdded onAdded, VidyoConnectorOnRemoteMicrophoneRemoved onRemoved, VidyoConnectorOnRemoteMicrophoneStateUpdated onStateUpdated);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterRemoteMicrophoneEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters remote microphone event notifications.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterRemoteMicrophoneEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterRemoteMicrophoneEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterRemoteMicrophoneFrameListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about microphone frames from other participants.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterRemoteMicrophoneFrameListener(VidyoConnector* c, VidyoConnectorOnRemoteMicrophoneFrame onFrame, const VidyoRemoteMicrophone* remoteMicrophone)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered for every frame from a remote microphone.}
	}
	{parameter:
		{name: remoteMicrophone}
		{description: The VidyoRemoteMicrophone for which to receive the frames.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterRemoteMicrophoneFrameListener(VidyoConnector* c, VidyoConnectorOnRemoteMicrophoneFrame onFrame, const VidyoRemoteMicrophone* remoteMicrophone);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterRemoteMicrophoneFrameListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters remote microphone frame notifications. 
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterRemoteMicrophoneFrameListener(VidyoConnector* c, const VidyoRemoteMicrophone* remoteMicrophone)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: remoteMicrophone}
		{description: The VidyoRemoteMicrophone for which to stop notifications.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorUnregisterRemoteMicrophoneFrameListener(VidyoConnector* c, const VidyoRemoteMicrophone* remoteMicrophone);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterRemoteMicrophoneEnergyListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about microphone energies from other participants.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterRemoteMicrophoneEnergyListener(VidyoConnector* c, VidyoConnectorOnRemoteMicrophoneEnergy onEnergy)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onEnergy}
		{description: Callback that is triggered for every frame with energy from a remote microphone.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterRemoteMicrophoneEnergyListener(VidyoConnector* c, VidyoConnectorOnRemoteMicrophoneEnergy onEnergy);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterRemoteMicrophoneEnergyListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters remote microphone frame notifications.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterRemoteMicrophoneEnergyListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorUnregisterRemoteMicrophoneEnergyListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterRemoteWindowShareEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about remote shreable-window events.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterRemoteWindowShareEventListener(VidyoConnector* c, VidyoConnectorOnRemoteWindowShareAdded onAdded, VidyoConnectorOnRemoteWindowShareRemoved onRemoved, VidyoConnectorOnRemoteWindowShareStateUpdated onStateUpdated)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered when another participant adds a window-share into a conference.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when another participant removes a window-share from a conference.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when another participant window-share changes state in a conference.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterRemoteWindowShareEventListener(VidyoConnector* c, VidyoConnectorOnRemoteWindowShareAdded onAdded, VidyoConnectorOnRemoteWindowShareRemoved onRemoved, VidyoConnectorOnRemoteWindowShareStateUpdated onStateUpdated);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterRemoteWindowShareEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters remote window share event notifications.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterRemoteWindowShareEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterRemoteWindowShareEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterRemoteWindowShareFrameListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about window-share frames from other participants.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterRemoteWindowShareFrameListener(VidyoConnector* c, VidyoConnectorOnRemoteWindowShareFrame onFrame, const VidyoRemoteWindowShare* remoteWindowShare, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onFrame}
		{description: Callback that is triggered for every frame from a remote window share.}
	}
	{parameter:
		{name: remoteWindowShare}
		{description: The VidyoRemoteWindowShare for which to receive the frames.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested frame.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested frame.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested frame.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: The extra parameters are used to determin the steam quality and the resulting frame might not match exactly what is requested.}
}
*/
LmiBool VidyoConnectorRegisterRemoteWindowShareFrameListener(VidyoConnector* c, VidyoConnectorOnRemoteWindowShareFrame onFrame, const VidyoRemoteWindowShare* remoteWindowShare, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterRemoteWindowShareFrameListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters remote window share frame notifications.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterRemoteWindowShareFrameListener(VidyoConnector* c, const VidyoRemoteWindowShare* remoteWindowShare)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: remoteWindowShare}
		{description: The VidyoRemoteWindowShare for which to stop notifications.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorUnregisterRemoteWindowShareFrameListener(VidyoConnector* c, const VidyoRemoteWindowShare* remoteWindowShare);

/**
{function apigroup="beta,enterprise":
 	{name: VidyoConnectorRegisterRemoteSpeakerEventListener}
 	{parent: VidyoConnector}
 	{description: This is a synchronous interface. Registers to get notified about remote speaker events.
 					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
 	{prototype: LmiBool VidyoConnectorRegisterRemoteSpeakerEventListener(VidyoConnector* c, VidyoConnectorOnRemoteSpeakerAdded onAdded, VidyoConnectorOnRemoteSpeakerRemoved onRemoved, VidyoConnectorOnRemoteSpeakerStateUpdated onStateUpdated)}
 	{parameter:
 		{name: c}
 		{description: The VidyoConnector object.}
 	}
 	{parameter:
 		{name: onAdded}
 		{description: Callback that is triggered when server is ready to accept audio stream from local microphone.}
 	}
 	{parameter:
 		{name: onRemoved}
 		{description: Callback that is triggered when server stops receiving audio stream from local microphone.}
 	}
 	{parameter:
 		{name: onStateUpdated}
 		{description: Callback that is triggered when server makes a state change on accepting audio stream.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterRemoteSpeakerEventListener(VidyoConnector* c, VidyoConnectorOnRemoteSpeakerAdded onAdded, VidyoConnectorOnRemoteSpeakerRemoved onRemoved, VidyoConnectorOnRemoteSpeakerStateUpdated onStateUpdated);

/**
{function apigroup="beta,enterprise":
 	{name: VidyoConnectorUnregisterRemoteSpeakerEventListener}
 	{parent: VidyoConnector}
 	{description: This is a synchronous interface. Unregisters remote speaker event notifications.
 					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
 	{prototype: LmiBool VidyoConnectorUnregisterRemoteSpeakerEventListener(VidyoConnector* c)}
 	{parameter:
 		{name: c}
 		{description: The VidyoConnector object.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterRemoteSpeakerEventListener(VidyoConnector* c);

/**
{function apigroup="beta,enterprise":
 	{name: VidyoConnectorRegisterRemoteRendererEventListener}
 	{parent: VidyoConnector}
 	{description: This is a synchronous interface. Registers to get notified about remote renderer events.
 					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
 	{prototype: LmiBool VidyoConnectorRegisterRemoteRendererEventListener(VidyoConnector* c, VidyoConnectorOnRemoteRendererAdded onAdded, VidyoConnectorOnRemoteRendererRemoved onRemoved, VidyoConnectorOnRemoteRendererStateUpdated onStateUpdated)}
 	{parameter:
 		{name: c}
 		{description: The VidyoConnector object.}
 	}
 	{parameter:
 		{name: onAdded}
 		{description: Callback that is triggered when server is ready to accept video stream from local camera.}
 	}
	{parameter:
 		{name: onRemoved}
 		{description: Callback that is triggered when server stops receiving video stream from local camera.}
 	}
 	{parameter:
 		{name: onStateUpdated}
 		{description: Callback that is triggered when server makes a state change on accepting video stream.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterRemoteRendererEventListener(VidyoConnector* c, VidyoConnectorOnRemoteRendererAdded onAdded, VidyoConnectorOnRemoteRendererRemoved onRemoved, VidyoConnectorOnRemoteRendererStateUpdated onStateUpdated);

/**
 	{function apigroup="beta,enterprise":
	{name: VidyoConnectorUnregisterRemoteRendererEventListener}
 	{parent: VidyoConnector}
 	{description: This is a synchronous interface. Unregisters remote renderer event notifications.
 					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
 	{prototype: LmiBool VidyoConnectorUnregisterRemoteRendererEventListener(VidyoConnector* c)}
	{parameter:
 		{name: c}
 		{description: The VidyoConnector object.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterRemoteRendererEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorAssignViewToRemoteWindowShare}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Shows a VidyoRemoteWindowShare in a view. On, iOS, this API must be called from main(UI) thread.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorAssignViewToRemoteWindowShare(VidyoConnector* c, const LmiViewId* viewId, const VidyoRemoteWindowShare* remoteWindowShare, LmiBool displayCropped, LmiBool allowZoom)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID where the VidyoRemoteWindowShare will be rendered.}
	}
	{parameter:
		{name: remoteWindowShare}
		{description: The VidyoRemoteWindowShare that will be shown.}
	}
	{parameter:
		{name: displayCropped}
		{description: Render the image as cropped. Otherwise letterbox.}
	}
	{parameter:
		{name: allowZoom}
		{description: Allow zooming of the image with mouse or touch.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: This API will work only if compositing had been disabled by passing a NULL LmiViewId to VidyoConnectorConstruct.}
	{note: The view will not be shown until VidyoConnectorShowViewAt is called with valid parameters.}
}
*/
LmiBool VidyoConnectorAssignViewToRemoteWindowShare(VidyoConnector* c, const LmiViewId* viewId, const VidyoRemoteWindowShare* remoteWindowShare, LmiBool displayCropped, LmiBool allowZoom);

/**
 {function apigroup="connector":
    {name: VidyoConnectorAssignViewToVirtualVideoSource}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Shows a VidyoVirtualVideoSource in a view. On, iOS, this API must be called from main(UI) thread.
    				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorAssignViewToVirtualVideoSource(VidyoConnector* c, const LmiViewId* viewId, const VidyoVirtualVideoSource* virtualVideoSource, LmiBool displayCropped, LmiBool allowZoom)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {parameter:
        {name: viewId}
        {description: A platform specific view ID where the VidyoVirtualVideoSource will be rendered.}
    }
    {parameter:
        {name: virtualVideoSource}
        {description: The VidyoVirtualVideoSource that will be shown.}
    }
    {parameter:
        {name: displayCropped}
        {description: Render the image as cropped. Otherwise letterbox.}
    }
    {parameter:
        {name: allowZoom}
        {description: Allow zooming of the image with mouse or touch.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
    {note: This API will work only if compositing had been disabled by passing a NULL LmiViewId to VidyoConnectorConstruct.}
    {note: The view will not be shown until VidyoConnectorShowViewAt is called with valid parameters.}
}
*/
LmiBool VidyoConnectorAssignViewToVirtualVideoSource(VidyoConnector* c, const LmiViewId* viewId, const VidyoVirtualVideoSource* virtualVideoSource, LmiBool displayCropped, LmiBool allowZoom);

/**
 {function apigroup="connector":
	{name: VidyoConnectorHideView}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Hides a view which is used for custom layout of sources. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorHideView(VidyoConnector* c, const LmiViewId* viewId)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: viewId}
		{description: The view ID of the view to hide.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorHideView(VidyoConnector* c, const LmiViewId* viewId);

/**
 {function apigroup="connector":
	{name: VidyoConnectorShowViewAt}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets the position of the view every time it's changed on the screen. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorShowViewAt(VidyoConnector* c, const LmiViewId* viewId, LmiInt x, LmiInt y, LmiUint width, LmiUint height)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID to be positioned.}
	}
	{parameter:
		{name: x}
		{description: X coordinate. }
	}
	{parameter:
		{name: y}
		{description: Y coordinate. }
	}
	{parameter:
		{name: width}
		{description: Width of the view. }
	}
	{parameter:
		{name: height}
		{description: Height of the view. }
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: The view will not be shown until this funciton is called with valid parameters.}
	{note: If viewId is NULL then the composite renderer constructed in VidyoConnectorConstruct will be positioned.}
}
*/
LmiBool VidyoConnectorShowViewAt(VidyoConnector* c, const LmiViewId* viewId, LmiInt x, LmiInt y, LmiUint width, LmiUint height);

/**
 {function apigroup="connector":
 {name: VidyoConnectorShowViewAtPoints}
 {parent: VidyoConnector}
 {description: This is a synchronous interface. Sets the position of the view every time it's changed on the screen. On, iOS, this API must be called from main(UI) thread.
 				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
 {prototype: LmiBool VidyoConnectorShowViewAtPoints(VidyoConnector* c, const LmiViewId* viewId, LmiInt x, LmiInt y, LmiUint width, LmiUint height)}
 {parameter:
 	{name: c}
 	{description: The VidyoConnector object.}
 }
 {parameter:
 	{name: viewId}
 	{description: A platform specific view ID to be positioned.}
 }
 {parameter:
 	{name: x}
 	{description: X coordinate (in points).}
 }
 {parameter:
 	{name: y}
 	{description: Y coordinate (in points).}
 }
 {parameter:
 	{name: width}
 	{description: Width of the view (in points).}
 }
 {parameter:
 	{name: height}
 	{description: Height of the view (in points).}
 }
 {return: LMI_TRUE on success, LMI_FALSE on failure.}
 	{note: The view will not be shown until this funciton is called with valid parameters.}
 	{note: If viewId is NULL then the composite renderer constructed in VidyoConnectorConstruct will be positioned.}
 }
 */
LmiBool VidyoConnectorShowViewAtPoints(VidyoConnector* c, const LmiViewId* viewId, LmiInt x, LmiInt y, LmiUint width, LmiUint height);

/**
 {function apigroup="connector":
	{name: VidyoConnectorShowViewLabel}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Shows or hides the label on the view which displays the participant name or preview. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorShowViewLabel(VidyoConnector* c, const LmiViewId* viewId, LmiBool showLabel)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID to be positioned.}
	}
	{parameter:
		{name: showLabel}
		{description: Boolean to show or hide the label. }
	}
	{return: }
	{note: If viewId is NULL then the composite renderer constructed in VidyoConnectorConstruct will be assumed.}
}
*/
LmiBool VidyoConnectorShowViewLabel(VidyoConnector* c, const LmiViewId* viewId, LmiBool showLabel);

/**
 {function apigroup="connector":
	{name: VidyoConnectorShowAudioMeters}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Shows or hides the audio meters in the view. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorShowAudioMeters(VidyoConnector* c, const LmiViewId* viewId, LmiBool showMeters)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID to be positioned.}
	}
	{parameter:
		{name: showMeters}
		{description: Boolean to show or hide the audio meters. }
	}
	{return: }
	{note: If viewId is NULL then the composite renderer constructed in VidyoConnectorConstruct will be assumed.}
}
*/
LmiBool VidyoConnectorShowAudioMeters(VidyoConnector* c, const LmiViewId* viewId, LmiBool showMeters);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSetViewBackgroundColor}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Set the background color of the view. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetViewBackgroundColor(VidyoConnector* c, const LmiViewId* viewId, LmiUint8 red, LmiUint8 green, LmiUint8 blue)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID to be positioned.}
	}
	{parameter:
		{name: red}
		{description: Red component of the color 0-255.}
	}
	{parameter:
		{name: green}
		{description: Green component of the color 0-255.}
	}
	{parameter:
		{name: blue}
		{description: Blue component of the color 0-255.}
	}
	{return: }
	{note: If viewId is NULL then the composite renderer constructed in VidyoConnectorConstruct will be assumed.}
}
*/
LmiBool VidyoConnectorSetViewBackgroundColor(VidyoConnector* c, const LmiViewId* viewId, LmiUint8 red, LmiUint8 green, LmiUint8 blue);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSetViewAnimationSpeed}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets the speed of transition animation for windows moving around inside the view. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetViewAnimationSpeed(VidyoConnector* c, const LmiViewId* viewId, LmiUint speedPercentage)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: viewId}
		{description: A platform specific view ID to be positioned.}
	}
	{parameter:
		{name: speedPercentage}
		{description: Percentage of animation speed with default of 50, 100 for slowest and 0 for instant. }
	}
	{return: }
	{note: Not supported in WebRTC.}
	{note: If viewId is NULL then the composite renderer constructed in VidyoConnectorConstruct will be assumed.}
}
*/
LmiBool VidyoConnectorSetViewAnimationSpeed(VidyoConnector* c, const LmiViewId* viewId, LmiUint speedPercentage);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterNetworkInterfaceEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about network interface events.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterNetworkInterfaceEventListener(VidyoConnector* c, VidyoConnectorOnNetworkInterfaceAdded onAdded, VidyoConnectorOnNetworkInterfaceRemoved onRemoved, VidyoConnectorOnNetworkInterfaceSelected onSelected, VidyoConnectorOnNetworkInterfaceStateUpdated onStateUpdated)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onAdded}
		{description: Callback that is triggered for every new or existing network interface added.}
	}
	{parameter:
		{name: onRemoved}
		{description: Callback that is triggered when the network interface is removed.}
	}
	{parameter:
		{name: onSelected}
		{description: Callback that is triggered when a network interface is selected for use.}
	}
	{parameter:
		{name: onStateUpdated}
		{description: Callback that is triggered when a network interface state changes.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterNetworkInterfaceEventListener(VidyoConnector* c, VidyoConnectorOnNetworkInterfaceAdded onAdded, VidyoConnectorOnNetworkInterfaceRemoved onRemoved, VidyoConnectorOnNetworkInterfaceSelected onSelected, VidyoConnectorOnNetworkInterfaceStateUpdated onStateUpdated);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterNetworkInterfaceEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters network interface event notifications.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterNetworkInterfaceEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorUnregisterNetworkInterfaceEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSelectNetworkInterfaceForSignaling}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Selects the network interface to be used for Signaling.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSelectNetworkInterfaceForSignaling(VidyoConnector* c, const VidyoNetworkInterface* networkInterface)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: networkInterface}
		{description: Network interface to be selected.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectNetworkInterfaceForSignaling(VidyoConnector* c, const VidyoNetworkInterface *networkInterface);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSelectDefaultNetworkInterfaceForSignaling}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Selects the default ANY network interface to be used for Signaling.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSelectDefaultNetworkInterfaceForSignaling(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectDefaultNetworkInterfaceForSignaling(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSelectNetworkInterfaceForMedia}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Selects the network interface to be used for Media.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSelectNetworkInterfaceForMedia(VidyoConnector* c, const VidyoNetworkInterface* networkInterface)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: networkInterface}
		{description: Network interface to be selected.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectNetworkInterfaceForMedia(VidyoConnector* c, const VidyoNetworkInterface *networkInterface);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSelectDefaultNetworkInterfaceForMedia}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Selects the default ANY network interface to be used for Media.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSelectDefaultNetworkInterfaceForMedia(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectDefaultNetworkInterfaceForMedia(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorConnect}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Connects to a live conference with the specified resourceId.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorConnect(VidyoConnector* c, const char* host, const char* token, const char* displayName, const char *resourceId, VidyoConnectorOnSuccess onSuccess, VidyoConnectorOnFailure onFailure, VidyoConnectorOnDisconnected onDisconnected)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: host}
		{description: Host to which the connection will be made.}
	}
	{parameter:
		{name: token}
		{description: Token generated by the backend that contains the user ID and other metadata that will be used for authentication.}
	}
	{parameter:
		{name: displayName}
		{description: A display name that will be used in the UI.}
	}
	{parameter:
		{name: resourceId}
		{description: An alphanumeric string. This resourceId does not need to be pre-provisioned but in order to create a connection between multiple participants, the same resourceId has to be specified.}
	}
	{parameter:
		{name: onSuccess}
		{description: Callback that is triggered when the connection is successfully established.}
	}
	{parameter:
		{name: onFailure}
		{description: Callback that is triggered when the connection establishemnt failed.}
	}
	{parameter:
		{name: onDisconnected}
		{description: Callback that is triggered when an established connection is disconnected.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorConnect(VidyoConnector* c, const char* host, const char* token, const char* displayName, const char *resourceId, VidyoConnectorOnSuccess onSuccess, VidyoConnectorOnFailure onFailure, VidyoConnectorOnDisconnected onDisconnected);

/**
 {function apigroup="beta,enterprise":
 {name: VidyoConnectorConnectToRoomWithTokenAndKey}
 {parent: VidyoConnector}
 {description: This is an asynchronous interface. Connects to a live conference with the specified room key.
 				This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
 {prototype: LmiBool VidyoConnectorConnectToRoomWithTokenAndKey(VidyoConnector* c, const char* host, const char* userName, const char* token, const char* roomKey, const char * roomPin, VidyoConnectorOnSuccess onSuccess, VidyoConnectorOnFailure onFailure, VidyoConnectorOnDisconnected onDisconnected)}
 {parameter:
 	{name: c}
 	{description: The VidyoConnector object.}
 }
 {parameter:
 	{name: host}
 	{description: Host to which the connection will be made.}
 }
 {parameter:
 	{name: userName}
 	{description: User name.}
 }
 {parameter:
 	{name: token}
 	{description: Access Token.}
 }
 {parameter:
 	{name: roomKey}
 	{description: Room Key.}
 }
 {parameter:
    {name: roomPin}
    {description: Room Pin/Password used to enter the room.}
 }
 {parameter:
 	{name: onSuccess}
 	{description: Callback that is triggered when the connection is successfully established.}
 }
 {parameter:
 	{name: onFailure}
 	{description: Callback that is triggered when the connection establishemnt failed.}
 }
 {parameter:
 	{name: onDisconnected}
 	{description: Callback that is triggered when an established connection is disconnected.}
 }
 {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorConnectToRoomWithTokenAndKey(VidyoConnector* c, const char* host, const char* userName, const char* token, const char* roomKey, const char* roomPin, VidyoConnectorOnSuccess onSuccess, VidyoConnectorOnFailure onFailure, VidyoConnectorOnDisconnected onDisconnected);

/**
 {function apigroup="beta,enterprise":
 {name: VidyoConnectorConnectToRoomWithTokenAndId}
 {parent: VidyoConnector}
 {description: This is an asynchronous interface. Connects to a live conference with the specified room ID on VidyoCloud. This API is not supported on Vidyo.io.
 				This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
 {prototype: LmiBool VidyoConnectorConnectToRoomWithTokenAndId(VidyoConnector* c, const char* host, const char* userName, const char* token, const char* roomId, const char* roomPin, VidyoConnectorOnSuccess onSuccess, VidyoConnectorOnFailure onFailure, VidyoConnectorOnDisconnected onDisconnected)}
 {parameter:
 	{name: c}
 	{description: The VidyoConnector object.}
 }
 {parameter:
 	{name: host}
 	{description: Host to which the connection will be made.}
 }
 {parameter:
 	{name: userName}
 	{description: User name.}
 }
 {parameter:
 	{name: token}
 	{description: Access Token.}
 }
 {parameter:
 	{name: roomId}
 	{description: Room ID.}
 }
 {parameter:
    {name: roomPin}
    {description: Room Pin/Password to enter the room.}
 }
 {parameter:
 	{name: onSuccess}
 	{description: Callback that is triggered when the connection is successfully established.}
 }
 {parameter:
 	{name: onFailure}
 	{description: Callback that is triggered when the connection establishemnt failed.}
 }
 {parameter:
 	{name: onDisconnected}
 	{description: Callback that is triggered when an established connection is disconnected.}
 }
 {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorConnectToRoomWithTokenAndId(VidyoConnector* c, const char* host, const char* userName, const char* token, const char* roomId, const char* roomPin, VidyoConnectorOnSuccess onSuccess, VidyoConnectorOnFailure onFailure, VidyoConnectorOnDisconnected onDisconnected);

/**
 {function apigroup="beta,enterprise":
 {name: VidyoConnectorConnectToRoomWithKey}
 {parent: VidyoConnector}
 {description: This is an asynchronous interface. Connects to a live conference with the specified room key on VidyoCloud. This API is not supported on Vidyo.io.
 				This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
 {prototype: LmiBool VidyoConnectorConnectToRoomWithKey(VidyoConnector* c, const char* host, const char* userName, const char* password, const char *roomKey, const char* roomPin, VidyoConnectorOnSuccess onSuccess, VidyoConnectorOnFailure onFailure, VidyoConnectorOnDisconnected onDisconnected)}
 {parameter:
 	{name: c}
 	{description: The VidyoConnector object.}
 }
 {parameter:
 	{name: host}
 	{description: Host to which the connection will be made.}
 }
 {parameter:
 	{name: userName}
 	{description: User name.}
 }
 {parameter:
 	{name: password}
 	{description: Password.}
 }
 {parameter:
 	{name: roomKey}
 	{description: Room key.}
 }
 {parameter:
    {name: roomPin}
    {description: Room Pin/Password to enter the room.}
 }
 {parameter:
 	{name: onSuccess}
 	{description: Callback that is triggered when the connection is successfully established.}
 }
 {parameter:
 	{name: onFailure}
 	{description: Callback that is triggered when the connection establishemnt failed.}
 }
 {parameter:
 	{name: onDisconnected}
 	{description: Callback that is triggered when an established connection is disconnected.}
 }
 {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorConnectToRoomWithKey(VidyoConnector* c, const char* host, const char* userName, const char* password, const char *roomKey, const char* roomPin, VidyoConnectorOnSuccess onSuccess, VidyoConnectorOnFailure onFailure, VidyoConnectorOnDisconnected onDisconnected);


/**
 {function apigroup="beta,enterprise":
 {name: VidyoConnectorConnectToRoomWithId}
 {parent: VidyoConnector}
 {description: This is an asynchronous interface. Connects to a live conference with the specified room ID on VidyoCloud. This API is not supported on Vidyo.io. 
 				This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
 {prototype: LmiBool VidyoConnectorConnectToRoomWithId(VidyoConnector* c, const char* host, const char* userName, const char* password, const char *roomId, const char* roomPin, VidyoConnectorOnSuccess onSuccess, VidyoConnectorOnFailure onFailure, VidyoConnectorOnDisconnected onDisconnected)}
 {parameter:
 	{name: c}
 	{description: The VidyoConnector object.}
 }
 {parameter:
 	{name: host}
 	{description: Host to which the connection will be made.}
 }
 {parameter:
 	{name: userName}
 	{description: User name.}
 }
 {parameter:
 	{name: password}
 	{description: Password.}
 }
 {parameter:
 	{name: roomId}
 	{description: Room ID.}
 }
 {parameter:
    {name: roomPin}
    {description: Room Pin/Password to enter the room.}
 }
 {parameter:
 	{name: onSuccess}
 	{description: Callback that is triggered when the connection is successfully established.}
 }
 {parameter:
 	{name: onFailure}
 	{description: Callback that is triggered when the connection establishemnt failed.}
 }
 {parameter:
 	{name: onDisconnected}
 	{description: Callback that is triggered when an established connection is disconnected.}
 }
 {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorConnectToRoomWithId(VidyoConnector* c, const char* host, const char* userName, const char* password, const char *roomId, const char* roomPin, VidyoConnectorOnSuccess onSuccess, VidyoConnectorOnFailure onFailure, VidyoConnectorOnDisconnected onDisconnected);

/**
 {function apigroup="beta,enterprise":
 {name: VidyoConnectorConnectToRoomAsGuest}
 {parent: VidyoConnector}
 {description: This is an asynchronous interface. Connects to a live conference as guest with the specified room key on VidyoCloud. This API is not supported on Vidyo.io.
 				This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
 {prototype: LmiBool VidyoConnectorConnectToRoomAsGuest(VidyoConnector* c, const char* host, const char* displayName, const char* roomKey, const char* roomPin, VidyoConnectorOnSuccess onSuccess, VidyoConnectorOnFailure onFailure, VidyoConnectorOnDisconnected onDisconnected)}
 {parameter:
 	{name: c}
 	{description: The VidyoConnector object.}
 }
 {parameter:
 	{name: host}
 	{description: Host to which the connection will be made.}
 }
 {parameter:
 	{name: displayName}
 	{description: A display name that will be used in the UI.}
 }
 {parameter:
 	{name: roomKey}
 	{description: Room key.}
 }
 {parameter:
    {name: roomPin}
    {description: Room Pin / Password to enter the room.}
 }
 {parameter:
 	{name: onSuccess}
 	{description: Callback that is triggered when the connection is successfully established.}
 }
 {parameter:
 	{name: onFailure}
 	{description: Callback that is triggered when the connection establishemnt failed.}
 }
 {parameter:
 	{name: onDisconnected}
 	{description: Callback that is triggered when an established connection is disconnected.}
 }
 {return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorConnectToRoomAsGuest(VidyoConnector* c, const char* host, const char* displayName, const char* roomKey, const char* roomPin, VidyoConnectorOnSuccess onSuccess, VidyoConnectorOnFailure onFailure, VidyoConnectorOnDisconnected onDisconnected);

/**
 {function apigroup="connector":
	{name: VidyoConnectorDisconnect}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Disconnects a live conference. The onDisconnected callback will be called on successful execution.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDisconnect(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorDisconnect(VidyoConnector* c);

/**
{function apigroup="connector":
	{name: VidyoConnectorDisableVideoForAll}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Set video disable state for all participants in the room.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDisableVideoForAll(VidyoConnector* c, LmiBool disable, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: disable}
		{description: Set to LMI_TRUE to stop all participants' video. Video cannot be resumed until another call with revoke set to LMI_FALSE.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorDisableVideoForAll(VidyoConnector* c, LmiBool disable, const LmiString* requestId);

/**
{function apigroup="connector":
	{name: VidyoConnectorDisableVideoForParticipant}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Set video disable state for a selected participant in the room.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDisableVideoForParticipant(VidyoConnector* c, const VidyoParticipant* participant, LmiBool disable, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: participant}
		{description: VidyoParticipant object whose audio is revoked/resumed.}
	}
	{parameter:
		{name: disable}
		{description: Set to LMI_TRUE to stop the participant's video. Video cannot be resumed until another call with revoke set to LMI_FALSE.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorDisableVideoForParticipant(VidyoConnector* c, const VidyoParticipant* participant, LmiBool disable, const LmiString* requestId);

/**
{function apigroup="connector":
	{name: VidyoConnectorDisableAudioForAll}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Set disable audio state for all participants in the room.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDisableAudioForAll(VidyoConnector* c, LmiBool disable, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: disable}
		{description: Set to LMI_TRUE to stop all participants' audio. Audio cannot be resumed until another call with revoke set to LMI_FALSE.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorDisableAudioForAll(VidyoConnector* c, LmiBool disable, const LmiString* requestId);

/**
{function apigroup="connector":
	{name: VidyoConnectorDisableAudioForParticipant}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Set disable audio state for a selected participant in the room.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDisableAudioForParticipant(VidyoConnector* c, const VidyoParticipant* participant, LmiBool disable, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: participant}
		{description: VidyoParticipant Object whose audio is revoked/resumed.}
	}
	{parameter:
		{name: disable}
		{description: Set to LMI_TRUE to stop the participant's audio. Audio cannot be resumed until another call with revoke set to LMI_FALSE.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorDisableAudioForParticipant(VidyoConnector* c, const VidyoParticipant* participant, LmiBool disable, const LmiString* requestId);

/**
{function apigroup="connector":
	{name: VidyoConnectorDisableVideoSilenceForAll}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Set disable video silence state for all participants in the room.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDisableVideoSilenceForAll(VidyoConnector* c, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorDisableVideoSilenceForAll(VidyoConnector* c, const LmiString* requestId);

/**
{function apigroup="connector":
	{name: VidyoConnectorDisableVideoSilenceForParticipant}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Set disable video silence state for a selected participant in the room.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDisableVideoSilenceForParticipant(VidyoConnector* c, const VidyoParticipant* participant, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: participant}
		{description: VidyoParticipant Object whose video is revoked/resumed.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorDisableVideoSilenceForParticipant(VidyoConnector* c, const VidyoParticipant* participant, const LmiString* requestId);

/**
{function apigroup="connector":
	{name: VidyoConnectorDisableAudioSilenceForAll}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Set disable audio silence state for all participants in the room.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDisableAudioSilenceForAll(VidyoConnector* c, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorDisableAudioSilenceForAll(VidyoConnector* c, const LmiString* requestId);

/**
{function apigroup="connector":
	{name: VidyoConnectorDisableAudioSilenceForParticipant}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Set disable audio silence state for a selected participant in the room.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDisableAudioSilenceForParticipant(VidyoConnector* c, const VidyoParticipant* participant, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: participant}
		{description: VidyoParticipant object whose audio is revoked/resumed.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorDisableAudioSilenceForParticipant(VidyoConnector* c, const VidyoParticipant* participant, const LmiString* requestId);

/**
{function apigroup="connector":
	{name: VidyoConnectorDropParticipant}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Drop one participant out of a room. Application needs to register VidyoConnectorRegisterModerationResultEventListener to get the result. 
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDropParticipant(VidyoConnector* c, const VidyoParticipant* participant, const LmiString* reason)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: participant}
		{description: VidyoParticipant Object.}
	}
	{parameter:
		{name: reason}
		{description: A message explaining the reason for the booting. This can be NULL if no message is desired.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorDropParticipant(VidyoConnector* c, const VidyoParticipant* participant, const LmiString* reason);

/**
{function apigroup="connector":
	{name: VidyoConnectorDropAllParticipants}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Drop all participant out of a room, including the moderator used this API.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDropAllParticipants(VidyoConnector* c, const LmiString* reason)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: reason}
		{description: A message explaining the reason for the dropping. This can be NULL if no message is desired.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorDropAllParticipants(VidyoConnector* c, const LmiString* reason);

/**
{function apigroup="connector":
	{name: VidyoConnectorRequestModeratorRole}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. This API provides the ability to request moderator privilege for the regular user.
					If the room is owned by this user or user is Admin/operator moderator pin can be empty string.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRequestModeratorRole(VidyoConnector* c, const LmiString* moderatorPIN, VidyoConnectorOnRequestModeratorRoleResult onRequestModeratorRoleResult)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: moderatorPIN}
		{description: set authentication scheme (moderator pIN) to get the moderatore role functionality. Empty string for admin/host.}
	}
	{parameter:
		{name: onRequestModeratorRoleResult}
		{description: Callback to be triggered when request of moderator role is completed }
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorRequestModeratorRole(VidyoConnector* c,const LmiString* moderatorPIN, VidyoConnectorOnRequestModeratorRoleResult onRequestModeratorRoleResult);

/**
{function apigroup="connector":
	{name: VidyoConnectorRemoveModeratorRole}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. This API provides the ability to remove moderator privilege.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRemoveModeratorRole(VidyoConnector* c, VidyoConnectorOnRemoveModeratorRoleResult onRemoveModeratorRoleResult)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: onRemoveModeratorRoleResult}
		{description: Callback to be triggered when remove of moderator role is completed }
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorRemoveModeratorRole(VidyoConnector* c, VidyoConnectorOnRemoveModeratorRoleResult onRemoveModeratorRoleResult);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetModeratorPIN}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. This API provides the ability to allow a room owner to set an authentication scheme (moderator pin) for moderator functionality. The same authentication scheme must be used by other users by calling VidyoConnectorRequestModeratorRole to get moderator privilege.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetModeratorPIN(VidyoConnector* c, const LmiString* moderatorPIN, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: moderatorPIN}
		{description: authentication scheme (moderator pin) for moderator functionality.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorSetModeratorPIN(VidyoConnector* c, const LmiString* moderatorPIN, const LmiString* requestId);

/**
{function apigroup="connector":
	{name: VidyoConnectorRemoveModeratorPIN}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. This API provides the ability for the room owner to remove authentication scheme for moderator functionality. Once moderator pin is removed, other users will not be able to moderate call on that room.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRemoveModeratorPIN(VidyoConnector* c, const LmiString* requestId)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorRemoveModeratorPIN(VidyoConnector* c, const LmiString* requestId);

/**
{function apigroup="connector":
	{name: VidyoConnectorInviteParticipant}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. This API provides the ability to the room owner, admin or moderators to invite other users to join this room for conference.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorInviteParticipant(VidyoConnector* c, const VidyoContactInfo* contact, const LmiString* message, VidyoConnectorOnInviteResult onInviteResult)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: contact}
		{description: VidyoContactInfo Object.}
	}
	{parameter:
		{name: message}
		{description: Invitation Message.}
	}
	{parameter:
		{name: onInviteResult}
		{description: Callback to be triggered for invitation result.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorInviteParticipant(VidyoConnector* c, const VidyoContactInfo* contact, const LmiString* message, VidyoConnectorOnInviteResult onInviteResult);

/**
{function apigroup="connector":
	{name: VidyoConnectorCancelInviteParticipant}
	{parent: VidyoConnector}
	{description: This is an synchronous interface. This API provides the ability to cancels invitation of a participant. This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorCancelInviteParticipant(VidyoConnector *c, const VidyoContactInfo *contact)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: contact} {description: VidyoContactInfo object.}}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorCancelInviteParticipant(VidyoConnector *c, const VidyoContactInfo *contact);

/**
{function apigroup="connector":
	{name: VidyoConnectorLockRoom}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. This api provides the ability lock the current room.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorLockRoom(VidyoConnector* c, VidyoConnectorOnLockRoomResult onLockRoomResult)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: onLockRoomResult}
		{description: Callback to be triggered for lock room result.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorLockRoom(VidyoConnector* c, VidyoConnectorOnLockRoomResult onLockRoomResult);

/**
{function apigroup="connector":
	{name: VidyoConnectorUnlockRoom}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. This api provides the ability unlock the current room.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnlockRoom(VidyoConnector* c, VidyoConnectorOnUnlockRoomResult onUnlockRoomResult)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: onUnlockRoomResult}
		{description: Callback to be triggered for UnlockRoom  Result.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorUnlockRoom(VidyoConnector* c, VidyoConnectorOnUnlockRoomResult onUnlockRoomResult);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetRoomPIN}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. This api provides the ability to set the room pin. Once room pin is set, all other users are required to enter this room pin to join this conference room.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetRoomPIN(VidyoConnector* c, const LmiString* roomPIN, VidyoConnectorOnSetRoomPINResult onSetRoomPinResult)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: roomPIN}
		{description: Pin to be set as room PIN or set NULL to remove existing room PIN }
	}
	{parameter:
		{name: onSetRoomPinResult}
		{description: Callback to be triggered for set room pin result.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorSetRoomPIN(VidyoConnector* c, const LmiString* roomPIN, VidyoConnectorOnSetRoomPINResult onSetRoomPinResult);

/**
{function apigroup="connector":
	{name: VidyoConnectorRemoveRoomPIN}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. This api provides the ability to remove the room pin. Once room pin is remove, all other users are not required to enter room pin to join conference room.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRemoveRoomPIN(VidyoConnector* c, VidyoConnectorOnRemoveRoomPINResult onRemoveRoomPinResult)}
	{parameter:
		{name: c}
		{description: VidyoConnector object.}
	}
	{parameter:
		{name: onRemoveRoomPinResult}
		{description: Callback to be triggered for remove room pin result.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorRemoveRoomPIN(VidyoConnector* c, VidyoConnectorOnRemoveRoomPINResult onChangeRoomPinResult);

/**
 {function apigroup="connector":
	{name: VidyoConnectorGetRecordingServiceProfiles}
	{parent: VidyoConnector}
	{description: Asynchronously retrieves a vector of available recording profiles. Each element of this vector is a LmiString that contains a description of a recording profile as provided by the active recording server.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.
					For example, STANDARD-QUALITY, HIGH-QUALITY, VOICE-ONLY. Calls a callback upon completion.}
	{prototype: LmiBool VidyoConnectorGetRecordingServiceProfiles(VidyoConnector* c, VidyoConnectorOnGetRecordingServiceProfiles onGetRecordingProfilesCallback)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector object.}
	}
    {parameter:
		{name: onGetRecordingProfilesCallback}
		{description: Callback called upon completion.}
	}
	{return: LMI_TRUE if the request has been scheduled successfully, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorGetRecordingServiceProfiles(VidyoConnector* c, VidyoConnectorOnGetRecordingServiceProfiles onGetRecordingProfilesCallback);

/**
 {function apigroup="connector":
	{name: VidyoConnectorStartRecording}
	{parent: VidyoConnector}
	{description: Asynchronously starts recording a conference using the specified recording profile. Must be called on behalf of a moderator. Fails in case another recording is in progress. Calls a callback upon completion.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorStartRecording(VidyoConnector* c, const LmiString* recordingProfilePrefix, VidyoConnectorOnRecordingServiceStartResult onRecordingServiceStartCallback)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector object.}
	}
	{parameter:
		{name: recordingProfilePrefix}
		{description: Recording profile prefix to start the recording with. Can be one of those provided by VidyoConnectorGetRecordingServiceProfiles or NULL - a default profile will be used.}
	}
	{parameter:
		{name: onRecordingServiceStartCallback}
		{description: Callback called upon completion.}
	}
	{return: LMI_TRUE if the request has been scheduled successfully, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorStartRecording(VidyoConnector* c, const LmiString* recordingProfilePrefix, VidyoConnectorOnRecordingServiceStartResult onRecordingServiceStartCallback);

/**
 {function apigroup="connector":
	{name: VidyoConnectorStopRecording}
	{parent: VidyoConnector}
	{description: Asynchronously stops recording a conference. Must be called on behalf of a moderator. Fails in case there is no recording in progress. Calls a callback upon completion.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorStopRecording(VidyoConnector* c, VidyoConnectorOnRecordingServiceStopResult onRecordingServiceStopResultCallback)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector object.}
	}
	{parameter:
		{name: onRecordingServiceStopResultCallback}
		{description: Callback called upon completion.}
	}
	{return: LMI_TRUE if the request has been scheduled successfully, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorStopRecording(VidyoConnector* c, VidyoConnectorOnRecordingServiceStopResult onRecordingServiceStopResultCallback);

/**
 {function apigroup="connector":
	{name: VidyoConnectorPauseRecording}
	{parent: VidyoConnector}
	{description: Asynchronously pauses recording a conference. Must be called on behalf of a moderator. Fails in case recording is not in progress or already paused. Calls a callback upon completion.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorPauseRecording(VidyoConnector* c, VidyoConnectorOnRecordingServicePauseResult onRecordingServicePauseResultCallback)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector object.}
	}
	{parameter:
		{name: onRecordingServicePauseResultCallback}
		{description: Callback called upon completion.}
	}
	{return: LMI_TRUE if the request has been scheduled successfully, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorPauseRecording(VidyoConnector* c, VidyoConnectorOnRecordingServicePauseResult onRecordingServicePauseResultCallback);

/**
 {function apigroup="connector":
	{name: VidyoConnectorResumeRecording}
	{parent: VidyoConnector}
	{description: Asynchronously resumes recording a conference. Must be called on behalf of a moderator. Fails in case recording is not in progress or not paused. Calls a callback upon completion.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another states, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorResumeRecording(VidyoConnector* c, VidyoConnectorOnRecordingServiceResumeResult onRecordingServiceResumeResultCallback)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector object.}
	}
	{parameter:
		{name: onRecordingServiceResumeResultCallback}
		{description: Callback called upon completion.}
	}
	{return: LMI_TRUE if the request has been scheduled successfully, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorResumeRecording(VidyoConnector* c, VidyoConnectorOnRecordingServiceResumeResult onRecordingServiceResumeResultCallback);

/**
{function apigroup="connector":
	{name: VidyoConnectorDisable}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Disables connector object, releases all devices as precursor to destruct. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disconnecting. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorDisable(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorDisable(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSendChatMessage}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sends a message to all the other participants.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSendChatMessage(VidyoConnector* c, const char* message)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: message}
		{description: A message to send to all the other participants.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSendChatMessage(VidyoConnector* c, const char* message);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSendPrivateChatMessage}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sends a message to the specific participant.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSendPrivateChatMessage(VidyoConnector* c, const VidyoParticipant* participant,  const char* message)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: participant}
		{description: A participant that will receive message.}
	}
	{parameter:
		{name: message}
		{description: A message to send to the specific participants.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSendPrivateChatMessage(VidyoConnector* c, const VidyoParticipant* participant, const char* message);

/**
 {function apigroup="connector":
	{name: VidyoConnectorGetStatsJson}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Retrieves VidyoConnector statistics in JSON format.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: const LmiString *VidyoConnectorGetStatsJson(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: JSON encoded string on success, NULL on failure.}
}
*/
const LmiString *VidyoConnectorGetStatsJson(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorGetVersion}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Retrieves version of the VidyoConnector library.}
	{prototype: const LmiString *VidyoConnectorGetVersion(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: Version on success, NULL on failure.}
}
*/
const LmiString *VidyoConnectorGetVersion(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorGetVersionWithoutBuildNumber}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Retrieves version of the VidyoConnector library wihtout the build number.}
	{prototype: const LmiString *VidyoConnectorGetVersionWithoutBuildNumber(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: Version on success, NULL on failure.}
}
*/
const LmiString *VidyoConnectorGetVersionWithoutBuildNumber(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorGetState}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Retrieves the state of the VidyoConnector.}
	{prototype: VidyoConnectorState VidyoConnectorGetState(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: The state of the VidyoConnector.}
}
*/
VidyoConnectorState VidyoConnectorGetState(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterResourceManagerEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about resource-management events.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterResourceManagerEventListener(VidyoConnector* c, VidyoConnectorOnAvailableResourcesChanged onAvailableResourcesChanged, VidyoConnectorOnMaxRemoteSourcesChanged onMaxRemoteSourcesChanged)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onAvailableResourcesChanged}
		{description: Callback that is triggered when local resource availability changes.}
	}
	{parameter:
		{name: onMaxRemoteSourcesChanged}
		{description: Callback that is triggered when the maximum number of decoded sources changes.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterResourceManagerEventListener(VidyoConnector* c, VidyoConnectorOnAvailableResourcesChanged onAvailableResourcesChanged, VidyoConnectorOnMaxRemoteSourcesChanged onMaxRemoteSourcesChanged);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterResourceManagerEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters resource manager event notifications.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterResourceManagerEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterResourceManagerEventListener(VidyoConnector* c);

/**
 {function apigroup="connector" deprecated="yes" replaced-by="VidyoConnectorRegisterConnectionPropertiesEventListener":
	{name: VidyoConnectorRegisterRecorderInCallEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified when a recorder joins or leaves the call.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterRecorderInCallEventListener(VidyoConnector* c, VidyoConnectorRecorderInCall onRecorderInCallChanged)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onRecorderInCallChanged}
		{description: Callback that is triggered when a recorder joins or leaves the call.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterRecorderInCallEventListener(VidyoConnector* c, VidyoConnectorRecorderInCall onRecorderInCallChanged );

/**
 {function apigroup="connector" deprecated="yes" replaced-by="VidyoConnectorUnregisterConnectionPropertiesEventListener":
	{name: VidyoConnectorUnregisterRecorderInCallEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters recorder in call event notifications.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterRecorderInCallEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterRecorderInCallEventListener(VidyoConnector* c);

/**
{function apigroup="connector" visibility="private":
	{name: VidyoConnectorConnectionPropertiesConstructDefault}
	{parent: VidyoConnectorConnectionProperties}
	{description: Default construct for VidyoConnectorConnectionProperties.}
	{prototype: VidyoConnectorConnectionProperties* VidyoConnectorConnectionPropertiesConstructDefault(VidyoConnectorConnectionProperties* connectionProperties, LmiAllocator* alloc)}
	{parameter:
		{name: connectionProperties}
		{description: VidyoConnectorConnectionProperties object.}
	}
	{parameter:
		{name: alloc}
		{description: LmiAllocator object.}
	}
	{return: VidyoConnectorConnectionProperties object on success otherwise NULL.}
}
*/
VidyoConnectorConnectionProperties* VidyoConnectorConnectionPropertiesConstructDefault(VidyoConnectorConnectionProperties* connectionProperties, LmiAllocator* alloc);

/**
{function apigroup="connector" visibility="private":
	{name: VidyoConnectorConnectionPropertiesConstructCopy}
	{parent: VidyoConnectorConnectionProperties}
	{description: Constructs an VidyoConnectorConnectionProperties object as a copy of an existing object.}
	{prototype: VidyoConnectorConnectionProperties* VidyoConnectorConnectionPropertiesConstructCopy(VidyoConnectorConnectionProperties* d, VidyoConnectorConnectionProperties* s)}
	{parameter:
		{name: d}
		{description: The VidyoConnectorConnectionProperties object to construct.}
	}
	{parameter:
		{name: s}
		{description: The VidyoCameraControlCapabilities object to make a copy of.}
	}
	{return: VidyoConnectorConnectionProperties object on success otherwise NULL.}
}
*/
VidyoConnectorConnectionProperties* VidyoConnectorConnectionPropertiesConstructCopy(VidyoConnectorConnectionProperties* d, VidyoConnectorConnectionProperties* s);

/**
{function apigroup="connector" visibility="private":
	{name: VidyoConnectorConnectionPropertiesDestruct}
	{parent: VidyoConnectorConnectionProperties}
	{description: Destructor for VidyoConnectorConnectionProperties.}
	{prototype: void VidyoConnectorConnectionPropertiesDestruct(VidyoConnectorConnectionProperties* connectionProperties)}
	{parameter:
		{name: connectionProperties}
		{description: VidyoConnectorConnectionProperties object.}
	}
}
*/
void VidyoConnectorConnectionPropertiesDestruct(VidyoConnectorConnectionProperties* connectionProperties);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterConnectionPropertiesEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified when a connection properties is changed.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterConnectionPropertiesEventListener(VidyoConnector* c, VidyoConnectorOnConnectionPropertiesChanged onConnectionPropertiesChanged)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onConnectionPropertiesChanged}
		{description: Callback that is triggered when a connection properties is changed.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterConnectionPropertiesEventListener(VidyoConnector* c, VidyoConnectorOnConnectionPropertiesChanged onConnectionPropertiesChanged);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterConnectionPropertiesEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters event listener, application will not receive an event about connection properties changes.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterConnectionPropertiesEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterConnectionPropertiesEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterWebProxyEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about web proxy events.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterWebProxyEventListener(VidyoConnector* c, VidyoConnectorOnWebProxyCredentialsRequest onWebProxyCredentialsRequest)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onWebProxyCredentialsRequest}
		{description: Callback that is triggered when web proxy credentials are requested.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterWebProxyEventListener(VidyoConnector* c, VidyoConnectorOnWebProxyCredentialsRequest onWebProxyCredentialsRequest);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterWebProxyEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters web proxy event notifications.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterWebProxyEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorUnregisterWebProxyEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSetWebProxyAddressCredentials}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets web proxy server address and credentials.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetWebProxyAddressCredentials(VidyoConnector* c, const char* proxyAddress, const char* username, const char* password)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: proxyAddress}
		{description: The web proxy server address.}
	}
	{parameter:
		{name: username}
		{description: The web proxy server user name.}
	}
	{parameter:
		{name: password}
		{description: The web proxy server password.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetWebProxyAddressCredentials(VidyoConnector* c, const char* proxyAddress, const char* username, const char* password);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSetCertificateAuthorityList}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets the list of certificate authority's certificates used to authenticate a TLS connection.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetCertificateAuthorityList(VidyoConnector* c, const char* certificateAuthorityList)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: certificateAuthorityList}
        {description: The string representing Certificate Authority lists. Note: If the string 'default' is passed here, VidyoConnector will use a built-in Certificate Authority list.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: The list must be a string in PEM format, or 'default'.}
}
*/
LmiBool VidyoConnectorSetCertificateAuthorityList(VidyoConnector* c, const char* certificateAuthorityList);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSetCertificateAuthorityFile}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets the file that contains certificate authority's certificates used to authenticate a TLS connection.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetCertificateAuthorityFile(VidyoConnector* c, const char* certificateAuthorityFile)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: certificateAuthorityFile}
		{description: The string representing Certificate Authority file. Note: The string must containt full path and name of file.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: The list must be a string in PEM format, or 'default'.}
}
*/
LmiBool VidyoConnectorSetCertificateAuthorityFile(VidyoConnector* c, const char* certificateAuthorityFile);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSetCpuTradeOffProfile}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Set CPU tradeoff profile.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetCpuTradeOffProfile(VidyoConnector *c, VidyoConnectorTradeOffProfile profile)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: profile} {description: The tradeoff profile.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorSetCpuTradeOffProfile(VidyoConnector *c, VidyoConnectorTradeOffProfile profile);

/**
 {function apigroup="connector":
	{name: VidyoConnectorGetCpuTradeOffProfile}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Get current CPU tradeoff profile setting.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: VidyoConnectorTradeOffProfile VidyoConnectorGetCpuTradeOffProfile(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: The current CPU tradeoff profile.}
 }
 */
VidyoConnectorTradeOffProfile VidyoConnectorGetCpuTradeOffProfile(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetAdvancedOptions}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets up advanced options on the VidyoConnector API.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetAdvancedOptions(VidyoConnector *c, const char *options)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter:
		{name: options}
		{description: The experimental optons in JSON format.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: This function is for advanced features and should not be used unless specifically instructed by Vidyo.}
}
*/
LmiBool VidyoConnectorSetAdvancedOptions(VidyoConnector *c, const char *options);

/**
{function apigroup="connector":
 	{name: VidyoConnectorSetCameraNudgeConfiguration}
 	{parent: VidyoConnector}
 	{description: This is a synchronous interface. Sets camera PTZ nudge configuration.
 					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
 	{prototype: LmiBool VidyoConnectorSetCameraNudgeConfiguration(VidyoConnector *c, const char *cameraNudgeConfig)}
 	{parameter: {name: c} {description: The VidyoConnector object.}}
 	{parameter:
 		{name: cameraNudgeConfig}
 		{description: Camera PTZ nudge configuration in JSON format.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorSetCameraNudgeConfiguration(VidyoConnector *c, const char *cameraNudgeConfig);

/**
{function apigroup="connector":
 	{name: VidyoConnectorSetUDPTransport}
 	{parent: VidyoConnector}
 	{description: This is a synchronous interface. Enable the use of UDP Transport when connecting.
 					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
 	{prototype: LmiBool VidyoConnectorSetUDPTransport(VidyoConnector *c, LmiBool enable)}
 	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
 	{parameter:
 		{name: enable}
 		{description: Enable the tranport.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 	{note: Enabled by default and preferred over TCP.}
 }
 */
LmiBool VidyoConnectorSetUDPTransport(VidyoConnector *c, LmiBool enable);

/**
{function apigroup="connector":
 	{name: VidyoConnectorSetTCPTransport}
 	{parent: VidyoConnector}
 	{description: This is a synchronous interface. Enable the use of TCP Transport when connecting.
 					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
 	{prototype: LmiBool VidyoConnectorSetTCPTransport(VidyoConnector *c, LmiBool enable)}
 	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
 	{parameter:
 		{name: enable}
 		{description: Enable the tranport.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 	{note: Enabled by default, used as a backup to UDP.}
 }
 */
LmiBool VidyoConnectorSetTCPTransport(VidyoConnector *c, LmiBool enable);

/**
{function apigroup="connector":
 	{name: VidyoConnectorSetWebProxyTransport}
 	{parent: VidyoConnector}
 	{description: This is a synchronous interface. Enable the use of WebProxy Transport when connecting.
 					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
 	{prototype: LmiBool VidyoConnectorSetWebProxyTransport(VidyoConnector *c, LmiBool enable)}
 	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
 	{parameter:
 		{name: enable}
 		{description: Enable the tranport.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 	{note: Enabled by default, used if set by the API or operating system. See VidyoConnectorSetWebProxyAddressCredentials(), VidyoConnectorRegisterWebProxyEventListener() }
 }
 */
LmiBool VidyoConnectorSetWebProxyTransport(VidyoConnector *c, LmiBool enable);

/**
{function apigroup="connector":
 	{name: VidyoConnectorSetWebProxyTransportAddress}
 	{parent: VidyoConnector}
 	{description: This is a synchronous interface. Enable the use of WebProxy Transport with provided web proxy server address.
 					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
 	{prototype: LmiBool VidyoConnectorSetWebProxyTransportAddress(VidyoConnector *c, const LmiString *name, LmiUint port)}
 	{parameter:
 		{name: c}
 		{description: The VidyoConnector object.}
 	}
 	{parameter:
 		{name: name}
 		{description: The name of web proxy server, it may also may be prefixed by a scheme ('http://proxy.company.com')}
 	}
 	{parameter:
 		{name: port}
 		{description: The port that web proxy server is running on}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 	{note: By default web proxy support is enabled, and web proxy server is auto detected from operating system. This API
		allows user to use a pre-defined web proxy server instead of querying operating system.}
 }
 */
LmiBool VidyoConnectorSetWebProxyTransportAddress(VidyoConnector *c, const LmiString *name, LmiUint port);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetTURNAddressCredentials}
 	{parent: VidyoConnector}
	{description: This is a synchronous interface. Set TURN server and credentials information.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetTURNAddressCredentials(VidyoConnector* c, const char* turnAddress, LmiUint port, const char* username, const char* password, VidyoConnectorBaseTransportType baseTransport)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
 	}
	{parameter:
		{name: turnAddress}
		{description: TURN server FQDN or IP address.}
	}
 	{parameter:
 		{name: port}
 		{description: TURN server port.}
 	}
 	{parameter:
 		{name: username}
 		{description: TURN server user name.}
 	}
 	{parameter:
 		{name: password}
 		{description: TURN server password.}
 	}
 	{parameter:
 		{name: baseTransport}
 		{description: Base transport type.}
 	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetTURNAddressCredentials(VidyoConnector* c, const char* turnAddress, LmiUint port, const char* username, const char* password, VidyoConnectorBaseTransportType baseTransport);

/**
{function apigroup="connector":
 	{name: VidyoConnectorSetTURNServerCertificateRequired}
 	{parent: VidyoConnector}
 	{description: This is a synchronous interface. Set TURN server certificate requirement.
 					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
 	{prototype: LmiBool VidyoConnectorSetTURNServerCertificateRequired(VidyoConnector* c, LmiBool required)}
 	{parameter:
 		{name: c}
 		{description: The VidyoConnector object.}
 	}
 	{parameter:
 		{name: required}
 		{description: LMI_TRUE if TURN server certificate is required.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetTURNServerCertificateRequired(VidyoConnector* c, LmiBool required);

/**
{function apigroup="connector":
    {name: VidyoConnectorSetFontFileName}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Set font file name. On, iOS, this API must be called from main(UI) thread.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorSetFontFileName(VidyoConnector* c, const char *fontFileName)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {parameter:
        {name: fontFileName}
        {description: Pointer to char array that contains font file name.
		 Note: on Windows this must be the name of a system TrueType font.  For example "Microsoft YaHei"}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetFontFileName(VidyoConnector* c, const char *fontFileName);

/**
{function apigroup="connector":
 	{name: VidyoConnectorReportLocalParticipantOnJoined}
 	{parent: VidyoConnector}
 	{description: This is a synchronous interface. By default, local participant joining/leaving are not reported to applicaiotn. Use this function to report those events.
 					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
 	{prototype: LmiBool VidyoConnectorReportLocalParticipantOnJoined(VidyoConnector* c, LmiBool reportLocalParticipant)}
 	{parameter:
 		{name: c}
 		{description: The VidyoConnector object.}
 	}
 	{parameter:
 		{name: reportLocalParticipant}
 		{description: Set to LMI_TRUE if application is interested in local participant to be included in participant event callbacks. Otherwise set to LMI_FALSE. Default is LMI_FALSE}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorReportLocalParticipantOnJoined(VidyoConnector* c, LmiBool reportLocalParticipant);

/**
{function apigroup="beta,enterprise":
 	{name: VidyoConnectorRequestToSpeak}
 	{parent: VidyoConnector}
 	{description: This is an asynchronous interface. Request to speak after sever paused audio stream from selected local microphone. On, iOS, this API must be called from main(UI) thread.
 					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
 	{prototype: LmiBool VidyoConnectorRequestToSpeak(VidyoConnector* c)}
	{parameter:
 		{name: c}
 		{description: The VidyoConnector object.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRequestToSpeak(VidyoConnector* c);

/**
{function apigroup="beta,enterprise":
 	{name: VidyoConnectorRequestToResumeVideo}
 	{parent: VidyoConnector}
 	{description: This is an asynchronous interface. Request to resume sending video after sever paused video stream from selected local camera. Will resume both local and remote streams and show preview in case it is on. Allowed only in case of soft mute. On, iOS, this API must be called from main(UI) thread.
 					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
 	{prototype: LmiBool VidyoConnectorRequestToResumeVideo(VidyoConnector* c)}
	{parameter:
 		{name: c}
 		{description: The VidyoConnector object.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRequestToResumeVideo(VidyoConnector* c);

/**
{function apigroup="connector":
    {name: VidyoConnectorRegisterVirtualVideoSourceEventListener}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Registers to get notified about virtual video source events. On, iOS, this API must be called from main(UI) thread.
    				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorRegisterVirtualVideoSourceEventListener(VidyoConnector* c, VidyoConnectorOnVirtualVideoSourceAdded onAdded, VidyoConnectorOnVirtualVideoSourceRemoved onRemoved, VidyoConnectorOnVirtualVideoSourceStateUpdated onStateUpdated, VidyoConnectorOnVirtualVideoSourceExternalMediaBufferReleased onExternalMediaBufferReleased)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {parameter:
        {name: onAdded}
        {description: Callback that is triggered each time when virtual video source is added into a conference.}
    }
    {parameter:
        {name: onRemoved}
        {description: Callback that is triggered each time when virtual video source is added from a conference.}
    }
    {parameter:
        {name: onStateUpdated}
        {description: Callback that is triggered each time when virtual video source chagnes state in a conference.}
    }
    {parameter:
        {name: onExternalMediaBufferReleased}
        {description: Callback that is triggered each time when VidyoClient is done with an application allocated buffer that holds raw video frame.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterVirtualVideoSourceEventListener(VidyoConnector* c, VidyoConnectorOnVirtualVideoSourceAdded onAdded, VidyoConnectorOnVirtualVideoSourceRemoved onRemoved, VidyoConnectorOnVirtualVideoSourceStateUpdated onStateUpdated, VidyoConnectorOnVirtualVideoSourceExternalMediaBufferReleased onExternalMediaBufferReleased);

/**
{function apigroup="connector":
    {name: VidyoConnectorUnregisterVirtualVideoSourceEventListener}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Unregisters virtual video source event notifications. On, iOS, this API must be called from main(UI) thread.
    				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorUnregisterVirtualVideoSourceEventListener(VidyoConnector* c)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterVirtualVideoSourceEventListener(VidyoConnector* c);

/**
{function apigroup="connector":
    {name: VidyoConnectorCreateVirtualVideoSource}
    {parent: VidyoConnector}
    {description: This is an asynchronous interface. Create virtual video source. The user can create a maximum of 10 virtual video sources. On, iOS, this API must be called from main(UI) thread.
    				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorCreateVirtualVideoSource(VidyoConnector* c, VidyoVirtualVideoSourceType type, const char *id, const char *name)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {parameter:
        {name: type}
        {description: Type of the virtual video source.}
    }
    {parameter:
        {name: id}
        {description: Unique ID.}
    }
    {parameter:
        {name: name}
        {description: Name of the virtual video source.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorCreateVirtualVideoSource(VidyoConnector* c, VidyoVirtualVideoSourceType type, const char *id, const char *name);

/**
{function apigroup="connector":
    {name: VidyoConnectorDestroyVirtualVideoSource}
    {parent: VidyoConnector}
    {description: This is an asynchronous interface. Destroy virtual video source. On, iOS, this API must be called from main(UI) thread.
    				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorDestroyVirtualVideoSource(VidyoConnector* c, const VidyoVirtualVideoSource* virtualVideoSource)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {parameter:
        {name: virtualVideoSource}
        {description: The virtual video source that will be destroyed.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorDestroyVirtualVideoSource(VidyoConnector* c, const VidyoVirtualVideoSource* virtualVideoSource);

/**
{function apigroup="connector":
    {name: VidyoConnectorSelectVirtualCamera}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Selects the virtual video source which is camera that will be shown to remote participants. The real camera will be unselected if it's selected. On, iOS, this API must be called from main(UI) thread.
    				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorSelectVirtualCamera(VidyoConnector* c, const VidyoVirtualVideoSource* virtualCamera)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {parameter:
        {name: virtualCamera}
        {description: The virtual video source that will be shown to remote participants.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectVirtualCamera(VidyoConnector* c, const VidyoVirtualVideoSource* virtualCamera);

/**
{function apigroup="connector":
    {name: VidyoConnectorSelectVirtualSourceWindowShare}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Selects the virtual video source which is window share that will be shown to remote participants. The local window share will be unselected if it's selected. On, iOS, this API must be called from main(UI) thread.
    				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorSelectVirtualSourceWindowShare(VidyoConnector* c, const VidyoVirtualVideoSource* virtualWindowShare)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {parameter:
        {name: virtualWindowShare}
        {description: The virtual video source that will be shown to remote participants.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectVirtualSourceWindowShare(VidyoConnector* c, const VidyoVirtualVideoSource* virtualWindowShare);

/**
{function apigroup="connector":
    {name: VidyoConnectorSetRendererOptionsForViewId}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Sets up the local renderer and conference tiles displaying options.
    				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorSetRendererOptionsForViewId(VidyoConnector *c, const LmiViewId* viewId, const char *options)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {parameter:
        {name: viewId}
        {description: A platform-specific view ID.}
    }
    {parameter:
        {name: options}
        {description: A char array that contains options in the JSON format.
            {p: EnableVerticalVideoCentering - Enable or disable the vertical centering of the video screen. Default is true.}
            {p: Example: \{\'EnableVerticalVideoCentering\':true\}.}
            {p: EnableFECCIconCustomLayout - Enable or disable the displaying of custom FECC icon in the right bottom corner. Default is false.}
            {p: Example: \{\'EnableFECCIconCustomLayout\':false\}.}
            {p: ViewingDistance - Sets viewing distance in feet. Default is 2.5. Minimum and Maximum values are 1f & 15f respectively.}
            {p: Example: \{\'ViewingDistance\':2.5\}.}
            {p: ShowAudioTiles - Enable or disable the showing of audio tiles. Default is true.}
            {p: Example: \{\'ShowAudioTiles\':false\}.}
            {p: SetPipPosition - Sets the position of the preview when it is displayed using picture-in-picture mode.
                        Possible values for x: "PipPositionLeft", "PipPositionCenter", "PipPositionRight".
                        Possible values for y: "PipPositionBottom", "PipPositionCenter", "PipPositionTop".
                        lockPip is a boolean indicating whether to lock the preview in this position (by default is false).
                        When false, the user may drag the preview to another location using the mouse or a touch.}
            {p: Example: \{\'SetPipPosition\':\{\'x\':\'PipPositionRight\', \'y\': \'PipPositionTop\', \'lockPip\':true\}\}.}
            {p: SetPixelDensity - Sets the rendering environment's pixel density.}
            {p: Example: \{\'SetPixelDensity\':326.0f.
                        However for latest rendering environment, being released for iOS only, this option is used to choose surface density with possible values as
                        0.5 (everything will be smaller) / 1.0 (default) / 2.0 (everything will be larger).\}}
            {p: SetTouchAsInputDevice - Enable or disable touch as a input device. Default is false.}
            {p: Example: \{\'SetTouchAsInputDevice\':false\}.}
            {p: EnableExpandedCameraControl - Enable or disable expanded camera control. Default is true for mobile devices and false for non-mobile devices.}
            {p: Example: \{\'EnableExpandedCameraControl\':false\}.}
            {p: BorderStyle - Sets the border style of the preview. None: Do not put a border on any tiles. Same: Put the same colored border on each tile. Highlight: Put a border on each tile, but tiles associated with the highest ranked participant will have a different colored border than the others. Flash: Tiles associated with the highest ranked participant have a different colored border than the others, When the highest ranked participant changes, the border of the new highest ranked participant flashes momentarily.
                        Possible values for BorderStyle: "None", "Same", "Highlight", "Flash"}
            {p: Example: \{\'BorderStyle\':\'Same\'}\}
            {p: enablePreviewMirroring  - Specifies whether mirroring will be enabled or disabled for local cameras. Possible values are true or false. The default is true.}
            {p: Example: \{\'EnablePreviewMirroring\' : true\}.}
        }
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetRendererOptionsForViewId(VidyoConnector *c, const LmiViewId* viewId, const char *options);

/**
{function apigroup="connector":
    {name: VidyoConnectorGetRendererOptionsForViewId}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Get the local renderer and conference tiles displaying options.
    				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiString* VidyoConnectorGetRendererOptionsForViewId(VidyoConnector *c, const LmiViewId* viewId)}
    {parameter:
        {name: c}
        {description: The VidyoConnector object.}
    }
    {parameter:
        {name: viewId}
        {description: A platform-specific view ID.}
    }
    {return: Pointer to LmiString that contains the local renderer and conference tiles displaying options in JSON format.}
}
*/
LmiString* VidyoConnectorGetRendererOptionsForViewId(VidyoConnector *c, const LmiViewId* viewId);

/**
{function apigroup="connector":
	{name: VidyoConnectorAddMessageClass}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Add message class that application should receive.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorAddMessageClass(VidyoConnector *c, const char *messageClass)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: messageClass}
		{description: Name of message class. MSGCLASS_HUNTER is added by default so application will receive only these message class.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorAddMessageClass(VidyoConnector *c, const char *messageClass);

/**
{callback apigroup="connector":
    {name: VidyoConnectorOnModerationCommandReceived}
    {parent: VidyoConnector}
    {description: Callback that will be tringgered when the moderation command will be recieved from the server.}
    {prototype: void (*VidyoConnectorOnModerationCommandReceived)(const VidyoConnector* c, VidyoDeviceType deviceType, VidyoRoomModerationType moderationType, LmiBool state)}
    {parameter:
        {name: c}
        {description: Pointer to the VidyoConnector that triggered the callback. }
    }
    {parameter:
        {name: deviceType}
        {description: Type of the device that was muted. Local camera or microphone.}
    }
    {parameter:
        {name: moderationType}
        {description: Type of moderation command, tells if device was hard muted/unmuted or soft muted. Note: soft unmute is not supported.}
    }
    {parameter:
        {name: state}
        {description: State of the mute. LMI_TRUE in case device was muted, LMI_FALSE in case he was unmuted.}
    }
}
*/
typedef void (*VidyoConnectorOnModerationCommandReceived)(const VidyoConnector* c, VidyoDeviceType deviceType, VidyoRoomModerationType moderationType, LmiBool state);

/**
{function:
    {name: VidyoConnectorRegisterModerationCommandEventListener}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Register a callback to receive moderation commands sent from the server.
    				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorRegisterModerationCommandEventListener(VidyoConnector* c, VidyoConnectorOnModerationCommandReceived onModerationCommand)}
    {parameter: {name: c} {description: VidyoConnector object.}}
    {parameter: {name: onModerationCommand} {description: Callback to be triggered upon receive moderation command from the server.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorRegisterModerationCommandEventListener(VidyoConnector* c, VidyoConnectorOnModerationCommandReceived onModerationCommand);

/**
{function:
    {name: VidyoConnectorUnregisterModerationCommandEventListener}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Unregister a callback to do not receive moderation commands sent from the server.
    				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorUnregisterModerationCommandEventListener(VidyoConnector* c)}
    {parameter: {name: c} {description: VidyoConnector object.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorUnregisterModerationCommandEventListener(VidyoConnector* c);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterReconnectEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers to get notified about reconnecting events. Note: You should not perform long action in the callbacks.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterReconnectEventListener(VidyoConnector *c, VidyoConnectorOnReconnecting onReconnecting, VidyoConnectorOnReconnected onReconnected, VidyoConnectorOnConferenceLost onConferenceLost)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onReconnecting}
		{description: Callback that is triggered each time the new attempt to reconnect will be started.}
	}
	{parameter:
		{name: onReconnected}
		{description: Callback that is triggered when reconnecting was successful.}
	}
	{parameter:
		{name: onConferenceLost}
		{description: Callback that is triggered when reconnecting failed.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterReconnectEventListener(VidyoConnector *c, VidyoConnectorOnReconnecting onReconnecting, VidyoConnectorOnReconnected onReconnected, VidyoConnectorOnConferenceLost onConferenceLost);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterReconnectEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters reconnecting event notifications.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterReconnectEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterReconnectEventListener(VidyoConnector* c);

/**
{function apigroup="connector":
	{name: VidyoConnectorPinParticipant}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Pin a participant's video.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorPinParticipant(VidyoConnector* c, const VidyoParticipant* participant, LmiBool pin)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: participant}
		{description: A participant that will pinned/unpinned.}
	}
	{parameter:
		{name: pin}
		{description: LMI_TRUE to pin participant, LMI_FALSE to unpin.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorPinParticipant(VidyoConnector* c, const VidyoParticipant* participant, LmiBool pin);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSwapStreamsBetweenViews}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Swap streams of the remote video sources between views.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSwapStreamsBetweenViews(VidyoConnector* c, const LmiViewId* first, const LmiViewId* second)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: first}
		{description: A platform specific view ID where the first remote video sources is displayed.}
	}
	{parameter:
		{name: second}
		{description: A platform specific view ID where the second remote video sources is displayed.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSwapStreamsBetweenViews(VidyoConnector* c, const LmiViewId* first, const LmiViewId* second);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRequestVideoForRemoteCamera}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Requests video for remote camera before assign.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRequestVideoForRemoteCamera(VidyoConnector* c, const VidyoRemoteCamera* remoteCamera, LmiUint width, LmiUint height, LmiTime frameInterval, VidyoConnectorOnRemoteCameraFirstFrameReceived onFirstFrameReceived)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: remoteCamera}
		{description: The VidyoRemoteCamera for which to receive the video.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested video.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested video.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested video.}
	}
	{parameter:
		{name: onFirstFrameReceived}
		{description: Callback that is triggered when first frame from a remote camera received.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRequestVideoForRemoteCamera(VidyoConnector* c, const VidyoRemoteCamera* remoteCamera, LmiUint width, LmiUint height, LmiTime frameInterval, VidyoConnectorOnRemoteCameraFirstFrameReceived onFirstFrameReceived);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRequestVideoForRemoteWindowShare}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Requests video for remote window share before assign.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRequestVideoForRemoteWindowShare(VidyoConnector* c, const VidyoRemoteWindowShare* remoteWindowShare, LmiUint width, LmiUint height, LmiTime frameInterval, VidyoConnectorOnRemoteWindowShareFirstFrameReceived onFirstFrameReceived)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: remoteWindowShare}
		{description: The VidyoRemoteWindowShare for which to receive the video.}
	}
	{parameter:
		{name: width}
		{description: Approximate width of the requested video.}
	}
	{parameter:
		{name: height}
		{description: Approximate height of the requested video.}
	}
	{parameter:
		{name: frameInterval}
		{description: Approximate frame interval of the requested video.}
	}
	{parameter:
		{name: onFirstFrameReceived}
		{description: Callback that is triggered when first frame from a remote window share received.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRequestVideoForRemoteWindowShare(VidyoConnector* c, const VidyoRemoteWindowShare* remoteWindowShare, LmiUint width, LmiUint height, LmiTime frameInterval, VidyoConnectorOnRemoteWindowShareFirstFrameReceived onFirstFrameReceived);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetOptions}
	{parent: VidyoConnector}
	{description: This is  a synchronous interface. Sets up options on the VidyoConnector API.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetOptions(VidyoConnector *c, const char *options)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter:
		{name: options}
		{description: The options in JSON format.
			{p: conferenceReferenceNumber - a special identifier that will show up in the Portal CDR record for the conference. Should be set before joining the conference.}
			{p: Example: \{\'conferenceReferenceNumber\' : \'reference number\'\}.}
			{p: microphoneMaxBoostLevel - sets max boost level that will be applied only in the conference for the selected microphone in case the system is set to the higher value. Possible values are devices specific, e.g. 0 - 20 dB. Default is 12 dB.}
			{p: Example: \{\'microphoneMaxBoostLevel\' : 10\}.}
			{p: minMicrophoneVolume - sets min volume that will be applied only in the conference for the selected microphone in case the system is set to the lower value. Possible values are 0 - 100 percentage. Default is 50 per cent.}
			{p: Example: \{\'minMicrophoneVolume\' : 70\}.}
			{p: Possible values for audio device modes are :
				audioSharedModeBoth - use both audio devices (Speaker and Microphone) in shared mode.
				audioExclusiveModeBoth - use both audio devices (Speaker and Microphone) in exclusive mode.
				audioExclusiveModeMic - use microphone in exclusive mode and speaker in shared mode.
				By default VidyoClient is working with audioSharedModeBoth. A new mode will be applied only for the next conference or in case another device is selected.}
			{p: Example: \{\'audioSharedModeBoth\' : true\}.}
			{p: Example: \{\'audioExclusiveModeBoth\' : true\}.}
			{p: Example: \{\'audioExclusiveModeMic\' : true\}.}
			{p: preferredAudioCodec - sets VidyoClient to use preferred audio codec. Possible value are OPUS RED, OPUS and SPEEX RED. Default is OPUS RED.}
			{p: Example: \{\'preferredAudioCodec\' : \'SPEEX RED\'\}.}
            {p: AudioBitrateMultiplier - A multiplier to be used when calculating the audio bitrate. Possible values are 0, 1 and 2. Default is 2.}
            {p: Example: \{\'AudioBitrateMultiplier\':1\}.}
            {p: AudioPacketInterval - It decies audio frame duration to encoder. It sets time interval between two packet of audio in miliseconds Possible values are 20, and 40 ms. Default is 40 ms.}
            {p: Example: \{\'AudioPacketInterval\':40\}.}
            {p: AudioPacketLossPercentage - it values is used to configure encoder. If given higher value then encoder is configured to resistant to loss on the expense of quality. Possible values are [0, 10, 20, 30] in percentage. Default is 10%.}
            {p: Example: \{\'AudioPacketLossPercentage\':20\}.}
            {p: statRate - sets VidyoRoom statistics gather interval in seconds. Possible values are 2, 5, 10, 20 and 30. Default is 10.}
            {p: Example: \{\'statRate\':10\}.}
			{p: enableStaticShareSvc - Specifies whether SVC will be enabled or disabled for static sharing. Possible values are true or false. The default is false. Note: SVC is enabled by default for interactive video and high frame rate sharing.}
			{p: Example: \{\'enableStaticShareSvc\' : true\}.}
		}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetOptions(VidyoConnector *c, const char *options);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetOptions}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Retrieves VidyoConnector options in JSON format.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: const LmiString *VidyoConnectorGetOptions(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: JSON encoded string on success, NULL on failure.}
}
*/
const LmiString *VidyoConnectorGetOptions(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorRegisterConferenceModeEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers the conference mode event listeners.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterConferenceModeEventListener(VidyoConnector *c, VidyoConnectorOnConferenceModeChanged conferenceModeChanged)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: conferenceModeChanged}
		{description: The VidyoConnectorOnConferenceModeChanged status as lecture,lobby or group.}
	}	
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterConferenceModeEventListener(VidyoConnector *c, VidyoConnectorOnConferenceModeChanged conferenceModeChanged);

/**
{function apigroup="connector":
	{name: VidyoConnectorUnregisterConferenceModeEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Un-Registers the conference mode event listeners.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterConferenceModeEventListener(VidyoConnector *c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterConferenceModeEventListener(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorRegisterHostEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers the host event listeners.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterHostEventListener(VidyoConnector* c, VidyoConnectorOnConferenceHostStatusChanged hostStatus)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: hostStatus}
		{description: The VidyoConnectorOnConferenceHostStatusChanged status i..e host joined or left }
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterHostEventListener(VidyoConnector* c, VidyoConnectorOnConferenceHostStatusChanged hostStatus);

/**
{function apigroup="connector":
	{name: VidyoConnectorUnregisterHostEventListener}
	{parent: VidyoConnector}
 	{description: This is a synchronous interface. Un-Registers the host event listeners.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterHostEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterHostEventListener(VidyoConnector* c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetAutoReconnect}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Enables or disables auto-reconnect.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling 
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetAutoReconnect(VidyoConnector *c, LmiBool enable)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: enable} {description: Indicates whether auto-reconnect is enabled or not.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoConnectorSetAutoReconnect(VidyoConnector *c, LmiBool enable);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetAutoReconnect}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets state of auto-reconnect. It's either enabled or disabled.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorGetAutoReconnect(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: LMI_TRUE if auto-reconnect enabled, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoConnectorGetAutoReconnect(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetAutoReconnectMaxAttempts}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets maximum attempts to be done for reconnecting. By default, it is set to 3.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling 
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetAutoReconnectMaxAttempts(VidyoConnector *c, LmiUint maxAttempts)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: maxAttempts} {description: Number of maximum attempts to be done for reconnecting. Min value is 1 and the max is 4.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoConnectorSetAutoReconnectMaxAttempts(VidyoConnector *c, LmiUint maxAttempts);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetAutoReconnectMaxAttempts}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets maximum attempts to be done for reconnecting.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiUint VidyoConnectorGetAutoReconnectMaxAttempts(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: amount of maximum attempts to be done for reconnecting}
}
*/
LmiUint VidyoConnectorGetAutoReconnectMaxAttempts(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetAutoReconnectAttemptBackOff}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets back off for next reconnect attempt. By default, it is set to 5 seconds. 
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling 
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetAutoReconnectAttemptBackOff(VidyoConnector *c, LmiUint backOff)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: backOff} {description: Time in seconds before next reconnect attempt. Min value is 2 and the max is 20 seconds.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoConnectorSetAutoReconnectAttemptBackOff(VidyoConnector *c, LmiUint backOff);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetAutoReconnectAttemptBackOff}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets back off for next reconnect attempt.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiUint VidyoConnectorGetAutoReconnectAttemptBackOff(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: the back off for next reconnect attempt.}
}
*/
LmiUint VidyoConnectorGetAutoReconnectAttemptBackOff(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetDisableVideoOnLowBandwidth}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. When enabled, all video feeds (video and content share) will be shut off to the client when the user experiences prolonged low network bandwidth.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetDisableVideoOnLowBandwidth(VidyoConnector *c, LmiBool enable)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: enable} {description: Indicates whether disable video on low bandwidth is enabled or not.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoConnectorSetDisableVideoOnLowBandwidth(VidyoConnector *c, LmiBool enable);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetDisableVideoOnLowBandwidth}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets state of video on low bandwidth. It's either enabled or disabled.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorGetDisableVideoOnLowBandwidth(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: LMI_TRUE if enabled video on low bandwidth, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoConnectorGetDisableVideoOnLowBandwidth(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetDisableVideoOnLowBandwidthResponseTime}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets time it takes to shut off video due to low bandwidth. By default, it is set to 30 seconds.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetDisableVideoOnLowBandwidthResponseTime(VidyoConnector *c, LmiUint responseTime)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: responseTime} {description: Time it takes to shut off video due to low bandwidth.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoConnectorSetDisableVideoOnLowBandwidthResponseTime(VidyoConnector *c, LmiUint responseTime);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetDisableVideoOnLowBandwidthResponseTime}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets time it takes to shut off video due to low bandwidth.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiUint VidyoConnectorGetDisableVideoOnLowBandwidthResponseTime(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: the time it takes to shut off video due to low bandwidth.}
}
*/
LmiUint VidyoConnectorGetDisableVideoOnLowBandwidthResponseTime(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetDisableVideoOnLowBandwidthSampleTime}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets how often to check for low bandwidth. By default, it is set to 5 seconds. 
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetDisableVideoOnLowBandwidthSampleTime(VidyoConnector *c, LmiUint sampleTime)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: sampleTime} {description: How often to check for low bandwidth. Must divide evenly into responseTime.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoConnectorSetDisableVideoOnLowBandwidthSampleTime(VidyoConnector *c, LmiUint sampleTime);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetDisableVideoOnLowBandwidthSampleTime}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets how often checking for low bandwidth happens.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiUint VidyoConnectorGetDisableVideoOnLowBandwidthSampleTime(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: how often check for low bandwidth happens.}
}
*/
LmiUint VidyoConnectorGetDisableVideoOnLowBandwidthSampleTime(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetDisableVideoOnLowBandwidthRecoveryTime}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets time it takes to recover when video is shut off. By default, it is set to 0 seconds which means never.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetDisableVideoOnLowBandwidthRecoveryTime(VidyoConnector *c, LmiUint recoveryTime)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: recoveryTime} {description: Time it takes to recover when video is shut off. Recovery can only happen when there are no available video sources.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoConnectorSetDisableVideoOnLowBandwidthRecoveryTime(VidyoConnector *c, LmiUint recoveryTime);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetDisableVideoOnLowBandwidthRecoveryTime}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets time it takes to recover when video is shut off.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiUint VidyoConnectorGetDisableVideoOnLowBandwidthRecoveryTime(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: time it takes to recover when video is shut off}
}
*/
LmiUint VidyoConnectorGetDisableVideoOnLowBandwidthRecoveryTime(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetDisableVideoOnLowBandwidthThreshold}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets low bandwidth threshold. By default, it is set to 100kpbs.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetDisableVideoOnLowBandwidthThreshold(VidyoConnector *c, LmiUint kbps)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: kbps} {description: Threshold for low bandwidth in kilobits/sec.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoConnectorSetDisableVideoOnLowBandwidthThreshold(VidyoConnector *c, LmiUint kbps);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetDisableVideoOnLowBandwidthThreshold}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets low bandwidth threshold.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiUint VidyoConnectorGetDisableVideoOnLowBandwidthThreshold(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: the low bandwidth threshold}
}
*/
LmiUint VidyoConnectorGetDisableVideoOnLowBandwidthThreshold(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetDisableVideoOnLowBandwidthAudioStreams}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets number of audio streams requested while video is shut off. By default, it is set to 3. 
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetDisableVideoOnLowBandwidthAudioStreams(VidyoConnector *c, LmiUint audioStreams)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: audioStreams} {description:  Number of audio streams requested while video is shut off. Min value is 1 and max is 3.}}
	{return: LMI_TRUE on success, otherwise LMI_FALSE.}
}
*/
LmiBool VidyoConnectorSetDisableVideoOnLowBandwidthAudioStreams(VidyoConnector *c, LmiUint audioStreams);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetDisableVideoOnLowBandwidthAudioStreams}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets number of audio streams requested while video is shut off.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle,VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiUint VidyoConnectorGetDisableVideoOnLowBandwidthAudioStreams(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: the number of audio streams requested while video is shut off.}
}
*/
LmiUint VidyoConnectorGetDisableVideoOnLowBandwidthAudioStreams(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetConferenceNumber}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets conference number - a special identifier that will show up in the Portal CDR record for the conference. 
					Should be set before joining the conference. This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetConferenceNumber(VidyoConnector *c, const LmiString *conferenceNumber)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: conferenceNumber} {description: Conference number to be set.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetConferenceNumber(VidyoConnector *c, const LmiString *conferenceNumber);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetConferenceNumber}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets conference number - a special identifier that is showed up in the Portal CDR record for the conference.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and empty string returned.}
	{prototype: const LmiString* VidyoConnectorGetConferenceNumber(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: Conference number.}
}
*/
const LmiString* VidyoConnectorGetConferenceNumber(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetMinMicrophoneVolume}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets min volume that will be applied only in the conference for the selected microphone in case the system is set to the lower value.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetMinMicrophoneVolume(VidyoConnector *c, LmiUint volumePercent)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: volumePercent} {description: Volume percent for which possible values are 0 - 100 percentage. Default is 50 percent.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetMinMicrophoneVolume(VidyoConnector *c, LmiUint volumePercent);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetMinMicrophoneVolume}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets minimum microphone value that is applied only in the conference for the selected microphone in case the system is set to the lower value.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return -1.}
	{prototype: LmiInt16 VidyoConnectorGetMinMicrophoneVolume(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: Minimum microphone value.}
}
*/
LmiInt16 VidyoConnectorGetMinMicrophoneVolume(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetMicrophoneMaxBoostLevel}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets max boost level that will be applied only in the conference for the selected microphone in case the system is set to the higher value.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetMicrophoneMaxBoostLevel(VidyoConnector *c, LmiFloat64 boostLevel)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: boostLevel} {description: Boost level for which possible values are devices specific, e.g. 0 - 20 dB. Default is 12 dB.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetMicrophoneMaxBoostLevel(VidyoConnector *c, LmiFloat64 boostLevel);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetMicrophoneMaxBoostLevel}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets maximum microphone boost level that is applied only in the conference for the selected microphone in case the system is set to the higher value.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return -1.}
	{prototype: LmiFloat64 VidyoConnectorGetMicrophoneMaxBoostLevel(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: Maximum microphone boost level.}
}
*/
LmiFloat64 VidyoConnectorGetMicrophoneMaxBoostLevel(VidyoConnector *c);

/**
 {type apigroup="connector" visibility="public":
	{name: VidyoConnectorAudioMode}
	{parent: VidyoConnector}
	{description: Indicates the audio mode.}
	{value: {name: VIDYO_CONNECTORAUDIOMODE_SharedModeBoth}
		{description: Both audio devices (Speaker and Microphone) used in shared mode.}}
	{value: {name: VIDYO_CONNECTORAUDIOMODE_ExclusiveModeBoth}
		{description: Both audio devices (Speaker and Microphone) used in exclusive mode.}}
	{value: {name: VIDYO_CONNECTORAUDIOMODE_ExclusiveModeMicrophone}
		{description: Microphone is used in exclusive mode and speaker in shared mode.}}
	{value: {name: VIDYO_CONNECTORAUDIOMODE_Unknown}
		{description: Unknown audio mode.}}
}
*/
typedef enum
{
	VIDYO_CONNECTORAUDIOMODE_SharedModeBoth,
	VIDYO_CONNECTORAUDIOMODE_ExclusiveModeBoth,
	VIDYO_CONNECTORAUDIOMODE_ExclusiveModeMicrophone,
	VIDYO_CONNECTORAUDIOMODE_Unknown
} VidyoConnectorAudioMode;

/**
{function apigroup="connector":
	{name: VidyoConnectorSetAudioMode}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Enables or disables usage of audio devices (speaker and microphone) in different audio modes. By default SharedModeBoth is enabled.
				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
				& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetAudioMode(VidyoConnector *c, VidyoConnectorAudioMode audioMode)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{parameter:
		{name: audioMode}
		{description: Indicates which audio mode is enabled/disabled for both audio devices. For possible values of the audio mode, refer VidyoConnectorAudioMode.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetAudioMode(VidyoConnector* c, VidyoConnectorAudioMode audioMode);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetAudioMode}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets the audio mode for both the audio devices (speaker and microphone).
				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
				& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return VIDYO_CONNECTORAUDIOMODE_Unknown.}
	{prototype: VidyoConnectorAudioMode VidyoConnectorGetAudioMode(VidyoConnector *c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{return: Audio Mode, refer VidyoConnectorAudioMode.}
}
*/
VidyoConnectorAudioMode VidyoConnectorGetAudioMode(VidyoConnector* c);

/**
 {type apigroup="connector" visibility="public":
	{name: VidyoConnectorPreferredAudioCodec}
	{parent: VidyoConnector}
	{description: Preferred audio codec to be set.}
	{value: {name: VIDYO_CONNECTORPREFERREDAUDIOCODEC_OpusRed}
		{description: Preferred audio codec is OPUS RED.}}
	{value: {name: VIDYO_CONNECTORPREFERREDAUDIOCODEC_Opus}
		{description: Preferred audio codec is OPUS.}}
	{value: {name: VIDYO_CONNECTORPREFERREDAUDIOCODEC_SpeexRed}
		{description: Preferred audio codec is SPEEX RED.}}
	{value: {name: VIDYO_CONNECTORPREFERREDAUDIOCODEC_Unknown}
		{description: Unknown preferred audio codec .}}
}
*/
typedef enum
{
	VIDYO_CONNECTORPREFERREDAUDIOCODEC_OpusRed,
	VIDYO_CONNECTORPREFERREDAUDIOCODEC_Opus,
	VIDYO_CONNECTORPREFERREDAUDIOCODEC_SpeexRed,
	VIDYO_CONNECTORPREFERREDAUDIOCODEC_Unknown
} VidyoConnectorPreferredAudioCodec;

/**
{function apigroup="connector":
	{name: VidyoConnectorSetPreferredAudioCodec}
	{parent: VidyoConnector}
	{description:This is a synchronous interface. Sets VidyoClient to use preferred audio codec.
				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
				& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetPreferredAudioCodec(VidyoConnector *c, VidyoConnectorPreferredAudioCodec preferredAudioCodec)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{parameter:
		{name: preferredAudioCodec}
		{description: Preferred audio codec to be set. For possible values of the audio codec, refer VidyoConnectorPreferredAudioCodec.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetPreferredAudioCodec(VidyoConnector* c, VidyoConnectorPreferredAudioCodec preferredAudioCodec);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetPreferredAudioCodec}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets preferred audio codec name.
				This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
				& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return VIDYO_CONNECTORPREFERREDAUDIOCODEC_Unknown.}
	{prototype: VidyoConnectorPreferredAudioCodec VidyoConnectorGetPreferredAudioCodec(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: Preferred audio codec, refer VidyoConnectorPreferredAudioCodec.}
}
*/
VidyoConnectorPreferredAudioCodec VidyoConnectorGetPreferredAudioCodec(VidyoConnector* c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetAudioPacketInterval}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets time interval between two packet of audio in miliseconds. Default is 40 ms.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetAudioPacketInterval(VidyoConnector *c, LmiUint packetInterval)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: packetInterval} {description: Packet interval for which possible values are 20, and 40 ms.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetAudioPacketInterval(VidyoConnector *c, LmiUint packetInterval);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetAudioPacketInterval}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets time interval between two packet of audio in miliseconds.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return -1.}
	{prototype: LmiInt VidyoConnectorGetAudioPacketInterval(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: Audio packet interval in miliseconds.}
}
*/
LmiInt VidyoConnectorGetAudioPacketInterval(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetAudioPacketLossPercentage}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets value that is used to configure encoder.
					If given higher value then encoder is configured to resistant to loss on the expense of quality. Default is 10%.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetAudioPacketLossPercentage(VidyoConnector *c, LmiUint lossPercentage)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: lossPercentage} {description: Loss percantage for which possible values are [0, 10, 20, 30] in percentage.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetAudioPacketLossPercentage(VidyoConnector *c, LmiUint lossPercentage);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetAudioPacketLossPercentage}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets audio loss percantage - value that is used to configure encoder.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return -1.}
	{prototype: LmiInt VidyoConnectorGetAudioPacketLossPercentage(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: Audio loss percentage.}
}
*/
LmiInt VidyoConnectorGetAudioPacketLossPercentage(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetAudioBitrateMultiplier}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets multiplier value to be used when calculating the audio bitrate. Default is 2.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetAudioBitrateMultiplier(VidyoConnector *c, LmiUint bitrateMultiplier)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: bitrateMultiplier} {description: Bitrate multiplier for which possible values are 0, 1 and 2.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetAudioBitrateMultiplier(VidyoConnector *c, LmiUint bitrateMultiplier);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetAudioBitrateMultiplier}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets audio bitrate multiplier which is used when calculating the audio bitrate.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return -1.}
	{prototype: LmiInt VidyoConnectorGetAudioBitrateMultiplier(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: Audio bitrate multiplier.}
}
*/
LmiInt VidyoConnectorGetAudioBitrateMultiplier(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetStatisticRate}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets conference room statistics gather interval in seconds. Default is 10.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetStatisticRate(VidyoConnector *c, LmiUint statRate)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: statRate} {description: Statistics gather interval for which possible values are 2, 5, 10, 20 and 30.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetStatisticRate(VidyoConnector *c, LmiUint statRate);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetStatisticRate}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets conference room statistics gather interval in seconds.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return -1.}
	{prototype: LmiInt VidyoConnectorGetStatisticRate(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: Statistics gather interval.}
}
*/
LmiInt VidyoConnectorGetStatisticRate(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetStaticShareSvcState}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Enables or disables SVC. By default SVC is disabled. Note: For interactive video and high frame rate sharing SVC is enabled by default.
	This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling & VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetStaticShareSvcState(VidyoConnector *c, LmiBool enable)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: enable} {description: Indicates whether enable SVC or not.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetStaticShareSvcState(VidyoConnector *c, LmiBool enable);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetStaticShareSvcState}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Gets whether SVC is enabled. The default is false.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Disabling
					& VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorGetStaticShareSvcState(VidyoConnector *c)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{return: LMI_TRUE if SVC enabled, otherwise LMI_FALSE}
}
*/
LmiBool VidyoConnectorGetStaticShareSvcState(VidyoConnector *c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSearchUsers}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Search for users in conference mode.}
	{prototype:LmiBool VidyoConnectorSearchUsers(VidyoConnector* c, const LmiString* searchText, LmiUint startIndex, LmiUint maxRecords, VidyoConnectorOnUserSearchResults onUserSearchResults)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object pointer.}
	}
	{parameter:
		{name: searchText}
		{description: The search text to search users.}
	}
	{parameter:
		{name: startIndex}
		{description: The starting index from where the result should be retrieved.}
	}
	{parameter:
		{name: maxRecords}
		{description: The max number of records in the result.}
	}
	{parameter:
		{name: onUserSearchResults}
		{description: The callback which will receive the results.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
*/
LmiBool VidyoConnectorSearchUsers(VidyoConnector* c, const LmiString* searchText, LmiUint startIndex, LmiUint maxRecords, VidyoConnectorOnUserSearchResults onUserSearchResults);

/**
{function apigroup="connector":
	{name: VidyoConnectorRegisterErrorEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers the advanced error event listeners.}
	{prototype: LmiBool VidyoConnectorRegisterErrorEventListener(VidyoConnector* c, VidyoConnectorOnError onErrorCallback)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onErrorCallback}
		{description: Callback that is triggered when an error occurs.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterErrorEventListener(VidyoConnector* c, VidyoConnectorOnError onErrorCallback);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterErrorEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters advanced error event notifications.}
	{prototype: LmiBool VidyoConnectorUnregisterErrorEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterErrorEventListener(VidyoConnector* c);

/**
{function apigroup="connector":
	{name: VidyoConnectorAddAudioDeviceToWhitelist}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Adds audio device to whitelist. This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorAddAudioDeviceToWhitelist(VidyoConnector* c, const LmiString* deviceName)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: deviceName}
		{description: The name of audio device that will be whitelisted.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorAddAudioDeviceToWhitelist(VidyoConnector* c, const LmiString* deviceName);

/**
{function apigroup="connector":
	{name: VidyoConnectorRemoveAudioDeviceFromWhitelist}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Removes audio device from whitelist. This interface is allowed to be called in all VidyoConnector states except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRemoveAudioDeviceFromWhitelist(VidyoConnector* c, const LmiString* deviceName)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: deviceName}
		{description: The name of audio device that will be removed from whitelist.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRemoveAudioDeviceFromWhitelist(VidyoConnector* c, const LmiString* deviceName);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetWhitelistedAudioDevices}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Gets the names of the audio devices that are whitelisted. This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorGetWhitelistedAudioDevices(VidyoConnector* c, VidyoConnectorOnGetWhitelistedAudioDevices onGetCallback)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onGetCallback}
		{description: Callback that will report an array with the names of audio devices that are whitelisted.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorGetWhitelistedAudioDevices(VidyoConnector* c, VidyoConnectorOnGetWhitelistedAudioDevices onGetCallback);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnGetActiveNetworkInterface}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when result of VidyoConnectorGetActiveNetworkInterface is received.}
	{prototype: void (*VidyoConnectorOnGetActiveNetworkInterface)(VidyoConnector *c, const VidyoNetworkInterface *signalingInterface, const VidyoNetworkInterface *mediaInterface)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}}
	{parameter:
		{name: signalingInterface}
		{description: The zeroed-out VidyoNetworkInterface object for signaling interface.}}
	{parameter:
		{name: mediaInterface}
		{description: The zeroed-out VidyoNetworkInterface object for media interface.}}
}
*/
typedef void (*VidyoConnectorOnGetActiveNetworkInterface)(VidyoConnector *c, const VidyoNetworkInterface *signalingInterface, const VidyoNetworkInterface *mediaInterface);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetActiveNetworkInterface}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Get the NetworkInterface for media and signaling.
		This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorGetActiveNetworkInterface(VidyoConnector* c, VidyoConnectorOnGetActiveNetworkInterface onGetActiveNetworkInterfaceCallback)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{parameter:
		{name: onGetActiveNetworkInterfaceCallback}
		{description: Callback called upon completion.}}
	{return: LMI_TRUE on success, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorGetActiveNetworkInterface(VidyoConnector* c, VidyoConnectorOnGetActiveNetworkInterface onGetActiveNetworkInterfaceCallback);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnGetAutoReconnectSetting}
	{parent: VidyoConnector}
	{description: Callback that is triggered each time when result of VidyoConnectorGetAutoReconnectSetting is received.}
	{prototype: void (*VidyoConnectorOnGetAutoReconnectSetting)(VidyoConnector *c, LmiBool enableAutoReconnect, LmiUint reconnectBackoff, LmiUint maxReconnectAttempts)}
	{parameter:
		{name: c}
		{description: The VidyoConnector that triggered the callback.}}
	{parameter:
		{name: enableAutoReconnect}
		{description: Represents the auto reconnect is enabled or not.}}
	{parameter:
		{name: reconnectBackoff}
		{description: Represents the time in seconds before next reconnect attempt.}}
	{parameter:
		{name: maxReconnectAttempts}
		{description: Represents the maximum reconnect attempts.}}
}
*/
typedef void (*VidyoConnectorOnGetAutoReconnectSetting)(VidyoConnector *c, LmiBool enableAutoReconnect, LmiUint reconnectBackoff, LmiUint maxReconnectAttempts);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetAutoReconnectSetting}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Gets the auto-reconnect settings.
		This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorGetAutoReconnectSetting(VidyoConnector *c, VidyoConnectorOnGetAutoReconnectSetting onGetAutoReconnectSettingCallback)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{parameter:
		{name: onGetAutoReconnectSettingCallback}
		{description: Callback called upon completion.}}
	{return: LMI_TRUE on success, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorGetAutoReconnectSetting(VidyoConnector *c, VidyoConnectorOnGetAutoReconnectSetting onGetAutoReconnectSettingCallback);

/**
{type apigroup="connector" visibility="public":
	{name: VidyoConnectorCameraEffectType}
	{parent: VidyoConnector}
	{description: Provides the supported camera background effects and used to enable or disable effect.}
	{value:
		{name: VIDYO_CONNECTORCAMERAEFFECTTYPE_None}
		{description: Disables active effects.}}
	{value:
		{name: VIDYO_CONNECTORCAMERAEFFECTTYPE_Blur}
		{description: Enables background blur effect.}}
	{value:
		{name: VIDYO_CONNECTORCAMERAEFFECTTYPE_VirtualBackground}
		{description: Enables virtual background effect.}}
}
*/
typedef enum
{
	VIDYO_CONNECTORCAMERAEFFECTTYPE_None,
	VIDYO_CONNECTORCAMERAEFFECTTYPE_Blur,
	VIDYO_CONNECTORCAMERAEFFECTTYPE_VirtualBackground
} VidyoConnectorCameraEffectType;

/**
{type apigroup="connector" visibility="public":
	{name:  VidyoConnectorCameraEffectInfo}
	{parent: Connector}
	{description: Represent the configuration of camera background effect.}
	{member:
		{name: effectType}
		{type: VidyoConnectorCameraEffectType}
		{description: Type of the camera background effect. In case VIDYO_CONNECTORCAMERAEFFECTTYPE_None is set no need to set other fields.}
	}
	{member:
		{name: token}
		{type: LmiString}
		{description: The token for enabling the camera background effect. The blank string is NOT allowed. In case the token is not valid or expired enabling of camera background effect will fail.}
	}
	{member:
		{name: pathToResources}
		{type: LmiString}
		{description: The path to resources like a model and its configuration. A directory with supported resources is packaged with VidyoClient library.}
	}
	{member:
		{name: pathToEffect}
		{type: LmiString}
		{description:The path to effect. A directory with supported effect is packaged with VidyoClient library.}
	}
	{member:
		{name: blurIntensity}
		{type: LmiUint}
		{description: Blur intensity. Valid valies is [3-8]. Default value is 5. Used only if effectType VIDYO_CONNECTORCAMERAEFFECTTYPE_Blur}
	}
	{member:
		{name: virtualBackgroundPicture}
		{type: LmiString}
		{description: The path to the picture that will be used for the virtual background. Used only if effectType VIDYO_CONNECTORCAMERAEFFECTTYPE_VirtualBackground.}
	}
}
*/
typedef struct
{
	VidyoConnectorCameraEffectType effectType;
	LmiString token;
	LmiString pathToResources;
	LmiString pathToEffect;
	LmiUint blurIntensity;
	LmiString virtualBackgroundPicture;
} VidyoConnectorCameraEffectInfo;

VidyoConnectorCameraEffectInfo* VidyoConnectorCameraEffectInfoConstructDefault(VidyoConnectorCameraEffectInfo* effectInfo, LmiAllocator* alloc);
VidyoConnectorCameraEffectInfo* VidyoConnectorCameraEffectInfoConstructCopy(VidyoConnectorCameraEffectInfo* dst, const VidyoConnectorCameraEffectInfo* src);
void VidyoConnectorCameraEffectInfoDestruct(VidyoConnectorCameraEffectInfo* effectInfo);

/**
{callback apigroup="simple":
	{name: VidyoConnectorOnGetCameraBackgroundEffectInfo}
	{parent: VidyoConnector}
	{description: Callback that reports the current configuration of camera background effect to the application.}
	{prototype: void (*VidyoConnectorOnGetCameraBackgroundEffectInfo)(VidyoConnector* c,  const VidyoConnectorCameraEffectInfo* effectInfo)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback. }
	}
	{parameter:
		{name: effectInfo}
		{description: Represents the current configuration of camera background effect.}
	}
}
*/
typedef void (*VidyoConnectorOnGetCameraBackgroundEffectInfo)(VidyoConnector* c, const VidyoConnectorCameraEffectInfo* effectInfo);

/**
{function apigroup="connector":
	{name: VidyoConnectorGetCameraBackgroundEffect}
	{parent: VidyoConnector}
	{description: This is a asynchronous interface. Gets the current configuration of camera background effect. This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE. This functionality is supported only for macOS 10.14 and above, iOS 10.0 and above and Windows VC++ 15 and above.}
	{prototype: LmiBool VidyoConnectorGetCameraBackgroundEffect(VidyoConnector* c, VidyoConnectorOnGetCameraBackgroundEffectInfo onGetEffectInfo)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onGetEffectInfo}
		{description: Callback that reports the current configuration of camera background effect to the application. In case some error occurs NULL will be reported.}
	}
	{return: Returns LMI_TRUE on success or LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorGetCameraBackgroundEffect(VidyoConnector* c, VidyoConnectorOnGetCameraBackgroundEffectInfo onGetEffectInfo);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetCameraBackgroundEffect}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Enables the specified by VidyoConnectorCameraEffectType camera background effect or disables active effect if VIDYO_CONNECTORCAMERAEFFECTTYPE_None passed. The application must provide token to enable this feature as part of VidyoConnectorCameraEffectInfo structure. This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE. This functionality is supported only for macOS 10.14 and above, iOS 10.0 and above and Windows VC++ 15 and above.}
	{prototype: LmiBool VidyoConnectorSetCameraBackgroundEffect(VidyoConnector* c, const VidyoConnectorCameraEffectInfo* effectInfo)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: effectInfo}
		{description: Represents the configuration of camera background effect.}
	}
	{return: Returns LMI_TRUE on success or LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorSetCameraBackgroundEffect(VidyoConnector* c, const VidyoConnectorCameraEffectInfo* effectInfo);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetBlurIntensity}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets the intensity of the background blur effect. It is allowed to change blur intensity while the effect is enabled. This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE. This functionality is supported only for macOS 10.14 and above, iOS 10.0 and above and Windows VC++ 15 and above.}
	{prototype: LmiBool VidyoConnectorSetBlurIntensity(VidyoConnector* c, LmiUint intensity)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: intensity}
		{description: The intensity of the background blur effect. Valid valies is [3-8]. Default value is 5.}
	}
	{return: Returns LMI_TRUE on success or LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorSetBlurIntensity(VidyoConnector* c, LmiUint intensity);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetVirtualBackgroundPicture}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets path and file name to the picture that will be used for the virtual background effect. It is allowed to change picture while the effect is enabled. This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE. This functionality is supported only for macOS 10.14 and above, iOS 10.0 and above and Windows VC++ 15 and above.}
	{prototype: LmiBool VidyoConnectorSetVirtualBackgroundPicture(VidyoConnector* c, const LmiString* pathToPicture)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: pathToPicture}
		{description: Path and file name to the picture. Currently, only PNG and JPG is supported.}
	}
	{return: Returns LMI_TRUE on success or LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorSetVirtualBackgroundPicture(VidyoConnector* c, const LmiString* pathToPicture);

/**
{type apigroup="connector":
	{name: VidyoConnectorCameraEffectError}
	{parent: VidyoConnector}
	{description: Provides the error codes for camera background effect.}
	{value:
		{name: VIDYO_CONNECTORCAMERAEFFECTERROR_None}
		{description: No errors.}}
	{value:
		{name: VIDYO_CONNECTORCAMERAEFFECTERROR_InvalidToken}
		{description: The token is empty or expired.}}
	{value:
		{name: VIDYO_CONNECTORCAMERAEFFECTERROR_InvalidResources}
		{description: The path to resources is empty or does not contain some files.}}
	{value:
		{name: VIDYO_CONNECTORCAMERAEFFECTERROR_InvalidEffect}
		{description: The path to effect is empty  or does not contain some files.}}
	{value:
		{name: VIDYO_CONNECTORCAMERAEFFECTERROR_InvalidBlurIntensity}
		{description: The blur intensity is out of allowed range.}}
	{value:
		{name: VIDYO_CONNECTORCAMERAEFFECTERROR_InvalidBackgroundPicture}
		{description: The path to background picture is empty or file doesn't exist or has unsupported format.}}
	{value:
		{name: VIDYO_CONNECTORCAMERAEFFECTERROR_LoadEffectFailed}
		{description: Failed to load effect.}}
	{value:
		{name: VIDYO_CONNECTORCAMERAEFFECTERROR_UnsupportedOpenGLVersion}
		{description: Effect was not started due to unsupported OpenGL version.}}
	{value:
		{name: VIDYO_CONNECTORCAMERAEFFECTERROR_UnsupportedLibraryVersion}
		{description: Effect was not started due to unsupported Banuba library version.}}
	{value:
		{name: VIDYO_CONNECTORCAMERAEFFECTERROR_NotAllowedForCurrectEffect}
		{description: Opperation is not allowed with the current effect.}}
	{value:
		{name: VIDYO_CONNECTORCAMERAEFFECTERROR_MiscError}
		{description: Miscellaneous error.}}
}
*/

typedef enum
{
	VIDYO_CONNECTORCAMERAEFFECTERROR_None,
	VIDYO_CONNECTORCAMERAEFFECTERROR_InvalidToken,
	VIDYO_CONNECTORCAMERAEFFECTERROR_InvalidResources,
	VIDYO_CONNECTORCAMERAEFFECTERROR_InvalidEffect,
	VIDYO_CONNECTORCAMERAEFFECTERROR_InvalidBlurIntensity,
	VIDYO_CONNECTORCAMERAEFFECTERROR_InvalidBackgroundPicture,
	VIDYO_CONNECTORCAMERAEFFECTERROR_LoadEffectFailed,
	VIDYO_CONNECTORCAMERAEFFECTERROR_UnsupportedOpenGLVersion,
	VIDYO_CONNECTORCAMERAEFFECTERROR_UnsupportedLibraryVersion,
	VIDYO_CONNECTORCAMERAEFFECTERROR_NotAllowedForCurrectEffect,
	VIDYO_CONNECTORCAMERAEFFECTERROR_MiscError
} VidyoConnectorCameraEffectError;

/**
{callback apigroup="simple":
	{name: VidyoConnectorOnCameraEffectError}
	{parent: VidyoConnector}
	{description: Callback that reports camera effect errors.}
	{prototype: void (*VidyoConnectorOnCameraEffectError)(VidyoConnector* c,  VidyoConnectorCameraEffectError error)}
	{parameter:
		{name: c}
		{description: Pointer to the VidyoConnector that triggered the callback.}
	}
	{parameter:
		{name: error}
		{description: The error occurred during applying or changing the camera effect.}
	}
}
*/
typedef void (*VidyoConnectorOnCameraEffectError)(VidyoConnector* c, VidyoConnectorCameraEffectError error);

/**
{function apigroup="connector":
	{name: VidyoConnectorRegisterCameraEffectErrorListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers callback to notify the user in case of any errors occur during enabling camera effect.}
	{prototype: LmiBool VidyoConnectorRegisterCameraEffectErrorListener(VidyoConnector* c, VidyoConnectorOnCameraEffectError onError)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onError}
		{description: The callback that will reports camera effect errors.}
	}
	{return: Returns LMI_TRUE on success or LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorRegisterCameraEffectErrorListener(VidyoConnector* c, VidyoConnectorOnCameraEffectError onError);

/**
{function apigroup="connector":
	{name: VidyoConnectorUnregisterCameraEffectErrorListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters callback to notify the user in case of any errors occur during enabling camera effect.}
	{prototype: LmiBool VidyoConnectorUnregisterCameraEffectErrorListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: Returns LMI_TRUE on success or LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorUnregisterCameraEffectErrorListener(VidyoConnector* c);

/**
{type apigroup="connector" visibility="public":
	{name:  VidyoConnectorShareOptions}
	{parent: Connector}
	{description: Represent the options for sharing.}
	{member: {name: enableAudio} {type: LmiBool} {description: A variable that indicates whether the sharing will be with or without audio.}}
	{member: {name: enableHighFramerate} {type: LmiBool} {description: A variable that indicates whether the sharing will be with or without high framerate share.}}
}
*/
typedef struct
{
	LmiBool enableAudio;
	LmiBool enableHighFramerate;
} VidyoConnectorShareOptions;

VidyoConnectorShareOptions* VidyoConnectorShareOptionsConstructDefault(VidyoConnectorShareOptions* options, LmiAllocator* alloc);
VidyoConnectorShareOptions* VidyoConnectorShareOptionsConstructCopy(VidyoConnectorShareOptions* dst, const VidyoConnectorShareOptions* src);
void VidyoConnectorShareOptionsDestruct(VidyoConnectorShareOptions* options);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSelectLocalMonitorAdvanced}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Selects the local monitor to share in a conference. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSelectLocalMonitorAdvanced(VidyoConnector* c, const VidyoLocalMonitor* localMonitor, VidyoConnectorShareOptions* options)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: localMonitor} {description: Local monitor to share in a conference. NULL if none of them should be shared.}}
	{parameter: {name: options} {description: A structure that represent the sharing options.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectLocalMonitorAdvanced(VidyoConnector* c, const VidyoLocalMonitor* localMonitor, VidyoConnectorShareOptions* options);

/**
 {function apigroup="connector":
	{name: VidyoConnectorSelectLocalWindowShareAdvanced}
	{parent: VidyoConnector}
	{description: This is an asynchronous interface. Selects the local window to share in a conference. On, iOS, this API must be called from main(UI) thread.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSelectLocalWindowShareAdvanced(VidyoConnector* c, const VidyoLocalWindowShare* localWindowShare, VidyoConnectorShareOptions* options)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: localWindowShare} {description: Local window to share in a conference. NULL if none of them should be shared.}}
	{parameter: {name: options} {description: A structure that represent the sharing options.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectLocalWindowShareAdvanced(VidyoConnector* c, const VidyoLocalWindowShare* localWindowShare, VidyoConnectorShareOptions* options);

/**
{type apigroup="connector":
	{name: VidyoConnectorInCallEventClass}
	{parent: VidyoConnector}
	{description: It identifies the class of in call event.}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCLASS_MediaTransport}
		{description: It indicate event belongs to media transportation.}}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCLASS_InCallCodec}
		{description: It indicate event belongs to InCallCodec.}}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCLASS_Dtls}
		{description: It indicate event belongs to Dtls Encryption.}}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCLASS_UnKnown}
		{description: It indicate Unknown event.}}
}
*/
typedef enum
{
	VIDYO_CONNECTORINCALLEVENTCLASS_MediaTransport,
	VIDYO_CONNECTORINCALLEVENTCLASS_InCallCodec,
	VIDYO_CONNECTORINCALLEVENTCLASS_Dtls,
	VIDYO_CONNECTORINCALLEVENTCLASS_UnKnown,
} VidyoConnectorInCallEventClass;

/**
{type apigroup="connector":
	{name: VidyoConnectorInCallEventCode}
	{parent: VidyoConnector}
	{description: It represents error code as call event.}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCODE_RTCPSendError}
		{description: RTCP send fail.}}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCODE_RTCPReceiveError}
		{description: RTCP receive fail.}}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCODE_RTPSendError}
		{description: RTP media packet send fail.}}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCODE_RTPReceiveError}
		{description: RTP media packet receive fail.}}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCODE_EncoderPausedOnLowBW}
		{description: Event when encoder paused on low bandwidth condition.}}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCODE_EncoderResumed}
		{description: Event when encoder resumed after pause.}}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCODE_InCallCodecVideoH264}
		{description: Event when H264 video codec is used in a conference call.}}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCODE_InCallCodecVideoH264SVC}
		{description: Event when H264-SVC video codec is used in a conference call.}}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCODE_InCallCodecAudioSPEEXRED}
		{description: Event when SPEEX RED audio codec is used in a conference call.}}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCODE_DtlsError}
		{description: Event when there is an error encrypting srtp packet because DTLSSRTP session is not yet established.}}
	{value: {name: VIDYO_CONNECTORINCALLEVENTCODE_UnKnown}
		{description: Unknown event code.}}
}
*/
typedef enum
{
	VIDYO_CONNECTORINCALLEVENTCODE_RTCPSendError,
	VIDYO_CONNECTORINCALLEVENTCODE_RTCPReceiveError,
	VIDYO_CONNECTORINCALLEVENTCODE_RTPSendError,
	VIDYO_CONNECTORINCALLEVENTCODE_RTPReceiveError,
	VIDYO_CONNECTORINCALLEVENTCODE_EncoderPausedOnLowBW,
	VIDYO_CONNECTORINCALLEVENTCODE_EncoderResumed,
	VIDYO_CONNECTORINCALLEVENTCODE_InCallCodecVideoH264,
	VIDYO_CONNECTORINCALLEVENTCODE_InCallCodecVideoH264SVC,
	VIDYO_CONNECTORINCALLEVENTCODE_InCallCodecAudioSPEEXRED,
	VIDYO_CONNECTORINCALLEVENTCODE_DtlsError,
	VIDYO_CONNECTORINCALLEVENTCODE_UnKnown,
} VidyoConnectorInCallEventCode;

/**
{type apigroup="connector" visibility="public":
	{name: VidyoConnectorInCallEvent}
	{parent: Connector}
	{description: It is the event which will be sent to application which register for incall events.}
	{member:
		{name: type}
		{type: VidyoConnectorInCallEventClass}
		{description: This indicate the module to which event belongs to.}
	}
	{member:
		{name: code}
		{type: VidyoConnectorInCallEventCode}
		{description: It is error code for the event.}
	}
	{member:
		{name: details}
		{type: LmiString}
		{description: It extra information for the event. It is optional and it may depend on type of event.}
	}
}
*/
typedef struct
{
	VidyoConnectorInCallEventClass type;
	VidyoConnectorInCallEventCode code;
	LmiString details;
} VidyoConnectorInCallEvent;

/**
{function visibility="private":
	{name: VidyoConnectorInCallEventConstructDefault}
	{parent: VidyoConnectorInCallEvent}
	{description: Constructs default in call event.}
	{prototype: VidyoConnectorInCallEvent* VidyoConnectorInCallEventConstructDefault(VidyoConnectorInCallEvent* ev, LmiAllocator* alloc)}
	{parameter:
		{name: ev}
		{description:  Pointer to VidyoConnectorInCallEvent object.}
	}
	{parameter:
		{name: alloc}
		{description: LmiAllocator object.}
	}
	{return: The pointer to constructed object on success, or NULL on failure}
}
*/
VidyoConnectorInCallEvent* VidyoConnectorInCallEventConstructDefault(VidyoConnectorInCallEvent* ev, LmiAllocator* alloc);

/**
{function visibility="private":
	{name: VidyoConnectorInCallEventConstructCopy}
	{parent: VidyoConnectorInCallEvent}
	{description: Constructs copy of VidyoConnectorInCallEvent.}
	{prototype: VidyoConnectorInCallEvent* VidyoConnectorInCallEventConstructCopy(VidyoConnectorInCallEvent* d, const VidyoConnectorInCallEvent* s)}
	{parameter:
		{name: d}
		{description: The object to construct.}
	}
	{parameter:
		{name: s}
		{description: The call object to make copy of.}
	}
	{return: The pointer to constructed object on success, or NULL on failure}
}
*/
VidyoConnectorInCallEvent* VidyoConnectorInCallEventConstructCopy(VidyoConnectorInCallEvent* d, const VidyoConnectorInCallEvent* s);

/**
{function visibility="private":
	{name: VidyoConnectorInCallEventAssign}
	{parent: VidyoConnectorInCallEvent}
	{description: Assigns one event  object the value of another.}
	{prototype: VidyoConnectorInCallEvent* VidyoConnectorInCallEventAssign(VidyoConnectorInCallEvent* d, const VidyoConnectorInCallEvent* s)}
	{parameter:
		{name: d}
		{description: The object to construct.}
	}
	{parameter:
		{name: s}
		{description: The call object to make copy of.}
	}
	{return: The pointer to constructed object on success, or NULL on failure}
}
*/
VidyoConnectorInCallEvent* VidyoConnectorInCallEventAssign(VidyoConnectorInCallEvent* d, const VidyoConnectorInCallEvent* s);

/**
{function visibility="private":
	{name: VidyoConnectorInCallEventDestruct}
	{parent: VidyoConnectorInCallEvent}
	{description: Destructs event object.}
	{prototype: void VidyoConnectorInCallEventDestruct(VidyoConnectorInCallEvent* ev)}
	{parameter:
		{name: ev}
		{description: Event for corresponding action.}
	}
}
*/
void VidyoConnectorInCallEventDestruct(VidyoConnectorInCallEvent* ev);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnInCallEvents}
	{parent: VidyoConnector}
	{description: A callback to be invoked when any in call event to be sent to application.}
	{prototype: void(*VidyoConnectorOnInCallEvents)(VidyoConnector* c, const VidyoConnectorInCallEvent* ev)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object that triggered the callback.}
	}
	{parameter:
		{name: ev}
		{description: Detail of event.}
	}
}
*/
typedef void(*VidyoConnectorOnInCallEvents)(VidyoConnector* c, const VidyoConnectorInCallEvent* ev);

/**
{function apigroup="connector":
	{name: VidyoConnectorRegisterInCallEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets the callback which will be triggered during a conference, these events may not stop the conferece but may degrade quality of call. This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterInCallEventListener(VidyoConnector* c, VidyoConnectorOnInCallEvents cb)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: cb}
		{description: The callback which will be triggered when error occur in conference.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterInCallEventListener(VidyoConnector* c, VidyoConnectorOnInCallEvents cb);

/**
{function apigroup="connector":
	{name: VidyoConnectorUnregisterInCallEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Resets the callback for in call event. This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterInCallEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterInCallEventListener(VidyoConnector* c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetLogFilePath}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets new path to log file.
	This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetLogFilePath(VidyoConnector* c, const char* filename)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: filename}
		{description: The name of the file to log to. Note: Can be passed just filename or with full path so that location will be changed.}
	}
	{return: LMI_TRUE on success, else LMI_FALSE.}
}
*/
LmiBool VidyoConnectorSetLogFilePath(VidyoConnector* c, const char* filename);

/**
{function apigroup="connector":
	{name: VidyoConnectorStartNextLogFile}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Stops logging into current file and start logging to the next one.
	This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorStartNextLogFile(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorStartNextLogFile(VidyoConnector* c);

/**
{function apigroup="connector":
	{name: VidyoConnectorSetFileLoggerRotationParameters}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets log file size limit, how often to initiate the wrap, and number of log files.
	This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetFileLoggerRotationParameters(VidyoConnector* c, LmiSizeT maxSize, LmiTime wrapPeriod, LmiSizeT wrapCount)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: maxSize}
		{description: The number of bytes of logged messages to save.}
	}
	{parameter:
		{name: wrapPeriod}
		{description: How often to initiate the wrap in nanoseconds. If this value is LMI_TIME_INFINITE, the log file will not be wrapped based on time.}
	}
	{parameter:
		{name: wrapCount}
		{description: The maximum number of files to keep.}
	}
	{return:  LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSetFileLoggerRotationParameters(VidyoConnector* c, LmiSizeT maxSize, LmiTime wrapPeriod, LmiSizeT wrapCount);

/**
{type apigroup="connector":
	{name: VidyoConnectorProperty}
	{parent: VidyoConnector}
	{description: VidyoConnector properties that can be set.}
	{value: {name: VIDYO_CONNECTORPROPERTY_ApplicationName}
		{description: Application name property.}}
	{value: {name: VIDYO_CONNECTORPROPERTY_ApplicationVersion}
		{description: Application version property.}}
}
*/
typedef enum {
	VIDYO_CONNECTORPROPERTY_ApplicationName,
	VIDYO_CONNECTORPROPERTY_ApplicationVersion
} VidyoConnectorProperty;

/**
{type apigroup="connector" visibility="public":
	{name: VidyoConnectorProductInformation}
	{parent: Connector}
	{description: Describes the information about the product.}
	{member:
		{name: propertyName}
		{type: VidyoConnectorProperty}
		{description: Describes the property name. Refer VidyoConnectorProperty.}
	}
	{member:
		{name: value}
		{type: LmiString}
		{description: This represents the value of the property.}
	}
}
*/
typedef struct
{	
	VidyoConnectorProperty propertyName;
	LmiString value;
} VidyoConnectorProductInformation;

/**
{function visibility="private":
	{name: VidyoConnectorProductInformationConstruct}
	{parent: VidyoConnectorProductInformation}
	{description: Constructs product with initial values (in CStr format).}
	{prototype: VidyoConnectorProductInformation* VidyoConnectorProductInformationConstruct(VidyoConnectorProductInformation* proInfo, VidyoConnectorProperty propertyName, const LmiString* value)}
		{parameter:
			{name: proInfo}
			{description: The VidyoConnectorProductInformation object to be constructed.}
		}
		{parameter:
			{name: propertyName}
			{description: The name of the property, refer VidyoConnectorProperty.}
		}
		{parameter:
			{name: value}
			{description: The value of the property.}
		}
	{return: p if the construct operation was successful; NULL otherwise.}
}
*/
VidyoConnectorProductInformation* VidyoConnectorProductInformationConstruct(VidyoConnectorProductInformation* proInfo, VidyoConnectorProperty propertyName, const LmiString* value);

/**
{function visibility="private":
	{name: VidyoConnectorProductInformationConstructDefault}
	{parent: VidyoConnectorProductInformation}
	{description: Constructs default Product Information object.}
	{prototype: VidyoConnectorProductInformation* VidyoConnectorProductInformationConstructDefault(VidyoConnectorProductInformation* proInfo, LmiAllocator* alloc)}
	{parameter:
		{name: proInfo}
		{description: The object to construct.}
	}
	{parameter:
		{name: alloc}
		{description: LmiAllocator object.}
	}
	{return: The pointer to constructed object on success, or NULL on failure}
}
*/
VidyoConnectorProductInformation* VidyoConnectorProductInformationConstructDefault(VidyoConnectorProductInformation* proInfo, LmiAllocator* alloc);

/**
{function visibility="private":
	{name: VidyoConnectorProductInformationConstructCopy}
	{parent: VidyoConnectorProductInformation}
	{description: Constructs copy of object.}
	{prototype: VidyoConnectorProductInformation* VidyoConnectorProductInformationConstructCopy(VidyoConnectorProductInformation* d, const VidyoConnectorProductInformation* s)}
	{parameter:
		{name: d}
		{description: The object to construct.}
	}
	{parameter:
		{name: s}
		{description: The object to make copy of.}
	}
	{return: The pointer to constructed object on success, or NULL on failure}
}
*/
VidyoConnectorProductInformation* VidyoConnectorProductInformationConstructCopy(VidyoConnectorProductInformation* d, const VidyoConnectorProductInformation* s);

/**
{function visibility="private":
	{name: VidyoConnectorProductInformationAssign}
	{parent: VidyoConnectorProductInformation}
	{description: Assigns one object the value of another.}
	{prototype: VidyoConnectorProductInformation* VidyoConnectorProductInformationAssign(VidyoConnectorProductInformation* d, const VidyoConnectorProductInformation* s)}
	{parameter:
		{name: d}
		{description: The object to assign to.}
	}
	{parameter:
		{name: s}
		{description: The object to assign from.}
	}
	{return: The pointer to constructed object on success, or NULL on failure}
}
*/
VidyoConnectorProductInformation* VidyoConnectorProductInformationAssign(VidyoConnectorProductInformation* d, const VidyoConnectorProductInformation* s);

/**
{function visibility="private":
	{name: VidyoConnectorProductInformationDestruct}
	{parent: VidyoConnectorProductInformation}
	{description: Destructs the object.}
	{prototype: void VidyoConnectorProductInformationDestruct(VidyoConnectorProductInformation* proInfo)}
	{parameter:
		{name: proInfo}
		{description: Product Information object.}
	}
}
*/
void VidyoConnectorProductInformationDestruct(VidyoConnectorProductInformation* proInfo);

Declare_LmiVector(VidyoConnectorProductInformation)

/**
 {function apigroup="connector":
	{name: VidyoConnectorSetProductInfo}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Sets the product information.
		This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorSetProductInfo(VidyoConnector* c, const LmiVector(VidyoConnectorProductInformation)* productInfo)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: productInfo}
		{description: Sets the product information. Supported Properties- ApplicationName, ApplicationVersion}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
 */
LmiBool VidyoConnectorSetProductInfo(VidyoConnector* c, const LmiVector(VidyoConnectorProductInformation)* productInfo);

/**
 {function apigroup="connector":
	{name: VidyoConnectorRegisterPlugin}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Registers plugin. This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return NULL.}
	{prototype: LmiVoidPtr VidyoConnectorRegisterPlugin(VidyoConnector* c, const LmiVoidPtr plugin)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: plugin} {description: Plugin to register.}}
	{return: Newly registered plugin handle or NULL on failure.}
}
*/
LmiVoidPtr VidyoConnectorRegisterPlugin(VidyoConnector* c, const LmiVoidPtr plugin);

/**
 {function apigroup="connector":
	{name: VidyoConnectorUnregisterPlugin}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Unregisters plugin. This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterPlugin(VidyoConnector* c, const LmiVoidPtr plugin)}
	{parameter: {name: c} {description: The VidyoConnector object.}}
	{parameter: {name: plugin} {description: Plugin to unregister.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterPlugin(VidyoConnector* c, const LmiVoidPtr plugin);

/**
{type:
	{name: VidyoConnectorTokenType}
	{parent: VidyoConnector}
	{description: An enumeration that represents the type of token.}
	{value:
		{name: VIDYO_CONNECTORTOKENTYPE_RefreshToken}
		{description: Represents the associated token is a refresh token.}}
	{value:
		{name: VIDYO_CONNECTORTOKENTYPE_JwtToken}
		{description: Represents the associated token is a jwt token.}}
	{value:
		{name: VIDYO_CONNECTORTOKENTYPE_Undefined}
		{description: Represents the associated token is undefined.}}
}
*/
typedef enum
{
	VIDYO_CONNECTORTOKENTYPE_RefreshToken,
	VIDYO_CONNECTORTOKENTYPE_JwtToken,
	VIDYO_CONNECTORTOKENTYPE_Undefined,
} VidyoConnectorTokenType;

/**
{type apigroup="connector" visibility="public":
	{name: VidyoConnectorTokenInfo}
	{parent: Connector}
	{description: Represents the structure associate with token info.}
	{member:
		{name: token}
		{type: LmiString}
		{description: Represents the value or data related to token.}}
	{member:
		{name: tokenType}
		{type: VidyoConnectorTokenType}
		{description: Represents the type of token, please refer VidyoConnectorTokenType.}}
}
*/
typedef struct
{
	LmiString token;
	VidyoConnectorTokenType tokenType;
}VidyoConnectorTokenInfo;

/**
{function visibility="private":
	{name: VidyoConnectorTokenInfoConstructDefault}
	{parent: VidyoConnectorTokenInfo}
	{description: Constructs an VidyoConnectorTokenInfo object.}
	{prototype: VidyoConnectorTokenInfo* VidyoConnectorTokenInfoConstructDefault(VidyoConnectorTokenInfo* tokenInfo, LmiAllocator* alloc)}
	{parameter:
		{name: tokenInfo}
		{description: Represents the VidyoConnectorTokenInfo object to construct.}}
	{parameter:
		{name: alloc}
		{description: Represents the LmiAllocator object.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoConnectorTokenInfo* VidyoConnectorTokenInfoConstructDefault(VidyoConnectorTokenInfo* tokenInfo, LmiAllocator* alloc);

/**
{function visibility="private":
	{name: VidyoConnectorTokenInfoConstruct}
	{parent: VidyoConnectorTokenInfo}
	{description: Constructs an VidyoConnectorTokenInfo object.}
	{prototype: VidyoConnectorTokenInfo* VidyoConnectorTokenInfoConstruct(VidyoConnectorTokenInfo* tokenInfo, const LmiString* token, VidyoConnectorTokenType tokenType, LmiAllocator* alloc)}
	{parameter:
		{name: tokenInfo}
		{description: Represents the VidyoConnectorTokenInfo object to construct.}}
	{parameter:
		{name: token}
		{description: Represents the token in string format.}}
	{parameter:
		{name: tokenType}
		{description: Represents the type of token, please refer VidyoConnectorTokenType.}}
	{parameter:
		{name: alloc}
		{description: Represents the LmiAllocator object.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoConnectorTokenInfo* VidyoConnectorTokenInfoConstruct(VidyoConnectorTokenInfo* tokenInfo, const LmiString* token, VidyoConnectorTokenType tokenType, LmiAllocator* alloc);

/**
{function visibility="private":
	{name: VidyoConnectorTokenInfoConstructCopy}
	{parent: VidyoConnectorTokenInfo}
	{description: Constructs an VidyoConnectorTokenInfo object as a copy of an existing object.}
	{prototype: VidyoConnectorTokenInfo* VidyoConnectorTokenInfoConstructCopy(VidyoConnectorTokenInfo* d, const VidyoConnectorTokenInfo* s)}
	{parameter:
		{name: d}
		{description: The VidyoConnectorTokenInfo object to construct.}}
	{parameter:
		{name: s}
		{description: The existing VidyoConnectorTokenInfo object to copy.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoConnectorTokenInfo* VidyoConnectorTokenInfoConstructCopy(VidyoConnectorTokenInfo* d, const VidyoConnectorTokenInfo* s);

/**
 {function visibility="private":
	{name: VidyoConnectorTokenInfoAssign}
	{parent: VidyoConnectorTokenInfo}
	{description: Assign the contents of one VidyoConnectorTokenInfo object to another.}
	{prototype: VidyoConnectorTokenInfo* VidyoConnectorTokenInfoAssign(VidyoConnectorTokenInfo* d, const VidyoConnectorTokenInfo* s)}
	{parameter:
		{name: d}
		{description: The VidyoConnectorTokenInfo object to assign to.}}
	{parameter:
		{name: s}
		{description: The VidyoConnectorTokenInfo object to copy.}}
	{return: A pointer to the assigned object on success, or NULL on failure.}
}
*/
VidyoConnectorTokenInfo* VidyoConnectorTokenInfoAssign(VidyoConnectorTokenInfo* d, const VidyoConnectorTokenInfo* s);

/**
{function visibility="private":
	{name: VidyoConnectorTokenInfoDestruct}
	{parent: VidyoConnectorTokenInfo}
	{description: Destructs an VidyoConnectorTokenInfo object.}
	{prototype: void VidyoConnectorTokenInfoDestruct(VidyoConnectorTokenInfo* tokenInfo)}
	{parameter:
		{name: tokenInfo}
		{description: Represents the VidyoConnectorTokenInfo object to destruct.}}
}
*/
void VidyoConnectorTokenInfoDestruct(VidyoConnectorTokenInfo* tokenInfo);

Declare_LmiVector(VidyoConnectorTokenInfo)

/**
{callback:
	{name: VidyoConnectorOnTokenReceived}
	{parent: VidyoConnector}
	{description: This is private callback that is triggered upon successful received tokens from the vidyo portal, for usage please contact vidyo api support team.}
	{prototype: void(*VidyoConnectorOnTokenReceived)(const VidyoConnector* c, const LmiVector(VidyoConnectorTokenInfo)* tokenInfo)}
	{parameter:
		{name: c}
		{description: Represents the VidyoConnector object that triggered the callback.}
	}
	{parameter:
		{name: tokenInfo}
		{description: Represents the vector of VidyoConnectorTokenInfo object.}
	}
}
*/
typedef void(*VidyoConnectorOnTokenReceived)(const VidyoConnector* c, const LmiVector(VidyoConnectorTokenInfo)* tokenInfo);

/**
{function:
	{name: VidyoConnectorRegisterTokenReceivedEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous and private interface, for usage please contact vidyo api support team. Application can register a private callback to receive tokens from the vidyo portal.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterTokenReceivedEventListener(VidyoConnector* c, VidyoConnectorOnTokenReceived onTokenReceived)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: onTokenReceived}
		{description: Callback that is triggered when the tokens are received from the vidyo portal.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterTokenReceivedEventListener(VidyoConnector* c, VidyoConnectorOnTokenReceived onTokenReceived);

/**
{function:
	{name: VidyoConnectorUnregisterTokenReceivedEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous and private interface, for usage please contact vidyo api support team. Application can unregister private callback to stop receiving tokens from the vidyo portal.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterTokenReceivedEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterTokenReceivedEventListener(VidyoConnector* c);

/**
{type apigroup="connector" visibility="public":
	{name:  VidyoConnectorBotInfo}
	{parent: VidyoConnector}
	{description: Represents the bot info structure.}
	{member:
		{name: name}
		{type: LmiString}
		{description: Represents the name associated with bot.}}
	{member:
		{name: id}
		{type: LmiString}
		{description: Represents the id associated with bot.}}
	{member:
		{name: data}
		{type: LmiString}
		{description: Represents the meta data associated with bot in json format string.}}
}
*/
typedef struct
{
	LmiString name;
	LmiString id;
	LmiString data;
} VidyoConnectorBotInfo;

/**
{function visibility="private":
	{name: VidyoConnectorBotInfoConstructDefault}
	{parent: VidyoConnectorBotInfo}
	{description: Constructs an VidyoConnectorBotInfo object.}
	{prototype: VidyoConnectorBotInfo* VidyoConnectorBotInfoConstructDefault(VidyoConnectorBotInfo* info, LmiAllocator* alloc)}
	{parameter:
		{name: info}
		{description: Represents the VidyoConnectorBotInfo object to construct.}}
	{parameter:
		{name: alloc}
		{description: Represents the LmiAllocator object.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoConnectorBotInfo* VidyoConnectorBotInfoConstructDefault(VidyoConnectorBotInfo* info, LmiAllocator* alloc);

/**
{function visibility="private":
	{name: VidyoConnectorBotInfoConstruct}
	{parent: VidyoConnectorBotInfo}
	{description: Constructs an VidyoConnectorBotInfo object.}
	{prototype: VidyoConnectorBotInfo* VidyoConnectorBotInfoConstruct(VidyoConnectorBotInfo* info, const LmiString* name, const LmiString* id, const LmiString* data, LmiAllocator* alloc)}
	{parameter:
		{name: info}
		{description: Represents the VidyoConnectorBotInfo object to construct.}}
	{parameter:
		{name: name}
		{description: Represents the name associated with bot.}}
	{parameter:
		{name: id}
		{description: Represents the id associated with bot.}}
	{parameter:
		{name: data}
		{description: Represents the meta data associated with bot in json format string.}}
	{parameter:
		{name: alloc}
		{description: Represents the LmiAllocator object.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoConnectorBotInfo* VidyoConnectorBotInfoConstruct(VidyoConnectorBotInfo* info, const LmiString* name, const LmiString* id, const LmiString* data, LmiAllocator* alloc);

/**
{function visibility="private":
	{name: VidyoConnectorBotInfoConstructCopy}
	{parent: VidyoConnectorBotInfo}
	{description: Constructs an VidyoConnectorBotInfo object as a copy of an existing object.}
	{prototype: VidyoConnectorBotInfo* VidyoConnectorBotInfoConstructCopy(VidyoConnectorBotInfo* d, const VidyoConnectorBotInfo* s)}
	{parameter:
		{name: d}
		{description: The VidyoConnectorBotInfo object to construct.}}
	{parameter:
		{name: s}
		{description: The existing VidyoConnectorBotInfo object to copy.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoConnectorBotInfo* VidyoConnectorBotInfoConstructCopy(VidyoConnectorBotInfo* d, const VidyoConnectorBotInfo* s);

/**
{function visibility="private":
	{name: VidyoConnectorBotInfoAssign}
	{parent: VidyoConnectorBotInfo}
	{description: Assign the contents of one VidyoConnectorBotInfo object to another.}
	{prototype: VidyoConnectorBotInfo* VidyoConnectorBotInfoAssign(VidyoConnectorBotInfo* d, const VidyoConnectorBotInfo* s)}
	{parameter:
		{name: d}
		{description: The VidyoConnectorBotInfo object to assign to.}}
	{parameter:
		{name: s}
		{description: The VidyoConnectorBotInfo object to copy.}}
	{return: A pointer to the assigned object on success, or NULL on failure.}
}
*/
VidyoConnectorBotInfo* VidyoConnectorBotInfoAssign(VidyoConnectorBotInfo* d, const VidyoConnectorBotInfo* s);

/**
{function visibility="private":
	{name: VidyoConnectorBotInfoDestruct}
	{parent: VidyoConnectorBotInfo}
	{description: Destructs an VidyoConnectorBotInfo object.}
	{prototype: void VidyoConnectorBotInfoDestruct(VidyoConnectorBotInfo* info)}
	{parameter:
		{name: info}
		{description: Represents the VidyoConnectorBotInfo object to destruct.}}
}
*/
void VidyoConnectorBotInfoDestruct(VidyoConnectorBotInfo* info);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnBotJoined}
	{parent: VidyoConnector}
	{description: Callback that is triggered when bot participant joins a conference.}
	{prototype: void(*VidyoConnectorOnBotJoined)(VidyoConnector* c, const VidyoConnectorBotInfo* info)}
	{parameter: {name: c} {description: The VidyoConnector object that triggered the callback.}}
	{parameter: {name: info} {description: Represents the VidyoConnectorBotInfo object.}}
}
*/
typedef void(*VidyoConnectorOnBotJoined)(VidyoConnector* c, const VidyoConnectorBotInfo* info);

/**
{callback apigroup="connector":
	{name: VidyoConnectorOnBotLeft}
	{parent: VidyoConnector}
	{description: Callback that is triggered when an existing bot participant leaves a conference.}
	{prototype: void(*VidyoConnectorOnBotLeft)(VidyoConnector* c, const VidyoConnectorBotInfo* info)}
	{parameter: {name: c} {description: The VidyoConnector object that triggered the callback.}}
	{parameter: {name: info} {description: Represents the VidyoConnectorBotInfo object.}}
}
*/
typedef void(*VidyoConnectorOnBotLeft)(VidyoConnector* c, const VidyoConnectorBotInfo* info);

/**
{function:
	{name: VidyoConnectorRegisterBotEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Application can register a callback to get notified about bot participants.
					This API is allowed only in VIDYO_CONNECTORSTATE_Ready state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorRegisterBotEventListener(VidyoConnector* c, VidyoConnectorOnBotJoined onJoined, VidyoConnectorOnBotLeft onLeft)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}}
	{parameter:
		{name: onJoined}
		{description: Callback that is triggered when bot participant joins a conference.}}
	{parameter:
		{name: onLeft}
		{description: Callback that is triggered when an existing bot participant leaves a conference.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorRegisterBotEventListener(VidyoConnector* c, VidyoConnectorOnBotJoined onJoined, VidyoConnectorOnBotLeft onLeft);

/**
{function:
	{name: VidyoConnectorUnregisterBotEventListener}
	{parent: VidyoConnector}
	{description: This is a synchronous interface. Application can unregister callback to stop bot participant event notifications.
					This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorUnregisterBotEventListener(VidyoConnector* c)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorUnregisterBotEventListener(VidyoConnector* c);

/**
{function:
	{name: VidyoConnectorShowStatisticsDialog}
	{parent: VidyoConnector}
	{description: This is a synchronous interface to display statistics dialog box for a conference on desktop devices. It is not applicable for handheld devices.
					This API is allowed only in VIDYO_CONNECTORSTATE_Connected state of VidyoConnector. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoConnectorShowStatisticsDialog(VidyoConnector* c, LmiBool showStatisticsDialog)}
	{parameter:
		{name: c}
		{description: The VidyoConnector object.}
	}
	{parameter:
		{name: showStatisticsDialog}
		{description: Boolean value that indicate to show or hide the statistics dialog.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorShowStatisticsDialog(VidyoConnector* c, LmiBool showStatisticsDialog);

/**
{function apigroup="connector":
    {name: VidyoConnectorSelectVirtualRenderer}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Selects the virtual renderer to receive all events required for rendering. On, iOS, this API must be called from main(UI) thread. This interface is allowed to be called in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle, VIDYO_CONNECTORSTATE_Dead and VIDYO_CONNECTORSTATE_Disabling. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorSelectVirtualRenderer(VidyoConnector* c, const VidyoVirtualRenderer* renderer)}
    {parameter: {name: c} {description: The VidyoConnector object.}}
    {parameter:
        {name: renderer}
        {description: The virtual renderer object to select, NULL to unselect.}
    }
    {return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoConnectorSelectVirtualRenderer(VidyoConnector* c, const VidyoVirtualRenderer* renderer);

/**
{type apigroup="connector" visibility="public":
    {name: VidyoConnectorNGRActionType}
    {parent: VidyoConnector}
    {description: Lists available actions for next generation renderer library.}
    {value:
        {name: VIDYO_CONNECTORNGRACTIONTYPE_SetLayout}
        {description: Sets the layout.}}
    {value:
        {name: VIDYO_CONNECTORNGRACTIONTYPE_SetViewSize}
        {description: Sets the view size.}}
    {value: 
        {name: VIDYO_CONNECTORNGRACTIONTYPE_SetDebugInfoVisible}
        {description: Sets the visibility of debug information.}}
    {value: 
        {name: VIDYO_CONNECTORNGRACTIONTYPE_SetLabelVisible}
        {description: Sets the visibility of labels on renderer.}}
    {value: 
        {name: VIDYO_CONNECTORNGRACTIONTYPE_SetAudioMeterVisible}
        {description: Sets the visibility of audio meter on tiles.}}
    {value: 
        {name: VIDYO_CONNECTORNGRACTIONTYPE_SetBackgroundColor}
        {description: Sets the background color.}}
    {value: 
        {name: VIDYO_CONNECTORNGRACTIONTYPE_SetDensity}
        {description: Sets the density.}}
    {value: 
        {name: VIDYO_CONNECTORNGRACTIONTYPE_SetMaxRemoteParticipants}
        {description: Sets the max remote participants.}}
    {value: 
        {name: VIDYO_CONNECTORNGRACTIONTYPE_PinParticipant}
        {description: Pin the participants.}}
    {value:
        {name: VIDYO_CONNECTORNGRACTIONTYPE_ShowAudioTiles}
        {description: Sets whether to show audio only tiles.}}
    {value:
        {name: VIDYO_CONNECTORNGRACTIONTYPE_Invalid} 
        {description: Invalid action.}}
}
*/

typedef enum
{
    VIDYO_CONNECTORNGRACTIONTYPE_SetLayout,
    VIDYO_CONNECTORNGRACTIONTYPE_SetViewSize,
    VIDYO_CONNECTORNGRACTIONTYPE_SetDebugInfoVisible,
    VIDYO_CONNECTORNGRACTIONTYPE_SetLabelVisible,
    VIDYO_CONNECTORNGRACTIONTYPE_SetAudioMeterVisible,
    VIDYO_CONNECTORNGRACTIONTYPE_SetBackgroundColor,
    VIDYO_CONNECTORNGRACTIONTYPE_SetDensity,
    VIDYO_CONNECTORNGRACTIONTYPE_SetMaxRemoteParticipants,
    VIDYO_CONNECTORNGRACTIONTYPE_PinParticipant,
    VIDYO_CONNECTORNGRACTIONTYPE_ShowAudioTiles,
    VIDYO_CONNECTORNGRACTIONTYPE_Invalid
} VidyoConnectorNGRActionType;

/**
{callback apigroup="connector":
    {name: VidyoConnectorOnNGRResult}
    {parent: VidyoConnector}
    {description: Callback providing final response for actions requested to next generation renderer.}
    {prototype: void(*VidyoConnectorOnNGRResult)(const VidyoConnector* c, VidyoConnectorNGRActionType type, LmiBool success, const LmiString* error)}
    {parameter:
        {name: c}
        {description: Represent the pointer to VidyoConnector object that triggered the callback. }}
    {parameter:
        {name: type}
        {description: Represent the type of action. Refer VidyoConnectorNGRActionType.}}
    {parameter:
        {name: success}
        {description: Represent the final response.}}
    {parameter:
        {name: error}
        {description: Represent the description of error, empty otherwise.}}
}
*/
typedef void(*VidyoConnectorOnNGRResult)(const VidyoConnector* c, VidyoConnectorNGRActionType type, LmiBool success, const LmiString* error);

/**
{function apigroup="connector":
    {name: VidyoConnectorRegisterNGRResultEventListener}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Register a callback to receive the final response for the Next Generation Renderer action.
                    This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorRegisterNGRResultEventListener(VidyoConnector* c, VidyoConnectorOnNGRResult onNGRResult)}
    {parameter: 
        {name: c}
        {description: VidyoConnector object.}}
    {parameter: 
        {name: onNGRResult}
        {description: Callback to be triggered upon receive the final response from next generation renderer library.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorRegisterNGRResultEventListener(VidyoConnector* c, VidyoConnectorOnNGRResult onNGRResult);

/**
{function apigroup="connector":
    {name: VidyoConnectorUnregisterNGRResultEventListener}
    {parent: VidyoConnector}
    {description: This is a synchronous interface. Unregister the callback to stop receiving the final response for the next generation renderer action.
                    This API is allowed in all states of VidyoConnector except VIDYO_CONNECTORSTATE_Idle and VIDYO_CONNECTORSTATE_Dead. If used in another state, no action will be taken and return LMI_FALSE.}
    {prototype: LmiBool VidyoConnectorUnregisterNGRResultEventListener(VidyoConnector* c)}
    {parameter: 
        {name: c}
        {description: VidyoConnector object.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoConnectorUnregisterNGRResultEventListener(VidyoConnector* c);

LMI_END_EXTERN_C

#if LMI_INLINE_NEED_HEADER_FILE_DEFINITIONS
#include "VidyoConnectorInline.h"
#endif

#endif /* VIDYO_CONNECTOR_H_ */

