/**
{file:
	{name: LmiCameraControl.h}
	{description: }
	{copyright:
		(c) 2006-2019 Vidyo, Inc.,
		433 Hackensack Avenue,
		Hackensack, NJ  07601.

		All rights reserved.

		The information contained herein is proprietary to Vidyo, Inc.
		and shall not be reproduced, copied (in whole or in part), adapted,
		modified, disseminated, transmitted, transcribed, stored in a retrieval
		system, or translated into any language in any form by any means
		without the express written consent of Vidyo, Inc.
		                  ***** CONFIDENTIAL *****
	}
}
*/
#ifndef LMI_CAMERACONTROL_H
#define LMI_CAMERACONTROL_H

#include <Lmi/Utils/LmiTypes.h>
#include <Lmi/Utils/LmiAllocator.h>
#include <Lmi/Utils/LmiString.h>
#include <Lmi/Utils/LmiVector.h>
#include <Lmi/Os/LmiTime.h>
#include <Lmi/Utils/LmiBasicTypesContainer.h>
#include <Lmi/Os/LmiDataBuffer.h>

LMI_BEGIN_EXTERN_C

/**
{package:
	{name: CameraControl}
	{parent: VideoCommon}
	{include: Lmi/Video/Common/LmiCameraControl.h}
	{description: This package contains commands for controlling cameras and objects that describe a camera's ability to be controlled.}
}
*/

/**
{type:
	{name: LmiCameraPreset}
	{parent: CameraControl}
	{include: Lmi/Video/Common/LmiCameraControl.h}
	{description: A camera's preset capabilities.}
}
*/
typedef struct
{
	LmiUint index;
	LmiString name;
} LmiCameraPreset;

/**
{function:
	{name: LmiCameraPresetConstruct}
	{parent: LmiCameraPreset}
	{description: Constructs an LmiCameraPreset object.}
	{prototype: LmiCameraPreset *LmiCameraPresetConstruct(LmiCameraPreset *preset, LmiUint index, const LmiString *name, LmiAllocator *alloc)}
	{parameter:
		{name: preset}
		{description: The LmiCameraPreset object to be constructed.}
	}
	{parameter:
		{name: index}
		{description: The index number of preset.}
	}
	{parameter:
		{name: name}
		{description: The friendly name of preset.}
	}
	{parameter:
		{name: alloc}
		{description: The LmiAllocator object.}
	}
	{return: LmiCameraPreset object on success otherwise NULL.}
}
*/
LmiCameraPreset *LmiCameraPresetConstruct(LmiCameraPreset *preset, LmiUint index, const LmiString *name, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraPresetConstructDefault}
	{parent: LmiCameraPreset}
	{description: Constructs default LmiCameraPreset object.}
	{prototype: LmiCameraPreset *LmiCameraPresetConstructDefault(LmiCameraPreset *preset, LmiAllocator *alloc)}
	{parameter:
		{name: preset}
		{description: The LmiCameraPreset object to be constructed.}
	}
	{parameter:
		{name: alloc}
		{description: LmiAllocator object.}
	}
	{return: LmiCameraPreset object on success otherwise NULL.}
}
*/
LmiCameraPreset *LmiCameraPresetConstructDefault(LmiCameraPreset *preset, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraPresetConstructCopy}
	{parent: LmiCameraPreset}
	{description: Constructs an LmiCameraPreset object as a copy of an existing object.}
	{prototype: LmiCameraPreset *LmiCameraPresetConstructCopy(LmiCameraPreset *d, const LmiCameraPreset *s)}
	{parameter:
		{name: d}
		{description: The LmiCameraPreset object to construct.}
	}
	{parameter:
		{name: s}
		{description: The LmiCameraPreset object to make a copy of.}
	}
	{return: LmiCameraPreset object on success otherwise NULL.}
}
*/
LmiCameraPreset *LmiCameraPresetConstructCopy(LmiCameraPreset *d, const LmiCameraPreset *s);

/**
{function:
	{name: LmiCameraPresetAssign}
	{parent: LmiCameraPreset}
	{description: Assign a LmiCameraPreset object as a copy of an existing object.}
	{prototype: LmiCameraPreset *LmiCameraPresetAssign(LmiCameraPreset *d, const LmiCameraPreset *s)}
	{parameter:
		{name: d}
		{description: The LmiCameraPreset object to be constructed as a copy.}
	}
	{parameter:
		{name: s}
		{description: An existing LmiCameraPreset object.}
	}
	{return: LmiCameraPreset object on success otherwise NULL.}
}
*/
LmiCameraPreset *LmiCameraPresetAssign(LmiCameraPreset *d, const LmiCameraPreset *s);

/**
{function:
	{name: LmiCameraPresetEqual}
	{parent: LmiCameraPreset}
	{description: Compare two LmiCameraPreset objects for equality.}
	{prototype: LmiBool LmiCameraPresetEqual(const LmiCameraPreset *preset, const LmiCameraPreset *other)}
	{parameter:
		{name: preset}
		{description: The first LmiCameraPreset object.}
	}
	{parameter:
		{name: other}
		{description: The second LmiCameraPreset object.}
	}
	{return: A boolean indicating whether the two objects are equal.}
}
*/
LmiBool LmiCameraPresetEqual(const LmiCameraPreset *preset, const LmiCameraPreset *other);

/**
{function:
	{name: LmiCameraPresetDestruct}
	{parent: LmiCameraPreset}
	{description: Destructs the LmiCameraPreset object.}
	{prototype: void LmiCameraPresetDestruct(LmiCameraPreset *preset)}
	{parameter:
		{name: preset}
		{description: The LmiCameraPreset object.}
	}
}
*/
void LmiCameraPresetDestruct(LmiCameraPreset *preset);

/**
{function:
    {name: LmiCameraPresetGetIndex}
    {parent: LmiCameraPreset}
    {description: Gets the preset index.}
    {prototype: LmiUint LmiCameraPresetGetIndex(const LmiCameraPreset *preset)}
    {parameter: {name: preset} {description: The LmiCameraPreset object.}}
    {return: The index number of preset.}
}
*/
LMI_INLINE_DECLARATION LmiUint LmiCameraPresetGetIndex(const LmiCameraPreset *preset);

/**
{function:
    {name: LmiCameraPresetGetName}
    {parent: LmiCameraPreset}
    {description: Gets the preset friendly name.}
    {prototype: const LmiString *LmiCameraPresetGetName(const LmiCameraPreset *preset)}
    {parameter: {name: preset} {description: The LmiCameraPreset object.}}
    {return: The friendly name of preset.}
}
*/
LMI_INLINE_DECLARATION const LmiString *LmiCameraPresetGetName(const LmiCameraPreset *preset);

Declare_LmiVector(LmiCameraPreset)
Declare_LmiVectorEqual(LmiCameraPreset)

/**
{type:
	{name: LmiCameraControlCapabilities}
	{parent: CameraControl}
	{include: Lmi/Video/Common/LmiCameraControl.h}
	{description: A camera's control capabilities.}
}
*/
typedef struct
{
	LmiBool hasAbsolutePanTilt;
	LmiRange(LmiFloat64) panDegreeRange;
	LmiRange(LmiFloat64) tiltDegreeRange;
	LmiBool hasAbsoluteZoom;
	LmiRange(LmiFloat64) zoomMagnificationRange;
	LmiBool hasRelativePanTilt;
	LmiRange(LmiInt) panSpeedRange;
	LmiRange(LmiInt) tiltSpeedRange;
	LmiBool hasRelativeZoom;
	LmiRange(LmiInt) zoomSpeedRange;
	LmiBool hasPhotoCapture;

	LmiBool hasOldEndpoint;
	LmiBool hasViscaSupport;
	LmiBool hasPresetSupport;

	LmiVector(LmiCameraPreset) preset;
} LmiCameraControlCapabilities;

/**
{function:
	{name: LmiCameraControlCapabilitiesConstructDefault}
	{parent: LmiCameraControlCapabilities}
	{description: Constructs an LmiCameraControlCapabilities object with no capabilities.}
	{prototype: LmiCameraControlCapabilities *LmiCameraControlCapabilitiesConstructDefault(LmiCameraControlCapabilities *caps, LmiAllocator *alloc)}
	{parameter: {name: caps} {description: The LmiCameraControlCapabilities object to construct.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraControlCapabilities *LmiCameraControlCapabilitiesConstructDefault(LmiCameraControlCapabilities *caps, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraControlCapabilitiesConstructCopy}
	{parent: LmiCameraControlCapabilities}
	{description: Constructs an LmiCameraControlCapabilities object as a copy of another.}
	{prototype: LmiCameraControlCapabilities *LmiCameraControlCapabilitiesConstructCopy(LmiCameraControlCapabilities *caps, const LmiCameraControlCapabilities *other)}
	{parameter: {name: caps} {description: The LmiCameraControlCapabilities object to construct.}}
	{parameter: {name: other} {description: The LmiCameraControlCapabilities object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraControlCapabilities *LmiCameraControlCapabilitiesConstructCopy(LmiCameraControlCapabilities *caps, const LmiCameraControlCapabilities *other);

/**
{function:
	{name: LmiCameraControlCapabilitiesDestruct}
	{parent: LmiCameraControlCapabilities}
	{description: Destructs an LmiCameraControlCapabilities object.}
	{prototype: void LmiCameraControlCapabilitiesDestruct(LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The LmiCameraControlCapabilities object to destruct.}}
}
*/
void LmiCameraControlCapabilitiesDestruct(LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesAssign}
	{parent: LmiCameraControlCapabilities}
	{description: Assigns one LmiCameraControlCapabilities object the value of another.}
	{prototype: LmiCameraControlCapabilities *LmiCameraControlCapabilitiesAssign(LmiCameraControlCapabilities *caps, const LmiCameraControlCapabilities *other)}
	{parameter: {name: caps} {description: The LmiCameraControlCapabilities object to assign to.}}
	{parameter: {name: other} {description: The LmiCameraControlCapabilities object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/
LmiCameraControlCapabilities *LmiCameraControlCapabilitiesAssign(LmiCameraControlCapabilities *caps, const LmiCameraControlCapabilities *other);

/**
{function:
	{name: LmiCameraControlCapabilitiesEqual}
	{parent: LmiCameraControlCapabilities}
	{description: Compare two LmiCameraControlCapabilities objects for equality.}
	{prototype: LmiBool LmiCameraControlCapabilitiesEqual(const LmiCameraControlCapabilities *caps, const LmiCameraControlCapabilities *other)}
	{parameter: {name: caps} {description: The first LmiCameraControlCapabilities object.}}
	{parameter: {name: other} {description: The second LmiCameraControlCapabilities object.}}
	{return: A boolean indicating whether the two objects are equal.}
}
*/
LmiBool LmiCameraControlCapabilitiesEqual(const LmiCameraControlCapabilities *caps, const LmiCameraControlCapabilities *other);

/**
{function:
	{name: LmiCameraControlCapabilitiesSwap}
	{parent: LmiCameraControlCapabilities}
	{description: Swap the values of two LmiCameraControlCapabilities objects.}
	{prototype: LmiBool LmiCameraControlCapabilitiesSwap(LmiCameraControlCapabilities *caps, LmiCameraControlCapabilities *other)}
	{parameter: {name: caps} {description: The first LmiCameraControlCapabilities object.}}
	{parameter: {name: other} {description: The second LmiCameraControlCapabilities object.}}
	{return: A boolean indicating if swap is successful or not.}
}
*/
LmiBool LmiCameraControlCapabilitiesSwap(LmiCameraControlCapabilities *caps, LmiCameraControlCapabilities *other);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetAbsolutePanTilt}
	{parent: LmiCameraControlCapabilities}
	{description: Set whether absolute pan-tilt is supported.}
	{prototype: void LmiCameraControlCapabilitiesSetAbsolutePanTilt(LmiCameraControlCapabilities *caps, LmiBool supported)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: supported} {description: Whether absolute pan-tilt is supported.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetAbsolutePanTilt(LmiCameraControlCapabilities *caps, LmiBool supported);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetAbsoluteZoom}
	{parent: LmiCameraControlCapabilities}
	{description: Set whether absolute zoom is supported.}
	{prototype: void LmiCameraControlCapabilitiesSetAbsoluteZoom(LmiCameraControlCapabilities *caps, LmiBool supported)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: supported} {description: Whether absolute zoom is supported.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetAbsoluteZoom(LmiCameraControlCapabilities *caps, LmiBool supported);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetRelativePanTilt}
	{parent: LmiCameraControlCapabilities}
	{description: Set whether relative pan-tilt is supported.}
	{prototype: void LmiCameraControlCapabilitiesSetRelativePanTilt(LmiCameraControlCapabilities *caps, LmiBool supported)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: supported} {description: Whether relative pan-tilt is supported.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetRelativePanTilt(LmiCameraControlCapabilities *caps, LmiBool supported);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetRelativeZoom}
	{parent: LmiCameraControlCapabilities}
	{description: Set whether relative zoom is supported.}
	{prototype: void LmiCameraControlCapabilitiesSetRelativeZoom(LmiCameraControlCapabilities *caps, LmiBool supported)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: supported} {description: Whether relative zoom is supported.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetRelativeZoom(LmiCameraControlCapabilities *caps, LmiBool supported);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetPhotoCapture}
	{parent: LmiCameraControlCapabilities}
	{description: Set whether photo capture is supported.}
	{prototype: void LmiCameraControlCapabilitiesSetPhotoCapture(LmiCameraControlCapabilities *caps, LmiBool supported)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: supported} {description: Whether photo capture is supported.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetPhotoCapture(LmiCameraControlCapabilities *caps, LmiBool supported);

/**
{function:
	{name: LmiCameraControlCapabilitiesClear}
	{parent: LmiCameraControlCapabilities}
	{description: Remove all capabilities from an LmiCameraControlCapabilities object.}
	{prototype: void LmiCameraControlCapabilitiesClear(LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
}
*/
void LmiCameraControlCapabilitiesClear(LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesHasAbsolutePanTilt}
	{parent: LmiCameraControlCapabilities}
	{description: Check whether absolute pan-tilt is supported.}
	{prototype: LmiBool LmiCameraControlCapabilitiesHasAbsolutePanTilt(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: A boolean indicating if absolute pan-tilt is supported.}
}
*/
LMI_INLINE_DECLARATION LmiBool LmiCameraControlCapabilitiesHasAbsolutePanTilt(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesHasAbsoluteZoom}
	{parent: LmiCameraControlCapabilities}
	{description: Check whether absolute zoom is supported.}
	{prototype: LmiBool LmiCameraControlCapabilitiesHasAbsoluteZoom(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: A boolean indicating if absolute zoom is supported.}
}
*/
LMI_INLINE_DECLARATION LmiBool LmiCameraControlCapabilitiesHasAbsoluteZoom(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesHasRelativePanTilt}
	{parent: LmiCameraControlCapabilities}
	{description: Check whether relative pan-tilt is supported.}
	{prototype: LmiBool LmiCameraControlCapabilitiesHasRelativePanTilt(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: A boolean indicating if relative pan-tilt is supported.}
}
*/
LMI_INLINE_DECLARATION LmiBool LmiCameraControlCapabilitiesHasRelativePanTilt(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesHasRelativeZoom}
	{parent: LmiCameraControlCapabilities}
	{description: Check whether relative zoom is supported.}
	{prototype: LmiBool LmiCameraControlCapabilitiesHasRelativeZoom(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: A boolean indicating if relative zoom is supported.}
}
*/
LMI_INLINE_DECLARATION LmiBool LmiCameraControlCapabilitiesHasRelativeZoom(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesHasPhotoCapture}
	{parent: LmiCameraControlCapabilities}
	{description: Check whether photo capture is supported.}
	{prototype: LmiBool LmiCameraControlCapabilitiesHasPhotoCapture(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: A boolean indicating if photo capture is supported.}
}
*/
LMI_INLINE_DECLARATION LmiBool LmiCameraControlCapabilitiesHasPhotoCapture(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesHasAny}
	{parent: LmiCameraControlCapabilities}
	{description: Check whether any control features are supported.}
	{prototype: LmiBool LmiCameraControlCapabilitiesHasAny(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: A boolean indicating if any control features are supported.}
}
*/
LmiBool LmiCameraControlCapabilitiesHasAny(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetPanRange}
	{parent: LmiCameraControlCapabilities}
	{description: Set the range of possible positions for panning.}
	{prototype: void LmiCameraControlCapabilitiesSetPanRange(LmiCameraControlCapabilities *caps, const LmiRange(LmiFloat64) *range)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: range} {description: The range of values. The values are in degrees, with positive
		values representing panning to the right.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetPanRange(LmiCameraControlCapabilities *caps, const LmiRange(LmiFloat64) *range);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetPanRangeValues}
	{parent: LmiCameraControlCapabilities}
	{description: Set the range of possible positions for panning.}
	{prototype: void LmiCameraControlCapabilitiesSetPanRangeValues(LmiCameraControlCapabilities *caps, LmiFloat64 minValue, LmiFloat64 maxValue)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: minValue} {description: The minimum pan position in degrees.}}
	{parameter: {name: maxValue} {description: The maximum pan position in degrees.}}
	{note: Positive values represent panning to the right.}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetPanRangeValues(LmiCameraControlCapabilities *caps, LmiFloat64 minValue, LmiFloat64 maxValue);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetTiltRange}
	{parent: LmiCameraControlCapabilities}
	{description: Set the range of possible positions for tilting.}
	{prototype: void LmiCameraControlCapabilitiesSetTiltRange(LmiCameraControlCapabilities *caps, const LmiRange(LmiFloat64) *range)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: range} {description: The range of values. The values are in degrees, with positive
		values representing tilting up.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetTiltRange(LmiCameraControlCapabilities *caps, const LmiRange(LmiFloat64) *range);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetTiltRangeValues}
	{parent: LmiCameraControlCapabilities}
	{description: Set the range of possible positions for tilting.}
	{prototype: void LmiCameraControlCapabilitiesSetTiltRangeValues(LmiCameraControlCapabilities *caps, LmiFloat64 minValue, LmiFloat64 maxValue)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: minValue} {description: The minimum tilt position in degrees.}}
	{parameter: {name: maxValue} {description: The maximum tilt position in degrees.}}
	{note: Positive values represent tilting up.}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetTiltRangeValues(LmiCameraControlCapabilities *caps, LmiFloat64 minValue, LmiFloat64 maxValue);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetZoomRange}
	{parent: LmiCameraControlCapabilities}
	{description: Set the range of possible positions for zooming.}
	{prototype: void LmiCameraControlCapabilitiesSetZoomRange(LmiCameraControlCapabilities *caps, const LmiRange(LmiFloat64) *range)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: range} {description: The range of values. The values are unitless, representing the
		magnification factor from the default magnification.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetZoomRange(LmiCameraControlCapabilities *caps, const LmiRange(LmiFloat64) *range);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetZoomRangeValues}
	{parent: LmiCameraControlCapabilities}
	{description: Set the range of possible positions for zooming.}
	{prototype: void LmiCameraControlCapabilitiesSetZoomRangeValues(LmiCameraControlCapabilities *caps, LmiFloat64 minValue, LmiFloat64 maxValue)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: minValue} {description: The minimum magnification factor.}}
	{parameter: {name: maxValue} {description: The maximum magnification factor.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetZoomRangeValues(LmiCameraControlCapabilities *caps, LmiFloat64 minValue, LmiFloat64 maxValue);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetPanRange}
	{parent: LmiCameraControlCapabilities}
	{description: Get the range of possible positions for panning.}
	{prototype: const LmiRange(LmiFloat64) *LmiCameraControlCapabilitiesGetPanRange(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The range of values. The values are in degrees, with positive
		values representing panning to the right.}
}
*/
LMI_INLINE_DECLARATION const LmiRange(LmiFloat64) *LmiCameraControlCapabilitiesGetPanRange(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetMinPanValue}
	{parent: LmiCameraControlCapabilities}
	{description: Get the minimum possible pan position.}
	{prototype: LmiFloat64 LmiCameraControlCapabilitiesGetMinPanValue(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The minimum possible pan position in degrees.}
	{note: Positive values represent panning to the right.}
}
*/
LMI_INLINE_DECLARATION LmiFloat64 LmiCameraControlCapabilitiesGetMinPanValue(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetMaxPanValue}
	{parent: LmiCameraControlCapabilities}
	{description: Get the maximum possible pan position.}
	{prototype: LmiFloat64 LmiCameraControlCapabilitiesGetMaxPanValue(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The maximum possible pan position in degrees.}
	{note: Positive values represent panning to the right.}
}
*/
LMI_INLINE_DECLARATION LmiFloat64 LmiCameraControlCapabilitiesGetMaxPanValue(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetTiltRange}
	{parent: LmiCameraControlCapabilities}
	{description: Get the range of possible positions for tilting.}
	{prototype: const LmiRange(LmiFloat64) *LmiCameraControlCapabilitiesGetTiltRange(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The range of values. The values are in degrees, with positive
		values representing tilting up.}
}
*/
LMI_INLINE_DECLARATION const LmiRange(LmiFloat64) *LmiCameraControlCapabilitiesGetTiltRange(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetMinTiltValue}
	{parent: LmiCameraControlCapabilities}
	{description: Get the minimum possible tilt position.}
	{prototype: LmiFloat64 LmiCameraControlCapabilitiesGetMinTiltValue(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The minimum possible tilt position in degrees.}
	{note: Positive values represent tilting up.}
}
*/
LMI_INLINE_DECLARATION LmiFloat64 LmiCameraControlCapabilitiesGetMinTiltValue(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetMaxTiltValue}
	{parent: LmiCameraControlCapabilities}
	{description: Get the maximum possible tilt position.}
	{prototype: LmiFloat64 LmiCameraControlCapabilitiesGetMaxTiltValue(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The maximum possible tilt position in degrees.}
	{note: Positive values represent tilting up.}
}
*/
LMI_INLINE_DECLARATION LmiFloat64 LmiCameraControlCapabilitiesGetMaxTiltValue(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetZoomRange}
	{parent: LmiCameraControlCapabilities}
	{description: Get the range of possible magnification factors for zooming.}
	{prototype: const LmiRange(LmiFloat64) *LmiCameraControlCapabilitiesGetZoomRange(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The range of values. The values are unitless, representing the
		magnification factor from the default magnification.}
}
*/
LMI_INLINE_DECLARATION const LmiRange(LmiFloat64) *LmiCameraControlCapabilitiesGetZoomRange(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetMinZoomValue}
	{parent: LmiCameraControlCapabilities}
	{description: Get the minimum possible magnification factor for zooming.}
	{prototype: LmiFloat64 LmiCameraControlCapabilitiesGetMinZoomValue(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The minimum possible magnification factor. This is unitless, representing the
		magnification factor from the default magnification.}
}
*/
LMI_INLINE_DECLARATION LmiFloat64 LmiCameraControlCapabilitiesGetMinZoomValue(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetMaxZoomValue}
	{parent: LmiCameraControlCapabilities}
	{description: Get the maximum possible magnification factor for zooming.}
	{prototype: LmiFloat64 LmiCameraControlCapabilitiesGetMaxZoomValue(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The maximum possible magnification factor. This is unitless, representing the
		magnification factor from the default magnification.}
}
*/
LMI_INLINE_DECLARATION LmiFloat64 LmiCameraControlCapabilitiesGetMaxZoomValue(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetPanSpeedRange}
	{parent: LmiCameraControlCapabilities}
	{description: Set the range of speed settings for panning.}
	{prototype: void LmiCameraControlCapabilitiesSetPanSpeedRange(LmiCameraControlCapabilities *caps, const LmiRange(LmiInt) *range)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: range} {description: The range of speed settings.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetPanSpeedRange(LmiCameraControlCapabilities *caps, const LmiRange(LmiInt) *range);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetPanSpeedRangeValues}
	{parent: LmiCameraControlCapabilities}
	{description: Set the range of speed settings for panning.}
	{prototype: void LmiCameraControlCapabilitiesSetPanSpeedRangeValues(LmiCameraControlCapabilities *caps, LmiInt minValue, LmiInt maxValue)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: minValue} {description: The lowest speed setting. This will typically be 1.}}
	{parameter: {name: maxValue} {description: The highest speed setting.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetPanSpeedRangeValues(LmiCameraControlCapabilities *caps, LmiInt minValue, LmiInt maxValue);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetTiltSpeedRange}
	{parent: LmiCameraControlCapabilities}
	{description: Set the range of speed settings for tilting.}
	{prototype: void LmiCameraControlCapabilitiesSetTiltSpeedRange(LmiCameraControlCapabilities *caps, const LmiRange(LmiInt) *range)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: range} {description: The range of speed settings.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetTiltSpeedRange(LmiCameraControlCapabilities *caps, const LmiRange(LmiInt) *range);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetTiltSpeedRangeValues}
	{parent: LmiCameraControlCapabilities}
	{description: Set the range of speed settings for tilting.}
	{prototype: void LmiCameraControlCapabilitiesSetTiltSpeedRangeValues(LmiCameraControlCapabilities *caps, LmiInt minValue, LmiInt maxValue)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: minValue} {description: The lowest speed setting. This will typically be 1.}}
	{parameter: {name: maxValue} {description: The highest speed setting.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetTiltSpeedRangeValues(LmiCameraControlCapabilities *caps, LmiInt minValue, LmiInt maxValue);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetZoomSpeedRange}
	{parent: LmiCameraControlCapabilities}
	{description: Set the range of speed settings for zooming.}
	{prototype: void LmiCameraControlCapabilitiesSetZoomSpeedRange(LmiCameraControlCapabilities *caps, const LmiRange(LmiInt) *range)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: range} {description: The range of speed settings.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetZoomSpeedRange(LmiCameraControlCapabilities *caps, const LmiRange(LmiInt) *range);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetZoomSpeedRangeValues}
	{parent: LmiCameraControlCapabilities}
	{description: Set the range of speed settings for zooming.}
	{prototype: void LmiCameraControlCapabilitiesSetZoomSpeedRangeValues(LmiCameraControlCapabilities *caps, LmiInt minValue, LmiInt maxValue)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: minValue} {description: The lowest speed setting. This will typically be 1.}}
	{parameter: {name: maxValue} {description: The highest speed setting.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetZoomSpeedRangeValues(LmiCameraControlCapabilities *caps, LmiInt minValue, LmiInt maxValue);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetPanSpeedRange}
	{parent: LmiCameraControlCapabilities}
	{description: Get the range of speed settings for panning.}
	{prototype: const LmiRange(LmiInt) *LmiCameraControlCapabilitiesGetPanSpeedRange(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The range of speed settings. These values represent the range of possible magnitudes when setting a nonzero speed.
		The set speed may be positive to pan right, negative to pan left, or zero to stop panning.}
	{note: No guarantees are made about the actual speed of one setting versus another. For example, the speeds may not necessarily
		be spaced at fixed increments.}
}
*/
LMI_INLINE_DECLARATION const LmiRange(LmiInt) *LmiCameraControlCapabilitiesGetPanSpeedRange(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetMinPanSpeed}
	{parent: LmiCameraControlCapabilities}
	{description: Get the lowest speed setting for panning.}
	{prototype: LmiInt LmiCameraControlCapabilitiesGetMinPanSpeed(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The lowest speed setting. This is typically 1.}
}
*/
LMI_INLINE_DECLARATION LmiInt LmiCameraControlCapabilitiesGetMinPanSpeed(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetMaxPanSpeed}
	{parent: LmiCameraControlCapabilities}
	{description: Get the highest speed setting for panning.}
	{prototype: LmiInt LmiCameraControlCapabilitiesGetMaxPanSpeed(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The highest speed setting. This value represents the largest possible magnitude when setting a nonzero speed.
		The set speed may be positive to pan right, negative to pan left, or zero to stop panning.}
}
*/
LMI_INLINE_DECLARATION LmiInt LmiCameraControlCapabilitiesGetMaxPanSpeed(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetTiltSpeedRange}
	{parent: LmiCameraControlCapabilities}
	{description: Get the range of speed settings for tilting.}
	{prototype: const LmiRange(LmiInt) *LmiCameraControlCapabilitiesGetTiltSpeedRange(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The range of speed settings. These values represent the range of possible magnitudes when setting a nonzero speed.
		The set speed may be positive to tilt up, negative to tilt down, or zero to stop tilting.}
	{note: No guarantees are made about the actual speed of one setting versus another. For example, the speeds may not necessarily
		be spaced at fixed increments.}
}
*/
LMI_INLINE_DECLARATION const LmiRange(LmiInt) *LmiCameraControlCapabilitiesGetTiltSpeedRange(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetMinTiltSpeed}
	{parent: LmiCameraControlCapabilities}
	{description: Get the lowest speed setting for tilting.}
	{prototype: LmiInt LmiCameraControlCapabilitiesGetMinTiltSpeed(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The lowest speed setting. This is typically 1.}
}
*/
LMI_INLINE_DECLARATION LmiInt LmiCameraControlCapabilitiesGetMinTiltSpeed(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetMaxTiltSpeed}
	{parent: LmiCameraControlCapabilities}
	{description: Get the highest speed setting for tilting.}
	{prototype: LmiInt LmiCameraControlCapabilitiesGetMaxTiltSpeed(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The highest speed setting. This value represents the largest possible magnitude when setting a nonzero speed.
		The set speed may be positive to tilt up, negative to tilt down, or zero to stop tilting.}
}
*/
LMI_INLINE_DECLARATION LmiInt LmiCameraControlCapabilitiesGetMaxTiltSpeed(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetZoomSpeedRange}
	{parent: LmiCameraControlCapabilities}
	{description: Get the range of speed settings for zooming.}
	{prototype: const LmiRange(LmiInt) *LmiCameraControlCapabilitiesGetZoomSpeedRange(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The range of speed settings. These values represent the range of possible magnitudes when setting a nonzero speed.
		The set speed may be positive to zoom in, negative to zoom out, or zero to stop zooming.}
	{note: No guarantees are made about the actual speed of one setting versus another. For example, the speeds may not necessarily
		be spaced at fixed increments.}
}
*/
LMI_INLINE_DECLARATION const LmiRange(LmiInt) *LmiCameraControlCapabilitiesGetZoomSpeedRange(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetMinZoomSpeed}
	{parent: LmiCameraControlCapabilities}
	{description: Get the lowest speed setting for zooming.}
	{prototype: LmiInt LmiCameraControlCapabilitiesGetMinZoomSpeed(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The lowest speed setting. This is typically 1.}
}
*/
LMI_INLINE_DECLARATION LmiInt LmiCameraControlCapabilitiesGetMinZoomSpeed(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesGetMaxZoomSpeed}
	{parent: LmiCameraControlCapabilities}
	{description: Get the highest speed setting for zooming.}
	{prototype: LmiInt LmiCameraControlCapabilitiesGetMaxZoomSpeed(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: The highest speed setting. This value represents the largest possible magnitude when setting a nonzero speed.
		The set speed may be positive to zoom in, negative to zoom out, or zero to stop zooming.}
}
*/
LMI_INLINE_DECLARATION LmiInt LmiCameraControlCapabilitiesGetMaxZoomSpeed(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesHasOldEndpoint}
	{parent: LmiCameraControlCapabilities}
	{description: Check whether old endpoint is supported.}
	{prototype: LmiBool LmiCameraControlCapabilitiesHasOldEndpoint(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: A boolean indicating if old endpoint is supported.}
}
*/
LMI_INLINE_DECLARATION LmiBool LmiCameraControlCapabilitiesHasOldEndpoint(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetOldEndpoint}
	{parent: LmiCameraControlCapabilities}
	{description: Set whether old endpoint is supported.}
	{prototype: void LmiCameraControlCapabilitiesSetOldEndpoint(LmiCameraControlCapabilities *caps, LmiBool supported)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: supported} {description: Whether old endpoint is supported.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetOldEndpoint(LmiCameraControlCapabilities *caps, LmiBool supported);

/**
{function:
	{name: LmiCameraControlCapabilitiesHasViscaSupport}
	{parent: LmiCameraControlCapabilities}
	{description: Check whether Visca protocol is supported.}
	{prototype: LmiBool LmiCameraControlCapabilitiesHasViscaSupport(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: A boolean indicating if Visca protocol is supported.}
}
*/
LMI_INLINE_DECLARATION LmiBool LmiCameraControlCapabilitiesHasViscaSupport(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetViscaSupport}
	{parent: LmiCameraControlCapabilities}
	{description: Set whether Visca protocol is supported.}
	{prototype: void LmiCameraControlCapabilitiesSetViscaSupport(LmiCameraControlCapabilities *caps, LmiBool supported)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: supported} {description: Whether Visca protocol is supported.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetViscaSupport(LmiCameraControlCapabilities *caps, LmiBool supported);

/**
{function:
	{name: LmiCameraControlCapabilitiesHasPresetSupport}
	{parent: LmiCameraControlCapabilities}
	{description: Check whether camera preset is supported.}
	{prototype: LmiBool LmiCameraControlCapabilitiesHasPresetSupport(const LmiCameraControlCapabilities *caps)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{return: A Boolean value that indicates whether the preset is supported.}
}
*/
LMI_INLINE_DECLARATION LmiBool LmiCameraControlCapabilitiesHasPresetSupport(const LmiCameraControlCapabilities *caps);

/**
{function:
	{name: LmiCameraControlCapabilitiesSetPresetSupport}
	{parent: LmiCameraControlCapabilities}
	{description: Set whether preset is supported.}
	{prototype: void LmiCameraControlCapabilitiesSetPresetSupport(LmiCameraControlCapabilities *caps, LmiBool supported)}
	{parameter: {name: caps} {description: The camera capabilities object.}}
	{parameter: {name: supported} {description: Whether preset is supported.}}
}
*/
LMI_INLINE_DECLARATION void LmiCameraControlCapabilitiesSetPresetSupport(LmiCameraControlCapabilities *caps, LmiBool supported);

/**
{type:
	{name: LmiCameraCommandType}
	{parent: LmiCameraCommand}
	{include: Lmi/Video/Common/LmiCameraControl.h}
	{description: Indicates the type of a command contained in an LmiCameraCommand object.}
	{value: {name: LMI_CAMERACOMMANDTYPE_Nudge}   {description: The command contained in the LmiCameraCommand object is of type LmiCameraNudgeCommand.}}
	{value: {name: LMI_CAMERACOMMANDTYPE_Set}     {description: The command contained in the LmiCameraCommand object is of type LmiCameraSetCommand.}}
	{value: {name: LMI_CAMERACOMMANDTYPE_Move}    {description: The command contained in the LmiCameraCommand object is of type LmiCameraMoveCommand.}}
	{value: {name: LMI_CAMERACOMMANDTYPE_Magnify} {description: The command contained in the LmiCameraCommand object is of type LmiCameraMagnifyCommand.}}
	{value: {name: LMI_CAMERACOMMANDTYPE_RelativePanTilt} {description: The command contained in the LmiCameraCommand object is of type LmiCameraRelativePanTiltCommand.}}
	{value: {name: LMI_CAMERACOMMANDTYPE_RelativeZoom}    {description: The command contained in the LmiCameraCommand object is of type LmiCameraRelativeZoomCommand.}}
	{value: {name: LMI_CAMERACOMMANDTYPE_ActivatePreset}  {description: The command contained in the LmiCameraCommand object is of type LmiCameraActivatePresetCommand.}}
	{value: {name: LMI_CAMERACOMMANDTYPE_ViscaControl}  {description: The command contained in the LmiCameraCommand object is of type LmiCameraViscaControlCommand.}}
}
*/
typedef enum
{
	LMI_CAMERACOMMANDTYPE_Nudge,
	LMI_CAMERACOMMANDTYPE_Set,
	LMI_CAMERACOMMANDTYPE_Move,
	LMI_CAMERACOMMANDTYPE_Magnify,
	LMI_CAMERACOMMANDTYPE_RelativePanTilt,
	LMI_CAMERACOMMANDTYPE_RelativeZoom,
	LMI_CAMERACOMMANDTYPE_ActivatePreset,
	LMI_CAMERACOMMANDTYPE_ViscaControl,
} LmiCameraCommandType;

/**
{type:
	{name: LmiCameraNudgeCommand}
	{parent: CameraControl}
	{include: Lmi/Video/Common/LmiCameraControl.h}
	{description: A command used to change a parameter by a small fixed amount.}
	{note: Nudge commands should only be directed at devices that support moves to a specified position of the parameter (absolute mode).
		Check LmiCameraControlCapabilitiesHasAbsolutePanTilt for pan-tilt and LmiCameraControlCapabilitiesHasAbsoluteZoom for zoom.}
}
*/
typedef struct
{
	LmiString parameter;
	LmiInt nudges;
} LmiCameraNudgeCommand;

/**
{function:
	{name: LmiCameraNudgeCommandConstruct}
	{parent: LmiCameraNudgeCommand}
	{description: Constructs an LmiCameraNudgeCommand object.}
	{prototype: LmiCameraNudgeCommand *LmiCameraNudgeCommandConstruct(LmiCameraNudgeCommand *c, const LmiString *parameter, LmiInt nudges, LmiAllocator *alloc)}
	{parameter: {name: c} {description: The LmiCameraNudgeCommand object to construct.}}
	{parameter: {name: parameter} {description: The parameter to nudge.}}
	{parameter: {name: nudges} {description: The number of nudges to change the parameter by. Positive numbers increase the parameter, and negative numbers
		decrease the parameter.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraNudgeCommand *LmiCameraNudgeCommandConstruct(LmiCameraNudgeCommand *c, const LmiString *parameter, LmiInt nudges, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraNudgeCommandConstructCStr}
	{parent: LmiCameraNudgeCommand}
	{description: Constructs an LmiCameraNudgeCommand object.}
	{prototype: LmiCameraNudgeCommand *LmiCameraNudgeCommandConstructCStr(LmiCameraNudgeCommand *c, const char *parameter, LmiInt nudges, LmiAllocator *alloc)}
	{parameter: {name: c} {description: The LmiCameraNudgeCommand object to construct.}}
	{parameter: {name: parameter} {description: The parameter to nudge.}}
	{parameter: {name: nudges} {description: The number of nudges to change the parameter by. Positive numbers increase the parameter, and negative numbers
		decrease the parameter.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraNudgeCommand *LmiCameraNudgeCommandConstructCStr(LmiCameraNudgeCommand *c, const char *parameter, LmiInt nudges, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraNudgeCommandConstructCopy}
	{parent: LmiCameraNudgeCommand}
	{description: Constructs an LmiCameraNudgeCommand object as a copy of another.}
	{prototype: LmiCameraNudgeCommand *LmiCameraNudgeCommandConstructCopy(LmiCameraNudgeCommand *c, const LmiCameraNudgeCommand *other)}
	{parameter: {name: c} {description: The LmiCameraNudgeCommand object to construct.}}
	{parameter: {name: other} {description: The LmiCameraNudgeCommand object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraNudgeCommand *LmiCameraNudgeCommandConstructCopy(LmiCameraNudgeCommand *c, const LmiCameraNudgeCommand *other);

/**
{function:
	{name: LmiCameraNudgeCommandDestruct}
	{parent: LmiCameraNudgeCommand}
	{description: Destructs an LmiCameraNudgeCommand object.}
	{prototype: void LmiCameraNudgeCommandDestruct(LmiCameraNudgeCommand *c)}
	{parameter: {name: c} {description: The LmiCameraNudgeCommand object to destruct.}}
}
*/
void LmiCameraNudgeCommandDestruct(LmiCameraNudgeCommand *c);

/**
{function:
	{name: LmiCameraNudgeCommandAssign}
	{parent: LmiCameraNudgeCommand}
	{description: Assigns one LmiCameraNudgeCommand object the value of another.}
	{prototype: LmiCameraNudgeCommand *LmiCameraNudgeCommandAssign(LmiCameraNudgeCommand *c, const LmiCameraNudgeCommand *other)}
	{parameter: {name: c} {description: The LmiCameraNudgeCommand object to assign to.}}
	{parameter: {name: other} {description: The LmiCameraNudgeCommand object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/
LmiCameraNudgeCommand *LmiCameraNudgeCommandAssign(LmiCameraNudgeCommand *c, const LmiCameraNudgeCommand *other);

/**
{function:
	{name: LmiCameraNudgeCommandGetParameter}
	{parent: LmiCameraNudgeCommand}
	{description: Gets the name of the parameter being modified by a Nudge command.}
	{prototype: const LmiString *LmiCameraNudgeCommandGetParameter(const LmiCameraNudgeCommand *c)}
	{parameter: {name: c} {description: The LmiCameraNudgeCommand object.}}
	{return: The name of the parameter.}
}
*/
LMI_INLINE_DECLARATION const LmiString *LmiCameraNudgeCommandGetParameter(const LmiCameraNudgeCommand *c);

/**
{function:
	{name: LmiCameraNudgeCommandGetNudges}
	{parent: LmiCameraNudgeCommand}
	{description: Gets the number of nudges that a parameter should be changed by a Nudge command.}
	{prototype: LmiInt LmiCameraNudgeCommandGetNudges(const LmiCameraNudgeCommand *c)}
	{parameter: {name: c} {description: The LmiCameraNudgeCommand object.}}
	{return: The number of nudges to change the parameter by. Positive numbers increase the parameter, and negative numbers
		decrease the parameter.}
}
*/
LMI_INLINE_DECLARATION LmiInt LmiCameraNudgeCommandGetNudges(const LmiCameraNudgeCommand *c);

/**
{type:
	{name: LmiCameraSetCommand}
	{parent: CameraControl}
	{include: Lmi/Video/Common/LmiCameraControl.h}
	{description: A command used to set a parameter to a specific value, without regard to its current value.}
	{note: Set commands should only be directed at devices that support moves to a specified position of the parameter (absolute mode).
		Check LmiCameraControlCapabilitiesHasAbsolutePanTilt for pan-tilt and LmiCameraControlCapabilitiesHasAbsoluteZoom for zoom.}
}
*/
typedef struct
{
	LmiString parameter;
	LmiFloat64 value;
} LmiCameraSetCommand;

/**
{function:
	{name: LmiCameraSetCommandConstruct}
	{parent: LmiCameraSetCommand}
	{description: Constructs an LmiCameraSetCommand object.}
	{prototype: LmiCameraSetCommand *LmiCameraSetCommandConstruct(LmiCameraSetCommand *c, const LmiString *parameter, LmiFloat64 value, LmiAllocator *alloc)}
	{parameter: {name: c} {description: The LmiCameraSetCommand object to construct.}}
	{parameter: {name: parameter} {description: The parameter to set.}}
	{parameter: {name: value} {description: The value to set the parameter to.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraSetCommand *LmiCameraSetCommandConstruct(LmiCameraSetCommand *c, const LmiString *parameter, LmiFloat64 value, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraSetCommandConstructCStr}
	{parent: LmiCameraSetCommand}
	{description: Constructs an LmiCameraSetCommand object.}
	{prototype: LmiCameraSetCommand *LmiCameraSetCommandConstructCStr(LmiCameraSetCommand *c, const char *parameter, LmiFloat64 value, LmiAllocator *alloc)}
	{parameter: {name: c} {description: The LmiCameraSetCommand object to construct.}}
	{parameter: {name: parameter} {description: The parameter to set.}}
	{parameter: {name: value} {description: The value to set the parameter to.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraSetCommand *LmiCameraSetCommandConstructCStr(LmiCameraSetCommand *c, const char *parameter, LmiFloat64 value, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraSetCommandConstructCopy}
	{parent: LmiCameraSetCommand}
	{description: Constructs an LmiCameraSetCommand object as a copy of another.}
	{prototype: LmiCameraSetCommand *LmiCameraSetCommandConstructCopy(LmiCameraSetCommand *c, const LmiCameraSetCommand *other)}
	{parameter: {name: c} {description: The LmiCameraSetCommand object to construct.}}
	{parameter: {name: other} {description: The LmiCameraSetCommand object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraSetCommand *LmiCameraSetCommandConstructCopy(LmiCameraSetCommand *c, const LmiCameraSetCommand *other);

/**
{function:
	{name: LmiCameraSetCommandDestruct}
	{parent: LmiCameraSetCommand}
	{description: Destructs an LmiCameraSetCommand object.}
	{prototype: void LmiCameraSetCommandDestruct(LmiCameraSetCommand *c)}
	{parameter: {name: c} {description: The LmiCameraSetCommand object to destruct.}}
}
*/
void LmiCameraSetCommandDestruct(LmiCameraSetCommand *c);

/**
{function:
	{name: LmiCameraSetCommandAssign}
	{parent: LmiCameraSetCommand}
	{description: Assigns one LmiCameraSetCommand object the value of another.}
	{prototype: LmiCameraSetCommand *LmiCameraSetCommandAssign(LmiCameraSetCommand *c, const LmiCameraSetCommand *other)}
	{parameter: {name: c} {description: The LmiCameraSetCommand object to assign to.}}
	{parameter: {name: other} {description: The LmiCameraSetCommand object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/
LmiCameraSetCommand *LmiCameraSetCommandAssign(LmiCameraSetCommand *c, const LmiCameraSetCommand *other);

/**
{function:
	{name: LmiCameraSetCommandGetParameter}
	{parent: LmiCameraSetCommand}
	{description: Gets the name of the parameter being modified by a Set command.}
	{prototype: const LmiString *LmiCameraSetCommandGetParameter(const LmiCameraSetCommand *c)}
	{parameter: {name: c} {description: The LmiCameraSetCommand object.}}
	{return: The name of the parameter.}
}
*/
LMI_INLINE_DECLARATION const LmiString *LmiCameraSetCommandGetParameter(const LmiCameraSetCommand *c);

/**
{function:
	{name: LmiCameraSetCommandGetValue}
	{parent: LmiCameraSetCommand}
	{description: Gets the value that a parameter should be set to by a Set command.}
	{prototype: LmiFloat64 LmiCameraSetCommandGetValue(const LmiCameraSetCommand *c)}
	{parameter: {name: c} {description: The LmiCameraSetCommand object.}}
	{return: The value the parameter should be set to.}
}
*/
LMI_INLINE_DECLARATION LmiFloat64 LmiCameraSetCommandGetValue(const LmiCameraSetCommand *c);

/**
{type:
	{name: LmiCameraMoveCommand}
	{parent: CameraControl}
	{include: Lmi/Video/Common/LmiCameraControl.h}
	{description: A command used to pan and tilt to a specified position within the currently displayed image.}
	{note: Move commands should only be directed at devices that support absolute pan-tilt. Check LmiCameraControlCapabilitiesHasAbsolutePanTilt.}
}
*/
typedef struct
{
	LmiFloat64 pan, tilt;
	LmiTime timeout;
} LmiCameraMoveCommand;

/**
{function:
	{name: LmiCameraMoveCommandConstruct}
	{parent: LmiCameraMoveCommand}
	{description: Constructs an LmiCameraMoveCommand object.}
	{prototype: LmiCameraMoveCommand *LmiCameraMoveCommandConstruct(LmiCameraMoveCommand *c, LmiFloat64 pan, LmiFloat64 tilt, LmiTime timeout, LmiAllocator *alloc)}
	{parameter: {name: c} {description: The LmiCameraMoveCommand object to construct.}}
	{parameter: {name: pan} {description: The amount to pan the camera. The magnitude of this value represents the fraction of the current frame width to pan.
		Positive values request a pan to the right, and negative values request a pan to the left. A value of 0. maintains the current pan setting.}}
	{parameter: {name: tilt} {description: The amount to tilt the camera. The magnitude of this value represents the fraction of the current frame height to tilt.
		Positive values request a tilt up, and negative values request a tilt down. A value of 0. maintains the current tilt setting.}}
	{parameter: {name: timeout} {description: The amount of time in milliseconds during which the camera will move in certain directions.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraMoveCommand *LmiCameraMoveCommandConstruct(LmiCameraMoveCommand *c, LmiFloat64 pan, LmiFloat64 tilt, LmiTime timeout, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraMoveCommandConstructCopy}
	{parent: LmiCameraMoveCommand}
	{description: Constructs an LmiCameraMoveCommand object as a copy of another.}
	{prototype: LmiCameraMoveCommand *LmiCameraMoveCommandConstructCopy(LmiCameraMoveCommand *c, const LmiCameraMoveCommand *other)}
	{parameter: {name: c} {description: The LmiCameraMoveCommand object to construct.}}
	{parameter: {name: other} {description: The LmiCameraMoveCommand object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraMoveCommand *LmiCameraMoveCommandConstructCopy(LmiCameraMoveCommand *c, const LmiCameraMoveCommand *other);

/**
{function:
	{name: LmiCameraMoveCommandDestruct}
	{parent: LmiCameraMoveCommand}
	{description: Destructs an LmiCameraMoveCommand object.}
	{prototype: void LmiCameraMoveCommandDestruct(LmiCameraMoveCommand *c)}
	{parameter: {name: c} {description: The LmiCameraMoveCommand object to destruct.}}
}
*/
void LmiCameraMoveCommandDestruct(LmiCameraMoveCommand *c);

/**
{function:
	{name: LmiCameraMoveCommandAssign}
	{parent: LmiCameraMoveCommand}
	{description: Assigns one LmiCameraMoveCommand object the value of another.}
	{prototype: LmiCameraMoveCommand *LmiCameraMoveCommandAssign(LmiCameraMoveCommand *c, const LmiCameraMoveCommand *other)}
	{parameter: {name: c} {description: The LmiCameraMoveCommand object to assign to.}}
	{parameter: {name: other} {description: The LmiCameraMoveCommand object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/
LmiCameraMoveCommand *LmiCameraMoveCommandAssign(LmiCameraMoveCommand *c, const LmiCameraMoveCommand *other);

/**
{function:
	{name: LmiCameraMoveCommandGetPan}
	{parent: LmiCameraMoveCommand}
	{description: Gets how much to pan the camera during a move operation.}
	{prototype: LmiFloat64 LmiCameraMoveCommandGetPan(const LmiCameraMoveCommand *c)}
	{parameter: {name: c} {description: The LmiCameraMoveCommand object.}}
	{return: The amount to pan the camera. The magnitude of this value represents the fraction of the current frame width to pan.
		Positive values request a pan to the right, and negative values request a pan to the left. A value of 0. maintains the current pan setting.}
}
*/
LMI_INLINE_DECLARATION LmiFloat64 LmiCameraMoveCommandGetPan(const LmiCameraMoveCommand *c);

/**
{function:
	{name: LmiCameraMoveCommandGetTilt}
	{parent: LmiCameraMoveCommand}
	{description: Gets how much to tilt the camera during a move operation.}
	{prototype: LmiFloat64 LmiCameraMoveCommandGetTilt(const LmiCameraMoveCommand *c)}
	{parameter: {name: c} {description: The LmiCameraMoveCommand object.}}
	{return: The magnitude of this value represents the fraction of the current frame height to tilt.
		Positive values request a tilt up, and negative values request a tilt down. A value of 0. maintains the current tilt setting.}
}
*/
LMI_INLINE_DECLARATION LmiFloat64 LmiCameraMoveCommandGetTilt(const LmiCameraMoveCommand *c);

/**
{function:
	{name: LmiCameraMoveCommandGetTimeout}
	{parent: LmiCameraMoveCommand}
	{description: Gets the amount of time in milliseconds at which camera will be moving in the specified directions.}
	{prototype: LmiTime LmiCameraMoveCommandGetTimeout(const LmiCameraMoveCommand *c)}
	{parameter: {name: c} {description: The LmiCameraMoveCommand object.}}
	{return: The amount of time in milliseconds during which the camera will move in certain directions.}
}
*/
LMI_INLINE_DECLARATION LmiTime LmiCameraMoveCommandGetTimeout(const LmiCameraMoveCommand *c);

/**
{type:
	{name: LmiCameraMagnifyCommand}
	{parent: CameraControl}
	{include: Lmi/Video/Common/LmiCameraControl.h}
	{description: A command used to zoom the camera such that the currently displayed image
		is magnified by a specified scale factor.}
	{note: Magnify commands should only be directed at devices that support absolute zoom. Check LmiCameraControlCapabilitiesHasAbsoluteZoom.}
}
*/
typedef struct
{
	LmiFloat64 zoom;
	LmiTime timeout;
} LmiCameraMagnifyCommand;

/**
{function:
	{name: LmiCameraMagnifyCommandConstruct}
	{parent: LmiCameraMagnifyCommand}
	{description: Constructs an LmiCameraMagnifyCommand object.}
	{prototype: LmiCameraMagnifyCommand *LmiCameraMagnifyCommandConstruct(LmiCameraMagnifyCommand *c, LmiFloat64 zoom, LmiTime timeout, LmiAllocator *alloc)}
	{parameter: {name: c} {description: The LmiCameraMagnifyCommand object to construct.}}
	{parameter: {name: zoom} {description: The amount to zoom the camera. This value represents how much to scale the current magnification of the camera.
		A value of 1. maintains the current zoom setting.}}
	{parameter: {name: timeout} {description: The amount of time in milliseconds during which the camera will move in certain directions.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraMagnifyCommand *LmiCameraMagnifyCommandConstruct(LmiCameraMagnifyCommand *c, LmiFloat64 zoom, LmiTime timeout, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraMagnifyCommandConstructCopy}
	{parent: LmiCameraMagnifyCommand}
	{description: Constructs an LmiCameraMagnifyCommand object as a copy of another.}
	{prototype: LmiCameraMagnifyCommand *LmiCameraMagnifyCommandConstructCopy(LmiCameraMagnifyCommand *c, const LmiCameraMagnifyCommand *other)}
	{parameter: {name: c} {description: The LmiCameraMagnifyCommand object to construct.}}
	{parameter: {name: other} {description: The LmiCameraMagnifyCommand object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraMagnifyCommand *LmiCameraMagnifyCommandConstructCopy(LmiCameraMagnifyCommand *c, const LmiCameraMagnifyCommand *other);

/**
{function:
	{name: LmiCameraMagnifyCommandDestruct}
	{parent: LmiCameraMagnifyCommand}
	{description: Destructs an LmiCameraMagnifyCommand object.}
	{prototype: void LmiCameraMagnifyCommandDestruct(LmiCameraMagnifyCommand *c)}
	{parameter: {name: c} {description: The LmiCameraMagnifyCommand object to destruct.}}
}
*/
void LmiCameraMagnifyCommandDestruct(LmiCameraMagnifyCommand *c);

/**
{function:
	{name: LmiCameraMagnifyCommandAssign}
	{parent: LmiCameraMagnifyCommand}
	{description: Assigns one LmiCameraMagnifyCommand object the value of another.}
	{prototype: LmiCameraMagnifyCommand *LmiCameraMagnifyCommandAssign(LmiCameraMagnifyCommand *c, const LmiCameraMagnifyCommand *other)}
	{parameter: {name: c} {description: The LmiCameraMagnifyCommand object to assign to.}}
	{parameter: {name: other} {description: The LmiCameraMagnifyCommand object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/
LmiCameraMagnifyCommand *LmiCameraMagnifyCommandAssign(LmiCameraMagnifyCommand *c, const LmiCameraMagnifyCommand *other);

/**
{function:
	{name: LmiCameraMagnifyCommandGetZoom}
	{parent: LmiCameraMagnifyCommand}
	{description: Gets how much to zoom the camera during a magnify operation.}
	{prototype: LmiFloat64 LmiCameraMagnifyCommandGetZoom(const LmiCameraMagnifyCommand *c)}
	{parameter: {name: c} {description: The LmiCameraMagnifyCommand object.}}
	{return: This value represents how much to scale the current magnification of the camera.
		A value of 1. maintains the current zoom setting.}
}
*/
LMI_INLINE_DECLARATION LmiFloat64 LmiCameraMagnifyCommandGetZoom(const LmiCameraMagnifyCommand *c);

/**
{function:
    {name: LmiCameraMagnifyCommandGetTimeout}
    {parent: LmiCameraMagnifyCommand}
    {description: Gets the amount of time in milliseconds at which camera will be zoom in/out.}
    {prototype: LmiTime LmiCameraMagnifyCommandGetTimeout(const LmiCameraMagnifyCommand *c)}
    {parameter: {name: c} {description: The LmiCameraMagnifyCommand object.}}
    {return: The amount of time in milliseconds during which the camera will zoom in/out.}
}
*/
LMI_INLINE_DECLARATION LmiTime LmiCameraMagnifyCommandGetTimeout(const LmiCameraMagnifyCommand* c);

/**
{type:
	{name: LmiCameraRelativePanTiltCommand}
	{parent: CameraControl}
	{include: Lmi/Video/Common/LmiCameraControl.h}
	{description: A command used to pan and tilt the camera at specified rates.}
	{note: Relative-pan-tilt commands should only be directed at devices that support them. Check LmiCameraControlCapabilitiesHasRelativePanTilt.}
}
*/
typedef struct
{
	LmiInt pan, tilt;
	LmiTime timeout;
} LmiCameraRelativePanTiltCommand;

/**
{function:
	{name: LmiCameraRelativePanTiltCommandConstruct}
	{parent: LmiCameraRelativePanTiltCommand}
	{description: Constructs an LmiCameraRelativePanTiltCommand object.}
	{prototype: LmiCameraRelativePanTiltCommand *LmiCameraRelativePanTiltCommandConstruct(LmiCameraRelativePanTiltCommand *c, LmiInt pan, LmiInt tilt, LmiTime timeout, LmiAllocator *alloc)}
	{parameter: {name: c} {description: The LmiCameraRelativePanTiltCommand object to construct.}}
	{parameter: {name: pan} {description: The speed setting at which the camera should be panned. Positive values pan the camera to the right, negative values pan the camera to the left
		and a value of zero stops panning. The magnitude of the value should be no greater than the maximum pan speed value advertised by the camera.}}
	{parameter: {name: tilt} {description: The speed setting at which the camera should be tilted. Positive values tilt the camera up, negative values tilt the camera down
		and a value of zero stops tilting. The magnitude of the value should be no greater than the maximum tilt speed value advertised by the camera.}}
	{parameter: {name: timeout} {description: The amount of time in milliseconds during which the camera will move in certain directions.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraRelativePanTiltCommand *LmiCameraRelativePanTiltCommandConstruct(LmiCameraRelativePanTiltCommand *c, LmiInt pan, LmiInt tilt, LmiTime timeout, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraRelativePanTiltCommandConstructCopy}
	{parent: LmiCameraRelativePanTiltCommand}
	{description: Constructs an LmiCameraRelativePanTiltCommand object as a copy of another.}
	{prototype: LmiCameraRelativePanTiltCommand *LmiCameraRelativePanTiltCommandConstructCopy(LmiCameraRelativePanTiltCommand *c, const LmiCameraRelativePanTiltCommand *other)}
	{parameter: {name: c} {description: The LmiCameraRelativePanTiltCommand object to construct.}}
	{parameter: {name: other} {description: The LmiCameraRelativePanTiltCommand object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraRelativePanTiltCommand *LmiCameraRelativePanTiltCommandConstructCopy(LmiCameraRelativePanTiltCommand *c, const LmiCameraRelativePanTiltCommand *other);

/**
{function:
	{name: LmiCameraRelativePanTiltCommandDestruct}
	{parent: LmiCameraRelativePanTiltCommand}
	{description: Destructs an LmiCameraRelativePanTiltCommand object.}
	{prototype: void LmiCameraRelativePanTiltCommandDestruct(LmiCameraRelativePanTiltCommand *c)}
	{parameter: {name: c} {description: The LmiCameraRelativePanTiltCommand object to destruct.}}
}
*/
void LmiCameraRelativePanTiltCommandDestruct(LmiCameraRelativePanTiltCommand *c);

/**
{function:
	{name: LmiCameraRelativePanTiltCommandAssign}
	{parent: LmiCameraRelativePanTiltCommand}
	{description: Assigns one LmiCameraRelativePanTiltCommand object the value of another.}
	{prototype: LmiCameraRelativePanTiltCommand *LmiCameraRelativePanTiltCommandAssign(LmiCameraRelativePanTiltCommand *c, const LmiCameraRelativePanTiltCommand *other)}
	{parameter: {name: c} {description: The LmiCameraRelativePanTiltCommand object to assign to.}}
	{parameter: {name: other} {description: The LmiCameraRelativePanTiltCommand object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/
LmiCameraRelativePanTiltCommand *LmiCameraRelativePanTiltCommandAssign(LmiCameraRelativePanTiltCommand *c, const LmiCameraRelativePanTiltCommand *other);

/**
{function:
	{name: LmiCameraRelativePanTiltCommandGetPan}
	{parent: LmiCameraRelativePanTiltCommand}
	{description: Gets the rate at which to pan the camera.}
	{prototype: LmiInt LmiCameraRelativePanTiltCommandGetPan(const LmiCameraRelativePanTiltCommand *c)}
	{parameter: {name: c} {description: The LmiCameraRelativePanTiltCommand object.}}
	{return: The speed setting at which the camera should be panned. Positive values pan the camera to the right, negative values pan the camera to the left
		and a value of zero stops panning. The magnitude of the value should be no greater than the maximum pan speed value advertised by the camera.}
}
*/
LMI_INLINE_DECLARATION LmiInt LmiCameraRelativePanTiltCommandGetPan(const LmiCameraRelativePanTiltCommand *c);

/**
{function:
	{name: LmiCameraRelativePanTiltCommandGetTilt}
	{parent: LmiCameraRelativePanTiltCommand}
	{description: Gets the rate at which to tilt the camera.}
	{prototype: LmiInt LmiCameraRelativePanTiltCommandGetTilt(const LmiCameraRelativePanTiltCommand *c)}
	{parameter: {name: c} {description: The LmiCameraRelativePanTiltCommand object.}}
	{return: The speed setting at which the camera should be tilted. Positive values tilt the camera up, negative values tilt the camera down
		and a value of zero stops tilting. The magnitude of the value should be no greater than the maximum tilt speed value advertised by the camera.}
}
*/
LMI_INLINE_DECLARATION LmiInt LmiCameraRelativePanTiltCommandGetTilt(const LmiCameraRelativePanTiltCommand *c);

/**
{function:
    {name: LmiCameraRelativePanTiltCommandGetTimeout}
    {parent: LmiCameraRelativePanTiltCommand}
    {description: Gets the amount of time in milliseconds at which camera will be move.}
    {prototype: LmiTime LmiCameraRelativePanTiltCommandGetTimeout(const LmiCameraRelativePanTiltCommand *c)}
    {parameter: {name: c} {description: The LmiCameraRelativePanTiltCommand object.}}
    {return: The amount of time in milliseconds during which the camera will move.}
}
*/
LMI_INLINE_DECLARATION LmiTime LmiCameraRelativePanTiltCommandGetTimeout(const LmiCameraRelativePanTiltCommand* c);

/**
{type:
	{name: LmiCameraRelativeZoomCommand}
	{parent: CameraControl}
	{include: Lmi/Video/Common/LmiCameraControl.h}
	{description: A command used to zoom the camera at a specified rate.}
	{note: Relative-zoom commands should only be directed at devices that support them. Check LmiCameraControlCapabilitiesHasRelativeZoom.}
}
*/
typedef struct
{
	LmiInt zoom;
	LmiTime timeout;
} LmiCameraRelativeZoomCommand;

/**
{function:
	{name: LmiCameraRelativeZoomCommandConstruct}
	{parent: LmiCameraRelativeZoomCommand}
	{description: Constructs an LmiCameraRelativeZoomCommand object.}
	{prototype: LmiCameraRelativeZoomCommand *LmiCameraRelativeZoomCommandConstruct(LmiCameraRelativeZoomCommand *c, LmiInt zoom, LmiTime timeout, LmiAllocator *alloc)}
	{parameter: {name: c} {description: The LmiCameraRelativeZoomCommand object to construct.}}
	{parameter: {name: zoom} {description: The speed setting at which the camera should be zoomed. Positive values zoom the camera in, negative values zoom the camera out
		and a value of zero stops zooming. The magnitude of the value should be no greater than the maximum zoom speed value advertised by the camera.}}
	{parameter: {name: timeout} {description: The amount of time in milliseconds during which the camera will zoom in/out.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraRelativeZoomCommand *LmiCameraRelativeZoomCommandConstruct(LmiCameraRelativeZoomCommand *c, LmiInt zoom, LmiTime timeout, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraRelativeZoomCommandConstructCopy}
	{parent: LmiCameraRelativeZoomCommand}
	{description: Constructs an LmiCameraRelativeZoomCommand object as a copy of another.}
	{prototype: LmiCameraRelativeZoomCommand *LmiCameraRelativeZoomCommandConstructCopy(LmiCameraRelativeZoomCommand *c, const LmiCameraRelativeZoomCommand *other)}
	{parameter: {name: c} {description: The LmiCameraRelativeZoomCommand object to construct.}}
	{parameter: {name: other} {description: The LmiCameraRelativeZoomCommand object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraRelativeZoomCommand *LmiCameraRelativeZoomCommandConstructCopy(LmiCameraRelativeZoomCommand *c, const LmiCameraRelativeZoomCommand *other);

/**
{function:
	{name: LmiCameraRelativeZoomCommandDestruct}
	{parent: LmiCameraRelativeZoomCommand}
	{description: Destructs an LmiCameraRelativeZoomCommand object.}
	{prototype: void LmiCameraRelativeZoomCommandDestruct(LmiCameraRelativeZoomCommand *c)}
	{parameter: {name: c} {description: The LmiCameraRelativeZoomCommand object to destruct.}}
}
*/
void LmiCameraRelativeZoomCommandDestruct(LmiCameraRelativeZoomCommand *c);

/**
{function:
	{name: LmiCameraRelativeZoomCommandAssign}
	{parent: LmiCameraRelativeZoomCommand}
	{description: Assigns one LmiCameraRelativeZoomCommand object the value of another.}
	{prototype: LmiCameraRelativeZoomCommand *LmiCameraRelativeZoomCommandAssign(LmiCameraRelativeZoomCommand *c, const LmiCameraRelativeZoomCommand *other)}
	{parameter: {name: c} {description: The LmiCameraRelativeZoomCommand object to assign to.}}
	{parameter: {name: other} {description: The LmiCameraRelativeZoomCommand object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/
LmiCameraRelativeZoomCommand *LmiCameraRelativeZoomCommandAssign(LmiCameraRelativeZoomCommand *c, const LmiCameraRelativeZoomCommand *other);

/**
{function:
	{name: LmiCameraRelativeZoomCommandGetZoom}
	{parent: LmiCameraRelativeZoomCommand}
	{description: Gets the rate at which to zoom the camera.}
	{prototype: LmiInt LmiCameraRelativeZoomCommandGetZoom(const LmiCameraRelativeZoomCommand *c)}
	{parameter: {name: c} {description: The LmiCameraRelativeZoomCommand object.}}
	{return: The speed setting at which the camera should be zoomed. Positive values zoom the camera in, negative values zoom the camera out
		and a value of zero stops zooming. The magnitude of the value should be no greater than the maximum zoom speed value advertised by the camera.}
}
*/
LMI_INLINE_DECLARATION LmiInt LmiCameraRelativeZoomCommandGetZoom(const LmiCameraRelativeZoomCommand *c);

/**
{function:
    {name: LmiCameraRelativeZoomCommandGetTimeout}
    {parent: LmiCameraRelativeZoomCommand}
    {description: Gets the amount of time in milliseconds at which camera will be zoom in/out.}
    {prototype: LmiTime LmiCameraRelativeZoomCommandGetTimeout(const LmiCameraRelativeZoomCommand *c)}
    {parameter: {name: c} {description: The LmiCameraRelativeZoomCommand object.}}
    {return: The amount of time in milliseconds during which the camera will zoom in/out.}
}
*/
LMI_INLINE_DECLARATION LmiTime LmiCameraRelativeZoomCommandGetTimeout(const LmiCameraRelativeZoomCommand* c);

/**
{type:
	{name: LmiCameraActivatePresetCommand}
	{parent: CameraControl}
	{include: Lmi/Video/Common/LmiCameraControl.h}
	{description: A command used to activate preset.}
	{note: Activate preset commands should only be directed at devices that support them. Check VidyoCameraControlCapabilitiesHasPresetSupport.}
}
*/
typedef struct
{
	LmiUint index;
} LmiCameraActivatePresetCommand;

/**
{function:
	{name: LmiCameraActivatePresetCommandConstruct}
	{parent: LmiCameraActivatePresetCommand}
	{description: Constructs an LmiCameraActivatePresetCommand object.}
	{prototype: LmiCameraActivatePresetCommand *LmiCameraActivatePresetCommandConstruct(LmiCameraActivatePresetCommand *c, LmiUint index, LmiAllocator *alloc)}
	{parameter: {name: c} {description: The LmiCameraActivatePresetCommand object to construct.}}
	{parameter: {name: index} {description: Represents the preset index.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraActivatePresetCommand *LmiCameraActivatePresetCommandConstruct(LmiCameraActivatePresetCommand *c, LmiUint index, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraActivatePresetCommandConstructCopy}
	{parent: LmiCameraActivatePresetCommand}
	{description: Constructs an LmiCameraActivatePresetCommand object as a copy of another.}
	{prototype: LmiCameraActivatePresetCommand *LmiCameraActivatePresetCommandConstructCopy(LmiCameraActivatePresetCommand *c, const LmiCameraActivatePresetCommand *other)}
	{parameter: {name: c} {description: The LmiCameraActivatePresetCommand object to construct.}}
	{parameter: {name: other} {description: The LmiCameraActivatePresetCommand object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraActivatePresetCommand *LmiCameraActivatePresetCommandConstructCopy(LmiCameraActivatePresetCommand *c, const LmiCameraActivatePresetCommand *other);

/**
{function:
	{name: LmiCameraActivatePresetCommandDestruct}
	{parent: LmiCameraActivatePresetCommand}
	{description: Destructs an LmiCameraActivatePresetCommand object.}
	{prototype: void LmiCameraActivatePresetCommandDestruct(LmiCameraActivatePresetCommand *c)}
	{parameter: {name: c} {description: The LmiCameraActivatePresetCommand object to destruct.}}
}
*/
void LmiCameraActivatePresetCommandDestruct(LmiCameraActivatePresetCommand *c);

/**
{function:
	{name: LmiCameraActivatePresetCommandAssign}
	{parent: LmiCameraActivatePresetCommand}
	{description: Assigns one LmiCameraActivatePresetCommand object the value of another.}
	{prototype: LmiCameraActivatePresetCommand *LmiCameraActivatePresetCommandAssign(LmiCameraActivatePresetCommand *c, const LmiCameraActivatePresetCommand *other)}
	{parameter: {name: c} {description: The LmiCameraActivatePresetCommand object to assign to.}}
	{parameter: {name: other} {description: The LmiCameraActivatePresetCommand object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/
LmiCameraActivatePresetCommand *LmiCameraActivatePresetCommandAssign(LmiCameraActivatePresetCommand *c, const LmiCameraActivatePresetCommand *other);

/**
{function:
	{name: LmiCameraActivatePresetCommandGetIndex}
	{parent: LmiCameraActivatePresetCommand}
	{description: Gets the preset index.}
	{prototype: LmiUint LmiCameraActivatePresetCommandGetIndex(const LmiCameraActivatePresetCommand *c)}
	{parameter: {name: c} {description: The LmiCameraActivatePresetCommand object.}}
	{return: The preset index.}
}
*/
LMI_INLINE_DECLARATION LmiUint LmiCameraActivatePresetCommandGetIndex(const LmiCameraActivatePresetCommand *c);

/**
{type:
	{name: LmiCameraViscaControlCommand}
	{parent: CameraControl}
	{include: Lmi/Video/Common/LmiCameraControl.h}
	{description: A command used to visca control.}
	{note: Visca control commands should only be directed at devices that support them. Check VidyoCameraControlCapabilitiesHasViscaSupport.}
}
*/
typedef struct
{
	LmiDataBuffer viscaMessage;
} LmiCameraViscaControlCommand;

/**
{function:
	{name: LmiCameraViscaControlCommandConstruct}
	{parent: LmiCameraViscaControlCommand}
	{description: Constructs an LmiCameraViscaControlCommand object.}
	{prototype: LmiCameraViscaControlCommand *LmiCameraViscaControlCommandConstruct(LmiCameraViscaControlCommand *c, const LmiString *viscaCmd, const LmiString *viscaCmdId, LmiAllocator *alloc)}
	{parameter: {name: c} {description: The LmiCameraViscaControlCommand object to construct.}}
	{parameter: {name: viscaCmd} {description: Represents the visca command, with maximum length as 128 characters. This should be a any valid visca command.For example: "81 01 04 3f 02 02 ff"}}
	{parameter: {name: viscaCmdId} {description: Represents the OPTIONAL visca command identifier, with maximum length as 128 characters. Application can set NULL OR any valid string identifier, which can be used at remote end for debugging.}}
	{parameter: {name: alloc} {description: The LmiAllocator object.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraViscaControlCommand *LmiCameraViscaControlCommandConstruct(LmiCameraViscaControlCommand *c, const LmiString *viscaCmd, const LmiString *viscaCmdId, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraViscaControlCommandConstructCopy}
	{parent: LmiCameraViscaControlCommand}
	{description: Constructs an LmiCameraViscaControlCommand object as a copy of another.}
	{prototype: LmiCameraViscaControlCommand *LmiCameraViscaControlCommandConstructCopy(LmiCameraViscaControlCommand *c, const LmiCameraViscaControlCommand *other)}
	{parameter: {name: c} {description: The LmiCameraViscaControlCommand object to construct.}}
	{parameter: {name: other} {description: The LmiCameraViscaControlCommand object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraViscaControlCommand *LmiCameraViscaControlCommandConstructCopy(LmiCameraViscaControlCommand *c, const LmiCameraViscaControlCommand *other);

/**
{function:
	{name: LmiCameraViscaControlCommandDestruct}
	{parent: LmiCameraActivatePresetCommand}
	{description: Destructs an LmiCameraViscaControlCommandDestruct object.}
	{prototype: void LmiCameraViscaControlCommandDestruct(LmiCameraViscaControlCommand *c)}
	{parameter: {name: c} {description: The LmiDataBuffer object to destruct.}}
}
*/
void LmiCameraViscaControlCommandDestruct(LmiCameraViscaControlCommand *c);

/**
{function:
	{name: LmiCameraViscaControlCommandAssign}
	{parent: LmiCameraViscaControlCommand}
	{description: Assigns one LmiCameraViscaControlCommand object the value of another.}
	{prototype: LmiCameraViscaControlCommand *LmiCameraViscaControlCommandAssign(LmiCameraViscaControlCommand *c, const LmiCameraViscaControlCommand *other)}
	{parameter: {name: c} {description: The LmiCameraViscaControlCommand object to assign to.}}
	{parameter: {name: other} {description: The LmiCameraViscaControlCommand object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/
LmiCameraViscaControlCommand *LmiCameraViscaControlCommandAssign(LmiCameraViscaControlCommand *c, const LmiCameraViscaControlCommand *other);

/**
{function:
	{name: LmiCameraViscaControlCommandGetViscaMessage}
	{parent: LmiCameraViscaControlCommand}
	{description: Gets the visca message.}
	{prototype:  const LmiDataBuffer *LmiCameraViscaControlCommandGetViscaMessage(const LmiCameraViscaControlCommand *c)}
	{parameter: {name: c} {description: The LmiCameraViscaControlCommand object.}}
	{return: The LmiDataBuffer object.}
}
*/
LMI_INLINE_DECLARATION const LmiDataBuffer *LmiCameraViscaControlCommandGetViscaMessage(const LmiCameraViscaControlCommand *c);

/**
{type:
	{name: LmiCameraCommand}
	{parent: CameraControl}
	{include: Lmi/Video/Common/LmiCameraControl.h}
	{description: An FECC (far-end camera control) command.}
}
*/
typedef struct
{
	LmiCameraCommandType type;
	union
	{
		LmiCameraNudgeCommand nudge;
		LmiCameraSetCommand set;
		LmiCameraMoveCommand move;
		LmiCameraMagnifyCommand magnify;
		LmiCameraRelativePanTiltCommand relativePanTilt;
		LmiCameraRelativeZoomCommand relativeZoom;
		LmiCameraActivatePresetCommand activatePreset;
		LmiCameraViscaControlCommand viscaControl;
	} command;
} LmiCameraCommand;

/**
{function:
	{name: LmiCameraCommandConstructNudge}
	{parent: LmiCameraCommand}
	{description: Constructs an LmiCameraCommand object with a "nudge" command.}
	{prototype: LmiCameraCommand *LmiCameraCommandConstructNudge(LmiCameraCommand *fc, const LmiCameraNudgeCommand *nudge, LmiAllocator *alloc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object to construct.}}
	{parameter: {name: nudge} {description: The nudge command.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraCommand *LmiCameraCommandConstructNudge(LmiCameraCommand *fc, const LmiCameraNudgeCommand *nudge, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraCommandConstructSet}
	{parent: LmiCameraCommand}
	{description: Constructs an LmiCameraCommand object with a "set" command.}
	{prototype: LmiCameraCommand *LmiCameraCommandConstructSet(LmiCameraCommand *fc, const LmiCameraSetCommand *set, LmiAllocator *alloc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object to construct.}}
	{parameter: {name: set} {description: The set command.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraCommand *LmiCameraCommandConstructSet(LmiCameraCommand *fc, const LmiCameraSetCommand *set, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraCommandConstructMove}
	{parent: LmiCameraCommand}
	{description: Constructs an LmiCameraCommand object with a move command.}
	{prototype: LmiCameraCommand *LmiCameraCommandConstructMove(LmiCameraCommand *fc, const LmiCameraMoveCommand *move, LmiAllocator *alloc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object to construct.}}
	{parameter: {name: move} {description: The move command.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraCommand *LmiCameraCommandConstructMove(LmiCameraCommand *fc, const LmiCameraMoveCommand *move, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraCommandConstructMagnify}
	{parent: LmiCameraCommand}
	{description: Constructs an LmiCameraCommand object with a magnify command.}
	{prototype: LmiCameraCommand *LmiCameraCommandConstructMagnify(LmiCameraCommand *fc, const LmiCameraMagnifyCommand *magnify, LmiAllocator *alloc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object to construct.}}
	{parameter: {name: magnify} {description: The magnify command.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraCommand *LmiCameraCommandConstructMagnify(LmiCameraCommand *fc, const LmiCameraMagnifyCommand *magnify, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraCommandConstructRelativePanTilt}
	{parent: LmiCameraCommand}
	{description: Constructs an LmiCameraCommand object with a relative-pan-tilt command.}
	{prototype: LmiCameraCommand *LmiCameraCommandConstructRelativePanTilt(LmiCameraCommand *fc, const LmiCameraRelativePanTiltCommand *relativePanTilt, LmiAllocator *alloc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object to construct.}}
	{parameter: {name: relativePanTilt} {description: The relative-pan-tilt command.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraCommand *LmiCameraCommandConstructRelativePanTilt(LmiCameraCommand *fc, const LmiCameraRelativePanTiltCommand *relativePanTilt, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraCommandConstructRelativeZoom}
	{parent: LmiCameraCommand}
	{description: Constructs an LmiCameraCommand object with a relative-zoom command.}
	{prototype: LmiCameraCommand *LmiCameraCommandConstructRelativeZoom(LmiCameraCommand *fc, const LmiCameraRelativeZoomCommand *relativeZoom, LmiAllocator *alloc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object to construct.}}
	{parameter: {name: relativeZoom} {description: The relative-zoom command.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraCommand *LmiCameraCommandConstructRelativeZoom(LmiCameraCommand *fc, const LmiCameraRelativeZoomCommand *relativeZoom, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraCommandConstructActivatePreset}
	{parent: LmiCameraCommand}
	{description: Constructs an LmiCameraCommand object with a activate preset command.}
	{prototype: LmiCameraCommand *LmiCameraCommandConstructActivatePreset(LmiCameraCommand *fc, const LmiCameraActivatePresetCommand *activatePreset, LmiAllocator *alloc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object to construct.}}
	{parameter: {name: activatePreset} {description: The activate preset command.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraCommand *LmiCameraCommandConstructActivatePreset(LmiCameraCommand *fc, const LmiCameraActivatePresetCommand *activatePreset, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraCommandConstructViscaControl}
	{parent: LmiCameraCommand}
	{description: Constructs an LmiCameraViscaControlCommand object with a visca control command.}
	{prototype: LmiCameraCommand *LmiCameraCommandConstructViscaControl(LmiCameraCommand *fc, const LmiCameraViscaControlCommand *viscaControl, LmiAllocator *alloc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object to construct.}}
	{parameter: {name: viscaControl} {description: The visca control command.}}
	{parameter: {name: alloc} {description: An allocator to use.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraCommand *LmiCameraCommandConstructViscaControl(LmiCameraCommand *fc, const LmiCameraViscaControlCommand *viscaControl, LmiAllocator *alloc);

/**
{function:
	{name: LmiCameraCommandConstructCopy}
	{parent: LmiCameraCommand}
	{description: Constructs an LmiCameraCommand object as a copy of another.}
	{prototype: LmiCameraCommand *LmiCameraCommandConstructCopy(LmiCameraCommand *fc, const LmiCameraCommand *other)}
	{parameter: {name: fc} {description: The LmiCameraCommand object to construct.}}
	{parameter: {name: other} {description: The LmiCameraCommand object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
LmiCameraCommand *LmiCameraCommandConstructCopy(LmiCameraCommand *fc, const LmiCameraCommand *other);

/**
{function:
	{name: LmiCameraCommandDestruct}
	{parent: LmiCameraCommand}
	{description: Destructs an LmiCameraCommand object.}
	{prototype: void LmiCameraCommandDestruct(LmiCameraCommand *fc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object to destruct.}}
}
*/
void LmiCameraCommandDestruct(LmiCameraCommand *fc);

/**
{function:
	{name: LmiCameraCommandAssign}
	{parent: LmiCameraCommand}
	{description: Assigns one LmiCameraCommand object the value of another.}
	{prototype: LmiCameraCommand *LmiCameraCommandAssign(LmiCameraCommand *fc, const LmiCameraCommand *other)}
	{parameter: {name: fc} {description: The LmiCameraCommand object to assign to.}}
	{parameter: {name: other} {description: The LmiCameraCommand object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/
LmiCameraCommand *LmiCameraCommandAssign(LmiCameraCommand *fc, const LmiCameraCommand *other);

/**
{function:
	{name: LmiCameraCommandGetType}
	{parent: LmiCameraCommand}
	{description: Gets the type of an LmiCameraCommand.}
	{prototype: LmiCameraCommandType LmiCameraCommandGetType(const LmiCameraCommand *fc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object.}}
	{return: The command type.}
}
*/
LMI_INLINE_DECLARATION LmiCameraCommandType LmiCameraCommandGetType(const LmiCameraCommand *fc);

/**
{function:
	{name: LmiCameraCommandGetNudge}
	{parent: LmiCameraCommand}
	{description: Get a Nudge command object from an LmiCameraCommand object.}
	{prototype: const LmiCameraNudgeCommand *LmiCameraCommandGetNudge(const LmiCameraCommand *fc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object.}}
	{return: The Nudge command object if the command is of type LMI_CAMERACOMMANDTYPE_Nudge, otherwise NULL.}
}
*/
LMI_INLINE_DECLARATION const LmiCameraNudgeCommand *LmiCameraCommandGetNudge(const LmiCameraCommand *fc);

/**
{function:
	{name: LmiCameraCommandGetSet}
	{parent: LmiCameraCommand}
	{description: Get a Set command object from an LmiCameraCommand object.}
	{prototype: const LmiCameraSetCommand *LmiCameraCommandGetSet(const LmiCameraCommand *fc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object.}}
	{return: The Set command object if the command is of type LMI_CAMERACOMMANDTYPE_Set, otherwise NULL.}
}
*/
LMI_INLINE_DECLARATION const LmiCameraSetCommand *LmiCameraCommandGetSet(const LmiCameraCommand *fc);

/**
{function:
	{name: LmiCameraCommandGetMove}
	{parent: LmiCameraCommand}
	{description: Get a Move command object from an LmiCameraCommand object.}
	{prototype: const LmiCameraMoveCommand *LmiCameraCommandGetMove(const LmiCameraCommand *fc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object.}}
	{return: The Move command object if the command is of type LMI_CAMERACOMMANDTYPE_Move, otherwise NULL.}
}
*/
LMI_INLINE_DECLARATION const LmiCameraMoveCommand *LmiCameraCommandGetMove(const LmiCameraCommand *fc);

/**
{function:
	{name: LmiCameraCommandGetMagnify}
	{parent: LmiCameraCommand}
	{description: Get a Magnify command object from an LmiCameraCommand object.}
	{prototype: const LmiCameraMagnifyCommand *LmiCameraCommandGetMagnify(const LmiCameraCommand *fc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object.}}
	{return: The Magnify command object if the command is of type LMI_CAMERACOMMANDTYPE_Magnify, otherwise NULL.}
}
*/
LMI_INLINE_DECLARATION const LmiCameraMagnifyCommand *LmiCameraCommandGetMagnify(const LmiCameraCommand *fc);

/**
{function:
	{name: LmiCameraCommandGetRelativePanTilt}
	{parent: LmiCameraCommand}
	{description: Get a Relative-Pan-Tilt command object from an LmiCameraCommand object.}
	{prototype: const LmiCameraRelativePanTiltCommand *LmiCameraCommandGetRelativePanTilt(const LmiCameraCommand *fc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object.}}
	{return: The Relative-Pan-Tilt command object if the command is of type LMI_CAMERACOMMANDTYPE_RelativePanTilt, otherwise NULL.}
}
*/
LMI_INLINE_DECLARATION const LmiCameraRelativePanTiltCommand *LmiCameraCommandGetRelativePanTilt(const LmiCameraCommand *fc);

/**
{function:
	{name: LmiCameraCommandGetRelativeZoom}
	{parent: LmiCameraCommand}
	{description: Get a Relative-Zoom command object from an LmiCameraCommand object.}
	{prototype: const LmiCameraRelativeZoomCommand *LmiCameraCommandGetRelativeZoom(const LmiCameraCommand *fc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object.}}
	{return: The Relative-Zoom command object if the command is of type LMI_CAMERACOMMANDTYPE_RelativeZoom, otherwise NULL.}
}
*/
LMI_INLINE_DECLARATION const LmiCameraRelativeZoomCommand *LmiCameraCommandGetRelativeZoom(const LmiCameraCommand *fc);

/**
{function:
	{name: LmiCameraCommandGetActivatePreset}
	{parent: LmiCameraCommand}
	{description: Get a Activate Preset command object from an LmiCameraCommand object.}
	{prototype: const LmiCameraActivatePresetCommand *LmiCameraCommandGetActivatePreset(const LmiCameraCommand *fc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object.}}
	{return: The Activate Preset command object if the command is of type LMI_CAMERACOMMANDTYPE_ActivatePreset, otherwise NULL.}
}
*/
LMI_INLINE_DECLARATION const LmiCameraActivatePresetCommand *LmiCameraCommandGetActivatePreset(const LmiCameraCommand *fc);

/**
{function:
	{name: LmiCameraCommandGetViscaControl}
	{parent: LmiCameraCommand}
	{description: Get a Visca Control command object from an LmiCameraCommand object.}
	{prototype: const LmiCameraViscaControlCommand *LmiCameraCommandGetViscaControl(const LmiCameraCommand *fc)}
	{parameter: {name: fc} {description: The LmiCameraCommand object.}}
	{return: The visca control command object if the command is of type LMI_CAMERACOMMANDTYPE_ViscaControl, otherwise NULL.}
}
*/
LMI_INLINE_DECLARATION const LmiCameraViscaControlCommand *LmiCameraCommandGetViscaControl(const LmiCameraCommand *fc);

Declare_LmiVector(LmiCameraCommand)

LMI_END_EXTERN_C

#if LMI_INLINE_NEED_HEADER_FILE_DEFINITIONS
#include <Lmi/Video/Common/LmiCameraControlInline.h>
#endif

#endif
