/** 
{file:
	{name: VidyoLocalMonitor.h}
	{description: Monitor available for sharing. }
	{copyright:
		(c) 2013-2016 Vidyo, Inc.,
		433 Hackensack Avenue,
		Hackensack, NJ  07601.
		All rights reserved.
		The information contained herein is proprietary to Vidyo, Inc.
		and shall not be reproduced, copied (in whole or in part), adapted,
		modified, disseminated, transmitted, transcribed, stored in a retrieval
		system, or translated into any language in any form by any means
		without the express written consent of Vidyo, Inc.}
}
*/
#ifndef VIDYO_LOCALMONITOR_H_ 
#define VIDYO_LOCALMONITOR_H_ 

#include <Lmi/Utils/LmiTypes.h>
#include <Lmi/Utils/LmiVector.h>
#include <Lmi/Utils/LmiList.h>
#include <Lmi/Utils/LmiRectangle.h>
#include <Lmi/Os/LmiSharedPtr.h>

LMI_BEGIN_EXTERN_C

struct VidyoLocalMonitorImpl_;
typedef struct VidyoLocalMonitorImpl_ VidyoLocalMonitorImpl;
Declare_LmiSharedPtr(VidyoLocalMonitorImpl)
Declare_LmiSharedPtrWrapper(VidyoLocalMonitor, VidyoLocalMonitorImpl)
Declare_LmiWeakPtrWrapper(VidyoLocalMonitorHandle, VidyoLocalMonitor, VidyoLocalMonitorImpl)
Declare_LmiVector(VidyoLocalMonitor)
Declare_LmiList(VidyoLocalMonitor)

LMI_END_EXTERN_C

#include "VidyoLocalRenderer.h"
#include "VidyoRemoteRenderer.h"
#include "VidyoLocalWindowShare.h"
#include "VidyoForwardDecl.h"
#include "VidyoDevice.h"

LMI_BEGIN_EXTERN_C


/**
{type apigroup="connector,simple" reference-counted="yes":
	{name: VidyoLocalMonitor}
	{parent: Device}
	{include: Lmi/VidyoClient/VidyoLocalMonitor.h}
	{description: This object represents a monitor on the local endpoint.}
	{member:
		{name: id}
		{type: LmiString}
		{description: Unique ID.}
	}
	{member:
		{name: name}
		{type: LmiString}
		{description: Name.}
	}
}
*/

/**
{type apigroup="connector,simple" visibility="private":
	{name: VidyoLocalMonitorHandle}
	{parent: VidyoLocalMonitor}
	{description: }
}
*/

/**
{type apigroup="connector,simple":
	{name: VidyoLocalMonitorState}
	{parent: VidyoLocalMonitor}
	{description: State of the monitor.}
	{value: {name: VIDYO_LOCALMONITORSTATE_Ok} {description: The monitor is visible and has been fully captured.}}
	{value: {name: VIDYO_LOCALMONITORSTATE_NotVisible} {description: The monitor exists but is not "visible".}}
	{value: {name: VIDYO_LOCALMONITORSTATE_MiscError} {description: Miscellaneous error has occured trying to capture the frame.}}
}
*/
typedef enum {
	VIDYO_LOCALMONITORSTATE_Ok,
	VIDYO_LOCALMONITORSTATE_NotVisible,
	VIDYO_LOCALMONITORSTATE_MiscError
}
VidyoLocalMonitorState;

/**
{callback visibility="private":
	{name: VidyoLocalMonitorCallback}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: void (*VidyoLocalMonitorCallback)(const VidyoLocalMonitor* m, const VidyoLmiVideoFrame* videoFrame)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: videoFrame}
		{description: }
	}
}
*/
typedef void (*VidyoLocalMonitorCallback)(const VidyoLocalMonitor* m, const VidyoLmiVideoFrame* videoFrame);

/**
{callback:
	{name: VidyoLocalMonitorOnPreviewDataUriComplete}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: void (*VidyoLocalMonitorOnPreviewDataUriComplete)(const VidyoLocalMonitor* m, const LmiString* videoFrame, VidyoLocalMonitorState state)}
	{parameter:
		{name: m}
		{description: }
	}
	{parameter:
		{name: videoFrame}
		{description: }
	}
	{parameter:
		{name: state}
		{description: }
	}
}
*/
typedef void (*VidyoLocalMonitorOnPreviewDataUriComplete)(const VidyoLocalMonitor* m, const LmiString* videoFrame, VidyoLocalMonitorState state);

/**
{callback:
	{name: VidyoLocalMonitorOnGetCapabilities}
	{parent: VidyoLocalMonitor}
	{description: The callback which will be triggered with the most optimal capabilities for the device.}
	{prototype: void (*VidyoLocalMonitorOnGetCapabilities)(const VidyoLocalMonitor *m, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter: {name: m} {description: The VidyoLocalMonitor object.}}
	{parameter: {name: width} {description: The most optimal width.}}
	{parameter: {name: height} {description: The most optimal height.}}
	{parameter: {name: frameInterval} {description: The most optimal frame interval.}}
}
*/
typedef void (*VidyoLocalMonitorOnGetCapabilities)(const VidyoLocalMonitor *m, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
{function visibility="private":
	{name: VidyoLocalMonitorConstruct}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: VidyoLocalMonitor* VidyoLocalMonitorConstruct(VidyoLocalMonitor* m, VidyoEndpoint *endpoint, const VidyoLmiScreen* screen, LmiAllocator* alloc)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: endpoint}
		{description: }
	}
	{parameter:
		{name: screen}
		{description: }
	}
	{parameter:
		{name: alloc}
		{description: }
	}
	{return: }
}
*/
VidyoLocalMonitor* VidyoLocalMonitorConstruct(VidyoLocalMonitor* m, VidyoEndpoint *endpoint, const VidyoLmiScreen* screen, LmiAllocator* alloc);

/**
{function apigroup="connector,simple":
	{name: VidyoLocalMonitorConstructCopy}
	{parent: VidyoLocalMonitor}
	{description: Constructs an VidyoLocalMonitor object as a copy of another.}
	{prototype: VidyoLocalMonitor *VidyoLocalMonitorConstructCopy(VidyoLocalMonitor *obj, const VidyoLocalMonitor *other)}
	{parameter: {name: obj} {description: The VidyoLocalMonitor object to construct.}}
	{parameter: {name: other} {description: The call object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/

/**
{function apigroup="connector,simple":
	{name: VidyoLocalMonitorDestruct}
	{parent: VidyoLocalMonitor}
	{description: Destructs an VidyoLocalMonitor object.}
	{prototype: void VidyoLocalMonitorDestruct(VidyoLocalMonitor *obj)}
	{parameter: {name: obj} {description: The VidyoLocalMonitor object to destruct.}}
}
*/

/**
{function apigroup="connector,simple" visibility="private":
	{name: VidyoLocalMonitorAssign}
	{parent: VidyoLocalMonitor}
	{description: Assigns one VidyoLocalMonitor object the value of another. }
	{prototype: VidyoLocalMonitor *VidyoLocalMonitorAssign(VidyoLocalMonitor *obj, const VidyoLocalMonitor *other)}
	{parameter: {name: obj} {description: The VidyoLocalMonitor object.}}
	{parameter: {name: other} {description: The call object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/

/**
{function apigroup="connector,simple" visibility="private":
	{name: VidyoLocalMonitorGetHandle}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: VidyoLocalMonitorHandle* VidyoLocalMonitorGetHandle(VidyoLocalMonitor* m)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{return: }
}
*/
VidyoLocalMonitorHandle* VidyoLocalMonitorGetHandle(VidyoLocalMonitor* m);

/**
{function apigroup="connector,simple":
	{name: VidyoLocalMonitorGetName}
	{parent: VidyoLocalMonitor}
	{description: Gets the name of the monitor.}
	{prototype: const LmiString* VidyoLocalMonitorGetName(const VidyoLocalMonitor* m)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{return: Name of the monitor.}
}
*/
const LmiString* VidyoLocalMonitorGetName(const VidyoLocalMonitor* m);

/**
{function apigroup="connector,simple":
	{name: VidyoLocalMonitorGetId}
	{parent: VidyoLocalMonitor}
	{description: Gets the unique ID of the monitor.}
	{prototype: const LmiString* VidyoLocalMonitorGetId(const VidyoLocalMonitor* m)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{return: Unique ID of the monitor. }
}
*/
const LmiString* VidyoLocalMonitorGetId(const VidyoLocalMonitor* m);

/**
{function visibility="private":
	{name: VidyoLocalMonitorGetPosition}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: const LmiRectangle* VidyoLocalMonitorGetPosition(const VidyoLocalMonitor* m)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{return: }
}
*/
const LmiRectangle* VidyoLocalMonitorGetPosition(const VidyoLocalMonitor* m);

/**
{function visibility="private":
	{name: VidyoLocalMonitorGetWorkArea}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: const LmiRectangle* VidyoLocalMonitorGetWorkArea(const VidyoLocalMonitor* m)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{return: }
}
*/
const LmiRectangle* VidyoLocalMonitorGetWorkArea(const VidyoLocalMonitor* m);

/**
{function:
	{name: VidyoLocalMonitorIsPrimary}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: LmiBool VidyoLocalMonitorIsPrimary(VidyoLocalMonitor* m)}
	{parameter:
		{name: m}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoLocalMonitorIsPrimary(VidyoLocalMonitor* m);

/**
{function:
	{name: VidyoLocalMonitorSetCaptureInterval}
	{parent: VidyoLocalMonitor}
	{description: Set frame interval(fps) for capturer.}
	{prototype: LmiBool VidyoLocalMonitorSetCaptureInterval(VidyoLocalMonitor* m, LmiTime frameInterval)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: frameInterval}
		{description: The interval in nanoseconds between consecutive frames. By default will be calculated based on encoder configuration.}
	}
	{return: }
}
*/
LmiBool VidyoLocalMonitorSetCaptureInterval(VidyoLocalMonitor* m, LmiTime frameInterval);

/**
{function apigroup="connector,simple" visibility="private":
	{name: VidyoLocalMonitorSetUserData}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: void VidyoLocalMonitorSetUserData(VidyoLocalMonitor* m, LmiVoidPtr userData)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: userData}
		{description: }
	}
}
*/
void VidyoLocalMonitorSetUserData(VidyoLocalMonitor* m, LmiVoidPtr userData);

/**
{function apigroup="connector,simple" visibility="private":
	{name: VidyoLocalMonitorGetUserData}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: LmiVoidPtr VidyoLocalMonitorGetUserData(const VidyoLocalMonitor* m)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{return: }
}
*/
LmiVoidPtr VidyoLocalMonitorGetUserData(const VidyoLocalMonitor* m);

/**
{function visibility="private":
	{name: VidyoLocalMonitorSetCallback}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: void VidyoLocalMonitorSetCallback(VidyoLocalMonitor* m, VidyoLocalMonitorCallback callback)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: callback}
		{description: }
	}
}
*/
void VidyoLocalMonitorSetCallback(VidyoLocalMonitor* m, VidyoLocalMonitorCallback callback);

/**
{function visibility="private":
	{name: VidyoLocalMonitorGetPreviewFrame}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: VidyoLocalMonitorState VidyoLocalMonitorGetPreviewFrame(VidyoLocalMonitor *m, LmiSizeT maxWidth, LmiSizeT maxHeight, VidyoLmiVideoFrame *videoFrame)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: maxWidth}
		{description: Maximum width of the preview image.}
	}
	{parameter:
		{name: maxHeight}
		{description: Maximum height of the preview image.}
	}
	{parameter:
		{name: videoFrame}
		{description: Video Frame to which the image will be assigned.}
	}
	{return: }
	{note: The aspect ratio will be preserved. }
	{note: The videoFrame will only contain the new image if VidyoLocalMonitorState VIDYO_LOCALMONITORSTATE_Ok, otherwise will will be assigned last valid preview image. }
}
*/
VidyoLocalMonitorState VidyoLocalMonitorGetPreviewFrame(VidyoLocalMonitor *m, LmiSizeT maxWidth, LmiSizeT maxHeight, VidyoLmiVideoFrame *videoFrame);

/**
{function:
	{name: VidyoLocalMonitorGetPreviewFrameDataUri}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: VidyoLocalMonitorState VidyoLocalMonitorGetPreviewFrameDataUri(VidyoLocalMonitor *m, LmiSizeT maxWidth, LmiSizeT maxHeight, LmiString *videoFrame)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: maxWidth}
		{description: Maximum width of the preview image.}
	}
	{parameter:
		{name: maxHeight}
		{description: Maximum height of the preview image.}
	}
	{parameter:
		{name: videoFrame}
		{description: Video Frame to which the image will be assigned.}
	}
	{return: }
	{note: The aspect ratio will be preserved. }
	{note: The videoFrame will only contain the new image if VidyoLocalMonitorState VIDYO_LOCALMONITORSTATE_Ok, otherwise will will be assigned last valid preview image. }
}
*/
VidyoLocalMonitorState VidyoLocalMonitorGetPreviewFrameDataUri(VidyoLocalMonitor *m, LmiSizeT maxWidth, LmiSizeT maxHeight, LmiString *videoFrame);

/**
{function:
	{name: VidyoLocalMonitorGetPreviewFrameDataUriAsync}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: LmiBool VidyoLocalMonitorGetPreviewFrameDataUriAsync(VidyoLocalMonitor* m, LmiSizeT maxWidth, LmiSizeT maxHeight, VidyoLocalMonitorOnPreviewDataUriComplete onComplete)}
	{parameter:
		{name: m}
		{description: }
	}
	{parameter:
		{name: maxWidth}
		{description: Maximum width of the preview image.}
	}
	{parameter:
		{name: maxHeight}
		{description: Maximum height of the preview image.}
	}
	{parameter:
		{name: onComplete}
		{description: Callback that is triggered when a preview frame is captured.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	{note: The aspect ratio will be preserved. }
}
*/
LmiBool VidyoLocalMonitorGetPreviewFrameDataUriAsync(VidyoLocalMonitor* m, LmiSizeT maxWidth, LmiSizeT maxHeight, VidyoLocalMonitorOnPreviewDataUriComplete onComplete);

/**
{function apigroup="connector,simple" deprecated="yes" replaced-by="VidyoLocalMonitorSetMinFrameInterval":
	{name: VidyoLocalMonitorSetFrameInterval}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: LmiBool VidyoLocalMonitorSetFrameInterval(VidyoLocalMonitor* m, LmiTime frameInterval)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: frameInterval}
		{description: The interval in nanoseconds between consecutive frames. Default is 333333333 which is 3 frames per second. }
	}
	{return: LMI_TRUE on success, LMI_FALSE otherwise.}
	{note: To set 3 frames per seconds use 1000000000/3 as the value.}
}
*/
LmiBool VidyoLocalMonitorSetFrameInterval(VidyoLocalMonitor* m, LmiTime frameInterval);

/**
{function visibility="private" deprecated="yes" replaced-by="VidyoLocalMonitorSetMinFrameInterval":
	{name: VidyoLocalMonitorSetFrameIntervalConstraints}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: LmiBool VidyoLocalMonitorSetFrameIntervalConstraints(VidyoLocalMonitor* c, LmiInt maxFrameInterval, LmiInt minFrameInterval, LmiFloat32 scaleFactor)}
	{parameter:
		{name: c}
		{description: }
	}
	{parameter:
		{name: maxFrameInterval}
		{description: Maximum time between frames in ns. This is a hint to the resource manager for how it decides to allocate internal resources (cpu,mem,etc)}
	}
	{parameter:
		{name: minFrameInterval}
		{description: Minimum time between frames in ns. This is a hint to the resource manager for how it decides to allocate internal resources (cpu,mem,etc)}
	}
	{parameter:
		{name: scaleFactor}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoLocalMonitorSetFrameIntervalConstraints(VidyoLocalMonitor* c, LmiInt maxFrameInterval, LmiInt minFrameInterval, LmiFloat32 scaleFactor);

/**
{function apigroup="connector,simple,beta"  deprecated="yes":
	{name: VidyoLocalMonitorSetLowLatencyProfile}
	{parent: VidyoLocalMonitor}
	{description: Sets the low latency profile of the monitor which treats the stream as a regular video source. This prioritizes latency over quality.}
	{prototype: LmiBool VidyoLocalMonitorSetLowLatencyProfile(VidyoLocalMonitor* m, LmiBool profile)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: profile}
		{description: Low latency profile toggle. Default is LMI_FALSE. }
	}
	{return: LMI_TRUE on success, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoLocalMonitorSetLowLatencyProfile(VidyoLocalMonitor* m, LmiBool profile);

/**
{function deprecated="yes" replaced-by="VidyoLocalMonitorSetMaxConstraints":
	{name: VidyoLocalMonitorSetBoundsConstraints}
	{parent: VidyoLocalMonitor}
	{description: Set a maximum width and height for the encoded stream and frame interval bounds. The stream size will be scaled down to fit the maxWidth/maxHeight if the capture is larger than either of those dimensions }
	{prototype: LmiBool VidyoLocalMonitorSetBoundsConstraints(VidyoLocalMonitor* m, LmiTime maxFrameInterval, LmiTime minFrameInterval, LmiUint maxWidth, LmiUint minWidth, LmiUint maxHeight, LmiUint minHeight)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: maxFrameInterval}
		{description: Maximum time between frames in ns. This is a hint to the resource manager for how it decides to allocate internal resources (cpu,mem,etc)}
	}
	{parameter:
		{name: minFrameInterval}
		{description: Minimum time between frames in ns. This is a hint to the resource manager for how it decides to allocate internal resources (cpu,mem,etc)}
	}
	{parameter:
		{name: maxWidth}
 		{description: Maximum width of the encoded stream. If the captured stream width is larger than this max the stream will be scaled before sending to fit}
	}
	{parameter:
		{name: minWidth}
		{description: Unused}
	}
	{parameter:
		{name: maxHeight}
		{description: Maximum height of the encoded stream. If the captured stream height is larger than this max the stream will be scaled before sending to fit }
	}
	{parameter:
		{name: minHeight}
		{description: Unusued}
	}
	{return: }
}
*/
LmiBool VidyoLocalMonitorSetBoundsConstraints(VidyoLocalMonitor* m, LmiTime maxFrameInterval, LmiTime minFrameInterval, LmiUint maxWidth, LmiUint minWidth, LmiUint maxHeight, LmiUint minHeight);

/**
{function deprecated="yes" replaced-by="VidyoLocalMonitorSetMaxConstraints":
	{name: VidyoLocalMonitorSetDiscreteConstraints}
	{parent: VidyoLocalMonitor}
	{description: Set the stream characteristics, causing the stream to be reconfigured. This is an override which allows teh application to control the stream reconfiguration. After calling SetDiscreteConstraints the stream will not longer be reconfigured unless SetDiscreteConstraints again or SetBoundsConstraints is used}
	{prototype: LmiBool VidyoLocalMonitorSetDiscreteConstraints(VidyoLocalMonitor* m, LmiTime maxFrameInterval, LmiTime minFrameInterval, LmiUint width, LmiUint height, LmiFloat32 scaleFactor)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: maxFrameInterval}
		{description: Maximum time between frames in ns. This is a hint to the resource manager for how it decides to allocate internal resources (cpu,mem,etc)}
	}
	{parameter:
		{name: minFrameInterval}
		{description: Minimum time between frames in ns. This is a hint to the resource manager for how it decides to allocate internal resources (cpu,mem,etc)}
	}
	{parameter:
		{name: width}
 		{description: The width of the stream after the stream is reconfigured }
	}
	{parameter:
		{name: height}
		{description: The height of the stream after the stream is reconfigured}
	}
	{parameter:
		{name: scaleFactor}
		{description: This should be 1 }
	}
	{return: }
}
*/
LmiBool VidyoLocalMonitorSetDiscreteConstraints(VidyoLocalMonitor* m, LmiTime maxFrameInterval, LmiTime minFrameInterval, LmiUint width, LmiUint height, LmiFloat32 scaleFactor);

/**
{function apigroup="connector,simple":
	{name: VidyoLocalMonitorSetMaxConstraints}
	{parent: VidyoLocalMonitor}
	{description: Set a maximum width, height, and minimum frame interval(max fps) for the encoded stream. The stream size will be scaled down with a preserving aspect ratio to fit the width/height if the capture is larger than either of those dimensions. If the monitor is in landscape mode width and height will be applied as max constraints, if the monitor is in portrait mode width and height will be swapped. Any values are allowed for width and height if they fit width multiply height equal to or less than 4K pixels(3840 x 2160 = 8294400 pixels) and higher than 180p (320 x 180 = 57600 pixels).}
	{prototype: LmiBool VidyoLocalMonitorSetMaxConstraints(VidyoLocalMonitor* m, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object}
	}
	{parameter:
		{name: width}
		{description: Maximum width of the encoded stream. If the captured stream width is larger than this max the stream will be scaled before sending to fit. The default value is 3840.}
	}
	{parameter:
		{name: height}
		{description: Maximum height of the encoded stream. If the captured stream height is larger than this max the stream will be scaled before sending to fit. The default value is 2160.}
	}
	{parameter:
		{name: frameInterval}
		{description: The interval in nanoseconds between consecutive frames. The default is 333333333 (3fps). To enable a high frame rate share set value less than 142857142 (7 fps). Range of allowed values 33333333(30fps) - 5000000000(1 frame per 5 secs).}
	}
	{return: LMI_TRUE on success, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoLocalMonitorSetMaxConstraints(VidyoLocalMonitor* m, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
{function apigroup="connector,simple":
	{name: VidyoLocalMonitorSetMinFrameInterval}
	{parent: VidyoLocalMonitor}
	{description: Set a minimum frame interval(maximum fps) for the encoded stream.}
	{prototype: LmiBool VidyoLocalMonitorSetMinFrameInterval(VidyoLocalMonitor* m, LmiTime frameInterval)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: frameInterval}
		{description: The interval in nanoseconds between consecutive frames. The default is 333333333 (3fps). To enable a high frame rate share set value less than 142857142 (7 fps). Range of allowed values 33333333(30fps) - 5000000000(1 frame per 5 secs).}
	}
	{return: LMI_TRUE on success, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoLocalMonitorSetMinFrameInterval(VidyoLocalMonitor* m, LmiTime frameInterval);

/**
{function apigroup="connector,simple":
	{name: VidyoLocalMonitorClearConstraints}
	{parent: VidyoLocalMonitor}
	{description: Clears existing constraints and set them to default values 3840x2160@3fps.}
	{prototype: LmiBool VidyoLocalMonitorClearConstraints(VidyoLocalMonitor* m)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoLocalMonitorClearConstraints(VidyoLocalMonitor* m);

/**
{function:
	{name: VidyoLocalMonitorAddToRemoteRenderer}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: LmiBool VidyoLocalMonitorAddToRemoteRenderer(VidyoLocalMonitor* m, const VidyoRemoteRenderer* remoteRenderer)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: remoteRenderer}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoLocalMonitorAddToRemoteRenderer(VidyoLocalMonitor* m, const VidyoRemoteRenderer* remoteRenderer);

/**
{function:
	{name: VidyoLocalMonitorRemoveFromRemoteRenderer}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: LmiBool VidyoLocalMonitorRemoveFromRemoteRenderer(VidyoLocalMonitor* m, const VidyoRemoteRenderer* remoteRenderer)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: remoteRenderer}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoLocalMonitorRemoveFromRemoteRenderer(VidyoLocalMonitor* m, const VidyoRemoteRenderer* remoteRenderer);

/**
{function:
	{name: VidyoLocalMonitorAddToLocalRenderer}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: LmiVoidPtr VidyoLocalMonitorAddToLocalRenderer(VidyoLocalMonitor* m, const VidyoLocalRenderer* renderer)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: renderer}
		{description: }
	}
	{return: }
}
*/
LmiVoidPtr VidyoLocalMonitorAddToLocalRenderer(VidyoLocalMonitor* m, const VidyoLocalRenderer* renderer);

/**
{function:
	{name: VidyoLocalMonitorRemoveFromLocalRenderer}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: LmiBool VidyoLocalMonitorRemoveFromLocalRenderer(VidyoLocalMonitor* m, const VidyoLocalRenderer* renderer)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: renderer}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoLocalMonitorRemoveFromLocalRenderer(VidyoLocalMonitor* m, const VidyoLocalRenderer* renderer);

/**
{function:
	{name: VidyoLocalMonitorSetPositionInLocalRenderer}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: LmiBool VidyoLocalMonitorSetPositionInLocalRenderer(VidyoLocalMonitor *m, const VidyoLocalRenderer *localRenderer, LmiInt x, LmiInt y, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: localRenderer}
		{description: }
	}
	{parameter:
		{name: x}
		{description: }
	}
	{parameter:
		{name: y}
		{description: }
	}
	{parameter:
		{name: width}
		{description: }
	}
	{parameter:
		{name: height}
		{description: }
	}
	{parameter:
		{name: frameInterval}
		{description: The interval in nanoseconds between consecutive frames. }
	}
	{return: }
}
*/
LmiBool VidyoLocalMonitorSetPositionInLocalRenderer(VidyoLocalMonitor *m, const VidyoLocalRenderer *localRenderer, LmiInt x, LmiInt y, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
{function:
	{name: VidyoLocalMonitorPrecludeWindow}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: void VidyoLocalMonitorPrecludeWindow(VidyoLocalMonitor* m, const VidyoLocalWindowShare* w)}
	{parameter:
		{name: m}
		{description: The VidyoLocalMonitor object. }
	}
	{parameter:
		{name: w}
		{description: }
	}
}
*/
void VidyoLocalMonitorPrecludeWindow(VidyoLocalMonitor* m, const VidyoLocalWindowShare* w);

/**
 {function:
	{name: VidyoLocalMonitorPrecludeApplication}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: void VidyoLocalMonitorPrecludeApplication(VidyoLocalMonitor* m, const VidyoLocalWindowShare* w)}
	{parameter:
		{name: m}
		{description: }
	}
	{parameter:
		{name: w}
		{description: }
	}
 }
 */
void VidyoLocalMonitorPrecludeApplication(VidyoLocalMonitor* m, const VidyoLocalWindowShare* w);


/**
{function visibility="private":
	{name: VidyoLocalMonitorConstructFromDevice}
	{parent: VidyoLocalMonitor}
	{description: }
	{prototype: VidyoLocalMonitor *VidyoLocalMonitorConstructFromDevice(VidyoLocalMonitor* vidyoLocalMonitor, const VidyoDevice *device, LmiAllocator* alloc)}
	{parameter:
		{name: vidyoLocalMonitor}
		{description: }
	}
	{parameter:
		{name: device}
		{description: }
	}
	{parameter:
		{name: alloc}
		{description: }
	}
	{return: }
}
*/
VidyoLocalMonitor *VidyoLocalMonitorConstructFromDevice(VidyoLocalMonitor* vidyoLocalMonitor, const VidyoDevice *device, LmiAllocator* alloc);

/**
{function visibility="private":
	{name: VidyoLocalMonitorGetStats}
	{parent: VidyoLocalMonitor}
	{description: Fill in stats data in VidyoLocalVideoSourceStats from a local monitor.}
	{prototype: LmiBool VidyoLocalMonitorGetStats(VidyoLocalMonitor *monitor, VidyoLocalVideoSourceStats *stats)}
	{parameter: {name: monitor} {description: The local monitor object to extract stats.}}
	{parameter: {name: stats} {description: The stats object to fill.}}
	{return: LMI_TRUE on success.}
}
*/
LmiBool VidyoLocalMonitorGetStats(VidyoLocalMonitor *monitor, VidyoLocalVideoSourceStats *stats);

/**
{function:
	{name: VidyoLocalMonitorExcludeWindow}
	{parent: VidyoLocalMonitor}
	{description: Enable/disable window exclude. Only top-level window related to the current process can be excluded.}
	{prototype: LmiBool VidyoLocalMonitorExcludeWindow(VidyoLocalMonitor *monitor, const VidyoLocalWindowShare *window, LmiBool exclude)}
	{parameter: {name: monitor} {description: The VidyoLocalMonitor object.}}
	{parameter: {name: window} {description: The VidyoLocalWindowShare object.}}
	{parameter: {name: exclude} {description: The variable indicates whether the window will be excluded or not.}}
	{return: LMI_TRUE on success, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoLocalMonitorExcludeWindow(VidyoLocalMonitor *monitor, const VidyoLocalWindowShare* window, LmiBool exclude);

/**
{function:
	{name: VidyoLocalMonitoreGetCapabilities}
	{parent: VidyoLocalMonitor}
	{description: Get the capabilities of the operating point that is currently used as the most optimal for the device. Note: These capabilities are used during starting the device.}
	{prototype: LmiBool VidyoLocalMonitoreGetCapabilities(VidyoLocalMonitor *monitor, VidyoLocalMonitorOnGetCapabilities onGetCapabilities)}
	{parameter: {name: monitor} {description: The VidyoLocalMonitor object.}}
	{parameter: {name: onGetCapabilities} {description: The callback to which we will pass the most optimal capabilities.}}
	{return: LMI_TRUE on success, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoLocalMonitoreGetCapabilities(VidyoLocalMonitor *monitor, VidyoLocalMonitorOnGetCapabilities onGetCapabilities);

#if LMI_INLINE_NEED_HEADER_FILE_DEFINITIONS
#include "VidyoLocalMonitorInline.h"
#endif

LMI_END_EXTERN_C

#endif /* VIDYO_LOCALMONITOR_H_ */
