/**
{file:
	{name: VidyoNetworkService.h}
	{description: Network Service Interface.}
	{copyright:
		(c) 2014-2016 Vidyo, Inc.,
		433 Hackensack Avenue,
		Hackensack, NJ  07601.

		All rights reserved.

		The information contained herein is proprietary to Vidyo, Inc.
		and shall not be reproduced, copied (in whole or in part), adapted,
		modified, disseminated, transmitted, transcribed, stored in a retrieval
		system, or translated into any language in any form by any means
		without the express written consent of Vidyo, Inc.}
}
*/
#ifndef VIDYO_NETWORKSERVICE_H_
#define VIDYO_NETWORKSERVICE_H_

#include <Lmi/Utils/LmiTypes.h>
#include <Lmi/Os/LmiSharedPtr.h>
#include <Lmi/Utils/LmiString.h>

LMI_BEGIN_EXTERN_C

struct VidyoNetworkServiceImpl_;
typedef struct VidyoNetworkServiceImpl_ VidyoNetworkServiceImpl;
Declare_LmiSharedPtr(VidyoNetworkServiceImpl)
Declare_LmiSharedPtrWrapper(VidyoNetworkService, VidyoNetworkServiceImpl)
Declare_LmiWeakPtrWrapper(VidyoNetworkServiceHandle, VidyoNetworkService, VidyoNetworkServiceImpl)

/**
{package apigroup="networkservice, simple":
	{name: NetworkService}
	{parent: VidyoClient}
	{include: Lmi/VidyoClient/VidyoNetworkService.h}
	{description: Package that encapsulates all the APIs required to use REST and SOAP protocol.}
}
*/
/**
{type apigroup="networkservice" reference-counted="yes":
	{name: VidyoNetworkService}
	{parent: NetworkService}
	{description: An optional module that handles all request related to REST and SOAP protocol.}
}
*/

/**
{type apigroup="networkservice" visibility="public":
	{name: VidyoNetworkServiceErrorCode}
	{parent: VidyoNetworkService}
	{description: NetworkService error codes.}
	{value: {name: VIDYO_NETWORKSERVICE_ERRORCODE_Ok}
			{description: Successful request.}}
	{value: {name: VIDYO_NETWORKSERVICE_ERRORCODE_Error}
			{description: Unsuccessful request.}}
	{value: {name: VIDYO_NETWORKSERVICE_ERRORCODE_MandatoryParamsMissing}
			{description: Mandatory parameters are missing from request.}}
	{value: {name: VIDYO_NETWORKSERVICE_ERRORCODE_BadRequest}
			{description: Invalid parameters.}}
	{value: {name: VIDYO_NETWORKSERVICE_ERRORCODE_MaxSimultaneousRequestsReached}
			{description: Maximum number of simultaneous requests reached. By default 10 simultaneous requests.}}
	{value: {name: VIDYO_NETWORKSERVICE_ERRORCODE_HttpRequestTimeOut}
			{description: Http request time out received.}}
}
*/
typedef enum
{
	VIDYO_NETWORKSERVICE_ERRORCODE_Ok,
	VIDYO_NETWORKSERVICE_ERRORCODE_Error,
	VIDYO_NETWORKSERVICE_ERRORCODE_MandatoryParamsMissing,
	VIDYO_NETWORKSERVICE_ERRORCODE_BadRequest,
	VIDYO_NETWORKSERVICE_ERRORCODE_MaxSimultaneousRequestsReached,
	VIDYO_NETWORKSERVICE_ERRORCODE_HttpRequestTimeOut
}VidyoNetworkServiceErrorCode;

/**
{type apigroup="networkservice" visibility="public":
	{name: VidyoNetworkServiceRequestHeader}
	{parent: VidyoNetworkService}
	{description: Represents the header structure of the http request.}
	{member:
		{name: requestUrl}
		{type: LmiString}
		{description: Represents the url of the request. Requirement: Mandatory.}}
	{member:
		{name: requestMethod}
		{type: LmiString}
		{description: Represents the type of the request. Default value: post. Requirement: Optional.}}
	{member:
		{name: requestPayload}
		{type: LmiString}
		{description: Represents the payload of the request. Default value: empty. Requirement: Optional.}}
	{member:
		{name: requestContentType}
		{type: LmiString}
		{description: Represents the content/encoding type of the request. Default value: empty. Requirement: Optional.}}
	{member:
		{name: requestNumber}
		{type: LmiString}
		{description: Represents the request number of the request. Default value: empty. Requirement: Optional.}}
	{member:
		{name: userAuthToken}
		{type: LmiString}
		{description: Represents the authorization token of the request. Default value: empty. Requirement: Optional.}}
	{member:
		{name: userAuthUserName}
		{type: LmiString}
		{description: Represents the authorization user name of the request. Default value: empty. Requirement: Optional.}}
	{member:
		{name: userAuthUserPassword}
		{type: LmiString}
		{description: Represents the authorization user password of the request. Default value: empty. Requirement: Optional.}}
	{member:
		{name: webProxyUserName}
		{type: LmiString}
		{description: Represents the user name of the webproxy. Default value: empty. Requirement: Optional.}}
	{member:
		{name: webProxyUserPassword}
		{type: LmiString}
		{description: Represents the user password of the webproxy. Default value: empty. Requirement: Optional.}}
}
*/
typedef struct
{
	LmiString requestUrl;
	LmiString requestMethod;
	LmiString requestPayload;
	LmiString requestContentType;
	LmiString requestNumber;
	LmiString userAuthToken;
	LmiString userAuthUserName;
	LmiString userAuthUserPassword;
	LmiString webProxyUserName;
	LmiString webProxyUserPassword;
}VidyoNetworkServiceRequestHeader;

/**
{type apigroup="networkservice" visibility="public":
	{name:  VidyoNetworkServiceCAInfo}
	{parent: NetworkService}
	{description: Represents a certificate authority structure. if not specify, by default OS certificates will be used.}
	{member:
		{name: caFileContent}
		{type: LmiString}
		{description: The content of the CA file that will be use by the library when connecting to the specified server. Default value: empty. Requirement: Optional.
					Each certificate authority content should be ended with '\n' or '\r\n'.
					Note: If the string 'default' is passed here, VidyoNetworkService will use a built-in Certificate Authority bundle.}}
	{member:
		{name: caFilePath}
		{type: LmiString}
		{description: The path to the CA certificate file that the library will use when connecting to the specified server. Default value: empty. Requirement: Optional.}}
}
*/
typedef struct
{
	LmiString caFileContent;
	LmiString caFilePath;
}VidyoNetworkServiceCAInfo;

/**
{type apigroup="networkservice" visibility="public":
	{name:  VidyoNetworkServiceResponseHeader}
	{parent: VidyoNetworkService}
	{description: Represents a http response header structure.}
	{member:
		{name: requestNumber}
		{type: LmiString}
		{description: Represents the unique request number for each Http request, provided/used in VidyoNetworkServiceHttpRequestASync.}}
	{member:
		{name: errorCode}
		{type: VidyoNetworkServiceErrorCode}
		{description: Represents the error code, see VidyoNetworkServiceErrorCode.}}
	{member:
		{name: httpStatusCode}
		{type: LmiUint}
		{description: Represents the http status code of the response.}}
	{member:
		{name: httpResponseContentSize}
		{type: LmiUint}
		{description: Represents the content size of the respons.}}
	{member:
		{name: httpResponseContent}
		{type: LmiString}
		{description: Represents the content of the response.}}
	{member:
		{name: httpResponseContentType}
		{type: LmiString}
		{description: Represents the content type of response.}}
	{member:
		{name: webProxyIPAddress}
		{type: LmiString}
		{description: Represents the discovered webproxy IP address.}}
	{member:
		{name: webProxyIPPort}
		{type: LmiUint}
		{description: Represents the discovered webproxy IP address port.}}
}
*/
typedef struct
{
	LmiString requestNumber;
	VidyoNetworkServiceErrorCode errorCode;
	LmiUint httpStatusCode;
	LmiUint httpResponseContentSize;
	LmiString httpResponseContent;
	LmiString httpResponseContentType;
	LmiString webProxyIPAddress;
	LmiUint webProxyIPPort;
}VidyoNetworkServiceResponseHeader;

/**
 {callback apigroup="networkservice":
	{name: VidyoNetworkServiceResponseCB}
	{parent: VidyoNetworkService}
	{description: This is Http Response CallBack, Application should define this callback in order to get response data.
					VidyoClient library will not process http response including status code.
					Application has to process the response and take necessary action.}
	{prototype: void(*VidyoNetworkServiceResponseCB)(VidyoNetworkService* networkService, VidyoNetworkServiceResponseHeader* httpResponseHeader, LmiVoidPtr userData)}
	{parameter:
		{name: networkService}
		{description: The VidyoNetworkService that triggered the callback.}}
	{parameter:
		{name: httpResponseHeader}
		{description: Http response header data.}}
	{parameter:
		{name: userData}
		{description: This is the same data that application passed in VidyoNetworkServiceHttpRequestASync.}}
}
*/
typedef void(*VidyoNetworkServiceResponseCB)(VidyoNetworkService* networkService, VidyoNetworkServiceResponseHeader* httpResponseHeader, LmiVoidPtr userData);

/**
{function apigroup="networkservice" visibility="private":
	{name: VidyoNetworkServiceCAInfoConstruct}
	{parent: VidyoNetworkServiceCAInfo}
	{description: Constructs an VidyoNetworkServiceCAInfo object.}
	{prototype: VidyoNetworkServiceCAInfo* VidyoNetworkServiceCAInfoConstruct(VidyoNetworkServiceCAInfo* caInfo, const LmiString *caFileContent, const LmiString *caFilePath, LmiAllocator* alloc)}
	{parameter:
		{name: caInfo}
		{description: The VidyoNetworkServiceCAInfo object to construct.}}
	{parameter:
		{name: caFileContent}
		{description: The path to the CA file content that the library will use when connecting to the specified server.}}
	{parameter:
		{name: caFilePath}
		{description: The path to the CA certificate file that the library will use when connecting to the specified server.}}
	{parameter:
		{name: alloc}
		{description: The LmiAllocator object.}}
	{return: VidyoNetworkServiceCAInfo object on success otherwise NULL.}
}
*/
VidyoNetworkServiceCAInfo* VidyoNetworkServiceCAInfoConstruct(VidyoNetworkServiceCAInfo* caInfo, const LmiString *caFileContent, const LmiString *caFilePath, LmiAllocator* alloc);

/**
{function apigroup="networkservice" visibility="private":
	{name: VidyoNetworkServiceCAInfoConstructDefault}
	{parent: VidyoNetworkServiceCAInfo}
	{description: Default construct for VidyoNetworkServiceCAInfo.}
	{prototype: VidyoNetworkServiceCAInfo* VidyoNetworkServiceCAInfoConstructDefault(VidyoNetworkServiceCAInfo* caInfo, LmiAllocator* alloc)}
	{parameter:
		{name: caInfo}
		{description: The VidyoNetworkServiceCAInfo object to construct.}}
	{parameter:
		{name: alloc}
		{description: The LmiAllocator object.}}
	{return: VidyoNetworkServiceCAInfo object on success otherwise NULL.}
}
*/
VidyoNetworkServiceCAInfo* VidyoNetworkServiceCAInfoConstructDefault(VidyoNetworkServiceCAInfo* caInfo, LmiAllocator* alloc);

/**
{function apigroup="networkservice" visibility="private":
	{name: VidyoNetworkServiceCAInfoConstructCopy}
	{parent: VidyoNetworkServiceCAInfo}
	{description: Constructs an VidyoNetworkServiceCAInfo object as a copy of an existing object.}
	{prototype: const VidyoNetworkServiceCAInfo* VidyoNetworkServiceCAInfoConstructCopy(VidyoNetworkServiceCAInfo* d, VidyoNetworkServiceCAInfo* s)}
	{parameter:
		{name: d}
		{description: The VidyoNetworkServiceCAInfo object to construct.}}
	{parameter:
		{name: s}
		{description: The VidyoNetworkServiceCAInfo object to make a copy of.}}
	{return: VidyoNetworkServiceCAInfo object on success otherwise NULL.}
}
*/
const VidyoNetworkServiceCAInfo* VidyoNetworkServiceCAInfoConstructCopy(VidyoNetworkServiceCAInfo* d, VidyoNetworkServiceCAInfo* s);

/**
{function apigroup="networkservice" visibility="private":
	{name: VidyoNetworkServiceCAInfoDestruct}
	{parent: VidyoNetworkServiceCAInfo}
	{description: Destructor for VidyoNetworkServiceCAInfo.}
	{prototype: void VidyoNetworkServiceCAInfoDestruct(VidyoNetworkServiceCAInfo* caInfo)}
	{parameter:
		{name: caInfo}
		{description: The VidyoNetworkServiceCAInfo object to destruct.}}
}
*/
void VidyoNetworkServiceCAInfoDestruct(VidyoNetworkServiceCAInfo* caInfo);

/**
{function apigroup="networkservice" visibility="private":
	{name: VidyoNetworkServiceRequestHeaderConstruct}
	{parent: VidyoNetworkServiceRequestHeader}
	{description: Construct an VidyoNetworkServiceRequestHeader object.}
	{prototype: VidyoNetworkServiceRequestHeader* VidyoNetworkServiceRequestHeaderConstruct(VidyoNetworkServiceRequestHeader* requestHeader,
		const LmiString *requestUrl, const LmiString *requestMethod, const LmiString *requestPayload, const LmiString *requestContentType,
		const LmiString *requestNumber, const LmiString *userAuthToken, const LmiString *userAuthUserName,
		const LmiString *userAuthUserPassword, const LmiString *webProxyUserName, const LmiString *webProxyUserPassword, LmiAllocator* alloc)}
	{parameter:
		{name: requestHeader}
		{description: The VidyoNetworkServiceRequestHeader object to construct.}}
	{parameter:
		{name: requestUrl}
		{description: Represents the url of the request.}}
	{parameter:
		{name: requestMethod}
		{description: Represents the type of the request.}}
	{parameter:
		{name: requestPayload}
		{description: Represents the payload of the request.}}
	{parameter:
		{name: requestContentType}
		{description: Represents the content/encoding type of the request.}}
	{parameter:
		{name: requestNumber}
		{description: Represents the request number of the request.}}
	{parameter:
		{name: userAuthToken}
		{description: Represents the authorization token of the request.}}
	{parameter:
		{name: userAuthUserName}
		{description: Represents the authorization user name of the request.}}
	{parameter:
		{name: userAuthUserPassword}
		{description: Represents the authorization user password of the request.}}
	{parameter:
		{name: webProxyUserName}
		{description: Represents the user name of the webproxy.}}
	{parameter:
		{name: webProxyUserPassword}
		{description: Represents the user password of the webproxy.}}
	{parameter:
		{name: alloc}
		{description: The LmiAllocator object.}}
	{return: VidyoNetworkServiceRequestHeader object on success otherwise NULL.}
}
*/
VidyoNetworkServiceRequestHeader* VidyoNetworkServiceRequestHeaderConstruct(VidyoNetworkServiceRequestHeader* requestHeader,
	const LmiString *requestUrl, const LmiString *requestMethod, const LmiString *requestPayload, const LmiString *requestContentType,
	const LmiString *requestNumber, const LmiString *userAuthToken, const LmiString *userAuthUserName,
	const LmiString *userAuthUserPassword, const LmiString *webProxyUserName, const LmiString *webProxyUserPassword, LmiAllocator* alloc);


/**
{function apigroup="networkservice" visibility="private":
	{name: VidyoNetworkServiceRequestHeaderConstructDefault}
	{parent: VidyoNetworkServiceRequestHeader}
	{description: Default construct for VidyoNetworkServiceRequestHeader.}
	{prototype: VidyoNetworkServiceRequestHeader* VidyoNetworkServiceRequestHeaderConstructDefault(VidyoNetworkServiceRequestHeader* requestHeader, LmiAllocator* alloc)}
	{parameter:
		{name: requestHeader}
		{description: The VidyoNetworkServiceRequestHeader object to construct.}}
	{parameter:
		{name: alloc}
		{description: The LmiAllocator object.}}
	{return: VidyoNetworkServiceRequestHeader object on success otherwise NULL.}
}
*/
VidyoNetworkServiceRequestHeader* VidyoNetworkServiceRequestHeaderConstructDefault(VidyoNetworkServiceRequestHeader* requestHeader, LmiAllocator* alloc);

/**
{function apigroup="networkservice" visibility="private":
	{name: VidyoNetworkServiceRequestHeaderConstructCopy}
	{parent: VidyoNetworkServiceRequestHeader}
	{description: Constructs an VidyoNetworkServiceRequestHeader object as a copy of an existing object.}
	{prototype: VidyoNetworkServiceRequestHeader* VidyoNetworkServiceRequestHeaderConstructCopy(VidyoNetworkServiceRequestHeader* d, const VidyoNetworkServiceRequestHeader* s)}
	{parameter:
		{name: d}
		{description: The VidyoNetworkServiceRequestHeader Object to construct.}}
	{parameter:
		{name: s}
		{description: The VidyoNetworkServiceRequestHeader object to make a copy of.}}
	{return: VidyoNetworkServiceRequestHeader object on success otherwise NULL.}
}
*/
VidyoNetworkServiceRequestHeader* VidyoNetworkServiceRequestHeaderConstructCopy(VidyoNetworkServiceRequestHeader* d, const VidyoNetworkServiceRequestHeader* s);

/**
{function apigroup="networkservice" visibility="private":
	{name: VidyoNetworkServiceRequestHeaderDestruct}
	{parent: VidyoNetworkServiceRequestHeader}
	{description: Destructs an VidyoNetworkServiceRequestHeader object.}
	{prototype: void VidyoNetworkServiceRequestHeaderDestruct(VidyoNetworkServiceRequestHeader* requestHeader)}
	{parameter:
		{name: requestHeader}
		{description: The VidyoNetworkServiceRequestHeader object to destruct.}}
}
*/
void VidyoNetworkServiceRequestHeaderDestruct(VidyoNetworkServiceRequestHeader* requestHeader);

/**
{function apigroup="networkservice" visibility="private":
	{name: VidyoNetworkServiceResponseHeaderConstruct}
	{parent: VidyoNetworkServiceResponseHeader}
	{description: Cconstruct an VidyoNetworkServiceResponseHeader object.}
	{prototype: VidyoNetworkServiceResponseHeader* VidyoNetworkServiceResponseHeaderConstruct(VidyoNetworkServiceResponseHeader* responseHeader,
		const LmiString *requestNumber, VidyoNetworkServiceErrorCode errorCode, LmiUint httpStatusCode, LmiUint httpResponseContentSize,
		const LmiString *httpResponseContent, const LmiString *httpResponseContentType, const LmiString *webProxyIPAddress,
		LmiUint webProxyIPPort, LmiAllocator* alloc)}
	{parameter:
		{name: responseHeader}
		{description: The VidyoNetworkServiceResponseHeader object to construct.}}
	{parameter:
		{name: requestNumber}
		{description: Represents the unique request number for each Http request.}}
	{parameter:
		{name: errorCode}
		{description: Represents the error code, see VidyoNetworkServiceErrorCode.}}
	{parameter:
		{name: httpStatusCode}
		{description: Represents the http status code of the response.}}
	{parameter:
		{name: httpResponseContentSize}
		{description: Represents the content size of the respons.}}
	{parameter:
		{name: httpResponseContent}
		{description: Represents the content of the response.}}
	{parameter:
		{name: httpResponseContentType}
		{description: Represents the content type of response.}}
	{parameter:
		{name: webProxyIPAddress}
		{description: Represents the discovered webproxy IP address.}}
	{parameter:
		{name: webProxyIPPort}
		{description: Represents the discovered webproxy IP address port.}}
	{parameter:
		{name: alloc}
		{description: The LmiAllocator object}}
	{return: VidyoNetworkServiceResponseHeader object on success otherwise NULL.}
}
*/
VidyoNetworkServiceResponseHeader* VidyoNetworkServiceResponseHeaderConstruct(VidyoNetworkServiceResponseHeader* responseHeader,
	const LmiString *requestNumber, VidyoNetworkServiceErrorCode errorCode, LmiUint httpStatusCode, LmiUint httpResponseContentSize,
	const LmiString *httpResponseContent, const LmiString *httpResponseContentType, const LmiString *webProxyIPAddress,
	LmiUint webProxyIPPort, LmiAllocator* alloc);

/**
{function apigroup="networkservice" visibility="private":
	{name: VidyoNetworkServiceResponseHeaderConstructDefault}
	{parent: VidyoNetworkServiceResponseHeader}
	{description: Default construct for VidyoNetworkServiceResponseHeader object.}
	{prototype: VidyoNetworkServiceResponseHeader* VidyoNetworkServiceResponseHeaderConstructDefault(VidyoNetworkServiceResponseHeader* responseHeader, LmiAllocator* alloc)}
	{parameter:
		{name: responseHeader}
		{description: The VidyoNetworkServiceResponseHeader object to construct.}}
	{parameter:
		{name: alloc}
		{description: The LmiAllocator object}}
	{return: VidyoNetworkServiceResponseHeader object on success otherwise NULL.}
}
*/
VidyoNetworkServiceResponseHeader* VidyoNetworkServiceResponseHeaderConstructDefault(VidyoNetworkServiceResponseHeader* responseHeader, LmiAllocator* alloc);

/**
{function apigroup="networkservice" visibility="private":
	{name: VidyoNetworkServiceResponseHeaderConstructCopy}
	{parent: VidyoNetworkServiceResponseHeader}
	{description: Constructs an VidyoNetworkServiceResponseHeader object as a copy of an existing object.}
	{prototype: VidyoNetworkServiceResponseHeader* VidyoNetworkServiceResponseHeaderConstructCopy(VidyoNetworkServiceResponseHeader* d, VidyoNetworkServiceResponseHeader* s)}
	{parameter:
		{name: d}
		{description: The VidyoNetworkServiceResponseHeader object to construct.}}
	{parameter:
		{name: s}
		{description: The VidyoNetworkServiceResponseHeader object to make a copy of.}}
	{return: VidyoNetworkServiceResponseHeader object on success otherwise NULL.}
}
*/
VidyoNetworkServiceResponseHeader* VidyoNetworkServiceResponseHeaderConstructCopy(VidyoNetworkServiceResponseHeader* d, VidyoNetworkServiceResponseHeader* s);

/**
{function apigroup="networkservice" visibility="private":
	{name: VidyoNetworkServiceResponseHeaderDestruct}
	{parent: VidyoNetworkServiceResponseHeader}
	{description: Destructs an VidyoNetworkServiceResponseHeader object.}
	{prototype: void VidyoNetworkServiceResponseHeaderDestruct(VidyoNetworkServiceResponseHeader* responseHeader)}
	{parameter:
		{name: responseHeader}
		{description: The VidyoNetworkServiceResponseHeader object to destruct.}}
}
*/
void VidyoNetworkServiceResponseHeaderDestruct(VidyoNetworkServiceResponseHeader* responseHeader);

/**
{function apigroup="networkservice":
	{name: VidyoNetworkServiceConstruct}
	{parent: VidyoNetworkService}
	{description: Constructs an NetworkService object.}
	{prototype: VidyoNetworkService* VidyoNetworkServiceConstruct(VidyoNetworkService* networkService, VidyoNetworkServiceCAInfo* caInfo)}
	{parameter:
		{name: networkService}
		{description: The VidyoNetworkService Object to construct.}}
	{parameter:
		{name: caInfo}
		{description: The VidyoNetworkServiceCAInfo object.}}
	{return: NetworkService object on success otherwise NULL.}
}
*/
VidyoNetworkService* VidyoNetworkServiceConstruct(VidyoNetworkService* networkService, VidyoNetworkServiceCAInfo* caInfo);

/**
{function apigroup="networkservice":
	{name: VidyoNetworkServiceConstructCopy}
	{parent: VidyoNetworkService}
	{description: Construct an VidyoNetworkService object as a copy of another.}
	{prototype: VidyoNetworkService *VidyoNetworkServiceConstructCopy(VidyoNetworkService* networkService, const VidyoNetworkService *other)}
	{parameter:
		{name: networkService} 
		{description: The VidyoNetworkService object to construct.}}
	{parameter: 
		{name: other}
		{description: The VidyoNetworkService object from which to construct a copy.}}
	{return: The constructed object on success otherwise NULL.}
}
*/

/**
{function apigroup="networkservice":
	{name: VidyoNetworkServiceDestruct}
	{parent: VidyoNetworkService}
	{description: Destructs an VidyoNetworkService object.}
	{prototype: void VidyoNetworkServiceDestruct(VidyoNetworkService* networkService)}
	{parameter: {name: networkService} {description: The VidyoNetworkService object to destruct.}}
}
*/

/**
{function apigroup="networkservice" visibility="private":
	{name: VidyoNetworkServiceSetUserData}
	{parent: VidyoNetworkService}
	{description: Associates user data with a VidyoNetworkService object, so that such data may be retrieved later.}
	{prototype: void VidyoNetworkServiceSetUserData(VidyoNetworkService* networkService, LmiVoidPtr userData)}
	{parameter:
		{name: networkService}
		{description: The VidyoNetworkService object.}}
	{parameter:
		{name: userData}
		{description: Arbitrary data.}}
}
*/
void VidyoNetworkServiceSetUserData(VidyoNetworkService* networkService, LmiVoidPtr userData);

/**
{function apigroup="networkservice" visibility="private":
	{name: VidyoNetworkServiceGetUserData}
	{parent: VidyoNetworkService}
	{description: Retreives any user data that was associated with a VidyoNetworkService object.}
	{prototype: LmiVoidPtr VidyoNetworkServiceGetUserData(const VidyoNetworkService* networkService)}
	{parameter:
		{name: networkService}
		{description: The VidyoNetworkService object.}}
	{return: User data on success. NULL on failure.}
}
*/
LmiVoidPtr VidyoNetworkServiceGetUserData(const VidyoNetworkService* networkService);

/**
 {function apigroup="networkservice":
	{name: VidyoNetworkServiceHttpRequestASync}
	{parent: VidyoNetworkService}
	{description: Http request function, this will be call by the application in order to placed any http request. This will be asynchronous call, Application call will not be blocked.}
	{prototype: VidyoNetworkServiceErrorCode VidyoNetworkServiceHttpRequestASync(VidyoNetworkService* networkService, const VidyoNetworkServiceRequestHeader* requestHeader, VidyoNetworkServiceResponseCB responseCB, LmiVoidPtr userData)}
	{parameter:
		{name: networkService}
		{description: The VidyoNetworkService object.}}
	{parameter:
		{name: requestHeader}
		{description: Http request data, user has to provide data for http request.}}
	{parameter:
		{name: responseCB}
		{description: Callback function through which application will get http response.}}
	{parameter:
		{name: userData}
		{description: Application specific data, same thing will be provided back to application through http response CallBack.}}
	{return: NetworkService error code.}
}
*/
VidyoNetworkServiceErrorCode VidyoNetworkServiceHttpRequestASync(VidyoNetworkService* networkService, const VidyoNetworkServiceRequestHeader* requestHeader, VidyoNetworkServiceResponseCB responseCB, LmiVoidPtr userData);

LMI_END_EXTERN_C

#if LMI_INLINE_NEED_HEADER_FILE_DEFINITIONS
#include "VidyoNetworkServiceInline.h"
#endif

#endif /* VIDYO_NETWORKSERVICE_H_ */
