/** 
{file:
	{name: VidyoParticipant.h}
	{description: Participant Entity Library. }
	{copyright:
		(c) 2013-2015 Vidyo, Inc.,
		433 Hackensack Avenue,
		Hackensack, NJ  07601.
		All rights reserved.
		The information contained herein is proprietary to Vidyo, Inc.
		and shall not be reproduced, copied (in whole or in part), adapted,
		modified, disseminated, transmitted, transcribed, stored in a retrieval
		system, or translated into any language in any form by any means
		without the express written consent of Vidyo, Inc.}
}
*/
#ifndef VIDYO_PARTICIPANT_H_ 
#define VIDYO_PARTICIPANT_H_ 

#include <Lmi/Os/LmiSharedPtr.h>
#include <Lmi/Utils/LmiVector.h>
#include <Lmi/Utils/LmiUnordered.h>
#include <Lmi/Utils/LmiString.h>

LMI_BEGIN_EXTERN_C

struct VidyoParticipantImpl_;
typedef struct VidyoParticipantImpl_ VidyoParticipantImpl;
Declare_LmiSharedPtr(VidyoParticipantImpl)
Declare_LmiSharedPtrWrapper(VidyoParticipant, VidyoParticipantImpl)
Declare_LmiWeakPtrWrapper(VidyoParticipantHandle, VidyoParticipant, VidyoParticipantImpl)
Declare_LmiUnorderedMap(LmiString, VidyoParticipant)
Declare_LmiVector(VidyoParticipant)
Declare_LmiVectorEqual(VidyoParticipant)
Declare_LmiList(VidyoParticipant)

/**
 {type visibility="public":
	{name: VidyoParticipantHandState}
	{parent: VidyoParticipant}
	{description: Structure that will contain a different hand state for a particular participant.}
	{value: {name: VIDYO_PARTICIPANTHANDSTATE_UNRAISED}
		{description: UnRaised State.}}
	{value: {name: VIDYO_PARTICIPANTHANDSTATE_RAISED}
		{description: when participate raise hand.}}
	{value: {name: VIDYO_PARTICIPANTHANDSTATE_APPROVED}
		{description: Participant request is approved.}}
	{value: {name: VIDYO_PARTICIPANTHANDSTATE_DISMISSED}
		{description: Participant request is dismissed.}}
}
*/
typedef enum
{
	VIDYO_PARTICIPANTHANDSTATE_UNRAISED,
	VIDYO_PARTICIPANTHANDSTATE_RAISED,
	VIDYO_PARTICIPANTHANDSTATE_APPROVED,
	VIDYO_PARTICIPANTHANDSTATE_DISMISSED 
}VidyoParticipantHandState;

LMI_END_EXTERN_C

#include "VidyoCall.h"
#include "VidyoRoom.h"
#include "VidyoContact.h"
#include "VidyoRemoteMicrophone.h"
#include "VidyoRemoteCamera.h"
#include "VidyoRemoteWindowShare.h"

LMI_BEGIN_EXTERN_C

/**
{type visibility="public":
	{name: VidyoParticipantTrust}
	{parent: VidyoParticipant}
	{description: This type indicates the trust status of a participant.}
	{value: {name: VIDYO_PARTICIPANTTRUST_Local}
		{description: Locally authenticated user.}}
	{value: {name: VIDYO_PARTICIPANTTRUST_Federated}
		{description: Federated from another trusted server.}}
	{value: {name: VIDYO_PARTICIPANTTRUST_Anonymous}
		{description: Untrusted guest user.}}
}
*/
typedef enum
{
	VIDYO_PARTICIPANTTRUST_Local,
	VIDYO_PARTICIPANTTRUST_Federated,
	VIDYO_PARTICIPANTTRUST_Anonymous,
} VidyoParticipantTrust;

/**
	{type visibility="public":
	{name: VidyoParticipantApplicationType}
	{parent: VidyoParticipant}
	{description: This type indicates the trust status of a participant.}
	{value: 
		{name: VIDYO_PARTICIPANT_APPLICATIONTYPE_None}
		{description: default application type.}}
	{value: 
		{name: VIDYO_PARTICIPANT_APPLICATIONTYPE_Recorder}
		{description: Recorder application type}}
	{value: 
		{name: VIDYO_PARTICIPANT_APPLICATIONTYPE_Gateway}
		{description: Gateway application type}}
	{value:
		{name: VIDYO_PARTICIPANT_APPLICATIONTYPE_Bot}
		{description: Bot application type}}
}
*/
typedef enum
{
	VIDYO_PARTICIPANT_APPLICATIONTYPE_None,
	VIDYO_PARTICIPANT_APPLICATIONTYPE_Recorder,
	VIDYO_PARTICIPANT_APPLICATIONTYPE_Gateway,
	VIDYO_PARTICIPANT_APPLICATIONTYPE_Bot,
} VidyoParticipantApplicationType;

/**
 {type visibility="public":
	{name: VidyoParticipantGetContactResult}
	{parent: VidyoParticipant}
	{description: Result of getting the contact.}
	{value: {name: VIDYO_PARTICIPANTGETCONTACTRESULT_SUCCESS}
		{description: Contact was got successfully.}}
	{value: {name: VIDYO_PARTICIPANTGETCONTACTRESULT_MiscError}
		{description: Getting the contact failed some miscellaneous reason.}}
}
*/
typedef enum
{
	VIDYO_PARTICIPANTGETCONTACTRESULT_SUCCESS,
	VIDYO_PARTICIPANTGETCONTACTRESULT_MiscError
} VidyoParticipantGetContactResult;

/**
	{type visibility="public":
	{name: VidyoParticipantClearanceType}
	{parent: VidyoParticipant}
	{description: Clearance of an participant. This is used to determine what the participant is allowed to do.}
	{value:
		{name: VIDYO_PARTICIPANT_CLEARANCETYPE_None}
		{description: Guest participant with no privileges.}
	}
	{value:
		{name: VIDYO_PARTICIPANT_CLEARANCETYPE_Member}
		{description: Normal participant with normal privileges.}
	}
	{value:
		{name: VIDYO_PARTICIPANT_CLEARANCETYPE_Owner}
		{description: Owner of the conference room, has full privileges.}
	}
	{value:
		{name: VIDYO_PARTICIPANT_CLEARANCETYPE_Admin}
		{description: Administrator of the conference room, has full privileges except it can not promote other participants to Admin clearance.}
	}
	{value:
		{name: VIDYO_PARTICIPANT_CLEARANCETYPE_Moderator}
		{description: Moderator of the conference, has ability to control all aspects of the conference.}
	}
}
*/
typedef enum
{
	VIDYO_PARTICIPANT_CLEARANCETYPE_None,
	VIDYO_PARTICIPANT_CLEARANCETYPE_Member,
	VIDYO_PARTICIPANT_CLEARANCETYPE_Owner,
	VIDYO_PARTICIPANT_CLEARANCETYPE_Admin,
	VIDYO_PARTICIPANT_CLEARANCETYPE_Moderator,
}VidyoParticipantClearanceType;

/**
 {callback:
	{name: VidyoParticipantOnGetContactComplete}
	{parent: VidyoParticipant}
	{description: Callback that is triggered when getting the contact completes.}
	{prototype: void (*VidyoParticipantOnGetContactComplete)(VidyoParticipant* p, const VidyoContact* contact, VidyoParticipantGetContactResult result)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{parameter:
		{name: contact}
		{description: The VidyoContact object.}
	}
	{parameter:
		{name: result}
		{description: Result of getting the contact.}
	}
}
*/
typedef void (*VidyoParticipantOnGetContactComplete)(VidyoParticipant* p, const VidyoContact* contact, VidyoParticipantGetContactResult result);

/**
{type apigroup="connector,simple" reference-counted="yes":
	{name: VidyoParticipant}
	{parent: Endpoint}
	{include: Lmi/VidyoClient/VidyoParticipant.h}
	{description: This object represents a participant in the room or call.}
	{member:
		{name: name}
		{type: LmiString}
		{description: Display name of the participant.}
	}
	{member:
		{name: id}
		{type: LmiString}
		{description: ID that is globally unique such as user@domain.}
	}
    {member deprecated="yes":
        {name: userId}
        {type: LmiString}
        {description: ID that is globally unique such as user@domain.}
    }
	{member:
		{name: scipId}
		{type: LmiString}
		{description: SCIP ID unique to the room or call only.}
	}
	{member:
		{name: trust}
		{type: VidyoParticipantTrust}
		{description: }
	}
}
*/

/**
{type apigroup="connector,simple" visibility="private":
	{name: VidyoParticipantHandle}
	{parent: VidyoParticipant}
	{description: }
}
*/

/**
{function visibility="private":
	{name: VidyoParticipantInitialize}
	{parent: VidyoParticipant}
	{description: }
	{prototype: LmiBool VidyoParticipantInitialize(void)}
	{return: }
}
*/
LmiBool VidyoParticipantInitialize(void);


/**
{function visibility="private":
	{name: VidyoParticipantUninitialize}
	{parent: VidyoParticipant}
	{description: }
	{prototype: void VidyoParticipantUninitialize(void)}
}
*/
void VidyoParticipantUninitialize(void);

/**
{function apigroup="connector,simple":
	{name: VidyoParticipantConstructCopy}
	{parent: VidyoParticipant}
	{description: Constructs an VidyoParticipant object as a copy of another.}
	{prototype: VidyoParticipant *VidyoParticipantConstructCopy(VidyoParticipant *obj, const VidyoParticipant *other)}
	{parameter: {name: obj} {description: The VidyoParticipant object to construct.}}
	{parameter: {name: other} {description: The call object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/

/**
{function apigroup="connector,simple":
	{name: VidyoParticipantDestruct}
	{parent: VidyoParticipant}
	{description: Destructs an VidyoParticipant object.}
	{prototype: void VidyoParticipantDestruct(VidyoParticipant *obj)}
	{parameter: {name: obj} {description: The VidyoParticipant object to destruct.}}
}
*/

/**
{function apigroup="connector,simple" visibility="private":
	{name: VidyoParticipantAssign}
	{parent: VidyoParticipant}
	{description: Assigns one VidyoParticipant object the value of another. }
	{prototype: VidyoParticipant *VidyoParticipantAssign(VidyoParticipant *obj, const VidyoParticipant *other)}
	{parameter: {name: obj} {description: The VidyoParticipant object.}}
	{parameter: {name: other} {description: The call object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/

/**
{function apigroup="connector,simple":
	{name: VidyoParticipantEqual}
	{parent: VidyoParticipant}
	{description: Compare two VidyoParticipant objects for equality.}
	{prototype: LmiBool VidyoParticipantEqual(const VidyoParticipant* a, const VidyoParticipant* b)}
	{parameter:
	{name: a}
	{description: A pointer to the first VidyoParticipant object
	to be compared.}}
	{parameter:
	{name: b}
	{description: A pointer to the second VidyoParticipant object
	to be compared.}}
	{return: LMI_TRUE if the objects pointed to are equal, otherwise
	LMI_FALSE.}
}
*/

/**
{function apigroup="connector,simple" visibility="private":
	{name: VidyoParticipantGetHandle}
	{parent: VidyoParticipant}
	{description: }
	{prototype: VidyoParticipantHandle* VidyoParticipantGetHandle(VidyoParticipant* p)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{return: }
}
*/
VidyoParticipantHandle* VidyoParticipantGetHandle(VidyoParticipant* p);

/**
{function apigroup="connector,simple":
	{name: VidyoParticipantGetTrust}
	{parent: VidyoParticipant}
	{description: }
	{prototype: VidyoParticipantTrust VidyoParticipantGetTrust(const VidyoParticipant* p)}
	{parameter:
		{name: p}
		{description: }
	}
	{return: }
}
*/
VidyoParticipantTrust VidyoParticipantGetTrust(const VidyoParticipant* p);

/**
{function apigroup="connector,simple":
	{name: VidyoParticipantGetName}
	{parent: VidyoParticipant}
	{description: Gets the display name of the participant.}
	{prototype: const LmiString* VidyoParticipantGetName(const VidyoParticipant* p)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{return: Display name of the participant. }
}
*/
const LmiString* VidyoParticipantGetName(const VidyoParticipant* p);

/**
{function apigroup="connector,simple":
	{name: VidyoParticipantGetId}
	{parent: VidyoParticipant}
    {description: Gets the ID that is globally unique such as user@domain. }
	{prototype: const LmiString* VidyoParticipantGetId(const VidyoParticipant* p)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{return: ID that is unique to the room or call only.}
}
*/
const LmiString* VidyoParticipantGetId(const VidyoParticipant* p);

/**
{function apigroup="connector,simple" deprecated="yes":
	{name: VidyoParticipantGetUserId}
	{parent: VidyoParticipant}
	{description: Gets the ID that is globally unique such as user@domain. }
	{prototype: const LmiString* VidyoParticipantGetUserId(const VidyoParticipant* p)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{return: ID that is globally unique such as user@domain.}
}
*/
const LmiString* VidyoParticipantGetUserId(const VidyoParticipant* p);

/**
{function apigroup="connector,simple":
    {name: VidyoParticipantGetScipId}
    {parent: VidyoParticipant}
    {description: Gets the SCIP ID unique to the room or call only.}
    {prototype: const LmiString* VidyoParticipantGetScipId(const VidyoParticipant* p)}
    {parameter:
        {name: p}
        {description: The VidyoParticipant object.}
    }
    {return: SCIP ID unique to the room or call only.}
}
*/
const LmiString* VidyoParticipantGetScipId(const VidyoParticipant* p);

/**
{function  deprecated="yes" replaced-by="VidyoParticipantGetContactAsync":
	{name: VidyoParticipantGetContact}
	{parent: VidyoParticipant}
	{description: }
	{prototype: VidyoContact* VidyoParticipantGetContact(const VidyoParticipant* p, VidyoContact* contact)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{parameter:
		{name: contact}
		{description: }
	}
	{return: }
}
*/
VidyoContact* VidyoParticipantGetContact(const VidyoParticipant* p, VidyoContact* contact);

/**
{function:
	{name: VidyoParticipantGetContactAsync}
	{parent: VidyoParticipant}
	{description: Gets conntact.}
	{prototype: LmiBool VidyoParticipantGetContactAsync(VidyoParticipant* p, VidyoParticipantOnGetContactComplete onComplete)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{parameter:
		{name: onComplete}
		{description: The callback that is triggered when getting contact completes.}
	}
	{return: }
}
*/
LmiBool VidyoParticipantGetContactAsync(VidyoParticipant* p, VidyoParticipantOnGetContactComplete onComplete);

/**
{function apigroup="connector,simple" visibility="private":
	{name: VidyoParticipantSetUserData}
	{parent: VidyoParticipant}
	{description: }
	{prototype: void VidyoParticipantSetUserData(VidyoParticipant* p, LmiVoidPtr userData)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{parameter:
		{name: userData}
		{description: }
	}
}
*/
void VidyoParticipantSetUserData(VidyoParticipant* p, LmiVoidPtr userData);

/**
{function:
	{name: VidyoParticipantIsLocal}
	{parent: VidyoParticipant}
	{description: }
	{prototype: LmiBool VidyoParticipantIsLocal(const VidyoParticipant *p)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{return: }
}
*/
LmiBool VidyoParticipantIsLocal(const VidyoParticipant *p);

/**
{function apigroup="connector,simple" visibility="private":
	{name: VidyoParticipantGetUserData}
	{parent: VidyoParticipant}
	{description: }
	{prototype: LmiVoidPtr VidyoParticipantGetUserData(const VidyoParticipant* p)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{return: }
}
*/
LmiVoidPtr VidyoParticipantGetUserData(const VidyoParticipant* p);

/**
{function apigroup="connector,simple" visibility="private":
	{name: VidyoParticipantGetRemoteCameras}
	{parent: VidyoParticipant}
	{description: }
	{prototype: LmiBool VidyoParticipantGetRemoteCameras(VidyoParticipant *p, LmiVector(VidyoRemoteCamera) *cameras)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{parameter:
		{name: cameras}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoParticipantGetRemoteCameras(VidyoParticipant *p, LmiVector(VidyoRemoteCamera) *cameras);

/**
{function apigroup="connector,simple" visibility="private":
	{name: VidyoParticipantHasRemoteCameras}
	{parent: VidyoParticipant}
	{description: }
	{prototype: LmiBool VidyoParticipantHasRemoteCameras(VidyoParticipant *p)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{return: }
}
*/
LmiBool VidyoParticipantHasRemoteCameras(VidyoParticipant *p);

/**
{function apigroup="connector,simple" visibility="private":
	{name: VidyoParticipantGetRemoteMicrophones}
	{parent: VidyoParticipant}
	{description: }
	{prototype: LmiBool VidyoParticipantGetRemoteMicrophones(VidyoParticipant *p, LmiVector(VidyoRemoteMicrophone) *microphones)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{parameter:
		{name: microphones}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoParticipantGetRemoteMicrophones(VidyoParticipant *p, LmiVector(VidyoRemoteMicrophone) *microphones);

/**
{function apigroup="connector,simple" visibility="private":
	{name: VidyoParticipantGetRemoteWindowShares}
	{parent: VidyoParticipant}
	{description: }
	{prototype: LmiBool VidyoParticipantGetRemoteWindowShares(VidyoParticipant *p, LmiVector(VidyoRemoteWindowShare) *windowShares)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{parameter:
		{name: windowShares}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoParticipantGetRemoteWindowShares(VidyoParticipant *p, LmiVector(VidyoRemoteWindowShare) *windowShares);

/**
{function visibility="private":
	{name: VidyoParticipantGetCall}
	{parent: VidyoParticipant}
	{description: }
	{prototype: VidyoCall *VidyoParticipantGetCall(VidyoParticipant *p)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{return: }
}
*/
VidyoCall *VidyoParticipantGetCall(VidyoParticipant *p);

/**
{function visibility="private":
	{name: VidyoParticipantGetRoomHandle}
	{parent: VidyoParticipant}
	{description: }
	{prototype: VidyoRoomHandle *VidyoParticipantGetRoomHandle(VidyoParticipant *p)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{return: }
}
*/
VidyoRoomHandle *VidyoParticipantGetRoomHandle(VidyoParticipant *p);

/**
{function visibility="private":
	{name: VidyoParticipantGetStats}
	{parent: VidyoParticipant}
	{description: Fill in data in VidyoParticipantGetStats from a participant.}
	{prototype: LmiBool VidyoParticipantGetStats(VidyoParticipant *p, VidyoParticipantStats *stats)}
	{parameter: {name: p} {description: The participant object to extract stats.}}
	{parameter: {name: stats} {description: The stats object to fill.}}
	{return: LMI_TRUE on success.}
}
*/
LmiBool VidyoParticipantGetStats(VidyoParticipant *p, VidyoParticipantStats *stats);

/**
{function apigroup="connector,simple":
	{name: VidyoParticipantIsRecording}
	{parent: VidyoParticipant}
	{description: Gets the recording state of the participant.}
	{prototype: LmiBool VidyoParticipantIsRecording(const VidyoParticipant* p)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{return: LMI_TRUE if the participant is recording, LMI_FALSE otherwise. }
}
*/
LmiBool VidyoParticipantIsRecording(const VidyoParticipant* p);

/**
{function apigroup="connector,simple":
	{name: VidyoParticipantIsHidden}
	{parent: VidyoParticipant}
	{description: Gets the hidden state of the participant.}
	{prototype: LmiBool VidyoParticipantIsHidden(const VidyoParticipant* p)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{return: LMI_TRUE if the participant is hidden, LMI_FALSE otherwise. }
}
*/
LmiBool VidyoParticipantIsHidden(const VidyoParticipant* p);

/**
{function apigroup="connector,simple":
	{name: VidyoParticipantIsSelectable}
	{parent: VidyoParticipant}
	{description: Gets the selectable state of the participant.}
	{prototype: LmiBool VidyoParticipantIsSelectable(const VidyoParticipant* p)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{return: LMI_TRUE if the participant is selectable, LMI_FALSE otherwise. }
}
*/
LmiBool VidyoParticipantIsSelectable(const VidyoParticipant* p);

/**
 {function apigroup="connector,simple" visibility="public":
 {name: VidyoParticipantGetApplicationType}
 {parent: VidyoParticipant}
 {description: }
 {prototype: VidyoParticipantApplicationType VidyoParticipantGetApplicationType(const VidyoParticipant* p)}
 {parameter:
 {name: p}
 {description: The VidyoParticipant object.}
 }
 {return: one of the enum value from VidyoParticipantApplicationType}
 }
 */
 VidyoParticipantApplicationType VidyoParticipantGetApplicationType(const VidyoParticipant* p);

/**
{function apigroup="connector, simple":
	{name: VidyoParticipantGetClearanceType}
	{parent: VidyoParticipant}
	{description: Gets the clearance type of the participant.}
	{prototype: VidyoParticipantClearanceType VidyoParticipantGetClearanceType(const VidyoParticipant* p)}
	{parameter:
		{name: p}
		{description: The VidyoParticipant object.}
	}
	{return: Clearance of participant. See VidyoParticipantClearanceType. }
}
*/
VidyoParticipantClearanceType VidyoParticipantGetClearanceType(const VidyoParticipant *p);

LMI_END_EXTERN_C

#if LMI_INLINE_NEED_HEADER_FILE_DEFINITIONS
#include "VidyoParticipantInline.h"
#endif

#endif /* VIDYO_PARTICIPANT_H_ */

