#ifndef SURFACE_H
#define SURFACE_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stdbool.h>

#include "NgrGeneralUtils.h"
#include "NgrGestureEvent.h"
#include "NgrTexturedRect.h"
#include "NgrTile.h"
#include "NgrTileSet.h"

typedef struct {
	uintptr_t value;
} NgrSurfaceHandle;

typedef void (*NgrSurfaceReleaseCallback) (const void* userData, const char* error);
typedef void (*NgrSurfaceSetDensityCallback) (const void* userData, const char* error);
typedef void (*NgrSurfaceSetBackgroundColorCallback) (const void* userData, const char* error);
typedef void (*NgrSurfaceSetMaxFpsCallback) (const void* userData, const char* error);
typedef void (*NgrSurfaceSetDebugInfoVisibleCallback) (const void* userData, const char* error);
typedef void (*NgrSurfaceCreateTexturedRectCallback) (const void* userData, NgrTexturedRectHandle* value, const char* error);
typedef void (*NgrSurfaceCreateTileCallback) (const void* userData, NgrTileHandle* value, const char* error);
typedef void (*NgrSurfaceCreateTileSetCallback) (const void* userData, NgrTileSetHandle* value, const char* error);
typedef void (*NgrSurfaceHandleGestureEventCallback) (const void* userData, const char* error);

void NgrSurfaceAcquireRef(NgrSurfaceHandle handle);
void NgrSurfaceReleaseRef(NgrSurfaceHandle handle);

/**
 * Release surface and all its child objects
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrSurfaceRelease(
	NgrSurfaceHandle handle,
	const NgrUserData callback_userData,
	NgrSurfaceReleaseCallback callback
);

/**
 * Put surface density
 *
 * It is used for calculating density-aware pixel sizes:
 * 1 dp * 0.5 density = 0.5 px (everything will be smaller)
 * 1 dp * 1.0 density = 1.0 px
 * 1 dp * 2.0 density = 2.0 px (everything will be larger)
 *
 * # Arguments
 * * `density` - Density value
 * * `callback` - error if occurred
 */
void NgrSurfaceSetDensity(
	NgrSurfaceHandle handle,
	float density,
	const NgrUserData callback_userData,
	NgrSurfaceSetDensityCallback callback
);

/**
 * Put surface background color
 *
 * # Arguments
 * * `color` - RGBA Background color
 * * `callback` - error if occurred
 */
void NgrSurfaceSetBackgroundColor(
	NgrSurfaceHandle handle,
	uint32_t color,
	const NgrUserData callback_userData,
	NgrSurfaceSetBackgroundColorCallback callback
);

/**
 * Put surface max frames-per-second update rate
 *
 * Surface will redraw itself lazily only when something is changed but not faster than this limit
 *
 * # Arguments
 * * `fps` - frames-per-second (cannot be 0)
 * * `callback` - error if occurred
 */
void NgrSurfaceSetMaxFps(
	NgrSurfaceHandle handle,
	uint32_t fps,
	const NgrUserData callback_userData,
	NgrSurfaceSetMaxFpsCallback callback
);

/**
 * Display debug stats on the surface
 *
 * # Arguments
 * * `visible` - Visibility of the debug stats
 * * `callback` - error if occurred
 */
void NgrSurfaceSetDebugInfoVisible(
	NgrSurfaceHandle handle,
	bool visible,
	const NgrUserData callback_userData,
	NgrSurfaceSetDebugInfoVisibleCallback callback
);

/**
 * Create rect that displays a texture
 *
 * # Arguments
 * * `callback` - TexturedRect or error otherwise
 */
void NgrSurfaceCreateTexturedRect(
	NgrSurfaceHandle handle,
	const NgrUserData callback_userData,
	NgrSurfaceCreateTexturedRectCallback callback
);

/**
 * Create tile that will display participant's info and video stream
 *
 * # Arguments
 * * `callback` - Tile or error otherwise
 */
void NgrSurfaceCreateTile(
	NgrSurfaceHandle handle,
	const NgrUserData callback_userData,
	NgrSurfaceCreateTileCallback callback
);

/**
 * Create tile set that will automatically manage tiles
 *
 * # Arguments
 * * `callback` - TileSet or error otherwise
 */
void NgrSurfaceCreateTileSet(
	NgrSurfaceHandle handle,
	const NgrUserData callback_userData,
	NgrSurfaceCreateTileSetCallback callback
);

/**
 * Handle gesture events
 *
 * # Arguments
 * * `event` - Gesture event
 * * `callback` - error if occurred
 */
void NgrSurfaceHandleGestureEvent(
	NgrSurfaceHandle handle,
	NgrGestureEvent event,
	const NgrUserData callback_userData,
	NgrSurfaceHandleGestureEventCallback callback
);


#ifdef __cplusplus
}
#endif

#endif /* SURFACE_H */
