#ifndef TEXT_TEXTURE_H
#define TEXT_TEXTURE_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stdbool.h>

#include "NgrGeneralUtils.h"
#include "NgrAlignment.h"
#include "NgrDimension.h"

typedef struct {
	uintptr_t value;
} NgrTextTextureHandle;

typedef void (*NgrTextTextureReleaseCallback) (const void* userData, const char* error);
typedef void (*NgrTextTextureSetAlphaMaskCallback) (const void* userData, const char* error);
typedef void (*NgrTextTextureSetTextCallback) (const void* userData, const char* error);
typedef void (*NgrTextTextureSetTextSizeCallback) (const void* userData, const char* error);
typedef void (*NgrTextTextureSetTextColorCallback) (const void* userData, const char* error);
typedef void (*NgrTextTextureSetFontCallback) (const void* userData, const char* error);
typedef void (*NgrTextTextureSetMaxWidthCallback) (const void* userData, const char* error);
typedef void (*NgrTextTextureSetPaddingsCallback) (const void* userData, const char* error);
typedef void (*NgrTextTextureSetAlignmentCallback) (const void* userData, const char* error);
typedef void (*NgrTextTextureSetShadowCallback) (const void* userData, const char* error);

void NgrTextTextureAcquireRef(NgrTextTextureHandle handle);
void NgrTextTextureReleaseRef(NgrTextTextureHandle handle);

/**
 * Release texture
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrTextTextureRelease(
	NgrTextTextureHandle handle,
	const NgrUserData callback_userData,
	NgrTextTextureReleaseCallback callback
);

/**
 * Put texture alpha mask
 *
 * # Arguments
 * * `value` - value of alpha mask
 * * `callback` - error if occurred
 */
void NgrTextTextureSetAlphaMask(
	NgrTextTextureHandle handle,
	bool value,
	const NgrUserData callback_userData,
	NgrTextTextureSetAlphaMaskCallback callback
);

/**
 * Put texture text
 *
 * # Arguments
 * * `value` - text value
 * * `callback` - error if occurred
 */
void NgrTextTextureSetText(
	NgrTextTextureHandle handle,
	const char* value,
	const NgrUserData callback_userData,
	NgrTextTextureSetTextCallback callback
);

/**
 * Put texture text size
 *
 * # Arguments
 * * `value` - text size value
 * * `callback` - error if occurred
 */
void NgrTextTextureSetTextSize(
	NgrTextTextureHandle handle,
	NgrDimension value,
	const NgrUserData callback_userData,
	NgrTextTextureSetTextSizeCallback callback
);

/**
 * Put texture text color
 *
 * Examples:
 * 0xbbggrraa - 4 bytes data
 * aa denotes the value of alpha
 * rr denotes the value of red
 * gg denotes the value of green
 * bb denotes the value of blue
 *
 * # Arguments
 * * `value` - text color value
 * * `callback` - error if occurred
 */
void NgrTextTextureSetTextColor(
	NgrTextTextureHandle handle,
	uint32_t value,
	const NgrUserData callback_userData,
	NgrTextTextureSetTextColorCallback callback
);

/**
 * Put texture font
 *
 * # Arguments
 * * `value` - texture font value in string
 * * `callback` - error if occurred
 */
void NgrTextTextureSetFont(
	NgrTextTextureHandle handle,
	const char* value,
	const NgrUserData callback_userData,
	NgrTextTextureSetFontCallback callback
);

/**
 * Put texture max width
 *
 * # Arguments
 * * `value` - texture dimension value
 * * `callback` - error if occurred
 */
void NgrTextTextureSetMaxWidth(
	NgrTextTextureHandle handle,
	NgrDimension value,
	const NgrUserData callback_userData,
	NgrTextTextureSetMaxWidthCallback callback
);

/**
 * Put texture paddings
 *
 * # Arguments
 * * `left` - left padding value
 * * `top` - top padding value
 * * `right` - right padding value
 * * `bottom` - bottom padding value
 * * `callback` - error if occurred
 */
void NgrTextTextureSetPaddings(
	NgrTextTextureHandle handle,
	NgrDimension left,
	NgrDimension top,
	NgrDimension right,
	NgrDimension bottom,
	const NgrUserData callback_userData,
	NgrTextTextureSetPaddingsCallback callback
);

/**
 * Put texture alignment
 *
 * # Arguments
 * * `value` - Alignment value
 * * `callback` - error if occurred
 */
void NgrTextTextureSetAlignment(
	NgrTextTextureHandle handle,
	NgrAlignment value,
	const NgrUserData callback_userData,
	NgrTextTextureSetAlignmentCallback callback
);

/**
 * Put texture shadow
 *
 * # Arguments
 * * `color` - Shadow color value
 * * `radius` - Shadow radius value
 * * `offset_x` - Shadow offset X value
 * * `offset_y` - Shadow offset Y value
 * * `callback` - error if occurred
 */
void NgrTextTextureSetShadow(
	NgrTextTextureHandle handle,
	uint32_t color,
	float radius,
	NgrDimension offsetX,
	NgrDimension offsetY,
	const NgrUserData callback_userData,
	NgrTextTextureSetShadowCallback callback
);


#ifdef __cplusplus
}
#endif

#endif /* TEXT_TEXTURE_H */
