#ifndef TILE_H
#define TILE_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stdbool.h>

#include "NgrGeneralUtils.h"
#include "NgrFECCPlacement.h"
#include "NgrFECCType.h"
#include "NgrPosition.h"
#include "NgrStreamKind.h"

typedef struct {
	uintptr_t value;
} NgrTileHandle;

typedef void (*NgrTileReleaseCallback) (const void* userData, const char* error);
typedef void (*NgrTileSetPositionCallback) (const void* userData, const char* error);
typedef void (*NgrTileSetDebugInfoVisibleCallback) (const void* userData, const char* error);
typedef void (*NgrTileSetLocalParticipantCallback) (const void* userData, const char* error);
typedef void (*NgrTileSetParticipantIdCallback) (const void* userData, const char* error);
typedef void (*NgrTileSetLocalVideoStreamCallback) (const void* userData, const char* error);
typedef void (*NgrTileSetVideoStreamDeviceIdCallback) (const void* userData, const char* error);
typedef void (*NgrTileSetVideoStreamParticipantIdCallback) (const void* userData, const char* error);
typedef void (*NgrTileSetVideoStreamPinnedCallback) (const void* userData, const char* error);
typedef void (*NgrTilePutFeccIconVisibleCallback) (const void* userData, const char* error);
typedef void (*NgrTilePutFeccControlVisibleCallback) (const void* userData, const char* error);
typedef void (*NgrTilePutFeccIconPlacementCallback) (const void* userData, const char* error);
typedef void (*NgrTileEnableFeccExtendedControlCallback) (const void* userData, const char* error);
typedef void (*NgrTilePutCustomFeccSvgCallback) (const void* userData, const char* error);

void NgrTileAcquireRef(NgrTileHandle handle);
void NgrTileReleaseRef(NgrTileHandle handle);

/**
 * Release tile
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrTileRelease(
	NgrTileHandle handle,
	const NgrUserData callback_userData,
	NgrTileReleaseCallback callback
);

/**
 * Put tile position relative to its parent
 *
 * # Arguments
 * * `position` - Position to use
 * * `callback` - error if occurred
 */
void NgrTileSetPosition(
	NgrTileHandle handle,
	NgrPosition position,
	const NgrUserData callback_userData,
	NgrTileSetPositionCallback callback
);

/**
 * Put tile debug info visibility
 *
 * # Arguments
 * * `visible` - Debug info visibility
 * * `callback` - error if occurred
 */
void NgrTileSetDebugInfoVisible(
	NgrTileHandle handle,
	bool visible,
	const NgrUserData callback_userData,
	NgrTileSetDebugInfoVisibleCallback callback
);

/**
 * Use local (current user) participant for displaying name and avatar
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrTileSetLocalParticipant(
	NgrTileHandle handle,
	const NgrUserData callback_userData,
	NgrTileSetLocalParticipantCallback callback
);

/**
 * Use participant with provided `participant_id` for displaying name and avatar
 *
 * # Arguments
 * * `participant_id` - Participant ID
 * * `callback` - error if occurred
 */
void NgrTileSetParticipantId(
	NgrTileHandle handle,
	const char* participantId,
	const NgrUserData callback_userData,
	NgrTileSetParticipantIdCallback callback
);

/**
 * Display video stream from local (current user) participant with specified `kind`
 *
 * # Arguments
 * * `kind` - Kind of video stream to display
 * * `callback` - error if occurred
 */
void NgrTileSetLocalVideoStream(
	NgrTileHandle handle,
	NgrStreamKind kind,
	const NgrUserData callback_userData,
	NgrTileSetLocalVideoStreamCallback callback
);

/**
 * Display video stream by `device_id`
 *
 * # Arguments
 * * `device_id` - Video stream device ID
 * * `callback` - error if occurred
 */
void NgrTileSetVideoStreamDeviceId(
	NgrTileHandle handle,
	const char* deviceId,
	const NgrUserData callback_userData,
	NgrTileSetVideoStreamDeviceIdCallback callback
);

/**
 * Display video stream for participant with `participant_id` and specified `kind`
 *
 * # Arguments
 * * `participant_id` - Video stream participant ID
 * * `kind` - Kind of video stream to display
 * * `callback` - error if occurred
 */
void NgrTileSetVideoStreamParticipantId(
	NgrTileHandle handle,
	const char* participantId,
	NgrStreamKind kind,
	const NgrUserData callback_userData,
	NgrTileSetVideoStreamParticipantIdCallback callback
);

/**
 * Put video stream pinned status
 *
 * NOTE: This will convert stream from dynamic to static
 *
 * # Arguments
 * * `pinned` - Pinned value
 * * `callback` - error if occurred
 */
void NgrTileSetVideoStreamPinned(
	NgrTileHandle handle,
	bool pinned,
	const NgrUserData callback_userData,
	NgrTileSetVideoStreamPinnedCallback callback
);

/**
 * Put FECC icon visibility
 *
 * # Arguments
 * * `visible` - FECC icon visibility
 * * `callback` - error if occurred
 */
void NgrTilePutFeccIconVisible(
	NgrTileHandle handle,
	bool visible,
	const NgrUserData callback_userData,
	NgrTilePutFeccIconVisibleCallback callback
);

/**
 * Put FECC control visibility
 *
 * # Arguments
 * * `visible` - FECC control visibility
 * * `callback` - error if occurred
 */
void NgrTilePutFeccControlVisible(
	NgrTileHandle handle,
	bool visible,
	const NgrUserData callback_userData,
	NgrTilePutFeccControlVisibleCallback callback
);

/**
 * Put FECC icon placement
 *
 * # Arguments
 * * `placement` - FECC icon placement
 * * `callback` - error if occurred
 */
void NgrTilePutFeccIconPlacement(
	NgrTileHandle handle,
	NgrFECCPlacement placement,
	const NgrUserData callback_userData,
	NgrTilePutFeccIconPlacementCallback callback
);

/**
 * Enable/Disable FECC extended control. Disabled by default
 *
 * # Arguments
 * * `enable` - FECC extended control enable
 * * `callback` - error if occurred
 */
void NgrTileEnableFeccExtendedControl(
	NgrTileHandle handle,
	bool enable,
	const NgrUserData callback_userData,
	NgrTileEnableFeccExtendedControlCallback callback
);

/**
 * Put SVG for the specific FECC texture
 *
 * # Arguments
 * * `fecc_type` - Fecc type
 * * `svg` - SVG of the FECC extended control down arrow texture
 * * `callback` - error if occurred
 */
void NgrTilePutCustomFeccSvg(
	NgrTileHandle handle,
	NgrFECCType feccType,
	const char* svg,
	const NgrUserData callback_userData,
	NgrTilePutCustomFeccSvgCallback callback
);


#ifdef __cplusplus
}
#endif

#endif /* TILE_H */
