#ifndef GRID_LAYOUT_H
#define GRID_LAYOUT_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stdbool.h>

#include "NgrGeneralUtils.h"
#include "NgrPreviewPosition.h"
#include "NgrTileStyle.h"

typedef struct {
	uintptr_t value;
} NgrGridLayoutHandle;

typedef void (*NgrGridLayoutReleaseCallback) (const void* userData, const char* error);
typedef void (*NgrGridLayoutActivateCallback) (const void* userData, const char* error);
typedef void (*NgrGridLayoutGetDefaultTileStyleCallback) (const void* userData, NgrTileStyleHandle* value, const char* error);
typedef void (*NgrGridLayoutGetShareTileStyleCallback) (const void* userData, NgrTileStyleHandle* value, const char* error);
typedef void (*NgrGridLayoutGetSelfTileStyleCallback) (const void* userData, NgrTileStyleHandle* value, const char* error);
typedef void (*NgrGridLayoutSetSelfPreviewVisibleCallback) (const void* userData, const char* error);
typedef void (*NgrGridLayoutSetMaxTilesCountCallback) (const void* userData, const char* error);
typedef void (*NgrGridLayoutSetPreferredTileAspectRatioCallback) (const void* userData, const char* error);
typedef void (*NgrGridLayoutSetPreviewPositionCallback) (const void* userData, const char* error);
typedef void (*NgrGridLayoutSetPipSizeCallback) (const void* userData, const char* error);
typedef void (*NgrGridLayoutSetPipLockCallback) (const void* userData, const char* error);
typedef void (*NgrGridLayoutSetPipToTileAfterNTilesCallback) (const void* userData, const char* error);

void NgrGridLayoutAcquireRef(NgrGridLayoutHandle handle);
void NgrGridLayoutReleaseRef(NgrGridLayoutHandle handle);

/**
 * Release layout
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrGridLayoutRelease(
	NgrGridLayoutHandle handle,
	const NgrUserData callback_userData,
	NgrGridLayoutReleaseCallback callback
);

/**
 * Activate layout
 *
 * Only one layout can be active at once
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrGridLayoutActivate(
	NgrGridLayoutHandle handle,
	const NgrUserData callback_userData,
	NgrGridLayoutActivateCallback callback
);

/**
 * Get default style for the tiles
 *
 * This style will be used for all tiles that don't relate to other styles or if other styles are not provided
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrGridLayoutGetDefaultTileStyle(
	NgrGridLayoutHandle handle,
	const NgrUserData callback_userData,
	NgrGridLayoutGetDefaultTileStyleCallback callback
);

/**
 * Get style for the screen share tiles
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrGridLayoutGetShareTileStyle(
	NgrGridLayoutHandle handle,
	const NgrUserData callback_userData,
	NgrGridLayoutGetShareTileStyleCallback callback
);

/**
 * Get style for the self preview tiles
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrGridLayoutGetSelfTileStyle(
	NgrGridLayoutHandle handle,
	const NgrUserData callback_userData,
	NgrGridLayoutGetSelfTileStyleCallback callback
);

/**
 * Specify whether to show self preview tile
 *
 * # Arguments
 * * `visible` - Visible or not
 * * `callback` - error if occurred
 */
void NgrGridLayoutSetSelfPreviewVisible(
	NgrGridLayoutHandle handle,
	bool visible,
	const NgrUserData callback_userData,
	NgrGridLayoutSetSelfPreviewVisibleCallback callback
);

/**
 * Specify maximum number of tiles to display
 *
 * Order of importance is following:
 * * pinned
 * * screen share
 * * loudest speaker
 * * self preview
 * * participants with dynamic streams
 * * other participants
 *
 * # Arguments
 * * `count` - Number of tiles to display
 * * `callback` - error if occurred
 */
void NgrGridLayoutSetMaxTilesCount(
	NgrGridLayoutHandle handle,
	uint32_t count,
	const NgrUserData callback_userData,
	NgrGridLayoutSetMaxTilesCountCallback callback
);

/**
 * Specify preferred aspect ratio of the tiles
 *
 * Layout will do its best to preserve it while still filling all available space
 *
 * # Arguments
 * * `aspect_ratio` - Aspect ration (width / height)
 * * `callback` - error if occurred
 */
void NgrGridLayoutSetPreferredTileAspectRatio(
	NgrGridLayoutHandle handle,
	float aspectRatio,
	const NgrUserData callback_userData,
	NgrGridLayoutSetPreferredTileAspectRatioCallback callback
);

/**
 * Specify preview tile position in Grid mode
 *
 * # Arguments
 * * `position` - Position of preview (TopLeft, CenterLeft, BottomLeft, TopRight, CenterRight, BottomRight, TopCenter, CenterCenter, BottomCenter)
 * * `callback` - error if occurred
 */
void NgrGridLayoutSetPreviewPosition(
	NgrGridLayoutHandle handle,
	NgrPreviewPosition position,
	const NgrUserData callback_userData,
	NgrGridLayoutSetPreviewPositionCallback callback
);

/**
 * Specify PIP size
 *
 * # Arguments
 * * `size` - Size of PIP (1 to 50)
 * * `callback` - error if occurred
 */
void NgrGridLayoutSetPipSize(
	NgrGridLayoutHandle handle,
	uint8_t size,
	const NgrUserData callback_userData,
	NgrGridLayoutSetPipSizeCallback callback
);

/**
 * Specify PIP lock state
 *
 * # Arguments
 * * `lock` - Lock state of PIP (true for locked, false for unlocked)
 * * `callback` - error if occurred
 */
void NgrGridLayoutSetPipLock(
	NgrGridLayoutHandle handle,
	bool lock,
	const NgrUserData callback_userData,
	NgrGridLayoutSetPipLockCallback callback
);

/**
 * Specify the number of participants after which PIP should switch to tile mode
 *
 * # Arguments
 * * `num_tiles` - Number of tiles after which PIP switches to tile mode
 * * `callback` - error if occurred
 */
void NgrGridLayoutSetPipToTileAfterNTiles(
	NgrGridLayoutHandle handle,
	uint32_t numTiles,
	const NgrUserData callback_userData,
	NgrGridLayoutSetPipToTileAfterNTilesCallback callback
);


#ifdef __cplusplus
}
#endif

#endif /* GRID_LAYOUT_H */
