#ifndef SPEAKER_LAYOUT_H
#define SPEAKER_LAYOUT_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stdbool.h>

#include "NgrGeneralUtils.h"
#include "NgrPreviewPosition.h"
#include "NgrTileStyle.h"

typedef struct {
	uintptr_t value;
} NgrSpeakerLayoutHandle;

typedef void (*NgrSpeakerLayoutReleaseCallback) (const void* userData, const char* error);
typedef void (*NgrSpeakerLayoutActivateCallback) (const void* userData, const char* error);
typedef void (*NgrSpeakerLayoutGetDefaultTileStyleCallback) (const void* userData, NgrTileStyleHandle* value, const char* error);
typedef void (*NgrSpeakerLayoutGetShareTileStyleCallback) (const void* userData, NgrTileStyleHandle* value, const char* error);
typedef void (*NgrSpeakerLayoutGetSpeakerTileStyleCallback) (const void* userData, NgrTileStyleHandle* value, const char* error);
typedef void (*NgrSpeakerLayoutGetLoudestTileStyleCallback) (const void* userData, NgrTileStyleHandle* value, const char* error);

void NgrSpeakerLayoutAcquireRef(NgrSpeakerLayoutHandle handle);
void NgrSpeakerLayoutReleaseRef(NgrSpeakerLayoutHandle handle);

/**
 * Release layout
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrSpeakerLayoutRelease(
	NgrSpeakerLayoutHandle handle,
	const NgrUserData callback_userData,
	NgrSpeakerLayoutReleaseCallback callback
);

/**
 * Activate layout
 *
 * Only one layout can be active at once
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrSpeakerLayoutActivate(
	NgrSpeakerLayoutHandle handle,
	const NgrUserData callback_userData,
	NgrSpeakerLayoutActivateCallback callback
);

/**
 * Get default style for the tiles
 *
 * This style will be used for all tiles that don't relate to other styles or if other styles are not provided
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrSpeakerLayoutGetDefaultTileStyle(
	NgrSpeakerLayoutHandle handle,
	const NgrUserData callback_userData,
	NgrSpeakerLayoutGetDefaultTileStyleCallback callback
);

/**
 * Get style for the screen share tiles
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrSpeakerLayoutGetShareTileStyle(
	NgrSpeakerLayoutHandle handle,
	const NgrUserData callback_userData,
	NgrSpeakerLayoutGetShareTileStyleCallback callback
);

/**
 * Get style for the speaker tiles
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrSpeakerLayoutGetSpeakerTileStyle(
	NgrSpeakerLayoutHandle handle,
	const NgrUserData callback_userData,
	NgrSpeakerLayoutGetSpeakerTileStyleCallback callback
);

/**
 * Get style for the loudest tiles
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrSpeakerLayoutGetLoudestTileStyle(
	NgrSpeakerLayoutHandle handle,
	const NgrUserData callback_userData,
	NgrSpeakerLayoutGetLoudestTileStyleCallback callback
);

/**
 * Specify whether to show self preview tile
 *
 * # Arguments
 * * `visible` - Visible or not
 * * `callback` - error if occurred
 */
void NgrSpeakerLayoutSetSelfPreviewVisible(
	NgrSpeakerLayoutHandle handle,
	bool visible
);

/**
 * Specify maximum number of side tiles in portrait mode
 *
 * # Arguments
 * * `count` - Number of tiles to display
 * * `callback` - error if occurred
 */
void NgrSpeakerLayoutSetPortraitMaxTilesCount(
	NgrSpeakerLayoutHandle handle,
	uint32_t count
);

/**
 * Specify maximum number of side tiles per row in portrait mode
 *
 * # Arguments
 * * `count` - Number of tiles to display
 * * `callback` - error if occurred
 */
void NgrSpeakerLayoutSetPortraitTilesPerRow(
	NgrSpeakerLayoutHandle handle,
	uint32_t count
);

/**
 * Specify preferred aspect ratio of the side tiles in portrait mode
 *
 * # Arguments
 * * `aspect_ratio` - Aspect ration (width / height)
 * * `callback` - error if occurred
 */
void NgrSpeakerLayoutSetPortraitTileAspectRatio(
	NgrSpeakerLayoutHandle handle,
	float aspectRatio
);

/**
 * Specify maximum number of side tiles in landscape mode
 *
 * # Arguments
 * * `count` - Number of tiles to display
 * * `callback` - error if occurred
 */
void NgrSpeakerLayoutSetLandscapeMaxTilesCount(
	NgrSpeakerLayoutHandle handle,
	uint32_t count
);

/**
 * Specify maximum number of side tiles per column in landscape mode
 *
 * # Arguments
 * * `count` - Number of tiles to display
 * * `callback` - error if occurred
 */
void NgrSpeakerLayoutSetLandscapeTilesPerColumn(
	NgrSpeakerLayoutHandle handle,
	uint32_t count
);

/**
 * Specify preferred aspect ratio of the side tiles in landscape mode
 *
 * # Arguments
 * * `aspect_ratio` - Aspect ration (width / height)
 * * `callback` - error if occurred
 */
void NgrSpeakerLayoutSetLandscapeTileAspectRatio(
	NgrSpeakerLayoutHandle handle,
	float aspectRatio
);

/**
 * Specify preview tile position in Speaker mode
 *
 * # Arguments
 * * `position` - Position of preview (TopLeft, BottomLeft, TopRight or BottomRight)
 */
void NgrSpeakerLayoutSetPreviewPosition(
	NgrSpeakerLayoutHandle handle,
	NgrPreviewPosition position
);

/**
 * Specify PIP size
 *
 * # Arguments
 * * `size` - Size of PIP (1 to 50)
 */
void NgrSpeakerLayoutSetPipSize(
	NgrSpeakerLayoutHandle handle,
	uint8_t size
);

/**
 * Specify PIP lock state
 *
 * # Arguments
 * * `lock` - Lock state of PIP (true for locked, false for unlocked)
 */
void NgrSpeakerLayoutSetPipLock(
	NgrSpeakerLayoutHandle handle,
	bool lock
);

/**
 * Specify the number of participants after which PIP should switch to tile mode
 *
 * # Arguments
 * * `num_tiles` - Number of tiles after which PIP switches to tile mode
 */
void NgrSpeakerLayoutSetPipToTileAfterNTiles(
	NgrSpeakerLayoutHandle handle,
	uint32_t numTiles
);


#ifdef __cplusplus
}
#endif

#endif /* SPEAKER_LAYOUT_H */
