#ifndef TEXTURED_RECT_H
#define TEXTURED_RECT_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stdbool.h>

#include "NgrGeneralUtils.h"
#include "NgrBlendMode.h"
#include "NgrDynamicTexture.h"
#include "NgrPosition.h"
#include "NgrScaleKind.h"

typedef struct {
	uintptr_t value;
} NgrTexturedRectHandle;

typedef void (*NgrTexturedRectReleaseCallback) (const void* userData, const char* error);
typedef void (*NgrTexturedRectSetDynamicTextureCallback) (const void* userData, const char* error);
typedef void (*NgrTexturedRectSetVisibleCallback) (const void* userData, const char* error);
typedef void (*NgrTexturedRectSetScaleTypeCallback) (const void* userData, const char* error);
typedef void (*NgrTexturedRectSetZOrderCallback) (const void* userData, const char* error);
typedef void (*NgrTexturedRectSetVerticalAlignmentCallback) (const void* userData, const char* error);
typedef void (*NgrTexturedRectSetHorizontalAlignmentCallback) (const void* userData, const char* error);
typedef void (*NgrTexturedRectSetPositionCallback) (const void* userData, const char* error);
typedef void (*NgrTexturedRectSetTextureFlipVerticallyCallback) (const void* userData, const char* error);
typedef void (*NgrTexturedRectSetTextureFlipHorizontallyCallback) (const void* userData, const char* error);
typedef void (*NgrTexturedRectSetBlendModeCallback) (const void* userData, const char* error);

void NgrTexturedRectAcquireRef(NgrTexturedRectHandle handle);
void NgrTexturedRectReleaseRef(NgrTexturedRectHandle handle);

/**
 * Release textured rect
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrTexturedRectRelease(
	NgrTexturedRectHandle handle,
	const NgrUserData callback_userData,
	NgrTexturedRectReleaseCallback callback
);

/**
 * Put a texture
 *
 * # Arguments
 * * `texture_id` - Texture ID
 * * `callback` - error if occurred
 */
void NgrTexturedRectSetDynamicTexture(
	NgrTexturedRectHandle handle,
	NgrDynamicTextureHandle texture,
	const NgrUserData callback_userData,
	NgrTexturedRectSetDynamicTextureCallback callback
);

/**
 * Put visibility flag
 *
 * # Arguments
 * * `visible` - Whether rect is visible or not
 * * `callback` - error if occurred
 */
void NgrTexturedRectSetVisible(
	NgrTexturedRectHandle handle,
	bool visible,
	const NgrUserData callback_userData,
	NgrTexturedRectSetVisibleCallback callback
);

/**
 * Put texture scale kind
 *
 * # Arguments
 * * `kind` - Scale kind to use
 * * `callback` - error if occurred
 */
void NgrTexturedRectSetScaleType(
	NgrTexturedRectHandle handle,
	NgrScaleKind kind,
	const NgrUserData callback_userData,
	NgrTexturedRectSetScaleTypeCallback callback
);

/**
 * Put z order
 *
 * Objects with higher z order will be drawn on top
 *
 * # Arguments
 * * `z_order` - z order value
 * * `callback` - error if occurred
 */
void NgrTexturedRectSetZOrder(
	NgrTexturedRectHandle handle,
	float zOrder,
	const NgrUserData callback_userData,
	NgrTexturedRectSetZOrderCallback callback
);

/**
 * Put texture vertical alignment
 *
 * Examples:
 * 0.0 - will align to the top
 * 0.5 - will align to the middle
 * 1.0 - will align to the bottom
 *
 * # Arguments
 * * `value` - Alignment value
 * * `callback` - error if occurred
 */
void NgrTexturedRectSetVerticalAlignment(
	NgrTexturedRectHandle handle,
	float value,
	const NgrUserData callback_userData,
	NgrTexturedRectSetVerticalAlignmentCallback callback
);

/**
 * Put texture horizontal alignment
 *
 * Examples:
 * 0.0 - will align to the left
 * 0.5 - will align to the middle
 * 1.0 - will align to the right
 *
 * # Arguments
 * * `value` - Alignment value
 * * `callback` - error if occurred
 */
void NgrTexturedRectSetHorizontalAlignment(
	NgrTexturedRectHandle handle,
	float value,
	const NgrUserData callback_userData,
	NgrTexturedRectSetHorizontalAlignmentCallback callback
);

/**
 * Put position relative to its parent
 *
 * # Arguments
 * * `position` - Position to use
 * * `callback` - error if occurred
 */
void NgrTexturedRectSetPosition(
	NgrTexturedRectHandle handle,
	NgrPosition position,
	const NgrUserData callback_userData,
	NgrTexturedRectSetPositionCallback callback
);

/**
 * Put texture vertical flip flag
 *
 * # Arguments
 * * `flip` - Whether to flip texture
 * * `callback` - error if occurred
 */
void NgrTexturedRectSetTextureFlipVertically(
	NgrTexturedRectHandle handle,
	bool flip,
	const NgrUserData callback_userData,
	NgrTexturedRectSetTextureFlipVerticallyCallback callback
);

/**
 * Put texture horizontal flip flag
 *
 * # Arguments
 * * `flip` - Whether to flip texture
 * * `callback` - error if occurred
 */
void NgrTexturedRectSetTextureFlipHorizontally(
	NgrTexturedRectHandle handle,
	bool flip,
	const NgrUserData callback_userData,
	NgrTexturedRectSetTextureFlipHorizontallyCallback callback
);

/**
 * Put blend mode
 *
 * # Arguments
 * * `mode` - PorterDuff blend mode
 * * `color` - Color to blend with
 * * `callback` - error if occurred
 */
void NgrTexturedRectSetBlendMode(
	NgrTexturedRectHandle handle,
	NgrBlendMode mode,
	uint32_t color,
	const NgrUserData callback_userData,
	NgrTexturedRectSetBlendModeCallback callback
);


#ifdef __cplusplus
}
#endif

#endif /* TEXTURED_RECT_H */
