#ifndef TILE_SET_H
#define TILE_SET_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stdbool.h>

#include "NgrGeneralUtils.h"
#include "NgrGridLayout.h"
#include "NgrPosition.h"
#include "NgrSelfPreviewLayout.h"
#include "NgrSpeakerLayout.h"

typedef struct {
	uintptr_t value;
} NgrTileSetHandle;

typedef void (*NgrTileSetReleaseCallback) (const void* userData, const char* error);
typedef void (*NgrTileSetSetPositionCallback) (const void* userData, const char* error);
typedef void (*NgrTileSetPinLocalParticipantCallback) (const void* userData, const char* error);
typedef void (*NgrTileSetPinParticipantByIdCallback) (const void* userData, const char* error);
typedef void (*NgrTileSetCreateSelfPreviewLayoutCallback) (const void* userData, NgrSelfPreviewLayoutHandle* value, const char* error);
typedef void (*NgrTileSetCreateGridLayoutCallback) (const void* userData, NgrGridLayoutHandle* value, const char* error);
typedef void (*NgrTileSetCreateSpeakerLayoutCallback) (const void* userData, NgrSpeakerLayoutHandle* value, const char* error);
typedef void (*NgrTileSetShowAudioTilesCallback) (const void* userData, const char* error);

void NgrTileSetAcquireRef(NgrTileSetHandle handle);
void NgrTileSetReleaseRef(NgrTileSetHandle handle);

/**
 * Release tile set
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrTileSetRelease(
	NgrTileSetHandle handle,
	const NgrUserData callback_userData,
	NgrTileSetReleaseCallback callback
);

/**
 * Put tile set position relative to its parent
 *
 * # Arguments
 * * `position` - TileSet position
 * * `callback` - error if occurred
 */
void NgrTileSetSetPosition(
	NgrTileSetHandle handle,
	NgrPosition position,
	const NgrUserData callback_userData,
	NgrTileSetSetPositionCallback callback
);

/**
 * Pins local (current user) participant
 *
 * # Arguments
 * * `callback` - error if occurred
 */
void NgrTileSetPinLocalParticipant(
	NgrTileSetHandle handle,
	const NgrUserData callback_userData,
	NgrTileSetPinLocalParticipantCallback callback
);

/**
 * Pins participant by its `participant_id`
 *
 * # Arguments
 * * `participant_id` - Participant ID
 * * `callback` - error if occurred
 */
void NgrTileSetPinParticipantById(
	NgrTileSetHandle handle,
	const char* participantId,
	const NgrUserData callback_userData,
	NgrTileSetPinParticipantByIdCallback callback
);

/**
 * Create self preview layout
 *
 * # Arguments
 * * `callback` - Layout or error otherwise
 */
void NgrTileSetCreateSelfPreviewLayout(
	NgrTileSetHandle handle,
	const NgrUserData callback_userData,
	NgrTileSetCreateSelfPreviewLayoutCallback callback
);

/**
 * Create grid layout
 *
 * # Arguments
 * * `callback` - Layout or error otherwise
 */
void NgrTileSetCreateGridLayout(
	NgrTileSetHandle handle,
	const NgrUserData callback_userData,
	NgrTileSetCreateGridLayoutCallback callback
);

/**
 * Create speaker layout
 *
 * # Arguments
 * * `callback` - Layout or error otherwise
 */
void NgrTileSetCreateSpeakerLayout(
	NgrTileSetHandle handle,
	const NgrUserData callback_userData,
	NgrTileSetCreateSpeakerLayoutCallback callback
);

/**
 * Show audio tiles
 *
 * # Arguments
 * * `enable` - enable or disable audio tiles
 * * `callback` - error if occurred
 */
void NgrTileSetShowAudioTiles(
	NgrTileSetHandle handle,
	bool enable,
	const NgrUserData callback_userData,
	NgrTileSetShowAudioTilesCallback callback
);


#ifdef __cplusplus
}
#endif

#endif /* TILE_SET_H */
