/** 
{file:
	{name: VidyoRoom.h}
	{description: Room Entity Library. }
	{copyright:
		(c) 2013-2016 Vidyo, Inc.,
		433 Hackensack Avenue,
		Hackensack, NJ  07601.
		All rights reserved.
		The information contained herein is proprietary to Vidyo, Inc.
		and shall not be reproduced, copied (in whole or in part), adapted,
		modified, disseminated, transmitted, transcribed, stored in a retrieval
		system, or translated into any language in any form by any means
		without the express written consent of Vidyo, Inc.}
}
*/
#ifndef VIDYO_ROOM_H_ 
#define VIDYO_ROOM_H_ 

#include <Lmi/Utils/LmiList.h>
#include <Lmi/Utils/LmiVector.h>
#include <Lmi/Utils/LmiUnordered.h>
#include <Lmi/Utils/LmiString.h>
#include <Lmi/Os/LmiSharedPtr.h>

LMI_BEGIN_EXTERN_C

struct VidyoRoomImpl_;
typedef struct VidyoRoomImpl_ VidyoRoomImpl;
Declare_LmiSharedPtr_DummyAlloc(VidyoRoomImpl)
Declare_LmiSharedPtrWrapper(VidyoRoom, VidyoRoomImpl)
Declare_LmiWeakPtrWrapper(VidyoRoomHandle, VidyoRoom, VidyoRoomImpl)
Declare_LmiUnorderedMap(LmiString, VidyoRoomHandle)
Declare_LmiList(VidyoRoom)
Declare_LmiVector(VidyoRoom)
Declare_LmiUnorderedMap(LmiString, VidyoRoom)

LMI_END_EXTERN_C
#include "VidyoEventSchedule.h"
#include "VidyoChatMessage.h"
#include "VidyoLocalSpeaker.h"
#include "VidyoLocalRenderer.h"

#include "VidyoLocalMicrophone.h"
#include "VidyoLocalCamera.h"
#include "VidyoLocalWindowShare.h"
#include "VidyoVirtualVideoSource.h"
#include "VidyoVirtualAudioSource.h"
#include "VidyoLocalMonitor.h"

#include "VidyoRemoteSpeaker.h"
#include "VidyoRemoteRenderer.h"

#include "VidyoRemoteMicrophone.h"
#include "VidyoRemoteCamera.h"
#include "VidyoRemoteWindowShare.h"

#include "VidyoUser.h"
#include "VidyoParticipant.h"
#include "VidyoRoomProperties.h"
#include "VidyoRoomInfo.h"

LMI_BEGIN_EXTERN_C
/**
{type apigroup="simple" visibility="public":
	{name: VidyoRoomSearchResult}
	{parent: VidyoRoom}
	{description: Result of get room history.}
	{value: {name: VIDYO_ROOMSEARCHRESULT_OK} {description: Get room history
		operation was successful.}}
	{value: {name: VIDYO_ROOMSEARCHRESULT_NoRecords} {description: Server did
		not return any records.}}
	{value: {name: VIDYO_ROOMSEARCHRESULT_NoResponse} {description: Server did
		not respond in a reasonable amount of time to the get room history request.}}
	{value: {name: VIDYO_ROOMSEARCHRESULT_Unauthorized} {description: User did
		not have the permission to get room history.}}
	{value: {name: VIDYO_ROOMSEARCHRESULT_MiscLocalError} {description: Server is
		rejecting the request to get room history due to some miscellaneous problem with the request.}}
	{value: {name: VIDYO_ROOMSEARCHRESULT_MiscRemoteError} {description: Server is
		rejecting the request to get room history  due to some miscellaneous problem of its own.}}
}
*/
typedef enum
{
	VIDYO_ROOMSEARCHRESULT_OK,
	VIDYO_ROOMSEARCHRESULT_NoRecords,
	VIDYO_ROOMSEARCHRESULT_NoResponse,
	VIDYO_ROOMSEARCHRESULT_Unauthorized,
	VIDYO_ROOMSEARCHRESULT_MiscLocalError,
	VIDYO_ROOMSEARCHRESULT_MiscRemoteError
} VidyoRoomSearchResult;

/**
{type apigroup="simple" visibility="public":
	{name: VidyoRoomEnterResult}
	{parent: VidyoRoom}
	{description: Result of an attempt to enter a VidyoRoom.}
	{value: {name: VIDYO_ROOMENTERRESULT_OK} {description: Room entered successfully.}}
	{value: {name: VIDYO_ROOMENTERRESULT_Cancelled} {description: Enter request was cancelled.}}
	{value: {name: VIDYO_ROOMENTERRESULT_NoResponse} {description: Server did
		not respond in a reasonable amount of time to the request to enter the room.}}
	{value: {name: VIDYO_ROOMENTERRESULT_InvalidPassword} {description: User did not
		provide a password when one was required, or provided an invalid password.}}
	{value: {name: VIDYO_ROOMENTERRESULT_UnknownRoom} {description: Room
		specified by the user does not exist.}}
	{value: {name: VIDYO_ROOMENTERRESULT_RoomFull} {description: Room
		is at full capacity and cannot accept additional members.}}
	{value: {name: VIDYO_ROOMENTERRESULT_RoomLocked} {description: Room
		is locked.}}
	{value: {name: VIDYO_ROOMENTERRESULT_NotMember} {description: Room allows
		only certain users to enter, and the user is not one of them.}}
	{value: {name: VIDYO_ROOMENTERRESULT_Banned} {description: User has
		been banned from entering the room.}}
	{value: {name: VIDYO_ROOMENTERRESULT_MiscLocalError} {description: Server is
		rejecting the request to enter due to some miscellaneous problem with the request.}}
	{value: {name: VIDYO_ROOMENTERRESULT_MiscRemoteError} {description: Server is
		rejecting the request to enter due to some miscellaneous problem of its own.}}
}
*/
typedef enum
{
	VIDYO_ROOMENTERRESULT_OK,
	VIDYO_ROOMENTERRESULT_Cancelled,
	VIDYO_ROOMENTERRESULT_NoResponse,
	VIDYO_ROOMENTERRESULT_InvalidPassword,
	VIDYO_ROOMENTERRESULT_UnknownRoom,
	VIDYO_ROOMENTERRESULT_RoomFull,
	VIDYO_ROOMENTERRESULT_RoomLocked,
	VIDYO_ROOMENTERRESULT_NotMember,
	VIDYO_ROOMENTERRESULT_Banned,
	VIDYO_ROOMENTERRESULT_MiscLocalError,
	VIDYO_ROOMENTERRESULT_MiscRemoteError
} VidyoRoomEnterResult;

/**
{type apigroup="simple" visibility="public":
	{name: VidyoRoomExitReason}
	{parent: VidyoRoom}
	{description: Reason why a user is no longer a participant of a VidyoRoom.}
	{value: {name: VIDYO_ROOMEXITREASON_Left} {description: Leave was invoked, and this action
		 completed successfully.}}
	{value: {name: VIDYO_ROOMEXITREASON_ConnectionLost} {description: Connectivity was lost and signaling
		 messages can no longer be sent or received.}}
	{value: {name: VIDYO_ROOMEXITREASON_SessionConnectionLost} {description: Session connectivity was lost.}}
	{value: {name: VIDYO_ROOMEXITREASON_MembershipConnectionLost} {description: Membership connectivity
		was lost.}}
	{value: {name: VIDYO_ROOMEXITREASON_NoResponse} {description: A signaling request was sent, but no
		 response was received within the allotted amount of time.}}
	{value: {name: VIDYO_ROOMEXITREASON_ProtocolViolation} {description: A signaling message was received
		 that could not be understood or otherwise violated the rules of the protocol in use.}}
	{value: {name: VIDYO_ROOMEXITREASON_Booted} {description: User was kicked out of the room. This
		 could be the result of an administrator's action.}}
	{value: {name: VIDYO_ROOMEXITREASON_ShuttingDown} {description: Room or the entire conferencing
		 service is going offline. All users are being removed from the room.}}
	{value: {name: VIDYO_ROOMEXITREASON_MiscLocalError} {description: Some miscellaneous problem occurred
		 with this application.}}
	{value: {name: VIDYO_ROOMEXITREASON_MiscRemoteError} {description: Some miscellaneous problem occurred
		 with the conferencing service.}}
}
*/
typedef enum
{
	VIDYO_ROOMEXITREASON_Left,
	VIDYO_ROOMEXITREASON_ConnectionLost,
	VIDYO_ROOMEXITREASON_SessionConnectionLost,
	VIDYO_ROOMEXITREASON_MembershipConnectionLost,
	VIDYO_ROOMEXITREASON_NoResponse,
	VIDYO_ROOMEXITREASON_ProtocolViolation,
	VIDYO_ROOMEXITREASON_Booted,
	VIDYO_ROOMEXITREASON_ShuttingDown,
	VIDYO_ROOMEXITREASON_MiscLocalError,
	VIDYO_ROOMEXITREASON_MiscRemoteError
} VidyoRoomExitReason;

/**
{type apigroup="simple" visibility="public":
	{name: VidyoRoomMediaFailReason}
	{parent: VidyoRoom}
	{description: This type indicates the reason why an attempt to enable media on a room failed.}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_Cancelled} {description: Media support was disabled (by calling VidyoRoomDisableMedia)
		before an attempt to enable it completed.}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_NoConference} {description: There was an error determining which conference to join.}}
	{value:	{name: VIDYO_ROOMMEDIAFAILREASON_SignalingConnectionLost} {description: Signaling Connection lost on current transport.}}
	{value:	{name: VIDYO_ROOMMEDIAFAILREASON_MediaConnectionLost} {description: Media Connection lost on current transport.}}
	{value:	{name: VIDYO_ROOMMEDIAFAILREASON_NoResponse} {description: Signaling messages were sent to enable media, but
		no response was received within the allotted amount of time.}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_InvalidPassword} {description: Media support was disabled because the room password has changed.}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_InvalidRoomPin} {description: Media support was disabled because the room pin has changed.}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_RoomIsFull} {description: Media support was disabled because the room is full.}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_InvalidRoomKey} {description: Media support was disabled because the room is invalid.}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_RoomDisabled} {description: Media support was disabled because the room is disabled.}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_AllLinesInUse} {description: Media support was disabled because all lines are in use.}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_ConnectionError} {description: Media support was disabled because the of a connection error.}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_InvalidRoom} {description: }}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_ConferenceLocked} {description: Media support was disabled because the room is locked.}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_SeatLicenseExpired} {description: Media support was disabled because the seat license has expired.}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_NotLicensed} {description: Media support was disabled because the endpoint is not licensed.}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_Rejected} {description: Attempt to enable media was rejected by the server}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_UnknownError} {description: Media support was disabled as there an unknown fault from the portal.}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_MiscLocalError} {description: Some miscellaneous problem occurred
		with this application.}}
	{value: {name: VIDYO_ROOMMEDIAFAILREASON_MiscRemoteError} {description: Some miscellaneous problem occurred
		with the conferencing service.}}
}
*/
typedef enum
{
	VIDYO_ROOMMEDIAFAILREASON_Cancelled,
	VIDYO_ROOMMEDIAFAILREASON_NoConference,
	VIDYO_ROOMMEDIAFAILREASON_SignalingConnectionLost,
	VIDYO_ROOMMEDIAFAILREASON_MediaConnectionLost,
	VIDYO_ROOMMEDIAFAILREASON_NoResponse,
	VIDYO_ROOMMEDIAFAILREASON_InvalidPassword,
	VIDYO_ROOMMEDIAFAILREASON_InvalidRoomPin,
	VIDYO_ROOMMEDIAFAILREASON_RoomIsFull,
	VIDYO_ROOMMEDIAFAILREASON_InvalidRoomKey,
	VIDYO_ROOMMEDIAFAILREASON_RoomDisabled,
	VIDYO_ROOMMEDIAFAILREASON_AllLinesInUse,
	VIDYO_ROOMMEDIAFAILREASON_ConnectionError,
	VIDYO_ROOMMEDIAFAILREASON_InvalidRoom,
	VIDYO_ROOMMEDIAFAILREASON_ConferenceLocked,
	VIDYO_ROOMMEDIAFAILREASON_SeatLicenseExpired,
	VIDYO_ROOMMEDIAFAILREASON_NotLicensed,
	VIDYO_ROOMMEDIAFAILREASON_Rejected,
	VIDYO_ROOMMEDIAFAILREASON_UnknownError,
	VIDYO_ROOMMEDIAFAILREASON_MiscLocalError,
	VIDYO_ROOMMEDIAFAILREASON_MiscRemoteError
} VidyoRoomMediaFailReason;

/**
{type apigroup="simple" visibility="public":
	{name: VidyoRoomMediaDisableReason}
	{parent: VidyoRoom}
	{description: This type indicates the reason why media support is now disabled on a room.}
	{value: {name: VIDYO_ROOMMEDIADISABLEREASON_Disabled} {description: Media support was disabled gracefully (after calling
		VidyoRoomEnableMedia).}}
	{value: {name: VIDYO_ROOMMEDIADISABLEREASON_SignalingConnectionLost} {description: Signaling connectivity to the conference server was lost.}}
	{value: {name: VIDYO_ROOMMEDIADISABLEREASON_MediaConnectionLost} {description: Media connectivity to the conference server was lost.}}
	{value: {name: VIDYO_ROOMMEDIADISABLEREASON_Booted} {description: The server is ousting this member
		from the conference. This could be the result of an administrator's action.}}
	{value: {name: VIDYO_ROOMMEDIADISABLEREASON_SessionTerminated} {description: The session that was
		carrying the conferencing protocol has been terminated.}}
	{value: {name: VIDYO_ROOMMEDIADISABLEREASON_ServerShuttingDown} {description: The conference server is
		shutting down. All members of all conferences are being booted.}}
	{value: {name: VIDYO_ROOMMEDIADISABLEREASON_ConferenceDestroyed} {description: The conference is
		being destroyed. All members of the conference are being booted.}}
	{value: {name: VIDYO_ROOMMEDIADISABLEREASON_MiscLocalError} {description: Some miscellaneous problem occurred
		with this application.}}
	{value: {name: VIDYO_ROOMMEDIADISABLEREASON_MiscRemoteError} {description: Some miscellaneous problem occurred
		with the conferencing service.}}
}
*/
typedef enum
{
	VIDYO_ROOMMEDIADISABLEREASON_Disabled,
	VIDYO_ROOMMEDIADISABLEREASON_SignalingConnectionLost,
	VIDYO_ROOMMEDIADISABLEREASON_MediaConnectionLost,
	VIDYO_ROOMMEDIADISABLEREASON_Booted,
	VIDYO_ROOMMEDIADISABLEREASON_SessionTerminated,
	VIDYO_ROOMMEDIADISABLEREASON_ServerShuttingDown,
	VIDYO_ROOMMEDIADISABLEREASON_ConferenceDestroyed,
	VIDYO_ROOMMEDIADISABLEREASON_MiscLocalError,
	VIDYO_ROOMMEDIADISABLEREASON_MiscRemoteError
} VidyoRoomMediaDisableReason;

/**
{type apigroup="simple" visibility="public":
	{name: VidyoRoomGetPropertiesResult}
	{parent: VidyoRoom}
	{description: This type indicates the result of get room properties.}
	{value: {name: VIDYO_ROOMGETPROPERTIESRESULT_OK} {description: The get properties
		operation was successful.}}
	{value: {name: VIDYO_ROOMGETPROPERTIESRESULT_NoResponse} {description: The server did
		not respond in a reasonable amount of time to the request to get room properties.}}
	{value: {name: VIDYO_ROOMGETPROPERTIESRESULT_Unauthorized} {description: The user did
		not have permission to get room properties.}}
	{value: {name: VIDYO_ROOMGETPROPERTIESRESULT_MiscLocalError} {description: The server is
		rejecting the request to get room properties due to some miscellaneous problem with the request.}}
	{value: {name: VIDYO_ROOMGETPROPERTIESRESULT_MiscRemoteError} {description: The server is
		rejecting the request to get room properties  due to some miscellaneous problem of its own.}}
}
*/
typedef enum
{
	VIDYO_ROOMGETPROPERTIESRESULT_OK,
	VIDYO_ROOMGETPROPERTIESRESULT_NoResponse,
	VIDYO_ROOMGETPROPERTIESRESULT_Unauthorized,
	VIDYO_ROOMGETPROPERTIESRESULT_MiscLocalError,
	VIDYO_ROOMGETPROPERTIESRESULT_MiscRemoteError
} VidyoRoomGetPropertiesResult;

/**
{type apigroup="simple" visibility="public":
	{name: VidyoRoomSetPropertiesResult}
	{parent: VidyoRoom}
	{description: This type indicates the result of set room properties.}
	{value: {name: VIDYO_ROOMSETPROPERTIESRESULT_OK} {description: The set properties
		operation was successful.}}
	{value: {name: VIDYO_ROOMSETPROPERTIESRESULT_NoResponse} {description: The server did
		not respond in a reasonable amount of time to the request to set room properties.}}
	{value: {name: VIDYO_ROOMSETPROPERTIESRESULT_Conflict} {description: There was a conflict
		in setting room properties.}}
	{value: {name: VIDYO_ROOMSETPROPERTIESRESULT_Unauthorized} {description: The user did
		not have permission to get room properties.}}
	{value: {name: VIDYO_ROOMSETPROPERTIESRESULT_NotOwnerOfRoom} {description: The user is not owner of the room.}}
	{value: {name: VIDYO_ROOMSETPROPERTIESRESULT_NotAcceptable} {description: The properties
		set on the room are not acceptable.}}
	{value: {name: VIDYO_ROOMSETPROPERTIESRESULT_NotAllowed} {description: The operation being
		performed on the room using set properties is not allowed.}}
	{value: {name: VIDYO_ROOMSETPROPERTIESRESULT_MiscLocalError} {description: The server is
		rejecting the request to set room properties due to some miscellaneous problem with the request.}}
	{value: {name: VIDYO_ROOMSETPROPERTIESRESULT_MiscRemoteError} {description: The server is
		rejecting the request to set room properties  due to some miscellaneous problem of its own.}}
}
*/
typedef enum
{
	VIDYO_ROOMSETPROPERTIESRESULT_OK,
	VIDYO_ROOMSETPROPERTIESRESULT_NoResponse,
	VIDYO_ROOMSETPROPERTIESRESULT_Conflict,
	VIDYO_ROOMSETPROPERTIESRESULT_Unauthorized,
	VIDYO_ROOMSETPROPERTIESRESULT_NotOwnerOfRoom,
	VIDYO_ROOMSETPROPERTIESRESULT_NotAcceptable,
	VIDYO_ROOMSETPROPERTIESRESULT_NotAllowed,
	VIDYO_ROOMSETPROPERTIESRESULT_MiscLocalError,
	VIDYO_ROOMSETPROPERTIESRESULT_MiscRemoteError
} VidyoRoomSetPropertiesResult;

/**
{type visibility="public":
    {name: VidyoRoomSetFavoriteResult}
    {parent: VidyoRoom}
    {description: This type indicates the result of set favorite room.}
    {value: {name: VIDYO_ROOMSETFAVORITERESULT_OK} {description: The set operation was successful.}}
    {value: {name: VIDYO_ROOMSETFAVORITERESULT_NoResponse} {description: The server did not respond in a reasonable amount of time to the request to set.}}
    {value: {name: VIDYO_ROOMSETFAVORITERESULT_Conflict} {description: There was a conflict in set.}}
    {value: {name: VIDYO_ROOMSETFAVORITERESULT_Unauthorized} {description: The user did not have permission to set.}}
    {value: {name: VIDYO_ROOMSETFAVORITERESULT_NotAcceptable} {description: The properties set on the room are not acceptable.}}
    {value: {name: VIDYO_ROOMSETFAVORITERESULT_NotAllowed} {description: The operation being performed on the room using set properties is not allowed.}}
    {value: {name: VIDYO_ROOMSETFAVORITERESULT_MiscLocalError} {description: The server is rejecting the request to set due to some miscellaneous problem with the request.}}
    {value: {name: VIDYO_ROOMSETFAVORITERESULT_MiscRemoteError} {description: The server isrejecting the request to set room properties due to some miscellaneous problem of its own.}}
}
*/
typedef enum
{
    VIDYO_ROOMSETFAVORITERESULT_OK,
    VIDYO_ROOMSETFAVORITERESULT_NoResponse,
    VIDYO_ROOMSETFAVORITERESULT_Conflict,
    VIDYO_ROOMSETFAVORITERESULT_Unauthorized,
    VIDYO_ROOMSETFAVORITERESULT_NotAcceptable,
    VIDYO_ROOMSETFAVORITERESULT_NotAllowed,
    VIDYO_ROOMSETFAVORITERESULT_MiscLocalError,
    VIDYO_ROOMSETFAVORITERESULT_MiscRemoteError
} VidyoRoomSetFavoriteResult;

/**
{type apigroup="simple" visibility="public":
	{name: VidyoRoomDeleteResult}
	{parent: VidyoRoom}
	{description: This type indicates the result of delete room.}
	{value: {name: VIDYO_ROOMDELETERESULT_OK} {description: The delete room operation was successful.}}
	{value: {name: VIDYO_ROOMDELETERESULT_NoResponse} {description: The server did not respond in a reasonable amount of time to the request to delete room.}}
	{value: {name: VIDYO_ROOMDELETERESULT_Unauthorized} {description: The user did not have permission to delete room.}}
	{value: {name: VIDYO_ROOMDELETERESULT_OutOfResources} {description: The server cannot
delete rooms at the current time because it is out of some resource.}}
	{value: {name: VIDYO_ROOMDELETERESULT_MiscError} {description: Some other error occurred.}}
}
*/
typedef enum
{
	VIDYO_ROOMDELETERESULT_OK,
	VIDYO_ROOMDELETERESULT_NoResponse,
	VIDYO_ROOMDELETERESULT_Unauthorized,
	VIDYO_ROOMDELETERESULT_OutOfResources,
	VIDYO_ROOMDELETERESULT_MiscError
} VidyoRoomDeleteResult;


/**
 {type visibility="public":
	{name: VidyoRoomInviteResult}
	{parent: VidyoRoom}
	{description: This type indicates the result of an attempt to invite an user to a room.}
	{value: {name: VIDYO_ROOMINVITERESULT_OK} {description: The invite operation was successful.}}
	{value: {name: VIDYO_ROOMINVITERESULT_NoResponse} {description: The server did not respond in a reasonable amount of time to the request to invite user to the room.}}
	{value: {name: VIDYO_ROOMINVITERESULT_Unauthorized} {description: The user does not have sufficient privileges to invite user to the room it requested.}}
	{value: {name: VIDYO_ROOMINVITERESULT_OutOfResources} {description: The server cannot delete rooms at the current time because it is out of some resource.}}
	{value: {name: VIDYO_ROOMINVITERESULT_MiscError} {description: Some other error occurred.}}
	{value: {name: VIDYO_ROOMINVITERESULT_UserIsOffline} {description: User we are attempting to invite to a room is offline.}}
	{value: {name: VIDYO_ROOMINVITERESULT_ServerError} {description: Server returned an error while attemptig to invite user to the room.}}
	{value: {name: VIDYO_ROOMINVITERESULT_RoomFull} {description: Room is currently full.}}
 }
 */
typedef enum
{
    VIDYO_ROOMINVITERESULT_OK,
    VIDYO_ROOMINVITERESULT_NoResponse,
    VIDYO_ROOMINVITERESULT_Unauthorized,
    VIDYO_ROOMINVITERESULT_OutOfResources,
    VIDYO_ROOMINVITERESULT_MiscError, 
    VIDYO_ROOMINVITERESULT_UserIsOffline,
    VIDYO_ROOMINVITERESULT_ServerError,
	VIDYO_ROOMINVITERESULT_RoomFull,
} VidyoRoomInviteResult;

/**
 {type visibility="public":
 {name: VidyoRoomShowThrottle}
 {parent: VidyoRoom}
 {description: Type of throttling to apply to show requests.}
 {value: {name: VIDYO_ROOMSHOWTHROTTLE_NONE} {description: Disable show throttling.}}
 {value: {name: VIDYO_ROOMSHOWTHROTTLE_TIMED} {description: Throttle show requests with timer.}}
 {value: {name: VIDYO_ROOMSHOWTHROTTLE_WINDOWSIZE} {description: Throttle show requests by window size change threshold.}}
 }
 */
typedef enum {
	VIDYO_ROOMSHOWTHROTTLE_NONE = 0,
	VIDYO_ROOMSHOWTHROTTLE_TIMED = 1,
	VIDYO_ROOMSHOWTHROTTLE_WINDOWSIZE = 2
} VidyoRoomShowThrottle;

/**
{type apigroup="simple" visibility="public":
	{name: VidyoRoomConferenceMode}
	{parent: VidyoRoom}
	{description: Sets the type of conference mode.}
	{value: {name: VIDYO_ROOMCONFERENCEMODE_LOBBY} {description: Tells us if the user has entered into a waiting room,i.e. presenter has not joined yet}}
	{value: {name: VIDYO_ROOMCONFERENCEMODE_LECTURE} {description: Tells us if conference mode is lecture mode}}
	{value: {name: VIDYO_ROOMCONFERENCEMODE_GROUP} {description: Tells us if the user is in group mode }}
}
*/
typedef enum
{
	VIDYO_ROOMCONFERENCEMODE_LOBBY,
	VIDYO_ROOMCONFERENCEMODE_LECTURE,
	VIDYO_ROOMCONFERENCEMODE_GROUP
} VidyoRoomConferenceMode;

/**
{type visibility="public":
	{name: VidyoRoomConferenceHostState}
	{parent: VidyoRoom}
	{description: Tell us the joining status of the host or owner of the room hosting the conference.}
	{value: {name: VIDYO_ROOMCONFERENCEHOSTSTATE_JOINED} {description: Tells us if the host has joined}}
	{value: {name: VIDYO_ROOMCONFERENCEHOSTSTATE_LEFT} {description: Tells us if the host has left the conference}}
}
*/
typedef enum
{
	VIDYO_ROOMCONFERENCEHOSTSTATE_JOINED,
	VIDYO_ROOMCONFERENCEHOSTSTATE_LEFT
} VidyoRoomConferenceHostState;

/**
{type visibility="public":
	{name: VidyoRoomRecordingState}
	{parent: VidyoRoom}
	{description: Provide the status of recording.}
	{value:
		{name: VIDYO_ROOMRECORDINGSTATE_NotRecording} 
		{description: Recording is in stop state.}}
	{value:
		{name: VIDYO_ROOMRECORDINGSTATE_RecordingPaused}
		{description: Recording is in pause state.}}
	{value:
		{name: VIDYO_ROOMRECORDINGSTATE_Recording}
		{description: Recording is in start/resume state.}}
}
*/
typedef enum
{
	VIDYO_ROOMRECORDINGSTATE_NotRecording,
	VIDYO_ROOMRECORDINGSTATE_RecordingPaused,
	VIDYO_ROOMRECORDINGSTATE_Recording
} VidyoRoomRecordingState;

/**
{type visibility="public":
	{name: VidyoRoomWebcastingState}
	{parent: VidyoRoom}
	{description: Provide the status of webcasting/broadcasting.}
	{value:
		{name: VIDYO_ROOMWEBCASTINGSTATE_NotWebcasting} 
		{description: Room is not broadcast or webcasting is not going on.}}
	{value:
		{name: VIDYO_ROOMWEBCASTINGSTATE_Webcasting}
		{description: Room is being broadcast or webcasting is going on.}}
}
*/
typedef enum
{
	VIDYO_ROOMWEBCASTINGSTATE_NotWebcasting,
	VIDYO_ROOMWEBCASTINGSTATE_Webcasting
} VidyoRoomWebcastingState;

/**
{type visibility="public":
	{name: VidyoRoomModerationActionType}
	{parent: VidyoRoom}
	{description: This type indicates the action of moderation request.}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_Mute}
		{description: Mute video or audio action.}
	}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_Unmute}
		{description: Unmute video or audio action.}
	}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_StartLectureMode}
		{description: Start lecture mode action.}
	}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_StopLectureMode}
		{description: Stop lecture mode action.}
	}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_SetPresenter}
		{description: Set presenter action.}
	}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_RemovePresenter}
		{description: Remove presenter action.}
	}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_RaiseHand}
		{description: Raise hand action.}
	}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_UnraiseHand}
		{description: Unraise hand action.}
	}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_DismissRaisedHand}
		{description: Dismiss raised hand action.}
	}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_DismissAllRaisedHands}
		{description: Dismiss all raised hands action.}
	}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_SetModeratorPin}
		{description: Set moderator pin.}
	}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_RemoveModeratorPin}
		{description: Remove moderator pin.}
	}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_DisconnectAll}
		{description: Disconnect all participants.}
	}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_DisconnectOne}
		{description: Disconnect one participant.}
	}
	{value:
		{name: VIDYO_ROOMMODERATIONACTIONTYPE_Invalid}
		{description: Wrong action.}
	}
}
*/
typedef enum
{
	VIDYO_ROOMMODERATIONACTIONTYPE_Mute,
	VIDYO_ROOMMODERATIONACTIONTYPE_Unmute,
	VIDYO_ROOMMODERATIONACTIONTYPE_StartLectureMode,
	VIDYO_ROOMMODERATIONACTIONTYPE_StopLectureMode,
	VIDYO_ROOMMODERATIONACTIONTYPE_SetPresenter,
	VIDYO_ROOMMODERATIONACTIONTYPE_RemovePresenter,
	VIDYO_ROOMMODERATIONACTIONTYPE_RaiseHand,
	VIDYO_ROOMMODERATIONACTIONTYPE_UnraiseHand,
	VIDYO_ROOMMODERATIONACTIONTYPE_DismissRaisedHand,
	VIDYO_ROOMMODERATIONACTIONTYPE_DismissAllRaisedHands,
	VIDYO_ROOMMODERATIONACTIONTYPE_SetModeratorPin,
	VIDYO_ROOMMODERATIONACTIONTYPE_RemoveModeratorPin,
	VIDYO_ROOMMODERATIONACTIONTYPE_DisconnectAll,
	VIDYO_ROOMMODERATIONACTIONTYPE_DisconnectOne,
	VIDYO_ROOMMODERATIONACTIONTYPE_Invalid
} VidyoRoomModerationActionType;

/**
{type visibility="public":
	{name:  VidyoRoomConnectionProperties}
	{parent: VidyoRoom}
	{description: Represents the connection properties for the room, which are based on information from the portal.}
	{member:
		{name: roomName}
		{type: LmiString}
		{description: Name of the room.}
	}
	{member:
		{name: recordingState}
		{type: VidyoRoomRecordingState}
		{description: State of Recording. See enum VidyoRoomRecordingState.}
	}
	{member:
		{name: webcastingState}
		{type: VidyoRoomWebcastingState}
		{description: State of Webcasting. See enum VidyoRoomWebcastingState.}
	}
	{member:
		{name: isRoomLocked}
		{type: LmiBool}
		{description: Return LMI_TRUE if room is lock otherwise LMI_FALSE.}
	}
	{member:
		{name: hasModeratorPin}
		{type: LmiBool}
		{description: Return LMI_TRUE if moderation pin is set otherwise LMI_FALSE.}
	}
	{member:
		{name: hasRoomPin}
		{type: LmiBool}
		{description: Return LMI_TRUE if room pin is set otherwise LMI_FALSE.}
	}
	{member:
		{name: isVp9Enabled}
		{type: LmiBool}
		{description: Return LMI_TRUE if VP9 is enabled otherwise LMI_FALSE.}
	}
	{member:
		{name: minimumRoomPinLength}
		{type: LmiInt}
		{description: Return the minimum room pin length if minimumRoomPinLength is set by default.}
	}
	{member:
		{name: maximumRoomPinLength}
		{type: LmiInt}
		{description: Return the maximum room pin length if maximumRoomPinLength is set by default.}
	}
	{member:
		{name: maxSendBandWidth}
		{type: LmiUint}
		{description: Represents the maximum send bandwidth.}
	}
	{member:
		{name: maxReceiveBandWidth}
		{type: LmiUint}
		{description: Represents the maximum receive bandwidth.}
	}
}
*/
typedef struct
{
	LmiString roomName;
	VidyoRoomRecordingState recordingState;
	VidyoRoomWebcastingState webcastingState;
	LmiBool isRoomLocked;
	LmiBool hasModeratorPin;
	LmiBool hasRoomPin;
	LmiBool isVp9Enabled;
	LmiInt minimumRoomPinLength;
	LmiInt maximumRoomPinLength;
	LmiUint maxSendBandWidth;
	LmiUint maxReceiveBandWidth;
} VidyoRoomConnectionProperties;

/**
{type:
	{name:  VidyoGenerationDiff}
	{parent: VidyoRoom}
	{description: Represents a remote participant whose selected ranking in a conference has changed.
	When the selected participant list changes, a set of these objects can be used to represent
	the differences between	the old list and the new list. Each object can indicate a new participant
	in the list, a participant that is no longer in the list, or a participant that has changed rank
	within the list.
	}
	{member:
		{name: participant}
		{type: VidyoParticipant}
		{description: }
	}
	{member:
		{name: newIndex}
		{type: LmiInt}
		{description: }
	}
	{member:
		{name: oldIndex}
		{type: LmiInt}
		{description: }
	}
}
*/
typedef struct
{
	VidyoParticipant participant;
	LmiInt newIndex;
	LmiInt oldIndex;
} VidyoGenerationDiff;

/**
{function visibility="private":
	{name: VidyoGenerationDiffConstructDefault}
	{parent: VidyoGenerationDiff}
	{description: Destructs a VidyoGenerationDiff object.}
	{prototype: VidyoGenerationDiff* VidyoGenerationDiffConstructDefault(VidyoGenerationDiff *rank, LmiAllocator *alloc)}
	{parameter: {name: rank} {description: The VidyoGenerationDiff object to construct.}}
	{parameter: {name: alloc} {description: An allocator to use for miscellaneous items.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoGenerationDiff* VidyoGenerationDiffConstructDefault(VidyoGenerationDiff *rank, LmiAllocator* alloc);

/**
{function visibility="private":
	{name: VidyoGenerationDiffConstruct}
	{parent: VidyoGenerationDiff}
	{description: Constructs a VidyoGenerationDiff object.}
	{prototype: VidyoGenerationDiff *VidyoGenerationDiffConstruct(VidyoGenerationDiff* rank, VidyoParticipant* p, LmiInt newIndex, LmiInt oldIndex, LmiAllocator *alloc)}
	{parameter: {name: rank} {description: The VidyoGenerationDiff object to construct.}}
	{parameter: {name: p} {description: The VidyoParticipant object whose rank changed.}}
	{parameter: {name: newIndex} {description: total number of users of the room.}}
	{parameter: {name: oldIndex} {description: total number of users of the room.}}
	{parameter: {name: alloc} {description: An allocator to use for miscellaneous items.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoGenerationDiff *VidyoGenerationDiffConstruct(VidyoGenerationDiff* rank, VidyoParticipant* p, LmiInt newIndex, LmiInt oldIndex, LmiAllocator *alloc);

/**
{function visibility="private":
	{name: VidyoGenerationDiffConstructCopy}
	{parent: VidyoGenerationDiff}
	{description: Constructs a VidyoGenerationDiff object as a copy of an existing object.}
	{prototype: VidyoGenerationDiff* VidyoGenerationDiffConstructCopy(VidyoGenerationDiff *d, const VidyoGenerationDiff *s)}
	{parameter: {name: d} {description: The VidyoGenerationDiff object to construct.}}
	{parameter: {name: s} {description: The existing VidyoGenerationDiff object to copy.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoGenerationDiff *VidyoGenerationDiffConstructCopy(VidyoGenerationDiff *d, const VidyoGenerationDiff *s);

/**
{function visibility="private":
	{name: VidyoGenerationDiffDestruct}
	{parent: VidyoGenerationDiff}
	{description: Destructs a VidyoGenerationDiff object.}
	{prototype: void VidyoGenerationDiffDestruct(VidyoGenerationDiff *p)}
	{parameter: {name: p} {description: The VidyoGenerationDiff object to destruct.}}
}
*/
void VidyoGenerationDiffDestruct(VidyoGenerationDiff *p);

/**
{function visibility="private":
	{name: VidyoGenerationDiffAssign}
	{parent: VidyoGenerationDiff}
	{description: Assign the contents of one VidyoGenerationDiff to another.}
	{prototype: VidyoGenerationDiff *VidyoGenerationDiffAssign(VidyoGenerationDiff *d, const VidyoGenerationDiff *s)}
	{parameter: {name: d} {description: The object to assign to.}}
	{parameter: {name: s} {description: The object to copy.}}
	{return: A pointer to the assigned object on success, or NULL on failure.}
}
*/
VidyoGenerationDiff *VidyoGenerationDiffAssign(VidyoGenerationDiff *d, const VidyoGenerationDiff *s);

/**
{function visibility="private":
	{name: VidyoGenerationDiffSwap}
	{parent: VidyoGenerationDiff}
	{description: Swap the contents of one VidyoGenerationDiff with other.}
	{prototype: void VidyoGenerationDiffSwap(VidyoGenerationDiff *a, VidyoGenerationDiff *b)}
	{parameter: {name: a} {description: The object to swap.}}
	{parameter: {name: b} {description: The object to swap.}}
}
*/
void VidyoGenerationDiffSwap(VidyoGenerationDiff *a, VidyoGenerationDiff *b);

/**
{function visibility="private":
	{name: VidyoGenerationDiffEqual}
	{parent: VidyoGenerationDiff}
	{description: Check the equality of VidyoGenerationDiff object.}
	{prototype: LmiBool VidyoGenerationDiffEqual(const VidyoGenerationDiff *a, const VidyoGenerationDiff *b)}
	{parameter: {name: a} {description: The VidyoGenerationDiff object to compare.}}
	{parameter: {name: b} {description: The VidyoGenerationDiff object to compare.}}
	{return: LMI_TRUE if equal LMI_FALSE otherwise.}
}
*/
LmiBool VidyoGenerationDiffEqual(const VidyoGenerationDiff *a, const VidyoGenerationDiff *b);

Declare_LmiVector(VidyoGenerationDiff)


/**
{type apigroup="simple" reference-counted="yes":
	{name: VidyoRoom}
	{parent: Endpoint}
	{description: Establishes and manages a conference room. See VidyoRoomType for sescription of room types.}
	{member:
		{name: id}
		{type: LmiString}
		{description: Unique ID that identifies a VidyoRoom at the server.}
	}
	{member:
		{name: type}
		{type: VidyoRoomType}
		{description: Type of VidyoRoom such as topic or Conversation.}
	}
}
*/

/**
{type visibility="private":
	{name: VidyoRoomHandle}
	{parent: VidyoRoom}
	{description: Weak reference to the VidyoRoom object.}
}
*/

/**
{callback apigroup="simple":
	{name: VidyoRoomOnEntered}
	{parent: VidyoRoom}
	{description: Callback that is triggered when entering the VidyoRoom as a result of VidyoRoomEnter() request.}
	{prototype: void (*VidyoRoomOnEntered)(VidyoRoom* room, VidyoRoomEnterResult result)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: result}
		{description: Result of an attempt to enter the VidyoRoom. }
	}
}
*/
typedef void (*VidyoRoomOnEntered)(VidyoRoom* room, VidyoRoomEnterResult result);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnExited}
	{parent: VidyoRoom}
	{description: Callback that is triggered when the VidyoUser exited the VidyoRoom or as a result of VidyoRoomLeave() request. }
	{prototype: void (*VidyoRoomOnExited)(VidyoRoom* room, VidyoRoomExitReason reason)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: reason}
		{description: Reason why VidyoUser exited VidyoRoom. }
	}
}
*/
typedef void(*VidyoRoomOnExited)(VidyoRoom* room, VidyoRoomExitReason reason);

/**
{callback:
	{name: VidyoRoomOnMediaRouteAcquired}
	{parent: VidyoRoom}
	{description: Callback that is triggered when media route is discovered as a result of VidyoRoomAcquireMediaRoute() request. }
	{prototype: void (*VidyoRoomOnMediaRouteAcquired)(VidyoRoom* room)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
}
*/
typedef void(*VidyoRoomOnMediaRouteAcquired)(VidyoRoom* room);

/**
 {callback apigroup="simple":
 	{name: VidyoRoomOnMediaRouteAcquireFailed}
 	{parent: VidyoRoom}
 	{description: Callback that is triggered when media enabling failed as a result of VidyoRoomAcquireMediaRoute() request. }
 	{prototype: void (*VidyoRoomOnMediaRouteAcquireFailed)(VidyoRoom* room, VidyoRoomMediaFailReason reason)}
 	{parameter:
 		{name: room}
 		{description: Pointer to the VidyoRoom that triggered the callback. }
 	}
 	{parameter:
 		{name: reason}
 		{description: Reason why media failed. }
 	}
 }
 */
typedef void(*VidyoRoomOnMediaRouteAcquireFailed)(VidyoRoom* room, VidyoRoomMediaFailReason reason);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnMediaEnabled}
	{parent: VidyoRoom}
	{description: Callback that is triggered when media was enabled as a result of VidyoRoomEnableMedia() request. }
	{prototype: void (*VidyoRoomOnMediaEnabled)(VidyoRoom* room)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
}
*/
typedef void(*VidyoRoomOnMediaEnabled)(VidyoRoom* room);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnMediaFailed}
	{parent: VidyoRoom}
	{description: Callback that is triggered when media enabling failed as a result of VidyoRoomEnableMedia() request. }
	{prototype: void (*VidyoRoomOnMediaFailed)(VidyoRoom* room, VidyoRoomMediaFailReason reason)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: reason}
		{description: Reason why media failed. }
	}
}
*/
typedef void(*VidyoRoomOnMediaFailed)(VidyoRoom* room, VidyoRoomMediaFailReason reason);


/**
{callback apigroup="simple":
	{name: VidyoRoomOnMediaDisabled}
	{parent: VidyoRoom}
	{description: Callback that is triggered when media has been disbled after a successfully enabling it with VidyoRoomEnableMedia(). }
	{prototype: void (*VidyoRoomOnMediaDisabled)(VidyoRoom* room, VidyoRoomMediaDisableReason reason)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: reason}
		{description: Reason why media had beed disabled. }
	}
}
*/
typedef void(*VidyoRoomOnMediaDisabled)(VidyoRoom* room, VidyoRoomMediaDisableReason reason);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnParticipantJoined}
	{parent: VidyoRoom}
	{description: Callback that is triggered when VidyoParticipant is entering or has been in the room when VidyoUser Entered.}
	{prototype: void (*VidyoRoomOnParticipantJoined)(VidyoRoom* room, VidyoParticipant* participant)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: participant}
		{description: Pointer to the VidyoParticipant object. }
	}
}
*/
typedef void(*VidyoRoomOnParticipantJoined)(VidyoRoom* room, VidyoParticipant* participant);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnParticipantLeft}
	{parent: VidyoRoom}
	{description: Callback that is triggered when VidyoParticipant is leaving or is being removed when this VidyoUser Exited.}
	{prototype: void (*VidyoRoomOnParticipantLeft)(VidyoRoom* room, VidyoParticipant* participant)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: participant}
		{description: Pointer to the VidyoParticipant object.}
	}
}
*/
typedef void(*VidyoRoomOnParticipantLeft)(VidyoRoom* room, VidyoParticipant* participant);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnDynamicParticipantChanged}
	{parent: VidyoRoom}
	{description: This callback is invoked to report that the list of dynamically selected participants in a conference has changed.}
	{prototype: void (*VidyoRoomOnDynamicParticipantChanged)(VidyoRoom* room, LmiVector(VidyoParticipant) *selectedParticipants)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: selectedParticipants}
		{description: Ordered list of most recently selected participants in the conference.}
	}
}
*/
typedef void(*VidyoRoomOnDynamicParticipantChanged)(VidyoRoom* room, LmiVector(VidyoParticipant) *selectedParticipants);

/**
 {callback apigroup="simple":
	{name: VidyoRoomOnLoudestParticipantChanged}
	{parent: VidyoRoom}
	{description: This callback is invoked to report that an audio-only participant has gain or lost loudest speaker status.}
	{prototype: void (*VidyoRoomOnLoudestParticipantChanged)(VidyoRoom* room, VidyoParticipant* participant, LmiBool isAudioOnly, LmiBool isStatic)}
	{parameter:
        {name: room}
        {description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
        {name: participant}
        {description: Pointer to an audio-only participant.}
	}
	{parameter:
        {name: isAudioOnly}
        {description: LMI_TRUE if the participant is audio only, otherwise LMI_FALSE}
	}
    {parameter:
        {name: isStatic}
        {description: LMI_TRUE if the participant is viewed statically, otherwise LMI_FALSE}
    }
 }
 */
typedef void(*VidyoRoomOnLoudestParticipantChanged)(VidyoRoom* room, VidyoParticipant* participant, LmiBool isAudioOnly, LmiBool isStatic);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnGetRoomPropertiesResult}
	{parent: VidyoRoom}
	{description: Callback that is triggered when VidyoRoomGetRoomProperties() request completes. }
	{prototype: void (*VidyoRoomOnGetRoomPropertiesResult)(VidyoRoom* room, VidyoRoomGetPropertiesResult result, const VidyoRoomProperties* roomProperties)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: result}
		{description: Result of the VidyoRoomGetRoomProperties request. }
	}
	{parameter:
		{name: roomProperties}
		{description: Pointer to the VidyoRoomProperties object or NULL on failure.}
	}
}
*/
typedef void(*VidyoRoomOnGetRoomPropertiesResult)(VidyoRoom* room, VidyoRoomGetPropertiesResult result, const VidyoRoomProperties* roomProperties);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnSetRoomPropertiesResult}
	{parent: VidyoRoom}
	{description: Callback that is triggered when VidyoRoomSetProperties request completes.}
	{prototype: void (*VidyoRoomOnSetRoomPropertiesResult)(VidyoRoom* room, VidyoRoomSetPropertiesResult result)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: result}
		{description: Result of the VidyoRoomSetProperties request.}
	}
}
*/
typedef void(*VidyoRoomOnSetRoomPropertiesResult)(VidyoRoom* room, VidyoRoomSetPropertiesResult result);

/**
 {callback apigroup="simple":
	{name: VidyoRoomOnRoomPropertiesChanged}
	{parent: VidyoRoom}
	{description: Callback that is triggered when VidyoRoomProperties is changed on the server side.}
	{prototype: void (*VidyoRoomOnRoomPropertiesChanged)(VidyoRoom* room, const VidyoRoomProperties* roomProperties)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: roomProperties}
		{description: Updated room properties.}
	}
 }
 */
typedef void(*VidyoRoomOnRoomPropertiesChanged)(VidyoRoom* room, const VidyoRoomProperties* roomProperties);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnSetSubject}
	{parent: VidyoRoom}
	{description: Callback that is triggered when the subject of the VidyoRoom has changed.}
	{prototype: void (*VidyoRoomOnSetSubject)(VidyoRoom* room, const char* subject)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: subject}
		{description: New subject of the VidyoRoom.}
	}
}
*/
typedef void(*VidyoRoomOnSetSubject)(VidyoRoom* room, const char* subject);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnMessageReceived}
	{parent: VidyoRoom}
	{description: Callback that is triggered for every VidyoChatMessage that was sent to the VidyoRoom by every other VidyoParticipant after the VidyoRoomOnEntered returned successfully. }
	{prototype: void (*VidyoRoomOnMessageReceived)(VidyoRoom* room, VidyoParticipant* participant, VidyoChatMessage* chatMessage)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: participant}
		{description: Pointer to the VidyoParticipant object that sent the message. It can be NULL when this information is unknown to the client due to timing.}
	}
	{parameter:
		{name: chatMessage}
		{description: Pointer to the VidyoChatMessage object containing the message, and sender name and id.}
	}
}
*/
typedef void(*VidyoRoomOnMessageReceived)(VidyoRoom* room, VidyoParticipant* participant, VidyoChatMessage* chatMessage);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnMessageAcknowledged}
	{parent: VidyoRoom}
	{description: Callback that is triggered when a VidyoChatMessage is delivered to the VidyoRoom on the server. }
	{prototype: void (*VidyoRoomOnMessageAcknowledged)(VidyoRoom* room, VidyoChatMessage* chatMessage)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: chatMessage}
		{description: Pointer to the VidyoChatMessage object created by the server from VidyoRoomSendMessage() call. }
	}
	{note: The VidyoChatMessage has the messageId that was assigned by the server which can be referenced by the VidyoRoomOnMessageRead callback. Currently the only way to correlate the message that was sent with VidyoRoomSendMessage() to the acknowledged message is with the message body. }
}
*/
typedef void(*VidyoRoomOnMessageAcknowledged)(VidyoRoom* room, VidyoChatMessage* chatMessage);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnMessageRead}
	{parent: VidyoRoom}
	{description: Callback that is triggered when a message that was acknowledged by the server was read by the other VidyoParticipant in the room. }
	{prototype: void (*VidyoRoomOnMessageRead)(VidyoRoom* room, VidyoParticipant* participant, LmiInt64 messageId)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: participant}
		{description: Pointer to the VidyoParticipant object that read the message.}
	}
	{parameter:
		{name: messageId}
		{description: The ID of the VidyoChatMessage that was read. }
	}
}
*/
typedef void(*VidyoRoomOnMessageRead)(VidyoRoom* room, VidyoParticipant* participant, LmiInt64 messageId);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnMessageTypingIndication}
	{parent: VidyoRoom}
	{description: Callback that is triggered when typing state of a remote VidyoParticipant changes. }
	{prototype: void (*VidyoRoomOnMessageTypingIndication)(VidyoRoom* room, VidyoParticipant* participant, VidyoChatMessageTypingIndication typingIndication)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: participant}
		{description: Pointer to the VidyoParticipant object whose typing state changed. }
	}
	{parameter:
		{name: typingIndication}
		{description: New typing state of the VidyoParticipant. }
	}
}
*/
typedef void(*VidyoRoomOnMessageTypingIndication)(VidyoRoom* room, VidyoParticipant* participant, VidyoChatMessageTypingIndication typingIndication);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnGetHistoryResults}
	{parent: VidyoRoom}
	{description:  Callback that is triggered as a result of VidyoRoomGetHistory() call.}
	{prototype: void (*VidyoRoomOnGetHistoryResults)(VidyoRoom* room, const LmiVector(VidyoChatMessage) *records,
		LmiInt64 historySize, VidyoRoomSearchResult result)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: records}
		{description: Pointer to the vector of VidyoChatMessage records.}
	}
	{parameter:
		{name: historySize}
		{description: Total number of records in the room history.}
	}
	{parameter:
		{name: result}
		{description: Result of VidyoRoomGetHistory() call.}
	}
}
*/
typedef void (*VidyoRoomOnGetHistoryResults)(VidyoRoom* room, const LmiVector(VidyoChatMessage) *records,
	LmiInt64 historySize, VidyoRoomSearchResult result);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnMessageSearchResults}
	{parent: VidyoRoom}
	{description: Callback that is triggered as a result of VidyoRoomSearch() call.}
	{prototype: void (*VidyoRoomOnMessageSearchResults)(VidyoRoom* room, const char* searchText, const LmiVector(VidyoChatMessage)* records, VidyoRoomSearchResult result)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: searchText}
		{description: Case insensitive text used for the search.}
	}
	{parameter:
		{name: records}
		{description: Pointer to the vector of VidyoChatMessage records.}
	}
	{parameter:
		{name: result}
		{description: Results of VidyoRoomSearch() call.}
	}
}
*/
typedef void(*VidyoRoomOnMessageSearchResults)(VidyoRoom* room, const char* searchText, const LmiVector(VidyoChatMessage)* records, VidyoRoomSearchResult result);

/**
 {callback:
	{name: VidyoRoomOnSetFavoriteResult}
	{parent: VidyoRoom}
	{description: Callback that is triggered when server acknowleges a VidyoRoom's favorite status for current logged-in user has changed. }
	{prototype: void (*VidyoRoomOnSetFavoriteResult)(VidyoRoom* room, LmiBool isFavorite, VidyoRoomSetFavoriteResult result)}
	{parameter:
        {name: room}
        {description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
        {name: isFavorite}
        {description: }
	}
    {parameter:
        {name: result}
        {description: }
	}
 }
 */
typedef void (*VidyoRoomOnSetFavoriteResult)(VidyoRoom* room, LmiBool isFavorite, VidyoRoomSetFavoriteResult result);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnDeleteResult}
	{parent: VidyoRoom}
	{description: Callback that is triggered when server acknowleges a VidyoRoom has been deleted on the server. }
	{prototype: void (*VidyoRoomOnDeleteResult)(VidyoRoom* room, VidyoRoomDeleteResult result)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: result}
		{description: result of the operation to delete the room on the server. }
	}
}
*/
typedef void(*VidyoRoomOnDeleteResult)(VidyoRoom* room, VidyoRoomDeleteResult result);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnAvailableResourcesChanged}
	{parent: VidyoRoom}
	{description: Callback that is triggered when the available resources such as CPU and bandwidth change when media is enabled. If there is no enough bandwidth for sources bandwidthSend and bandwidthReceive will be 0.
		If has enough bandwidth for sources bandwidthSend and bandwidthReceive will be 1-70%. If has more bandwidth than required for sources bandwidthSend and bandwidthReceive will be 70-100%.}
	{prototype: void (*VidyoRoomOnAvailableResourcesChanged)(VidyoRoom* room, LmiUint cpuEncode, LmiUint cpuDecode, LmiUint bandwidthSend, LmiUint bandwidthReceive)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: cpuEncode}
		{description: Percent of maximum CPU available to encode the streams.}
	}
	{parameter:
		{name: cpuDecode}
		{description: Percent of maximum CPU available to decode the streams.}
	}
	{parameter:
		{name: bandwidthSend}
		{description: Percentage of maximum bandwidth avaialble to transmit the streams.}
	}
	{parameter:
		{name: bandwidthReceive}
		{description: Percentage of maximum bandwidth avaialble to receive the streams.}
	}
}
*/
typedef void(*VidyoRoomOnAvailableResourcesChanged)(VidyoRoom* room, LmiUint cpuEncode, LmiUint cpuDecode, LmiUint bandwidthSend, LmiUint bandwidthReceive);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnMaxRemoteSourcesChanged}
	{parent: VidyoRoom}
	{description: Callback that is triggered when maximum decoding capability changes when media is enabled. }
	{prototype: void (*VidyoRoomOnMaxRemoteSourcesChanged)(VidyoRoom* room, LmiUint maxRemoteSources)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: maxRemoteSources}
		{description: Maximum number of remote video sources that can be recived both statically and dynamically.}
	}
}
*/
typedef void(*VidyoRoomOnMaxRemoteSourcesChanged)(VidyoRoom* room, LmiUint maxRemoteSources);

/**
 {callback:
	{name: VidyoRoomOnInviteResult}
	{parent: VidyoRoom}
	{description: Callback that is triggered when server acknowleges a VidyoRoom invite has been sent to the user. }
	{prototype: void (*VidyoRoomOnInviteResult)(VidyoRoom* room, LmiString* inviteeId, VidyoRoomInviteResult result)}
	{parameter:
        {name: room}
        {description: Pointer to the VidyoRoom that triggered the callback. }
	}
    {parameter:
        {name: inviteeId}
        {description: Id of the user who was invited. }
	}
	{parameter:
        {name: result}
        {description: The number of received streams. }
	}
 }
 */
typedef void(*VidyoRoomOnInviteResult)(VidyoRoom* room, LmiString* inviteeId, VidyoRoomInviteResult result);

/**
{callback:
	{name: VidyoRoomOnLockRoomResult}
	{parent: VidyoRoom}
	{description: Callback that is triggered each tome when the result of lock room request is received.}
	{prototype: void(*VidyoRoomOnLockRoomResult)(VidyoRoom* room, const VidyoRoomSetPropertiesResult result)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: The result of lock room request.}
	}
}
*/
typedef void(*VidyoRoomOnLockRoomResult)(VidyoRoom* room, const VidyoRoomSetPropertiesResult result);

/**
{callback:
	{name: VidyoRoomOnUnlockRoomResult}
	{parent: VidyoRoom}
	{description: Callback that is triggered each tome when the result of unlock room request is received.}
	{prototype: void(*VidyoRoomOnUnlockRoomResult)(VidyoRoom* room, const VidyoRoomSetPropertiesResult result)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: The result of unlock room request.}
	}
}
*/
typedef void(*VidyoRoomOnUnlockRoomResult)(VidyoRoom* room, const VidyoRoomSetPropertiesResult result);

/**
{callback:
	{name: VidyoRoomOnSetRoomPINResult}
	{parent: VidyoRoom}
	{description: Callback that is triggered each time when result of set room pin is received.}
	{prototype: void(*VidyoRoomOnSetRoomPINResult)(VidyoRoom* room, const VidyoRoomSetPropertiesResult result)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: The result of set room pin Request.}
	}
}
*/
typedef void(*VidyoRoomOnSetRoomPINResult)(VidyoRoom* room, const VidyoRoomSetPropertiesResult result);

/**
{callback:
	{name: VidyoRoomOnRemoveRoomPINResult}
	{parent: VidyoRoom}
	{description: Callback that is triggered each time when result of remove room pin is received.}
	{prototype: void(*VidyoRoomOnRemoveRoomPINResult)(VidyoRoom* room, const VidyoRoomSetPropertiesResult result)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: The result of remove room pin Request.}
	}
}
*/
typedef void(*VidyoRoomOnRemoveRoomPINResult)(VidyoRoom* room, const VidyoRoomSetPropertiesResult result);

/**
 {callback:
	{name: VidyoRoomOnUnprocessedAudioSupportChanged}
	{parent: VidyoRoom}
	{description: Callback that is triggered when support of unprocessed audio is changed }
	{prototype: void (*VidyoRoomOnUnprocessedAudioSupportChanged)(VidyoRoom* room, LmiBool supported)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: supported}
		{description: unprocessed audio is supported or not. }
	}
 }
 */
typedef void(*VidyoRoomOnUnprocessedAudioSupportChanged)(VidyoRoom* room, LmiBool supported);

/**
 {callback:
	{name: VidyoRoomOnUnprocessedAudioStarted}
	{parent: VidyoRoom}
	{description: Callback that is triggered when support of unprocessed audio is changed }
	{prototype: void (*VidyoRoomOnUnprocessedAudioStarted)(VidyoRoom* room, LmiBool started)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: started}
		{description: unprocessed audio is started or not. }
	}
 }
*/
typedef void(*VidyoRoomOnUnprocessedAudioStarted)(VidyoRoom* room, LmiBool started);

/**
 {function apigroup="simple":
 {name: VidyoRoomRegisterUnprocessedAudioEventListener}
 	{parent: VidyoRoom}
 	{description: .}
 	{prototype: LmiBool VidyoRoomRegisterUnprocessedAudioEventListener(VidyoRoom* r, VidyoRoomOnUnprocessedAudioSupportChanged onUnprocessedAudioSupportChanged, VidyoRoomOnUnprocessedAudioStarted onUnprocessedAudioStarted)}
 	{parameter:
 		{name: r}
 		{description: The VidyoRoom object.}
 	}
 	{parameter:
 		{name: onUnprocessedAudioSupportChanged}
 		{description: .}
 	}
 	{parameter:
 		{name: onUnprocessedAudioStarted}
 		{description: .}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
*/
LmiBool VidyoRoomRegisterUnprocessedAudioEventListener(VidyoRoom* r, VidyoRoomOnUnprocessedAudioSupportChanged onUnprocessedAudioSupportChanged, VidyoRoomOnUnprocessedAudioStarted onUnprocessedAudioStarted);

/**
 {callback:
	{name: VidyoRoomOnGetSelectedLocalCamerasComplete}
	{parent: VidyoRoom}
	{description: Callback that is triggered when getting the selected local cameras completes.}
	{prototype: void (*VidyoRoomOnGetSelectedLocalCamerasComplete)(VidyoRoom* r, const LmiVector(VidyoLocalCamera)* cameras)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: cameras}
		{description: Array of selected local cameras.}
	}
}
*/
typedef void (*VidyoRoomOnGetSelectedLocalCamerasComplete)(VidyoRoom* r, const LmiVector(VidyoLocalCamera)* cameras);

/**
 {callback:
	{name: VidyoRoomOnGetSelectedLocalMicrophonesComplete}
	{parent: VidyoRoom}
	{description: Callback that is triggered when getting the selected local microphones completes.}
	{prototype: void (*VidyoRoomOnGetSelectedLocalMicrophonesComplete)(VidyoRoom* r, const LmiVector(VidyoLocalMicrophone)* microphones)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: microphones}
		{description: Array of selected local microphones.}
	}
}
*/
typedef void (*VidyoRoomOnGetSelectedLocalMicrophonesComplete)(VidyoRoom* r, const LmiVector(VidyoLocalMicrophone)* microphones);

/**
 {callback:
	{name: VidyoRoomOnGetSelectedLocalSpeakersComplete}
	{parent: VidyoRoom}
	{description: Callback that is triggered when getting the selected local speakers completes.}
	{prototype: void (*VidyoRoomOnGetSelectedLocalSpeakersComplete)(VidyoRoom* r, const LmiVector(VidyoLocalSpeaker)* speakers)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: speakers}
		{description: Array of selected local speakers.}
	}
}
*/
typedef void (*VidyoRoomOnGetSelectedLocalSpeakersComplete)(VidyoRoom* r, const LmiVector(VidyoLocalSpeaker)* speakers);

/**
 {callback:
	{name: VidyoRoomOnGetSelectedLocalRenderersComplete}
	{parent: VidyoRoom}
	{description: Callback that is triggered when getting the selected local renderers completes.}
	{prototype: void (*VidyoRoomOnGetSelectedLocalRenderersComplete)(VidyoRoom* r, const LmiVector(VidyoLocalRenderer)* renderers)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: renderers}
		{description: Array of selected local renderers.}
	}
}
*/
typedef void (*VidyoRoomOnGetSelectedLocalRenderersComplete)(VidyoRoom* r, const LmiVector(VidyoLocalRenderer)* renderers);

/**
 {callback:
	{name: VidyoRoomOnGetSelectedLocalWindowSharesComplete}
	{parent: VidyoRoom}
	{description: Callback that is triggered when getting the selected local window shares completes.}
	{prototype: void (*VidyoRoomOnGetSelectedLocalWindowSharesComplete)(VidyoRoom* r, const LmiVector(VidyoLocalWindowShare)* windowShares)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: windowShares}
		{description: Array of selected local window shares.}
	}
}
*/
typedef void (*VidyoRoomOnGetSelectedLocalWindowSharesComplete)(VidyoRoom* r, const LmiVector(VidyoLocalWindowShare)* windowShares);

/**
 {callback:
	{name: VidyoRoomOnGetSelectedLocalMonitorsComplete}
	{parent: VidyoRoom}
	{description: Callback that is triggered when getting the selected local monitors completes.}
	{prototype: void (*VidyoRoomOnGetSelectedLocalMonitorsComplete)(VidyoRoom* r, const LmiVector(VidyoLocalMonitor)* monitors)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: monitors}
		{description: Array of selected local monitors.}
	}
}
*/
typedef void (*VidyoRoomOnGetSelectedLocalMonitorsComplete)(VidyoRoom* r, const LmiVector(VidyoLocalMonitor)* monitors);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnConferenceModeChanged}
	{parent: VidyoRoom}
	{description: Callback that will be triggered when conference mode changes}
	{prototype: void(*VidyoRoomOnConferenceModeChanged)(VidyoRoom *r, VidyoRoomConferenceMode mode)}
	{parameter:
		{name: r}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: mode}
		{description: The state of vidyoconference i.e. VIDYO_ROOMMODE_LOBBY or VIDYO_ROOMMODE_LECTURE or VIDYO_ROOMMODE_GROUP}
	}
}
*/
typedef void(*VidyoRoomOnConferenceModeChanged)(VidyoRoom *r, VidyoRoomConferenceMode mode);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnConferenceHostStatusChanged}
	{parent: VidyoRoom}
	{description: Callback that will be triggered when conference host joins or leaves}
	{prototype:void(*VidyoRoomOnConferenceHostStatusChanged)(VidyoRoom* r, const VidyoParticipant* host, VidyoRoomConferenceHostState hostState)}
	{parameter:
		{name: r}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: host}
		{description: The participant object for conference host}
	}
	{parameter:
		{name: hostState}
		{description:  whether the host is leaving or joining the conference}
	}
}
*/
typedef void(*VidyoRoomOnConferenceHostStatusChanged)(VidyoRoom* r, const VidyoParticipant* host, VidyoRoomConferenceHostState hostState);

/**
 {function apigroup="simple":
 	{name: VidyoRoomUnregisterUnprocessedAudioEventListener}
 	{parent: VidyoRoom}
 	{description: .}
 	{prototype: LmiBool VidyoRoomUnregisterUnprocessedAudioEventListener(VidyoRoom* r)}
 	{parameter:
 		{name: r}
 		{description: The VidyoRoom object.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
*/
LmiBool VidyoRoomUnregisterUnprocessedAudioEventListener(VidyoRoom* r);

/**
{callback:
	{name: VidyoRoomOnPipLocationChanged}
	{parent: VidyoRoom}
	{description: Callback that is triggerred when picture-in-picture view size is changed.}
	{prototype: void VidyoRoomOnPipLocationChanged(VidyoRoom* room, LmiInt x, LmiInt y, LmiUint width, LmiUint height, LmiBool isExpanded)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: x}
		{description: The horizontal position of the picture-in-picture view.}
	}
	{parameter:
		{name: y}
		{description: The vertical position of the picture-in-picture view.}
	}
	{parameter:
		{name: width}
		{description: Width of the picture-in-picture view.}
	}
	{parameter:
		{name: height}
		{description: Height of the picture-in-picture view.}
	}
	{parameter:
		{name: isExpanded}
		{description: LMI_TRUE if picture-in-picture is expanded and LMI_FALSE if collapsed.}
	}
}
*/
typedef void (*VidyoRoomOnPipLocationChanged)(VidyoRoom* room, LmiInt x, LmiInt y, LmiUint width, LmiUint height, LmiBool isExpanded);

/**
 {callback apigroup="simple":
	{name: VidyoRoomOnRecordingServiceStartResult}
	{parent: VidyoRoom}
	{description: Callback that is triggered upon completion of a recording service start request.}
	{prototype: void(*VidyoRoomOnRecordingServiceStartResult)(VidyoRoom* r, VidyoRoomSetPropertiesResult result)}
	{parameter:
		{name: r}
		{description: Pointer to the VidyoRoom that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: Result of the request. Can be one of VidyoRoomSetPropertiesResult values.}
	}
}
*/
typedef void(*VidyoRoomOnRecordingServiceStartResult)(VidyoRoom* r, VidyoRoomSetPropertiesResult result);

/**
 {callback apigroup="simple":
	{name: VidyoRoomOnRecordingServiceStopResult}
	{parent: VidyoRoom}
	{description: Callback that is triggered upon completion of a recording service stop request.}
	{prototype: void(*VidyoRoomOnRecordingServiceStopResult)(VidyoRoom* r, VidyoRoomSetPropertiesResult result)}
	{parameter:
		{name: r}
		{description: Pointer to the VidyoRoom that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: Result of the request. Can be one of VidyoRoomSetPropertiesResult values.}
	}
}
*/
typedef void(*VidyoRoomOnRecordingServiceStopResult)(VidyoRoom* r, VidyoRoomSetPropertiesResult result);

/**
 {callback apigroup="simple":
	{name: VidyoRoomOnRecordingServicePauseResult}
	{parent: VidyoRoom}
	{description: Callback that is triggered upon completion of a recording service pause request.}
	{prototype: void(*VidyoRoomOnRecordingServicePauseResult)(VidyoRoom* r, VidyoRoomSetPropertiesResult result)}
	{parameter:
		{name: r}
		{description: Pointer to the VidyoRoom that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: Result of the request. Can be one of VidyoRoomSetPropertiesResult values.}
	}
}
*/
typedef void(*VidyoRoomOnRecordingServicePauseResult)(VidyoRoom* r, VidyoRoomSetPropertiesResult result);

/**
 {callback apigroup="simple":
	{name: VidyoRoomOnRecordingServiceResumeResult}
	{parent: VidyoRoom}
	{description: Callback that is triggered upon completion of a recording service resume request.}
	{prototype: void(*VidyoRoomOnRecordingServiceResumeResult)(VidyoRoom* r, VidyoRoomSetPropertiesResult result)}
	{parameter:
		{name: r}
		{description: Pointer to the VidyoRoom that triggered the callback.}
	}
	{parameter:
		{name: result}
		{description: Result of the request. Can be one of VidyoRoomSetPropertiesResult values.}
	}
}
*/
typedef void(*VidyoRoomOnRecordingServiceResumeResult)(VidyoRoom* r, VidyoRoomSetPropertiesResult result);

/**
 {function apigroup="simple":
 	{name: VidyoRoomRegisterPipEventListener}
 	{parent: VidyoRoom}
 	{description: Registers to get notified when a a call is being recorded.}
 	{prototype: LmiBool VidyoRoomRegisterPipEventListener(VidyoRoom* r, VidyoRoomOnPipLocationChanged onPipLocationChanged)}
 	{parameter:
 		{name: r}
 		{description: The VidyoRoom object.}
 	}
 	{parameter:
 		{name: onPipLocationChanged}
 		{description: .}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
*/
LmiBool VidyoRoomRegisterPipEventListener(VidyoRoom* r, VidyoRoomOnPipLocationChanged onPipLocationChanged);

/**
 {function apigroup="simple":
 	{name: VidyoRoomUnregisterPipEventListener}
 	{parent: VidyoRoom}
 	{description: Unregisters recorder in call event notifications.}
 	{prototype: LmiBool VidyoRoomUnregisterPipEventListener(VidyoRoom* r)}
	{parameter:
 		{name: r}
 		{description: The VidyoRoom object.}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
*/
LmiBool VidyoRoomUnregisterPipEventListener(VidyoRoom* r);


LmiBool VidyoRoomInitialize(void);

/**
{function visibility="private":
	{name: VidyoRoomUninitialize}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomUninitialize(void)}
}
*/
void VidyoRoomUninitialize(void);

/**
{type:
	{name: VidyoRoomState}
	{parent: VidyoRoom}
	{description: }
	{value: {name: VIDYO_ROOMSTATE_Creating} {description: }}
	{value: {name: VIDYO_ROOMSTATE_Created} {description: }}
	{value: {name: VIDYO_ROOMSTATE_Entering} {description: }}
	{value: {name: VIDYO_ROOMSTATE_Entered} {description: }}
	{value: {name: VIDYO_ROOMSTATE_MediaRouteAcquiring} {description: }}
	{value: {name: VIDYO_ROOMSTATE_MediaRouteAcquired} {description: }}
	{value: {name: VIDYO_ROOMSTATE_MediaEnabling} {description: }}
	{value: {name: VIDYO_ROOMSTATE_MediaEnabled} {description: }}
	{value: {name: VIDYO_ROOMSTATE_MediaDisabling} {description: }}
	{value: {name: VIDYO_ROOMSTATE_MediaDisabled} {description: }}
	{value: {name: VIDYO_ROOMSTATE_Exiting} {description: }}
	{value: {name: VIDYO_ROOMSTATE_Exited} {description: }}
}
*/
typedef enum {
	VIDYO_ROOMSTATE_Creating,
	VIDYO_ROOMSTATE_Created,
	VIDYO_ROOMSTATE_Entering,
	VIDYO_ROOMSTATE_Entered,
	VIDYO_ROOMSTATE_MediaRouteAcquiring,
	VIDYO_ROOMSTATE_MediaRouteAcquired,
	VIDYO_ROOMSTATE_MediaEnabling,
	VIDYO_ROOMSTATE_MediaEnabled,
	VIDYO_ROOMSTATE_MediaDisabling,
	VIDYO_ROOMSTATE_MediaDisabled,
	VIDYO_ROOMSTATE_Exiting,
	VIDYO_ROOMSTATE_Exited
} VidyoRoomState;


/**
{function apigroup="simple":
	{name: VidyoRoomConstructCopy}
	{parent: VidyoRoom}
	{description: Constructs an VidyoRoom object as a copy of another.}
	{prototype: VidyoRoom *VidyoRoomConstructCopy(VidyoRoom *obj, const VidyoRoom *other)}
	{parameter: {name: obj} {description: The VidyoRoom object to construct.}}
	{parameter: {name: other} {description: The call object to make a copy of.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/

/**
{function apigroup="simple":
	{name: VidyoRoomDestruct}
	{parent: VidyoRoom}
	{description: Destructs an VidyoRoom object.}
	{prototype: void VidyoRoomDestruct(VidyoRoom *obj)}
	{parameter: {name: obj} {description: The VidyoRoom object to destruct.}}
}
*/

/**
{function visibility="private":
	{name: VidyoRoomAssign}
	{parent: VidyoRoom}
	{description: Assigns one VidyoRoom object the value of another. }
	{prototype: VidyoRoom *VidyoRoomAssign(VidyoRoom *obj, const VidyoRoom *other)}
	{parameter: {name: obj} {description: The VidyoRoom object.}}
	{parameter: {name: other} {description: The call object to make a copy of.}}
	{return: A pointer to the destination object on success, or NULL on failure.}
}
*/

/**
{function visibility="private":
	{name: VidyoRoomGetHandle}
	{parent: VidyoRoom}
	{description: }
	{prototype: const VidyoRoomHandle* VidyoRoomGetHandle(const VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: }
	}
	{return: }
}
*/
const VidyoRoomHandle* VidyoRoomGetHandle(const VidyoRoom* r);

/**
{function visibility="private":
	{name: VidyoRoomSetUserData}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomSetUserData(VidyoRoom* r, LmiVoidPtr userData)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: userData}
		{description: }
	}
}
*/
void VidyoRoomSetUserData(VidyoRoom* r, LmiVoidPtr userData);

/**
{function visibility="private":
	{name: VidyoRoomGetUserData}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiVoidPtr VidyoRoomGetUserData(const VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: }
	}
	{return: }
}
*/
LmiVoidPtr VidyoRoomGetUserData(const VidyoRoom* r);

/**
 {function apigroup="simple":
	{name: VidyoRoomRegisterParticipantEventListener}
	{parent: VidyoRoom}
	{description: Registers to get notified about participant events.}
	{prototype: LmiBool VidyoRoomRegisterParticipantEventListener(VidyoRoom* r, VidyoRoomOnParticipantJoined onParticipantJoined, VidyoRoomOnParticipantLeft onParticipantLeft, VidyoRoomOnDynamicParticipantChanged onDynamicParticipantChanged, VidyoRoomOnLoudestParticipantChanged onLoudestParticipantChanged)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
 	{parameter:
		{name: onParticipantJoined}
		{description: Callback that is triggered when another participant joins a conference.}
	}
	{parameter:
		{name: onParticipantLeft}
		{description: Callback that is triggered when an existing participant leaves a conference.}
	}
	{parameter:
		{name: onDynamicParticipantChanged}
		{description: Callback that is triggered when the order of participants has changed, based on their importance according to active speech detection.}
	}
	{parameter:
		{name: onLoudestParticipantChanged}
		{description: Callback that is triggered when a new participant becomes the loudest, based on active speech detection.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomRegisterParticipantEventListener(VidyoRoom* r, VidyoRoomOnParticipantJoined onParticipantJoined, VidyoRoomOnParticipantLeft onParticipantLeft, VidyoRoomOnDynamicParticipantChanged onDynamicParticipantChanged, VidyoRoomOnLoudestParticipantChanged onLoudestParticipantChanged);

/**
 {function apigroup="simple":
 	{name: VidyoRoomReportLocalParticipantOnJoined}
 	{parent: VidyoRoom}
 	{description: By default, local participant joining/leaving conference are not reported to applicaiotn. Use this function to report those events.}
 	{prototype: LmiBool VidyoRoomReportLocalParticipantOnJoined(VidyoRoom* r, LmiBool reportLocalParticipant)}
 	{parameter:
 		{name: r}
 		{description: The VidyoRoom object.}
 	}
 	{parameter:
 		{name: reportLocalParticipant}
 		{description: Set to LMI_TRUE if application is interested in local participant to be included in participant event callbacks.
 			Otherwise set to LMI_FALSE. Default is LMI_FALSE}
 	}
 	{return: LMI_TRUE on success, LMI_FALSE on failure.}
 }
*/
LmiBool VidyoRoomReportLocalParticipantOnJoined(VidyoRoom* r, LmiBool reportLocalParticipant);

/**
 {function apigroup="simple":
	{name: VidyoRoomUnregisterParticipantEventListener}
	{parent: VidyoRoom}
	{description: Unregisters participant event notifications.}
	{prototype: LmiBool VidyoRoomUnregisterParticipantEventListener(VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
	}
*/
LmiBool VidyoRoomUnregisterParticipantEventListener(VidyoRoom* r);

/**
 {function apigroup="simple":
	{name: VidyoRoomRegisterResourceManagerEventListener}
	{parent: VidyoRoom}
	{description: Registers to get notified about resource-management events.}
	{prototype: LmiBool VidyoRoomRegisterResourceManagerEventListener(VidyoRoom* r, VidyoRoomOnAvailableResourcesChanged onAvailableResourcesChanged, VidyoRoomOnMaxRemoteSourcesChanged onMaxRemoteSourcesChanged)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: onAvailableResourcesChanged}
		{description: Callback that is triggered when local resource availability changes.}
	}
	{parameter:
		{name: onMaxRemoteSourcesChanged}
		{description: Callback that is triggered when the maximum number of decoded sources changes.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomRegisterResourceManagerEventListener(VidyoRoom* r, VidyoRoomOnAvailableResourcesChanged onAvailableResourcesChanged, VidyoRoomOnMaxRemoteSourcesChanged onMaxRemoteSourcesChanged);

/**
 {function apigroup="simple":
	{name: VidyoRoomUnregisterResourceManagerEventListener}
	{parent: VidyoRoom}
	{description: Unregisters resource manager event notifications.}
	{prototype: LmiBool VidyoRoomUnregisterResourceManagerEventListener(VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomUnregisterResourceManagerEventListener(VidyoRoom* r);

/**
{function apigroup="simple":
	{name: VidyoRoomGetId}
	{parent: VidyoRoom}
	{description: }
	{prototype: const LmiString* VidyoRoomGetId(const VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: }
	}
	{return: }
}
*/
const LmiString* VidyoRoomGetId(const VidyoRoom* r);

/**
{function apigroup="simple":
	{name: VidyoRoomGetType}
	{parent: VidyoRoom}
	{description: }
	{prototype: VidyoRoomType VidyoRoomGetType(const VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: }
	}
	{return: }
}
*/
VidyoRoomType VidyoRoomGetType(const VidyoRoom* r);

/**
{function:
	{name: VidyoRoomGetUser}
	{parent: VidyoRoom}
	{description: }
	{prototype: VidyoUser *VidyoRoomGetUser(VidyoRoom *r)}
	{parameter:
		{name: r}
		{description: }
	}
	{return: }
}
*/
VidyoUser *VidyoRoomGetUser(VidyoRoom *r);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnIncomingEntered}
	{parent: VidyoRoom}
	{description: Callback that is triggered when entering the VidyoRoom as a result of VidyoRoomAcceptIncoming() request.}
	{prototype: void (*VidyoRoomOnIncomingEntered)(VidyoRoom* room, VidyoRoomEnterResult result)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: result}
		{description: Result of an attempt to enter the VidyoRoom. }
	}
}
*/
typedef void (*VidyoRoomOnIncomingEntered)(VidyoRoom* room, VidyoRoomEnterResult result);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnIncomingExited}
	{parent: VidyoRoom}
	{description: Callback that is triggered when the VidyoUser exited the VidyoRoom or as a result of VidyoRoomLeave() request. }
	{prototype: void (*VidyoRoomOnIncomingExited)(VidyoRoom* room, VidyoRoomExitReason reason)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: reason}
		{description: Reason why VidyoUser exited VidyoRoom. }
	}
}
*/
typedef void(*VidyoRoomOnIncomingExited)(VidyoRoom* room, VidyoRoomExitReason reason);

/**
{function apigroup="simple":
	{name: VidyoRoomAcceptIncoming}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomAcceptIncoming(VidyoRoom* r, VidyoRoomOnIncomingEntered onIncomingEntered, VidyoRoomOnIncomingExited onIncomingExited)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: onIncomingEntered}
		{description: Callback that is triggered when the enter operation completes. This could be a successful enter or a failure due to permissions, etc.}
	}
	{parameter:
		{name: onIncomingExited}
		{description: Callback that is triggered when the room has been exited.}
	}
	{return: }
}
*/
LmiBool VidyoRoomAcceptIncoming(VidyoRoom* r, VidyoRoomOnIncomingEntered onIncomingEntered, VidyoRoomOnIncomingExited onIncomingExited);

/**
{function apigroup="simple":
	{name: VidyoRoomRejectIncoming}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomRejectIncoming(VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoRoomRejectIncoming(VidyoRoom* r);

/**
{function apigroup="simple":
	{name: VidyoRoomLeave}
	{parent: VidyoRoom}
	{description: Leaves the room. This triggers VidyoRoomOnExited.}
	{prototype: LmiBool VidyoRoomLeave(VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoRoomLeave(VidyoRoom* r);

/**
{function apigroup="simple":
	{name: VidyoRoomEnter}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomEnter(VidyoRoom *r, const char* password, VidyoRoomOnEntered onEntered, VidyoRoomOnExited onExited)}
	{parameter:
		{name: r}
		{description: }
	}
    {parameter:
        {name: password}
        {description: }
	}
	{parameter:
		{name: onEntered}
		{description: Callback that is triggered when the enter operation completes. This could be a successful enter or a failure due to permissions, etc.}
	}
	{parameter:
		{name: onExited}
		{description: Callback that is triggered when the room has been exited.}
	}
	{return: }
}
*/
LmiBool VidyoRoomEnter(VidyoRoom *r, const char* password, VidyoRoomOnEntered onEntered, VidyoRoomOnExited onExited);

/**
 {function apigroup="simple":
	{name: VidyoRoomRegisterRoomPropertiesEventListener}
	{parent: VidyoRoom}
	{description: Registers to get notified about participant events.}
	{prototype: LmiBool VidyoRoomRegisterRoomPropertiesEventListener(VidyoRoom* r, VidyoRoomOnRoomPropertiesChanged onRoomPropertiesChanged)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: onRoomPropertiesChanged}
		{description: Callback that is triggered when room properties have changed.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomRegisterRoomPropertiesEventListener(VidyoRoom* r, VidyoRoomOnRoomPropertiesChanged onRoomPropertiesChanged);

/**
 {function apigroup="simple":
	{name: VidyoRoomUnregisterRoomPropertiesEventListener}
	{parent: VidyoRoom}
	{description: Unregisters room properties event notifications.}
	{prototype: LmiBool VidyoRoomUnregisterRoomPropertiesEventListener(VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomUnregisterRoomPropertiesEventListener(VidyoRoom* r);

/**
{function apigroup="simple":
	{name: VidyoRoomGetRoomProperties}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomGetRoomProperties(VidyoRoom* r, VidyoRoomOnGetRoomPropertiesResult onGetRoomPropertiesResult)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: onGetRoomPropertiesResult}
		{description: Callback that is triggered when get properties operation completes.}
	}
	{return: }
}
*/
LmiBool VidyoRoomGetRoomProperties(VidyoRoom* r, VidyoRoomOnGetRoomPropertiesResult onGetRoomPropertiesResult);

/**
{function apigroup="simple":
	{name: VidyoRoomSetRoomProperties}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomSetRoomProperties(VidyoRoom* r, const VidyoRoomProperties *roomProperties, VidyoRoomOnSetRoomPropertiesResult onSetRoomPropertiesResult)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: roomProperties}
		{description: }
	}
	{parameter:
		{name: onSetRoomPropertiesResult}
		{description: Callback that is triggered when set properties operation completes.}
	}
	{return: }
}
*/
LmiBool VidyoRoomSetRoomProperties(VidyoRoom* r, const VidyoRoomProperties *roomProperties, VidyoRoomOnSetRoomPropertiesResult onSetRoomPropertiesResult);

/**
{function:
	{name: VidyoRoomAcquireMediaRoute}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomAcquireMediaRoute(VidyoRoom* r, VidyoRoomOnMediaRouteAcquired onMediaRouteAcquired, VidyoRoomOnMediaRouteAcquireFailed onMediaRouteAcquireFailed)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: onMediaRouteAcquired}
		{description: Callback that is triggered when the acquire media route operation is complete.}
	}
 	{parameter:
 		{name: onMediaRouteAcquireFailed}
 		{description: Callback that is triggered when the could not be enabled.}
 	}
	{return: }
}
*/
LmiBool VidyoRoomAcquireMediaRoute(VidyoRoom *r, VidyoRoomOnMediaRouteAcquired onMediaRouteAcquired, VidyoRoomOnMediaRouteAcquireFailed onMediaRouteAcquireFailed);

/**
{function apigroup="simple":
	{name: VidyoRoomEnableMedia}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomEnableMedia(VidyoRoom* r, VidyoRoomOnMediaEnabled onMediaEnabled, VidyoRoomOnMediaFailed onMediaFailed, VidyoRoomOnMediaDisabled onMediaDisabled)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: onMediaEnabled}
		{description: Callback that is triggered when the media has been successfully enabled.}
	}
	{parameter:
		{name: onMediaFailed}
		{description: Callback that is triggered when the could not be enabled.}
	}
	{parameter:
		{name: onMediaDisabled}
		{description: Callback that is triggered when the media has been disabled after being previously enabled successfully.}
	}
	{return: }
}
*/
LmiBool VidyoRoomEnableMedia(VidyoRoom *r, VidyoRoomOnMediaEnabled onMediaEnabled, VidyoRoomOnMediaFailed onMediaFailed, VidyoRoomOnMediaDisabled onMediaDisabled);

/**
{function apigroup="simple":
	{name: VidyoRoomDisableMedia}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomDisableMedia(VidyoRoom* r)}
	{parameter:
		{name: r}
        {description: The room object.}
	}
	{return: }
}
*/
LmiBool VidyoRoomDisableMedia(VidyoRoom* r);

/**
{function:
	{name: VidyoRoomSetMaxSendBitRate}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomSetMaxSendBitRate(VidyoRoom *r, LmiUint bitRate)}
	{parameter:
		{name: r}
        {description: The room object.}
	}
    {parameter:
        {name: bitRate}
        {description: The maximum send  bit rate request sent to server, in bps.}
	}
}
*/
void VidyoRoomSetMaxSendBitRate(VidyoRoom *r, LmiUint bitRate);

/**
{function:
    {name: VidyoRoomSetMaxReceiveBitRate}
    {parent: VidyoRoom}
    {description: }
    {prototype: void VidyoRoomSetMaxReceiveBitRate(VidyoRoom *r, LmiUint bitRate)}
    {parameter:
        {name: r}
        {description: The room object.}
    }
    {parameter:
        {name: bitRate}
        {description: The maximum receiving bit rate request sent to server, in bps.}
    }
}
*/
void VidyoRoomSetMaxReceiveBitRate(VidyoRoom *r, LmiUint bitRate);

/**
{function:
	{name: VidyoRoomSetMaxAudioSources}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomSetMaxAudioSources(VidyoRoom* r, LmiUint count)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: count}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoRoomSetMaxAudioSources(VidyoRoom* r, LmiUint count);

/**
{function:
	{name: VidyoRoomSetMaxVideoSources}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomSetMaxVideoSources(VidyoRoom* r, LmiUint count)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: count}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoRoomSetMaxVideoSources(VidyoRoom* r, LmiUint count);

/**
{function:
	{name: VidyoRoomSetDynamicLocalRenderer}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomSetDynamicLocalRenderer(VidyoRoom* r, const VidyoLocalRenderer* renderer)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: renderer}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoRoomSetDynamicLocalRenderer(VidyoRoom *r, const VidyoLocalRenderer* renderer);

/**
{function:
	{name: VidyoRoomSetDynamicGenerationPosition}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomSetDynamicGenerationPosition(VidyoRoom *r, LmiUint generation, LmiInt x, LmiInt y, LmiUint width, LmiUint height, LmiTime frameInterval)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: generation}
		{description: }
	}
	{parameter:
		{name: x}
		{description: }
	}
	{parameter:
		{name: y}
		{description: }
	}
	{parameter:
		{name: width}
		{description: }
	}
	{parameter:
		{name: height}
		{description: }
	}
	{parameter:
		{name: frameInterval}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoRoomSetDynamicGenerationPosition(VidyoRoom *r, LmiUint generation, LmiInt x, LmiInt y, LmiUint width, LmiUint height, LmiTime frameInterval);

/**
 {function apigroup="simple":
	{name: VidyoRoomRegisterSubjectEventListener}
	{parent: VidyoRoom}
	{description: Registers to get notified about subject events.}
	{prototype: LmiBool VidyoRoomRegisterSubjectEventListener(VidyoRoom* r, VidyoRoomOnSetSubject onSetSubject)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: onSetSubject}
		{description: Callback that is triggered when the subject has been changed.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomRegisterSubjectEventListener(VidyoRoom* r, VidyoRoomOnSetSubject onSetSubject);

/**
 {function apigroup="simple":
	{name: VidyoRoomUnregisterSubjectEventListener}
	{parent: VidyoRoom}
	{description: Unregisters subject event notifications.}
	{prototype: LmiBool VidyoRoomUnregisterSubjectEventListener(VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomUnregisterSubjectEventListener(VidyoRoom* r);

/**
{function apigroup="simple":
	{name: VidyoRoomSetSubject}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomSetSubject(VidyoRoom* r, const char* subject)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: subject}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoRoomSetSubject(VidyoRoom* r, const char* subject);

/**
 {function apigroup="simple":
	{name: VidyoRoomRegisterMessageEventListener}
	{parent: VidyoRoom}
	{description: Registers to get notified about message events.}
	{prototype: LmiBool VidyoRoomRegisterMessageEventListener(VidyoRoom* r, VidyoRoomOnMessageReceived onMessageReceived, VidyoRoomOnMessageAcknowledged onMessageAcknowledged, VidyoRoomOnMessageRead onMessageRead, VidyoRoomOnMessageTypingIndication onMessageTypingIndication)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: onMessageReceived}
		{description: Callback that is triggered when a new message has been received.}
	}
	{parameter:
		{name: onMessageAcknowledged}
		{description: Callback that is triggered when a sent message has been acknowledged.}
	}
	{parameter:
		{name: onMessageRead}
		{description: Callback that is triggered when a sent message has been read.}
	}
	{parameter:
		{name: onMessageTypingIndication}
		{description: Callback that is triggered when a remote participant is typing.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomRegisterMessageEventListener(VidyoRoom* r, VidyoRoomOnMessageReceived onMessageReceived, VidyoRoomOnMessageAcknowledged onMessageAcknowledged, VidyoRoomOnMessageRead onMessageRead, VidyoRoomOnMessageTypingIndication onMessageTypingIndication);

/**
 {function apigroup="simple":
	{name: VidyoRoomUnregisterMessageEventListener}
	{parent: VidyoRoom}
	{description: Unregisters message event notifications.}
	{prototype: LmiBool VidyoRoomUnregisterMessageEventListener(VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomUnregisterMessageEventListener(VidyoRoom* r);

/**
 {callback apigroup="simple" deprecated="yes" replaced-by="VidyoRoomOnConnectionPropertiesChanged":
	{name: VidyoRoomRecorderInCall}
	{parent: VidyoRoom}
	{description: Callback that is triggered when the call is recorded or when call recording stops.}
	{prototype: void (*VidyoRoomRecorderInCall)(VidyoRoom* r, LmiBool hasRecorder, LmiBool isPaused )}
	{parameter:
		{name: r}
		{description: The VidyoRoom that triggered the callback.}
	}
	{parameter:
		{name: hasRecorder}
		{description: LMI_TRUE when the call is recorded, LMI_FALSE when recording stops}
	}
    {parameter:
        {name: isPaused}
        {description: LMI_TRUE when a recorder joined but is paused, LMI_FALSE otherwise}
    }
}
*/
typedef void(*VidyoRoomRecorderInCall)(VidyoRoom* r, LmiBool hasRecorder, LmiBool isPaused);

/**
 {callback apigroup="simple":
	{name: VidyoRoomOnConnectionPropertiesChanged}
	{parent: VidyoRoom}
	{description: Callback that is triggered each time when connection properties is changed.}
	{prototype: void (*VidyoRoomOnConnectionPropertiesChanged)(VidyoRoom* r, const VidyoRoomConnectionProperties* connectionProperties)}
	{parameter:
		{name: r}
		{description: The VidyoRoom that triggered the callback.}
	}
	{parameter:
		{name: connectionProperties}
		{description: VidyoRoomConnectionProperties object.}
	}
}
*/
typedef void (*VidyoRoomOnConnectionPropertiesChanged)(VidyoRoom* r, const VidyoRoomConnectionProperties* connectionProperties);

/**
 {function apigroup="simple" deprecated="yes" replaced-by="VidyoRoomRegisterConnectionPropertiesEventListener":
	{name: VidyoRoomRegisterRecorderInCallEventListener}
	{parent: VidyoRoom}
	{description: Registers to get notified when a a call is being recorded.}
	{prototype: LmiBool VidyoRoomRegisterRecorderInCallEventListener(VidyoRoom* r, VidyoRoomRecorderInCall onRecorderInCallChanged)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: onRecorderInCallChanged}
		{description: Callback that is triggered when a recorder joins or leaves the call.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomRegisterRecorderInCallEventListener(VidyoRoom* r, VidyoRoomRecorderInCall onRecorderInCallChanged );

/**
 {function apigroup="simple" deprecated="yes" replaced-by="VidyoRoomUnregisterConnectionPropertiesEventListener":
	{name: VidyoRoomUnregisterRecorderInCallEventListener}
	{parent: VidyoRoom}
	{description: Unregisters recorder in call event notifications.}
	{prototype: LmiBool VidyoRoomUnregisterRecorderInCallEventListener(VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomUnregisterRecorderInCallEventListener(VidyoRoom* r);

/**
{function apigroup="simple" visibility="private":
	{name: VidyoRoomConnectionPropertiesConstructDefault}
	{parent: VidyoRoomConnectionProperties}
	{description: Default construct for VidyoRoomConnectionProperties.}
	{prototype: VidyoRoomConnectionProperties* VidyoRoomConnectionPropertiesConstructDefault(VidyoRoomConnectionProperties* connectionProperties, LmiAllocator* alloc)}
	{parameter:
		{name: connectionProperties}
		{description: VidyoRoomConnectionProperties object.}
	}
	{parameter:
		{name: alloc}
		{description: LmiAllocator object.}
	}
	{return: VidyoRoomConnectionProperties object on success otherwise NULL.}
}
*/
VidyoRoomConnectionProperties* VidyoRoomConnectionPropertiesConstructDefault(VidyoRoomConnectionProperties* connectionProperties, LmiAllocator* alloc);

/**
{function apigroup="simple" visibility="private":
	{name: VidyoRoomConnectionPropertiesConstructCopy}
	{parent: VidyoRoomConnectionProperties}
	{description: Constructs an VidyoRoomConnectionProperties object as a copy of an existing objec.}
	{prototype: VidyoRoomConnectionProperties* VidyoRoomConnectionPropertiesConstructCopy(VidyoRoomConnectionProperties* d, VidyoRoomConnectionProperties* s)}
	{parameter:
		{name: d}
		{description: The VidyoRoomConnectionProperties object to construct.}
	}
	{parameter:
		{name: s}
		{description: The VidyoRoomConnectionProperties object to make a copy of.}
	}
	{return: VidyoRoomConnectionProperties object on success otherwise NULL.}
}
*/
VidyoRoomConnectionProperties* VidyoRoomConnectionPropertiesConstructCopy(VidyoRoomConnectionProperties* d, VidyoRoomConnectionProperties* s);

/**
{function apigroup="simple" visibility="private":
	{name: VidyoRoomConnectionPropertiesDestruct}
	{parent: VidyoRoomConnectionProperties}
	{description: Destructor for VidyoRoomConnectionProperties.}
	{prototype: void VidyoRoomConnectionPropertiesDestruct(VidyoRoomConnectionProperties* connectionProperties)}
	{parameter:
		{name: connectionProperties}
		{description: VidyoRoomConnectionProperties object.}
	}
}
*/
void VidyoRoomConnectionPropertiesDestruct(VidyoRoomConnectionProperties* connectionProperties);

/**
 {function apigroup="simple":
	{name: VidyoRoomRegisterConnectionPropertiesEventListener}
	{parent: VidyoRoom}
	{description: Registers to get notified when a connection properties is changed.}
	{prototype: LmiBool VidyoRoomRegisterConnectionPropertiesEventListener(VidyoRoom* r, VidyoRoomOnConnectionPropertiesChanged onConnectionPropertiesChanged)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: onConnectionPropertiesChanged}
		{description: Callback that is triggered when a connection properties is changed.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomRegisterConnectionPropertiesEventListener(VidyoRoom* r, VidyoRoomOnConnectionPropertiesChanged onConnectionPropertiesChanged);

/**
 {function apigroup="simple":
	{name: VidyoRoomUnregisterConnectionPropertiesEventListener}
	{parent: VidyoRoom}
	{description: Unregisters event listener, application will not receive an event about connection properties changes.}
	{prototype: LmiBool VidyoRoomUnregisterConnectionPropertiesEventListener(VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomUnregisterConnectionPropertiesEventListener(VidyoRoom* r);

/**
{function apigroup="simple":
	{name: VidyoRoomSendMessage}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomSendMessage(VidyoRoom* r, const char* message)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: message}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoRoomSendMessage(VidyoRoom* r, const char* message);

/**
{function apigroup="simple":
	{name: VidyoRoomSendPrivateMessage}
	{parent: VidyoRoom}
	{description: Send a private message. Only specific participant will receive this.}
	{prototype: LmiBool VidyoRoomSendPrivateMessage(VidyoRoom* r, const VidyoParticipant* participant, const char* message)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: participant}
		{description: The participant which will receive message.}
	}
	{parameter:
		{name: message}
		{description: Message that need to be sent.}
	}
	{return: }
}
*/
LmiBool VidyoRoomSendPrivateMessage(VidyoRoom* r, const VidyoParticipant* participant, const char* message);

/**
{function apigroup="simple":
	{name: VidyoRoomSendMessageAcknowledged}
	{parent: VidyoRoom}
	{description: Send the acknowledgement to message in a conference room.}
	{prototype: LmiBool VidyoRoomSendMessageAcknowledged(VidyoRoom *r, LmiInt64 messageId)}
	{parameter: {name: r} {description: The room object.}}
	{parameter: {name: messageId} {description: The message id of message being acknowledged.}}
	{return: LMI_TRUE on success, LMI_FALSE if the server doesn't support the feature.}
	{note: This function is used to acknowledge all the messages until the curser specified by messageId.}
}
*/
LmiBool VidyoRoomSendMessageAcknowledged(VidyoRoom *r, LmiInt64 messageId);

/**
{function apigroup="simple":
	{name: VidyoRoomSendMessageTypingIndication}
	{parent: VidyoRoom}
	{description: Send a typing indication state to all participants of a room.}
	{prototype: LmiBool VidyoRoomSendMessageTypingIndication(VidyoRoom *r, VidyoChatMessageTypingIndication typingIndication)}
	{parameter: {name: r} {description: The room object.}}
	{parameter: {name: typingIndication} {description: The typing indication state.}}
	{return: LMI_TRUE on success, LMI_FALSE on an error.}
}
*/
LmiBool VidyoRoomSendMessageTypingIndication(VidyoRoom *r, VidyoChatMessageTypingIndication typingIndication);

/**
{function apigroup="simple":
	{name: VidyoRoomInvite}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomInvite(VidyoRoom* r, const LmiString* userId, const char* message, VidyoRoomOnInviteResult onInviteResult)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: userId}
		{description: }
	}
	{parameter:
		{name: message}
		{description: }
	}
 	{parameter:
 		{name: onInviteResult}
 		{description: Callback that is triggered when another participant joins a conference.}
 	}
	{return: }
}
*/
LmiBool VidyoRoomInvite(VidyoRoom* r, const LmiString* userId, const char* message, VidyoRoomOnInviteResult onInviteResult);

/**
{function apigroup="simple":
	{name: VidyoRoomInviteN}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomInviteN(VidyoRoom* r, const LmiVector(LmiString)* userIds, const char* message, VidyoRoomOnInviteResult onInviteResult)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: userIds}
		{description: }
	}
	{parameter:
		{name: message}
		{description: }
	}
 	{parameter:
 		{name: onInviteResult}
 		{description: Callback that is triggered when another participant joins a conference.}
 	}
	{return: }
}
*/
LmiBool VidyoRoomInviteN(VidyoRoom* r, const LmiVector(LmiString)* userIds, const char* message, VidyoRoomOnInviteResult onInviteResult);

/**
{function:
	{name: VidyoRoomInviteAll}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomInviteAll(VidyoRoom* r, const char* message, VidyoRoomOnInviteResult onInviteResult)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: message}
		{description: }
	}
 	{parameter:
 		{name: onInviteResult}
 		{description: Callback that is triggered when another participant joins a conference.}
 	}
	{return: }
}
*/
LmiBool VidyoRoomInviteAll(VidyoRoom* r, const char* message, VidyoRoomOnInviteResult onInviteResult);

/**
{function:
	{name: VidyoRoomLockRoom}
	{parent: VidyoRoom}
	{description: This api provides the ability lock the current room.}
	{prototype: LmiBool VidyoRoomLockRoom(VidyoRoom* r, VidyoRoomOnLockRoomResult onLockRoomResult)}
	{parameter:
		{name: r}
		{description: VidyoRoom object.}
	}
	{parameter:
		{name: onLockRoomResult}
		{description: Callback to be triggered for lock room result.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomLockRoom(VidyoRoom* r, VidyoRoomOnLockRoomResult onLockRoomResult);

/**
{function:
	{name: VidyoRoomUnlockRoom}
	{parent: VidyoRoom}
	{description: This api provides the ability unlock the current room.}
	{prototype: LmiBool VidyoRoomUnlockRoom(VidyoRoom* r, VidyoRoomOnUnlockRoomResult onUnlockRoomResult)}
	{parameter:
		{name: r}
		{description: VidyoRoom object.}
	}
	{parameter:
		{name: onUnlockRoomResult}
		{description: Callback to be triggered for unlock room result.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomUnlockRoom(VidyoRoom* r, VidyoRoomOnUnlockRoomResult onUnlockRoomResult);

/**
{function:
	{name: VidyoRoomSetRoomPIN}
	{parent: VidyoRoom}
	{description: This api provides the ability to set the room pin. Once room pin is set, all other users are required to enter this room pin to join this conference room.}
	{prototype: LmiBool VidyoRoomSetRoomPIN(VidyoRoom* r, const LmiString* roomPIN, VidyoRoomOnSetRoomPINResult onSetRoomPINResult)}
	{parameter:
		{name: r}
		{description: VidyoRoom object.}
	}
	{parameter:
		{name: roomPIN}
		{description: Pin to be set as room pin or set NULL to remove existing room pin.}
	}
	{parameter:
		{name: onSetRoomPINResult}
		{description: Callback to be triggered for set room pin result.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomSetRoomPIN(VidyoRoom* r, const LmiString* roomPIN, VidyoRoomOnSetRoomPINResult onSetRoomPINResult);

/**
{function:
	{name: VidyoRoomRemoveRoomPIN}
	{parent: VidyoRoom}
	{description: This api provides the ability to remove the room pin. Once room pin is remove, all other users are not required to enter room pin to join the conference room.}
	{prototype: LmiBool VidyoRoomRemoveRoomPIN(VidyoRoom* r, VidyoRoomOnRemoveRoomPINResult onRemoveRoomPINResult)}
	{parameter:
		{name: r}
		{description: VidyoRoom object.}
	}
	{parameter:
		{name: onRemoveRoomPINResult}
		{description: Callback to be triggered for remove room pin result.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomRemoveRoomPIN(VidyoRoom* r, VidyoRoomOnRemoveRoomPINResult onRemoveRoomPINResult);

/**
{function:
	{name: VidyoRoomStartRecording}
	{parent: VidyoRoom}
	{description: Asynchronously starts recording a conference using the specified recording profile. Must be called on behalf of a moderator. Fails in case another recording is in progress. Calls a callback upon completion.}
	{prototype: LmiBool VidyoRoomStartRecording(VidyoRoom* r, const LmiString* recordingProfilePrefix, VidyoRoomOnRecordingServiceStartResult onRecordingServiceStartCallback)}
	{parameter:
		{name: r}
		{description: VidyoRoom object.}
	}
	{parameter:
		{name: recordingProfilePrefix}
		{description: Recording profile prefix to start the recording with.}
	}
	{parameter:
		{name: onRecordingServiceStartCallback}
		{description: Callback called upon completion.}
	}
	{return: LMI_TRUE if the request has been scheduled successfully, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomStartRecording(VidyoRoom* r, const LmiString* recordingProfilePrefix, VidyoRoomOnRecordingServiceStartResult OnRecordingServiceStartCallback);

/**
{function:
	{name: VidyoRoomStopRecording}
	{parent: VidyoRoom}
	{description: Asynchronously stops recording a conference. Must be called on behalf of a moderator. Fails in case there is no recording in progress. Calls a callback upon completion.}
	{prototype: LmiBool VidyoRoomStopRecording(VidyoRoom* r, VidyoRoomOnRecordingServiceStopResult onRecordingServiceStopResultCallback)}
	{parameter:
		{name: r}
		{description: VidyoRoom object.}
	}
	{parameter:
		{name: onRecordingServiceStopResultCallback}
		{description: Callback called upon completion.}
	}
	{return: LMI_TRUE if the request has been scheduled successfully, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomStopRecording(VidyoRoom* r, VidyoRoomOnRecordingServiceStopResult onRecordingServiceStopResultCallback);

/**
{function:
	{name: VidyoRoomPauseRecording}
	{parent: VidyoRoom}
	{description: Asynchronously pauses recording a conference. Must be called on behalf of a moderator. Fails in case recording is not in progress or already paused. Calls a callback upon completion.}
	{prototype: LmiBool VidyoRoomPauseRecording(VidyoRoom* r, VidyoRoomOnRecordingServicePauseResult onRecordingServicePauseResultCallback)}
	{parameter:
		{name: r}
		{description: VidyoRoom object.}
	}
	{parameter:
		{name: onRecordingServicePauseResultCallback}
		{description: Callback called upon completion.}
	}
	{return: LMI_TRUE if the request has been scheduled successfully, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomPauseRecording(VidyoRoom* r, VidyoRoomOnRecordingServicePauseResult onRecordingServicePauseResultCallback);

/**
{function:
	{name: VidyoRoomResumeRecording}
	{parent: VidyoRoom}
	{description: Asynchronously resumes recording a conference. Must be called on behalf of a moderator. Fails in case recording is not in progress or not paused. Calls a callback upon completion.}
	{prototype: LmiBool VidyoRoomResumeRecording(VidyoRoom* r, VidyoRoomOnRecordingServiceResumeResult onRecordingServiceResumeResultCallback)}
	{parameter:
		{name: r}
		{description: VidyoRoom object.}
	}
	{parameter:
		{name: onRecordingServiceResumeResultCallback}
		{description: Callback called upon completion.}
	}
	{return: LMI_TRUE if the request has been scheduled successfully, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomResumeRecording(VidyoRoom* r, VidyoRoomOnRecordingServiceResumeResult onRecordingServiceResumeResultCallback);

/**
{function visibility="private":
	{name: VidyoRoomGetRemoteRenderers}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomGetRemoteRenderers(VidyoRoom *r, LmiVector(VidyoRemoteRenderer) *renderers)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: renderers}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoRoomGetRemoteRenderers(VidyoRoom *r, LmiVector(VidyoRemoteRenderer) *renderers);

/**
{function visibility="private":
	{name: VidyoRoomGetRemoteSpeakers}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomGetRemoteSpeakers(VidyoRoom *r, LmiVector(VidyoRemoteSpeaker) *speakers)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: speakers}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoRoomGetRemoteSpeakers(VidyoRoom *r, LmiVector(VidyoRemoteSpeaker) *speakers);

/**
{function visibility="private":
	{name: VidyoRoomGetParticipants}
	{parent: VidyoRoom}
	{description: }
	{prototype: LmiBool VidyoRoomGetParticipants(VidyoRoom *r, LmiVector(VidyoParticipant) *participants)}
	{parameter:
		{name: r}
		{description: }
	}
	{parameter:
		{name: participants}
		{description: }
	}
	{return: }
}
*/
LmiBool VidyoRoomGetParticipants(VidyoRoom *r, LmiVector(VidyoParticipant) *participants);

/**
{function apigroup="simple":
	{name: VidyoRoomGetHistory}
	{parent: VidyoRoom}
	{description: Gets the latest history of the room.}
	{prototype: LmiBool VidyoRoomGetHistory(VidyoRoom *r, LmiUint32 numMessages, VidyoRoomOnGetHistoryResults onGetHistoryResults)}
	{parameter:
		{name: r}
		{description: The room object.}
	}
	{parameter:
		{name: numMessages}
		{description: The number of latest messages to request.}
	}
	{parameter:
		{name: onGetHistoryResults}
		{description: Callback that is triggered when history results are ready.}
	}
	{return: LMI_TRUE if the operation is successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomGetHistory(VidyoRoom *r, LmiUint32 numMessages, VidyoRoomOnGetHistoryResults onGetHistoryResults);

/**
{function apigroup="simple":
	{name: VidyoRoomGetHistoryById}
	{parent: VidyoRoom}
	{description: Gets the history of the room before and after the messageId.}
	{prototype: LmiBool VidyoRoomGetHistoryById(VidyoRoom *r, LmiInt64 messageId, LmiUint32 numMessagesBefore, LmiUint32 numMessagesAfter, VidyoRoomOnGetHistoryResults onGetHistoryResults)}
	{parameter: {name: r} {description: The room object.}}
	{parameter: {name: messageId} {description: The ID of the message to use as an apex.}}
	{parameter: {name: numMessagesBefore} {description: The number of messages before the apex.}}
	{parameter: {name: numMessagesAfter} {description: The number of messages after the apex.}}
	{parameter:
		{name: onGetHistoryResults}
		{description: Callback that is triggered when history results are ready.}
	}
	{return: LMI_TRUE if the operation is successful, LMI_FALSE otherwise.}}
*/
LmiBool VidyoRoomGetHistoryById(VidyoRoom *r, LmiInt64 messageId, LmiUint32 numMessagesBefore, LmiUint32 numMessagesAfter, VidyoRoomOnGetHistoryResults onGetHistoryResults);

/**
{function apigroup="simple":
	{name: VidyoRoomGetHistoryByTime}
	{parent: VidyoRoom}
	{description: Gets the history of the room before and after the messageId.}
	{prototype: LmiBool VidyoRoomGetHistoryByTime(VidyoRoom *r, LmiTime time, LmiUint32 numMessagesBefore, LmiUint32 numMessagesAfter, VidyoRoomOnGetHistoryResults onGetHistoryResults)}
	{parameter: {name: r} {description: The room object.}}
	{parameter: {name: time} {description: The time to use as an apex.}}
	{parameter: {name: numMessagesBefore} {description: The number of messages before the apex.}}
	{parameter: {name: numMessagesAfter} {description: The number of messages after the apex.}}
	{parameter:
		{name: onGetHistoryResults}
		{description: Callback that is triggered when history results are ready.}
	}
	{return: LMI_TRUE if the operation is successful, LMI_FALSE otherwise.}}
*/
LmiBool VidyoRoomGetHistoryByTime(VidyoRoom *r, LmiTime time, LmiUint32 numMessagesBefore, LmiUint32 numMessagesAfter, VidyoRoomOnGetHistoryResults onGetHistoryResults);

/**
{function apigroup="simple":
	{name: VidyoRoomMessageSearch}
	{parent: VidyoRoom}
	{description: Searches the room for text messages.}
	{prototype: LmiBool VidyoRoomMessageSearch(VidyoRoom *r, const char* searchText, LmiUint index, LmiUint pageSize, VidyoRoomOnMessageSearchResults onMessageSearchResults)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: searchText}
		{description: The text to search the messages in the room.}
	}
	{parameter:
		{name: index}
		{description: The start index from which the recent rooms are required. A value of '0' represents
			rooms starting from the beginning. A value of 'n' represents rooms starting from 'n'th record.
			This is used for paging the response.}
	}
	{parameter:
		{name: pageSize}
		{description: The maximum number of room records that can be sent in the response. 
			This is used for paging the response.}
	}
	{parameter:
		{name: onMessageSearchResults}
		{description: Callback that is triggered when message search results are ready.}
	}
	{return: }
}
*/
LmiBool VidyoRoomMessageSearch(VidyoRoom *r, const char *searchText, LmiUint index, LmiUint pageSize, VidyoRoomOnMessageSearchResults onMessageSearchResults);

/**
{function:
	{name: VidyoRoomAddLocalCamera}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomAddLocalCamera(VidyoRoom *r, const VidyoLocalCamera *localCamera, LmiBool removeOthers)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: localCamera}
		{description:}
	}
	{parameter:
		{name: removeOthers}
		{description:}
	}
}
*/
void VidyoRoomAddLocalCamera(VidyoRoom *r, const VidyoLocalCamera *localCamera, LmiBool removeOthers);

/**
{function:
	{name: VidyoRoomRemoveLocalCamera}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomRemoveLocalCamera(VidyoRoom *r, const VidyoLocalCamera *localCamera)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: localCamera}
		{description:}
	}
}
*/
void VidyoRoomRemoveLocalCamera(VidyoRoom *r, const VidyoLocalCamera *localCamera);

/**
{function:
	{name: VidyoRoomAddLocalMicrophone}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomAddLocalMicrophone(VidyoRoom *r, const VidyoLocalMicrophone *localMicrophone, LmiBool removeOthers, LmiBool isAES)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: localMicrophone}
		{description:}
	}
	{parameter:
		{name: removeOthers}
		{description:}
	}
	{parameter:
		{name: isAES}
		{description:}
	}
}
*/
void VidyoRoomAddLocalMicrophone(VidyoRoom *r, const VidyoLocalMicrophone *localMicrophone, LmiBool removeOthers, LmiBool isAES);

/**
{function:
	{name: VidyoRoomRemoveLocalMicrophone}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomRemoveLocalMicrophone(VidyoRoom *r, const VidyoLocalMicrophone *localMicrophone)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: localMicrophone}
		{description:}
	}
}
*/
void VidyoRoomRemoveLocalMicrophone(VidyoRoom *r, const VidyoLocalMicrophone *localMicrophone);

/**
{function:
	{name: VidyoRoomAddLocalSpeaker}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomAddLocalSpeaker(VidyoRoom *r, const VidyoLocalSpeaker *localSpeaker, LmiBool removeOthers, LmiBool isAES)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: localSpeaker}
		{description:}
	}
	{parameter:
		{name: removeOthers}
		{description:}
	}
	{parameter:
		{name: isAES}
		{description:}
	}
}
*/
void VidyoRoomAddLocalSpeaker(VidyoRoom *r, const VidyoLocalSpeaker *localSpeaker, LmiBool removeOthers, LmiBool isAES);

/**
{function:
	{name: VidyoRoomRemoveLocalSpeaker}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomRemoveLocalSpeaker(VidyoRoom *r, const VidyoLocalSpeaker *localSpeaker)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: localSpeaker}
		{description:}
	}
}
*/
void VidyoRoomRemoveLocalSpeaker(VidyoRoom *r, const VidyoLocalSpeaker *localSpeaker);

/**
{function:
	{name: VidyoRoomAddLocalRenderer}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomAddLocalRenderer(VidyoRoom *r, const VidyoLocalRenderer *localRenderer, LmiBool removeOthers)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: localRenderer}
		{description:}
	}
	{parameter:
		{name: removeOthers}
		{description:}
	}
}
*/
void VidyoRoomAddLocalRenderer(VidyoRoom *r, const VidyoLocalRenderer *localRenderer, LmiBool removeOthers);

/**
{function:
	{name: VidyoRoomRemoveLocalRenderer}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomRemoveLocalRenderer(VidyoRoom *r, const VidyoLocalRenderer *localRenderer)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: localRenderer}
		{description:}
	}
}
*/
void VidyoRoomRemoveLocalRenderer(VidyoRoom *r, const VidyoLocalRenderer *localRenderer);

/**
{function:
	{name: VidyoRoomAddLocalWindowShare}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomAddLocalWindowShare(VidyoRoom *r, const VidyoLocalWindowShare *localWindowShare, LmiBool removeOthers)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: localWindowShare}
		{description:}
	}
	{parameter:
		{name: removeOthers}
		{description:}
	}
}
*/
void VidyoRoomAddLocalWindowShare(VidyoRoom *r, const VidyoLocalWindowShare *localWindowShare, LmiBool removeOthers);

/**
{function:
	{name: VidyoRoomRemoveLocalWindowShare}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomRemoveLocalWindowShare(VidyoRoom *r, const VidyoLocalWindowShare *localWindowShare)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: localWindowShare}
		{description:}
	}
}
*/
void VidyoRoomRemoveLocalWindowShare(VidyoRoom *r, const VidyoLocalWindowShare *localWindowShare);

/**
{function:
	{name: VidyoRoomAddVirtualVideoSource}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomAddVirtualVideoSource(VidyoRoom *r, const VidyoVirtualVideoSource *virtualVideoSource, LmiBool removeOthers)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: virtualVideoSource}
		{description:}
	}
	{parameter:
		{name: removeOthers}
		{description:}
	}
}
*/
void VidyoRoomAddVirtualVideoSource(VidyoRoom *r, const VidyoVirtualVideoSource *virtualVideoSource, LmiBool removeOthers);

/**
{function:
	{name: VidyoRoomRemoveVirtualVideoSource}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomRemoveVirtualVideoSource(VidyoRoom *r, const VidyoVirtualVideoSource *virtualVideoSource)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: virtualVideoSource}
		{description:}
	}
}
*/
void VidyoRoomRemoveVirtualVideoSource(VidyoRoom *r, const VidyoVirtualVideoSource *virtualVideoSource);

/**
{function:
	{name: VidyoRoomAddLocalMonitor}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomAddLocalMonitor(VidyoRoom *r, const VidyoLocalMonitor *localMonitor, LmiBool removeOthers)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: localMonitor}
		{description:}
	}
	{parameter:
		{name: removeOthers}
		{description:}
	}
}
*/
void VidyoRoomAddLocalMonitor(VidyoRoom *r, const VidyoLocalMonitor *localMonitor, LmiBool removeOthers);

/**
{function:
	{name: VidyoRoomRemoveLocalMonitor}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomRemoveLocalMonitor(VidyoRoom *r, const VidyoLocalMonitor *localMonitor)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: localMonitor}
		{description:}
	}
}
*/
void VidyoRoomRemoveLocalMonitor(VidyoRoom *r, const VidyoLocalMonitor *localMonitor);

/**
{function:
	{name: VidyoRoomGetSelectedLocalCameras}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomGetSelectedLocalCameras(VidyoRoom *r, VidyoRoomOnGetSelectedLocalCamerasComplete onComplete)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: onComplete}
		{description:}
	}
}
*/
void VidyoRoomGetSelectedLocalCameras(VidyoRoom *r, VidyoRoomOnGetSelectedLocalCamerasComplete onComplete);

/**
{function:
	{name: VidyoRoomGetSelectedLocalMicrophones}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomGetSelectedLocalMicrophones(VidyoRoom *r, VidyoRoomOnGetSelectedLocalMicrophonesComplete onComplete)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: onComplete}
		{description:}
	}
}
*/
void VidyoRoomGetSelectedLocalMicrophones(VidyoRoom *r, VidyoRoomOnGetSelectedLocalMicrophonesComplete onComplete);

/**
{function:
	{name: VidyoRoomGetSelectedLocalSpeakers}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomGetSelectedLocalSpeakers(VidyoRoom *r, VidyoRoomOnGetSelectedLocalSpeakersComplete onComplete)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: onComplete}
		{description:}
	}
}
*/
void VidyoRoomGetSelectedLocalSpeakers(VidyoRoom *r, VidyoRoomOnGetSelectedLocalSpeakersComplete onComplete);

/**
{function:
	{name: VidyoRoomGetSelectedLocalRenderers}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomGetSelectedLocalRenderers(VidyoRoom *r, VidyoRoomOnGetSelectedLocalRenderersComplete onComplete)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: onComplete}
		{description:}
	}
}
*/
void VidyoRoomGetSelectedLocalRenderers(VidyoRoom *r, VidyoRoomOnGetSelectedLocalRenderersComplete onComplete);

/**
{function:
	{name: VidyoRoomGetSelectedLocalWindowShares}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomGetSelectedLocalWindowShares(VidyoRoom *r, VidyoRoomOnGetSelectedLocalWindowSharesComplete onComplete)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: onComplete}
		{description:}
	}
}
*/
void VidyoRoomGetSelectedLocalWindowShares(VidyoRoom *r, VidyoRoomOnGetSelectedLocalWindowSharesComplete onComplete);

/**
{function:
	{name: VidyoRoomGetSelectedLocalMonitors}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomGetSelectedLocalMonitors(VidyoRoom *r, VidyoRoomOnGetSelectedLocalMonitorsComplete monitors)}
	{parameter:
		{name: r}
		{description: Vidyo room object.}
	}
	{parameter:
		{name: monitors}
		{description:}
	}
}
*/
void VidyoRoomGetSelectedLocalMonitors(VidyoRoom *r, VidyoRoomOnGetSelectedLocalMonitorsComplete monitors);

/**
 {function:
	{name: VidyoRoomSetFavorite}
	{parent: VidyoRoom}
	{description: }
	{prototype: void VidyoRoomSetFavorite(VidyoRoom *r, LmiBool isFavorite, VidyoRoomOnSetFavoriteResult onSetFavoriteResult)}
	{parameter:
        {name: r}
        {description: Vidyo room object.}
	}
	{parameter:
        {name: isFavorite}
        {description:}
	}
	{parameter:
		{name: onSetFavoriteResult}
		{description: Callback that is triggered when set favorite operation completes.}
	}
 }
 */
void VidyoRoomSetFavorite(VidyoRoom *r, LmiBool isFavorite, VidyoRoomOnSetFavoriteResult onSetFavoriteResult);

/**
 {function apigroup="simple":
	{name: VidyoRoomDelete}
	{parent: VidyoRoom}
	{description: Delete a room. }
	{prototype: void VidyoRoomDelete(VidyoRoom *r, VidyoRoomOnDeleteResult onDeleteResult)}
	{parameter:
        {name: r}
        {description: Vidyo room object.}
	}
	{parameter:
		{name: onDeleteResult}
		{description: Callback that is triggered when delete operation completes.}
	}
 }
 */
void VidyoRoomDelete(VidyoRoom *r, VidyoRoomOnDeleteResult onDeleteResult);

/**
{function apigroup="simple":
	{name: VidyoRoomShowStatisticsDialog}
	{parent: VidyoRoom}
	{description: Shows statistics for a room. }
	{prototype: void VidyoRoomShowStatisticsDialog(VidyoRoom *r)}
	{parameter:
        {name: r}
        {description: Vidyo room object.}
	}
	{note: The dialog is a platform-specific native window that will pop up outside of the application.}
 }
 */
void VidyoRoomShowStatisticsDialog(VidyoRoom *r);

/**
{function apigroup="simple":
	{name: VidyoRoomHideStatisticsDialog}
	{parent: VidyoRoom}
	{description: Hides statistics for a room. }
	{prototype: void VidyoRoomHideStatisticsDialog(VidyoRoom *r)}
	{parameter:
        {name: r}
        {description: Vidyo room object.}
	}
 }
 */
void VidyoRoomHideStatisticsDialog(VidyoRoom *r);

/**
{function apigroup="simple":
	{name: VidyoRoomResizeStatisticsDialog}
	{parent: VidyoRoom}
	{description: Resizes statistics dialog for a room. }
	{prototype: void VidyoRoomResizeStatisticsDialog(VidyoRoom *r, LmiInt width, LmiInt height)}
	{parameter:
        {name: r}
        {description: Vidyo room object.}
	}
	{parameter:
        {name: width}
        {description: Width of the dialog.}
	}
	{parameter:
        {name: height}
        {description: height of the dialog.}
	}
 }
 */
void VidyoRoomResizeStatisticsDialog(VidyoRoom *r, LmiInt width, LmiInt height);

/**
{function visibility="private":
	{name: VidyoRoomGetStats}
	{parent: VidyoRoom}
	{description: Fill in data in VidyoRoomStats from a room.
 		This API is allowed in VIDYO_ROOMSTATE_MediaEnabled state. If used in another state, no action will be taken and return LMI_FALSE.}
	{prototype: LmiBool VidyoRoomGetStats(VidyoRoom *r, VidyoRoomStats *stats)}
	{parameter: {name: r} {description: The room object to extract stats.}}
	{parameter: {name: stats} {description: The stats object to fill.}}
	{return: LMI_TRUE on success.}
}
*/
LmiBool VidyoRoomGetStats(VidyoRoom *r, VidyoRoomStats *stats);

/**
{function:
	{name: VidyoRoomCancelInvite}
	{parent: VidyoRoom}
	{description: Cancel invitation to a VidyoRoom.}
	{prototype: LmiBool VidyoRoomCancelInvite(VidyoRoom *r, const LmiString *inviteeId)}
	{parameter: {name: r} {description: The room object to extract stats.}}
	{parameter: {name: inviteeId} {description: Id of the invited user.}}
 	{return: LMI_TRUE on success.}
}
*/
LmiBool VidyoRoomCancelInvite(VidyoRoom *r, const LmiString *inviteeId);

/**
 {function:
	{name: VidyoRoomBootParticipant}
	{parent: VidyoRoom}
	{description: Kick one participant out of a room. }
	{prototype: LmiBool VidyoRoomBootParticipant(VidyoRoom *r, const LmiString* userId, const LmiString* reason, const LmiString* requestId)}
	{parameter: {name: r} {description: Vidyo room object.}}
	{parameter: {name: userId} {description: ID of users to be kicked out of a room.}}
	{parameter: {name: reason} {description: A message explaining the reason for the booting. This can be NULL if no message is desired.}}
	{parameter: {name: requestId} {description: Application provide identifier to track this request.}}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
 }
 */
LmiBool VidyoRoomBootParticipant(VidyoRoom *r, const LmiString* userId, const LmiString* reason, const LmiString* requestId);

/**
 {function:
	{name: VidyoRoomBootAllParticipants}
	{parent: VidyoRoom}
	{description: Kick all participants out of a room. }
	{prototype: LmiBool VidyoRoomBootAllParticipants(VidyoRoom *r, const LmiString* reason, const LmiString* requestId)}
	{parameter: {name: r} {description: Vidyo room object.}}
	{parameter: {name: reason} {description: A message explaining the reason for the booting. This can be NULL if no message is desired.}}
	{parameter: {name: requestId} {description: Application provide identifier to track this request.}}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
 }
 */
LmiBool VidyoRoomBootAllParticipants(VidyoRoom *r, const LmiString* reason, const LmiString* requestId);


/**
 {type:
 	{name: VidyoRole}
 	{parent: VidyoRoom}
 	{description: }
 		{value: {name: VIDYO_ROLE_Moderator} {description: }}
 		{value: {name: VIDYO_ROLE_None} {description: }}
 }
*/
typedef enum {
	VIDYO_ROLE_Moderator,
	VIDYO_ROLE_None
} VidyoRole;

/**
 {type:
 	{name:  VidyoRoleAuthorization}
 	{parent: VidyoRoom}
 	{description: }
 	{member:
 		{name: hasPassCode}
 		{type: LmiBool}
 		{description: hasPassCode set to LMI_TRUE means 'passCode' field contains a valid input.}
 	}
 	{member:
 		{name: passCode}
 		{type: LmiString}
 		{description: }
 	}
 }
 */

typedef struct VidyoRoleAuthorization_ {
	LmiBool hasPassCode;
	LmiString passCode;
} VidyoRoleAuthorization;

/**
{function visibility="private":
	{name: VidyoRoleAuthorizationConstructDefault}
	{parent: VidyoRoom}
	{description: }
	{prototype:VidyoRoleAuthorization* VidyoRoleAuthorizationConstructDefault(VidyoRoleAuthorization* auth, LmiAllocator* alloc)}
	{parameter:
		{name: auth}
		{description:}
	}
	{parameter:
		{name: alloc}
		{description:}
	}
	{return: }
}
*/
VidyoRoleAuthorization* VidyoRoleAuthorizationConstructDefault(VidyoRoleAuthorization* auth, LmiAllocator* alloc);

/**
{function visibility="private":
	{name: VidyoRoleAuthorizationConstructCopy}
	{parent: VidyoRoom}
	{description: Constructs an VidyoRoleAuthorization object as a copy of an existing objec.}
	{prototype:VidyoRoleAuthorization* VidyoRoleAuthorizationConstructCopy(VidyoRoleAuthorization* d, VidyoRoleAuthorization* s)}
	{parameter:
		{name: d}
		{description: The VidyoRoleAuthorization object to construct.}
	}
	{parameter:
		{name: s}
		{description: The VidyoRoleAuthorization object to make a copy of.}
	}
	{return: The VidyoRoleAuthorization object on success, otherwise NULL.}
}
*/
VidyoRoleAuthorization* VidyoRoleAuthorizationConstructCopy(VidyoRoleAuthorization* d, VidyoRoleAuthorization* s);

/**
 {function visibility="private":
 	{name: VidyoRoleAuthorizationDestruct}
 	{parent: VidyoRoom}
 	{description: }
 	{prototype: void VidyoRoleAuthorizationDestruct(VidyoRoleAuthorization* auth)}
 	{parameter:
 		{name: auth}
 		{description:}
 	}
 }
 */
void VidyoRoleAuthorizationDestruct(VidyoRoleAuthorization* auth);

/**
{type apigroup="simple" visibility="public":
    {name: VidyoRoomRequestRoleChangeResult}
    {parent: VidyoRoom}
    {description: This type indicates the result of request role change.}
    {value: {name: VIDYO_ROOMREQUESTROLECHANGERESULT_OK} {description: Request role change was successful.}}
    {value: {name: VIDYO_ROOMREQUESTROLECHANGERESULT_NoResponse} {description: The server did not respond in a reasonable amount of time to the role change request.}}
    {value: {name: VIDYO_ROOMREQUESTROLECHANGERESULT_Unauthorized} {description: The user did not have permission to request role change.}}
    {value: {name: VIDYO_ROOMREQUESTROLECHANGERESULT_MiscLocalError} {description: Request role change failed due to some miscellaneous problem with the request.}}
    {value: {name: VIDYO_ROOMREQUESTROLECHANGERESULT_MiscRemoteError} {description: The server is rejecting the role change request due to some miscellaneous problem of its own.}}
}
*/
typedef enum
{
    VIDYO_ROOMREQUESTROLECHANGERESULT_OK,
    VIDYO_ROOMREQUESTROLECHANGERESULT_NoResponse,
    VIDYO_ROOMREQUESTROLECHANGERESULT_Unauthorized,
    VIDYO_ROOMREQUESTROLECHANGERESULT_MiscLocalError,
    VIDYO_ROOMREQUESTROLECHANGERESULT_MiscRemoteError
} VidyoRoomRequestRoleChangeResult;

/**
 {callback apigroup="simple":
 	{name: VidyoRoomOnRequestRoleChangeResult}
 	{parent: VidyoRoom}
 	{description: }
 	{prototype: void (*VidyoRoomOnRequestRoleChangeResult)(const VidyoRoom* room, VidyoRoomRequestRoleChangeResult result, const LmiString* requestId)}
 	{parameter:
 		{name: room}
 		{description: Pointer to the VidyoRoom that triggered the callback. }
 	}
 	{parameter:
 		{name: result}
 		{description: Result of the role change request.}
 	}
 	{parameter:
 		{name: requestId}
 		{description: }
 	}
 }
 */
typedef void (*VidyoRoomOnRequestRoleChangeResult)(const VidyoRoom* room, VidyoRoomRequestRoleChangeResult result, const LmiString* requestId);

/**
 {function:
 	{name: VidyoRoomRequestRoleChange}
 	{parent: VidyoRoom}
 	{description: This API provides the ability to request moderator functionality, providing the user who is not a room owner, the ability to control remote source transmitting, for other users in the conference. Remote sources currently include camera's and microphones.}
 	{prototype: LmiBool VidyoRoomRequestRoleChange(const VidyoRoom* room, VidyoRole newRole, const VidyoRoleAuthorization* auth, VidyoRoomOnRequestRoleChangeResult onRequestRoleChangeResult, const LmiString* requestId)}
 	{parameter: {name: room} {description: Vidyo room object.}}
 	{parameter: {name: newRole} {description: .}}
 	{parameter: {name: auth} {description: .}}
 	{parameter: {name: onRequestRoleChangeResult} {description: .}}
 	{parameter: {name: requestId} {description: .}}
 	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
 }
*/
LmiBool VidyoRoomRequestRoleChange(const VidyoRoom* room, VidyoRole newRole, const VidyoRoleAuthorization* auth, VidyoRoomOnRequestRoleChangeResult onRequestRoleChangeResult, const LmiString* requestId);

/**
{function:
	{name: VidyoRoomSetRoleChangeAuthorization}
	{parent: VidyoRoom}
	{description: This API provides the ability to allow a room owner to set an authentication scheme for moderator functionality. The same authentication scheme must be used by other participants who calls VidyoRoomRequestRoleChange to control remote source transmitting}
	{prototype: LmiBool VidyoRoomSetRoleChangeAuthorization(VidyoRoom* room, VidyoRole role, const VidyoRoleAuthorization* auth, const LmiString* requestId)}
	{parameter: {name: room} {description: Vidyo room object.}}
	{parameter: {name: role} {description: VidyoRole that provided auth is assoicted with.}}
	{parameter: {name: auth} {description: Represent the authentication scheme (e.g use passCode to obtain moderator functionality).}}
	{parameter: {name: requestId} {description: Application provide identifier to track this request.}}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomSetRoleChangeAuthorization(VidyoRoom* room, VidyoRole role, const VidyoRoleAuthorization* auth, const LmiString* requestId);

/**
{function:
	{name: VidyoRoomRemoveRoleChangeAuthorization}
	{parent: VidyoRoom}
	{description: This API provides the ability to allow a room owner to remove authentication scheme for moderator functionality}
	{prototype: LmiBool VidyoRoomRemoveRoleChangeAuthorization(VidyoRoom* room, VidyoRole role, const LmiString* requestId)}
	{parameter: {name: room} {description: Vidyo room object.}}
	{parameter: {name: role} {description: VidyoRole that provided auth is assoicted with.}}
	{parameter: {name: requestId} {description: Application provide identifier to track this request.}}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomRemoveRoleChangeAuthorization(VidyoRoom* room, VidyoRole role, const LmiString* requestId);

/**
{type apigroup="simple" visibility="public":
    {name: VidyoRoomModerationResult}
    {parent: VidyoRoom}
    {description: This type indicates the result of room moderation}
    {value: {name: VIDYO_ROOMMODERATIONRESULT_OK} {description: The moderation operation was successful.}}
    {value: {name: VIDYO_ROOMMODERATIONRESULT_NoResponse} {description: The server did not respond in a reasonable amount of time to the moderation request.}}
    {value: {name: VIDYO_ROOMMODERATIONRESULT_Unauthorized} {description: The user did not have permission to moderate.}}
	{value: {name: VIDYO_ROOMMODERATIONRESULT_InvalidInput} {description: The input of role change authorization is invalid.}}
    {value: {name: VIDYO_ROOMMODERATIONRESULT_MiscLocalError} {description: Moderation request failed due to some miscellaneous problem with the request.}}
    {value: {name: VIDYO_ROOMMODERATIONRESULT_MiscRemoteError} {description: The server is rejecting the moderation request due to some miscellaneous problem of its own.}}
}
*/
typedef enum
{
    VIDYO_ROOMMODERATIONRESULT_OK,
    VIDYO_ROOMMODERATIONRESULT_NoResponse,
    VIDYO_ROOMMODERATIONRESULT_Unauthorized,
	VIDYO_ROOMMODERATIONRESULT_InvalidInput,
    VIDYO_ROOMMODERATIONRESULT_MiscLocalError,
    VIDYO_ROOMMODERATIONRESULT_MiscRemoteError
} VidyoRoomModerationResult;

/**
{callback apigroup="simple":
	{name: VidyoRoomOnModerationResult}
	{parent: VidyoRoom}
	{description: }
	{prototype: void (*VidyoRoomOnModerationResult)(const VidyoRoom* room, const VidyoParticipant* participant, VidyoRoomModerationResult result, VidyoRoomModerationActionType actionType, const LmiString* requestId)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: participant}
		{description: Participant for which moderation request was performed.}
	}
	{parameter:
		{name: result}
		{description: Result of the room moderation request.}
	}
	{parameter:
		{name: actionType}
		{description: Action which was performed.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
}
*/
typedef void (*VidyoRoomOnModerationResult)(const VidyoRoom* room, const VidyoParticipant* participant, VidyoRoomModerationResult result, VidyoRoomModerationActionType actionType, const LmiString* requestId);

/**
{function:
    {name: VidyoRoomRegisterModerationResultEventListener}
    {parent: VidyoRoom}
    {description: Register a callback if application needs to know result of moderation request.}
    {prototype: LmiBool VidyoRoomRegisterModerationResultEventListener(VidyoRoom* room, VidyoRoomOnModerationResult onModerationResult)}
    {parameter: {name: room} {description: Vidyo room object.}}
    {parameter: {name: onModerationResult} {description: Callback to be triggered upon receive moderation request result.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomRegisterModerationResultEventListener(VidyoRoom* room, VidyoRoomOnModerationResult onModerationResult);

/**
{function:
    {name: VidyoRoomUnregisterModerationResultEventListener}
    {parent: VidyoRoom}
    {description:}
    {prototype: LmiBool VidyoRoomUnregisterModerationResultEventListener(VidyoRoom* room)}
    {parameter: {name: room} {description: Vidyo room object.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomUnregisterModerationResultEventListener(VidyoRoom* room);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnPresenterChanged}
	{parent: VidyoRoom}
	{description: Callback that is triggered when the presenter was changed. }
	{prototype: void (*VidyoRoomOnPresenterChanged)(VidyoRoom* room, const VidyoParticipant* participant)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: participant}
		{description: The current presenter. }
	}
}
*/
typedef void (*VidyoRoomOnPresenterChanged)(VidyoRoom* room, const VidyoParticipant* participant);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnHandRaised}
	{parent: VidyoRoom}
	{description: Callback that is triggered for the moderator every time someone raises their hand. }
	{prototype: void (*VidyoRoomOnHandRaised)(VidyoRoom* room, const LmiVector(VidyoParticipant)* participant)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: participant}
		{description: Participant whose hand was raised. }
	}
}
*/
typedef void (*VidyoRoomOnHandRaised)(VidyoRoom* room, const LmiVector(VidyoParticipant)* participant);

/**
{function:
	{name: VidyoRoomRegisterLectureModeEventListener}
	{parent: VidyoRoom}
	{description: Register a callback if application needs to know result of moderation request.}
	{prototype: LmiBool VidyoRoomRegisterLectureModeEventListener(VidyoRoom* room, VidyoRoomOnPresenterChanged presenterChanged, VidyoRoomOnHandRaised handRaised)}
	{parameter: {name: room} {description: Vidyo room object.}}
	{parameter: {name: presenterChanged} {description: Callback that is triggered when presenter changed.}}
	{parameter: {name: handRaised} {description: Callback that is triggered for Moderator when some hand raised.}}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomRegisterLectureModeEventListener(VidyoRoom* room, VidyoRoomOnPresenterChanged presenterChanged, VidyoRoomOnHandRaised handRaised);

/**
{function:
	{name: VidyoRoomUnregisterLectureModeEventListener}
	{parent: VidyoRoom}
	{description:}
	{prototype: LmiBool VidyoRoomUnregisterLectureModeEventListener(VidyoRoom* room)}
	{parameter: {name: room} {description: Vidyo room object.}}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomUnregisterLectureModeEventListener(VidyoRoom* room);

/**
{function:
	{name: VidyoRoomStartLectureMode}
	{parent: VidyoRoom}
	{description: Switch room mode to the Lecture mode.}
	{prototype: LmiBool VidyoRoomStartLectureMode(VidyoRoom* room, const LmiString* requestId)}
	{parameter: {name: room} {description: Vidyo room object.}}
	{parameter: {name: requestId} {description: Application provide identifier to track this request.}}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomStartLectureMode(VidyoRoom* room, const LmiString* requestId);

/**
{function:
	{name: VidyoRoomStopLectureMode}
	{parent: VidyoRoom}
	{description: Switch room mode to the Group mode.}
	{prototype: LmiBool VidyoRoomStopLectureMode(VidyoRoom* room, const LmiString* requestId)}
	{parameter: {name: room} {description: Vidyo room object.}}
	{parameter: {name: requestId} {description: Application provide identifier to track this request.}}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomStopLectureMode(VidyoRoom* room, const LmiString* requestId);

/**
{function:
	{name: VidyoRoomSetPresenter}
	{parent: VidyoRoom}
	{description: Set a specific participant as a presenter.}
	{prototype: LmiBool VidyoRoomSetPresenter(VidyoRoom* room, const VidyoParticipant* participant, const LmiString* requestId)}
	{parameter: {name: room} {description: Vidyo room object.}}
	{parameter: {name: participant} {description:The VidyoParticipant object who will be a new presenter.}}
	{parameter: {name: requestId} {description: Application provide identifier to track this request.}}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomSetPresenter(VidyoRoom* room, const VidyoParticipant* participant, const LmiString* requestId);

/**
{function:
	{name: VidyoRoomRemovePresenter}
	{parent: VidyoRoom}
	{description: Remove presenter rights from a specific participant.}
	{prototype: LmiBool VidyoRoomRemovePresenter(VidyoRoom* room, const LmiString* requestId)}
	{parameter: {name: room} {description: Vidyo room object.}}
	{parameter: {name: requestId} {description: Application provide identifier to track this request.}}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomRemovePresenter(VidyoRoom* room, const LmiString* requestId);

/**
 {callback apigroup="simple":
	{name: VidyoRoomOnRaiseHandResponse}
	{parent: VidyoRoom}
	{description: Callback that is triggered each time when response of hand raise request}
	{prototype: void (*VidyoRoomOnRaiseHandResponse)(VidyoRoom *r, const VidyoParticipantHandState handState)}
	{parameter:
		{name: r}
		{description: The VidyoRoom that triggered the callback.}
	}
	{parameter:
		{name: handState}
		{description: State of the VidyoParticipantHandState.}
	}
}
*/
typedef void (*VidyoRoomOnRaiseHandResponse)(VidyoRoom *r, const VidyoParticipantHandState handState);

/**
 {function:
	{name: VidyoRoomRaiseHand}
	{parent: VidyoRoom}
	{description: Each participant can send a moderation request to raise their hand to request to be unmuted. The Moderator and Presenter will be notified.}
	{prototype: LmiBool VidyoRoomRaiseHand(VidyoRoom* r, VidyoRoomOnRaiseHandResponse raiseHandResponse, const LmiString* requestId)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: raiseHandResponse}
		{description: Callback that is triggered each time when response of hand raise request.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomRaiseHand(VidyoRoom* r, VidyoRoomOnRaiseHandResponse raiseHandResponse, const LmiString* requestId);

/**
 {function:
	{name: VidyoRoomUnraiseHand}
	{parent: VidyoRoom}
	{description: Each participant can send a moderation request to remove his raised hand to cancel the request to be unmuted.}
	{prototype: LmiBool VidyoRoomUnraiseHand(VidyoRoom* r, const LmiString* requestId)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomUnraiseHand(VidyoRoom* r, const LmiString* requestId);

/**
 {function:
	{name: VidyoRoomApproveRaisedHand}
	{parent: VidyoRoom}
	{description: The Moderator can send a moderation request to approve a raised hand, in which case the participant whose hand was raised will be notified.}
	{prototype: LmiBool VidyoRoomApproveRaisedHand(VidyoRoom* r, const VidyoParticipant* participant, const LmiString* requestId)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: participant}
		{description: Selected participant whose raised hand request is approved.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomApproveRaisedHand(VidyoRoom* r, const VidyoParticipant* participant, const LmiString* requestId);

/**
 {function:
	{name: VidyoRoomDismissRaisedHand}
	{parent: VidyoRoom}
	{description: The Moderator can send a moderation request to decline a raised hands, in which case the participant whose hand was raised will be notified.}
	{prototype: LmiBool VidyoRoomDismissRaisedHand(VidyoRoom* r, const LmiVector(VidyoParticipant)* participants, const LmiString* requestId)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: participants}
		{description: List of selected participants whose raised hand request is going to dismiss.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomDismissRaisedHand(VidyoRoom* r, const LmiVector(VidyoParticipant)* participants, const LmiString* requestId);

/**
 {function:
	{name: VidyoRoomDismissAllRaisedHands}
	{parent: VidyoRoom}
	{description: The Moderator can send a moderation request to reject all raised hand, in which case all participants whose hand was raised will be notified.}
	{prototype: LmiBool VidyoRoomDismissAllRaisedHands(VidyoRoom* r, const LmiString* requestId)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: requestId}
		{description: Application provide identifier to track this request.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomDismissAllRaisedHands(VidyoRoom* r, const LmiString* requestId);

/**
{function:
    {name: VidyoRoomRevokeAudioForAll}
    {parent: VidyoRoom}
    {description: Set audio revoke state for all participants in the room.}
    {prototype: LmiBool VidyoRoomRevokeAudioForAll(VidyoRoom* room, LmiBool revoke, const LmiString* requestId)}
    {parameter: {name: room} {description: Vidyo room object.}}
    {parameter: {name: revoke} {description: Set to LMI_TRUE to stop all participants' audio. Audio cannot be resumed until another call with revoke set to LMI_FALSE.}}
    {parameter: {name: requestId} {description: Application provide identifier to track this request.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomRevokeAudioForAll(VidyoRoom* room, LmiBool revoke, const LmiString* requestId);

/**
{function:
    {name: VidyoRoomRevokeAudioForParticipant}
    {parent: VidyoRoom}
    {description: Set audio revoke state for a single participant in the room.}
    {prototype: LmiBool VidyoRoomRevokeAudioForParticipant(VidyoRoom* room, const LmiString* userId, LmiBool revoke, const LmiString* requestId)}
    {parameter: {name: room} {description: Vidyo room object.}}
    {parameter: {name: userId} {description: The ID of participant whose audio is revoked/resumed.}}
    {parameter: {name: revoke} {description: Set to LMI_TRUE to stop the participant's audio. Audio cannot be resumed until another call with revoke set to LMI_FALSE.}}
    {parameter: {name: requestId} {description: Application provide identifier to track this request.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomRevokeAudioForParticipant(VidyoRoom* room, const LmiString* userId, LmiBool revoke, const LmiString* requestId);

/**
{function:
    {name: VidyoRoomRevokeVideoForAll}
    {parent: VidyoRoom}
    {description: Set video revoke state for all participants in the room.}
    {prototype: LmiBool VidyoRoomRevokeVideoForAll(VidyoRoom* room, LmiBool revoke, const LmiString* requestId)}
    {parameter: {name: room} {description: Vidyo room object.}}
    {parameter: {name: revoke} {description: Set to LMI_TRUE to stop all participants' video. Video cannot be resumed until another call with revoke set to LMI_FALSE.}}
    {parameter: {name: requestId} {description: Application provide identifier to track this request.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomRevokeVideoForAll(VidyoRoom* room, LmiBool revoke, const LmiString* requestId);

/**
{function:
    {name: VidyoRoomRevokeVideoForParticipant}
    {parent: VidyoRoom}
    {description: Set video revoke state for a single participant in the room.}
    {prototype: LmiBool VidyoRoomRevokeVideoForParticipant(VidyoRoom* room, const LmiString* userId, LmiBool revoke, const LmiString* requestId)}
    {parameter: {name: room} {description: Vidyo room object.}}
    {parameter: {name: userId} {description: The ID of participant whose audio is revoked/resumed.}}
    {parameter: {name: revoke} {description: Set to LMI_TRUE to stop the participant's video. Video cannot be resumed until another call with revoke set to LMI_FALSE.}}
    {parameter: {name: requestId} {description: Application provide identifier to track this request.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomRevokeVideoForParticipant(VidyoRoom* room, const LmiString* userId, LmiBool revoke, const LmiString* requestId);

/**
{function:
    {name: VidyoRoomRequestAudioSilenceForAll}
    {parent: VidyoRoom}
    {description: Reqeust all participants in the room to stop sending audio. And any participant is allowed to restart audio after it is stopped first.}
    {prototype: LmiBool VidyoRoomRequestAudioSilenceForAll(VidyoRoom* room, const LmiString* requestId)}
    {parameter: {name: room} {description: Vidyo room object.}}
    {parameter: {name: requestId} {description: Application provide identifier to track this request.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool  VidyoRoomRequestAudioSilenceForAll(VidyoRoom* room, const LmiString* requestId);

/**
{function:
    {name: VidyoRoomRequestAudioSilenceForParticipant}
    {parent: VidyoRoom}
    {description: Reqeust a single participant in the room to stop sending audio. This participant is allowed to restart audio after it is stopped first.}
    {prototype: LmiBool VidyoRoomRequestAudioSilenceForParticipant(VidyoRoom* room, const LmiString* userId, const LmiString* requestId)}
    {parameter: {name: room} {description: Vidyo room object.}}
    {parameter: {name: userId} {description: The ID of participant whose audio is requested to stop.}}
    {parameter: {name: requestId} {description: Application provide identifier to track this request.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool  VidyoRoomRequestAudioSilenceForParticipant(VidyoRoom* room, const LmiString* userId, const LmiString* requestId);

/**
{function:
    {name: VidyoRoomRequestVideoSilenceForAll}
    {parent: VidyoRoom}
    {description: Reqeust all participants in the room to stop sending video. And any participant is allowed to restart video after it is stopped first.}
    {prototype: LmiBool VidyoRoomRequestVideoSilenceForAll(VidyoRoom* room, const LmiString* requestId)}
    {parameter: {name: room} {description: Vidyo room object.}}
    {parameter: {name: requestId} {description: Application provide identifier to track this request.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool  VidyoRoomRequestVideoSilenceForAll(VidyoRoom* room, const LmiString* requestId);

/**
{function:
    {name: VidyoRoomRequestVideoSilenceForParticipant}
    {parent: VidyoRoom}
    {description: Reqeust a single participant in the room to stop sending video. This participant is allowed to restart video after it is stopped first.}
    {prototype: LmiBool VidyoRoomRequestVideoSilenceForParticipant(VidyoRoom* room, const LmiString* userId, const LmiString* requestId)}
    {parameter: {name: room} {description: Vidyo room object.}}
    {parameter: {name: userId} {description: The ID of participant whose video is requested to stop.}}
    {parameter: {name: requestId} {description: Application provide identifier to track this request.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool  VidyoRoomRequestVideoSilenceForParticipant(VidyoRoom* room, const LmiString* userId, const LmiString* requestId);

/**
 {function:
 {name: VidyoRoomSetShowThrottle}
 {parent: VidyoRoom}
 {description: Set show throttle type.}
 {prototype: void VidyoRoomSetShowThrottle(VidyoRoom *room, VidyoRoomShowThrottle t)}
 {parameter: {name: room} {description: Vidyo room object.}}
 {parameter: {name: t} {description: The throttle type.}}
 }
 */
void VidyoRoomSetShowThrottle(VidyoRoom *room, VidyoRoomShowThrottle t);

/**
 {function:
 {name: VidyoRoomGetShowThrottle}
 {parent: VidyoRoom}
 {description: Get show throttle type.}
 {prototype: VidyoRoomShowThrottle VidyoRoomGetShowThrottle(VidyoRoom *room)}
 {parameter: {name: room} {description: Vidyo room object.}}
 {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
 }
 */
VidyoRoomShowThrottle VidyoRoomGetShowThrottle(VidyoRoom *room);

/**
 {function:
 {name: VidyoRoomSetShowThrottleTimerInterval}
 {parent: VidyoRoom}
 {description: Set show throttle timer interval.}
 {prototype: void VidyoRoomSetShowThrottleTimerInterval(VidyoRoom *room, LmiUint seconds)}
 {parameter: {name: room} {description: Vidyo room object.}}
 {parameter: {name: seconds} {description: The time interval in between shows.}}
 }
 */
void VidyoRoomSetShowThrottleTimerInterval(VidyoRoom *room, LmiUint seconds);

/**
 {function:
 {name: VidyoRoomGetShowThrottleTimerInterval}
 {parent: VidyoRoom}
 {description: Get show throttle timer interval.}
 {prototype: LmiUint VidyoRoomGetShowThrottleTimerInterval(VidyoRoom *room)}
 {parameter: {name: room} {description: Vidyo room object.}}
 {return: Show throttle timer interval in number of seconds.}
 }
 */
LmiUint VidyoRoomGetShowThrottleTimerInterval(VidyoRoom *room);

/**
 {function:
 {name: VidyoRoomSetShowThrottleWindowSizeChangeThreshold}
 {parent: VidyoRoom}
 {description: Set show throttle window size change threshold.}
 {prototype: void VidyoRoomSetShowThrottleWindowSizeChangeThreshold(VidyoRoom *room, LmiUint percentage)}
 {parameter: {name: room} {description: Vidyo room object.}}
 {parameter: {name: percentage} {description: The threshold as a percentage of window size change.}}
 }
 */
void VidyoRoomSetShowThrottleWindowSizeChangeThreshold(VidyoRoom *room, LmiUint percentage);

/**
 {function:
 {name: VidyoRoomGetShowThrottleWindowSizeChangeThreshold}
 {parent: VidyoRoom}
 {description: Get show throttle threshold for window size change.}
 {prototype: LmiUint VidyoRoomGetShowThrottleWindowSizeChangeThreshold(VidyoRoom *room)}
 {parameter: {name: room} {description: Vidyo room object.}}
 {return: Show throttle timer interval in number of seconds.}
 }
 */
LmiUint VidyoRoomGetShowThrottleWindowSizeChangeThreshold(VidyoRoom *room);

/**
{type apigroup="simple" visibility="public":
	{name: VidyoRoomModerationType}
	{parent: VidyoRoom}
	{description: Sets the type of moderation command received from the server.}
	{value: {name: VIDYO_ROOMMODERATIONTYPE_SoftMute} {description: Moderation command which tells that device was muted by the server and he is allowed to unmute.}}
	{value: {name: VIDYO_ROOMMODERATIONTYPE_HardMute} {description: Moderation command which tells that device was muted by the server and he is NOT allowed to unmute.}}
	{value: {name: VIDYO_ROOMMODERATIONTYPE_None} {description: Moderation command which tells that device was not muted by the server and he is allowed to unmute.}}
}
*/
typedef enum {
	VIDYO_ROOMMODERATIONTYPE_SoftMute,
	VIDYO_ROOMMODERATIONTYPE_HardMute,
	VIDYO_ROOMMODERATIONTYPE_None
}
VidyoRoomModerationType;

/**
{callback apigroup="simple":
    {name: VidyoRoomOnModerationCommandReceived}
    {parent: VidyoRoom}
    {description: Callback that will be tringgered when the moderation command will be recieved from the server.}
    {prototype: void (*VidyoRoomOnModerationCommandReceived)(const VidyoRoom* room, VidyoDeviceType deviceType, VidyoRoomModerationType moderationType, LmiBool state)}
    {parameter:
        {name: room}
        {description: Pointer to the VidyoRoom that triggered the callback. }
    }
    {parameter:
        {name: deviceType}
        {description: Type of the device that was muted. Local camera or microphone.}
    }
    {parameter:
        {name: moderationType}
        {description: Type of moderation command, tells if device was hard muted/unmuted or soft muted. Note: soft unmute is not supported.}
    }
    {parameter:
        {name: state}
        {description: State of the mute. LMI_TRUE in case device was muted, LMI_FALSE in case he was unmuted.}
    }
}
*/
typedef void (*VidyoRoomOnModerationCommandReceived)(const VidyoRoom* room, VidyoDeviceType deviceType, VidyoRoomModerationType moderationType, LmiBool state);

/**
{function apigroup="simple":
    {name: VidyoRoomRegisterModerationCommandEventListener}
    {parent: VidyoRoom}
    {description: Register a callback to receive moderation commands sent from the server.}
    {prototype: LmiBool VidyoRoomRegisterModerationCommandEventListener(VidyoRoom* room, VidyoRoomOnModerationCommandReceived onModerationCommand)}
    {parameter: {name: room} {description: Vidyo room object.}}
    {parameter: {name: onModerationCommand} {description: Callback to be triggered upon receive moderation command from the server.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomRegisterModerationCommandEventListener(VidyoRoom* room, VidyoRoomOnModerationCommandReceived onModerationCommand);

/**
{function apigroup="simple":
    {name: VidyoRoomUnregisterModerationCommandEventListener}
    {parent: VidyoRoom}
    {description: Unregister a callback to do not receive moderation commands sent from the server.}
    {prototype: LmiBool VidyoRoomUnregisterModerationCommandEventListener(VidyoRoom* room)}
    {parameter: {name: room} {description: Vidyo room object.}}
    {return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomUnregisterModerationCommandEventListener(VidyoRoom* room);

/**
{function apigroup="simple":
	{name: VidyoRoomRegisterConferenceModeEventListener}
	{parent: VidyoRoom}
	{description: Registers callbacks to get notified about conference mode events,i.e. lobby mode/lecture mode/waiting room.}
	{prototype: LmiBool VidyoRoomRegisterConferenceModeEventListener(VidyoRoom* r, VidyoRoomOnConferenceModeChanged conferenceModeChanged)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: conferenceModeChanged}
		{description: The conference mode i.e. Lecture mode/lobby room/group mode}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomRegisterConferenceModeEventListener(VidyoRoom* r, VidyoRoomOnConferenceModeChanged conferenceModeChanged);

/**
{function apigroup="simple":
	{name: VidyoRoomUnregisterConferenceModeEventListener}
	{parent: VidyoRoom}
	{description: UnRegisters callbacks to get notified about conference mode events.}
	{prototype: LmiBool VidyoRoomUnregisterConferenceModeEventListener(VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomUnregisterConferenceModeEventListener(VidyoRoom* r);

/**
{function apigroup="simple":
	{name: VidyoRoomRegisterHostEventListener}
	{parent: VidyoRoom}
	{description: Registers callbacks to get notified about room host joinin or leaving events}
	{prototype: LmiBool VidyoRoomRegisterHostEventListener(VidyoRoom* r,  VidyoRoomOnConferenceHostStatusChanged hostStatus)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: hostStatus}
		{description: The host status whether joining or leaving}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomRegisterHostEventListener(VidyoRoom* r,  VidyoRoomOnConferenceHostStatusChanged hostStatus);

/**
{function apigroup="simple":
	{name: VidyoRoomUnregisterHostEventListener}
	{parent: VidyoRoom}
	{description: UnRegisters callbacks to get notified about host joining or leaving events.}
	{prototype: LmiBool VidyoRoomUnregisterHostEventListener(VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomUnregisterHostEventListener(VidyoRoom* r);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnSetRoomPropertyResult}
	{parent: VidyoRoom}
	{description: Callback that is triggered when a VidyoRoomSetProperty request completes.}
	{prototype: void (*VidyoRoomOnSetRoomPropertyResult)(VidyoRoom* room, VidyoRoomSetPropertiesResult result)}
	{parameter:
		{name: room}
		{description: Pointer to the VidyoRoom that triggered the callback. }
	}
	{parameter:
		{name: result}
		{description: Result of the VidyoRoomSetProperty request.}
	}
}
*/
typedef void(*VidyoRoomOnSetRoomPropertyResult)(VidyoRoom* room, VidyoRoomSetPropertiesResult result);

/**
{function apigroup="simple" visibility="public":
	{name: VidyoRoomCreateParticipantProfile}
	{parent: VidyoRoom}
	{description: Creates a VidyoRoomParticipantProfile object.}
	{prototype: VidyoRoomParticipantProfile *VidyoRoomCreateParticipantProfile(VidyoRoom *r, VidyoRoomParticipantProfile *p, const char* userId, VidyoRoomProfileControlMode microphone, VidyoRoomProfileControlMode camera, VidyoRoomProfileControlMode windowShare, VidyoRoomProfileControlMode chat)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: p}
		{description: The VidyoRoomParticipantProfile object to construct.}
	}
	{parameter:
		{name: userId}
		{description: User ID if this is not a default profile.}
	}
	{parameter:
		{name: microphone}
		{description: The Microphone Profile.}
	}
	{parameter:
		{name: camera}
		{description: The Camera Profile.}
	}
	{parameter:
		{name: windowShare}
		{description: The WindowShare Profile.}
	}
	{parameter:
		{name: chat}{description: The Chat Profile.}
	}
	{return: A pointer to a created object on success, or NULL on failure.}
}
*/
VidyoRoomParticipantProfile *VidyoRoomCreateParticipantProfile(VidyoRoom *r, VidyoRoomParticipantProfile *p, const char* userId, VidyoRoomProfileControlMode microphone, VidyoRoomProfileControlMode camera, VidyoRoomProfileControlMode windowShare, VidyoRoomProfileControlMode chat);

/**
{function apigroup="simple" visibility="public":
	{name: VidyoRoomDestroyParticipantProfile}
	{parent: VidyoRoom}
	{description: Destroys a VidyoRoomParticipantProfile object.}
	{prototype: void VidyoRoomDestroyParticipantProfile(VidyoRoom *r, VidyoRoomParticipantProfile *p)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: p}
		{description: The VidyoRoomParticipantProfile object to construct.}
	}
}
*/
void VidyoRoomDestroyParticipantProfile(VidyoRoom *r, VidyoRoomParticipantProfile *p);

/**
{function apigroup="simple" visibility="public":
	{name: VidyoRoomSetDefaultProfile}
	{parent: VidyoRoom}
	{description: Set the default profile for all participants.
	 Only one SetRoomProperty request is allowed at a time.}
	{prototype: LmiBool VidyoRoomSetDefaultProfile(VidyoRoom *r, const VidyoRoomParticipantProfile *profile, VidyoRoomOnSetRoomPropertyResult onSetRoomPropertyResult)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: profile}
		{description: The default profile for all participants.}
	}
	{parameter:
		{name: onSetRoomPropertyResult}
		{description: Callback that is triggered when set property operation completes.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomSetDefaultProfile(VidyoRoom *r, const VidyoRoomParticipantProfile *profile, VidyoRoomOnSetRoomPropertyResult onSetRoomPropertyResult);

/**
{function apigroup="simple" visibility="public":
	{name: VidyoRoomSetParticipantProfiles}
	{parent: VidyoRoom}
	{description:Set the profiles for specific participants.
	 Only one SetRoomProperty request is allowed at a time.}
	{prototype: LmiBool VidyoRoomSetParticipantProfiles(VidyoRoom *r, const LmiVector(VidyoRoomParticipantProfile) *profiles, VidyoRoomOnSetRoomPropertyResult onSetRoomPropertyResult)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: profiles}
		{description: Set the profiles for specific participants.}
	}
	{parameter:
		{name: onSetRoomPropertyResult}
		{description: Callback that is triggered when set property operation completes.}
	}
	{return: LMI_TRUE if successful, LMI_FALSE otherwise.}
}
*/
LmiBool VidyoRoomSetParticipantProfiles(VidyoRoom *r, const LmiVector(VidyoRoomParticipantProfile) *profiles, VidyoRoomOnSetRoomPropertyResult onSetRoomPropertyResult);

/**
{function:
	{name: VidyoRoomAddVirtualAudioSource}
	{parent: VidyoRoom}
	{description: Add the virtual audio source to the room.}
	{prototype: void VidyoRoomAddVirtualAudioSource(VidyoRoom *r, const VidyoVirtualAudioSource *virtualAudioSource, LmiBool removeOthers)}
	{parameter: {name: r} {description: Vidyo room object.}}
	{parameter: {name: virtualAudioSource} {description: The VidyoVirtualAudioSource object.}}
	{parameter: {name: removeOthers} {description: Indicates that previously added virtual audio sources will be removed.}}
}
*/
void VidyoRoomAddVirtualAudioSource(VidyoRoom *r, const VidyoVirtualAudioSource *virtualAudioSource, LmiBool removeOthers);

/**
{function:
	{name: VidyoRoomRemoveVirtualAudioSource}
	{parent: VidyoRoom}
	{description: Remove the virtual audio source from the room.}
	{prototype: void VidyoRoomRemoveVirtualAudioSource(VidyoRoom *r, const VidyoVirtualAudioSource *virtualAudioSource)}
	{parameter: {name: r} {description: Vidyo room object.}}
	{parameter: {name: virtualAudioSource} {description: The VidyoVirtualAudioSource object.}}
}
*/
void VidyoRoomRemoveVirtualAudioSource(VidyoRoom *r, const VidyoVirtualAudioSource *virtualAudioSource);

/**
{type apigroup="simple":
	{name: VidyoRoomInCallEventClass}
	{parent: VidyoRoom}
	{description: It identifies the class of in call event.}
	{value: {name: VIDYO_ROOMINCALLEVENTCLASS_MediaTransport}
		{description: It indicate event belongs to media transportation.}}
	{value: {name: VIDYO_ROOMINCALLEVENTCLASS_InCallCodec}
		{description: It indicate event belongs to InCallCodec.}}
	{value: {name: VIDYO_ROOMINCALLEVENTCLASS_Dtls}
		{description: It indicate event belongs to Dtls.}}
	{value: {name: VIDYO_ROOMINCALLEVENTCLASS_UnKnown}
		{description: It indicate Unknown event.}}
}
*/
typedef enum
{
	VIDYO_ROOMINCALLEVENTCLASS_MediaTransport,
	VIDYO_ROOMINCALLEVENTCLASS_InCallCodec,
	VIDYO_ROOMINCALLEVENTCLASS_Dtls,
	VIDYO_ROOMINCALLEVENTCLASS_UnKnown,
} VidyoRoomInCallEventClass;

/**
{type apigroup="simple":
	{name: VidyoRoomInCallEventCode}
	{parent: VidyoRoom}
	{description: It represents error code in call event.}
	{value: {name: VIDYO_ROOMINCALLEVENTCODE_RTCPSendError}
		{description: RTCP send fail.}}
	{value: {name: VIDYO_ROOMINCALLEVENTCODE_RTCPReceiveError}
		{description: RTCP receive fail.}}
	{value: {name: VIDYO_ROOMINCALLEVENTCODE_RTPSendError}
		{description: RTP media packet send fail.}}
	{value: {name: VIDYO_ROOMINCALLEVENTCODE_RTPReceiveError}
		{description: RTP media packet receive fail.}}
	{value: {name: VIDYO_ROOMINCALLEVENTCODE_EncoderPausedOnLowBW}
		{description: Notification when encoder pauses when low bandwidth condition occur.}}
	{value: {name: VIDYO_ROOMINCALLEVENTCODE_EncoderResumed}
		{description: Notification when encoder resumes after pause.}}
	{value: {name: VIDYO_ROOMINCALLEVENTCODE_InCallCodecVideoH264}
		{description: Notification when H264 video codec is used in a conference call.}}
	{value: {name: VIDYO_ROOMINCALLEVENTCODE_InCallCodecVideoH264SVC}
		{description: Notification when H264-SVC video codec is used in a conference call.}}
	{value: {name: VIDYO_ROOMINCALLEVENTCODE_InCallCodecAudioSPEEXRED}
		{description: Notification when SPEEX RED audio codec is used in a conference call.}}
	{value: {name: VIDYO_ROOMINCALLEVENTCODE_DtlsError}
		{description: Notification when Dtls encryption fails.}}
	{value: {name: VIDYO_ROOMINCALLEVENTCODE_UnKnown}
		{description: Unknown event code.}}
}
*/
typedef enum
{
	VIDYO_ROOMINCALLEVENTCODE_RTCPSendError,
	VIDYO_ROOMINCALLEVENTCODE_RTCPReceiveError,
	VIDYO_ROOMINCALLEVENTCODE_RTPSendError,
	VIDYO_ROOMINCALLEVENTCODE_RTPReceiveError,
	VIDYO_ROOMINCALLEVENTCODE_EncoderPausedOnLowBW,
	VIDYO_ROOMINCALLEVENTCODE_EncoderResumed,
	VIDYO_ROOMINCALLEVENTCODE_InCallCodecVideoH264,
	VIDYO_ROOMINCALLEVENTCODE_InCallCodecVideoH264SVC,
	VIDYO_ROOMINCALLEVENTCODE_InCallCodecAudioSPEEXRED,
	VIDYO_ROOMINCALLEVENTCODE_DtlsError,
	VIDYO_ROOMINCALLEVENTCODE_UnKnown,
} VidyoRoomInCallEventCode;

/**
{type apigroup="simple":
	{name: VidyoRoomInCallEvent}
	{parent: VidyoRoom}
	{description: It is the event which will be sent to application which register for incall events.}
	{member:
		{name: type}
		{type: VidyoRoomInCallEventClass}
		{description: This indicate the module to which event belongs to.}
	}
	{member:
		{name: code}
		{type: VidyoRoomInCallEventCode}
		{description: It is error code for the event.}
	}
	{member:
		{name: details}
		{type: LmiString}
		{description: It extra information for the event. It is optional and it may depend on type of event.}
	}
}
*/
typedef struct
{
	VidyoRoomInCallEventClass type;
	VidyoRoomInCallEventCode code;
	LmiString details;
} VidyoRoomInCallEvent;

/**
{function visibility="private":
	{name: VidyoRoomInCallEventConstructDefault}
	{parent: VidyoRoomInCallEvent}
	{description: Default constructor for VidyoRoomInCallEvent.}
	{prototype: VidyoRoomInCallEvent* VidyoRoomInCallEventConstructDefault(VidyoRoomInCallEvent* ev, LmiAllocator* alloc)}
	{parameter:
		{name: ev}
		{description: VidyoRoomInCallEvent object.}
	}
	{parameter:
		{name: alloc}
		{description: LmiAllocator object.}
	}
	{return: VidyoRoomInCallEvent object on success otherwise NULL.}
}
*/
VidyoRoomInCallEvent* VidyoRoomInCallEventConstructDefault(VidyoRoomInCallEvent* ev, LmiAllocator* alloc);

/**
{function visibility="private":
	{name: VidyoRoomInCallEventConstruct}
	{parent: VidyoRoomInCallEvent}
	{description: Constructs an VidyoRoomInCallEvent object as a copy of an existing objec.}
	{prototype: VidyoRoomInCallEvent* VidyoRoomInCallEventConstruct(VidyoRoomInCallEvent *ev, VidyoRoomInCallEventClass type, VidyoRoomInCallEventCode code, const char *description, LmiAllocator *alloc)}
	{parameter: {name: ev} {description: The VidyoRoomInCallEvent object to construct.}}
	{parameter: {name: type} {description: This indicate the module to which event belongs to.}}
	{parameter: {name: code} {description: It is error code for the event.}}
	{parameter: {name: description} {description: It extra information for the event. It is optional and it may depend on type of event.}}
	{parameter: {name: alloc} {description: LmiAllocator object.}}
	{return: VidyoRoomInCallEvent object on success otherwise NULL.}
}
*/
VidyoRoomInCallEvent* VidyoRoomInCallEventConstruct(VidyoRoomInCallEvent *ev, VidyoRoomInCallEventClass type, VidyoRoomInCallEventCode code, const char *description, LmiAllocator *alloc);

/**
{function visibility="private":
	{name: VidyoRoomInCallEventConstructCopy}
	{parent: VidyoRoomInCallEvent}
	{description: Constructs an VidyoRoomInCallEvent object as a copy of an existing objec.}
	{prototype: VidyoRoomInCallEvent* VidyoRoomInCallEventConstructCopy(VidyoRoomInCallEvent* d, VidyoRoomInCallEvent* s)}
	{parameter:
		{name: d}
		{description: The VidyoRoomInCallEvent object to construct.}
	}
	{parameter:
		{name: s}
		{description: The VidyoRoomInCallEvent object to make a copy of.}
	}
	{return: VidyoRoomInCallEvent object on success otherwise NULL.}
}
*/
VidyoRoomInCallEvent* VidyoRoomInCallEventConstructCopy(VidyoRoomInCallEvent* d, VidyoRoomInCallEvent* s);

/**
{function visibility="private":
	{name: VidyoRoomInCallEventDestruct}
	{parent: VidyoRoomInCallEvent}
	{description: Destructor for VidyoRoomInCallEvent.}
	{prototype: void VidyoRoomInCallEventDestruct(VidyoRoomInCallEvent* ev)}
	{parameter:
		{name: ev}
		{description: VidyoRoomInCallEvent object.}
	}
}
*/
void VidyoRoomInCallEventDestruct(VidyoRoomInCallEvent* ev);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnInCallEvents}
	{parent: VidyoRoom}
	{description: A callback to be invoked when any in call event to be sent to application.}
	{prototype: void(*VidyoRoomOnInCallEvents)(VidyoRoom* r, const VidyoRoomInCallEvent* ev)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object that triggered the callback.}
	}
	{parameter:
		{name: ev}
		{description: Detail of event.}
	}
}
*/
typedef void(*VidyoRoomOnInCallEvents)(VidyoRoom* r, const VidyoRoomInCallEvent* ev);

/**
{function apigroup="simple":
	{name: VidyoRoomRegisterInCallEventListener}
	{parent: VidyoRoom}
	{description: Registers a callback to be will be triggered during a conference, these events may not stop the conferece but may degrade quality of call.}
	{prototype: LmiBool VidyoRoomRegisterInCallEventListener(VidyoRoom* r, VidyoRoomOnInCallEvents cb)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{parameter:
		{name: cb}
		{description: Callback which will be triggered when error occur in conference.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomRegisterInCallEventListener(VidyoRoom* r, VidyoRoomOnInCallEvents cb);

/**
{function apigroup="simple":
	{name: VidyoRoomUnregisterInCallEventListener}
	{parent: VidyoRoom}
	{description: Unregisters callback for in call events.}
	{prototype: LmiBool VidyoRoomUnregisterInCallEventListener(VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomUnregisterInCallEventListener(VidyoRoom* r);

/**
{type visibility="public":
	{name:  VidyoRoomBotInfo}
	{parent: VidyoRoom}
	{description: Represents the bot info structure.}
	{member:
		{name: name}
		{type: LmiString}
		{description: Represents the name associated with bot.}}
	{member:
		{name: id}
		{type: LmiString}
		{description: Represents the id associated with bot.}}
	{member:
		{name: data}
		{type: LmiString}
		{description: Represents the meta data associated with bot in json format string.}}
}
*/
typedef struct
{
	LmiString name;
	LmiString id;
	LmiString data;
} VidyoRoomBotInfo;

/**
{function visibility="private":
	{name: VidyoRoomBotInfoConstructDefault}
	{parent: VidyoRoomBotInfo}
	{description: Constructs an VidyoRoomBotInfo object.}
	{prototype: VidyoRoomBotInfo* VidyoRoomBotInfoConstructDefault(VidyoRoomBotInfo* info, LmiAllocator* alloc)}
	{parameter:
		{name: info}
		{description: Represents the VidyoRoomBotInfo object to construct.}}
	{parameter:
		{name: alloc}
		{description: Represents the LmiAllocator object.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoRoomBotInfo* VidyoRoomBotInfoConstructDefault(VidyoRoomBotInfo* info, LmiAllocator* alloc);

/**
{function visibility="private":
	{name: VidyoRoomBotInfoConstruct}
	{parent: VidyoRoomBotInfo}
	{description: Constructs an VidyoRoomBotInfo object.}
	{prototype: VidyoRoomBotInfo* VidyoRoomBotInfoConstruct(VidyoRoomBotInfo* info, const LmiString* name, const LmiString* id, const LmiString* data, LmiAllocator* alloc)}
	{parameter:
		{name: info}
		{description: Represents the VidyoRoomBotInfo object to construct.}}
	{parameter:
		{name: name}
		{description: Represents the name associated with bot.}}
	{parameter:
		{name: id}
		{description: Represents the id associated with bot.}}
	{parameter:
		{name: data}
		{description: Represents the meta data associated with bot in json fomat string.}}
	{parameter:
		{name: alloc}
		{description: Represents the LmiAllocator object.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoRoomBotInfo* VidyoRoomBotInfoConstruct(VidyoRoomBotInfo* info, const LmiString* name, const LmiString* id, const LmiString* data, LmiAllocator* alloc);

/**
{function visibility="private":
	{name: VidyoRoomBotInfoConstructCopy}
	{parent: VidyoRoomBotInfo}
	{description: Constructs an VidyoRoomBotInfo object as a copy of an existing object.}
	{prototype: VidyoRoomBotInfo* VidyoRoomBotInfoConstructCopy(VidyoRoomBotInfo* d, const VidyoRoomBotInfo* s)}
	{parameter:
		{name: d}
		{description: The VidyoRoomBotInfo object to construct.}}
	{parameter:
		{name: s}
		{description: The existing VidyoRoomBotInfo object to copy.}}
	{return: A pointer to a constructed object on success, or NULL on failure.}
}
*/
VidyoRoomBotInfo* VidyoRoomBotInfoConstructCopy(VidyoRoomBotInfo* d, const VidyoRoomBotInfo* s);

/**
{function visibility="private":
	{name: VidyoRoomBotInfoAssign}
	{parent: VidyoRoomBotInfo}
	{description: Assign the contents of one VidyoRoomBotInfo object to another.}
	{prototype: VidyoRoomBotInfo* VidyoRoomBotInfoAssign(VidyoRoomBotInfo* d, const VidyoRoomBotInfo* s)}
	{parameter:
		{name: d}
		{description: The VidyoRoomBotInfo object to assign to.}}
	{parameter:
		{name: s}
		{description: The VidyoRoomBotInfo object to copy.}}
	{return: A pointer to the assigned object on success, or NULL on failure.}
}
*/
VidyoRoomBotInfo* VidyoRoomBotInfoAssign(VidyoRoomBotInfo* d, const VidyoRoomBotInfo* s);

/**
{function visibility="private":
	{name: VidyoRoomBotInfoDestruct}
	{parent: VidyoRoomBotInfo}
	{description: Destructs an VidyoRoomBotInfo object.}
	{prototype: void VidyoRoomBotInfoDestruct(VidyoRoomBotInfo* info)}
	{parameter:
		{name: info}
		{description: Represents the VidyoRoomBotInfo object to destruct.}}
}
*/
void VidyoRoomBotInfoDestruct(VidyoRoomBotInfo* info);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnBotJoined}
	{parent: VidyoRoom}
	{description: Callback that is triggered when bot participant joins a conference.}
	{prototype: void(*VidyoRoomOnBotJoined)(VidyoRoom* r, const VidyoRoomBotInfo* info)}
	{parameter: {name: r} {description: The VidyoRoom object that triggered the callback.}}
	{parameter: {name: info} {description: Represents the VidyoRoomBotInfo object.}}
}
*/
typedef void(*VidyoRoomOnBotJoined)(VidyoRoom* r, const VidyoRoomBotInfo* info);

/**
{callback apigroup="simple":
	{name: VidyoRoomOnBotLeft}
	{parent: VidyoRoom}
	{description: Callback that is triggered when bot participant leaves a conference.}
	{prototype: void(*VidyoRoomOnBotLeft)(VidyoRoom* r, const VidyoRoomBotInfo* info)}
	{parameter: {name: r} {description: The VidyoRoom object that triggered the callback.}}
	{parameter: {name: info} {description: Represents the VidyoRoomBotInfo object.}}
}
*/
typedef void(*VidyoRoomOnBotLeft)(VidyoRoom* r, const VidyoRoomBotInfo* info);

/**
{function apigroup="simple":
	{name: VidyoRoomRegisterBotEventListener}
	{parent: VidyoRoom}
	{description: This is a synchronous interface. Application can register a callback to get notified about bot participants.}
	{prototype: LmiBool VidyoRoomRegisterBotEventListener(VidyoRoom* r, VidyoRoomOnBotJoined onJoined, VidyoRoomOnBotLeft onLeft)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}}
	{parameter:
		{name: onJoined}
		{description: Callback that is triggered when bot participant joins a conference.}}
	{parameter:
		{name: onLeft}
		{description: Callback that is triggered when an existing bot participant leaves a conference.}}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomRegisterBotEventListener(VidyoRoom* r, VidyoRoomOnBotJoined onJoined, VidyoRoomOnBotLeft onLeft);

/**
{function apigroup="simple":
	{name: VidyoRoomUnregisterBotEventListener}
	{parent: VidyoRoom}
	{description: This is a synchronous interface. Application can unregister callback to stop bot participant event notifications.}
	{prototype: LmiBool VidyoRoomUnregisterBotEventListener(VidyoRoom* r)}
	{parameter:
		{name: r}
		{description: The VidyoRoom object.}
	}
	{return: LMI_TRUE on success, LMI_FALSE on failure.}
}
*/
LmiBool VidyoRoomUnregisterBotEventListener(VidyoRoom* r);

#if LMI_INLINE_NEED_HEADER_FILE_DEFINITIONS
#include "VidyoRoomInline.h"
#endif

LMI_END_EXTERN_C

#endif /* VIDYO_ROOM_H_ */

