#pragma once

#include <stdint.h>

#include <bnb/error.h>
#include <bnb/export.h>

#ifdef __cplusplus
extern "C"
{
#endif

    /*
     * utility_manager
     */
    /**
     * opaque object representing utility_manager
     */
    typedef struct utility_manager_holder utility_manager_holder_t;

    /**
     * Creates a new instance of utility_manager
     *
     * @param resource_paths NULL terminated array of strings, e.g. {"path1", "path2", NULL}
     */
    BNB_EXPORT utility_manager_holder_t* bnb_utility_manager_init(const char* const* resource_paths, const char* client_token, bnb_error**);
    /**
     * Destroys utility_manager
     */
    BNB_EXPORT void bnb_utility_manager_release(utility_manager_holder_t*, bnb_error**);

    typedef enum
    {
        BNB_LOG_DEBUG,
        BNB_LOG_INFO,
        BNB_LOG_WARNING,
        BNB_LOG_ERROR,
    } bnb_severity_level_t;

    /**
     * log messages listener
     */
    typedef void (*bnb_log_record_callback_t)(bnb_severity_level_t level, const char* domain, const char* message, void* user_data);

    BNB_EXPORT void bnb_utility_manager_set_log_level(utility_manager_holder_t*, bnb_severity_level_t level, bnb_error**);
    /**
     * Set callback to recieve log events from Banuba SDK
     *
     * @param cb callback
     * @param level recieve logs with specified and higher levels
     * @param user_data user data will be passed to callback
     */
    BNB_EXPORT void bnb_utility_manager_set_log_record_callback(utility_manager_holder_t*, bnb_log_record_callback_t cb, bnb_severity_level_t level, void* user_data, bnb_error**);

    /*
     * version info
     */
    /**
     * Returns SDK version as string
     *
     * Returned string must be freed using bnb_memory_free()
     */
    BNB_EXPORT void bnb_get_banuba_sdk_version_string(char** version, bnb_error**);
    /**
     * Returns SDK version as number
     *
     * major = version / 10000000
     * minor = (version - major * 10000000) / 100000
     * patch = (version - major * 10000000 - minor * 100000) / 1000
     * revision = version % 1000
     */
    BNB_EXPORT int32_t bnb_get_banuba_sdk_version_int();

    /**
     * [Function is deprecated]
     * Returns resources version as string
     *
     * Returned string must be freed using bnb_memory_free()
     */
    BNB_EXPORT void bnb_get_banuba_sdk_nn_version_string(char** version, bnb_error**);

    /**
     * Returns resources version as string
     *
     * Returned string must be freed using bnb_memory_free()
     */
    BNB_EXPORT void bnb_get_banuba_sdk_resources_version_string(char** version, bnb_error**);
    /**
     * Returns resources version as number
     *
     * major = version / 10000000
     * minor = (version - major * 10000000) / 100000
     * patch = (version - major * 10000000 - minor * 100000) / 1000
     * revision = version % 1000
     */
    BNB_EXPORT int32_t bnb_get_banuba_sdk_resources_version_int();

#ifdef __cplusplus
} // extern "C"
#endif
