'use strict';

const prefabs = require("bnb_js/prefabs");

const AlphaMode = {
    /* 1 */ COMBINED: 1,
    /* 2 */ SEPARATE: 2,
}

class VideoTexture extends prefabs.Base {
    constructor(faceIndex=0) {
        super();
        const assets = bnb.scene.getAssetManager();
        const materialName = `mat_video_texture`;
        const material = assets.findMaterial(materialName);
        if (!material) {
            bnb.log(`[WARN] Unable to find material "${materialName}"`);
            return;
        }
        this._video_texture = assets.findImage("video_texture_src")?.asVideo();
        this._alpha_texture = assets.findImage("alpha_texture_src")?.asVideo();

        const parameters = material.getParameters();
        this._alpha = parameters.find((parameter) => parameter.getName() == "use_alpha");

        const entity = bnb.scene.getRoot().findChildByName(`video_texture:${faceIndex}`);
        this._transform = entity.getComponent(bnb.ComponentType.TRANSFORMATION).asTransformation();

        this._alpha_param = 0.;
        this._video_name = "";

        bnb.eventListener.on("onActivate", () => {
            if (this._alpha_param != 0)
                this._alpha.setVector4(new bnb.Vec4(this._alpha_param, 0., 0., 1.));

            if (this._video_name != "")
            {
                this._video_texture.load(this._video_name, (success) => {
                    if (success) {
                        if (this._alpha_param == AlphaMode.SEPARATE) {
                            this._alpha_texture.asMedia().setLooped(true);
                            this._alpha_texture.asMedia().play();
                        }
                        this._video_texture.asMedia().setLooped(true);
                        this._video_texture.asMedia().play();
                    }
                });
            }
        })
    }

    _mesh() {}

    video(filename) {
        this._video_name = filename;
        return this;
    }
    
    alpha(filename) {
        this._alpha_texture.load(filename, (success) => {
            if (success) {
                this._alpha_texture.asMedia().setLooped(true);
                this._alpha_texture.asMedia().play();
            }
        });
        return this;
    }

    useSeparateAlpha(use) {        
        if (use)
            this._alpha_param = AlphaMode.SEPARATE;
        else
            this._alpha_param = AlphaMode.COMBINED;
        return this;
    }

    translation(tr) {
        const [x, y, z] = tr.split(" ").map((c) => parseFloat(c));
        this._transform.setTranslation(new bnb.Vec3(x, y, z));
        return this;
    }

    scale(scale) {
        const [x, y, z] = scale.split(" ").map((c) => parseFloat(c));
        this._transform.setScale(new bnb.Vec3(x, y, z));
        return this;
    }

    rotation(rot) {
        const [x, y, z] = rot.split(" ").map((c) => parseFloat(c) * (Math.PI / 180));
        this._transform.setRotation(new bnb.Vec3(x, y, z));
        return this;
    }

    /** Resets any settings applied */
    clear() {
        this.translation("0 0 0");
        this.scale("1 1 1");
        this.rotation("0 0 0");
    }
}

exports = {
    VideoTexture
}