/** {file:
      {name: LmiBattery.h}
      {description: OS independent interface to battery information.}
      {copyright:
      	(c) 2006-2017 Vidyo, Inc.,
      	433 Hackensack Avenue,
      	Hackensack, NJ  07601.

      	All rights reserved.

      	The information contained herein is proprietary to Vidyo, Inc.
      	and shall not be reproduced, copied (in whole or in part), adapted,
      	modified, disseminated, transmitted, transcribed, stored in a retrieval
      	system, or translated into any language in any form by any means
      	without the express written consent of Vidyo, Inc.
      	                  ***** CONFIDENTIAL *****
      }
    }
*/
#ifndef LMI_BATTERY_H_
#define LMI_BATTERY_H_

#include <Lmi/Utils/LmiTypes.h>
#include <Lmi/Utils/LmiAllocator.h>
#include <Lmi/Utils/LmiString.h>

LMI_BEGIN_EXTERN_C

/**
 {type:
	{name: LmiBatteryState}
	{parent: Os}
	{include: Lmi/Os/LmiBattery.h}
 	{description: The state of a battery.}
 	{value: {name: LMI_BATTERYSTATE_Inactive} {description: Indicates that the battery is currently inactive (no power is being drawn).}}
 	{value: {name: LMI_BATTERYSTATE_Active} {description: Indicates that the battery is currently active (power is being drawn).}}
 	{value: {name: LMI_BATTERYSTATE_Unknown} {description: Indicates that the battery is currently in an unknowns state.  For example battery monitoring is off or there is not battery in the device.}}
}
*/
typedef enum {
	LMI_BATTERYSTATE_Inactive,
	LMI_BATTERYSTATE_Active,
	LMI_BATTERYSTATE_Unknown
} LmiBatteryState;

/**
{type:
	{name: LmiBattery}
	{parent: Os}
	{include: Lmi/Os/LmiBattery.h}
 	{description: A class representing the device\'s battery.}
}
*/
typedef struct {
	void* implementation;
	LmiAllocator* a;
} LmiBattery;

/**
{function:
	{name: LmiBatteryConstructDefault}
	{parent: LmiBattery}
	{description: Constructs a battery object. Applications are advised to retrieve the default battery object via a call to
 	  LmiBatteryGetDefault rather than constructing a new battery object with this function.}
	{prototype: LmiBattery* LmiBatteryConstructDefault(LmiBattery* x, LmiAllocator* a)}
	{parameter:
		{name: x}
		{description: The battery to construct.}
	}
 	{parameter:
		{name: a}
		{description: An allocator.}
	}
	{return: Returns a pointer to the battery object, or NULL on failure.}
}
*/
LmiBattery* LmiBatteryConstructDefault(LmiBattery* x, LmiAllocator* a);

/**
{function:
	{name: LmiBatteryDestruct}
	{parent: LmiBattery}
	{description: Destructs a battery object. Applications are advised to retrieve the default battery object via a call to
 	  LmiBatteryGetDefault rather than constructing/destructing a new battery object.}
	{prototype: void LmiBatteryDestruct(LmiBattery* x)}
	{parameter:
		{name: x}
		{description: The battery to destruct.}
	}
}
*/
void LmiBatteryDestruct(LmiBattery* x);

/**
{function:
	{name: LmiBatteryGetDefault}
	{parent: LmiBattery}
 	{description: Retrieve the default battery object. Applications are advised to use this method rather than constructing/destructing a new battery object.  The battery retrieved by this method must {bold:not} be destructed. }
	{prototype: const LmiBattery* LmiBatteryGetDefault()}
 	{return: Returns a pointer to the default battery object, or NULL on failure.}
}
*/
const LmiBattery* LmiBatteryGetDefault(void);

/**
{function:
	{name: LmiBatteryIsMonitoringEnabled}
	{parent: LmiBattery}
 	{description: Determine whether or not battery monitoring is enabled. If battery monitoring is not enabled LmiBatteryGetState will return
		LMI_BATTERYSTATE_Unknown and LmiBatteryGetLevel will return -1.0 (unknown). On iOS enabling battery monitoring is an application wide,
		global state and therefore must be enabled by the application by calling: [[UIDevice currentDevice] setBatteryMonitoringEnabled: YES].
		On macOS, Windows, and Linux battery monitoring is always enabled.}
	{prototype: LmiBool LmiBatteryIsMonitoringEnabled(const LmiBattery* x)}
	{parameter:
		{name: x}
		{description: The battery object.}
	}
	{return: Returns LMI_TRUE on success or LMI_FALSE on failure.}
}
*/
LmiBool LmiBatteryIsMonitoringEnabled(const LmiBattery* x);

/**
{function:
	{name: LmiBatteryGetState}
	{parent: LmiBattery}
	{description: Determine the battery state.  If battery monitoring is not enabled this method returns LMI_BATTERYSTATE_Unknown.}
	{prototype: LmiBatteryState LmiBatteryGetState(const LmiBattery* x)}
	{parameter:
		{name: x}
		{description: The battery object.}
	}
	{return: Returns the state of the battery.}
}
*/
LmiBatteryState LmiBatteryGetState(const LmiBattery* x);

/**
{function:
	{name: LmiBatteryGetLevel}
	{parent: LmiBattery}
	{description: Determine the battery level.  If battery monitoring is not enabled this method returns -1.0 indicating
 	  the level is unknown.}
	{prototype: LmiFloat64 LmiBatteryGetLevel(const LmiBattery* x)}
	{parameter:
		{name: x}
		{description: The battery object.}
	}
	{return: Returns a value between 0.0 (depleted) to 1.0 (fully charged) or -1.0 if the level could not be determined.}
}
*/
LmiFloat64 LmiBatteryGetLevel(const LmiBattery* x);

/* OS dependent interface */
LmiBattery* LmiBatteryConstructDefault_(LmiBattery* x, LmiAllocator* a);
void LmiBatteryDestruct_(LmiBattery* x);
LmiBool LmiBatteryIsMonitoringEnabled_(const LmiBattery* x);
LmiBatteryState LmiBatteryGetState_(const LmiBattery* x);
LmiFloat64 LmiBatteryGetLevel_(const LmiBattery* x);

LMI_END_EXTERN_C

#endif /* LMI_BATTERY_H_ */

